// -*- c++ -*-
#ifndef _GLIBMM_CONTAINERS_H
#define _GLIBMM_CONTAINERS_H

/* $Id: containers.h,v 1.13 2002/04/10 05:55:37 daniel Exp $ */

/* containers.h
 *
 * Copyright (C) 1998-2001 The Gtk-- Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <glib/glist.h>
#include <glib/gslist.h>
#include <glibmm/sarray.h> /* for backward compatibility */

#include <iterator>
#include <glibmmconfig.h>

GTKMM_USING_STD(bidirectional_iterator_tag)
GTKMM_USING_STD(forward_iterator_tag)


#ifndef DOXYGEN_SHOULD_SKIP_THIS

namespace Glib
{

template <class T> struct List_Iterator;
template <class T> struct List_ConstIterator;
template <class T> struct List_ReverseIterator;
/*
// Most of these methods in the non-template classes needs to be moved
// to implementation.

//: (internal) GList wrapper for gtk-- owned lists.
class ListImpl
{
  GList *list_;
  ListImpl(const ListImpl&);
public:
  ListImpl(): list_(0) {}
  ~ListImpl()         {g_list_free(list_);}

  GList* const &  list() const {return list_;}
  GList*& list() {return list_;} // is it allowed, if we own the list??

  int index_of(gpointer data)
    {return g_list_index(list_,data);}
  int position(GList *pos)
    {return g_list_position(list_,pos);}
  GList* first() const
    {return g_list_first(list_);}
  GList* last() const
    {return g_list_last(list_);}

  size_t size() const
    { return g_list_length(list_); }
  bool empty() const
    { return size()==0; }

  void reverse()
    {list_=g_list_reverse(list_);}

  GList* insert(GList* pos,gpointer data);

  void remove(gpointer data)
    {list_=g_list_remove(list_,data);}

  GList* erase(GList* pos);

  void clear()
    {
      g_list_free(list_);
      list_=0;
    }
};

//: (internal) GList wrapper for gtk+ owned lists.
class ListWrap
{
  GList*& list_;
public:
  ListWrap(GList*& l)           : list_(l)       {}
  ListWrap(const ListWrap& l) : list_(l.list_) {}

  GList*& list() {return list_;}
  GList* const& list() const {return list_;}

  int index_of(gpointer data)
    {return g_list_index(list_,data);}
  int position(GList *pos)
    {return g_list_position(list_,pos);}
  GList* first()
    {return g_list_first(list_);}
  GList* last()
    {return g_list_last(list_);}

  size_t size() const
    { return g_list_length(list_); }
  bool empty() const
    { return size()==0; }

  void reverse()
    {list_=g_list_reverse(list_);}

  GList* insert(GList* pos,gpointer data);

  void remove(gpointer data)
    {list_=g_list_remove(list_,data);}

  GList* erase(GList* pos);

  void clear()
    {
      g_list_free(list_);
      list_=0;
    }
};

//: STL C++ style wrapper for GList
//- This wraps simple types for use in Glist.  It is not intended
//- to replace STL lists, only be used where a GList is needed to wrap
//- gtk+ internals.  It can only hold C types.  They must not have a dtor,
//- nor should be larger than the size of a pointer.
template <class T,class Impl=ListImpl>
class List
{
public:
  typedef List<T,Impl>                         Self;
  typedef List_Iterator<T>                     iterator;
  typedef List_ConstIterator<iterator>         const_iterator;
  typedef List_ReverseIterator<iterator>       reverse_iterator;
  typedef List_ReverseIterator<const_iterator> reverse_const_iterator;
  typedef size_t                                 size_type;

private:
  Impl impl;
  List(const Self& x);

  gpointer to_pointer(const T& d)
    { return const_cast<gpointer&>(reinterpret_cast<const void* const&>(d)); }

public:
  List()              : impl()  {}
  List(const Impl& i) : impl(i) {}
  ~List()                       {}

  GList* & glist() const { return impl.list(); }

  iterator begin()             { return iterator(impl.list(),impl.first()); }
  iterator end()               { return iterator(impl.list(),(GList*)0); }
  const_iterator begin() const { return const_iterator(impl.list(),impl.first()); }
  const_iterator end()   const { return const_iterator(impl.list(),(GList*)0); }
  reverse_iterator rbegin()    { return reverse_const_iterator(end()); }
  reverse_iterator rend()      { return reverse_const_iterator(begin()); }
  reverse_const_iterator rbegin() const { return reverse_const_iterator(end());}
  reverse_const_iterator rend()   const { return reverse_const_iterator(begin());}

  iterator insert(iterator pos,const T& data)
    { return iterator(impl.list(),impl.insert(pos.node,to_pointer(data))); }

  iterator erase(iterator pos)
    { return iterator(impl.erase(pos));}
  void remove(const T& data)
    { impl.remove(to_pointer(data)); }
  void clear() {impl.clear();}

  void push_front(const T& data) { insert(begin(), data); }
  void push_back(const T& data)  { insert(end(), data); }
  void pop_front()               { erase(begin()); }
  void pop_back()                { erase(impl.last());}

  void reverse()                 { impl.reverse();}

  size_type size() const         { return impl.size(); }
  bool empty() const             { return impl.empty(); }

};
*/

/********************************************************************/


extern gpointer glibmm_null_pointer;

template <class T>
struct List_Iterator
{
  typedef std::bidirectional_iterator_tag iterator_category;
  typedef size_t size_type;
  typedef ptrdiff_t difference_type;

  typedef T  value_type;
  typedef T* pointer;
  typedef T& reference;

  GList * const *head;
  GList * node;

  typedef List_Iterator<T> Self;

  List_Iterator(GList* const& h,GList *n) : head(&h)    ,node(n)      {}
  List_Iterator()                   : head(0)     ,node(0)      {}
  List_Iterator(const Self& x)      : head(x.head),node(x.node) {}

  bool operator==(const Self& x) const { return node == x.node; }
  bool operator!=(const Self& x) const { return node != x.node; }

  Self&  operator++()
  {
    if (!node)
      node=g_list_first(*head);
    else
      node=(GList*)g_list_next(node);
    return *this;
  }

  Self operator++(int)
  {
    Self tmp = *this;
    ++*this;
    return tmp;
  }

  Self&  operator--()
  {
    if (!node)
      node=g_list_last(*head);
    else
      node=(GList*)g_list_previous(node);
    return *this;
  }

  Self operator--(int)
  {
    Self tmp = *this;
    --*this;
    return tmp;
  }

  reference operator*()  const 
  { 
    return (T&)((node)?(*node).data:glibmm_null_pointer);
  }
  
  pointer operator -> () const { return &operator*(); }
};

template <class T>
struct SList_Iterator
{
  typedef std::forward_iterator_tag iterator_category;
  typedef size_t size_type;
  typedef ptrdiff_t difference_type;

  typedef T  value_type;
  typedef T* pointer;
  typedef T& reference;

  GSList *node;
  typedef SList_Iterator<T> Self;

  SList_Iterator(GSList *n)     : node(n)      {}
  SList_Iterator()              : node(0)      {}
  SList_Iterator(const Self& x) : node(x.node) {}

  bool operator==(const Self& x) const { return node == x.node; }
  bool operator!=(const Self& x) const { return node != x.node; }

  Self&  operator++()
  {
    node = g_slist_next(node);
    return *this;
  }

  Self operator++(int)
  {
    Self tmp = *this;
    ++*this;
    return tmp;
  }

  reference operator*()  const { return reinterpret_cast<T&>((node)?(*node).data:glibmm_null_pointer); }
  pointer operator -> () const { return &operator*(); }
};

// this iterator variation returns interf (wrapped from impl)
//  Equivelency  List_Cpp_Iterator<GtkWidget,Gtk_Widget>
//     => std::list<Gtk_Widget*>::iterator
template<class impl, class interf>
struct List_Cpp_Iterator {
  typedef std::bidirectional_iterator_tag iterator_category;
  typedef size_t size_type;
  typedef ptrdiff_t difference_type;

  typedef interf*  value_type;
  typedef interf** pointer;
  typedef interf*& reference;

  typedef List_Cpp_Iterator<impl,interf> Self;

  GList **head;
  GList *node;

  bool operator==(const Self& x) const { return node == x.node; }
  bool operator!=(const Self& x) const { return node != x.node; }

  List_Cpp_Iterator(GList*& h,GList *n) : head(&h)    ,node(n)      {}
  List_Cpp_Iterator()                   : head(0)     ,node(0)      {}
  List_Cpp_Iterator(const Self& x)      : head(x.head),node(x.node) {}

  value_type operator*() const
  {
    if (node && node->data)
      return Glib::wrap(static_cast<impl*>((*node).data));
    return 0;
  }

  Self&  operator++()
  {
    if (!node)
      node=g_list_first(*head);
    else
      node = (GList *)g_list_next(node);
    return *this;
  }

  Self operator++(int)
  {
    Self tmp = *this;
    ++*this;
    return tmp;
  }

  Self&  operator--()
  {
    if (!node)
      node=g_list_last(*head);
    else
      node = (GList *)g_list_previous(node);
    return *this;
  }

  Self operator--(int)
  {
    Self tmp = *this;
    --*this;
    return tmp;
  }

};

template <class B>
struct List_ReverseIterator: private B
{
  typedef typename B::iterator_category iterator_category;
  typedef typename B::size_type         size_type;
  typedef typename B::difference_type   difference_type;

  typedef typename B::value_type        value_type;
  typedef typename B::pointer           pointer;
  typedef typename B::reference         reference;

  typedef List_ReverseIterator<B>    Self;

  bool operator==(const Self& x) const { return B::operator==(x); }
  bool operator!=(const Self& x) const { return B::operator!=(x); }

  List_ReverseIterator(GList* const& h,GList *n) : B(h,n) {}
  List_ReverseIterator()                   : B()    {}
  List_ReverseIterator(const Self& x)      : B(x)   {}
  List_ReverseIterator(const B& x)         : B(x)   { ++(*this); }

  Self& operator++()   {B::operator--(); return *this;}
  Self& operator--()   {B::operator++(); return *this;}
  Self operator++(int) {Self s=*this; B::operator--(); return s;}
  Self operator--(int) {Self s=*this; B::operator++(); return s;}

  reference operator*() const { return B::operator*(); }
  pointer operator->()  const { return B::operator->(); }
};

template <class B>
struct List_ConstIterator: public B
{
  typedef typename B::iterator_category iterator_category;
  typedef typename B::size_type         size_type;
  typedef typename B::difference_type   difference_type;

  typedef const typename B::value_type  value_type;
  typedef const typename B::pointer     pointer;
  typedef const typename B::reference   reference;

  typedef List_ConstIterator<B>    Self;

  bool operator==(const Self& x) const { return B::operator==(x); }
  bool operator!=(const Self& x) const { return B::operator!=(x); }

  List_ConstIterator(GList* const& h,GList *n) : B(h,n) {}
  List_ConstIterator()                   : B()    {}
  List_ConstIterator(const Self& x)      : B(x)   {}
  List_ConstIterator(const B& x)         : B(x)   {}

  Self& operator++()   {B::operator++(); return *this;}
  Self& operator--()   {B::operator--(); return *this;}
  Self operator++(int) {Self s=*this; B::operator++(); return s;}
  Self operator--(int) {Self s=*this; B::operator--(); return s;}

  value_type operator*() const { return B::operator*(); }
  pointer operator->()   const { return B::operator->(); }
};

// for compatablity
//typedef List<int> IntList;
//typedef List_Iterator<int> IntList_Iterator;

} // namespace Glib

#endif /* DOXYGEN_SHOULD_SKIP_THIS */

#endif /* _GLIBMM_CONTAINERS_H */

