
#include <gtkmm/style.h>
#include <gtkmm/private/style_p.h>

// -*- c++ -*-
/* $Id: style.ccg,v 1.20 2002/04/16 13:01:52 murrayc Exp $ */

/* 
 *
 * Copyright 1998-1999 The Gtk-- Development Team
 * Copyright 2001      Free Software Foundation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtkstyle.h>

namespace Gtk
{

/* 
Design notes:

Okay these are my notes on how a GtkStyle works.
They are not meant to be taken as documentation as I didn't
write the code.  

styles keeps a copy of itself for each color depth.  Thus
changing a style after it has been attached wont work!

At allocation time a Gtk Style has 
  - all GC set to 0 as they will be allocated later
  - has all color values set to default values.
  - a ref count of 1 and an attach count of 0 (floating?)
  - a properly referenced font.
  - colormap and depth are invalid.
  - The style list and rcstyle are 0. (??? styles wasn't set explicitly!)

It returns to this state if the style is detatched from
all widgets.

Attach acts to sink the object removing it from the floating state.

attaching a style for the first time initializes it. 
Initializing a style
  - sets the colormap and depth.
  - sets the mid colors. (thus allowing user to set these would be pointless)
  - gets black and white from the colormap.
  - allocates all the colors.
  - uses gtk_gc_get to share a gc if there is a matching one.

Conclusions, we need to rework the concept of Gdk to allow
for const objects.  

*/

/*
Style*
Style::copy() const
{
  return (Style*)(gtk_style_copy(const_cast<GtkStyle*>(gobj())));
}
*/

void Style::set_font(const Pango::FontDescription& font_desc)
{
  g_return_if_fail(font_desc.gobj() != 0);

  // It will be freed when it goes out of scope.
  const Pango::FontDescription fontBefore (gobj()->font_desc, false);

  gobj()->font_desc = font_desc.gobj_copy();
}

Pango::FontDescription Style::get_font() const
{
  // Direct struct access seems the only way.
  return Pango::FontDescription(gobj()->font_desc, true); // true = make a copy.
}

void Style::set_bg(StateType state_type, const Gdk::Color& new_color)
{
  gobj()->bg[state_type] = *new_color.gobj();
}

void Style::set_light(StateType state_type, const Gdk::Color& new_color)
{
  gobj()->light[state_type] = *new_color.gobj();
}

void Style::set_dark(StateType state_type, const Gdk::Color& new_color)
{
  gobj()->dark[state_type] = *new_color.gobj();
}

void Style::set_mid(StateType state_type, const Gdk::Color& new_color)
{
  gobj()->mid[state_type] = *new_color.gobj();
}

void Style::set_base(StateType state_type, const Gdk::Color& new_color)
{
  gobj()->base[state_type] = *new_color.gobj();
}

void Style::set_fg(StateType state_type, const Gdk::Color& new_color)
{
  gobj()->fg[state_type] = *new_color.gobj();
}

void Style::set_text(StateType state_type, const Gdk::Color& new_color)
{
  gobj()->text[state_type] = *new_color.gobj();
}

void
Style::set_background(const Glib::RefPtr<const Gdk::Window>& window, StateType state_type)
{
  gtk_style_set_background(const_cast<GtkStyle*>(gobj()),
      const_cast<GdkWindow*>(window->gobj()), (GtkStateType)state_type);
}

/* deprecated:
void
Style::draw_hline(const Glib::RefPtr<Gdk::Window>     &window,
			   StateType   state_type,
			   int           x1,
			   int           x2,
			   int           y) const
{
  gtk_draw_hline(const_cast<GtkStyle*>(gobj()), window->gobj(), (GtkStateType)state_type, x1, x2, y);
}

void
Style::draw_vline(const Glib::RefPtr<Gdk::Window>     &window,
			   StateType   state_type,
			   int           y1,
			   int           y2,
			   int           x) const
{
  gtk_draw_vline(const_cast<GtkStyle*>(gobj()), window->gobj(), (GtkStateType)state_type, y1, y2, x);
}

void
Style::draw_shadow(const Glib::RefPtr<Gdk::Window>     &window,
			    StateType   state_type,
			    ShadowType  shadow_type,
			    int           x,
			    int           y,
			    int           width,
			    int           height) const
{
  gtk_draw_shadow(
      const_cast<GtkStyle*>(gobj()), window->gobj(),
      (GtkStateType)state_type, (GtkShadowType)shadow_type, x, y, width, height);
}

void
Style::draw_polygon(const Glib::RefPtr<Gdk::Window>     &window,
			     StateType   state_type,
			     ShadowType  shadow_type,
			     GdkPoint      *points,
			     int           npoints,
			     int           fill) const
{
  gtk_draw_polygon(
      const_cast<GtkStyle*>(gobj()), window->gobj(),
      (GtkStateType)state_type, (GtkShadowType)shadow_type, points, npoints, fill);
}

void
Style::draw_arrow(const Glib::RefPtr<Gdk::Window>     &window,
			   StateType   state_type,
			   ShadowType  shadow_type,
			   ArrowType   arrow_type,
			   int           fill,
			   int           x,
			   int           y,
			   int           width,
			   int           height) const
{
  gtk_draw_arrow(
      const_cast<GtkStyle*>(gobj()), window->gobj(),
      (GtkStateType)state_type, (GtkShadowType)shadow_type, (GtkArrowType)arrow_type,
      fill, x, y, width, height);
}

void
Style::draw_diamond(const Glib::RefPtr<Gdk::Window>     &window,
			     StateType   state_type,
			     ShadowType  shadow_type,
			     int           x,
			     int           y,
			     int           width,
			     int           height) const
{
  gtk_draw_diamond(
      const_cast<GtkStyle*>(gobj()), window->gobj(),
      (GtkStateType)state_type, (GtkShadowType)shadow_type,
      x, y, width, height);
}

void
Style::draw_string(const Glib::RefPtr<Gdk::Window>     &window,
		       StateType   state_type,
		       int x, int y,
		       const Glib::ustring& str) const
{
  gtk_draw_string(
      const_cast<GtkStyle*>(gobj()), window->gobj(),
      (GtkStateType)state_type, x, y, str.c_str());
}
*/

void
Style::apply_default_background(const Glib::RefPtr<const Gdk::Window>& window,
				    bool set_bg,
				    StateType state_type,
				    const Gdk::Rectangle& area,
				    int x, int y,
				    int width, int height)
{
  gtk_style_apply_default_background(
      gobj(), const_cast<GdkWindow*>(window->gobj()), set_bg,
      (GtkStateType)state_type, const_cast<GdkRectangle*>(area.gobj()),
      x, y, width, height);
}

Glib::RefPtr<Gdk::GC> Style::get_fg_gc  ( StateType state_type)
{
  return Glib::wrap(gobj()->fg_gc[state_type], true);
}

Glib::RefPtr<Gdk::GC> Style::get_bg_gc  ( StateType state_type)
{
  return Glib::wrap(gobj()->bg_gc[state_type], true);
}

Glib::RefPtr<Gdk::GC> Style::get_light_gc( StateType state_type)
{
  return Glib::wrap(gobj()->light_gc[state_type], true);
}

Glib::RefPtr<Gdk::GC> Style::get_dark_gc( StateType state_type)
{
  return Glib::wrap(gobj()->dark_gc[state_type], true);
}

Glib::RefPtr<Gdk::GC> Style::get_mid_gc ( StateType state_type)
{
  return Glib::wrap(gobj()->mid_gc[state_type], true);
}

Glib::RefPtr<Gdk::GC> Style::get_text_gc( StateType state_type)
{
  return Glib::wrap(gobj()->text_gc[state_type], true);
}

Glib::RefPtr<Gdk::GC> Style::get_base_gc( StateType state_type)
{
  return Glib::wrap(gobj()->base_gc[state_type], true);
}

Gdk::Color Style::get_background(StateType state_type) const
{
  GdkColor* pGdkColor = const_cast<GdkColor*>(&(gobj()->bg[state_type]));
  return Glib::wrap(pGdkColor, true); //true = take_copy.
}

Glib::RefPtr<Gdk::Pixmap> Style::get_background_pixmap(StateType state_type)
{
  return Gdk::Pixmap::wrap_specific_type(gobj()->bg_pixmap[state_type], true); //true = take_copy.
}

} // namespace Gtk


namespace
{
} // anonymous namespace


namespace Glib
{

Glib::RefPtr<Gtk::Style> wrap(GtkStyle* object, bool take_copy /* = false */)
{
  return Glib::RefPtr<Gtk::Style>( dynamic_cast<Gtk::Style*> (Glib::wrap_auto ((GObject*)(object), take_copy)) );
  //We use dynamic_cast<> in case of multiple inheritance.
}

} /* namespace Glib */


namespace Gtk
{


/* The *_Class implementation: */

GType Style_Class::get_type()
{
  if(!gtype_) // create the GType if necessary
  {
    // TODO: This is currently just optimized away, apparently with no harm.
    // Is it actually needed?
    // Make sure that the parent type has been created.
    CppClassParent::CppObjectType::get_type();

    // Create the wrapper type, with the same class/instance size as the base type.
    register_derived_type(gtk_style_get_type(), (GClassInitFunc) &class_init_function);

    // Add derived versions of interfaces, if the C type implements any interfaces:
  }

  return gtype_;
}

void Style_Class::class_init_function(BaseClassType* klass)
{
  CppClassParent::class_init_function((CppClassParent::BaseClassType*) klass);
}


Glib::ObjectBase* Style_Class::wrap_new(GObject* o)
{
  return new Style((GtkStyle*)(o));
}


/* The implementation: */

GtkStyle* Style::gobj_copy()
{
  reference();
  return gobj();
}

Glib::RefPtr<Style> Style::wrap_specific_type(GtkStyle* gobject, bool take_copy /* = false */) //static
{
  Glib::RefPtr<Style> refPtr;

  if(gobject)
  {
    //Check for an existing wrapper:
    Style* pCppObject = dynamic_cast<Style*>(Glib::ObjectBase::_get_current_wrapper(G_OBJECT(gobject)));
    if(pCppObject)
    {
      //Return the existing wrapper:
      refPtr = Glib::RefPtr<Style>(pCppObject);
    }
    else
    {
      //Create a new wrapper:
      refPtr = Glib::RefPtr<Style>( new Style(gobject) );
    }

    if(take_copy && refPtr)
      refPtr->reference();
  }

  return refPtr;
}

Style::Style(GtkStyle* castitem)
: Glib::Object((GObject*)(castitem))
{}

Style::~Style()
{
}

Style::CppClassType Style::style_class_; //Initialize static member.

GType Style::get_type()
{
  return style_class_.get_type();
}

GType Style::get_base_type()
{
  return gtk_style_get_type();
}

Glib::RefPtr<Gdk::GC> Style::get_black_gc()
{
  Glib::RefPtr<Gdk::GC> ref_ptr(Glib::wrap(gobj()->black_gc));

  if(!ref_ptr.is_null())
    ref_ptr->reference();

  return ref_ptr;
}

Glib::RefPtr<const Gdk::GC> Style::get_black_gc() const
{
  Glib::RefPtr<const Gdk::GC> ref_ptr(Glib::wrap(gobj()->black_gc));

  if(!ref_ptr.is_null())
    ref_ptr->reference();

  return ref_ptr;
}

Glib::RefPtr<Gdk::GC> Style::get_white_gc()
{
  Glib::RefPtr<Gdk::GC> ref_ptr(Glib::wrap(gobj()->white_gc));

  if(!ref_ptr.is_null())
    ref_ptr->reference();

  return ref_ptr;
}

Glib::RefPtr<const Gdk::GC> Style::get_white_gc() const
{
  Glib::RefPtr<const Gdk::GC> ref_ptr(Glib::wrap(gobj()->white_gc));

  if(!ref_ptr.is_null())
    ref_ptr->reference();

  return ref_ptr;
}


} // namespace Gtk

