/*
 *  Copyright (C) 2005 Robert Staudinger
 *
 *  This software is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "nsString.h"
#include "nsIDOMHTMLFormElement.h"

#include "gme-dom-html-form-element-private.h"
#include "gme-dom-html-collection-private.h"


enum {
	PROP_0,
	PROP_WRAPPED_PTR,
	_NUM_PROPS
};

static GmeDOMHTMLElementClass *gme_dom_html_form_element_parent_class = NULL;

static void
instance_init (GmeDOMHTMLFormElement *self)
{
	self->wrapped_ptr = NULL;
	self->is_disposed = FALSE;
}

static void
instance_dispose (GObject *instance)
{
	GmeDOMHTMLFormElement *self = GME_DOM_HTML_FORM_ELEMENT (instance);

	if (self->is_disposed)
		return;

	if (self->wrapped_ptr) NS_RELEASE (self->wrapped_ptr);
	self->wrapped_ptr = NULL;
	self->is_disposed = TRUE;

	gme_dom_html_form_element_parent_class->dispose (G_OBJECT (self));
}

static void
set_property (GObject      *object,
	      guint         prop_id,
	      GValue const *value,
	      GParamSpec   *pspec)
{
	GmeDOMHTMLFormElement *self = GME_DOM_HTML_FORM_ELEMENT (object);
	nsISupports *ptr = NULL;
	nsresult rv;

	switch (prop_id) {
	case PROP_WRAPPED_PTR:
		ptr = reinterpret_cast<nsISupports*>(g_value_get_pointer (value));
		rv = CallQueryInterface (ptr, &self->wrapped_ptr);
		if (NS_SUCCEEDED (rv) && self->wrapped_ptr) {
			NS_ADDREF (self->wrapped_ptr);
			/* constuction param, init parent */
			gme_dom_html_element_private_set_wrapped_ptr (GME_DOM_HTML_ELEMENT (self), self->wrapped_ptr);
		}
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
	}
}

static void
get_property (GObject    *object,
	      guint       prop_id,
	      GValue     *value,
	      GParamSpec *pspec)
{
	GmeDOMHTMLFormElement *self = GME_DOM_HTML_FORM_ELEMENT (object);

	switch (prop_id) {
	case PROP_WRAPPED_PTR:
		g_value_set_pointer (value, (gpointer) self->wrapped_ptr);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
	}
}

static void
class_init (GmeDOMHTMLFormElementClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
	GmeDOMHTMLElementClass *node_class = GME_DOM_HTML_ELEMENT_CLASS (klass);

	/* vfuncs */
	klass->dispose = instance_dispose;

	/* hook gobject vfuncs */
	gobject_class->dispose = instance_dispose;

	gme_dom_html_form_element_parent_class = (GmeDOMHTMLElementClass*) g_type_class_peek_parent (klass);

	gobject_class->set_property = set_property;
	gobject_class->get_property = get_property;

	g_object_class_install_property (gobject_class,
		PROP_WRAPPED_PTR,
		g_param_spec_pointer ("wrapped-ptr", _("Wrapped Pointer"),
			_("Pointer to the wrapped c++ object"),
			(GParamFlags)(G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY)));
}

GType
gme_dom_html_form_element_get_gtype (void)
{
        static GType type = 0;
        if (!type) {
                static const GTypeInfo info = {
                        sizeof (GmeDOMHTMLFormElementClass),
                        NULL,           /* base_init */
                        NULL,           /* base_finalize */
                        (GClassInitFunc) class_init,
                        NULL,           /* class_finalize */
                        NULL,           /* class_data */
                        sizeof (GmeDOMHTMLFormElement),
                        0,              /* n_preallocs */
                        (GInstanceInitFunc) instance_init,
                };
                type = g_type_register_static (GME_TYPE_DOM_HTML_ELEMENT, "GmeDOMHTMLFormElement", &info, (GTypeFlags)0);
        }
        return type;
}

void 
gme_dom_html_form_element_private_set_wrapped_ptr (GmeDOMHTMLFormElement *self, 
				      nsIDOMHTMLFormElement *wrapped_ptr)
{
	g_assert (self && wrapped_ptr);
	self->wrapped_ptr = wrapped_ptr;
	NS_ADDREF (self->wrapped_ptr);
	/* constuction param, init parent */
	gme_dom_html_element_private_set_wrapped_ptr (GME_DOM_HTML_ELEMENT (self), self->wrapped_ptr);
}

GmeDOMHTMLFormElement* 
gme_dom_html_form_element_new (nsIDOMHTMLFormElement *wrapped_ptr)
{
	return GME_DOM_HTML_FORM_ELEMENT (g_object_new (GME_TYPE_DOM_HTML_FORM_ELEMENT, "wrapped-ptr", wrapped_ptr, NULL));
}

GmeDOMHTMLCollection* 
gme_dom_html_form_element_get_elements (GmeDOMHTMLFormElement *self)
{
	nsIDOMHTMLCollection *param = NULL;
	GmeDOMHTMLCollection *object = NULL;
	nsresult rv;
	g_assert (self);
	
	rv = self->wrapped_ptr->GetElements (&param);
	if (NS_SUCCEEDED (rv)) {
		object = gme_dom_html_collection_new (param);
	}
	return object;
}

gint64 
gme_dom_html_form_element_get_length (GmeDOMHTMLFormElement *self)
{
	gint32 param;
	nsresult rv;
	g_assert (self);

	rv = self->wrapped_ptr->GetLength (&param);
	if (NS_SUCCEEDED (rv)) {
		return param;
	}
	return -1;
}

gchar* 
gme_dom_html_form_element_get_name (GmeDOMHTMLFormElement *self)
{
	nsAutoString param;
	nsresult rv;
	g_assert (self);

	rv = self->wrapped_ptr->GetName (param);
	if (NS_SUCCEEDED (rv)) {
		 return g_strdup (NS_ConvertUTF16toUTF8 (param).get ());
	}
	return NULL;
}

gboolean 
gme_dom_html_form_element_set_name (GmeDOMHTMLFormElement *self, const gchar *param)
{
	nsAutoString p;
	nsresult rv;
	g_assert (self);

	p.AssignWithConversion (param);
	rv = self->wrapped_ptr->SetName (p);
	return NS_SUCCEEDED (rv);
}

gchar* 
gme_dom_html_form_element_get_accept_charset (GmeDOMHTMLFormElement *self)
{
	nsAutoString param;
	nsresult rv;
	g_assert (self);

	rv = self->wrapped_ptr->GetAcceptCharset (param);
	if (NS_SUCCEEDED (rv)) {
		 return g_strdup (NS_ConvertUTF16toUTF8 (param).get ());
	}
	return NULL;
}

gboolean 
gme_dom_html_form_element_set_accept_charset (GmeDOMHTMLFormElement *self, const gchar *param)
{
	nsAutoString p;
	nsresult rv;
	g_assert (self);

	p.AssignWithConversion (param);
	rv = self->wrapped_ptr->SetAcceptCharset (p);
	return NS_SUCCEEDED (rv);
}

gchar* 
gme_dom_html_form_element_get_action (GmeDOMHTMLFormElement *self)
{
	nsAutoString param;
	nsresult rv;
	g_assert (self);

	rv = self->wrapped_ptr->GetAction (param);
	if (NS_SUCCEEDED (rv)) {
		 return g_strdup (NS_ConvertUTF16toUTF8 (param).get ());
	}
	return NULL;
}

gboolean 
gme_dom_html_form_element_set_action (GmeDOMHTMLFormElement *self, const gchar *param)
{
	nsAutoString p;
	nsresult rv;
	g_assert (self);

	p.AssignWithConversion (param);
	rv = self->wrapped_ptr->SetAction (p);
	return NS_SUCCEEDED (rv);
}

gchar* 
gme_dom_html_form_element_get_enctype (GmeDOMHTMLFormElement *self)
{
	nsAutoString param;
	nsresult rv;
	g_assert (self);

	rv = self->wrapped_ptr->GetEnctype (param);
	if (NS_SUCCEEDED (rv)) {
		 return g_strdup (NS_ConvertUTF16toUTF8 (param).get ());
	}
	return NULL;
}

gboolean 
gme_dom_html_form_element_set_enctype (GmeDOMHTMLFormElement *self, const gchar *param)
{
	nsAutoString p;
	nsresult rv;
	g_assert (self);

	p.AssignWithConversion (param);
	rv = self->wrapped_ptr->SetEnctype (p);
	return NS_SUCCEEDED (rv);
}

gchar* 
gme_dom_html_form_element_get_method (GmeDOMHTMLFormElement *self)
{
	nsAutoString param;
	nsresult rv;
	g_assert (self);

	rv = self->wrapped_ptr->GetMethod (param);
	if (NS_SUCCEEDED (rv)) {
		 return g_strdup (NS_ConvertUTF16toUTF8 (param).get ());
	}
	return NULL;
}

gboolean 
gme_dom_html_form_element_set_method (GmeDOMHTMLFormElement *self, const gchar *param)
{
	nsAutoString p;
	nsresult rv;
	g_assert (self);

	p.AssignWithConversion (param);
	rv = self->wrapped_ptr->SetMethod (p);
	return NS_SUCCEEDED (rv);
}

gchar* 
gme_dom_html_form_element_get_target (GmeDOMHTMLFormElement *self)
{
	nsAutoString param;
	nsresult rv;
	g_assert (self);

	rv = self->wrapped_ptr->GetTarget (param);
	if (NS_SUCCEEDED (rv)) {
		 return g_strdup (NS_ConvertUTF16toUTF8 (param).get ());
	}
	return NULL;
}

gboolean 
gme_dom_html_form_element_set_target (GmeDOMHTMLFormElement *self, const gchar *param)
{
	nsAutoString p;
	nsresult rv;
	g_assert (self);

	p.AssignWithConversion (param);
	rv = self->wrapped_ptr->SetTarget (p);
	return NS_SUCCEEDED (rv);
}

gboolean 
gme_dom_html_form_element_submit (GmeDOMHTMLFormElement *self)
{
	nsresult rv;
	g_assert (self);

	rv = self->wrapped_ptr->Submit ();
	return NS_SUCCEEDED (rv);
}

gboolean 
gme_dom_html_form_element_reset (GmeDOMHTMLFormElement *self)
{
	nsresult rv;
	g_assert (self);

	rv = self->wrapped_ptr->Reset ();
	return NS_SUCCEEDED (rv);
}

