/*
 *  Copyright (C) 2005 Robert Staudinger
 *
 *  This software is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "nsString.h"
#include "nsIDOMCSSStyleDeclaration.h"

#include "gme-dom-css-style-declaration-private.h"
#include "gme-dom-css-value-private.h"
#include "gme-dom-css-rule-private.h"

enum {
	PROP_0,
	PROP_WRAPPED_PTR,
	_NUM_PROPS
};

static GmeSupportsClass *gme_dom_css_style_declaration_parent_class = NULL;

static void
instance_init (GmeDOMCSSStyleDeclaration *self)
{
	self->wrapped_ptr = NULL;
	self->is_disposed = FALSE;
}

static void
instance_dispose (GObject *instance)
{
	GmeDOMCSSStyleDeclaration *self = GME_DOM_CSS_STYLE_DECLARATION (instance);

	if (self->is_disposed)
		return;

	if (self->wrapped_ptr) NS_RELEASE (self->wrapped_ptr);
	self->wrapped_ptr = NULL;
	self->is_disposed = TRUE;

	gme_dom_css_style_declaration_parent_class->dispose (G_OBJECT (self));
}

static void
set_property (GObject      *object,
	      guint         prop_id,
	      GValue const *value,
	      GParamSpec   *pspec)
{
	GmeDOMCSSStyleDeclaration *self = GME_DOM_CSS_STYLE_DECLARATION (object);
	nsISupports *ptr = NULL;
	nsresult rv;

	switch (prop_id) {
	case PROP_WRAPPED_PTR:
		ptr = reinterpret_cast<nsISupports*>(g_value_get_pointer (value));
		rv = CallQueryInterface (ptr, &self->wrapped_ptr);
		if (NS_SUCCEEDED (rv) && self->wrapped_ptr) {
			NS_ADDREF (self->wrapped_ptr);
			/* constuction param, init parent */
			gme_supports_private_set_wrapped_ptr (GME_SUPPORTS (self), self->wrapped_ptr);
		}
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
	}
}

static void
get_property (GObject    *object,
	      guint       prop_id,
	      GValue     *value,
	      GParamSpec *pspec)
{
	GmeDOMCSSStyleDeclaration *self = GME_DOM_CSS_STYLE_DECLARATION (object);

	switch (prop_id) {
	case PROP_WRAPPED_PTR:
		g_value_set_pointer (value, (gpointer) self->wrapped_ptr);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
	}
}

static void
class_init (GmeDOMCSSStyleDeclarationClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

	/* vfuncs */
	klass->dispose = instance_dispose;

	/* hook gobject vfuncs */
	gobject_class->dispose = instance_dispose;

	gme_dom_css_style_declaration_parent_class = (GmeSupportsClass*) g_type_class_peek_parent (klass);

	gobject_class->set_property = set_property;
	gobject_class->get_property = get_property;

	g_object_class_install_property (gobject_class,
		PROP_WRAPPED_PTR,
		g_param_spec_pointer ("wrapped-ptr", _("Wrapped Pointer"),
			_("Pointer to the wrapped c++ object"),
			(GParamFlags)(G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY)));
}

GType
gme_dom_css_style_declaration_get_gtype (void)
{
        static GType type = 0;
        if (!type) {
                static const GTypeInfo info = {
                        sizeof (GmeDOMCSSStyleDeclarationClass),
                        NULL,           /* base_init */
                        NULL,           /* base_finalize */
                        (GClassInitFunc) class_init,
                        NULL,           /* class_finalize */
                        NULL,           /* class_data */
                        sizeof (GmeDOMCSSStyleDeclaration),
                        0,              /* n_preallocs */
                        (GInstanceInitFunc) instance_init,
                };
                type = g_type_register_static (GME_TYPE_SUPPORTS, "GmeDOMCSSStyleDeclaration", &info, (GTypeFlags)0);
        }
        return type;
}

void 
gme_dom_css_style_declaration_private_set_wrapped_ptr (GmeDOMCSSStyleDeclaration *self, 
				      nsIDOMCSSStyleDeclaration *wrapped_ptr)
{
	g_assert (self && wrapped_ptr);
	self->wrapped_ptr = wrapped_ptr;
	NS_ADDREF (self->wrapped_ptr);
	/* constuction param, init parent */
	gme_supports_private_set_wrapped_ptr (GME_SUPPORTS (self), self->wrapped_ptr);
}

GmeDOMCSSStyleDeclaration* 
gme_dom_css_style_declaration_new (nsIDOMCSSStyleDeclaration *wrapped_ptr)
{
	return GME_DOM_CSS_STYLE_DECLARATION (g_object_new (GME_TYPE_DOM_CSS_STYLE_DECLARATION, "wrapped-ptr", wrapped_ptr, NULL));
}

gchar* 
gme_dom_css_style_declaration_get_css_text (GmeDOMCSSStyleDeclaration *self)
{
	nsAutoString s;
	nsresult rv;
	g_assert (self);

	rv = self->wrapped_ptr->GetCssText (s);
	if (NS_SUCCEEDED (rv)) {
		return g_strdup (NS_ConvertUTF16toUTF8 (s).get ());
	}
	return NULL;
}

gboolean 
gme_dom_css_style_declaration_set_css_text (GmeDOMCSSStyleDeclaration *self, const gchar *param)
{
	nsAutoString s;
	nsresult rv;
	g_assert (self);

	s.AssignWithConversion (param);
	rv = self->wrapped_ptr->SetCssText (s);
	return NS_SUCCEEDED (rv);
}

gchar* 
gme_dom_css_style_declaration_get_property_value (GmeDOMCSSStyleDeclaration *self, const gchar *name)
{
	nsAutoString n;
	nsAutoString s;
	nsresult rv;
	g_assert (self);

	n.AssignWithConversion (name);
	rv = self->wrapped_ptr->GetPropertyValue (n, s);
	if (NS_SUCCEEDED (rv)) {
		return g_strdup (NS_ConvertUTF16toUTF8 (s).get ());
	}
	return NULL;
}

GmeDOMCSSValue* 
gme_dom_css_style_declaration_get_property_css_value (GmeDOMCSSStyleDeclaration *self, const gchar *name)
{
	nsIDOMCSSValue *v = NULL;
	GmeDOMCSSValue *value = NULL;
	nsAutoString n;
	nsresult rv;
	g_assert (self);

	n.AssignWithConversion (name);
	rv = self->wrapped_ptr->GetPropertyCSSValue (n, &v);
	if (NS_SUCCEEDED (rv)) {
		value = gme_dom_css_value_new (v);
	}
	return value;
}

gchar* 
gme_dom_css_style_declaration_remove_property (GmeDOMCSSStyleDeclaration *self, const gchar *name)
{
	nsAutoString n;
	nsAutoString s;
	nsresult rv;
	g_assert (self);

	n.AssignWithConversion (name);
	rv = self->wrapped_ptr->RemoveProperty (n, s);
	if (NS_SUCCEEDED (rv)) {
		return g_strdup (NS_ConvertUTF16toUTF8 (s).get ());
	}
	return NULL;
}

gchar* 
gme_dom_css_style_declaration_get_property_priority (GmeDOMCSSStyleDeclaration *self, const gchar *name)
{
	nsAutoString n;
	nsAutoString s;
	nsresult rv;
	g_assert (self);

	n.AssignWithConversion (name);
	rv = self->wrapped_ptr->GetPropertyPriority (n, s);
	if (NS_SUCCEEDED (rv)) {
		return g_strdup (NS_ConvertUTF16toUTF8 (s).get ());
	}
	return NULL;
}

gboolean 
gme_dom_css_style_declaration_set_property (GmeDOMCSSStyleDeclaration *self, 
					    const gchar *name, 
					    const gchar *value, 
					    const gchar *priority)
{
	nsAutoString n;
	nsAutoString v;
	nsAutoString p;
	nsresult rv;
	g_assert (self);

	n.AssignWithConversion (name);
	v.AssignWithConversion (value);
	p.AssignWithConversion (priority);
	rv = self->wrapped_ptr->SetProperty (n, v, p);
	return NS_SUCCEEDED (rv);
}

gint64 
gme_dom_css_style_declaration_get_length (GmeDOMCSSStyleDeclaration *self)
{
	guint32 length;
	nsresult rv;
	g_assert (self);

	rv = self->wrapped_ptr->GetLength (&length);
	if (NS_SUCCEEDED (rv)) {
		return length;
	}
	return -1;
}

gchar* 
gme_dom_css_style_declaration_item (GmeDOMCSSStyleDeclaration *self, guint32 index)
{
	nsAutoString s;
	nsresult rv;
	g_assert (self);

	rv = self->wrapped_ptr->Item (index, s);
	if (NS_SUCCEEDED (rv)) {
		return g_strdup (NS_ConvertUTF16toUTF8 (s).get ());
	}
	return NULL;
}

GmeDOMCSSRule* 
gme_dom_css_style_declaration_get_parent_rule (GmeDOMCSSStyleDeclaration *self)
{
	nsIDOMCSSRule *r = NULL;
	GmeDOMCSSRule *rule = NULL;
	nsresult rv;
	g_assert (self);

	rv = self->wrapped_ptr->GetParentRule (&r);
	if (NS_SUCCEEDED (rv)) {
		rule = gme_dom_css_rule_new (r);
	}
	return rule;
}
