/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/*
 * Hitori
 * Copyright (C) Philip Withnall 2007 <philip@tecnocode.co.uk>
 * 
 * Hitori is free software.
 * 
 * You may redistribute it and/or modify it under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * Hitori is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Hitori.  If not, write to:
 * 	The Free Software Foundation, Inc.,
 * 	51 Franklin Street, Fifth Floor
 * 	Boston, MA  02110-1301, USA.
 */

#include <config.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include "main.h"
#include "interface.h"
#include "generator.h"

void
hitori_new_game (Hitori *hitori)
{
	hitori_generate_board (hitori);
	hitori_clear_undo_stack (hitori);
	hitori_draw_board_simple (hitori, FALSE);
}

void
hitori_clear_undo_stack (Hitori *hitori)
{	
	/* Clear the undo stack */
	if (hitori->undo_stack != NULL) {
		while (hitori->undo_stack->redo != NULL)
			hitori->undo_stack = hitori->undo_stack->redo;

		while (hitori->undo_stack->undo != NULL) {
			hitori->undo_stack = hitori->undo_stack->undo;
			g_free (hitori->undo_stack->redo);
			if (hitori->undo_stack->type == UNDO_NEW_GAME)
				break;
		}
	}

	/* Reset the "new game" item */
	hitori->undo_stack->undo = NULL;
	hitori->undo_stack->redo = NULL;

	gtk_action_set_sensitive (hitori->undo_action, FALSE);
	gtk_action_set_sensitive (hitori->redo_action, FALSE);
}

void
hitori_print_board (Hitori *hitori)
{
	if (hitori->debug) {
		guint x, y;

		for (y = 0; y < BOARD_SIZE; y++) {
			for (x = 0; x < BOARD_SIZE; x++) {
				if (hitori->board[x][y].painted == FALSE)
					printf ("%u ", hitori->board[x][y].num);
				else
					printf ("X ");
			}
			printf ("\n");
		}
	}
}

void
hitori_enable_events (Hitori *hitori)
{
	hitori->processing_events = TRUE;

	if (hitori->undo_stack->redo != NULL)
		gtk_action_set_sensitive (hitori->redo_action, TRUE);
	if (hitori->undo_stack->undo != NULL)
		gtk_action_set_sensitive (hitori->redo_action, TRUE);
	gtk_action_set_sensitive (hitori->hint_action, TRUE);
}

void
hitori_disable_events (Hitori *hitori)
{
	hitori->processing_events = FALSE;
	gtk_action_set_sensitive (hitori->redo_action, FALSE);
	gtk_action_set_sensitive (hitori->undo_action, FALSE);
	gtk_action_set_sensitive (hitori->hint_action, FALSE);
}

void
hitori_quit (Hitori *hitori)
{
	hitori_clear_undo_stack (hitori);
	g_free (hitori->undo_stack); /* Clear the new game element */
	if (hitori->window != NULL)
		gtk_widget_destroy (hitori->window);

	if (gtk_main_level () > 0)
		gtk_main_quit ();

	exit (0);
}

int
main (int argc, char *argv[])
{
	Hitori *hitori;
	HitoriUndo *undo;
	GOptionContext *context;
	GError *error = NULL;
	gboolean debug = FALSE;

	const GOptionEntry options[] = {
	        { "debug", 0, 0, G_OPTION_ARG_NONE, &debug, N_("Enable debug mode"), NULL },
	        { NULL }
	};

#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	gtk_set_locale ();
	gtk_init (&argc, &argv);
	g_set_application_name (_("Hitori"));
	gtk_window_set_default_icon_name ("hitori");

	/* Options */
	context = g_option_context_new (_("- Play a game of Hitori"));
	g_option_context_set_translation_domain (context, GETTEXT_PACKAGE);
	g_option_context_add_main_entries (context, options, GETTEXT_PACKAGE);

	if (g_option_context_parse (context, &argc, &argv, &error) == FALSE) {
		/* Show an error */
		GtkWidget *dialog = gtk_message_dialog_new (NULL,
				GTK_DIALOG_MODAL,
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_OK,
				_("Command-line options could not be parsed. Error: %s"), error->message);
		g_signal_connect_swapped (dialog, "response", 
				G_CALLBACK (gtk_widget_destroy),
				dialog);
		gtk_widget_show_all (dialog);

		g_error_free (error);
		exit (1);
	}

	/* Setup */
	hitori = g_new (Hitori, 1);
	hitori->debug = debug;
	hitori->hint_status = 0;
	hitori->hint_x = 0;
	hitori->hint_y = 0;

	undo = g_new (HitoriUndo, 1);
	undo->type = UNDO_NEW_GAME;
	undo->x = 0;
	undo->y = 0;
	undo->redo = NULL;
	undo->undo = NULL;
	hitori->undo_stack = undo;

	/* Showtime! */
	srand (time (0));
	hitori_create_interface (hitori);
	hitori_generate_board (hitori);
	gtk_widget_show (hitori->window);

	g_option_context_free (context);

	gtk_main ();
	return 0;
}
