/* -*- tab-width: 4; c-basic-offset: 4 -*- */

using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

public class DiskSim : DiskModel, IDisposable
{
    const string disksimlib = "disksim";

    class Request : IDisposable
    {
        public enum Type : int { Read = 'r', Write = 'w' };
        [DllImport (disksimlib)]
        static extern IntPtr idisksim_request_new (Type type, uint blockNum, int byteCount);
        [DllImport (disksimlib)]
        static extern void idisksim_request_free (IntPtr req);

        public DiskModel.IORequest ioReq;

        internal IntPtr req;
        public Request (DiskModel.IORequest ioReq)
        {
            Type type = Type.Read;
            this.req = idisksim_request_new (type, (uint) ioReq.BlkId, ioReq.BlkCount);
            this.ioReq = ioReq;
        }
        public void Dispose()
        {
            if (this.req != IntPtr.Zero) {
                idisksim_request_free (this.req);
                this.req = IntPtr.Zero;
            }
        }
    }

    delegate void DiskSimCallback (IntPtr sim, IntPtr closure, double time);
    delegate void DiskSimScheduleCallback (IntPtr sim, DiskSimCallback cb,
                                           IntPtr cb_closure, double time,
                                           IntPtr closure);
    delegate void DiskSimDeScheduleCallback (IntPtr sim, DiskSimCallback cb,
                                             IntPtr closure);
    delegate void DiskSimDoneIONotify (IntPtr sim, IntPtr req, double time,
                                       IntPtr closure);

    [DllImport (disksimlib)]
    static extern IntPtr idisksim_init (string profile,
                                        DiskSimScheduleCallback scheduleFn,
                                        DiskSimDeScheduleCallback deScheduleFn,
                                        DiskSimDoneIONotify ioDoneNotify,
                                        IntPtr closure);
    [DllImport (disksimlib)]
    static extern int idisksim_get_blk_size (IntPtr sim);
    [DllImport (disksimlib)]
    static extern int idisksim_get_blk_count (IntPtr sim);
    [DllImport (disksimlib)]
    static extern void idisksim_free (IntPtr sim);
    [DllImport (disksimlib)]
    static extern void idisksim_submit (IntPtr sim, IntPtr req, double time);
    [DllImport (disksimlib)]
    static extern void idisksim_iterate (IntPtr sim, double time);

    GCHandle thisHandle;
    IntPtr   sim;

    static DiskSim Get (IntPtr closure)
    {
        return ((GCHandle)closure).Target as DiskSim;
    }

    public DiskSim (string model)
        : base (0)
    {
        this.thisHandle = GCHandle.Alloc (this);
        this.sim = idisksim_init (model, ScheduleCallback,
                                  DeScheduleCallback, DoneIONotify,
                                  (IntPtr) thisHandle);
        this.sectorSize = idisksim_get_blk_size (this.sim);
        this.pendingRequests = new List<Request>();
        this.completedRequests = new List<Request>();
    }

    static double usToDiskTime (double timeUs)
    {
        return timeUs / 1e6;
    }

    static double diskTimeToUs (double timeDisk)
    {
        return timeDisk * 1e6;
    }

    void SubmitRequest (Request req, double time)
    {
        idisksim_submit (this.sim, req.req, usToDiskTime (time));
    }

    // FIXME: unused ...
    public void Iterate (long timeNow)
    {
        idisksim_iterate (this.sim, usToDiskTime (timeNow));
    }

    public void Dispose()
    {
        if (this.thisHandle.IsAllocated)
            this.thisHandle.Free();
        if (this.sim != IntPtr.Zero) {
            idisksim_free (this.sim);
            this.sim = IntPtr.Zero;
        }
    }

    List <Request> pendingRequests;
    List <Request> completedRequests;
    protected override void SubmitRequest (IORequest req)
    {
//        Console.WriteLine ("submit new request");
        Request simReq = new Request (req);
        this.pendingRequests.Add (simReq);
        SubmitRequest (simReq, req.proc.CurrentTimeAsDouble);
        req.WaitToComplete();
    }

    public void SubmitTestRequest (IORequest req, double time)
    {
        Request simReq = new Request (req);
        this.pendingRequests.Add (simReq);
        SubmitRequest (simReq, time);
    }

    DiskSimCallback callback;
    IntPtr          callbackClosure;
    double          callbackTime;
    static void ScheduleCallback (IntPtr sim, DiskSimCallback cb,
                                  IntPtr cb_closure, double time,
                                  IntPtr closure)
    {
        DiskSim ds = Get (closure);
        ds.callback = cb;
        ds.callbackClosure = cb_closure;
        ds.callbackTime = diskTimeToUs (time);
//        Console.WriteLine ("callback scheduled for " + ds.callbackTime + " t " + time);
    }
    void DeScheduleWithCallback (bool doCall, double callTime)
    {
        DiskSimCallback callback = this.callback;
        IntPtr callbackClosure = this.callbackClosure;

        this.callback = null;
        this.callbackClosure = IntPtr.Zero;
        this.callbackTime = -1;

        if (doCall && callback != null)
            callback (this.sim, callbackClosure,
                      usToDiskTime (callTime));
    }
    static void DeScheduleCallback (IntPtr sim, DiskSimCallback cb,
                                    IntPtr closure)
    {
//        Console.WriteLine ("callback de-scheduled ...");
        DiskSim ds = Get (closure);
        ds.DeScheduleWithCallback (false, 0);
    }
    Request FindRequest (IntPtr cReq)
    {
        foreach (Request req in pendingRequests) {
            if (req.req == cReq)
                return req;
        }
        return null;
    }
    static void DoneIONotify (IntPtr sim, IntPtr cReq, double time,
                              IntPtr closure)
    {
        DiskSim ds = Get (closure);
//        Console.WriteLine ("i/o notify ! " + time);
        Request req = ds.FindRequest (cReq);
        ds.pendingRequests.Remove (req);
        ds.completedRequests.Add (req);
    }

    public override void Wait(ref double currentTime)
    {
//        Console.WriteLine ("Start Wait " + currentTime);
        while (//this.completedRequests.Count == 0 &&
               this.callbackTime >= 0) {
            // advance time
//            Console.WriteLine ("Wait until " + this.callbackTime);
            currentTime = this.callbackTime;
            DeScheduleWithCallback (true, currentTime);
        }

        // just assume all blocks have come back ...
        // (what an odd disk)
        foreach (Request req in completedRequests) {
            req.ioReq.SignalCompleted();
            req.Dispose();
        }
        completedRequests.Clear();

//        Console.WriteLine ("Time on exit " + currentTime);
    }

    public override bool CheckMaxSector (long maxSector)
    {
//        Console.WriteLine ("Blk count " + idisksim_get_blk_count (this.sim) + " maxss " + maxSector);
        return idisksim_get_blk_count (this.sim) >= maxSector;
    }
}
