#include <stdio.h>
#include "disksim_global.h"
#include "disksim_ioface.h"
#include "disksim_iosim.h"
#include "disksim_disk.h"

#include "interface.h"

// DiskSim time is in milliseconds (MS)
#define SYSSIMTIME_TO_MS(syssimtime)    (syssimtime*1e3)
#define MS_TO_SYSSIMTIME(curtime)       (curtime/1e3)

/*
 * FIXME - this interface is an idealised API, unfortunately
 * the disksim implementation is sufficiently lame that it
 * cannot easily be realised fully.
 *
 * FIXME: peel back disksim_interface.c to work out where
 * the evil truly comes from.
 */
static IDiskSim *urgh_global_state;
#define SET_GLOBAL_DISKSIM(s) disksim = &(s)->disksim;

struct _IDiskSim {
    IDiskSimScheduleCallback   schedule_fn;
    IDiskSimDeScheduleCallback de_schedule_fn;
    IDiskSimDoneIONotify       notify_fn;
    void                      *closure;
    int                        debug;
        
    disksim_t disksim;
};

#define IDISKSIM_OFFSET (((unsigned char *)&((sim_impl_t *) 0)->disksim))
#define IDISKSIM_FROM_DISKSIM(s) ((IDiskSim *)((unsigned char *)(s) - SIM_IMPL_DISKSIM_OFFSET))

struct _IDiskSimRequest {
    IDiskSim *ids;
    IDiskSimRequestType type;
    unsigned long blkno;
    int blkcount;
    int devno;
};

/* DiskSim is just unbelievable ! */
static double
idisksim_get_simtime (IDiskSim *sim)
{
    disksim_t *disksim = &sim->disksim;
    return simtime; /* macro, and also element name */
}

static void idisksim_io_done_notify (ioreq_event *curr)
{
    IDiskSimRequest *req = curr->buf;
    IDiskSim *ids = req->ids;

    ids->notify_fn (ids, req,
                    MS_TO_SYSSIMTIME (idisksim_get_simtime (ids)),
                    ids->closure);
}

IDiskSim *
idisksim_init (const char                *profile_name,
               IDiskSimScheduleCallback   schedule_fn,
               IDiskSimDeScheduleCallback de_schedule_fn,
               IDiskSimDoneIONotify       notify_fn,
               void                      *closure)
{
    IDiskSim *ids;
    char *argv[6];

    if (!schedule_fn || !de_schedule_fn || !notify_fn) {
        fprintf (stderr, "Error - missing callback\n");
        return NULL;
    }

    if (!(ids = calloc (1, sizeof (IDiskSim))))
        return NULL;

    ids->schedule_fn = schedule_fn;
    ids->de_schedule_fn = de_schedule_fn;
    ids->notify_fn = notify_fn;
    ids->closure = closure;
    ids->debug = getenv ("DISKSIM_DEBUG") != NULL;

//    if (ids->debug)
    fprintf (stderr, "idisksim_init '%s'\n", profile_name);
    disksim_initialize_disksim_structure (&ids->disksim, sizeof (disksim_t));

    argv[0] = "disksim";
    argv[1] = strdup (profile_name);
    argv[2] = "/dev/null";
    argv[3] = "external";
    argv[4] = "0"; // no io trace file
    argv[5] = "0"; // synthio

    SET_GLOBAL_DISKSIM (ids);
    disksim_setup_disksim (6, (char **)argv);
    disksim_set_external_io_done_notify (idisksim_io_done_notify);

    urgh_global_state = ids;

    if (ids->debug)
        fprintf (stderr, "idisksim_init done\n");

    return ids;
}

int
idisksim_get_blk_size (IDiskSim *ids)
{
    return 512;
}

int
idisksim_get_blk_count (IDiskSim *ids)
{
    int blks = 0, i, disks;
    SET_GLOBAL_DISKSIM (ids);
    /* normal disks */
    disks = disk_get_numdisks();
    for (i = 0; i < disks; i++) 
        blks += disk_get_number_of_blocks (i);
    /* simple disks */
    disks = simpledisk_get_numdisks();
    for (i = 0; i < disks; i++) 
        blks += simpledisk_get_number_of_blocks (i);
    return blks;
}

void
idisksim_free (IDiskSim *ids)
{
    if (ids)
        free (ids);
}

IDiskSimRequest *
idisksim_request_new (IDiskSimRequestType type,
                      unsigned int blkno,
                      int blkcount)
{
    IDiskSimRequest *req;

    if (!(req = calloc (1, sizeof (IDiskSimRequest))))
        return NULL;

    req->type = type;
    req->devno = 0;
    req->blkno = blkno;
    req->blkcount = blkcount;

    return req;
}

void
idisksim_request_free (IDiskSimRequest *req)
{
    if (req)
        free (req);
}

static void
idisksim_callback (IDiskSim        *ids,
                   void            *closure,
                   IDiskSimTime     time)
{
//    fprintf (stderr, "idisksim_callback %p %p %g\n", ids, closure, time);
    idisksim_iterate (ids, time);
}

void
idisksim_submit (IDiskSim        *ids,
                 IDiskSimRequest *req,
                 IDiskSimTime     time)
{
    double curtime = SYSSIMTIME_TO_MS (time);
    ioreq_event *new;

    SET_GLOBAL_DISKSIM (ids);

    new = (ioreq_event *) getfromextraq();
    
    assert (new != NULL);
    new->type = IO_REQUEST_ARRIVE;
    new->time = curtime;
    new->busno = 0;
    new->devno = req->devno;
    new->blkno = req->blkno;
    new->flags = req->type == IDISKSIM_READ ? READ : WRITE;
    new->bcount = req->blkcount;
    new->flags |= TIME_CRITICAL; // non-background ...
    new->cause = 0;
    new->opid = 0;

    if (ids->debug)
        fprintf (stderr, "submit request 0x%8.x (%d) at time %.6g(ms)\n",
                 req->blkno, req->blkcount, curtime);

    // closure
    req->ids = ids;
    new->buf = req;
    
    io_map_trace_request (new);

    /* issue it into simulator */
    if (ids->disksim.intq)
        ids->de_schedule_fn (ids, idisksim_callback, ids->closure);
    addtointq ((event *)new);
    
    idisksim_iterate (ids, time);
}

void
idisksim_iterate (IDiskSim    *ids,
                  IDiskSimTime time)
{
    double curtime = SYSSIMTIME_TO_MS (time);

    /* if we missed an event - error out ... */
    if (ids->disksim.intq != NULL && (ids->disksim.intq->time + 0.0001) < curtime) {
        fprintf (stderr, "external time is ahead of disksim time: %f > %f\n",
                 curtime, ids->disksim.intq->time);
        return;
    }

    if (ids->disksim.intq != NULL) {
        if (ids->disksim.intq->time + 0.0001 < idisksim_get_simtime (ids)) {
            fprintf (stderr, "Error: time out of sync %g vs %g\n",
                     ids->disksim.intq->time + 0.0001,
                     idisksim_get_simtime (ids));
            ASSERT (ids->disksim.intq->time + 0.0001 >= idisksim_get_simtime (ids));
        }
    }
    
    /* process events until the next event is either in the future, or nonexistent */
    while ((ids->disksim.intq != NULL) 
           && (ids->disksim.intq->time <= (curtime + 0.0001))) 
    {
        SET_GLOBAL_DISKSIM (ids);
        disksim_simulate_event (0 /* count for debugging */);
    }
    
   if (disksim->intq)
       ids->schedule_fn (ids, idisksim_callback, NULL,
                         MS_TO_SYSSIMTIME(disksim->intq->time),
                         ids->closure);
}
