/* -*- tab-width: 4; c-basic-offset: 4 -*- */

using System;
using System.Collections.Generic;

public abstract class DiskModel {
    public class IORequest {
        public SimEvent   ev;
        public SimProcess proc;
        public DiskModel  model;
        public long       BlkId;
        public int        BlkCount;
        public IORequest (DiskModel model, SimProcess proc,
                          SimEvent sime)
        {
            this.ev = sime;
            this.proc = proc;
            this.model = model;
            this.BlkId = 0;
            this.BlkCount = -1;
        }
        public void WaitToComplete()
        {
            this.ev.StartTime = proc.CurrentTime;
            this.ev.EndTime = -1;
            this.proc.IOSleep();
        }
        public virtual void SignalCompleted()
        {
            this.proc.IOWakeup (this.ev, this.BlkCount * this.model.sectorSize);
        }
    }
    public class TestRequest : IORequest {
        public TestRequest (DiskModel model, SimEvent sime, long blkId, int count)
            : base (model, null, sime)
        {
            this.BlkId = blkId;
            this.BlkCount = count;
        }
        public override void SignalCompleted() { }
    }
    protected internal int sectorSize;

	public DiskModel(int sectorSize)
	{
        this.sectorSize = sectorSize;
	}

    public int SectorSize { get { return this.sectorSize; } }

    public void ReadBlocks (SimProcess proc, SimEvent ev, long block, int count)
    {
        IORequest req = new IORequest (this, proc, ev);
        req.BlkId = block;
        req.BlkCount = count;
        SubmitRequest (req);
    }

    protected abstract void SubmitRequest (IORequest req);
    public abstract void Wait(ref double currentTime);    
    public abstract bool CheckMaxSector (long maxSector);
}

/*
 *    An almost totally moronic disk, for a quick and
 * dirty answer without requiring DiskSim.
 */
public class DiskModelSimple : DiskModel
{
	double seekLatency; // fixed time to seek
	int    trackSize;   // accesses in the same track are fast
    List<IORequest> requests;
	
	// noddy cache
	long     lastTrack;

    public DiskModelSimple ()
        : base (512)
	{
		this.seekLatency = 10e-3; // 10ms
		this.trackSize = 32;
        this.requests = new List<IORequest>();
	}

    public override bool CheckMaxSector (long maxSector)
    {
        return true;
    }

    protected override void SubmitRequest (IORequest req)
    {
//        Console.WriteLine ("IOReq 0x{0:x} cnt 0x{1:x}",
//                           req.BlkId, req.BlkCount);
        if (req.BlkId >= this.lastTrack &&
            req.BlkId  <= this.lastTrack + this.trackSize) {
            int count = req.BlkCount;
            count -= (int) (req.BlkId - this.lastTrack);
            count -= this.trackSize;
            // satisfied by the simple-minded cache
            if (count < 0)
                return;
            req.BlkCount -= count;
        }
        requests.Add (req);
        req.WaitToComplete();
    }

    public override void Wait(ref double currentTime)
    {
        currentTime += seekLatency;

        // just assume all blocks have come back ...
        // (what an odd disk)
        foreach (IORequest req in requests)
            req.SignalCompleted();
        requests.Clear();
    }
}
