/* -*- tab-width: 4; c-basic-offset: 4 -*- */

using System;
using System.Collections.Generic;
using FS;

public class ScribbleView : FSBase {
    enum StoreFields : int { SimBlk, Name, SectId, SectLen, Elapsed };
    Gtk.TreeView  treeView;
	Gtk.ListStore listStore;
    SimulationData sd;
    List<SimBlk> blkSelection;

    public ScribbleView (SimulationData sd)
        : base (sd.FileSystem)
    {
        this.sd = sd;
        this.blkSelection = new List<SimBlk>();
        FillTree();
        Zoom (0.25);
        UpdateTimeDisplay();
    }
  
    void FillTree()
    {
        foreach (SimBlk blk in this.sd.Blocks)
            this.listStore.AppendValues (blk, blk.BlockName, blk.LocationStr,
                                         blk.FsBlkLen, blk.TimeStr);
        this.overview.UpdateBlkSelection (this.blkSelection);
    }

    void UpdateTimeDisplay ()
    {
        if (this.blkSelection.Count == 0)
            this.selectionDisplay.LabelProp = 
                "Total time - " + Utils.PrettyTime (this.sd.TotalTime);
        else {
            long time = 0;
            foreach (SimBlk blk in this.blkSelection) {
                if (blk.IsSequence)
                    continue;
                time += blk.Ev.Elapsed;
            }
            this.selectionDisplay.LabelProp = 
                "Selected time - " + Utils.PrettyTime (time);
        }
    }

    void TreeSelectionChanged (object obj, System.EventArgs args)
    {
        this.blkSelection.Clear();
        this.treeView.Selection.SelectedForeach (
            delegate (Gtk.TreeModel model, Gtk.TreePath path, Gtk.TreeIter iter)
            {
                this.blkSelection.Add (this.listStore.GetValue (iter, (int) StoreFields.SimBlk) as SimBlk);
            }
            );
        this.overview.UpdateBlkSelection (this.blkSelection);

        if (this.blkSelection.Count > 0) {
            long minBlk = Int64.MaxValue;
            long maxBlk = 0;
            foreach (SimBlk blk in this.blkSelection)
            {
                minBlk = Math.Min (minBlk, blk.FsBlkStart);
                maxBlk = Math.Max (maxBlk, blk.FsBlkStart + blk.FsBlkLen);
            }
            double min = this.renderer.BlkToPixY (minBlk);
            double max = this.renderer.BlkToPixY (maxBlk);
            double val = this.vAdjustment.Value;
            if (val < min || val > max)
                MoveToBlock (minBlk);
        }
        UpdateTimeDisplay();
        QueueDraw();
    }

	protected internal override void RenderContent ()
    {
        base.RenderContent ();
        // FIXME: we want to ensure the selection is contiguous [!]
        Cairo.Color red = new Cairo.Color (1, 0, 0, 0.5);
        base.renderer.RenderScribble (this.blkSelection, red, 2.0);
    }

    public override Gtk.Widget CreateTree ()
    {
		this.listStore = new Gtk.ListStore (typeof (SimBlk), typeof (string),
                                            typeof (string), typeof (int),
                                            typeof (string));
        this.treeView = new Gtk.TreeView (this.listStore);
		this.treeView.Selection.Mode = Gtk.SelectionMode.Multiple;
		this.treeView.RulesHint = true;
        this.treeView.EnableSearch = true;
        this.treeView.FixedHeightMode = false;
        this.treeView.Selection.Changed += TreeSelectionChanged;

		Gtk.CellRendererText rendererText = new Gtk.CellRendererText ();
		rendererText.Xalign = 1.0f;
        Gtk.TreeViewColumn nameColumn;
		nameColumn = new Gtk.TreeViewColumn ("file", rendererText, 
											 "text", (int) StoreFields.Name);
        nameColumn.Resizable = true;
        this.treeView.InsertColumn (nameColumn, 0);

        rendererText = new Gtk.CellRendererText ();
        this.treeView.InsertColumn (1, "sect", rendererText, "text", (int) StoreFields.SectId);
        this.treeView.InsertColumn (2, "slen", rendererText, "text", (int) StoreFields.SectLen);
        this.treeView.InsertColumn (3, "dlta", rendererText, "text", (int) StoreFields.Elapsed);

		Gtk.ScrolledWindow scrolledWindow = new Gtk.ScrolledWindow ();

		scrolledWindow.SetPolicy (Gtk.PolicyType.Automatic,
								  Gtk.PolicyType.Automatic);

		scrolledWindow.Add (this.treeView);
        scrolledWindow.ShowAll();

        scrolledWindow.SizeAllocated += delegate {
            Gtk.Adjustment adj = scrolledWindow.Hadjustment;
            adj.Value = adj.Upper - adj.PageSize;
        };

        return scrolledWindow;
    }
}
