#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <ext2fs/ext2fs.h>

static int pretty_depth = 0;

static void
print_append (const char *format, ...)
{
    va_list args;
    va_start (args, format);
    vfprintf (stdout, format, args);
    va_end (args);
}

static void
print_tag (int depthinc, const char *format, ...)
{
    va_list args;
    int i;
    
    va_start (args, format);
    if (depthinc < 0)
        pretty_depth += depthinc;
    for (i = 0; i < pretty_depth; i++)
        fprintf (stdout, " ");
    vfprintf (stdout, format, args);
    if (depthinc > 0)
        pretty_depth += depthinc;
    va_end (args);
}

static struct struct_io_manager extfs_handle;
typedef struct {
    long  *blocks;
    long   block_len;
    long   block_alloc;
} BlockList;
static BlockList context;


static void
blocks_init (BlockList *blist)
{
    // No closure for various io manager methods (sadly)
    if (!blist)
        blist = &context;
    blist->blocks = NULL;
    blist->block_len = 0;
    blist->block_alloc = 0;
}

static void
blocks_free (BlockList *blist)
{
    if (!blist)
        blist = &context;
    if (blist->blocks)
        free (blist->blocks);
}

static void
blocks_push (BlockList *blist, long block)
{
    if (!blist)
        blist = &context;
    if (blist->block_len >= blist->block_alloc) {
        if (!blist->block_alloc)
            blist->block_alloc = 16;
        blist->block_alloc *= 2;
        blist->blocks = realloc (blist->blocks,
                                  sizeof (long) * blist->block_alloc);
    }
    blist->blocks[blist->block_len++] = block;
}

static void
blocks_reset (BlockList *blist)
{
    if (!blist)
        blist = &context;
    blist->block_len = 0;
}

// resets queue ...
static char *
blocks_as_string (BlockList *blist)
{
#define ELEM_SIZE (sizeof ("i0x, ") + sizeof (long) * 2)
    char *buf;
    char *p;
    int i;

    if (!blist)
        blist = &context;

    buf = malloc (blist->block_len * ELEM_SIZE + 1);
    buf[0] = '\0';

    // FIXME: we need to check if some blocks are contiguous
    // and crunch them into a span ... (+5) ...
    for (i = 0, p = buf; i < blist->block_len; i++)
    {
        int indirect = 0;
        long j, blk = blist->blocks[i];

        // look for spans
        if (blk >= 0)
        {
            for (j = 0; i + j < blist->block_len; j++) {
                if (blist->blocks[i+j] != blk + j)
                    break;
            }
        } else {
            blk = -blk;
            indirect = 1;
            j = 1;
        }
        if (j <= 1)
            p += sprintf (p, "%s0x%lx", indirect ? "i" : "", blk);
        else
            p += sprintf (p, "0x%lx-0x%lx", blk, blk + j - 1);
        i += j - 1;
        if (i < blist->block_len - 1)
            p = strcat (p, ", ") + 2;
    }
    blocks_reset (blist);
    return buf;
#undef ELEM_SIZE
}


static errcode_t
io_manager_open (const char *name, int flags, io_channel *channel)
{
    errcode_t ret;
    ret = unix_io_manager->open (name, flags, channel);
    if (ret)
        return ret;
    (*channel)->manager = &extfs_handle;
    return ret;
}

static errcode_t
io_manager_close (io_channel channel)
{
    return unix_io_manager->close (channel);
}

static errcode_t
io_manager_set_blksize (io_channel channel, int blksize)
{
//    fprintf (stderr, "SetBlckSize %d\n", blksize);
    return unix_io_manager->set_blksize (channel, blksize);
}

static errcode_t
io_manager_read_blk (io_channel channel, unsigned long block,
                     int count, void *data)
{
    int i;
    for (i = 0; i < count; i++)
        blocks_push (NULL, block + i);
    return unix_io_manager->read_blk (channel, block, count, data);
}

static errcode_t
io_manager_write_blk (io_channel channel, unsigned long block,
                      int count, const void *data)
{
    return unix_io_manager->write_blk (channel, block, count, data);
}

static errcode_t
io_manager_flush (io_channel channel)
{
    return unix_io_manager->flush (channel);
}

static errcode_t
io_manager_write_byte (io_channel channel, unsigned long offset,
                       int count, const void *data)
{
    return unix_io_manager->write_byte (channel, offset, count, data);
}

static errcode_t
io_manager_set_option (io_channel channel, const char *option, 
                       const char *arg)
{
    return unix_io_manager->set_option (channel, option, arg);
}

static struct struct_io_manager extfs_handle = {
    EXT2_ET_MAGIC_IO_MANAGER, "gzip",
    io_manager_open,
    io_manager_close,
    io_manager_set_blksize,
    io_manager_read_blk,
    io_manager_write_blk,
    io_manager_flush,
    io_manager_write_byte,
    io_manager_set_option,
};

static char *
read_inode_attrs (ext2_filsys fs, ext2_ino_t inode)
{
    char *ret;
    char *blocks;
    struct ext2_inode in_data;

    blocks_reset (NULL);
    ext2fs_flush_icache (fs);
	ext2fs_read_inode (fs, inode, &in_data);

    blocks = blocks_as_string (NULL);
    ret = malloc (strlen (blocks) + sizeof ("inode=\"0x00000000\" "
                                            "iblk=\"\"") + 1);
    sprintf (ret, " inode=\"0x%x\" iblk=\"%s\"", inode, blocks);
    free (blocks);

    return ret;
}

static int
block_account_func (ext2_filsys fs, blk_t *blocknr,
                    int	blockcnt, void *priv_data)
{
    BlockList *blist = priv_data;
    long blk = *blocknr;

//    fprintf (stdout, "act: 0x%lx (%d)\n", (long)*blocknr, blockcnt);
    // -N is N'th indirection & the block is an indirect block
    if (blockcnt < 0)
        blk = -blk;
    blocks_push (blist, blk);

    return 0;
}

struct dump_context_t {
    ext2_filsys fs;
    int (*filter_fn) (struct dump_context_t *dc, const char *name);
    const char *subtree;
    int depth;
};

static void dump_item (struct dump_context_t *dc, const char *name,
                       ext2_ino_t inode, char *inode_str);

static int dump_dir_ent (struct ext2_dir_entry *dirent,
                         int offset, int blocksize,
                         char *buf, void *priv_data)
{
    struct dump_context_t *dc = priv_data;
    char name[EXT2_NAME_LEN+1];
    int real_len = dirent->name_len & 0xff;
    int type = dirent->name_len >> 8;
    strncpy (name, dirent->name, real_len);
    name[real_len] = '\0';

//    fprintf (stderr, "Name '%s' inode 0x%x type %d\n",
//             name, dirent->inode, type);

    char *inode = read_inode_attrs (dc->fs, dirent->inode);
    if (type == EXT2_FT_DIR ||
        type == EXT2_FT_REG_FILE ||
        type == EXT2_FT_SYMLINK) {
        if (strcmp (name, ".") && strcmp (name, "..")
            && (!dc->filter_fn || !dc->filter_fn (dc, name)))
            dump_item (dc, name, dirent->inode, inode);
        // else: to first order '.' and '..' are not useful

    } else // These guys take no space - only one inode ...
        print_tag (0, "<special name=\"%s\" type=\"0x%x\" %s/>\n",
                   name, type, inode);

    free (inode);
    return 0;
}

static char *
read_link (ext2_filsys fs, 	struct ext2_inode *ino, char **blocks)
{
    char *str;
	char *pathname;
	char *block = NULL;

    blocks_reset (NULL);
	if (ext2fs_inode_data_blocks (fs, ino)) {
		if (ext2fs_get_mem (fs->blocksize, &block))
            return NULL;

		if (io_channel_read_blk (fs->io, ino->i_block[0], 1, block))
            pathname = NULL;
        else
            pathname = strdup (block);
        ext2fs_free_mem (&block);
	} else
		pathname = strdup ((char *)ino->i_block);

    str = malloc (strlen (pathname) + sizeof (" link=\"\"") + 1);
    strcpy (str, " link=\"");
    strcat (str, pathname);
    strcat (str, "\"");
    free (pathname);
    *blocks = blocks_as_string (NULL);
    blocks_reset (NULL);
    
	return str;
}

static void
dump_item (struct dump_context_t *dc, const char *name,
           ext2_ino_t inode, char *inode_str)
{
	struct ext2_inode ino;
    char *blocks = NULL;
    const char *node_type;
    char *link = NULL;
    BlockList item_list;

	ext2fs_read_inode (dc->fs, inode, &ino);

    if (LINUX_S_ISDIR (ino.i_mode))
        node_type = "dir";
    else if (LINUX_S_ISREG (ino.i_mode))
        node_type = "file";
    else if (LINUX_S_ISLNK (ino.i_mode)) {
        node_type = "symlink";
        link = read_link (dc->fs, &ino, &blocks);
    } else {
//        fprintf (stderr, "Abormal - directory lied wrt. "
//                 "type of '%s'\n", name);
        node_type = "odd";
        return;
    }

    if (!link) {
        blocks_init (&item_list);
        ext2fs_block_iterate (dc->fs, inode, 0, NULL, block_account_func, &item_list);
        blocks = blocks_as_string (&item_list);
        blocks_free (&item_list);
    }
    print_tag (0, "<%s name=\"%s\" size=\"0x%lx\"%s%s",
               node_type, name, (long)ino.i_size,
               link ? link : "", inode_str);
    if (!LINUX_S_ISDIR (ino.i_mode)) {
        if (blocks[0] == '\0')
            print_append ("/>\n");
        else
            print_append (">%s</%s>\n", blocks, node_type);
    } else {
        print_append ("><blks>%s</blks>\n", blocks);
        pretty_depth++;
        dc->depth++;
        ext2fs_dir_iterate (dc->fs, inode, 0, NULL, dump_dir_ent, dc);
        dc->depth--;
        print_tag (-1, "</%s>\n", node_type);
    }
    if (link)
        free (link);
    free (blocks);
}

static const char *my_strnchr (const char *str, char c, int num)
{
    int i;
    const char *sube = str;
    for (i = 0, sube = str; i < num && sube != NULL; i++) {
        const char *nxt = strchr (sube, '/');
        if (!nxt)
            return NULL;
        sube = nxt + 1;
    }
    return sube;
}

static int
subtree_filter_fn (struct dump_context_t *dc, const char *name)
{
    int len;
    const char *subp, *end;
    subp = my_strnchr (dc->subtree, '/', dc->depth);
    if (!subp) // inside the dir
        return 0;
    
    if ((end = strchr (subp, '/')))
        len = end - subp;
    else
        len = strlen (subp);

    return strncmp (subp, name, len);
}


static void usage_exit (const char *warning)
{
    if (warning)
        fprintf (stderr, "Warning: %s\n\n", warning);
    fprintf (stderr, "Usage: ext2dump </dev/path> [/sub/directory]\n");
    fprintf (stderr, "dump the contents of an ext2/3 filesystem as an XML block description.\n");
    fprintf (stderr, "optionally just a sub-tree of that file-system\n");
    exit (warning != NULL);
}

int main (int argc, char **argv)
{
    ext2_filsys fs;
	int flags;
    errcode_t ret;
    char *fname;
    struct dump_context_t dc = { 0, };
    io_manager io_mgr = &extfs_handle;
    int i;

    fname = NULL;
    dc.subtree = NULL;
    for (i = 1; i < argc; i++) {
        if (!strcmp (argv[i], "-h") ||
            !strcmp (argv[i], "--help"))
            usage_exit (NULL);
        else if (fname == NULL)
            fname = argv[i];
        else
            dc.subtree = argv[i];
    }

    if (!fname)
        usage_exit ("requires a file name");

    flags = EXT2_FLAG_JOURNAL_DEV_OK;

	ret = ext2fs_open (fname, flags, 0, 0, io_mgr, &fs);
    if (ret) {
        fprintf (stderr, "Failed to open '%s'\n", fname);
        return 1;
    }

    blocks_init (NULL);

    print_tag (1, "<root name=\"%s\" blksize=\"0x%x\">\n",
               fname, fs->blocksize);
    dc.fs = fs;
    dc.depth = 0;
    if (dc.subtree)
        dc.filter_fn = subtree_filter_fn;
    dump_item (&dc, "", EXT2_ROOT_INO, "");
    print_tag (-1, "</root>\n");

    blocks_free (NULL);

    ext2fs_close (fs);
    return 0;
}
