/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003 Richard Hult <richard@imendio.com>
 * Copyright (C) 2003 Anders Carlsson <andersca@gnome.org>
 * Copyright (C) 2003 Johan Dahlin <jdahlin@gnome.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <vorbis/vorbisfile.h>
#include "song-db.h"
#include "song.h"
#include "id3-tag.h"

#define VERSION_KEY "jamboree-version"
#define VERSION_KEY_LENGTH (16)

#define _ALIGN_VALUE(this, boundary) \
  (( ((unsigned long)(this)) + (((unsigned long)(boundary)) -1)) & (~(((unsigned long)(boundary))-1)))
	
#define _ALIGN_ADDRESS(this, boundary) \
  ((void*)_ALIGN_VALUE(this, boundary))


static void song_db_class_init (SongDBClass *klass);
static void song_db_init       (SongDB      *db);
static void song_db_finalize   (GObject     *object);


static GObjectClass *parent_class;


GType
song_db_get_type (void)
{
  static GType type = 0;
	
  if (!type)
    {
      static const GTypeInfo info =
	{
	  sizeof (SongDBClass),
	  NULL,           /* base_init */
	  NULL,           /* base_finalize */
	  (GClassInitFunc) song_db_class_init,
	  NULL,           /* class_finalize */
	  NULL,           /* class_data */
	  sizeof (SongDB),
	  0,
	  (GInstanceInitFunc) song_db_init,
	};

      type = g_type_register_static (G_TYPE_OBJECT,
				     "SongDB",
				     &info, 0);
    }

  return type;
}

static void
song_db_class_init (SongDBClass *klass)
{
  GObjectClass *object_class;

  parent_class = g_type_class_peek_parent (klass);
  object_class = (GObjectClass*) klass;

  object_class->finalize = song_db_finalize;
}

static void
song_db_init (SongDB *db)
{
}

static void
song_db_finalize (GObject *object)
{
  SongDB *db = SONG_DB (object);

#ifdef BDB  
  if (db->handle)
    db->handle->close (db->handle, 0);
#endif
  
  g_list_foreach (db->songs, (GFunc) song_free, NULL);
  g_list_free (db->songs);
  
  if (G_OBJECT_CLASS (parent_class)->finalize)
    (* G_OBJECT_CLASS (parent_class)->finalize) (object);
}

static gpointer
unpack_string (gpointer p, char **str)
{
  int len;
  
  p = _ALIGN_ADDRESS (p, 4);
  
  len = *(int *)p;

  if (str)
    *str = g_malloc (len + 1);
  
  p += 4;

  if (str)
    {
      memcpy (*str, p, len);
      (*str)[len] = 0;
    }
  
  return p + len + 1;
}

static gpointer
unpack_int (gpointer p, int *val)
{
  p = _ALIGN_ADDRESS (p, 4);

  if (val)
    *val = *(int *)p;
   
  p += 4;
  
  return p;
}

static gpointer
unpack_uint64 (gpointer p, guint64 *val)
{
  p = _ALIGN_ADDRESS (p, 8);

  if (val)
    *val = *(guint64 *)p;

  p += 8;

  return p;
}

static gpointer
unpack_playlists (gpointer p, GList **playlists)
{
  char *str;
  char **strv;
  int i = 0;
  int id;

  p = unpack_string (p, &str);

  strv = g_strsplit (str, ",", -1);

  while (strv[i])
    {
      id = atoi (strv[i]);
      *playlists = g_list_append (*playlists, GINT_TO_POINTER (id));

      i++;
    }

  g_strfreev (strv);

  return p;
}

static Song *
unpack_song (gpointer p)
{
  Song *song;
  
  song = g_new0 (Song, 1);

  p = unpack_string (p, &song->title);
  p = unpack_string (p, &song->artist);
  p = unpack_string (p, &song->album);
  p = unpack_int (p, &song->genre);
  p = unpack_int (p, &song->year);
  p = unpack_int (p, &song->length); 
  p = unpack_int (p, &song->bitrate);
  p = unpack_int (p, &song->samplerate);
  p = unpack_int (p, &song->track_number);
  p = unpack_int (p, (int*)&song->date_added);
  p = unpack_int (p, (int*)&song->date_modified);
  p = unpack_int (p, &song->rating);
  p = unpack_int (p, &song->play_count);
  p = unpack_int (p, (int*)&song->last_played);
  p = unpack_uint64 (p, &song->filesize);
  p = unpack_playlists (p, &song->playlists);

  return song;
}

static void
string_align (GString *string, int boundary)
{
  gpointer p;
  int padding;
  int i;

  p = string->str + string->len;

  padding = _ALIGN_ADDRESS (p, boundary) - p;

  for (i = 0; i < padding; i++)
    g_string_append_c (string, 0);
}

static void
pack_int (GString *string, int val)
{
  string_align (string, 4);

  g_string_append_len (string, (char *)&val, 4);
}

static void
pack_uint64 (GString *string, guint64 val)
{
  string_align (string, 8);

  g_string_append_len (string, (char *)&val, 8);
}

static void
pack_string (GString *string, const char *str)
{
  int len;
  
  if (str)
    len = strlen (str);
  else
    len = 0;
  
  pack_int (string, len);

  if (str)
    g_string_append (string, str);

  g_string_append_c (string, 0);
}

static void
pack_playlists (GString *string, GList *playlists)
{
  GString *str;
  GList *l;

  str = g_string_new (NULL);

  for (l = playlists; l; l = l->next)
    {
      g_string_append_printf (str, "%d", GPOINTER_TO_INT (l->data));
      g_string_append_c (str, ',');
    }

  pack_string (string, str->str);

  g_string_free (str, TRUE);
}

static gpointer
pack_song (Song *song, int *len)
{
  GString *string;

  string = g_string_new ("");

  pack_string (string, song->title);
  pack_string (string, song->artist);
  pack_string (string, song->album);
  pack_int (string, song->genre);
  pack_int (string, song->year);
  pack_int (string, song->length);
  pack_int (string, song->bitrate);
  pack_int (string, song->samplerate);
  pack_int (string, song->track_number);
  pack_int (string, song->date_added);
  pack_int (string, song->date_modified);
  pack_int (string, song->rating);
  pack_int (string, song->play_count);
  pack_int (string, song->last_played); 
  pack_uint64 (string, song->filesize);
  pack_playlists (string, song->playlists);

  if (len)
    *len = string->len;
  
  return g_string_free (string, FALSE);
}

static void
read_songs (SongDB *db)
{
  datum key, data;
  Song *song;
  
  memset (&key, 0, sizeof (key));
  memset (&data, 0, sizeof (data));

  key = gdbm_firstkey (db->dbf);

  while (1)
    {
      key = gdbm_nextkey (db->dbf, key);
      if (key.dptr == NULL)
	break;

      if (((char*)key.dptr)[0] == VERSION_KEY[0] &&
	  strncmp (key.dptr, VERSION_KEY, VERSION_KEY_LENGTH) == 0)
	continue;
      
      data = gdbm_fetch (db->dbf, key);
      
      song = unpack_song (data.dptr);
      song->filename = g_strndup (key.dptr, key.dsize);
      _song_calculate_collate_keys (song);

      db->songs = g_list_prepend (db->songs, song);
    }

#if 0  
  if (ret != DB_NOTFOUND)
    g_warning ("Error while getting songs: %s", db_strerror (ret));
#endif
  
}

SongDB *
song_db_new (const char *filename)
{
  SongDB *db;

  db = g_object_new (TYPE_SONG_DB, NULL);

  /* FIXME: Set error output and error prefix */
  
  db->dbf = gdbm_open ((char*)filename, 4096, GDBM_NOLOCK | GDBM_WRCREAT | GDBM_SYNC, 04644, NULL);
  
  if (!db->dbf)
    {
      g_print ("Could not open database: %s\n", gdbm_strerror (gdbm_errno));
      return NULL;
    }
  
  return db;
}

void
song_db_read_songs (SongDB *db)
{
  g_return_if_fail (IS_SONG_DB (db));

  g_list_foreach (db->songs, (GFunc) song_free, NULL);
  g_list_free (db->songs);
  db->songs = NULL;
  
  read_songs (db);
}

static int
unpack_int32 (const unsigned char *data)
{
  return (data[0] << 24) +
    (data[1] << 16) +
    (data[2] << 8) +
    data[3];
}

static void
assign_song_info_mp3 (Song *song)
{
  FILE *file;
  struct id3_file *id3_file;
  struct id3_tag *tag;
  unsigned char buf[8192];
  int i, bytes_read;
  gboolean found = FALSE;
  int bitrates[2][16] = {
    {0,8,16,24,32,40,48,56,64,80,96,112,128,144,160,},          /* MPEG2 */
    {0,32,40,48, 56, 64, 80, 96,112,128,160,192,224,256,320,}}; /* MPEG1 */
  int samprates[2][3] = {
    { 22050, 24000, 16000 },  /* MPEG2 */
    { 44100, 48000, 32000 }}; /* MPEG1 */
  int ver = 0, bitrate = 0, samprate = 0, samprate_index = 0;
      
  file = fopen (song->filename, "r");
  if (!file)
    return;
  
  id3_file = id3_file_fdopen (fileno (file), ID3_FILE_MODE_READONLY);
  if (id3_file)
    {
      tag = id3_file_tag (id3_file);
      if (tag)
	{  
	  song->title = id3_tag_get_title (tag);
	  song->artist = id3_tag_get_artist (tag);
	  song->album = id3_tag_get_album (tag);
	  song->genre = id3_tag_get_genre (tag);
	  song->year = id3_tag_get_year (tag);
	  song->track_number = id3_tag_get_number (tag);
	}
    }
  
  /* Now look for a header */
  bytes_read = fread (buf, 1, sizeof (buf), file);

  for (i = 0; i < bytes_read - 4; i++)
    { 
      if (buf[i] == 0xff &&
	  (buf[i + 1] & 0xf0) == 0xf0)
	{
	  ver = buf[i + 1] >> 3 & 0x1;
	  bitrate = bitrates[ver][buf[i + 2] >> 4 & 0xf];
	  samprate_index = buf[i + 2] >> 2 & 0x3;
	  samprate = samprates[ver][samprate_index];

	  found = TRUE;
	  break;
	}
    }

  if (!found)
    goto done;
  
  /* Now look for a xing header */
  for (i = 0; i < bytes_read - 16; i++)
    {
      if (buf[i] == 'X' &&
	  buf[i + 1] == 'i' &&
	  buf[i + 2] == 'n' &&
	  buf[i + 3] == 'g')
	{
	  int frames = unpack_int32 (buf + i + 8);
	  int bytes = unpack_int32 (buf + i +12);
	  long long total_bytes, magic1, magic2;

	  if (frames <= 0)
	    {
	      bitrate = 0;
	      goto done;
	    }
	  
	  total_bytes = (long long) samprates[ver][samprate_index] * (long long) bytes;
	  magic1 = total_bytes / (long long) (576 + ver * 576);
	  magic2 = magic1 / (long long) frames;
	  bitrate = (int) ((long long) magic2 / (long long) 125);
	}
    }
 
done:
  song->bitrate = bitrate;
  song->samplerate = samprate;
  song->length = ((double) song->filesize / 1024.0f) / ((double) song->bitrate / 8000.0f);

  if (id3_file)
    id3_file_close (id3_file);
  
  fclose (file);
}

static void
assign_song_info_ogg (Song *song)
{
  FILE *f;
  OggVorbis_File vf;
  vorbis_comment *comment;
  vorbis_info *info;
  int i;
  
  f = fopen (song->filename, "r");
  if (!f)
    return;
  
  if (ov_open (f, &vf, NULL, 0) < 0)
    {
      fclose (f);
      return;
    }

  comment = ov_comment (&vf, -1);
  if (!comment)
    {
      ov_clear (&vf);
      return;
    }

  song->genre = -1;
  
  for (i = 0; i < comment->comments; i++)
    {
      if (strncasecmp (comment->user_comments[i], "title=", 6) == 0)
	song->title = g_strdup (comment->user_comments[i] + 6);
      else if (strncasecmp (comment->user_comments[i], "artist=", 7) == 0)
	song->artist = g_strdup (comment->user_comments[i] + 7);
      else if (strncasecmp (comment->user_comments[i], "album=", 6) == 0)
	song->album = g_strdup (comment->user_comments[i] + 6);
      else if (strncasecmp (comment->user_comments[i], "tracknumber=", 12) == 0)
	song->track_number = strtol (comment->user_comments[i] + 12, NULL, 10);
      else if (strncasecmp (comment->user_comments[i], "date=", 5) == 0)
	song->year = strtol (comment->user_comments[i] + 5, NULL, 10);
    }

  info = ov_info (&vf, -1);
  
  song->length = ov_time_total (&vf, -1) * 1000;
  song->bitrate = info->bitrate_nominal;
  song->samplerate = info->rate;
  ov_clear (&vf);
}

static void
cleanup_name (char *name)
{
  char *p;
  gunichar c;
  
  if (g_str_has_suffix (name, ".mp3") || g_str_has_suffix (name, ".ogg"))
    {
      p = strrchr (name, '.'); 
      if (p)
	*p = 0;
    }
  
  p = name;
  while (*p)
    {
      c = g_utf8_get_char (p);
      
      if (c == '_')
	*p = ' ';
      
      p = g_utf8_next_char (p);
    }
}

/*
 * FIXME: Update the song if the tags has changed
 * FIXME: Handle covers
 */
static gboolean
add_song (SongDB* db, const char *filename, gboolean overwrite)
{
  datum key, data;
  int ret;
  gpointer p;
  gsize len;
  Song *song;
  struct stat buf;
  time_t now;

  if (access (filename, R_OK))
    return FALSE;
  
  song = g_new0 (Song, 1);
  song->filename = g_strdup (filename);

  memset (&key, 0, sizeof (key));
  key.dptr = song->filename;
  key.dsize = strlen (key.dptr);

  if (!overwrite)
    {
      if (gdbm_exists (db->dbf, key))
	{
	  g_free (song->filename);
	  g_free (song);
	  return FALSE;
	}
  }

  if (stat (filename, &buf) == 0)
    song->filesize = buf.st_size;
  else
    song->filesize = 0;
  
  if (g_str_has_suffix (filename, ".mp3"))
    assign_song_info_mp3 (song);    
  else if (g_str_has_suffix (filename, ".ogg"))
    assign_song_info_ogg (song);

  if (!song->title || !song->title[0])
    {
      char *utf8;
      char *tmp;

      g_free (song->title);

      /* FIXME: This doesn't always work. Should try without converting first. */
      utf8 = g_filename_to_utf8 (song->filename, -1, NULL, NULL, NULL);
      if (!utf8)
	{
	  song_free (song);
	  return FALSE;
	}
      
      tmp = g_path_get_basename (utf8);
      g_free (utf8);

      song->title = tmp;
    }

  cleanup_name (song->title);
  
  if (!song->artist)
    song->artist = g_strdup ("");

  if (!song->album)
    song->album = g_strdup ("");

  now = time (NULL);
  song->date_added = now;
  song->date_modified = now; 
  
  p = pack_song (song, &len);

  memset (&data, 0, sizeof (data));
  data.dptr = p;
  data.dsize = len;

  ret = gdbm_store (db->dbf, key, data, overwrite ? 0 : GDBM_REPLACE);
  db->songs = g_list_prepend (db->songs, song);

  g_free (p);

  if (song)
    _song_calculate_collate_keys (song);
  
  return ret == 0;
}

static gboolean
add_dir (SongDB                *db,
	 const char            *path,
	 SongDBAddProgressFunc  progress_callback,
	 gpointer               user_data)
{
  GDir *dir;
  const char *name;
  char *full;

  dir = g_dir_open (path, 0, NULL);

  while ((name = g_dir_read_name (dir)))
    {
      full = g_build_filename (path, name, NULL);
      
      if (g_file_test (full, G_FILE_TEST_IS_DIR))
	{
	  if (!add_dir (db, full, progress_callback, user_data))
	    return FALSE;
	}
      else
	{
	  /* FIXME: Perhaps sniff this info from the song data. */
	  if (g_str_has_suffix (name, ".mp3") || g_str_has_suffix (name, ".ogg"))
	    {
	      if (progress_callback)
		if (!progress_callback (db, full, user_data))
		  return FALSE;

	      add_song (db, full, FALSE);
	    }
	}

      g_free (full);
    }

  g_dir_close (dir);

  return TRUE;
}

void
song_db_add_dir (SongDB                *db,
		 const char            *path,
		 SongDBAddProgressFunc  progress_callback,
		 gpointer               user_data)
{
  if (g_file_test (path, G_FILE_TEST_IS_DIR))
    add_dir (db, path, progress_callback, user_data);
  else if (g_file_test (path, G_FILE_TEST_IS_REGULAR))
    {
      if (progress_callback)
	if (!progress_callback (db, path, user_data))
	  return;

      add_song (db, path, FALSE);
    }
  else
    {
      g_warning ("Don't know how to handle: %s", path);
      return;
    }
}

void
song_db_dump (SongDB *db)
{
  GList *l;
  
  g_return_if_fail (IS_SONG_DB (db));

  g_print ("Dump of song database\n");
  g_print ("---------------------\n");
  
  for (l = db->songs; l; l = l->next)
    _song_dump (l->data);
}

void
song_db_add_file (SongDB     *db,
		  const char *filename)
{
  g_return_if_fail (IS_SONG_DB (db));

  add_song (db, filename, FALSE);
}

void
song_db_update_song (SongDB *db, Song *song)
{
  int ret;
  datum song_key, song_data;
  gpointer p;
  int len;
  
  g_return_if_fail (IS_SONG_DB (db));
  g_return_if_fail (song != NULL);

  song->date_modified = time (NULL);
  
  p = pack_song (song, &len);

  memset (&song_key, 0, sizeof (song_key));
  song_key.dptr = song->filename;
  song_key.dsize = strlen (song_key.dptr);
    
  memset (&song_data, 0, sizeof (song_data));
  song_data.dptr = p;
  song_data.dsize = len;

  ret = gdbm_store (db->dbf, song_key, song_data, GDBM_REPLACE);
}

gboolean
song_db_remove_song (SongDB *db,
		     Song   *song)
{
  datum key;
  int ret;
    
  g_return_val_if_fail (IS_SONG_DB (db), FALSE);

  memset (&key, 0, sizeof (key));
  key.dptr = g_strdup (song->filename);
  key.dsize = strlen (key.dptr);

  ret = gdbm_delete (db->dbf, key);
  
  db->songs = g_list_remove (db->songs, song);

  song_free (song);
  
  return TRUE;
}

int
song_db_get_version (SongDB *db)
{
  datum key, data;
  int ret;

  g_return_val_if_fail (IS_SONG_DB (db), -1);

  memset (&key, 0, sizeof (key));
  key.dptr = VERSION_KEY;
  key.dsize = strlen (key.dptr);
  
  data = gdbm_fetch (db->dbf, key);
  if (!data.dptr)
    return -1;
  
  unpack_int (data.dptr, &ret);
  
  return ret;
}

void
song_db_set_version (SongDB *db, int version)
{
  GString *string;
  datum key, data;
  int ret;
  
  g_return_if_fail (IS_SONG_DB (db));

  string = g_string_new (NULL);

  pack_int (string, version);
  
  memset (&key, 0, sizeof (key));
  key.dptr = VERSION_KEY;
  key.dsize = strlen (key.dptr);
    
  memset (&data, 0, sizeof (data));
  data.dptr = string->str;
  data.dsize = string->len;

  ret = gdbm_store (db->dbf, key, data, GDBM_REPLACE);
  if (ret)
    {
      g_warning ("Could not update database version");
      return;
    }
  
}

void
song_db_rebuild (SongDB *db)
{
  datum key, data;
  GList *songs = NULL;
  GList *l;

  g_return_if_fail (IS_SONG_DB (db));

  memset (&key, 0, sizeof (key));
  memset (&data, 0, sizeof (data));
  
  while (1)
    {
      key = gdbm_nextkey (db->dbf, key);
      if (key.dptr == NULL)
	break;

      if (((char*)key.dptr)[0] == VERSION_KEY[0] &&
	  strncmp (key.dptr, VERSION_KEY, VERSION_KEY_LENGTH) == 0)
	continue;
      
      songs = g_list_prepend (songs, g_strndup (key.dptr, key.dsize));
    }

#if 0  
  if (ret != DB_NOTFOUND)
    g_warning ("Error while getting songs: %s", db_strerror (ret));
#endif
  
  for (l = songs; l; l = l->next)
    add_song (db, l->data, TRUE);

  g_list_foreach (songs, (GFunc) g_free, NULL);
  g_list_free (songs);

}
