/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003 Richard Hult <richard@imendio.com>
 * Copyright (C) 2003 Johan Dahlin <johan@gnome.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include "song-list-model.h"
#include "song.h"
#include "id3-tag.h"
#include "eggtreemultidnd.h"

static void song_list_model_finalize    (GObject       *object);
static void song_list_model_remove_iter (SongListModel *model,
					 GtkTreeIter   *iter);
static void clear_indices               (SongListModel *model);
static void ensure_indices              (SongListModel *model);


enum {
  TARGET_SONG_LIST
};

static const GtkTargetEntry drag_types[] = {
  {  "x-special/jamboree-song-list", 0, TARGET_SONG_LIST },
};

static GObjectClass *parent_class;

static GtkTreeModelFlags
song_list_model_get_flags (GtkTreeModel *tree_model)
{
  return GTK_TREE_MODEL_ITERS_PERSIST | GTK_TREE_MODEL_LIST_ONLY;
}

static int
song_list_model_get_n_columns (GtkTreeModel *tree_model)
{
  return 1;
}

static GType
song_list_model_get_column_type (GtkTreeModel *tree_model, int index)
{
  switch (index) {
  case 0:
    return G_TYPE_POINTER;
  default:
    return G_TYPE_INVALID;
  }
}

static gboolean
song_list_model_get_iter (GtkTreeModel *tree_model,
			  GtkTreeIter  *iter,
			  GtkTreePath  *path)
{
  SongListModel *model;
  GSequencePtr ptr;
  int i;

  model = (SongListModel *) tree_model;
  
  i = gtk_tree_path_get_indices (path)[0];

  if (i >= g_sequence_get_length (model->songs))
    return FALSE;

  ptr = g_sequence_get_ptr_at_pos (model->songs, i);

  iter->stamp = model->stamp;
  iter->user_data = ptr;

  return TRUE;
}

static GtkTreePath *
song_list_model_get_path (GtkTreeModel *tree_model,
			  GtkTreeIter  *iter)
{
  SongListModel *model;
  GtkTreePath *path;
  
  model = (SongListModel *) tree_model;

  g_return_val_if_fail (model->stamp == iter->stamp, NULL);

  if (g_sequence_ptr_is_end (iter->user_data))
    return NULL;
  
  path = gtk_tree_path_new ();
  gtk_tree_path_append_index (path, g_sequence_ptr_get_position (iter->user_data));

  return path;
}

static void
song_list_model_get_value (GtkTreeModel *tree_model,
			   GtkTreeIter  *iter,
			   int           column,
			   GValue       *value)
{
  SongListModel *model;
  Song *song;
   
  model = (SongListModel *) tree_model;

  g_return_if_fail (model->stamp == iter->stamp);

  song = g_sequence_ptr_get_data (iter->user_data);
  
  switch (column)
    {
    case 0:
      g_value_init (value, G_TYPE_POINTER);
      g_value_set_pointer (value, song);
      break;

    default:
      g_assert_not_reached ();
      break;
    }
}

static gboolean
song_list_model_iter_nth_child (GtkTreeModel *tree_model,
				GtkTreeIter  *iter,
				GtkTreeIter  *parent,
				int           n)
{
  SongListModel *model;
  GSequencePtr child;

  if (parent)
    return FALSE;

  model = (SongListModel *) tree_model;

  child = g_sequence_get_ptr_at_pos (model->songs, n);

  if (g_sequence_ptr_is_end (child))
    return FALSE;
  
  iter->stamp = model->stamp;
  iter->user_data = child;

  return TRUE;
}

static gboolean
song_list_model_iter_next (GtkTreeModel *tree_model,
			   GtkTreeIter  *iter)
{
  SongListModel *model;
  
  model = (SongListModel *) tree_model;

  g_return_val_if_fail (model->stamp == iter->stamp, FALSE);

  iter->user_data = g_sequence_ptr_next (iter->user_data);

  return !g_sequence_ptr_is_end (iter->user_data);
}

static gboolean
song_list_model_iter_children (GtkTreeModel *tree_model,
			       GtkTreeIter  *iter,
			       GtkTreeIter  *parent)
{
  SongListModel *model;
  
  if (parent)
    return FALSE;
  
  model = (SongListModel *) tree_model;
  
  if (g_sequence_get_length (model->songs) == 0)
    return FALSE;

  iter->stamp = model->stamp;
  iter->user_data = g_sequence_get_begin_ptr (model->songs);
  
  return TRUE;
}

static int
song_list_model_iter_n_children (GtkTreeModel *tree_model,
				 GtkTreeIter  *iter)
{
  SongListModel *model;
  
  model = (SongListModel *) tree_model;
  
  if (iter == NULL)
    return g_sequence_get_length (model->songs);

  g_return_val_if_fail (model->stamp == iter->stamp, -1);

  return 0;
}

static gboolean
song_list_model_iter_parent (GtkTreeModel *tree_model,
			     GtkTreeIter  *iter,
			     GtkTreeIter  *child)
{
  return FALSE;
}

static gboolean
song_list_model_iter_has_child (GtkTreeModel *tree_model,
				GtkTreeIter  *iter)
{
  return FALSE;
}

static gboolean
song_list_model_multi_row_draggable (EggTreeMultiDragSource *drag_source,
				     GList                  *path_list)
{
  return TRUE;
}

static GtkTargetList *drag_target_list = NULL;

typedef struct {
  SongListModel *model;
  GList *path_list;
} DragData;

static gboolean
song_list_model_multi_drag_data_get (EggTreeMultiDragSource *drag_source, 
				     GList                  *path_list, 
				     GtkSelectionData       *selection_data)
{
  SongListModel *model;
  DragData data;
  guint target_info;

  model = (SongListModel *) drag_source;
	
  data.model = model;
  data.path_list = path_list;
  
  if (!drag_target_list)
    drag_target_list = gtk_target_list_new (drag_types, G_N_ELEMENTS (drag_types));
  
  if (gtk_target_list_find (drag_target_list, selection_data->target, &target_info))
    {
      GList *l;
      Song *song;
      GString *str;

      g_assert (target_info == TARGET_SONG_LIST);

      str = g_string_new (NULL);
      
      for (l = path_list; l; l = l->next)
	{
	  GtkTreeIter iter;
	  GtkTreePath *path = gtk_tree_row_reference_get_path (l->data);
	  
	  gtk_tree_model_get_iter (GTK_TREE_MODEL (model), &iter, path);

	  song = g_sequence_ptr_get_data (iter.user_data);

	  g_string_append_printf (str, "%p", song);

	  if (l->next)
	    g_string_append (str, "\r\n");
	}

      gtk_selection_data_set (selection_data,
			      selection_data->target,
			      8,
			      str->str,
			      strlen (str->str));

      g_string_free (str, TRUE);
      
      return TRUE;
    }
  else
    {
      return FALSE;
    }
}

static gboolean
song_list_model_multi_drag_data_delete (EggTreeMultiDragSource *drag_source, GList *path_list)
{
  return TRUE;
}

static void
song_list_model_tree_model_init (GtkTreeModelIface *iface)
{
  iface->get_flags = song_list_model_get_flags;
  iface->get_n_columns = song_list_model_get_n_columns;
  iface->get_column_type = song_list_model_get_column_type;
  iface->get_iter = song_list_model_get_iter;
  iface->get_path = song_list_model_get_path;
  iface->get_value = song_list_model_get_value;
  iface->iter_nth_child = song_list_model_iter_nth_child;
  iface->iter_next = song_list_model_iter_next;
  iface->iter_has_child = song_list_model_iter_has_child;
  iface->iter_n_children = song_list_model_iter_n_children;
  iface->iter_children = song_list_model_iter_children;
  iface->iter_parent = song_list_model_iter_parent;
}

static void
song_list_model_multi_drag_source_init (EggTreeMultiDragSourceIface *iface)
{
  iface->row_draggable = song_list_model_multi_row_draggable;
  iface->drag_data_get = song_list_model_multi_drag_data_get;
  iface->drag_data_delete = song_list_model_multi_drag_data_delete;
}

static void
song_list_model_class_init (SongListModelClass *klass)
{
  GObjectClass *object_class;

  parent_class = g_type_class_peek_parent (klass);
  object_class = (GObjectClass*) klass;

  object_class->finalize = song_list_model_finalize;
}

static void
song_list_model_init (SongListModel *model)
{
  model->songs = g_sequence_new (NULL);
  model->reverse_map = g_hash_table_new (NULL, NULL);
  model->stamp = g_random_int ();
  model->sort_id = SONG_SORT_ID_DEFAULT;
  model->sort_type = GTK_SORT_ASCENDING;

  model->current_index = -1;
}

static void
song_list_model_finalize (GObject *object)
{
  SongListModel *model = SONG_LIST_MODEL (object);
  
  g_sequence_free (model->songs);
  g_hash_table_destroy (model->reverse_map);

  if (G_OBJECT_CLASS (parent_class)->finalize)
    G_OBJECT_CLASS (parent_class)->finalize (object);
}

GType
song_list_model_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo object_info = {
	sizeof (SongListModelClass),
	NULL,		/* base_init */
	NULL,		/* base_finalize */
	(GClassInitFunc) song_list_model_class_init,
	NULL,		/* class_finalize */
	NULL,		/* class_data */
	sizeof (SongListModel),
	0,              /* n_preallocs */
	(GInstanceInitFunc) song_list_model_init
      };
	    
      static const GInterfaceInfo tree_model_info = {
	(GInterfaceInitFunc) song_list_model_tree_model_init,
	NULL,
	NULL
      };
	    
      static const GInterfaceInfo multi_drag_source_info = {
	(GInterfaceInitFunc) song_list_model_multi_drag_source_init,
	NULL,
	NULL
      };
	    
      type = g_type_register_static (G_TYPE_OBJECT,
				     "SongListModel",
				     &object_info, 0);
	    
      g_type_add_interface_static (type,
				   GTK_TYPE_TREE_MODEL,
				   &tree_model_info);
	    
      g_type_add_interface_static (type,
				   EGG_TYPE_TREE_MULTI_DRAG_SOURCE,
				   &multi_drag_source_info);
    }
  
  return type;
}

GtkTreeModel *
song_list_model_new (void)
{
  return g_object_new (TYPE_SONG_LIST_MODEL, NULL);
}

static int
compare_int (int a, int b)
{
  if (a < b)
    return -1;
  else if (a > b)
    return 1;
  else
    return 0;
}

static int
sort_artist_album_track (Song *song_a, Song *song_b)
{
  int ret;

  ret = strcmp (string_entry_get_collated (song_a->artist),
		string_entry_get_collated (song_b->artist));
  if (ret == 0)
    {
      ret = strcmp (string_entry_get_collated (song_a->album),
		    string_entry_get_collated (song_b->album));
      if (ret == 0)
	ret = compare_int (song_a->track_number, song_b->track_number);
    }

  return ret;
}

static int
sort_album_track (Song *song_a, Song *song_b)
{
  int ret;

  ret = strcmp (string_entry_get_collated (song_a->album),
		string_entry_get_collated (song_b->album));
  if (ret == 0)
    ret = compare_int (song_a->track_number, song_b->track_number);
  
  return ret;
}

static int
song_compare_func (Song *song_a, Song *song_b, SongListModel *model)
{
  int ret, second = 0;

  switch (model->sort_id) {
  case SONG_SORT_ID_DEFAULT:
  case SONG_SORT_ID_ARTIST:
    ret = strcmp (string_entry_get_collated (song_a->artist),
		  string_entry_get_collated (song_b->artist));
    if (ret == 0)
      second = sort_album_track (song_a, song_b);
    break;
    
  case SONG_SORT_ID_TITLE:
    ret = strcmp (string_entry_get_collated (song_a->title),
		  string_entry_get_collated (song_b->title));
    if (ret == 0)
      second = sort_artist_album_track (song_a, song_b);
    break;

  case SONG_SORT_ID_TIME:
    ret = compare_int (song_a->length, song_b->length);
    if (ret == 0)
      second = sort_artist_album_track (song_a, song_b);
    break;

  case SONG_SORT_ID_ALBUM:
    ret = strcmp (string_entry_get_collated (song_a->album),
		  string_entry_get_collated (song_b->album));
    if (ret == 0)
      second = compare_int (song_a->track_number, song_b->track_number);
    break;

  case SONG_SORT_ID_GENRE:
    ret = strcasecmp (id3_genres[song_a->genre], id3_genres[song_b->genre]);
    if (ret == 0)
      second = sort_artist_album_track (song_a, song_b);
    break;
    
  case SONG_SORT_ID_YEAR:
    ret = compare_int (song_a->year, song_b->year);
    if (ret == 0)
      second = sort_artist_album_track (song_a, song_b);
    break;
    
  case SONG_SORT_ID_TRACK_NUMBER:
    ret = compare_int (song_a->track_number, song_b->track_number);
    break;
    
  case SONG_SORT_ID_RATING:
    ret = compare_int (song_a->rating, song_b->rating);
    break;

  case SONG_SORT_ID_PLAY_COUNT:
    ret = compare_int (song_a->play_count, song_b->play_count);
    break;

  default:
    ret = 0;
    break;
  }
  
  if (model->sort_type == GTK_SORT_DESCENDING)
      ret = -ret;

  if (ret == 0)
    ret = second;
  
  return ret;
}

gboolean
song_list_model_add (SongListModel *model, Song *song)
{
  GtkTreeIter iter;
  GtkTreePath *path;
  GSequencePtr new_ptr;

  if (g_hash_table_lookup (model->reverse_map, song))
    return FALSE;

  clear_indices (model);
  
  new_ptr = g_sequence_insert_sorted (model->songs, song,
				      (GCompareDataFunc) song_compare_func,
				      model);
  
  g_hash_table_insert (model->reverse_map, song, new_ptr);
	
  iter.stamp = model->stamp;
  iter.user_data = new_ptr;
  
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), &iter);
  gtk_tree_model_row_inserted (GTK_TREE_MODEL (model), path, &iter);
  gtk_tree_path_free (path);

  return TRUE;
}

void
song_list_model_remove_iter (SongListModel *model, GtkTreeIter *iter)
{
  GSequencePtr ptr;
  GtkTreePath *path;

  clear_indices (model);

  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), iter);
  ptr = iter->user_data;

  g_hash_table_remove (model->reverse_map, g_sequence_ptr_get_data (ptr));
  g_sequence_remove (ptr);
  
  model->stamp++;

  gtk_tree_model_row_deleted (GTK_TREE_MODEL (model), path);
  gtk_tree_path_free (path);
}

void
song_list_model_remove (SongListModel *model, Song *song)
{
  GtkTreeIter iter;
  
  if (song_list_model_song_get_iter (model, song, &iter))
    song_list_model_remove_iter (model, &iter);
}

void
song_list_model_clear (SongListModel *model)
{
  GtkTreeIter iter;
  
  g_return_if_fail (model != NULL);

  clear_indices (model);
  
  while (g_sequence_get_length (model->songs) > 0)
    {
      iter.stamp = model->stamp;
      iter.user_data = g_sequence_get_begin_ptr (model->songs);
      song_list_model_remove_iter (model, &iter);
    }
}

static void
song_list_model_sort (SongListModel *model)
{
  GSequence *songs;
  GSequencePtr *old_order;
  GtkTreePath *path;
  int *new_order;
  int length;
  int i;

  songs = model->songs;
  length = g_sequence_get_length (songs);

  if (length <= 1)
    return;
  
  /* Generate old order of GSequencePtrs. */
  old_order = g_new (GSequencePtr, length);
  for (i = 0; i < length; ++i)
    old_order[i] = g_sequence_get_ptr_at_pos (songs, i);

  g_sequence_sort (songs, (GCompareDataFunc) song_compare_func, model);
  
  /* Generate new order. */
  new_order = g_new (int, length);
  for (i = 0; i < length; ++i)
    new_order[i] = g_sequence_ptr_get_position (old_order[i]);

  path = gtk_tree_path_new ();
  
  gtk_tree_model_rows_reordered (GTK_TREE_MODEL (model), path, NULL, new_order);
  
  gtk_tree_path_free (path);
  g_free (old_order);
  g_free (new_order);
}

void
song_list_model_set_sorting (SongListModel *model,
			     SongSortId     id,
			     GtkSortType    type)
{
  if (id == model->sort_id && type == model->sort_type)
    return;

  model->sort_id = id;
  model->sort_type = type;

  song_list_model_sort (model);
}

gboolean
song_list_model_song_get_iter (SongListModel *model,
			       Song          *song,
			       GtkTreeIter   *iter)
{
  GSequencePtr ptr;

  ptr = g_hash_table_lookup (model->reverse_map, song);
  if (!ptr)
    return FALSE;
  
  if (iter != NULL)
    {
      iter->stamp = model->stamp;
      iter->user_data = ptr;
    }
  
  return TRUE;
}

Song *
song_list_model_get_song (SongListModel *model, GtkTreeIter *iter)
{
  g_return_val_if_fail (model->stamp == iter->stamp, NULL);
  
  return g_sequence_ptr_get_data (iter->user_data);
}

GList *
song_list_model_get_songs (SongListModel *model)
{
  GList *list = NULL;
  GSequencePtr ptr;

  ptr = g_sequence_get_begin_ptr (model->songs);
  while (!g_sequence_ptr_is_end (ptr))
    {
      list = g_list_prepend (list, g_sequence_ptr_get_data (ptr));
      ptr = g_sequence_ptr_next (ptr);
    }

  return g_list_reverse (list);
}

static void
remove_ptr (SongListModel *model, GSequencePtr ptr)
{
  GtkTreePath *path;
  
  path = gtk_tree_path_new ();
  gtk_tree_path_append_index (path, g_sequence_ptr_get_position (ptr));
  
  g_hash_table_remove (model->reverse_map, g_sequence_ptr_get_data (ptr));
  
  g_sequence_remove (ptr);

  /* FIXME: Why oh why is this needed? The length is reported as 0 if we don't
   * do this after removing a ptr. It's probably due to some missing splay?
   */
  /*  g_sequence_get_begin_ptr (model->songs);*/

  model->stamp++;
  
  gtk_tree_model_row_deleted (GTK_TREE_MODEL (model), path);
  gtk_tree_path_free (path);
}

void
song_list_model_remove_delta (SongListModel *model, GList *songs)
{
  GHashTable *hash;
  GList *l, *remove = NULL;
  Song *song;
  GSequencePtr ptr;
  
  if (g_sequence_get_length (model->songs) == 0)
    return;

  if (!songs)
    {
      song_list_model_clear (model);
      return;
    }

  hash = g_hash_table_new (NULL, NULL);
 
  for (l = songs; l; l = l->next)
    g_hash_table_insert (hash, l->data, GINT_TO_POINTER (TRUE));
 
  ptr = g_sequence_get_begin_ptr (model->songs);
  while (!g_sequence_ptr_is_end (ptr))
    {
      song = g_sequence_ptr_get_data (ptr);
      if (!g_hash_table_lookup (hash, song))
	remove = g_list_prepend (remove, ptr);
      
      ptr = g_sequence_ptr_next (ptr);
    }

  for (l = remove; l; l = l->next)
    remove_ptr (model, l->data);

  g_list_free (remove);
  g_hash_table_destroy (hash);

  clear_indices (model);
}

Song *
song_list_model_get_current (SongListModel *model)
{
  int pos;
  GSequencePtr ptr;
  
  g_return_val_if_fail (IS_SONG_LIST_MODEL (model), NULL);

  if (g_sequence_get_length (model->songs) == 0)
    return NULL;
  
  ensure_indices (model);

  if (model->current_index == -1)
    return NULL;

  pos = model->indices[model->current_index];

  ptr = g_sequence_get_ptr_at_pos (model->songs, pos);
  
  if (!ptr)
    return NULL;

  return g_sequence_ptr_get_data (ptr);
}

gboolean
song_list_model_set_current (SongListModel *model, Song *song)
{
  GSequencePtr ptr;
  int i, len, pos;
  
  g_return_val_if_fail (IS_SONG_LIST_MODEL (model), FALSE);

  if (!song)
    {
      model->current_index = -1;
      return TRUE;
    }
  
  len = g_sequence_get_length (model->songs);
  
  if (len == 0)
    return FALSE;
  
  ensure_indices (model);

  ptr = g_hash_table_lookup (model->reverse_map, song);
  if (!ptr)
    return FALSE;

  pos = g_sequence_ptr_get_position (ptr);

  if (!model->shuffle)
    {
      model->current_index = pos;
      return TRUE;
    }

  for (i = 0; i < len; i++)
    if (model->indices[i] == pos)
      {
	model->current_index = i;
	return TRUE;
      }
  
  return FALSE;
}

Song *
song_list_model_next (SongListModel *model)
{
  GSequencePtr ptr;
  Song *song = NULL;
  
  g_return_val_if_fail (IS_SONG_LIST_MODEL (model), NULL);

  if (!song_list_model_has_next (model))
    return NULL;

  ptr = g_sequence_get_ptr_at_pos (model->songs, model->indices[model->current_index + 1]);
  if (ptr)
    {
      song = g_sequence_ptr_get_data (ptr);
      model->current_index++;
    }

  return song;
}

Song *
song_list_model_prev (SongListModel *model)
{
  GSequencePtr ptr;
  Song *song = NULL;
  
  g_return_val_if_fail (IS_SONG_LIST_MODEL (model), NULL);

  if (!song_list_model_has_prev (model))
    return NULL;

  ptr = g_sequence_get_ptr_at_pos (model->songs, model->indices[model->current_index - 1]);
  if (ptr)
    {
      song = g_sequence_ptr_get_data (ptr);
      model->current_index--;
    }

  return song;
}

Song *
song_list_model_first (SongListModel *model)
{
  GSequencePtr ptr;
  Song *song = NULL;
  
  g_return_val_if_fail (IS_SONG_LIST_MODEL (model), NULL);

  if (g_sequence_get_length (model->songs) == 0)
    return NULL;
  
  ensure_indices (model);

  ptr = g_sequence_get_ptr_at_pos (model->songs, model->indices[0]);
  if (ptr)
    {
      song = g_sequence_ptr_get_data (ptr);
      model->current_index = 0;
    }

  return song;
}

typedef struct
{
  int random;
  int index;
} RandomData;

static int
compare_random (gconstpointer ptr_a, gconstpointer ptr_b)
{
  RandomData *a = (RandomData *) ptr_a;
  RandomData *b = (RandomData *) ptr_b;

  if (a->random < b->random)
    return -1;
  else if (a->random > b->random)
    return 1;
  else
    return 0;
}

static void
clear_indices (SongListModel *model)
{
  model->current_index = -1;
  g_free (model->indices);
  model->indices = NULL;
}

static void
ensure_indices (SongListModel *model)
{
  int len, i;
  RandomData data;
  GArray *array;

  if (model->indices)
    return;
  
  len = g_sequence_get_length (model->songs);
  model->indices = g_new (int, len);

  if (model->shuffle)
    {
      array = g_array_sized_new (FALSE, FALSE, sizeof (RandomData), len);

      for (i = 0; i < len; i++)
	{
	  data.random = g_random_int_range (0, len);
	  data.index = i;
	  
	  g_array_append_val (array, data);
	}
      
      g_array_sort (array, compare_random);

      for (i = 0; i < len; i++)
	model->indices[i] = g_array_index (array, RandomData, i).index;
      
      g_array_free (array, TRUE);
    }
  else
    for (i = 0; i < len; i++)
      model->indices[i] = i;
}

void
song_list_model_set_shuffle (SongListModel *model, gboolean shuffle)
{
  Song *song;
  
  g_return_if_fail (IS_SONG_LIST_MODEL (model));
  
  model->shuffle = shuffle;
  
  song = song_list_model_get_current (model);
  
  clear_indices (model);

  /* Keep the same current song when going from shuffle to normal. */
  if (!shuffle && song)
    song_list_model_set_current (model, song);
}

gboolean
song_list_model_has_prev (SongListModel *model)
{
  int len;
  GSequencePtr ptr;

  g_return_val_if_fail (IS_SONG_LIST_MODEL (model), FALSE);

  len = g_sequence_get_length (model->songs);

  if (len == 0)
    return FALSE;

  ensure_indices (model);

  /* Caller must decide if we should wrap. */
  if (model->current_index <= 0)
    return FALSE;

  ptr = g_sequence_get_ptr_at_pos (model->songs, model->indices[model->current_index - 1]);
  
  return ptr != NULL;
}

gboolean
song_list_model_has_next (SongListModel *model)
{
  int len;
  GSequencePtr ptr;

  g_return_val_if_fail (IS_SONG_LIST_MODEL (model), FALSE);

  len = g_sequence_get_length (model->songs);

  if (len == 0)
    return FALSE;

  ensure_indices (model);

  /* Caller must decide if we should wrap. */
  if (model->current_index + 1 >= len)
    return FALSE;

  ptr = g_sequence_get_ptr_at_pos (model->songs, model->indices[model->current_index + 1]);
  
  return ptr != NULL;
}

