/*
 * Copyright © 2007-2009 Emmanuel Pacaud
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Emmanuel Pacaud <emmanuel@gnome.org>
 */

#include <lsmdomdocument.h>
#include <lsmdomelement.h>
#include <lsmdebug.h>
#include <lsmdomtext.h>

/* LsmDomNode implementation */

static const char *
lsm_dom_document_get_node_name (LsmDomNode *node)
{
	return "#document";
}

static LsmDomNodeType
lsm_dom_document_get_node_type (LsmDomNode *node)
{
	return LSM_DOM_NODE_TYPE_DOCUMENT_NODE;
}

/* LsmDomDocument implementation */

LsmDomElement *
lsm_dom_document_get_document_element (LsmDomDocument *self)
{
	g_return_val_if_fail (LSM_IS_DOM_DOCUMENT (self), NULL);

	return LSM_DOM_ELEMENT (lsm_dom_node_get_first_child (LSM_DOM_NODE (self)));
}

LsmDomElement *
lsm_dom_document_create_element (LsmDomDocument *document, const char *tag_name)
{
	LsmDomDocumentClass *document_class;

	g_return_val_if_fail (LSM_IS_DOM_DOCUMENT (document), NULL);

	document_class = LSM_DOM_DOCUMENT_GET_CLASS (document);
	if (document_class->create_element != NULL)
		return document_class->create_element (document, tag_name);

	return NULL;
}

LsmDomText *
lsm_dom_document_create_text_node_base (LsmDomDocument *document, const char *data)
{
	return LSM_DOM_TEXT (lsm_dom_text_new (data));
}

LsmDomText *
lsm_dom_document_create_text_node (LsmDomDocument *document, const char *data)
{
	g_return_val_if_fail (LSM_IS_DOM_DOCUMENT (document), NULL);

	return LSM_DOM_DOCUMENT_GET_CLASS (document)->create_text_node (document, data);
}

LsmDomView *
lsm_dom_document_create_view (LsmDomDocument *self)
{
	g_return_val_if_fail (LSM_IS_DOM_DOCUMENT (self), NULL);

	return LSM_DOM_DOCUMENT_GET_CLASS (self)->create_view (self);
}

double
lsm_dom_document_get_resolution (LsmDomDocument *self)
{
	g_return_val_if_fail (LSM_IS_DOM_DOCUMENT (self), 0.0);

	return self->resolution_ppi;
}

void
lsm_dom_document_set_resolution (LsmDomDocument *self, double ppi)
{
	g_return_if_fail (LSM_IS_DOM_DOCUMENT (self));

	if (ppi < 0.0)
		self->resolution_ppi = LSM_DOM_DOCUMENT_DEFAULT_RESOLUTION;
	else
		self->resolution_ppi = ppi;
}

void
lsm_dom_document_set_viewport (LsmDomDocument *self, const LsmBox *viewport_pt)
{
	g_return_if_fail (LSM_IS_DOM_DOCUMENT (self));
	g_return_if_fail (viewport_pt != NULL);

	self->viewport_pt = *viewport_pt;
}

void
lsm_dom_document_set_viewport_px (LsmDomDocument *self, const LsmBox *viewport)
{
	g_return_if_fail (LSM_IS_DOM_DOCUMENT (self));
	g_return_if_fail (viewport != NULL);

	self->viewport_pt.x      = viewport->x      * 72.0 / self->resolution_ppi;
	self->viewport_pt.y      = viewport->y      * 72.0 / self->resolution_ppi;
	self->viewport_pt.width  = viewport->width  * 72.0 / self->resolution_ppi;
	self->viewport_pt.height = viewport->height * 72.0 / self->resolution_ppi;
}

LsmBox
lsm_dom_document_get_viewport (LsmDomDocument *self)
{
	static const LsmBox null_viewport = {0, 0, 0, 0};

	g_return_val_if_fail (LSM_IS_DOM_DOCUMENT (self), null_viewport);

	return self->viewport_pt;
}

LsmBox
lsm_dom_document_get_viewport_px (LsmDomDocument *self)
{
	LsmBox viewport = {0, 0, 0, 0};

	g_return_val_if_fail (LSM_IS_DOM_DOCUMENT (self), viewport);

	viewport.x      = self->viewport_pt.x      * self->resolution_ppi / 72.0;
	viewport.y      = self->viewport_pt.y      * self->resolution_ppi / 72.0;
	viewport.width  = self->viewport_pt.width  * self->resolution_ppi / 72.0;
	viewport.height = self->viewport_pt.height * self->resolution_ppi / 72.0;

	return viewport;
}

LsmDomElement *
lsm_dom_document_get_element_by_id (LsmDomDocument *self, const char *id)
{
	g_return_val_if_fail (LSM_IS_DOM_DOCUMENT (self), NULL);
	g_return_val_if_fail (id != NULL, NULL);

	lsm_debug ("[LsmDomDocument::get_element_by_id] Lookup '%s'", id);

	return g_hash_table_lookup (self->ids, id);
}

void
lsm_dom_document_register_element (LsmDomDocument *self, LsmDomElement *element, const char *id)
{
	char *old_id;

	g_return_if_fail (LSM_IS_DOM_DOCUMENT (self));

	old_id = g_hash_table_lookup (self->elements, element);
	if (old_id != NULL) {
		lsm_debug ("[LsmDomDocument::register_element] Unregister '%s'", old_id);

		g_hash_table_remove (self->elements, element);
		g_hash_table_remove (self->ids, old_id);
	}

	if (id != NULL) {
		char *new_id = g_strdup (id);

		lsm_debug ("[LsmDomDocument::register_element] Register '%s'", id);

		g_hash_table_replace (self->ids, new_id, element);
		g_hash_table_replace (self->elements, element, new_id);
	}
}

static void
lsm_dom_document_init (LsmDomDocument *document)
{
	document->ids = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
	document->elements = g_hash_table_new_full (g_direct_hash, g_direct_equal, NULL, NULL);

	document->resolution_ppi = LSM_DOM_DOCUMENT_DEFAULT_RESOLUTION;
	document->viewport_pt.x = 0;
	document->viewport_pt.y = 0;
	document->viewport_pt.width  = LSM_DOM_DOCUMENT_DEFAULT_VIEWBOX_WIDTH;
	document->viewport_pt.height = LSM_DOM_DOCUMENT_DEFAULT_VIEWBOX_HEIGHT;
}

static void
lsm_dom_document_finalize (GObject *object)
{
	LsmDomDocument *document = LSM_DOM_DOCUMENT (object);

	g_hash_table_unref (document->elements);
	g_hash_table_unref (document->ids);
}

/* LsmDomDocument class */

static void
lsm_dom_document_class_init (LsmDomDocumentClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	LsmDomNodeClass *node_class = LSM_DOM_NODE_CLASS (klass);

	object_class->finalize = lsm_dom_document_finalize;

	node_class->get_node_name = lsm_dom_document_get_node_name;
	node_class->get_node_type = lsm_dom_document_get_node_type;

	klass->create_text_node = lsm_dom_document_create_text_node_base;
}

G_DEFINE_ABSTRACT_TYPE (LsmDomDocument, lsm_dom_document, LSM_TYPE_DOM_NODE)

