/* arraylist.vala
 *
 * Copyright (C) 2004-2005  Novell, Inc
 * Copyright (C) 2005  David Waite
 * Copyright (C) 2007-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <string.h>


#define GEE_TYPE_ITERABLE (gee_iterable_get_type ())
#define GEE_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERABLE, GeeIterable))
#define GEE_IS_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERABLE))
#define GEE_ITERABLE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERABLE, GeeIterableIface))

typedef struct _GeeIterable GeeIterable;
typedef struct _GeeIterableIface GeeIterableIface;

#define GEE_TYPE_ITERATOR (gee_iterator_get_type ())
#define GEE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERATOR, GeeIterator))
#define GEE_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERATOR))
#define GEE_ITERATOR_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERATOR, GeeIteratorIface))

typedef struct _GeeIterator GeeIterator;
typedef struct _GeeIteratorIface GeeIteratorIface;

#define GEE_TYPE_COLLECTION (gee_collection_get_type ())
#define GEE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION, GeeCollection))
#define GEE_IS_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION))
#define GEE_COLLECTION_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_COLLECTION, GeeCollectionIface))

typedef struct _GeeCollection GeeCollection;
typedef struct _GeeCollectionIface GeeCollectionIface;

#define GEE_TYPE_ABSTRACT_COLLECTION (gee_abstract_collection_get_type ())
#define GEE_ABSTRACT_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ABSTRACT_COLLECTION, GeeAbstractCollection))
#define GEE_ABSTRACT_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_ABSTRACT_COLLECTION, GeeAbstractCollectionClass))
#define GEE_IS_ABSTRACT_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ABSTRACT_COLLECTION))
#define GEE_IS_ABSTRACT_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_ABSTRACT_COLLECTION))
#define GEE_ABSTRACT_COLLECTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_ABSTRACT_COLLECTION, GeeAbstractCollectionClass))

typedef struct _GeeAbstractCollection GeeAbstractCollection;
typedef struct _GeeAbstractCollectionClass GeeAbstractCollectionClass;
typedef struct _GeeAbstractCollectionPrivate GeeAbstractCollectionPrivate;

#define GEE_TYPE_LIST (gee_list_get_type ())
#define GEE_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_LIST, GeeList))
#define GEE_IS_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_LIST))
#define GEE_LIST_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_LIST, GeeListIface))

typedef struct _GeeList GeeList;
typedef struct _GeeListIface GeeListIface;

#define GEE_TYPE_ABSTRACT_LIST (gee_abstract_list_get_type ())
#define GEE_ABSTRACT_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ABSTRACT_LIST, GeeAbstractList))
#define GEE_ABSTRACT_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_ABSTRACT_LIST, GeeAbstractListClass))
#define GEE_IS_ABSTRACT_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ABSTRACT_LIST))
#define GEE_IS_ABSTRACT_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_ABSTRACT_LIST))
#define GEE_ABSTRACT_LIST_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_ABSTRACT_LIST, GeeAbstractListClass))

typedef struct _GeeAbstractList GeeAbstractList;
typedef struct _GeeAbstractListClass GeeAbstractListClass;
typedef struct _GeeAbstractListPrivate GeeAbstractListPrivate;

#define GEE_TYPE_ARRAY_LIST (gee_array_list_get_type ())
#define GEE_ARRAY_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ARRAY_LIST, GeeArrayList))
#define GEE_ARRAY_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_ARRAY_LIST, GeeArrayListClass))
#define GEE_IS_ARRAY_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ARRAY_LIST))
#define GEE_IS_ARRAY_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_ARRAY_LIST))
#define GEE_ARRAY_LIST_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_ARRAY_LIST, GeeArrayListClass))

typedef struct _GeeArrayList GeeArrayList;
typedef struct _GeeArrayListClass GeeArrayListClass;
typedef struct _GeeArrayListPrivate GeeArrayListPrivate;

#define GEE_ARRAY_LIST_TYPE_ITERATOR (gee_array_list_iterator_get_type ())
#define GEE_ARRAY_LIST_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_ARRAY_LIST_TYPE_ITERATOR, GeeArrayListIterator))
#define GEE_ARRAY_LIST_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_ARRAY_LIST_TYPE_ITERATOR, GeeArrayListIteratorClass))
#define GEE_ARRAY_LIST_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_ARRAY_LIST_TYPE_ITERATOR))
#define GEE_ARRAY_LIST_IS_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_ARRAY_LIST_TYPE_ITERATOR))
#define GEE_ARRAY_LIST_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_ARRAY_LIST_TYPE_ITERATOR, GeeArrayListIteratorClass))

typedef struct _GeeArrayListIterator GeeArrayListIterator;
typedef struct _GeeArrayListIteratorClass GeeArrayListIteratorClass;
typedef struct _GeeArrayListIteratorPrivate GeeArrayListIteratorPrivate;

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIteratorIface {
	GTypeInterface parent_iface;
	gboolean (*next) (GeeIterator* self);
	gpointer (*get) (GeeIterator* self);
};

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIterableIface {
	GTypeInterface parent_iface;
	GeeIterator* (*iterator) (GeeIterable* self);
	GType (*get_element_type) (GeeIterable* self);
};

/**
 * Serves as the base interface for implementing collection classes. Defines
 * size, iteration, and modification methods.
 */
struct _GeeCollectionIface {
	GTypeInterface parent_iface;
	gboolean (*contains) (GeeCollection* self, gconstpointer item);
	gboolean (*add) (GeeCollection* self, gconstpointer item);
	gboolean (*remove) (GeeCollection* self, gconstpointer item);
	void (*clear) (GeeCollection* self);
	gboolean (*add_all) (GeeCollection* self, GeeCollection* collection);
	gboolean (*contains_all) (GeeCollection* self, GeeCollection* collection);
	gboolean (*remove_all) (GeeCollection* self, GeeCollection* collection);
	gboolean (*retain_all) (GeeCollection* self, GeeCollection* collection);
	gpointer* (*to_array) (GeeCollection* self, int* result_length1);
	gint (*get_size) (GeeCollection* self);
	gboolean (*get_is_empty) (GeeCollection* self);
};

/**
 * Serves as the base class for implementing collection classes.
 */
struct _GeeAbstractCollection {
	GObject parent_instance;
	GeeAbstractCollectionPrivate * priv;
};

struct _GeeAbstractCollectionClass {
	GObjectClass parent_class;
	gboolean (*contains) (GeeAbstractCollection* self, gconstpointer item);
	gboolean (*add) (GeeAbstractCollection* self, gconstpointer item);
	gboolean (*remove) (GeeAbstractCollection* self, gconstpointer item);
	void (*clear) (GeeAbstractCollection* self);
	gpointer* (*to_array) (GeeAbstractCollection* self, int* result_length1);
	gboolean (*add_all) (GeeAbstractCollection* self, GeeCollection* collection);
	gboolean (*contains_all) (GeeAbstractCollection* self, GeeCollection* collection);
	gboolean (*remove_all) (GeeAbstractCollection* self, GeeCollection* collection);
	gboolean (*retain_all) (GeeAbstractCollection* self, GeeCollection* collection);
	GeeIterator* (*iterator) (GeeAbstractCollection* self);
	gint (*get_size) (GeeAbstractCollection* self);
	gboolean (*get_is_empty) (GeeAbstractCollection* self);
};

/**
 * Represents a collection of items in a well-defined order.
 */
struct _GeeListIface {
	GTypeInterface parent_iface;
	gpointer (*get) (GeeList* self, gint index);
	void (*set) (GeeList* self, gint index, gconstpointer item);
	gint (*index_of) (GeeList* self, gconstpointer item);
	void (*insert) (GeeList* self, gint index, gconstpointer item);
	void (*remove_at) (GeeList* self, gint index);
	GeeList* (*slice) (GeeList* self, gint start, gint stop);
	gpointer (*first) (GeeList* self);
	gpointer (*last) (GeeList* self);
	void (*insert_all) (GeeList* self, gint index, GeeCollection* collection);
};

/**
 * Serves as the base class for implementing list classes.
 */
struct _GeeAbstractList {
	GeeAbstractCollection parent_instance;
	GeeAbstractListPrivate * priv;
};

struct _GeeAbstractListClass {
	GeeAbstractCollectionClass parent_class;
	gpointer (*get) (GeeAbstractList* self, gint index);
	void (*set) (GeeAbstractList* self, gint index, gconstpointer item);
	gint (*index_of) (GeeAbstractList* self, gconstpointer item);
	void (*insert) (GeeAbstractList* self, gint index, gconstpointer item);
	void (*remove_at) (GeeAbstractList* self, gint index);
	GeeList* (*slice) (GeeAbstractList* self, gint start, gint stop);
	gpointer (*first) (GeeAbstractList* self);
	gpointer (*last) (GeeAbstractList* self);
	void (*insert_all) (GeeAbstractList* self, gint index, GeeCollection* collection);
};

/**
 * Arrays of arbitrary elements which grow automatically as elements are added.
 */
struct _GeeArrayList {
	GeeAbstractList parent_instance;
	GeeArrayListPrivate * priv;
};

struct _GeeArrayListClass {
	GeeAbstractListClass parent_class;
};

struct _GeeArrayListPrivate {
	GType g_type;
	GBoxedCopyFunc g_dup_func;
	GDestroyNotify g_destroy_func;
	GEqualFunc _equal_func;
	gpointer* _items;
	gint _items_length1;
	gint _items_size;
	gint _size;
	gint _stamp;
};

struct _GeeArrayListIterator {
	GObject parent_instance;
	GeeArrayListIteratorPrivate * priv;
};

struct _GeeArrayListIteratorClass {
	GObjectClass parent_class;
};

struct _GeeArrayListIteratorPrivate {
	GType g_type;
	GBoxedCopyFunc g_dup_func;
	GDestroyNotify g_destroy_func;
	GeeArrayList* _list;
	gint _index;
	gint _stamp;
};


static gpointer gee_array_list_iterator_parent_class = NULL;
static GeeIteratorIface* gee_array_list_iterator_gee_iterator_parent_iface = NULL;
static gpointer gee_array_list_parent_class = NULL;

GType gee_iterator_get_type (void);
GType gee_iterable_get_type (void);
GType gee_collection_get_type (void);
GType gee_abstract_collection_get_type (void);
GType gee_list_get_type (void);
GType gee_abstract_list_get_type (void);
GType gee_array_list_get_type (void);
#define GEE_ARRAY_LIST_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TYPE_ARRAY_LIST, GeeArrayListPrivate))
enum  {
	GEE_ARRAY_LIST_DUMMY_PROPERTY,
	GEE_ARRAY_LIST_SIZE,
	GEE_ARRAY_LIST_EQUAL_FUNC,
	GEE_ARRAY_LIST_G_TYPE,
	GEE_ARRAY_LIST_G_DUP_FUNC,
	GEE_ARRAY_LIST_G_DESTROY_FUNC
};
static void gee_array_list_set_equal_func (GeeArrayList* self, GEqualFunc value);
GeeArrayList* gee_array_list_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GEqualFunc equal_func);
GeeArrayList* gee_array_list_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GEqualFunc equal_func);
static GeeArrayListIterator* gee_array_list_iterator_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeArrayList* list);
static GeeArrayListIterator* gee_array_list_iterator_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeArrayList* list);
static GType gee_array_list_iterator_get_type (void);
static GeeIterator* gee_array_list_real_iterator (GeeAbstractCollection* base);
gint gee_abstract_list_index_of (GeeAbstractList* self, gconstpointer item);
static gboolean gee_array_list_real_contains (GeeAbstractCollection* base, gconstpointer item);
GEqualFunc gee_array_list_get_equal_func (GeeArrayList* self);
static gint gee_array_list_real_index_of (GeeAbstractList* base, gconstpointer item);
static gpointer gee_array_list_real_get (GeeAbstractList* base, gint index);
static void gee_array_list_real_set (GeeAbstractList* base, gint index, gconstpointer item);
static void gee_array_list_grow_if_needed (GeeArrayList* self, gint new_count);
static gboolean gee_array_list_real_add (GeeAbstractCollection* base, gconstpointer item);
static void gee_array_list_shift (GeeArrayList* self, gint start, gint delta);
static void gee_array_list_real_insert (GeeAbstractList* base, gint index, gconstpointer item);
void gee_abstract_list_remove_at (GeeAbstractList* self, gint index);
static gboolean gee_array_list_real_remove (GeeAbstractCollection* base, gconstpointer item);
static void gee_array_list_real_remove_at (GeeAbstractList* base, gint index);
static void gee_array_list_real_clear (GeeAbstractCollection* base);
gboolean gee_abstract_collection_add (GeeAbstractCollection* self, gconstpointer item);
gpointer gee_abstract_list_get (GeeAbstractList* self, gint index);
static GeeList* gee_array_list_real_slice (GeeAbstractList* base, gint start, gint stop);
gboolean gee_collection_get_is_empty (GeeCollection* self);
gint gee_collection_get_size (GeeCollection* self);
GeeIterator* gee_iterable_iterator (GeeIterable* self);
gboolean gee_iterator_next (GeeIterator* self);
gpointer gee_iterator_get (GeeIterator* self);
static gboolean gee_array_list_real_add_all (GeeAbstractCollection* base, GeeCollection* collection);
gboolean gee_collection_contains (GeeCollection* self, gconstpointer item);
static gboolean gee_array_list_real_remove_all (GeeAbstractCollection* base, GeeCollection* collection);
static gboolean gee_array_list_real_retain_all (GeeAbstractCollection* base, GeeCollection* collection);
static void gee_array_list_set_capacity (GeeArrayList* self, gint value);
#define GEE_ARRAY_LIST_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_ARRAY_LIST_TYPE_ITERATOR, GeeArrayListIteratorPrivate))
enum  {
	GEE_ARRAY_LIST_ITERATOR_DUMMY_PROPERTY,
	GEE_ARRAY_LIST_ITERATOR_LIST,
	GEE_ARRAY_LIST_ITERATOR_G_TYPE,
	GEE_ARRAY_LIST_ITERATOR_G_DUP_FUNC,
	GEE_ARRAY_LIST_ITERATOR_G_DESTROY_FUNC
};
static void gee_array_list_iterator_set_list (GeeArrayListIterator* self, GeeArrayList* value);
static gboolean gee_array_list_iterator_real_next (GeeIterator* base);
static gpointer gee_array_list_iterator_real_get (GeeIterator* base);
static void gee_array_list_iterator_finalize (GObject* obj);
static void gee_array_list_iterator_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void gee_array_list_iterator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void gee_array_list_finalize (GObject* obj);
static void gee_array_list_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void gee_array_list_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_move (gpointer array, gsize element_size, gint src, gint dest, gint length);



GeeArrayList* gee_array_list_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GEqualFunc equal_func) {
	GParameter * __params;
	GParameter * __params_it;
	GeeArrayList * self;
	__params = g_new0 (GParameter, 4);
	__params_it = __params;
	__params_it->name = "g-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, g_type);
	__params_it++;
	__params_it->name = "g-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, g_dup_func);
	__params_it++;
	__params_it->name = "g-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, g_destroy_func);
	__params_it++;
	__params_it->name = "equal-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, equal_func);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


GeeArrayList* gee_array_list_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GEqualFunc equal_func) {
	return gee_array_list_construct (GEE_TYPE_ARRAY_LIST, g_type, g_dup_func, g_destroy_func, equal_func);
}


static GeeIterator* gee_array_list_real_iterator (GeeAbstractCollection* base) {
	GeeArrayList * self;
	GeeIterator* result;
	self = (GeeArrayList*) base;
	result = (GeeIterator*) gee_array_list_iterator_new (self->priv->g_type, (GBoxedCopyFunc) self->priv->g_dup_func, self->priv->g_destroy_func, self);
	return result;
}


static gboolean gee_array_list_real_contains (GeeAbstractCollection* base, gconstpointer item) {
	GeeArrayList * self;
	gboolean result;
	self = (GeeArrayList*) base;
	result = gee_abstract_list_index_of ((GeeAbstractList*) self, item) != (-1);
	return result;
}


static gint gee_array_list_real_index_of (GeeAbstractList* base, gconstpointer item) {
	GeeArrayList * self;
	gint result;
	self = (GeeArrayList*) base;
	{
		gint index;
		index = 0;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				if (!_tmp0_) {
					index++;
				}
				_tmp0_ = FALSE;
				if (!(index < self->priv->_size)) {
					break;
				}
				if (gee_array_list_get_equal_func (self) (self->priv->_items[index], item)) {
					result = index;
					return result;
				}
			}
		}
	}
	result = -1;
	return result;
}


static gpointer gee_array_list_real_get (GeeAbstractList* base, gint index) {
	GeeArrayList * self;
	gpointer result;
	gconstpointer _tmp0_;
	self = (GeeArrayList*) base;
	g_assert (index >= 0);
	g_assert (index < self->priv->_size);
	_tmp0_ = NULL;
	result = (_tmp0_ = self->priv->_items[index], ((_tmp0_ == NULL) || (self->priv->g_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->g_dup_func ((gpointer) _tmp0_));
	return result;
}


static void gee_array_list_real_set (GeeAbstractList* base, gint index, gconstpointer item) {
	GeeArrayList * self;
	gpointer _tmp1_;
	gconstpointer _tmp0_;
	self = (GeeArrayList*) base;
	g_assert (index >= 0);
	g_assert (index < self->priv->_size);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_items[index] = (_tmp1_ = (_tmp0_ = item, ((_tmp0_ == NULL) || (self->priv->g_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->g_dup_func ((gpointer) _tmp0_)), ((self->priv->_items[index] == NULL) || (self->priv->g_destroy_func == NULL)) ? NULL : (self->priv->_items[index] = (self->priv->g_destroy_func (self->priv->_items[index]), NULL)), _tmp1_);
}


static gboolean gee_array_list_real_add (GeeAbstractCollection* base, gconstpointer item) {
	GeeArrayList * self;
	gboolean result;
	gpointer _tmp2_;
	gpointer *_tmp1_;
	gconstpointer _tmp0_;
	self = (GeeArrayList*) base;
	if (self->priv->_size == self->priv->_items_length1) {
		gee_array_list_grow_if_needed (self, 1);
	}
	_tmp2_ = NULL;
	_tmp0_ = NULL;
	_tmp1_ = &self->priv->_items[self->priv->_size++];
	(*_tmp1_) = (_tmp2_ = (_tmp0_ = item, ((_tmp0_ == NULL) || (self->priv->g_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->g_dup_func ((gpointer) _tmp0_)), (((*_tmp1_) == NULL) || (self->priv->g_destroy_func == NULL)) ? NULL : ((*_tmp1_) = (self->priv->g_destroy_func ((*_tmp1_)), NULL)), _tmp2_);
	self->priv->_stamp++;
	result = TRUE;
	return result;
}


static void gee_array_list_real_insert (GeeAbstractList* base, gint index, gconstpointer item) {
	GeeArrayList * self;
	gpointer _tmp1_;
	gconstpointer _tmp0_;
	self = (GeeArrayList*) base;
	g_assert (index >= 0);
	g_assert (index <= self->priv->_size);
	if (self->priv->_size == self->priv->_items_length1) {
		gee_array_list_grow_if_needed (self, 1);
	}
	gee_array_list_shift (self, index, 1);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_items[index] = (_tmp1_ = (_tmp0_ = item, ((_tmp0_ == NULL) || (self->priv->g_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->g_dup_func ((gpointer) _tmp0_)), ((self->priv->_items[index] == NULL) || (self->priv->g_destroy_func == NULL)) ? NULL : (self->priv->_items[index] = (self->priv->g_destroy_func (self->priv->_items[index]), NULL)), _tmp1_);
	self->priv->_stamp++;
}


static gboolean gee_array_list_real_remove (GeeAbstractCollection* base, gconstpointer item) {
	GeeArrayList * self;
	gboolean result;
	self = (GeeArrayList*) base;
	{
		gint index;
		index = 0;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				if (!_tmp0_) {
					index++;
				}
				_tmp0_ = FALSE;
				if (!(index < self->priv->_size)) {
					break;
				}
				if (gee_array_list_get_equal_func (self) (self->priv->_items[index], item)) {
					gee_abstract_list_remove_at ((GeeAbstractList*) self, index);
					result = TRUE;
					return result;
				}
			}
		}
	}
	result = FALSE;
	return result;
}


static void gee_array_list_real_remove_at (GeeAbstractList* base, gint index) {
	GeeArrayList * self;
	gpointer _tmp0_;
	self = (GeeArrayList*) base;
	g_assert (index >= 0);
	g_assert (index < self->priv->_size);
	_tmp0_ = NULL;
	self->priv->_items[index] = (_tmp0_ = NULL, ((self->priv->_items[index] == NULL) || (self->priv->g_destroy_func == NULL)) ? NULL : (self->priv->_items[index] = (self->priv->g_destroy_func (self->priv->_items[index]), NULL)), _tmp0_);
	gee_array_list_shift (self, index + 1, -1);
	self->priv->_stamp++;
}


static void gee_array_list_real_clear (GeeAbstractCollection* base) {
	GeeArrayList * self;
	self = (GeeArrayList*) base;
	{
		gint index;
		index = 0;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				gpointer _tmp1_;
				if (!_tmp0_) {
					index++;
				}
				_tmp0_ = FALSE;
				if (!(index < self->priv->_size)) {
					break;
				}
				_tmp1_ = NULL;
				self->priv->_items[index] = (_tmp1_ = NULL, ((self->priv->_items[index] == NULL) || (self->priv->g_destroy_func == NULL)) ? NULL : (self->priv->_items[index] = (self->priv->g_destroy_func (self->priv->_items[index]), NULL)), _tmp1_);
			}
		}
	}
	self->priv->_size = 0;
	self->priv->_stamp++;
}


static GeeList* gee_array_list_real_slice (GeeAbstractList* base, gint start, gint stop) {
	GeeArrayList * self;
	GeeList* result;
	GeeArrayList* slice;
	self = (GeeArrayList*) base;
	g_return_val_if_fail (start <= stop, NULL);
	g_return_val_if_fail (start >= 0, NULL);
	g_return_val_if_fail (stop <= self->priv->_size, NULL);
	slice = gee_array_list_new (self->priv->g_type, (GBoxedCopyFunc) self->priv->g_dup_func, self->priv->g_destroy_func, gee_array_list_get_equal_func (self));
	{
		gint i;
		i = start;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				gpointer _tmp1_;
				if (!_tmp0_) {
					i++;
				}
				_tmp0_ = FALSE;
				if (!(i < stop)) {
					break;
				}
				_tmp1_ = NULL;
				gee_abstract_collection_add ((GeeAbstractCollection*) slice, _tmp1_ = gee_abstract_list_get ((GeeAbstractList*) self, i));
				((_tmp1_ == NULL) || (self->priv->g_destroy_func == NULL)) ? NULL : (_tmp1_ = (self->priv->g_destroy_func (_tmp1_), NULL));
			}
		}
	}
	result = (GeeList*) slice;
	return result;
}


static gboolean gee_array_list_real_add_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeArrayList * self;
	gboolean result;
	self = (GeeArrayList*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	if (gee_collection_get_is_empty (collection)) {
		result = FALSE;
		return result;
	}
	gee_array_list_grow_if_needed (self, gee_collection_get_size (collection));
	{
		GeeIterator* _item_it;
		_item_it = gee_iterable_iterator ((GeeIterable*) collection);
		while (TRUE) {
			gpointer item;
			gpointer _tmp2_;
			gpointer *_tmp1_;
			gconstpointer _tmp0_;
			if (!gee_iterator_next (_item_it)) {
				break;
			}
			item = gee_iterator_get (_item_it);
			_tmp2_ = NULL;
			_tmp0_ = NULL;
			_tmp1_ = &self->priv->_items[self->priv->_size++];
			(*_tmp1_) = (_tmp2_ = (_tmp0_ = item, ((_tmp0_ == NULL) || (self->priv->g_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->g_dup_func ((gpointer) _tmp0_)), (((*_tmp1_) == NULL) || (self->priv->g_destroy_func == NULL)) ? NULL : ((*_tmp1_) = (self->priv->g_destroy_func ((*_tmp1_)), NULL)), _tmp2_);
			((item == NULL) || (self->priv->g_destroy_func == NULL)) ? NULL : (item = (self->priv->g_destroy_func (item), NULL));
		}
		(_item_it == NULL) ? NULL : (_item_it = (g_object_unref (_item_it), NULL));
	}
	self->priv->_stamp++;
	result = TRUE;
	return result;
}


static gboolean gee_array_list_real_remove_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeArrayList * self;
	gboolean result;
	gboolean changed;
	self = (GeeArrayList*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	changed = FALSE;
	{
		gint index;
		index = 0;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				if (!_tmp0_) {
					index++;
				}
				_tmp0_ = FALSE;
				if (!(index < self->priv->_size)) {
					break;
				}
				if (gee_collection_contains (collection, self->priv->_items[index])) {
					gee_abstract_list_remove_at ((GeeAbstractList*) self, index);
					index--;
					changed = TRUE;
				}
			}
		}
	}
	result = changed;
	return result;
}


static gboolean gee_array_list_real_retain_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeArrayList * self;
	gboolean result;
	gboolean changed;
	self = (GeeArrayList*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	changed = FALSE;
	{
		gint index;
		index = 0;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				if (!_tmp0_) {
					index++;
				}
				_tmp0_ = FALSE;
				if (!(index < self->priv->_size)) {
					break;
				}
				if (!gee_collection_contains (collection, self->priv->_items[index])) {
					gee_abstract_list_remove_at ((GeeAbstractList*) self, index);
					index--;
					changed = TRUE;
				}
			}
		}
	}
	result = changed;
	return result;
}


static void gee_array_list_shift (GeeArrayList* self, gint start, gint delta) {
	g_return_if_fail (self != NULL);
	g_assert (start >= 0);
	g_assert (start <= self->priv->_size);
	g_assert (start >= (-delta));
	_vala_array_move (self->priv->_items, sizeof (gpointer), start, start + delta, self->priv->_size - start);
	self->priv->_size = self->priv->_size + delta;
}


static void gee_array_list_grow_if_needed (GeeArrayList* self, gint new_count) {
	gint minimum_size;
	g_return_if_fail (self != NULL);
	g_assert (new_count >= 0);
	minimum_size = self->priv->_size + new_count;
	if (minimum_size > self->priv->_items_length1) {
		gint _tmp0_;
		_tmp0_ = 0;
		if (new_count > self->priv->_items_length1) {
			_tmp0_ = minimum_size;
		} else {
			_tmp0_ = 2 * self->priv->_items_length1;
		}
		/* double the capacity unless we add even more items at this time*/
		gee_array_list_set_capacity (self, _tmp0_);
	}
}


static void gee_array_list_set_capacity (GeeArrayList* self, gint value) {
	gint _tmp0_;
	g_return_if_fail (self != NULL);
	g_assert (value >= self->priv->_size);
	_tmp0_ = value;
	self->priv->_items = g_renew (gpointer, self->priv->_items, value);
	(_tmp0_ > self->priv->_items_length1) ? memset (self->priv->_items + self->priv->_items_length1, 0, sizeof (gpointer) * (_tmp0_ - self->priv->_items_length1)) : NULL;
	self->priv->_items_length1 = _tmp0_;
}


static gint gee_array_list_real_get_size (GeeAbstractCollection* base) {
	gint result;
	GeeArrayList* self;
	self = (GeeArrayList*) base;
	result = self->priv->_size;
	return result;
}


GEqualFunc gee_array_list_get_equal_func (GeeArrayList* self) {
	GEqualFunc result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_equal_func;
	return result;
}


static void gee_array_list_set_equal_func (GeeArrayList* self, GEqualFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_equal_func = value;
	g_object_notify ((GObject *) self, "equal-func");
}


static GeeArrayListIterator* gee_array_list_iterator_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeArrayList* list) {
	GParameter * __params;
	GParameter * __params_it;
	GeeArrayListIterator * self;
	g_return_val_if_fail (list != NULL, NULL);
	__params = g_new0 (GParameter, 4);
	__params_it = __params;
	__params_it->name = "g-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, g_type);
	__params_it++;
	__params_it->name = "g-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, g_dup_func);
	__params_it++;
	__params_it->name = "g-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, g_destroy_func);
	__params_it++;
	__params_it->name = "list";
	g_value_init (&__params_it->value, GEE_TYPE_ARRAY_LIST);
	g_value_set_object (&__params_it->value, list);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


static GeeArrayListIterator* gee_array_list_iterator_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeArrayList* list) {
	return gee_array_list_iterator_construct (GEE_ARRAY_LIST_TYPE_ITERATOR, g_type, g_dup_func, g_destroy_func, list);
}


static gboolean gee_array_list_iterator_real_next (GeeIterator* base) {
	GeeArrayListIterator * self;
	gboolean result;
	self = (GeeArrayListIterator*) base;
	g_assert (self->priv->_stamp == self->priv->_list->priv->_stamp);
	if (self->priv->_index < self->priv->_list->priv->_size) {
		self->priv->_index++;
	}
	result = self->priv->_index < self->priv->_list->priv->_size;
	return result;
}


static gpointer gee_array_list_iterator_real_get (GeeIterator* base) {
	GeeArrayListIterator * self;
	gpointer result;
	gboolean _tmp0_;
	self = (GeeArrayListIterator*) base;
	g_assert (self->priv->_stamp == self->priv->_list->priv->_stamp);
	_tmp0_ = FALSE;
	if (self->priv->_index < 0) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = self->priv->_index >= self->priv->_list->priv->_size;
	}
	if (_tmp0_) {
		result = NULL;
		return result;
	}
	result = gee_abstract_list_get ((GeeAbstractList*) self->priv->_list, self->priv->_index);
	return result;
}


static void gee_array_list_iterator_set_list (GeeArrayListIterator* self, GeeArrayList* value) {
	GeeArrayList* _tmp1_;
	GeeArrayList* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_list = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_list == NULL) ? NULL : (self->priv->_list = (g_object_unref (self->priv->_list), NULL)), _tmp1_);
	self->priv->_stamp = self->priv->_list->priv->_stamp;
	g_object_notify ((GObject *) self, "list");
}


static void gee_array_list_iterator_class_init (GeeArrayListIteratorClass * klass) {
	gee_array_list_iterator_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeArrayListIteratorPrivate));
	G_OBJECT_CLASS (klass)->get_property = gee_array_list_iterator_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_array_list_iterator_set_property;
	G_OBJECT_CLASS (klass)->finalize = gee_array_list_iterator_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_ITERATOR_G_TYPE, g_param_spec_gtype ("g-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_ITERATOR_G_DUP_FUNC, g_param_spec_pointer ("g-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_ITERATOR_G_DESTROY_FUNC, g_param_spec_pointer ("g-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_ITERATOR_LIST, g_param_spec_object ("list", "list", "list", GEE_TYPE_ARRAY_LIST, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


static void gee_array_list_iterator_gee_iterator_interface_init (GeeIteratorIface * iface) {
	gee_array_list_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_array_list_iterator_real_next;
	iface->get = gee_array_list_iterator_real_get;
}


static void gee_array_list_iterator_instance_init (GeeArrayListIterator * self) {
	self->priv = GEE_ARRAY_LIST_ITERATOR_GET_PRIVATE (self);
	self->priv->_index = -1;
	self->priv->_stamp = 0;
}


static void gee_array_list_iterator_finalize (GObject* obj) {
	GeeArrayListIterator * self;
	self = GEE_ARRAY_LIST_ITERATOR (obj);
	(self->priv->_list == NULL) ? NULL : (self->priv->_list = (g_object_unref (self->priv->_list), NULL));
	G_OBJECT_CLASS (gee_array_list_iterator_parent_class)->finalize (obj);
}


static GType gee_array_list_iterator_get_type (void) {
	static GType gee_array_list_iterator_type_id = 0;
	if (gee_array_list_iterator_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeArrayListIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_array_list_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeArrayListIterator), 0, (GInstanceInitFunc) gee_array_list_iterator_instance_init, NULL };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_array_list_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_array_list_iterator_type_id = g_type_register_static (G_TYPE_OBJECT, "GeeArrayListIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_array_list_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_array_list_iterator_type_id;
}


static void gee_array_list_iterator_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GeeArrayListIterator * self;
	gpointer boxed;
	self = GEE_ARRAY_LIST_ITERATOR (object);
	switch (property_id) {
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_array_list_iterator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GeeArrayListIterator * self;
	self = GEE_ARRAY_LIST_ITERATOR (object);
	switch (property_id) {
		case GEE_ARRAY_LIST_ITERATOR_LIST:
		gee_array_list_iterator_set_list (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
		case GEE_ARRAY_LIST_ITERATOR_G_TYPE:
		self->priv->g_type = g_value_get_gtype (value);
		break;
		case GEE_ARRAY_LIST_ITERATOR_G_DUP_FUNC:
		self->priv->g_dup_func = g_value_get_pointer (value);
		break;
		case GEE_ARRAY_LIST_ITERATOR_G_DESTROY_FUNC:
		self->priv->g_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static void gee_array_list_class_init (GeeArrayListClass * klass) {
	gee_array_list_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeArrayListPrivate));
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->iterator = gee_array_list_real_iterator;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->contains = gee_array_list_real_contains;
	GEE_ABSTRACT_LIST_CLASS (klass)->index_of = gee_array_list_real_index_of;
	GEE_ABSTRACT_LIST_CLASS (klass)->get = gee_array_list_real_get;
	GEE_ABSTRACT_LIST_CLASS (klass)->set = gee_array_list_real_set;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->add = gee_array_list_real_add;
	GEE_ABSTRACT_LIST_CLASS (klass)->insert = gee_array_list_real_insert;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->remove = gee_array_list_real_remove;
	GEE_ABSTRACT_LIST_CLASS (klass)->remove_at = gee_array_list_real_remove_at;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->clear = gee_array_list_real_clear;
	GEE_ABSTRACT_LIST_CLASS (klass)->slice = gee_array_list_real_slice;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->add_all = gee_array_list_real_add_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->remove_all = gee_array_list_real_remove_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->retain_all = gee_array_list_real_retain_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->get_size = gee_array_list_real_get_size;
	G_OBJECT_CLASS (klass)->get_property = gee_array_list_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_array_list_set_property;
	G_OBJECT_CLASS (klass)->finalize = gee_array_list_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_G_TYPE, g_param_spec_gtype ("g-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_G_DUP_FUNC, g_param_spec_pointer ("g-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_G_DESTROY_FUNC, g_param_spec_pointer ("g-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_override_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_SIZE, "size");
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_EQUAL_FUNC, g_param_spec_pointer ("equal-func", "equal-func", "equal-func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


static void gee_array_list_instance_init (GeeArrayList * self) {
	self->priv = GEE_ARRAY_LIST_GET_PRIVATE (self);
	self->priv->_items = g_new0 (gpointer, 4);
	self->priv->_items_length1 = 4;
	self->priv->_stamp = 0;
}


static void gee_array_list_finalize (GObject* obj) {
	GeeArrayList * self;
	self = GEE_ARRAY_LIST (obj);
	self->priv->_items = (_vala_array_free (self->priv->_items, self->priv->_items_length1, (GDestroyNotify) self->priv->g_destroy_func), NULL);
	G_OBJECT_CLASS (gee_array_list_parent_class)->finalize (obj);
}


GType gee_array_list_get_type (void) {
	static GType gee_array_list_type_id = 0;
	if (gee_array_list_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeArrayListClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_array_list_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeArrayList), 0, (GInstanceInitFunc) gee_array_list_instance_init, NULL };
		gee_array_list_type_id = g_type_register_static (GEE_TYPE_ABSTRACT_LIST, "GeeArrayList", &g_define_type_info, 0);
	}
	return gee_array_list_type_id;
}


static void gee_array_list_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GeeArrayList * self;
	gpointer boxed;
	self = GEE_ARRAY_LIST (object);
	switch (property_id) {
		case GEE_ARRAY_LIST_SIZE:
		g_value_set_int (value, gee_abstract_collection_get_size ((GeeAbstractCollection*) self));
		break;
		case GEE_ARRAY_LIST_EQUAL_FUNC:
		g_value_set_pointer (value, gee_array_list_get_equal_func (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_array_list_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GeeArrayList * self;
	self = GEE_ARRAY_LIST (object);
	switch (property_id) {
		case GEE_ARRAY_LIST_EQUAL_FUNC:
		gee_array_list_set_equal_func (self, g_value_get_pointer (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
		case GEE_ARRAY_LIST_G_TYPE:
		self->priv->g_type = g_value_get_gtype (value);
		break;
		case GEE_ARRAY_LIST_G_DUP_FUNC:
		self->priv->g_dup_func = g_value_get_pointer (value);
		break;
		case GEE_ARRAY_LIST_G_DESTROY_FUNC:
		self->priv->g_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static void _vala_array_move (gpointer array, gsize element_size, gint src, gint dest, gint length) {
	g_memmove (((char*) array) + (dest * element_size), ((char*) array) + (src * element_size), length * element_size);
	if (src < dest) {
		memset (((char*) array) + (src * element_size), 0, (dest - src) * element_size);
	} else {
		memset (((char*) array) + ((dest + length) * element_size), 0, (src - dest) * element_size);
	}
}




