/* 
 * GNetwork Library: libgnetwork/gnetwork-interfaces.c
 *
 * Copyright (C) 2003 James M. Cape
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; version 2.1 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */

#include "gnetwork-interfaces.h"

#include "gnetwork-ip-address.h"
#include "gnetwork-utils.h"
#include "gnetwork-type-builtins.h"

#include <sys/types.h>

#include <unistd.h>

/* Interfaces & Addresses */
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>

#ifdef __linux__
# include <netpacket/packet.h>
# include <netinet/ether.h>
#else
# include <net/ethernet.h>
#endif

#include <ifaddrs.h>
#include <arpa/inet.h>

#include <stdio.h>
#include <string.h>


/* *************** *
 *  Private Types  *
 * *************** */

struct _GNetworkInterfaceInfo
{
  GTypeClass g_class;

  gint ref;

  guint64 index_;
  gchar *name;

  GNetworkIpAddress ip4_address;
  GNetworkIpAddress ip4_netmask;
  GNetworkIpAddress ip4_broadcast_or_destination;
  GSList *ip4_multicasts;

  GNetworkIpAddress ip6_address;
  GNetworkIpAddress ip6_netmask;
  GNetworkIpAddress ip6_destination;
  GSList *ip6_multicasts;

  gchar *hw_address;
  gchar *hw_broadcast_or_destination;

  GNetworkInterfaceFlags flags:16;
  GNetworkProtocols protocols:3;
};


/* ******************* *
 *  Utility Functions  *
 * ******************* */

static void
append_iface_info_from_interface (GNetworkInterfaceInfo * info, struct ifaddrs *data)
{
  GNetworkIpAddress addr = GNETWORK_IP_ADDRESS_INIT;

  switch (data->ifa_addr->sa_family)
    {
    case AF_INET:
      info->protocols |= GNETWORK_PROTOCOL_IPv4;

      _gnetwork_ip_address_set_from_sockaddr (&addr, data->ifa_addr);

      if (gnetwork_ip_address_is_multicast (&addr))
	{
	  info->ip4_multicasts = g_slist_prepend (info->ip4_multicasts,
						  gnetwork_ip_address_dup (&addr));
	}
      else
	{
	  memcpy (&(info->ip4_address), &addr, sizeof (GNetworkIpAddress));
	}

      _gnetwork_ip_address_set_from_sockaddr (&(info->ip4_address), data->ifa_addr);
      _gnetwork_ip_address_set_from_sockaddr (&(info->ip4_netmask), data->ifa_netmask);
      _gnetwork_ip_address_set_from_sockaddr (&(info->ip4_broadcast_or_destination),
					     data->ifa_dstaddr);
      break;
    case AF_INET6:
      info->protocols |= GNETWORK_PROTOCOL_IPv6;

      _gnetwork_ip_address_set_from_sockaddr (&addr, data->ifa_addr);

      if (gnetwork_ip_address_is_multicast (&addr))
	{
	  info->ip6_multicasts = g_slist_prepend (info->ip6_multicasts,
						  gnetwork_ip_address_dup (&addr));
	}
      else
	{
	  memcpy (&(info->ip6_address), &addr, sizeof (GNetworkIpAddress));
	}

      _gnetwork_ip_address_set_from_sockaddr (&(info->ip6_netmask), data->ifa_netmask);
      _gnetwork_ip_address_set_from_sockaddr (&(info->ip6_destination), data->ifa_dstaddr);
      break;
#ifdef AF_PACKET
    case AF_PACKET:
      info->protocols |= GNETWORK_PROTOCOL_PACKET;

      info->index_ = ((struct sockaddr_ll *) (data->ifa_addr))->sll_ifindex;

      info->hw_address = _gnetwork_sockaddr_get_address (data->ifa_addr);
      info->hw_broadcast_or_destination = _gnetwork_sockaddr_get_address (data->ifa_dstaddr);
      break;
#endif
    }
}


static GNetworkInterfaceInfo *
create_info_from_interface (struct ifaddrs *data)
{
  GNetworkInterfaceInfo *info = g_new0 (GNetworkInterfaceInfo, 1);

  info->g_class.g_type = GNETWORK_TYPE_INTERFACE_INFO;
  info->ref = 1;

  info->name = g_strdup (data->ifa_name);
  info->ip4_multicasts = NULL;
  info->ip6_multicasts = NULL;

  info->flags = GNETWORK_INTERFACE_NONE;

  /* Flags */
  if (data->ifa_flags & IFF_UP)
    info->flags |= GNETWORK_INTERFACE_IS_UP;

  if (data->ifa_flags & IFF_RUNNING)
    info->flags |= GNETWORK_INTERFACE_IS_RUNNING;

  if (data->ifa_flags & IFF_DEBUG)
    info->flags |= GNETWORK_INTERFACE_IS_DEBUGGING;

  if (data->ifa_flags & IFF_LOOPBACK)
    info->flags |= GNETWORK_INTERFACE_IS_LOOPBACK;

  if (data->ifa_flags & IFF_POINTOPOINT)
    info->flags |= GNETWORK_INTERFACE_IS_POINT_TO_POINT;

#ifdef IFF_MASTER
  if (data->ifa_flags & IFF_MASTER)
    info->flags |= GNETWORK_INTERFACE_IS_LOAD_MASTER;
#endif /* IFF_MASTER */

#ifdef IFF_SLAVE
  if (data->ifa_flags & IFF_SLAVE)
    info->flags |= GNETWORK_INTERFACE_IS_LOAD_SLAVE;
#endif /* IFF_SLAVE */

  if (data->ifa_flags & IFF_BROADCAST)
    info->flags |= GNETWORK_INTERFACE_CAN_BROADCAST;

  if (data->ifa_flags & IFF_MULTICAST)
    info->flags |= GNETWORK_INTERFACE_CAN_MULTICAST;

#ifdef IFF_NOTRAILERS
  if (data->ifa_flags & IFF_NOTRAILERS)
    info->flags |= GNETWORK_INTERFACE_NO_TRAILERS;
#endif /* IFF_NOTRAILERS */

  if (data->ifa_flags & IFF_NOARP)
    info->flags |= GNETWORK_INTERFACE_NO_ARP;

#ifdef IFF_PORTSEL
  if (data->ifa_flags & IFF_PORTSEL)
    info->flags |= GNETWORK_INTERFACE_CAN_SET_MEDIA;
#endif /* IFF_PORTSEL */

#ifdef IFF_ALTPHYS
  if (data->ifa_flags & IFF_ALTPHYS)
    info->flags |= GNETWORK_INTERFACE_ALTERNATE_LINK;
#endif /* IFF_ALTPHYS */

#ifdef IFF_AUTOMEDIA
  if (data->ifa_flags & IFF_AUTOMEDIA)
    info->flags |= GNETWORK_INTERFACE_AUTOSELECTED_MEDIA;
#endif /* IFF_AUTOMEDIA */

  if (data->ifa_flags & IFF_PROMISC)
    info->flags |= GNETWORK_INTERFACE_RECV_ALL_PACKETS;

  if (data->ifa_flags & IFF_ALLMULTI)
    info->flags |= GNETWORK_INTERFACE_RECV_ALL_MULTICAST;

  return info;
}


/* ************************************************************************** *
 *  Public API                                                                *
 * ************************************************************************** */

GType
gnetwork_interface_info_get_type (void)
{
  static GType type = G_TYPE_INVALID;

  if (type == G_TYPE_INVALID)
    {
      type = g_boxed_type_register_static ("GNetworkInterfaceInfo",
					   (GBoxedCopyFunc) gnetwork_interface_info_ref,
					   (GBoxedFreeFunc) gnetwork_interface_info_unref);
    }

  return type;
}


/**
 * gnetwork_interface_get_info:
 * @name: the name of the interface.
 * 
 * Retrieves the information for the interface referred to by @name. The @name
 * should be something like "%eth0", "%lo", etc.
 * 
 * Returns: a structure describing a local interface.
 * 
 * Since: 1.0
 **/
GNetworkInterfaceInfo *
gnetwork_interface_get_info (const gchar * name)
{
  struct ifaddrs *interfaces, *current;
  GNetworkInterfaceInfo *retval;

  g_return_val_if_fail (name != NULL, NULL);
  g_return_val_if_fail (strncmp (name, "sit", 3) != 0, NULL);

  interfaces = NULL;

  if (getifaddrs (&interfaces) < 0)
    return NULL;

  retval = NULL;

  for (current = interfaces; current != NULL; current = current->ifa_next)
    {
      if (g_ascii_strcasecmp (current->ifa_name, name) == 0)
	{
	  if (retval == NULL)
	    {
	      retval = create_info_from_interface (current);
	    }

	  append_iface_info_from_interface (retval, current);
	}
    }

  freeifaddrs (interfaces);

  return retval;
}


typedef struct
{
  gboolean is_ipaddr;
  gconstpointer addr;
  GNetworkInterfaceInfo *retval;
}
FindByAddrData;


static void
find_by_address (gpointer key, GNetworkInterfaceInfo *info, FindByAddrData *find_data)
{
  if (find_data->is_ipaddr)
    {
      if (gnetwork_ip_address_is_ipv4 (find_data->addr))
	{
	  if (gnetwork_ip_address_equal (&(info->ip4_address), find_data->addr) ||
	      gnetwork_ip_address_equal (&(info->ip4_broadcast_or_destination), find_data->addr) ||
	      gnetwork_ip_address_equal (&(info->ip4_netmask), find_data->addr))
	    {
	      find_data->retval = gnetwork_interface_info_ref (info);
	    }
	}
      else if (gnetwork_ip_address_equal (&(info->ip6_address), find_data->addr))
	{
	  find_data->retval = gnetwork_interface_info_ref (info);
	}
    }
  else if (g_ascii_strcasecmp (find_data->addr, info->hw_address) == 0)
    {
      find_data->retval = gnetwork_interface_info_ref (info);
    }
}


/**
 * gnetwork_interface_get_info_by_address:
 * @address: a valid address string.
 * 
 * Retrieves the #GNetworkInterfaceInfo which uses the address in @address.
 * 
 * Returns: a list of local interfaces.
 * 
 * Since: 1.0
 **/
GNetworkInterfaceInfo *
gnetwork_interface_get_info_by_address (const gchar * address)
{
  GNetworkIpAddress addr;
  GHashTable *table;
  struct ifaddrs *interfaces, *current;
  guint i;
  FindByAddrData find_data = {
    FALSE, NULL, NULL
  };

  interfaces = NULL;

  if (getifaddrs (&interfaces) < 0)
    return NULL;

  /* Create the interface information. */
  table = g_hash_table_new_full (g_str_hash, g_str_equal, NULL,
				 (GDestroyNotify) gnetwork_interface_info_unref);

  for (i = 0, current = interfaces; current != NULL; current = current->ifa_next, i++)
    {
      if (strncmp (current->ifa_name, "sit", 3) != 0)
	{
	  GNetworkInterfaceInfo *data = g_hash_table_lookup (table, current->ifa_name);

	  if (data == NULL)
	    {
	      data = create_info_from_interface (current);
	      g_hash_table_insert (table, data->name, data);
	    }

	  append_iface_info_from_interface (data, current);
	}
    }

  freeifaddrs (interfaces);

  find_data.is_ipaddr = gnetwork_ip_address_set_from_string (&addr, address);

  if (find_data.is_ipaddr)
    find_data.addr = &addr;
  else
    find_data.addr = address;

  g_hash_table_foreach (table, (GHFunc) find_by_address, &find_data);
  g_hash_table_destroy (table);

  return find_data.retval;
}


/**
 * gnetwork_interface_get_all_interfaces:
 * 
 * Retrieves a list of #GNetworkInterfaceInfo structures representing the local
 * interfaces for this host. The returned list and list data should be freed
 * using the following code:
 * 
 * <informalexample><programlisting>g_slist_foreach (list, (GFunc) gnetwork_interface_info_unref, NULL);
 * g_slist_free (list);
 * </programlisting></informalexample>
 * 
 * Returns: a list of local interfaces.
 * 
 * Since: 1.0
 **/
GSList *
gnetwork_interface_get_all_interfaces (void)
{
  GHashTable *table;
  struct ifaddrs *interfaces, *current;
  GSList *retval;
  guint i;

  interfaces = NULL;

  if (getifaddrs (&interfaces) < 0)
    return NULL;

  table = g_hash_table_new (g_str_hash, g_str_equal);

  for (i = 0, current = interfaces; current != NULL; current = current->ifa_next, i++)
    {
      if (strncmp (current->ifa_name, "sit", 3) != 0)
	{
	  GNetworkInterfaceInfo *data = g_hash_table_lookup (table, current->ifa_name);

	  if (data == NULL)
	    {
	      data = create_info_from_interface (current);
	      g_hash_table_insert (table, data->name, data);
	    }

	  append_iface_info_from_interface (data, current);
	}
    }

  freeifaddrs (interfaces);

  retval = NULL;
  g_hash_table_foreach (table, _gnetwork_slist_from_hash_table, &retval);
  g_hash_table_destroy (table);

  return g_slist_sort (retval, (GCompareFunc) gnetwork_interface_info_collate);
}


/**
 * gnetwork_interface_info_get_name:
 * @info: the interface information to examine.
 * 
 * Retrieves the configured name of the interface described by @info (e.g.
 * "eth0"). If @info is invalid, %NULL will be returned.
 * 
 * Returns: the name of @info, or %NULL.
 * 
 * Since: 1.0
 **/
gchar *
gnetwork_interface_info_get_name (const GNetworkInterfaceInfo * info)
{
  g_return_val_if_fail (GNETWORK_IS_INTERFACE_INFO (info), NULL);

  return g_strdup (info->name);
}


/**
 * gnetwork_interface_info_get_protocols:
 * @info: the interface information to examine.
 * 
 * Retrieves the protocols used by the interface at @info. If @info is invalid,
 * or the interface does not support any known protcols, %GNETWORK_PROTOCOL_NONE
 * will be returned.
 * 
 * Returns: the protocols used by @info.
 * 
 * Since: 1.0
 **/
GNetworkProtocols
gnetwork_interface_info_get_protocols (const GNetworkInterfaceInfo * info)
{
  g_return_val_if_fail (GNETWORK_IS_INTERFACE_INFO (info), GNETWORK_PROTOCOL_NONE);

  return info->protocols;
}


/**
 * gnetwork_interface_info_get_address:
 * @info: the interface information to examine.
 * @protocol: the protocol type to use.
 * 
 * Retrieves the configured @protocol address of the interface described by
 * @info (e.g. "127.0.0.1" for IPv4, "::1" for IPv6, or "00:00:00:00:00:00" for
 * hardware). If @info is invalid, @protocol contains more than one flag, or if
 * @info does not support @protocol, %NULL will be returned.
 * 
 * Returns: the @protocol address of @info, or %NULL.
 * 
 * Since: 1.0
 **/
gconstpointer
gnetwork_interface_info_get_address (const GNetworkInterfaceInfo * info, GNetworkProtocols protocol)
{
  g_return_val_if_fail (GNETWORK_IS_INTERFACE_INFO (info), NULL);
  g_return_val_if_fail (_gnetwork_flags_value_is_valid (GNETWORK_TYPE_PROTOCOLS, protocol), NULL);

  switch (protocol)
    {
    case GNETWORK_PROTOCOL_PACKET:
      return info->hw_address;
    case GNETWORK_PROTOCOL_IPv4:
      return &(info->ip4_address);
    case GNETWORK_PROTOCOL_IPv6:
      return &(info->ip6_address);

    default:
      break;
    }

  g_return_val_if_reached (NULL);
}


/**
 * gnetwork_interface_info_get_broadcast_address:
 * @info: the interface information to examine.
 * @protocol: the protocol type to use.
 * 
 * Retrieves the broadcast address of the interface described by @info (e.g.
 * "127.0.0.255" for IPv4 or "00:00:00:00:00:00" for hardware). If @info is
 * invalid or is a "point-to-point" interface, or if @protocol does not support
 * broadcasting (like IPv6), %NULL will be returned.
 * 
 * Returns: the @protocol broadcast address of @info, or %NULL.
 * 
 * Since: 1.0
 **/
gconstpointer
gnetwork_interface_info_get_broadcast_address (const GNetworkInterfaceInfo * info,
					       GNetworkProtocols protocol)
{
  g_return_val_if_fail (GNETWORK_IS_INTERFACE_INFO (info), NULL);
  g_return_val_if_fail (_gnetwork_flags_value_is_valid (GNETWORK_TYPE_PROTOCOLS, protocol), NULL);

  switch (protocol)
    {
    case GNETWORK_PROTOCOL_PACKET:
      return (!(info->flags & GNETWORK_INTERFACE_IS_POINT_TO_POINT) ?
	      info->hw_broadcast_or_destination : NULL);
    case GNETWORK_PROTOCOL_IPv4:
      return (!(info->flags & GNETWORK_INTERFACE_IS_POINT_TO_POINT) ?
	      &(info->ip4_broadcast_or_destination) : NULL);
    case GNETWORK_PROTOCOL_IPv6:
      return NULL;

    default:
      break;
    }

  g_return_val_if_reached (NULL);
}


/**
 * gnetwork_interface_info_get_destination:
 * @info: the interface information to examine.
 * @protocol: the protocol type to use.
 * 
 * Retrieves the destination address of the interface described by @info, (e.g.
 * "127.0.0.1" for IPv4, "::1" for IPv6, or "00:00:00:00:00:00" for hardware).
 * The returned data should not be modified or freed. If @info is invalid or
 * is not a "point-to-point" interface, %NULL will be returned.
 * 
 * See also: gnetwork_interface_info_get_flags().
 * 
 * Returns: the @protocol destination address of @info, or %NULL.
 * 
 * Since: 1.0
 **/
gconstpointer
gnetwork_interface_info_get_destination (const GNetworkInterfaceInfo * info,
					 GNetworkProtocols protocol)
{
  g_return_val_if_fail (GNETWORK_IS_INTERFACE_INFO (info), NULL);
  g_return_val_if_fail (_gnetwork_flags_value_is_valid (GNETWORK_TYPE_PROTOCOLS, protocol), NULL);

  switch (protocol)
    {
    case GNETWORK_PROTOCOL_PACKET:
      return (info->flags & GNETWORK_INTERFACE_IS_POINT_TO_POINT ?
	      info->hw_broadcast_or_destination : NULL);
    case GNETWORK_PROTOCOL_IPv4:
      return (info->flags & GNETWORK_INTERFACE_IS_POINT_TO_POINT ?
	      &(info->ip4_broadcast_or_destination) : NULL);
    case GNETWORK_PROTOCOL_IPv6:
      return (info->flags & GNETWORK_INTERFACE_IS_POINT_TO_POINT ?
	      &(info->ip6_destination) : NULL);

    default:
      break;
    }

  g_return_val_if_reached (NULL);
}


/**
 * gnetwork_interface_info_get_netmask:
 * @info: the interface information to examine.
 * @protocol: the protocol to use.
 * 
 * Retrieves the @protocol network mask of the interface described by @info
 * (e.g. "255.255.255.255" for IPv4, or "ffff:ffff:ffff:ffff" for IPv6). If
 * @info is invalid, %NULL will be returned.
 * 
 * Returns: the network mask of @info, or %NULL.
 * 
 * Since: 1.0
 **/
gconstpointer
gnetwork_interface_info_get_netmask (const GNetworkInterfaceInfo * info, GNetworkProtocols protocol)
{
  g_return_val_if_fail (GNETWORK_IS_INTERFACE_INFO (info), NULL);
  g_return_val_if_fail (_gnetwork_flags_value_is_valid (GNETWORK_TYPE_PROTOCOLS, protocol), NULL);

  switch (protocol)
    {
    case GNETWORK_PROTOCOL_IPv4:
      return &(info->ip4_netmask);
    case GNETWORK_PROTOCOL_IPv6:
      return &(info->ip6_netmask);
    case GNETWORK_PROTOCOL_PACKET:
      return NULL;

    default:
      break;
    }

  g_return_val_if_reached (NULL);
}


/**
 * gnetwork_interface_info_get_multicasts:
 * @info: the interface information to examine.
 * @protocol: the protocol to use.
 * 
 * Retrieves a list of current multicast IP addresses for @protocol from the
 * interface described by @info (e.g. "224.0.0.1" for IPv4, or "ff02::1" for
 * IPv6). If @info or @protocol is invalid, or @protocol does not support
 * multicasting (like the packet protocol), %NULL will be returned.
 * 
 * Returns: the multicast addresses of @info, or %NULL.
 * 
 * Since: 1.0
 **/
G_CONST_RETURN GSList *
gnetwork_interface_info_get_multicasts (const GNetworkInterfaceInfo * info,
					GNetworkProtocols protocol)
{
  g_return_val_if_fail (GNETWORK_IS_INTERFACE_INFO (info), NULL);
  g_return_val_if_fail (_gnetwork_flags_value_is_valid (GNETWORK_TYPE_PROTOCOLS, protocol), NULL);

  switch (protocol)
    {
    case GNETWORK_PROTOCOL_IPv4:
      return info->ip4_multicasts;
    case GNETWORK_PROTOCOL_IPv6:
      return info->ip6_multicasts;
    case GNETWORK_PROTOCOL_PACKET:
      return NULL;

    default:
      break;
    }

  g_return_val_if_reached (NULL);
}


/**
 * gnetwork_interface_info_get_flags:
 * @info: the interface information to examine.
 * 
 * Retrieves the flags set on interface described by @info. If @info is invalid
 * or no flags have been set, %GNETWORK_INTERFACE_NONE is returned.
 * 
 * Returns: the flags set on @interface.
 * 
 * Since: 1.0
 **/
GNetworkInterfaceFlags
gnetwork_interface_info_get_flags (const GNetworkInterfaceInfo * info)
{
  g_return_val_if_fail (GNETWORK_IS_INTERFACE_INFO (info), GNETWORK_INTERFACE_NONE);

  return info->flags;
}


/**
 * gnetwork_interface_info_get_index:
 * @info: the interface information to examine.
 * 
 * Retrieves the index of the interface described by @info. If @info is invalid
 * or the index is unknown, %0 is returned.
 * 
 * Returns: the index of @interface.
 * 
 * Since: 1.0
 **/
guint
gnetwork_interface_info_get_index (const GNetworkInterfaceInfo * info)
{
  g_return_val_if_fail (GNETWORK_IS_INTERFACE_INFO (info), 0);

  return info->index_;
}


/**
 * gnetwork_interface_info_ref:
 * @info: the data to reference.
 * 
 * Creates a reference to the data in @info. When no longer needed, this
 * reference should be released with gnetwork_interface_info_unref().
 * 
 * Returns: a reference to @info, or %NULL.
 * 
 * Since: 1.0
 **/
GNetworkInterfaceInfo *
gnetwork_interface_info_ref (GNetworkInterfaceInfo * info)
{
  g_return_val_if_fail (info == NULL || GNETWORK_IS_INTERFACE_INFO (info), NULL);

  if (info != NULL && info->ref > 0)
    {
      info->ref++;
    }

  return info;
}


/**
 * gnetwork_interface_info_unref:
 * @info: the local interface reference to release.
 * 
 * Releases a reference to the data in @info. When all references have been
 * released, the data in @info will be destroyed.
 * 
 * Since: 1.0
 **/
void
gnetwork_interface_info_unref (GNetworkInterfaceInfo * info)
{
  g_return_if_fail (info == NULL || GNETWORK_IS_INTERFACE_INFO (info));

  if (info == NULL)
    return;

  info->ref--;

  if (info->ref == 0)
    {
      g_free (info->name);

      g_slist_foreach (info->ip4_multicasts, (GFunc) g_free, NULL);
      g_slist_free (info->ip4_multicasts);

      g_slist_foreach (info->ip6_multicasts, (GFunc) g_free, NULL);
      g_slist_free (info->ip4_multicasts);

      g_free (info->hw_address);
      g_free (info->hw_broadcast_or_destination);
      g_free (info);
    }
}


/**
 * gnetwork_interface_info_collate:
 * @info1: a structure describing a local interface.
 * @info2: a structure describing a local interface.
 * 
 * Determines which interface of the arguments is "greater" (should be sorted
 * before) than the other, using the name.
 * 
 * Returns: %-1 if @info1 should be sorted first, %1 if @info2 should be sorted first, or %0 if they are equal.
 * 
 * Since: 1.0
 **/
gint
gnetwork_interface_info_collate (const GNetworkInterfaceInfo * info1,
				 const GNetworkInterfaceInfo * info2)
{
  gint retval;

  g_return_val_if_fail (info1 == NULL || GNETWORK_IS_INTERFACE_INFO (info1), 0);
  g_return_val_if_fail (info2 == NULL || GNETWORK_IS_INTERFACE_INFO (info2), 0);

  if (info1 == NULL && info2 != NULL)
    retval = 1;
  else if (info1 != NULL && info2 == NULL)
    retval = -1;
  else if (info1 == info2)
    retval = 0;
  else if (info1->name == NULL && info2->name != NULL)
    retval = 1;
  else if (info1->name != NULL && info2->name == NULL)
    retval = -1;
  else if (info1->name == info2->name)
    retval = 0;
  else
    retval = g_utf8_collate (info1->name, info2->name);

  return retval;
}


/* GNetworkProtocols
gnetwork_str_to_protocol (const gchar * address)
{
  GNetworkProtocols retval;
  gpointer ptr;

  if (address == NULL || address[0] == '\0')
    return GNETWORK_PROTOCOL_NONE;

  ptr = g_malloc0 (MAX (sizeof (struct in_addr), sizeof (struct in6_addr)));
  if (inet_pton (AF_INET6, address, ptr) > -1)
    retval = GNETWORK_PROTOCOL_IPv6;
  if (inet_pton (AF_INET, address, ptr) > -1)
    retval = GNETWORK_PROTOCOL_IPv4;
  else if (ether_aton (address) != NULL)
    retval = GNETWORK_PROTOCOL_PACKET;
  else
    retval = GNETWORK_PROTOCOL_NONE;

  g_free (ptr);

  return retval;
} */
