/* GNOME-DB Components
 * Copyright (C) 2000-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkstock.h>
#include <libgnomedb/libgnomedb.h>
#include "db-shell.h"
#include <libgnomeui/gnome-client.h>
#include <libgnomedb/gnome-db-i18n.h>
#include <libgnomeui/gnome-window-icon.h>
#include "database-capplet.h"

#include "../libgnomedb/gnome-db-window.h"

static void
prepare_menu (GnomeDbWindow *window, GtkNotebook *notebook)
{
	GtkWidget *current_widget, *wdsn;
	gboolean dsn, selection;
	BonoboUIComponent *uic;
	gint current;
	
	current = gtk_notebook_get_current_page (notebook);
	g_return_if_fail (current != -1);
	current_widget = gtk_notebook_get_nth_page (notebook, current);
	g_return_if_fail (current_widget != NULL);

	wdsn = g_object_get_data (G_OBJECT (notebook), DSN_PAGE); 
	dsn = current_widget == wdsn;
	selection = database_capplet_has_selection (GTK_WIDGET (notebook));

	uic = gnome_db_window_get_ui_component (window);
	bonobo_ui_component_set_prop (uic, "/commands/DatabaseProperties",
				      "sensitive", dsn && selection ? "1" : "0",
				      NULL);
	bonobo_ui_component_set_prop (uic, "/commands/EditDelete",
				      "sensitive", dsn && selection ? "1" : "0",
				      NULL);
	bonobo_ui_component_set_prop (uic, "/commands/ViewDataSources",
				      "state", dsn ? "1" : "0", NULL);
	bonobo_ui_component_set_prop (uic, "/commands/ViewProviders",
				      "state", dsn ? "0" : "1", NULL);
}

/*
 * Callbacks
 */

/* Database */
static void
menu_database_cb (GtkWidget *shell, gpointer user_data)
{
	prepare_menu (GNOME_DB_WINDOW (shell), GTK_NOTEBOOK (user_data));
}

/* Database/New */
static void
druid_finished_cb (GnomeDbDsnConfigDruid *druid, gboolean error, gpointer user_data)
{
	const GdaDataSourceInfo *dsn_info;

	if (!error) {
		dsn_info = gnome_db_dsn_config_druid_get_dsn (druid);
		if (dsn_info) {
			gnome_db_config_save_data_source (dsn_info->name,
							  dsn_info->provider,
							  dsn_info->cnc_string,
							  dsn_info->description,
							  dsn_info->username,
							  dsn_info->password);
		}
		else
			gnome_db_show_error (_("No valid data source info was created"));
	}

	gtk_widget_destroy (GTK_WIDGET (user_data));
}

static void
file_new_cb (GtkWidget *shell, gpointer user_data)
{
	GtkWidget *druid, *dialog;
	GdkPixbuf *icon;

	/* create the window to contain the druid */
	dialog = gtk_dialog_new_with_buttons (
		_("New data source"),
		GTK_WINDOW (shell),
		0, NULL);

	icon = gdk_pixbuf_new_from_file (LIBGNOMEDB_ICONSDIR "/gnome-db.png", NULL);
	if (icon) {
		gtk_window_set_icon (GTK_WINDOW (dialog), icon);
		g_object_unref (icon);
	}
	gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);

	druid = gnome_db_dsn_config_druid_new ();
	g_signal_connect (G_OBJECT (druid), "finished",
			  G_CALLBACK (druid_finished_cb), dialog);
	gtk_widget_show (druid);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), druid, TRUE, TRUE, 0);

	/* run the dialog */
	gtk_widget_show (dialog);
}

/* Database/Properties */
static void
file_properties_cb (GtkWidget *shell, gpointer user_data)
{
	database_capplet_edit_properties (GTK_WIDGET (user_data));
}

/* Database/Close */
static void
window_closed_cb (GnomeDbWindow *window, gpointer user_data)
{
	gnome_db_main_quit ();
}

/* Edit */
static void
menu_edit_cb (GtkWidget *shell, gpointer user_data)
{
	prepare_menu (GNOME_DB_WINDOW (shell), GTK_NOTEBOOK (user_data));
}

/* Edit/Delete */
static void
edit_delete_cb (GtkWidget *shell, gpointer user_data)
{
	database_capplet_delete (GTK_WIDGET (user_data));
}

/* Edit/Select All */
static void
edit_select_all_cb (GtkWidget *shell, gpointer user_data)
{
	database_capplet_select_all (GTK_WIDGET (user_data));
}

/* Edit/Clear Selection */
static void
edit_deselect_all_cb (GtkWidget *shell, gpointer user_data)
{
	database_capplet_deselect_all (GTK_WIDGET (user_data));
}

/* View */
static void
menu_view_cb (GtkWidget *shell, gpointer user_data)
{
	prepare_menu (GNOME_DB_WINDOW (shell), GTK_NOTEBOOK (user_data));
}

/* View/Data Sources */
static void
view_data_sources_cb (GtkWidget *shell, gpointer user_data)
{
	database_capplet_switch (GTK_WIDGET (user_data), DSN_PAGE);
}

/* View/Providers */
static void
view_providers_cb (GtkWidget *shell, gpointer user_data)
{
	database_capplet_switch (GTK_WIDGET (user_data), PROVIDER_PAGE);
}

static void
create_main_window (gpointer user_data)
{
	DbShellCallbacks *callbacks;
	GtkWidget *window;
	GtkWidget *capplet;

	capplet = database_capplet_new ();
        gtk_container_set_border_width (GTK_CONTAINER (capplet), 6);

	/* setup callbacks */
	callbacks = g_new0 (DbShellCallbacks, 1);
	memset (callbacks, 0, sizeof (DbShellCallbacks));
	callbacks->menu_database_func = menu_database_cb;
	callbacks->new_func = file_new_cb;
	callbacks->edit_properties_func = file_properties_cb;
	callbacks->menu_edit_func = menu_edit_cb;
	callbacks->delete_func = edit_delete_cb;
	callbacks->select_all_func = edit_select_all_cb;
	callbacks->deselect_all_func = edit_deselect_all_cb;
	callbacks->menu_view_func = menu_view_cb;
	callbacks->view_data_sources_func = view_data_sources_cb;
	callbacks->view_providers_func = view_providers_cb;

	/* create the dialog */
	window = db_shell_new (_("Database Access Properties"),
			       capplet,
			       NULL,
			       callbacks,
			       capplet);
	gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
	gnome_db_window_show (GNOME_DB_WINDOW (window));

	g_signal_connect (G_OBJECT (window), "close",
			  G_CALLBACK (window_closed_cb), NULL);
}

int
main (int argc, char *argv[])
{
	GnomeClient *session_client;

	bindtextdomain (GETTEXT_PACKAGE, LIBGNOMEDB_LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	gnome_db_init ("gnome-database-properties", PACKAGE_VERSION, argc, argv);

	/* session management */
	session_client = gnome_master_client ();
	gnome_client_set_restart_style (session_client, GNOME_RESTART_NEVER);

	/* application loop */
	gnome_db_main_run ((GdaInitFunc) create_main_window, NULL);

	return 0;
}
