/* gnome-db-grid.c
 *
 * Copyright (C) 2002 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include <gdk/gdkkeysyms.h>

#include "gnome-db-parameter.h"
#include "gnome-db-data-set.h"
#include "gnome-db-result-set.h"
#include "gnome-db-data-proxy.h"
#include "gnome-db-data-model.h"
#include "gnome-db-model-wrapper.h"
#include "gnome-db-gray-bar.h"

#include "marshal.h"
#include "gnome-db-server.h"
#include "gnome-db-server-data-type.h"
#include "gnome-db-data-handler.h"
#include "gnome-db-grid.h"
#include "gnome-db-data-widget.h"
#include "gnome-db-query.h"
#include "gnome-db-target.h"
#include "gnome-db-entity.h"
#include "gnome-db-renderer.h"
#include "gnome-db-basic-form.h"
#include "gnome-db-field.h"
#include "gnome-db-qfield.h"
#include "handlers/gnome-db-data-cell-renderer-info.h"
#include "handlers/gnome-db-data-cell-renderer-combo.h"
#include "utility.h"
#include "gnome-db-util.h"

#ifdef debug
#include "gnome-db-graphviz.h"
#endif

static void gnome_db_grid_class_init (GnomeDbGridClass * class);
static void gnome_db_grid_init (GnomeDbGrid * wid);
static void gnome_db_grid_dispose (GObject   * object);

static void gnome_db_grid_set_property (GObject              *object,
					     guint                 param_id,
					     const GValue         *value,
					     GParamSpec           *pspec);
static void gnome_db_grid_get_property (GObject              *object,
					     guint                 param_id,
					     GValue               *value,
					     GParamSpec           *pspec);

static void gnome_db_grid_initialize (GnomeDbGrid *grid);
static void create_new_tree_view (GnomeDbGrid *grid);

static GtkWidget *modif_buttons_make (GnomeDbGrid *grid);
static void       modif_buttons_update (GnomeDbGrid *grid);

static void work_param_changed_cb (GnomeDbParameter *param, GnomeDbGrid *grid);
static void tree_model_row_changed_cb (GtkTreeModel *treemodel, GtkTreePath *arg1, GtkTreeIter *arg2, GnomeDbGrid *grid);



/* GnomeDbDataWidget interface */
static void            gnome_db_grid_widget_init           (GnomeDbDataWidgetIface *iface);
static void            gnome_db_grid_set_mode              (GnomeDbDataWidget *iface, guint mode);
static void            gnome_db_grid_col_set_show          (GnomeDbDataWidget *iface, gint column, gboolean shown);
static void            gnome_db_grid_set_column_editable   (GnomeDbDataWidget *iface, gint column, gboolean editable);
static void            gnome_db_grid_show_column_actions   (GnomeDbDataWidget *iface, gint column, gboolean show_actions);
static void            gnome_db_grid_show_global_actions   (GnomeDbDataWidget *iface, gboolean show_actions);
static GtkActionGroup *gnome_db_grid_get_actions_group     (GnomeDbDataWidget *iface);
static GnomeDbDataSet *gnome_db_grid_widget_get_params_set (GnomeDbDataWidget *iface);
static GnomeDbDataSet *gnome_db_grid_widget_get_data_set   (GnomeDbDataWidget *iface);

static GdaDataModel   *gnome_db_grid_widget_get_gda_model             (GnomeDbDataWidget *iface);
static void            gnome_db_grid_widget_set_gda_model             (GnomeDbDataWidget *iface, GnomeDbDict *dict, GdaDataModel *model);
static void            gnome_db_grid_widget_set_title                 (GnomeDbDataWidget *iface, const gchar *title);
static void            gnome_db_grid_widget_set_show_title            (GnomeDbDataWidget *iface, gboolean show);
static void            gnome_db_grid_widget_set_title_icon_from_stock (GnomeDbDataWidget *iface, const gchar *stock_id, GtkIconSize size);

typedef struct {
	GnomeDbDataSetNode *node;
	GtkCellRenderer    *data_cell;
	GtkCellRenderer    *info_cell;
	gboolean            info_shown;
	gboolean            data_locked; /* no modification allowed on that column */
} ColumnData;
#define COLUMN_DATA(x) ((ColumnData *)(x))

static ColumnData *get_column_data (GnomeDbGrid *grid, GnomeDbDataSetNode *node);

struct _GnomeDbGridPriv
{
	GnomeDbDataProxy  *proxy;
	GnomeDbDataModel  *data_model;
	GnomeDbDataSet    *data_set;

	GSList            *nodes_infos; /* list of DataSetNodeInfo structures */

	GHashTable        *nodes_gtk_row_refs; /* key = node, value = list of GtkTreeRowReference holding list of values for that node */
	GHashTable        *row_ref_values;     /* key = GtkTreeRowReference, value = list of values for that key */
	GSList            *gtk_tree_row_references; /* list of GtkTreeRowReference used as key in 'row_ref_values' */

	guint              mode;
	GtkTooltips       *tooltips;
	GSList            *columns_data; /* list of ColumnData */
	gboolean           internal_params_change;

	GtkWidget         *title;
	GtkWidget         *scroll;
	GtkWidget         *treeview;
	gboolean           titles_shown;

	gboolean           default_show_info_cell;
	gboolean           default_column_editable;
	gboolean           default_show_global_actions;

	GtkUIManager      *uimanager;
	GtkActionGroup    *actions_group;
	GtkWidget         *widget_actions;
	GtkWidget         *widget_current_sample;
	GtkWidget         *widget_group;

	gint               export_type; /* used by the export dialog */
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

/* signals */
enum
{
        SELECTION_CHANGED,
	DOUBLE_CLICKED,
	CREATE_POPUP_MENU,
        LAST_SIGNAL
};

static gint gnome_db_grid_signals[LAST_SIGNAL] = { 0, 0, 0 };


/* properties */
enum
{
        PROP_0,
        PROP_TITLE_VISIBLE,
	PROP_TITLE_STRING,
	PROP_ACTIONS_VISIBLE,
	PROP_INFO_CELL_VISIBLE
};

GType
gnome_db_grid_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGridClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_grid_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGrid),
			0,
			(GInstanceInitFunc) gnome_db_grid_init
		};		

		static const GInterfaceInfo work_widget_info = {
                        (GInterfaceInitFunc) gnome_db_grid_widget_init,
                        NULL,
                        NULL
                };
		
		type = g_type_register_static (GTK_TYPE_VBOX, "GnomeDbGrid", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_DATA_WIDGET_TYPE, &work_widget_info);
	}

	return type;
}

static void
gnome_db_grid_widget_init (GnomeDbDataWidgetIface *iface)
{
	iface->set_mode = gnome_db_grid_set_mode;
	iface->col_set_show = gnome_db_grid_col_set_show;
	iface->set_column_editable = gnome_db_grid_set_column_editable;
	iface->show_column_actions = gnome_db_grid_show_column_actions;
	iface->show_global_actions = gnome_db_grid_show_global_actions;
	iface->get_actions_group = gnome_db_grid_get_actions_group;
	iface->get_params_set = gnome_db_grid_widget_get_params_set;
	iface->get_data_set = gnome_db_grid_widget_get_data_set;
	iface->get_gda_model = gnome_db_grid_widget_get_gda_model;
	iface->set_gda_model = gnome_db_grid_widget_set_gda_model;
	iface->set_show_title = gnome_db_grid_widget_set_show_title;
	iface->set_title = gnome_db_grid_widget_set_title;
	iface->set_t_i_from_stock = gnome_db_grid_widget_set_title_icon_from_stock;
}

static void
gnome_db_grid_class_init (GnomeDbGridClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	gnome_db_grid_signals[SELECTION_CHANGED] = 
		g_signal_new ("selection_changed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbGridClass, selection_changed),
                              NULL, NULL,
                              marshal_VOID__BOOLEAN, G_TYPE_NONE,
                              1, G_TYPE_BOOLEAN);
	gnome_db_grid_signals[DOUBLE_CLICKED] = 
		g_signal_new ("double_clicked",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbGridClass, double_clicked),
                              NULL, NULL,
                              marshal_VOID__INT, G_TYPE_NONE,
                              1, G_TYPE_INT);
	gnome_db_grid_signals[CREATE_POPUP_MENU] = 
		g_signal_new ("create_popup_menu",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbGridClass, create_popup_menu),
                              NULL, NULL,
                              marshal_VOID__POINTER, G_TYPE_NONE,
                              1, G_TYPE_POINTER);

	object_class->dispose = gnome_db_grid_dispose;

	/* Properties */
        object_class->set_property = gnome_db_grid_set_property;
        object_class->get_property = gnome_db_grid_get_property;
	g_object_class_install_property (object_class, PROP_TITLE_VISIBLE,
                                         g_param_spec_boolean ("title_visible", NULL, NULL, FALSE,
                                                               G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_TITLE_STRING,
                                         g_param_spec_string ("title_string", NULL, NULL, NULL,
							      G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_ACTIONS_VISIBLE,
                                         g_param_spec_boolean ("actions_visible", NULL, NULL, FALSE,
                                                               G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_INFO_CELL_VISIBLE,
                                         g_param_spec_boolean ("info_cell_visible", NULL, NULL, FALSE,
                                                               G_PARAM_WRITABLE));
}

static void
gnome_db_grid_init (GnomeDbGrid * wid)
{
	wid->priv = g_new0 (GnomeDbGridPriv, 1);
	wid->priv->proxy = NULL;
	wid->priv->default_show_info_cell = TRUE;
	wid->priv->default_show_global_actions = TRUE;
	wid->priv->default_column_editable = TRUE;

	wid->priv->columns_data = NULL;
	wid->priv->internal_params_change = FALSE;

	wid->priv->treeview = NULL;
	wid->priv->titles_shown = TRUE;

	wid->priv->mode = 0;
	wid->priv->tooltips = NULL;

	wid->priv->export_type = 2;

	/* title */
	wid->priv->title = gnome_db_gray_bar_new (_("No title"));
        gtk_box_pack_start (GTK_BOX (wid), wid->priv->title, FALSE, TRUE, 2);
}

/**
 * gnome_db_grid_new
 *
 * Creates a new #GnomeDbGrid widget
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_grid_new (void)
{
	GObject *obj;
	
	obj = g_object_new (GNOME_DB_GRID_TYPE, NULL);

	return GTK_WIDGET (obj);
}

/**
 * gnome_db_grid_new_with_gda_model
 * @dict: a #GnomeDbDict dictionary, or %NULL
 * @model: a #GdaDataModel
 *
 * Creates a new #GnomeDbGrid widget suitable to display the data in @model
 *
 *  Returns: the new widget
 */
GtkWidget *
gnome_db_grid_new_with_gda_model (GnomeDbDict *dict, GdaDataModel *model)
{
	GtkWidget *grid;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);

	grid = gnome_db_grid_new ();
	gnome_db_grid_widget_set_gda_model (GNOME_DB_DATA_WIDGET (grid), ASSERT_DICT (dict), model);

	return grid;
}

/**
 * gnome_db_grid_new_with_select_query
 * @query: a #GnomeDbQuery object
 * @modified: a #GnomeDbTarget object, or %NULL
 *
 * Creates a new #GnomeDbGrid widget.
 *
 * @query must be a SELECT query (no union, etc selection query)
 *
 * The @modified target must belong to @query and represent
 * modifiable entity (a #GnomeDbTable for example). If @modified is %NULL then
 * no modification will be allowed.
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_grid_new_with_select_query (GnomeDbQuery *query, GnomeDbTarget *modified)
{
	GObject *obj;
	GnomeDbGrid *grid;
	GnomeDbResultSet *rs;
	GnomeDbDataProxy *proxy;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (gnome_db_query_get_query_type (query) == GNOME_DB_QUERY_TYPE_SELECT, NULL);
	
	if (modified) {
		g_return_val_if_fail (IS_GNOME_DB_TARGET (modified), NULL);
		g_return_val_if_fail (gnome_db_target_get_query (modified) == query, NULL);
		g_return_val_if_fail (gnome_db_entity_is_writable (gnome_db_target_get_represented_entity (modified)), NULL);
	}
	
	obj = g_object_new (GNOME_DB_GRID_TYPE, NULL);
	grid = GNOME_DB_GRID (obj);

	rs = GNOME_DB_RESULT_SET (gnome_db_result_set_new (query, modified));
	proxy = GNOME_DB_DATA_PROXY (gnome_db_data_proxy_new (GNOME_DB_DATA_MODEL (rs)));
	g_object_unref (rs);
	grid->priv->data_model = GNOME_DB_DATA_MODEL (rs);
	grid->priv->proxy = proxy;
	
	grid->priv->data_set = gnome_db_data_model_get_new_data_set (GNOME_DB_DATA_MODEL (rs));
	grid->priv->default_column_editable = !gnome_db_data_proxy_is_read_only (grid->priv->proxy);

	gnome_db_grid_initialize (grid);

#ifdef debug_NO
	DEBUG_HEADER;
	gnome_db_base_dump (rs, 5);
#endif

	return GTK_WIDGET (obj);
}

static void gnome_db_grid_clean (GnomeDbGrid *grid);
static void
gnome_db_grid_dispose (GObject *object)
{
	GnomeDbGrid *grid;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_GRID (object));
	grid = GNOME_DB_GRID (object);

	if (grid->priv) {
		gnome_db_grid_clean (grid);
		/* the private area itself */
		g_free (grid->priv);
		grid->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void
gnome_db_grid_clean_row_refs (GnomeDbGrid *grid)
{
	if (grid->priv->nodes_gtk_row_refs) {
		g_hash_table_destroy (grid->priv->nodes_gtk_row_refs);
		grid->priv->nodes_gtk_row_refs = NULL;
	}

	if (grid->priv->row_ref_values) {
		g_hash_table_destroy (grid->priv->row_ref_values);
		grid->priv->row_ref_values = NULL;
	}
	
	if (grid->priv->gtk_tree_row_references) {
		GSList *list = grid->priv->gtk_tree_row_references;
		while (list) {
			gtk_tree_row_reference_free ((GtkTreeRowReference *)(list->data));
			list = g_slist_next (list);
		}
		g_slist_free (grid->priv->gtk_tree_row_references);
	}
}

static void
gnome_db_grid_clean (GnomeDbGrid *grid)
{
	/* internally stored values */
	gnome_db_grid_clean_row_refs (grid);

	/* nodes_infos */
	if (grid->priv->nodes_infos) {
		GSList *list = grid->priv->nodes_infos;
		while (list) {
			utility_data_set_node_info_destroy (DATA_SET_NODE_INFO (list->data));
			list = g_slist_next (list);
		}
		g_slist_free (grid->priv->nodes_infos);
		grid->priv->nodes_infos = NULL;
	}

	/* info cells */
	if (grid->priv->columns_data) {
		GSList *list = grid->priv->columns_data;
		while (list) {
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (grid->priv->columns_data);
		grid->priv->columns_data = NULL;
	}
	
	/* private data set */
	if (grid->priv->data_set) {
		g_object_unref (grid->priv->data_set);
		grid->priv->data_set = NULL;
	}
	
	/* proxy */
	if (grid->priv->proxy) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (grid->priv->proxy),
						      G_CALLBACK (tree_model_row_changed_cb), grid);
		
		g_object_unref (grid->priv->proxy);
		grid->priv->proxy = NULL;
	}
	
	/* tooltips */
	if (grid->priv->tooltips) {
		gtk_object_destroy (GTK_OBJECT (grid->priv->tooltips));
		grid->priv->tooltips = NULL;
	}
	
	/* UI */
	if (grid->priv->actions_group) {
		g_object_unref (G_OBJECT (grid->priv->actions_group));
		grid->priv->actions_group = NULL;
	}
	
	if (grid->priv->uimanager) {
		g_object_unref (G_OBJECT (grid->priv->uimanager));
		grid->priv->uimanager = NULL;
	}
}


static void
gnome_db_grid_set_property (GObject              *object,
				 guint                 param_id,
				 const GValue         *value,
				 GParamSpec           *pspec)
{
	GnomeDbGrid *grid;

        grid = GNOME_DB_GRID (object);
        if (grid->priv) {
                switch (param_id) {
                case PROP_TITLE_VISIBLE:
			if (g_value_get_boolean (value))
				gtk_widget_show (grid->priv->title);
			else
				gtk_widget_hide (grid->priv->title);
                        break;
                case PROP_TITLE_STRING:
			gnome_db_grid_widget_set_title (GNOME_DB_DATA_WIDGET (grid), g_value_get_string (value));
                        break;
		case PROP_ACTIONS_VISIBLE:
			grid->priv->default_show_global_actions = g_value_get_boolean (value);
			if (grid->priv->widget_group) {
				if (g_value_get_boolean (value)) 
					gtk_widget_show (grid->priv->widget_group);
				else
					gtk_widget_hide (grid->priv->widget_group);
			}
			break;
		case PROP_INFO_CELL_VISIBLE: 
		{
			GSList *list = grid->priv->columns_data;
			gboolean show = g_value_get_boolean (value);
			grid->priv->default_show_info_cell = show;

			while (list) {
				COLUMN_DATA (list->data)->info_shown = show;
				g_object_set (G_OBJECT (COLUMN_DATA (list->data)->info_cell), "visible", show, NULL);
				list = g_slist_next (list);
			}
		}
		break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }
}

static void
gnome_db_grid_get_property (GObject              *object,
				 guint                 param_id,
				 GValue               *value,
				 GParamSpec           *pspec)
{
	GnomeDbGrid *grid;

        grid = GNOME_DB_GRID (object);
        if (grid->priv) {
                switch (param_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }	
}

/**
 * gnome_db_grid_get_selection
 * @grid: a #GnomeDbGrid widget
 *
 * Returns the list of the currently selected rows in a #GnomeDbGrid widget. 
 * The returned value is a list of integers, which represent each of the selected rows.
 *
 * If new rows have been inserted, then those new rows will have a row number equal to -1.
 *
 * Returns: a new list, should be freed (by calling g_list_free) when no longer needed.
 */
GList *
gnome_db_grid_get_selection (GnomeDbGrid *grid)
{
	GtkTreeSelection *selection;
	GList *selected_rows;

	g_return_val_if_fail (grid && IS_GNOME_DB_GRID (grid), NULL);
	g_return_val_if_fail (grid->priv, NULL);

	if (!grid->priv->treeview)
		return NULL;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
	selected_rows = gtk_tree_selection_get_selected_rows (selection, NULL);
	if (selected_rows) {
		GList *list, *retlist = NULL;
		GtkTreeIter iter;
		gint row;

		list = selected_rows;
		while (list) {
			if (gtk_tree_model_get_iter (GTK_TREE_MODEL (grid->priv->proxy), &iter,
						     (GtkTreePath *)(list->data))) {
				gtk_tree_model_get (GTK_TREE_MODEL (grid->priv->proxy), &iter, 
						    PROXY_COL_MODEL_ROW, &row, -1);
				retlist = g_list_prepend (retlist, GINT_TO_POINTER (row));
			}
			list = g_list_next (list);
		}
		g_list_foreach (selected_rows, (GFunc) gtk_tree_path_free, NULL);
		g_list_free (selected_rows);
		return g_list_reverse (retlist);
	}
	else
		return NULL;
}

/**
 * gnome_db_grid_foreach_selected
 */
void
gnome_db_grid_foreach_selected (GnomeDbGrid *grid, GnomeDbGridForeachFunc foreach_func, gpointer user_data)
{
	GList *selected_rows, *list;

	g_return_if_fail (grid && IS_GNOME_DB_GRID (grid));
	g_return_if_fail (grid->priv);
	g_return_if_fail (foreach_func);

	selected_rows = gnome_db_grid_get_selection (grid);
	list = selected_rows;
	while (list) {
		foreach_func (grid, (gint) (list->data), user_data);
		list = g_list_next (list);
	}
	g_list_free (selected_rows);
}


/**
 * gnome_db_grid_select_all
 */
void
gnome_db_grid_select_all (GnomeDbGrid *grid)
{
	GtkTreeSelection *selection;
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
	
	gtk_tree_selection_select_all (selection);
}

/**
 * gnome_db_grid_unselect_all
 */
void
gnome_db_grid_unselect_all (GnomeDbGrid *grid)
{
	GtkTreeSelection *selection;
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
	
	gtk_tree_selection_unselect_all (selection);
}

/* 
 * take care of param dependencies: look at all the parameters in the data set and 
 * if @param is a dependency of a parameter P, then P is set to NULL 
 */
static void
work_param_changed_cb (GnomeDbParameter *param, GnomeDbGrid *grid)
{
	GSList *list;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;
	GList *sel_rows, *cur_row;

	if (!grid->priv->treeview)
		return;
	if (grid->priv->internal_params_change)
		return;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
	sel_rows = gtk_tree_selection_get_selected_rows (selection, &model);
	cur_row = sel_rows;
	while (cur_row) {
		gtk_tree_model_get_iter (model, &iter, (GtkTreePath *) (cur_row->data));
		list = grid->priv->data_set->parameters;
		while (list) {
			GSList *depend = gnome_db_parameter_get_dependencies (GNOME_DB_PARAMETER (list->data));
			if (g_slist_find (depend, param)) {
				gint col;
				
				col = gnome_db_data_model_get_column_at_param (grid->priv->data_model, 
									       grid->priv->data_set,
									       GNOME_DB_PARAMETER (list->data));
				gnome_db_data_proxy_set_value (grid->priv->proxy, &iter, col, NULL);
			}
			list = g_slist_next (list);
		}

		cur_row = g_list_next (cur_row);
	}
	g_list_foreach (sel_rows, (GFunc) gtk_tree_path_free, NULL);
	g_list_free (sel_rows);
	modif_buttons_update (grid);
}

static void
free_local_stored_values (GSList *values)
{
	GSList *list = values;
	while (list) {
		if (list->data)
			gda_value_free ((GdaValue *)list->data);
		list = g_slist_next (list);
	}
	g_slist_free (values);
}

/*
 * Real initialization
 */
static void 
gnome_db_grid_initialize (GnomeDbGrid *grid)
{
	GtkWidget *group, *sw;
	GSList *list;

	/* clean any widget, etc that were previously here */
	if (grid->priv->title) {
		gtk_widget_ref (grid->priv->title);
		gtk_container_remove (GTK_CONTAINER (grid), grid->priv->title);
	}
	if (grid->priv->scroll) {
		gtk_widget_destroy (grid->priv->scroll);
		grid->priv->treeview = NULL;
	}
	if (grid->priv->tooltips)
		gtk_object_destroy (GTK_OBJECT (grid->priv->tooltips));
	if (grid->priv->widget_actions)
		gtk_widget_destroy (grid->priv->widget_actions);
	if (grid->priv->widget_current_sample)
		gtk_widget_destroy (grid->priv->widget_current_sample);
	if (grid->priv->widget_group)
		gtk_widget_destroy (grid->priv->widget_group);

	/* DataSetNodeInfo info */
	list = grid->priv->data_set->nodes;
	while (list) {
		if (GNOME_DB_DATA_SET_NODE (list->data)->params) {
			DataSetNodeInfo *info;
			info = utility_data_set_node_info_create (grid->priv->data_set, GNOME_DB_DATA_SET_NODE (list->data));
			grid->priv->nodes_infos = g_slist_append (grid->priv->nodes_infos, info);
		}
		list = g_slist_next (list);
	}

	/* locally stored values */
	grid->priv->nodes_gtk_row_refs = g_hash_table_new_full (NULL, NULL, NULL, (GDestroyNotify) g_slist_free);
	grid->priv->row_ref_values = g_hash_table_new_full (NULL, NULL, NULL, (GDestroyNotify) free_local_stored_values);
	
	/* title */
	if (!grid->priv->title) {
		grid->priv->title = gnome_db_gray_bar_new (_("No title"));
		gtk_box_pack_start (GTK_BOX (grid), grid->priv->title, FALSE, TRUE, 2);
	}
	else {
		gtk_box_pack_start (GTK_BOX (grid), grid->priv->title, FALSE, TRUE, 2);
		gtk_widget_unref (grid->priv->title);
	}

	/*
	 * TreeView preparation
	 */
	sw = gtk_scrolled_window_new (NULL, NULL);
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw), GTK_SHADOW_IN);
	gtk_box_pack_start (GTK_BOX (grid), sw, TRUE, TRUE, 0);
	gtk_widget_show (sw);
	grid->priv->scroll = sw;
	create_new_tree_view (grid);

	/* signals connecting */
	if (grid->priv->data_set) {
		GSList *list = grid->priv->data_set->parameters;
		while (list) {
			g_signal_connect (G_OBJECT (list->data), "changed",
					  G_CALLBACK (work_param_changed_cb), grid);
			list = g_slist_next (list);
		}
	}

	/*
	 * the actions part
	 */
	group = modif_buttons_make (grid);
	gtk_box_pack_start (GTK_BOX (grid), group, FALSE, FALSE, 0);
	if (grid->priv->default_show_global_actions)
		gtk_widget_show (group);
	grid->priv->widget_group = group;
	modif_buttons_update (grid);

	/* tooltips */
	grid->priv->tooltips = gtk_tooltips_new ();
}

/*
 * creates a new string where underscores '_' are replaced by double underscores '__'
 * WARNING: the old string is free'ed so it is possible to do "str=double_underscores(str);"
 */
static gchar *
replace_double_underscores (gchar *str)
{
        gchar **arr;
        gchar *ret;
	
        arr = g_strsplit (str, "_", 0);
        ret = g_strjoinv ("__", arr);
        g_strfreev (arr);
	g_free (str);
	
        return ret;
}

static void     cell_renderer_value_set_attributes (GtkTreeViewColumn *tree_column, GtkCellRenderer *cell,
						    GtkTreeModel *tree_model, GtkTreeIter *iter, GnomeDbGrid *grid);
static void     cell_renderer_info_set_attributes  (GtkTreeViewColumn *tree_column, GtkCellRenderer *cell,
						    GtkTreeModel *tree_model, GtkTreeIter *iter, GnomeDbGrid *grid);

static gboolean tree_view_event_cb (GtkWidget *treeview, GdkEvent *event, GnomeDbGrid *grid);
static gint     tree_view_popup_button_pressed_cb (GtkWidget *widget, GdkEventButton *event, GnomeDbGrid *grid);

static void     tree_view_selection_changed_cb (GtkTreeSelection *selection, GnomeDbGrid *grid);
static void     tree_view_row_activated_cb     (GtkTreeView *tree_view, GtkTreePath *path, 
						GtkTreeViewColumn *column, GnomeDbGrid *grid);
static void     data_cell_value_changed (GtkCellRenderer *renderer, const gchar *path, 
					 const GdaValue *new_value, GnomeDbGrid *grid);
static void     data_cell_values_changed (GtkCellRenderer *renderer, const gchar *path, 
					  GSList *new_values, GSList *all_new_values, GnomeDbGrid *grid);
static void     data_cell_status_changed (GtkCellRenderer *renderer, const gchar *path, 
					  GnomeDbValueAttribute requested_action, GnomeDbGrid *grid);
static void     gnome_db_grid_set_params_from_iter (GnomeDbGrid *grid, GtkTreeIter *iter);

/*
 * Creates the GtkTreeView component and its columns, from the grid->priv->proxy
 * GtkTreeModel
 */
static void
create_new_tree_view (GnomeDbGrid *grid)
{
	gint i;
	GtkTreeView *tree_view;
	GtkTreeSelection *selection;
	GSList *list;
	GnomeDbDict *dict;
	GSList *nodes_infos_list;

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (grid->priv->proxy));

	/* Actual GtkTreeView widget creation */
	tree_view = GTK_TREE_VIEW (gtk_tree_view_new_with_model (GTK_TREE_MODEL (grid->priv->proxy)));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);
	gtk_tree_view_set_enable_search (GTK_TREE_VIEW (tree_view), TRUE);
	g_assert (!grid->priv->treeview);
	grid->priv->treeview = GTK_WIDGET (tree_view);
	gtk_container_add (GTK_CONTAINER (grid->priv->scroll), grid->priv->treeview);
	gtk_widget_show (grid->priv->treeview);

	g_signal_connect (G_OBJECT (tree_view), "event",
			  G_CALLBACK (tree_view_event_cb), grid);
	g_signal_connect (G_OBJECT (tree_view), "button_press_event",
                          G_CALLBACK (tree_view_popup_button_pressed_cb), grid);

	/* selection and signal handling */
	selection = gtk_tree_view_get_selection (tree_view);
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);
	g_signal_connect (G_OBJECT (selection), "changed",
			  G_CALLBACK (tree_view_selection_changed_cb), grid);
	g_signal_connect (G_OBJECT (tree_view), "row_activated",
			  G_CALLBACK (tree_view_row_activated_cb), grid);
	g_signal_connect (G_OBJECT (grid->priv->proxy), "row_changed",
			  G_CALLBACK (tree_model_row_changed_cb), grid);

	/* Creation of the columns in the treeview, to fit the parameters in the context */
	nodes_infos_list = grid->priv->nodes_infos;
	list = grid->priv->data_set->nodes;
	i = 0;
	while (list) {
		GnomeDbParameter *param;
		GnomeDbDataSetNode *node;
		GtkTreeViewColumn *column;
		GtkCellRenderer *renderer;
		ColumnData *column_data;

		node = GNOME_DB_DATA_SET_NODE (list->data);

		/* update the list of columns data */
		column_data = get_column_data (grid, node);
		if (!column_data) {
			column_data = g_new0 (ColumnData, 1);
			column_data->node = node;
			column_data->info_shown = grid->priv->default_show_info_cell;
			column_data->data_locked = !grid->priv->default_column_editable;
			grid->priv->columns_data = g_slist_append (grid->priv->columns_data, column_data);
		}

		/* create renderers */
		if ((param = node->param)) { /* single direct parameter */
			GnomeDbServerDataType *type;
			GnomeDbDataHandler *dh = NULL;
			gchar *plugin = NULL;
			gchar *title;
			
			type = gnome_db_parameter_get_data_type (param);
			g_assert (type);

			title = gnome_db_base_get_name (GNOME_DB_BASE (param));
			if (title && *title)
				title = replace_double_underscores (g_strdup (title));
			else
				title = NULL;
			if (!title)
				title = g_strdup (_("No title"));
			
			g_object_get (G_OBJECT (param), "handler_plugin", &plugin, NULL);
			if (plugin) {
				dh = gnome_db_server_get_handler_by_name (gnome_db_dict_get_server (dict), plugin);

				/* test if plugin can handle the parameter's data type */
				if (!gnome_db_data_handler_accepts_gda_type (dh, 
									     gnome_db_server_data_type_get_gda_type (type)))
					dh = NULL;
			}

			if (!dh)
				dh = gnome_db_server_data_type_get_handler (type);
						
			renderer = gnome_db_data_handler_get_cell_renderer (dh, gnome_db_server_data_type_get_gda_type (type));
			column_data->data_cell = renderer;
			gtk_tree_view_insert_column_with_data_func (tree_view, i, title, renderer,
								    (GtkTreeCellDataFunc) cell_renderer_value_set_attributes, 
								    grid, NULL);
			column = gtk_tree_view_get_column (tree_view, i);
			g_free (title);

			g_signal_connect (G_OBJECT (renderer), "changed", 
					  G_CALLBACK (data_cell_value_changed), grid);

		}
		else { /* parameters depending on a GnomeDbDataModel */
			gchar *title;
			gboolean nullok = TRUE;
			GSList *params;
			
			params = node->params;
			while (params && nullok) {
				if (gnome_db_parameter_get_not_null (GNOME_DB_PARAMETER (params->data)))
					nullok = FALSE;
				params = g_slist_next (params);
			}
			
			title = gnome_db_base_get_name (GNOME_DB_BASE (node->data_for_param));
			if (title)
				title = replace_double_underscores (g_strdup (title));
			else 
				/* FIXME: find a better label */
				title = g_strdup (_("Value"));

			/* FIXME: if nullok is TRUE, then set the column title in bold */

			renderer = gnome_db_data_cell_renderer_combo_new (grid->priv->data_set, node);
			column_data->data_cell = renderer;
			gtk_tree_view_insert_column_with_data_func (tree_view, i, title, renderer,
								    (GtkTreeCellDataFunc) cell_renderer_value_set_attributes, 
								    grid, NULL);
			column = gtk_tree_view_get_column (tree_view, i);
			g_free (title);

			g_signal_connect (G_OBJECT (renderer), "changed", 
					  G_CALLBACK (data_cell_values_changed), grid);

			g_assert (nodes_infos_list);
			g_object_set_data (G_OBJECT (column), "node_info", nodes_infos_list->data);
			nodes_infos_list = g_slist_next (nodes_infos_list);
		}
		g_object_set_data (G_OBJECT (column), "data_renderer", renderer);
		g_object_set (G_OBJECT (renderer), "editable", !column_data->data_locked, NULL);

		/* settings and signals */
		g_object_set (G_OBJECT (renderer), "editable", !column_data->data_locked, NULL);
		if (g_object_class_find_property (G_OBJECT_GET_CLASS (renderer), "set_default_if_invalid"))
			g_object_set (G_OBJECT (renderer), "set_default_if_invalid", TRUE, NULL);
		g_object_set_data (G_OBJECT (renderer), "context-node", node);
		g_object_set_data (G_OBJECT (column), "context-node", node);

		/* Adding the GdaValue's information cell as another GtkCellRenderer */
		renderer = gnome_db_data_cell_renderer_info_new (grid->priv->proxy, grid->priv->data_set, node);
		column_data->info_cell = renderer;
		gtk_tree_view_column_pack_end (column, renderer, FALSE);
		gtk_tree_view_column_set_cell_data_func (column, renderer, 
							 (GtkTreeCellDataFunc) cell_renderer_info_set_attributes, 
							 grid, NULL);
		g_signal_connect (G_OBJECT (renderer), "status_changed",
				  G_CALLBACK (data_cell_status_changed), grid);
		g_object_set (G_OBJECT (renderer), "visible", column_data->info_shown, NULL);
		g_object_set_data (G_OBJECT (renderer), "context-node", node);

		/* Sorting data */
		/* gtk_tree_view_column_set_sort_column_id (column, i); */
		/* gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (tree_model), */
		/* i, (GtkTreeIterCompareFunc) tree_sortable_sort_values, node, NULL); */
		list = g_slist_next (list);
		i++;
	}

	/*
	 * Hiding some columns of the grid
	 */
	list = grid->priv->data_set->parameters;
	while (list) {
		guint hint;
		hint = GPOINTER_TO_UINT (g_hash_table_lookup (grid->priv->data_set->hints, list->data));
		if (hint & GNOME_DB_DATA_SET_PARAM_HIDE) {
			GtkTreeViewColumn *viewcol;
			GnomeDbDataSetNode *cnode;
			gint pos;

			cnode = gnome_db_data_set_find_node_for_param (grid->priv->data_set, list->data);	
			pos = g_slist_index (grid->priv->data_set->nodes, cnode);
			g_assert (pos >= 0);

			viewcol = gtk_tree_view_get_column (GTK_TREE_VIEW (grid->priv->treeview), pos);
			gtk_tree_view_column_set_visible (viewcol, FALSE);
		}
		list = g_slist_next (list);
	}
}

static GtkTreeRowReference *
find_gtk_tree_row_reference (GnomeDbGrid *grid, GtkTreeIter *iter, GnomeDbDataSetNode *node, 
			     gboolean create_if_missing, gboolean *created)
{
	GSList *list;
	GtkTreeRowReference *ref = NULL;
	GtkTreePath *refpath, *path;
	
	refpath = gtk_tree_model_get_path (GTK_TREE_MODEL (grid->priv->proxy), iter);
	list = g_hash_table_lookup (grid->priv->nodes_gtk_row_refs, node);
	while (list && !ref) {
		path = gtk_tree_row_reference_get_path ((GtkTreeRowReference *) (list->data));
		if (!path) {
			/* remove this GtkTreeRowReference as it references an invalid path */
			GtkTreeRowReference *rem_ref = (GtkTreeRowReference *) (list->data);

			if (g_slist_length (list) <= 1) {
				g_hash_table_remove (grid->priv->nodes_gtk_row_refs, node);
				list = NULL;
			}
			else {
				GSList *tmp = g_slist_copy (list);
				tmp = g_slist_remove (tmp, rem_ref);
				g_hash_table_insert (grid->priv->nodes_gtk_row_refs, node, tmp);
				list = tmp;
			}

			g_hash_table_remove (grid->priv->row_ref_values, rem_ref);
			grid->priv->gtk_tree_row_references = g_slist_remove (grid->priv->gtk_tree_row_references, rem_ref);
			gtk_tree_row_reference_free (rem_ref);
		}
		else {
			if (! gtk_tree_path_compare (path, refpath)) 
				ref = (GtkTreeRowReference *) (list->data);
			
			gtk_tree_path_free (path);
			list = g_slist_next (list);
		}
	}

	if (created)
		*created = FALSE;
	
	if (!ref && create_if_missing) {
		list = g_hash_table_lookup (grid->priv->nodes_gtk_row_refs, node);
		ref = gtk_tree_row_reference_new (GTK_TREE_MODEL (grid->priv->proxy), refpath);
		list = g_slist_prepend (list, ref);
		g_hash_table_insert (grid->priv->nodes_gtk_row_refs, node, list);
		grid->priv->gtk_tree_row_references = g_slist_prepend (grid->priv->gtk_tree_row_references, ref);

		if (created)
			*created = TRUE;
	}

	gtk_tree_path_free (refpath);

	return ref;
}

/*
 * Set the attributes for each cell renderer which is not the information cell renderer, 
 * called by each cell renderer before actually displaying anything.
 */
static void
cell_renderer_value_set_attributes (GtkTreeViewColumn *tree_column,
				    GtkCellRenderer *cell,
				    GtkTreeModel *tree_model,
				    GtkTreeIter *iter, GnomeDbGrid *grid)
{
	GnomeDbDataSetNode *node;
	guint attributes;
	gboolean to_be_deleted = FALSE;
	ColumnData *column_data;
	GdaValue *value;
	gint col;
	gint offset = gnome_db_data_proxy_get_n_columns (grid->priv->proxy);

	node = g_object_get_data (G_OBJECT (tree_column), "context-node");
	column_data = get_column_data (grid, node);
	
	if (node->param) { /* single parameter */
		col = gnome_db_data_model_get_column_at_param (grid->priv->data_model, grid->priv->data_set,
							       node->param);
		gtk_tree_model_get (GTK_TREE_MODEL (grid->priv->proxy), iter, 
				    PROXY_COL_TO_DELETE, &to_be_deleted,
				    col, &value, 
				    offset + col, &attributes, -1);
		g_object_set (G_OBJECT (cell), 
			      "value_attributes", attributes,
			      "value", value,
			      "editable", !column_data->data_locked,
			      "cell_background", GNOME_DB_COLOR_NORMAL_MODIF,
			      "cell_background-set", ! (attributes & GNOME_DB_VALUE_IS_UNCHANGED),
			      "to_be_deleted", to_be_deleted, 
			      NULL);
	}
	else { /* multiple parameters depending on a data model */
		GList *values = NULL;
		DataSetNodeInfo *info;
		gboolean values_to_free = TRUE;

		info = g_object_get_data (G_OBJECT (tree_column), "node_info");

		/* NOTE:
		 * For performances reasons we want to provide, if possible, all the values required by the combo cell
		 * renderer to draw whatever it wants, without further making it search for the values it wants in
		 * node->data_for_params.
		 *
		 * For this purpose, we try to get a complete list of values making one row of the node->data_for_params
		 * data model, so that the combo cell renderer has all the values it wants.
		 *
		 * This optimization is possible if the utility_data_set_node_info_compute_values_ext() function
		 * returns a non NULL list.
		 * 
		 * If it does then we try to look into the locally stored lists of values which we stored when
		 * values of the combo cell renderer have changed (when data_cell_values_changed() was called), 
		 * and we only store (for optimizations), such lists if the combo cell renderer made some changes.
		 *
		 * NOTE2:
		 * This optimization is required anyway when node->data_for_params can be changed depending on
		 * external events and we can't know when it has changed.
		 */

		attributes = utility_data_set_node_info_compute_attributes (info, grid->priv->proxy, iter, &to_be_deleted);
		values = utility_data_set_node_info_compute_values_ext (info, grid->priv->proxy, iter);
		if (!values) {
			/* try if found in locally stored values */
			GtkTreeRowReference *ref;
			ref = find_gtk_tree_row_reference (grid, iter, node, FALSE, NULL);
			if (ref) {
				values = g_hash_table_lookup (grid->priv->row_ref_values, ref);
				values_to_free = FALSE;
			}
		}
		if (values) {
			g_object_set (G_OBJECT (cell), 
				      "value_attributes", attributes,
				      "values_display", values,
				      "editable", !column_data->data_locked,
				      "cell_background", GNOME_DB_COLOR_NORMAL_MODIF,
				      "cell_background-set", ! (attributes & GNOME_DB_VALUE_IS_UNCHANGED),
				      "to_be_deleted", to_be_deleted, 
				      NULL);
			if (values_to_free)
				g_list_free (values);
		}
		else {
			values = utility_data_set_node_info_compute_values (info, grid->priv->proxy, iter);
			g_object_set (G_OBJECT (cell), 
				      "value_attributes", attributes,
				      "values", values,
				      "editable", !column_data->data_locked,
				      "cell_background", GNOME_DB_COLOR_NORMAL_MODIF,
				      "cell_background-set", ! (attributes & GNOME_DB_VALUE_IS_UNCHANGED),
				      "to_be_deleted", to_be_deleted, 
				      NULL);
			g_list_free (values);
		}
	}
}

/*
 * Set the attributes for each information cell renderer, 
 * called by each cell renderer before actually displaying anything.
 */
static void
cell_renderer_info_set_attributes (GtkTreeViewColumn *tree_column,
				   GtkCellRenderer *cell,
				   GtkTreeModel *tree_model,
				   GtkTreeIter *iter, GnomeDbGrid *grid)
{
	GnomeDbDataSetNode *node;
	guint attributes;
	gboolean to_be_deleted = FALSE;
	ColumnData *column_data;
	gint col;
	gint offset = gnome_db_data_proxy_get_n_columns (grid->priv->proxy);

	node = g_object_get_data (G_OBJECT (tree_column), "context-node");
	column_data = get_column_data (grid, node);
	
	if (node->param) { /* single parameter */
		col = gnome_db_data_model_get_column_at_param (grid->priv->data_model, grid->priv->data_set,
							       node->param);
		gtk_tree_model_get (GTK_TREE_MODEL (grid->priv->proxy), iter, 
				    PROXY_COL_TO_DELETE, &to_be_deleted,
				    offset + col, &attributes, -1);
		g_object_set (G_OBJECT (cell), 
			      "editable", !column_data->data_locked,
			      "value_attributes", attributes,
			      "cell_background", GNOME_DB_COLOR_NORMAL_MODIF,
			      "cell_background-set", ! (attributes & GNOME_DB_VALUE_IS_UNCHANGED),
			      "to_be_deleted", to_be_deleted, 
			      NULL);
	}
	else { /* multiple parameters depending on a data model */
		DataSetNodeInfo *info;

		info = g_object_get_data (G_OBJECT (tree_column), "node_info");
		attributes = utility_data_set_node_info_compute_attributes (info, grid->priv->proxy, iter, &to_be_deleted);

		g_object_set (G_OBJECT (cell), 
			      "editable", !column_data->data_locked,
			      "value_attributes", attributes,
			      "cell_background", GNOME_DB_COLOR_NORMAL_MODIF,
			      "cell_background-set", ! (attributes & GNOME_DB_VALUE_IS_UNCHANGED),
			      "to_be_deleted", to_be_deleted, 
			      NULL);
	}
}

static gboolean set_iter_from_path (GnomeDbGrid *grid, const gchar *path, GtkTreeIter *iter);

/*
 * Callback when a value displayed as a GtkCellRenderer has been changed, for single parameter cell renderers
 *
 * Apply the new_value to the GTkTreeModel (grid->priv->proxy)
 */
static void
data_cell_value_changed (GtkCellRenderer *renderer, const gchar *path, const GdaValue *new_value, GnomeDbGrid *grid)
{
	GtkTreeIter iter;
	GnomeDbDataSetNode *node;
	
	node = g_object_get_data (G_OBJECT (renderer), "context-node");
	g_return_if_fail (node->param);

	if (set_iter_from_path (grid, path, &iter)) {
		    gint col;
		    col = gnome_db_data_model_get_column_at_param (grid->priv->data_model, grid->priv->data_set, node->param);
		    gnome_db_data_proxy_set_value (grid->priv->proxy, &iter, col, new_value);

		    /* synchronize the parameters in data_set with the new values for the row */
		    gnome_db_grid_set_params_from_iter (grid, &iter);
	}

#ifdef debug_NO
	DEBUG_HEADER;
	gnome_db_base_dump (grid->priv->proxy, 5);
#endif
}


static void update_cached_gtk_tree_row_references (GnomeDbGrid *grid, GtkTreeIter *iter, GnomeDbDataSetNode *node, GSList *all_new_values);

/*
 * Callback when a value displayed as a GnomeDbDataCellRendererCombo has been changed.
 *
 * Apply the new_value to the GTkTreeModel (grid->priv->proxy)
 */
static void
data_cell_values_changed (GtkCellRenderer *renderer, const gchar *path, 
			  GSList *new_values, GSList *all_new_values, GnomeDbGrid *grid)
{
	GtkTreeIter iter;
	GnomeDbDataSetNode *node;
	
	node = g_object_get_data (G_OBJECT (renderer), "context-node");
	g_return_if_fail (node->params);

	if (new_values)
		g_return_if_fail (g_slist_length (node->params) == g_slist_length (new_values));
	else
		/* the reason for not having any value is that the GnomeDbDataCellRendererCombo had no selected item */
		return;
	
#ifdef debug_NO
	DEBUG_HEADER;
	gnome_db_base_dump (grid->priv->proxy, 0);
#endif

	if (set_iter_from_path (grid, path, &iter)) {
		GSList *list, *params;
		gint col;

		params = node->params;
		list = new_values;
		while (list) {
			col = gnome_db_data_model_get_column_at_param (grid->priv->data_model, grid->priv->data_set, 
								       GNOME_DB_PARAMETER (params->data));
			gnome_db_data_proxy_set_value (grid->priv->proxy, &iter, col, (GdaValue *)(list->data));
			list = g_slist_next (list);
			params = g_slist_next (params);
		}
		
		/* we need to internally store all these new values to be able to retreive them next time
		 * we need them to redraw a particular cell renderer. See the explanation in 
		 * cell_renderer_value_set_attributes()
		 */
		update_cached_gtk_tree_row_references (grid, &iter, node, all_new_values);

		/* synchronize the parameters in data_set with the new values for the row */
		gnome_db_grid_set_params_from_iter (grid, &iter);
	}

#ifdef debug_NO
	DEBUG_HEADER;
	gnome_db_base_dump (grid->priv->proxy, 0);
#endif
}

static gboolean
set_iter_from_path (GnomeDbGrid *grid, const gchar *path, GtkTreeIter *iter)
{
	GtkTreePath *treepath;

	g_assert (path);

	treepath = gtk_tree_path_new_from_string (path);
	if (! gtk_tree_model_get_iter (GTK_TREE_MODEL (grid->priv->proxy), iter, treepath)) {
		gtk_tree_path_free (treepath);
		g_warning ("Can't get iter for path %s", path);
		return FALSE;
	}
	gtk_tree_path_free (treepath);
	
	return TRUE;
}


static void
update_cached_gtk_tree_row_references (GnomeDbGrid *grid, GtkTreeIter *iter, GnomeDbDataSetNode *node, GSList *all_new_values)
{
	GSList *list;
	gboolean modified = FALSE;
	GtkTreeRowReference *ref;
	
	if (all_new_values) {
		gtk_tree_model_get (GTK_TREE_MODEL (grid->priv->proxy), iter, PROXY_COL_MODIFIED, &modified, -1);
		if (modified) {
			gboolean created;
			GSList *tmp;
			
			ref = find_gtk_tree_row_reference (grid, iter, node, TRUE, &created);
			list = NULL;
			tmp = all_new_values;
			while (tmp) {
				if (tmp->data)
					list = g_slist_append (list, gda_value_copy ((GdaValue *)(tmp->data)));
				else
					list = g_slist_append (list, NULL);
				tmp = g_slist_next (tmp);
			}
			g_hash_table_insert (grid->priv->row_ref_values, ref, list);
		}
	}
	if (!modified) {
		/* row is now considered as non-modified */
		ref = find_gtk_tree_row_reference (grid, iter, node, FALSE, NULL);
		if (ref) {
			list = g_hash_table_lookup (grid->priv->nodes_gtk_row_refs, node);
			if (g_slist_length (list) <= 1)
				g_hash_table_remove (grid->priv->nodes_gtk_row_refs, node);
			else {
				GSList *tmp = g_slist_copy (list);
				tmp = g_slist_remove (tmp, ref);
				g_hash_table_insert (grid->priv->nodes_gtk_row_refs, node, tmp);
			}
			
			g_hash_table_remove (grid->priv->row_ref_values, ref);
			grid->priv->gtk_tree_row_references = g_slist_remove (grid->priv->gtk_tree_row_references, ref);
			gtk_tree_row_reference_free (ref);
		}
	}
}

/*
 * Apply the requested_action to the GtkTreeModel (grid->priv->proxy)
 */
static void
data_cell_status_changed (GtkCellRenderer *renderer, const gchar *path, GnomeDbValueAttribute requested_action,
			  GnomeDbGrid *grid)
{
	GtkTreePath *treepath;
	GtkTreeIter iter;
	GnomeDbDataSetNode *node;
	GtkTreeModel *tree_model;
	gint col;
	gint offset;
	GdaValue *attribute;

        offset = gnome_db_data_proxy_get_n_columns (grid->priv->proxy);
	node = g_object_get_data (G_OBJECT (renderer), "context-node");
	tree_model = GTK_TREE_MODEL (grid->priv->proxy);

	treepath = gtk_tree_path_new_from_string (path);
	if (! gtk_tree_model_get_iter (tree_model, &iter, treepath)) {
		gtk_tree_path_free (treepath);
		g_warning ("Can't get iter for path %s", path);
		return;
	}
	gtk_tree_path_free (treepath);
	
	attribute = gda_value_new_uinteger (requested_action);
	if (node->param) {
		col = gnome_db_data_model_get_column_at_param (grid->priv->data_model, grid->priv->data_set, node->param);
		gnome_db_data_proxy_set_value (grid->priv->proxy, &iter, offset + col, attribute);
	}
	else {
		GSList *list = node->params;
		while (list) {
			col = gnome_db_data_model_get_column_at_param (grid->priv->data_model, grid->priv->data_set, 
								       GNOME_DB_PARAMETER (list->data));
			gnome_db_data_proxy_set_value (grid->priv->proxy, &iter, offset + col, attribute);
			list = g_slist_next (list);
		}
		
		/* we need to internally remove any cached list of values */
		update_cached_gtk_tree_row_references (grid, &iter, node, NULL);
	}
	gda_value_free (attribute);

	/* synchronize the parameters in data_set with the new values for the row */
	gnome_db_grid_set_params_from_iter (grid, &iter);

#ifdef debug_NO
	DEBUG_HEADER;
	gnome_db_base_dump (grid->priv->proxy, 0);
#endif
}

/* static gint */
/* tree_sortable_sort_values (GtkTreeModel *model, GtkTreeIter *itera, GtkTreeIter *iterb, GnomeDbDataSetNode *node) */
/* { */
/*         GdaValue *valuea, *valueb; */
/*         gint retval = 1; */
/* 	GnomeDbGrid *grid; */

/* 	grid = 	g_object_get_data (G_OBJECT (model), "grid"); */

/* 	valuea = utility_grid_model_get_value (model, itera, grid->priv->core, node, FALSE, NULL); */
/* 	valueb = utility_grid_model_get_value (model, iterb, grid->priv->core, node, FALSE, NULL); */

/* 	if (gda_value_get_type (valuea) == gda_value_get_type (valueb)) { */
/* 		if (gda_value_get_type (valuea) == GDA_VALUE_TYPE_LIST) { */
/* 			const GList *lista, *listb; */

/* 			lista = gda_value_get_list (valuea); */
/* 			listb = gda_value_get_list (valueb); */
/* 			retval = 0; */

/* 			g_assert (g_list_length (lista) == g_list_length (listb)); */
/* 			while (lista && listb && !retval) { */
/* 				GdaValue *vala = (GdaValue*) lista->data, *valb = (GdaValue*) listb->data; */
/* 				if (gda_value_is_null (vala)) { */
/* 					if (gda_value_is_null (valb)) */
/* 						retval = 0; */
/* 					else */
/* 						retval = 1; */
/* 				} */
/* 				else { */
/* 					if (gda_value_is_null (valb)) */
/* 						retval = -1; */
/* 					else */
/* 						retval = gda_value_compare (vala, valb);; */
/* 				} */
/* 				lista = g_list_next (lista); */
/* 				listb = g_list_next (listb); */
/* 			} */
/* 		} */
/* 		else */
/* 			retval = gda_value_compare (valuea, valueb); */
/* 	} */
/* 	else { */
/* 		if (gda_value_is_null (valuea)) */
/* 			retval = -1; */
/* 		else */
/* 			retval = 1; */
/* 	} */

/* 	gda_value_free (valuea); */
/* 	gda_value_free (valueb); */

/*         return retval; */
/* } */


/*
 * Catch any event in the GtkTreeView widget
 */
static gboolean 
tree_view_event_cb (GtkWidget *treeview, GdkEvent *event, GnomeDbGrid *grid)
{
	gboolean done = FALSE;

	if (event->type == GDK_KEY_PRESS) {
		GdkEventKey *ekey = (GdkEventKey *) event;
		guint modifiers = gtk_accelerator_get_default_mod_mask ();		

		/* Tab to move one column left or right */
		if (ekey->keyval == GDK_Tab) {
			GtkTreeViewColumn *column;
			GtkTreePath *path;

			/* FIXME: if a column is currently edited, then make sure the editing of that cell is not cancelled */
			gtk_tree_view_get_cursor (GTK_TREE_VIEW (treeview), &path, &column);
			if (column && path) {
				GList *columns = gtk_tree_view_get_columns (GTK_TREE_VIEW (treeview));
				GList *col;
				GtkCellRenderer *renderer;

				/* change column */
				col = g_list_find (columns, column);
				g_return_val_if_fail (col, FALSE);

				if (((ekey->state & modifiers) == GDK_SHIFT_MASK) || ((ekey->state & modifiers) == GDK_CONTROL_MASK))
					col = g_list_previous (col); /* going to previous column */
				else 
					col = g_list_next (col); /* going to next column */

				if (col) {
					renderer = g_object_get_data (G_OBJECT (col->data), "data_renderer");
					gtk_tree_view_set_cursor_on_cell (GTK_TREE_VIEW (treeview), path, 
									  GTK_TREE_VIEW_COLUMN (col->data), 
									  renderer, FALSE);
					gtk_widget_grab_focus (treeview);
					done = TRUE;
				}
				g_list_free (columns);
			}
			if (path)
				gtk_tree_path_free (path);
		}
		
		/* DELETE to delete the selected row */
		if (ekey->keyval == GDK_Delete) {
			GtkTreeIter iter;
			GtkTreeSelection *selection;
			GtkTreeModel *model;
			GList *sel_rows, *cur_row;

			selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
			sel_rows = gtk_tree_selection_get_selected_rows (selection, &model);
			cur_row = sel_rows;
			while (cur_row) {
				gtk_tree_model_get_iter (model, &iter, (GtkTreePath *) (cur_row->data));
				if (((ekey->state & modifiers) == GDK_SHIFT_MASK) || 
				    ((ekey->state & modifiers) == GDK_CONTROL_MASK))
					gnome_db_data_proxy_undelete (grid->priv->proxy, &iter);
				else
					gnome_db_data_proxy_delete (grid->priv->proxy, &iter);
				cur_row = g_list_next (cur_row);
			}
			g_list_foreach (sel_rows, (GFunc) gtk_tree_path_free, NULL);
			g_list_free (sel_rows);
			
			done = TRUE;
		}
	}

	return done;
}

static void menu_select_all_cb (GtkWidget *widget, GnomeDbGrid *grid);
static void menu_unselect_all_cb (GtkWidget *widget, GnomeDbGrid *grid);
static void menu_show_columns_cb (GtkWidget *widget, GnomeDbGrid *grid);
static void menu_save_as_cb (GtkWidget *widget, GnomeDbGrid *grid);

static gint
tree_view_popup_button_pressed_cb (GtkWidget *widget, GdkEventButton *event, GnomeDbGrid *grid)
{
	GtkWidget *menu;
	GtkTreeSelection *selection;
	GtkSelectionMode sel_mode;

        if (event->button != 3)
                return FALSE;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
	sel_mode = gtk_tree_selection_get_mode (selection);

        /* create the menu */
        menu = gtk_menu_new ();
	if (sel_mode == GTK_SELECTION_MULTIPLE)
		gtk_menu_append (GTK_MENU (menu),
				 gnome_db_new_menu_item (_("Select _All"), FALSE, 
							 G_CALLBACK (menu_select_all_cb), grid));

	if ((sel_mode == GTK_SELECTION_SINGLE) || (sel_mode == GTK_SELECTION_MULTIPLE))
		gtk_menu_append (GTK_MENU (menu),
				 gnome_db_new_menu_item (_("_Clear Selection"), FALSE,
							 G_CALLBACK (menu_unselect_all_cb), grid));
        gtk_menu_append (GTK_MENU (menu),
			 gnome_db_new_check_menu_item (_("Show Column _Titles"),
						       grid->priv->titles_shown,
						       G_CALLBACK (menu_show_columns_cb), grid));

	if (sel_mode != GTK_SELECTION_NONE) {
		gtk_menu_append (GTK_MENU (menu), gtk_separator_menu_item_new ());
		gtk_menu_append (GTK_MENU (menu), gnome_db_new_menu_item (GTK_STOCK_SAVE_AS, TRUE,
					       G_CALLBACK (menu_save_as_cb), grid));
	}

	/* allow listeners to add their custom menu items */
        g_signal_emit (G_OBJECT (grid), gnome_db_grid_signals [CREATE_POPUP_MENU], 0, GTK_MENU (menu));
        gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, event->button, event->time);
        gtk_widget_show_all (menu);
	
        return TRUE;
}

static void
menu_select_all_cb (GtkWidget *widget, GnomeDbGrid *grid)
{
        gnome_db_grid_select_all (grid);
}

static void
menu_unselect_all_cb (GtkWidget *widget, GnomeDbGrid *grid)
{
        gnome_db_grid_unselect_all (grid);
}

static void
menu_show_columns_cb (GtkWidget *widget, GnomeDbGrid *grid)
{
        GtkCheckMenuItem *item;

        item = (GtkCheckMenuItem *) widget;

        g_return_if_fail (GTK_IS_CHECK_MENU_ITEM (item));
	grid->priv->titles_shown = gtk_check_menu_item_get_active (item);

	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (grid->priv->treeview), grid->priv->titles_shown);
}

static void save_as_response_cb (GtkDialog *dialog, guint response_id, GnomeDbGrid *grid);

static void
menu_save_as_cb (GtkWidget *widget, GnomeDbGrid *grid)
{
	GtkWidget *dialog;
	GtkWidget *label;
	GtkWidget *filename;
	GtkWidget *types;
	GtkWidget *hbox;
	char *str;

	/* create dialog box */
	dialog = gtk_dialog_new_with_buttons (_("Saving Data"),
					      GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (grid))), 0,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_SAVE, GTK_RESPONSE_OK,
					      NULL);
	gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);

	str = g_strdup_printf ("<big><b>%s:</b></big>\n%s", _("Saving data to a file"), 
			       _("The data will be exported without any of the modifications which may "
				 "have been made and have not been comitted."));
        label = gtk_label_new ("");
        gtk_label_set_markup (GTK_LABEL (label), str);
        gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
        g_free (str);
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, FALSE, TRUE, 2);

	str = g_strdup_printf ("<b>%s:</b>", _("File name"));
        label = gtk_label_new ("");
        gtk_label_set_markup (GTK_LABEL (label), str);
        gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
        g_free (str);
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, FALSE, TRUE, 2);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, FALSE, FALSE, 5);
        gtk_widget_show (hbox);
        label = gtk_label_new ("    ");
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
        gtk_widget_show (label);

	filename = gtk_file_chooser_widget_new (GTK_FILE_CHOOSER_ACTION_SAVE);
	g_object_set_data (G_OBJECT (dialog), "filename", filename);
	gtk_box_pack_start (GTK_BOX (hbox), filename, TRUE, TRUE, 0);

	str = g_strdup_printf ("<b>%s:</b>", _("File type"));
        label = gtk_label_new ("");
        gtk_label_set_markup (GTK_LABEL (label), str);
        gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
        g_free (str);
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, FALSE, TRUE, 2);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, FALSE, FALSE, 5);
        gtk_widget_show (hbox);
        label = gtk_label_new ("    ");
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
        gtk_widget_show (label);

	types = gtk_combo_box_new_text ();
	gtk_box_pack_start (GTK_BOX (hbox), types, TRUE, TRUE, 0);
	g_object_set_data (G_OBJECT (dialog), "types", types);

	gtk_combo_box_append_text (GTK_COMBO_BOX (types), _("Tab-delimited"));
	gtk_combo_box_append_text (GTK_COMBO_BOX (types), _("Comma-delimited"));
	gtk_combo_box_append_text (GTK_COMBO_BOX (types), _("XML"));
	gtk_combo_box_set_active (GTK_COMBO_BOX (types), grid->priv->export_type);
	
	/* run the dialog */
	g_signal_connect (G_OBJECT (dialog), "response", G_CALLBACK (save_as_response_cb), grid);
	gtk_widget_show_all (dialog);
}

static gboolean confirm_file_overwrite (GtkWindow *parent, const gchar *path);

static void
save_as_response_cb (GtkDialog *dialog, guint response_id, GnomeDbGrid *grid)
{
	GtkWidget *types;
	gint export_type;
	GtkWidget *filename;
	
	if (response_id == GTK_RESPONSE_OK) {
		gchar *body;
		gchar *path;
		GList *columns, *list;
		gint *cols, nb_cols;

		types = g_object_get_data (G_OBJECT (dialog), "types");
		filename = g_object_get_data (G_OBJECT (dialog), "filename");

		/* output columns computation */
		columns = gtk_tree_view_get_columns (GTK_TREE_VIEW (grid->priv->treeview));
		cols = g_new (gint, gda_data_model_get_n_columns (GDA_DATA_MODEL (grid->priv->data_model)));
		nb_cols = 0;
		list = columns;
		while (list) {
			if (gtk_tree_view_column_get_visible (GTK_TREE_VIEW_COLUMN (list->data))) {
				GnomeDbDataSetNode *node;

				node = g_object_get_data (G_OBJECT (list->data), "context-node");
				if (node->param) {
					cols [nb_cols] = gnome_db_data_model_get_column_at_param (grid->priv->data_model,
												  grid->priv->data_set,
												  node->param);
					nb_cols ++;
				}
				else {
					GSList *params = node->params;
					while (params) {
						cols [nb_cols] = gnome_db_data_model_get_column_at_param (grid->priv->data_model,
										              grid->priv->data_set,
											      GNOME_DB_PARAMETER (params->data));
						nb_cols ++;
						params = g_slist_next (params);
					}
				}
			}
			list = g_list_next (list);
		}
		g_list_free (columns);
		
		/* Actual ouput computations */
		export_type = gtk_combo_box_get_active (GTK_COMBO_BOX (types));
		grid->priv->export_type = export_type;
		switch (export_type) {
		case 0:
			body = gda_data_model_to_text_separated (GDA_DATA_MODEL (grid->priv->data_model), 
								 cols, nb_cols, '\t');
			break;
		case 1:
			body = gda_data_model_to_text_separated (GDA_DATA_MODEL (grid->priv->data_model), 
								 cols, nb_cols, ',');
			break;
		case 2:
			body = gda_data_model_to_xml (GDA_DATA_MODEL (grid->priv->data_model), cols, nb_cols,
						      gnome_db_base_get_name (GNOME_DB_BASE (grid->priv->data_model)));
			break;
		default:
			g_assert_not_reached ();
			break;
		}
		g_free (cols);

		/* saving */
		if (body) {
			path = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (filename));
			if (path) {
				if (g_file_test (path, G_FILE_TEST_EXISTS)) {
					if (! confirm_file_overwrite (GTK_WINDOW (dialog), path)) {
						g_free (body);
						g_free (path);
						return;
					}
				}

				if (! gda_file_save (path, body, strlen (body))) {
					gnome_db_show_error (_("Could not save file %s"), path);
					g_free (body);
					g_free (path);
					return;
				}
				g_free (path);
			} 
			else {
				gnome_db_show_error (_("You must specify a file name"));
				g_free (body);
				return;
			}
			g_free (body);
		} else
			gnome_db_show_error (_("Got empty file while converting the data"));
	}

	gtk_widget_destroy (GTK_WIDGET (dialog));
}

static gboolean
confirm_file_overwrite (GtkWindow *parent, const gchar *path)
{
	GtkWidget *dialog, *button;
	gboolean yes;
	gchar *msg;

	msg = g_strdup_printf (_("File '%s' already exists.\n"
				 "Do you want to overwrite it?"), path);

	/* build the dialog */
	dialog = gnome_db_new_alert (
		NULL,
		GTK_MESSAGE_QUESTION,
		msg,
		_("If you choose yes, the contents will be lost."));
	button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_dialog_add_action_widget (GTK_DIALOG (dialog),
				      button,
				      GTK_RESPONSE_NO);
	gtk_dialog_add_action_widget (GTK_DIALOG (dialog),
				      gtk_button_new_from_stock (GTK_STOCK_YES),
				      GTK_RESPONSE_YES);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog),
					 GTK_RESPONSE_NO);

	/* run the dialog */
	gtk_widget_show_all (dialog);
	yes = gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_YES;

	/* free memory */
	gtk_widget_destroy (dialog);
	g_free (msg);
	return yes;
}


static void
tree_view_row_activated_cb (GtkTreeView *tree_view, GtkTreePath *path, GtkTreeViewColumn *column, GnomeDbGrid *grid)
{
	gint *indices;

	indices = gtk_tree_path_get_indices (path);
#ifdef debug_signal
	g_print (">> 'DOUBLE_CLICKED' from %s %p\n", G_OBJECT_TYPE_NAME (grid), grid);
#endif
	g_signal_emit (G_OBJECT (grid), gnome_db_grid_signals[DOUBLE_CLICKED], 0, *indices);
#ifdef debug_signal
	g_print ("<< 'DOUBLE_CLICKED' from %s %p\n", G_OBJECT_TYPE_NAME (grid), grid);
#endif	
}

/*
 * Synchronize the values of the parameters in priv->data_set with the currently selected row
 */
static void
tree_view_selection_changed_cb (GtkTreeSelection *selection, GnomeDbGrid *grid)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	gboolean row_selected = FALSE;
	gint has_selection;
	
	if (gtk_tree_selection_get_mode (selection) == GTK_SELECTION_MULTIPLE) {
		has_selection = gtk_tree_selection_count_selected_rows (selection);
		if (has_selection == 1) {
			GList *sel_rows;

			sel_rows = gtk_tree_selection_get_selected_rows (selection, &model);
			has_selection = gtk_tree_model_get_iter (model, &iter, (GtkTreePath *) (sel_rows->data)) ? 1 : 0;
			g_list_foreach (sel_rows, (GFunc) gtk_tree_path_free, NULL);
			g_list_free (sel_rows);
		}
	}
	else
		has_selection = gtk_tree_selection_get_selected (selection, &model, &iter) ? 1 : 0;

	if (has_selection == 1) {
		grid->priv->internal_params_change = TRUE;
		gnome_db_grid_set_params_from_iter (grid, &iter);
		grid->priv->internal_params_change = FALSE;
		row_selected = TRUE;
	}
	else {
		/* render all the parameters invalid */
		grid->priv->internal_params_change = TRUE;
		GSList *list = grid->priv->data_set->parameters;
		while (list) {
			GnomeDbParameter *tmp;
			g_object_get (G_OBJECT (list->data), "simple_bind", &tmp, NULL);
			if (! tmp) 
				gnome_db_parameter_declare_invalid (GNOME_DB_PARAMETER (list->data));
			
			list = g_slist_next (list);
		}
		grid->priv->internal_params_change = FALSE;
	}

	/* update the actions buttons */
	modif_buttons_update (grid);

#ifdef debug_signal
	g_print (">> 'SELECTION_CHANGED' from %s %p\n", G_OBJECT_TYPE_NAME (grid), grid);
#endif
	g_signal_emit (G_OBJECT (grid), gnome_db_grid_signals[SELECTION_CHANGED], 0, row_selected);
#ifdef debug_signal
	g_print ("<< 'SELECTION_CHANGED' from %s %p\n", G_OBJECT_TYPE_NAME (grid), grid);
#endif
}

static void
tree_model_row_changed_cb (GtkTreeModel *treemodel, GtkTreePath *arg1, GtkTreeIter *arg2, GnomeDbGrid *grid)
{
	GtkTreeSelection *selection;

	/* if the changed row is the current selection, then update the data set */
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
	if (gtk_tree_selection_iter_is_selected (selection, arg2) &&
	    (gtk_tree_selection_count_selected_rows (selection) == 1))
		tree_view_selection_changed_cb (selection, grid);
	else
		/* only update the actions buttons */
		modif_buttons_update (grid);	    
}

/*
 * Sets the parameters in 'data_set' from the values stored for the row provided by @iter.
 */
static void
gnome_db_grid_set_params_from_iter (GnomeDbGrid *grid, GtkTreeIter *iter)
{
	GSList *list;
	gint col;
	gint offset = gnome_db_data_proxy_get_n_columns (grid->priv->proxy);

	list = grid->priv->data_set->parameters;
	while (list) {
		GnomeDbParameter *param = GNOME_DB_PARAMETER (list->data);
		GdaValue *value;
		guint attributes;
		GnomeDbParameter *tmp;
		
		g_object_get (G_OBJECT (param), "simple_bind", &tmp, NULL);
		if (! tmp) {
			col = gnome_db_data_model_get_column_at_param (grid->priv->data_model, grid->priv->data_set, param);
			g_assert (col >= 0);
			gtk_tree_model_get (GTK_TREE_MODEL (grid->priv->proxy), iter, 
					    col, &value, 
					    offset + col, &attributes, -1);
			gnome_db_parameter_set_value (param, value);
			g_object_set (G_OBJECT (param), "use_default_value", 
				      attributes & GNOME_DB_VALUE_IS_DEFAULT ? TRUE : FALSE, NULL);
		}
		
		list = g_slist_next (list);
	}
}

static ColumnData *
get_column_data (GnomeDbGrid *grid, GnomeDbDataSetNode *node)
{
	ColumnData *retval = NULL;
	GSList *list = grid->priv->columns_data;
	while (list && !retval) {
		if (COLUMN_DATA (list->data)->node == node)
			retval = COLUMN_DATA (list->data);

		list = g_slist_next (list);
	}

	return retval;
}

/**
 * gnome_db_grid_set_sample_size
 * @grid:
 * @sample_size:
 *
 */
void
gnome_db_grid_set_sample_size (GnomeDbGrid *grid, gint sample_size)
{
	g_return_if_fail (grid && IS_GNOME_DB_GRID (grid));
	g_return_if_fail (grid->priv);

	gnome_db_data_proxy_set_sample_size (grid->priv->proxy, sample_size);
}

/**
 * gnome_db_grid_set_sample_start
 * @grid:
 * @sample_start:
 *
 */
void
gnome_db_grid_set_sample_start (GnomeDbGrid *grid, gint sample_start)
{
	g_return_if_fail (grid && IS_GNOME_DB_GRID (grid));
	g_return_if_fail (grid->priv);
	
	gnome_db_data_proxy_set_sample_start (grid->priv->proxy, sample_start);
}


/*
 *
 * Modification buttons (Commit changes, Reset grid, New entry, Delete)
 *
 */
static void action_new_cb (GtkAction *action, GnomeDbGrid *grid);
static void action_delete_cb (GtkAction *action, GnomeDbGrid *grid);
static void action_undelete_cb (GtkAction *action, GnomeDbGrid *grid);
static void action_commit_cb (GtkAction *action, GnomeDbGrid *grid);
static void action_reset_cb (GtkAction *action, GnomeDbGrid *grid);
static void action_first_chunck_cb (GtkAction *action, GnomeDbGrid *grid);
static void action_prev_chunck_cb (GtkAction *action, GnomeDbGrid *grid);
static void action_next_chunck_cb (GtkAction *action, GnomeDbGrid *grid);
static void action_last_chunck_cb (GtkAction *action, GnomeDbGrid *grid);


static GtkActionEntry ui_actions[] = {
	{ "ActionNew", GTK_STOCK_NEW, "_New", NULL, "Create a new data entry", G_CALLBACK (action_new_cb)},
	{ "ActionDelete", GTK_STOCK_DELETE, "_Delete", NULL, "Delete the selected entry", G_CALLBACK (action_delete_cb)},
	{ "ActionUndelete", GTK_STOCK_UNDELETE, "_Undelete", NULL, "Cancels the deletion of the selected entry", 
	  G_CALLBACK (action_undelete_cb)},
	{ "ActionCommit", GTK_STOCK_SAVE, "_Commit", NULL, "Commit the latest changes", G_CALLBACK (action_commit_cb)},
	{ "ActionReset", GTK_STOCK_REFRESH, "_Reset", NULL, "Reset the data", G_CALLBACK (action_reset_cb)},
	{ "ActionFirstChunck", GTK_STOCK_GOTO_FIRST, "_First chunck", NULL, "Go to first chunck of records", 
	  G_CALLBACK (action_first_chunck_cb)},
	{ "ActionLastChunck", GTK_STOCK_GOTO_LAST, "_Last chunck", NULL, "Go to last chunck of records", 
	  G_CALLBACK (action_last_chunck_cb)},
	{ "ActionPrevChunck", GTK_STOCK_GO_BACK, "_Previous chunck", NULL, "Go to previous chunck of records", 
	  G_CALLBACK (action_prev_chunck_cb)},
	{ "ActionNextChunck", GTK_STOCK_GO_FORWARD, "Ne_xt chunck", NULL, "Go to next chunck of records",
	  G_CALLBACK (action_next_chunck_cb)}
};

static const gchar *ui_actions_info =
	"<ui>"
	"  <toolbar  name='ToolBar'>"
	"    <toolitem action='ActionNew'/>"
	"    <toolitem action='ActionDelete'/>"
	"    <toolitem action='ActionUndelete'/>"
	"    <toolitem action='ActionCommit'/>"
	"    <toolitem action='ActionReset'/>"
	"    <separator/>"
	"    <toolitem action='ActionFirstChunck'/>"
	"    <toolitem action='ActionPrevChunck'/>"
	"    <toolitem action='ActionNextChunck'/>"
	"    <toolitem action='ActionLastChunck'/>"
	"  </toolbar>"
	"</ui>";


static GtkWidget *
modif_buttons_make (GnomeDbGrid *grid)
{
	GtkActionGroup *actions;
	GtkUIManager *ui;
	GtkWidget *table, *wid;

	table = gtk_table_new (1, 2, FALSE);

	/* action buttons */
	actions = gtk_action_group_new ("Actions");
	grid->priv->actions_group = actions;

	gtk_action_group_add_actions (actions, ui_actions, G_N_ELEMENTS (ui_actions), grid);

	ui = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group (ui, actions, 0);
	gtk_ui_manager_add_ui_from_string (ui, ui_actions_info, -1, NULL);
	grid->priv->uimanager = ui;
	grid->priv->widget_actions = gtk_ui_manager_get_widget (ui, "/ToolBar");
	g_object_set (G_OBJECT (grid->priv->widget_actions), "toolbar-style", GTK_TOOLBAR_ICONS, NULL);
	gtk_table_attach_defaults (GTK_TABLE (table), grid->priv->widget_actions, 0, 1, 0, 1);
	gtk_widget_show (grid->priv->widget_actions);

	/* samples counter */
	wid = gtk_label_new ("? - ? / ?");
	gtk_widget_show (wid);
	grid->priv->widget_current_sample = wid;
	gtk_table_attach (GTK_TABLE (table), wid, 1, 2, 0, 1, 0, 0, 5, 0);

	return table;
}

static void
modif_buttons_update (GnomeDbGrid *grid)
{
	GtkTreeSelection *select;
	GtkTreeIter iter;
	GtkTreeModel *model;
	gboolean changed = FALSE;
	gint has_selection;
	GtkSelectionMode sel_mode;
	gchar *str;
	GtkAction *action;
	gint row_count = 0;
	gboolean params_valid;
	GnomeDbDataSet *params;

	gint sample_size, sample_first_row, sample_last_row;

	params = gnome_db_data_model_get_params (grid->priv->data_model);
	params_valid = params ? gnome_db_data_set_is_valid (params) : TRUE;

	if (!grid->priv->treeview || !grid->priv->proxy) {
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionDelete");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionUndelete");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionCommit");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionReset");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionNew");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionDelete");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionFirstChunck");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionPrevChunck");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionNextChunck");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionLastChunck");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);

		gtk_label_set_text (GTK_LABEL (grid->priv->widget_current_sample), "? - ? / ?");
		gtk_widget_set_sensitive (grid->priv->widget_current_sample, FALSE);

		return;
	}

	changed = gnome_db_data_proxy_has_been_modified (grid->priv->proxy);
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
	sel_mode = gtk_tree_selection_get_mode (select);
	if (sel_mode == GTK_SELECTION_MULTIPLE) {
		has_selection = gtk_tree_selection_count_selected_rows (select);
		if (has_selection == 1) {
			GList *sel_rows;

			sel_rows = gtk_tree_selection_get_selected_rows (select, &model);
			has_selection = gtk_tree_model_get_iter (model, &iter, (GtkTreePath *) (sel_rows->data)) ? 1 : 0;
			g_list_foreach (sel_rows, (GFunc) gtk_tree_path_free, NULL);
			g_list_free (sel_rows);
		}
	}
	else
		has_selection = gtk_tree_selection_get_selected (select, &model, &iter) ? 1 : 0;

	/* FIXME: if the working mode is to allow the edition of one line at a time, then 
	 * we should check that the corresponding INSERT query is valid (or UPDATE, if INSERT is valid then
	 * UPDATE will also be valid, by construction) */
	action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionCommit");
	g_object_set (G_OBJECT (action), "sensitive", changed ? TRUE : FALSE, NULL);

	action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionReset");
	g_object_set (G_OBJECT (action), "sensitive", params_valid, NULL);

	action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionNew");
	g_object_set (G_OBJECT (action), "sensitive", 
		      ! gnome_db_data_proxy_is_read_only (grid->priv->proxy) && params_valid, NULL);

	if ((has_selection == 0) || gnome_db_data_proxy_is_read_only (grid->priv->proxy)) {
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionDelete");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionUndelete");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
	}
	else {
		if (has_selection > 1) {
			action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionDelete");
			g_object_set (G_OBJECT (action), "sensitive", TRUE, NULL);
			
			action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionUndelete");
			g_object_set (G_OBJECT (action), "sensitive", TRUE, NULL);
		}
		else {
			gboolean to_be_deleted = FALSE;
			gtk_tree_model_get (model, &iter, PROXY_COL_TO_DELETE, &to_be_deleted, -1);
			
			action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionDelete");
			g_object_set (G_OBJECT (action), "sensitive", !to_be_deleted, NULL);
			
			action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionUndelete");
			g_object_set (G_OBJECT (action), "sensitive", to_be_deleted, NULL);				
		}
	}

	sample_size = gnome_db_data_proxy_get_sample_size (grid->priv->proxy);
	row_count = gda_data_model_get_n_rows (GDA_DATA_MODEL (grid->priv->data_model));
	sample_first_row = gnome_db_data_proxy_get_sample_start (grid->priv->proxy);
	sample_last_row = gnome_db_data_proxy_get_sample_end (grid->priv->proxy);

	if (sample_size > 0) {
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionFirstChunck");
		g_object_set (G_OBJECT (action), "sensitive", 
			      sample_first_row > 0 ? TRUE : FALSE, 
			      "visible", grid->priv->mode & GNOME_DB_ACTION_NAVIGATION_ARROWS ? TRUE : FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionPrevChunck");
		g_object_set (G_OBJECT (action), "sensitive", 
			      sample_first_row > 0 ? TRUE : FALSE,  
			      "visible", grid->priv->mode & GNOME_DB_ACTION_NAVIGATION_ARROWS ? TRUE : FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionNextChunck");
		g_object_set (G_OBJECT (action), "sensitive", 
			      sample_last_row < row_count - 1 ? TRUE : FALSE,
			      "visible", grid->priv->mode & GNOME_DB_ACTION_NAVIGATION_ARROWS ? TRUE : FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionLastChunck");
		g_object_set (G_OBJECT (action), "sensitive", 
			      sample_last_row < row_count - 1 ? TRUE : FALSE,
			      "visible", grid->priv->mode & GNOME_DB_ACTION_NAVIGATION_ARROWS ? TRUE : FALSE, NULL);
	}
	else {
		/* disabled the chuncks navigation */
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionFirstChunck");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionPrevChunck");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionNextChunck");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);
		action = gtk_ui_manager_get_action (grid->priv->uimanager, "/ToolBar/ActionLastChunck");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);
	}
	
	if (sample_last_row == 0)
		str = g_strdup_printf (_("0 - 0 / 0"));
	else
		str = g_strdup_printf ("%d - %d / %d", sample_first_row + 1, sample_last_row + 1, row_count);
	gtk_label_set_text (GTK_LABEL (grid->priv->widget_current_sample), str);
	g_free (str);
	gtk_widget_set_sensitive (grid->priv->widget_current_sample, TRUE);
	if (grid->priv->mode & GNOME_DB_ACTION_NAVIGATION_ARROWS)
		gtk_widget_show (grid->priv->widget_current_sample);
	else
		gtk_widget_hide (grid->priv->widget_current_sample);

	gtk_ui_manager_ensure_update (grid->priv->uimanager); 
}

static void
action_new_cb (GtkAction *action, GnomeDbGrid *grid)
{
	GtkTreeIter iter;
	GtkTreePath *path;
	gnome_db_data_proxy_append (grid->priv->proxy, &iter);

	path = gtk_tree_model_get_path (GTK_TREE_MODEL (grid->priv->proxy), &iter);
	gtk_tree_view_set_cursor (GTK_TREE_VIEW (grid->priv->treeview), path, NULL, FALSE);
	gtk_tree_path_free (path);
}

static void
action_delete_cb (GtkAction *action, GnomeDbGrid *grid)
{
	GtkTreeIter iter;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GList *sel_rows, *cur_row;
	
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
	sel_rows = gtk_tree_selection_get_selected_rows (select, &model);
	cur_row = sel_rows;
	while (cur_row) {
		gtk_tree_model_get_iter (model, &iter, (GtkTreePath *) (cur_row->data));
		gnome_db_data_proxy_delete (grid->priv->proxy, &iter);
		cur_row = g_list_next (cur_row);
	}
	g_list_foreach (sel_rows, (GFunc) gtk_tree_path_free, NULL);
	g_list_free (sel_rows);
}

static void
action_undelete_cb (GtkAction *action, GnomeDbGrid *grid)
{
	GtkTreeIter iter;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GList *sel_rows, *cur_row;
	
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->treeview));
	sel_rows = gtk_tree_selection_get_selected_rows (select, &model);
	cur_row = sel_rows;
	while (cur_row) {
		gtk_tree_model_get_iter (model, &iter, (GtkTreePath *) (cur_row->data));
		gnome_db_data_proxy_undelete (grid->priv->proxy, &iter);
		cur_row = g_list_next (cur_row);
	}
	g_list_foreach (sel_rows, (GFunc) gtk_tree_path_free, NULL);
	g_list_free (sel_rows);
}

static void
action_commit_cb (GtkAction *action, GnomeDbGrid *grid)
{
	gnome_db_data_proxy_commit_all (grid->priv->proxy, NULL);
}

static void
action_reset_cb (GtkAction *action, GnomeDbGrid *grid)
{
	GnomeDbDataModel *model;

	/* reset the proxy */
	gnome_db_data_proxy_reset_all (grid->priv->proxy);

	/* refresh the data if possible */
	model = gnome_db_data_proxy_get_model (grid->priv->proxy);
	if (gnome_db_data_model_get_status (model) & GNOME_DB_DATA_MODEL_CAN_BE_REFRESHED)
		gnome_db_data_model_refresh (model, NULL);
}

static void
action_first_chunck_cb (GtkAction *action, GnomeDbGrid *grid)
{
	gnome_db_grid_set_sample_start (grid, 0);	
}

static void
action_prev_chunck_cb (GtkAction *action, GnomeDbGrid *grid)
{
	gint sample_size, sample_start;

	sample_size = gnome_db_data_proxy_get_sample_size (grid->priv->proxy);
	if (sample_size > 0) {
		sample_start = gnome_db_data_proxy_get_sample_start (grid->priv->proxy);
		sample_start -= sample_size;
		gnome_db_data_proxy_set_sample_start (grid->priv->proxy, sample_start);
	}
}

static void
action_next_chunck_cb (GtkAction *action, GnomeDbGrid *grid)
{
	gint sample_size, sample_start;

	sample_size = gnome_db_data_proxy_get_sample_size (grid->priv->proxy);
	if (sample_size > 0) {
		sample_start = gnome_db_data_proxy_get_sample_start (grid->priv->proxy);
		sample_start += sample_size;
		gnome_db_data_proxy_set_sample_start (grid->priv->proxy, sample_start);
	}
}

static void
action_last_chunck_cb (GtkAction *action, GnomeDbGrid *grid)
{
	gnome_db_data_proxy_set_sample_start (grid->priv->proxy, G_MAXINT);
}



/*
 * GnomeDbDataWidget interface
 */
static void
gnome_db_grid_set_mode (GnomeDbDataWidget *iface, guint mode)
{
	GnomeDbGrid *grid;

	g_return_if_fail (iface && IS_GNOME_DB_GRID (iface));
	grid = GNOME_DB_GRID (iface);
	g_return_if_fail (grid->priv);

	grid->priv->mode = mode;

	/* updating the various possible actions */
	modif_buttons_update (grid);
}


static void
gnome_db_grid_set_column_editable (GnomeDbDataWidget *iface, gint column, gboolean editable)
{
	GnomeDbGrid *grid;
	GnomeDbParameter *param;
	ColumnData *column_data;
	GnomeDbDataSetNode *node;

	g_return_if_fail (iface && IS_GNOME_DB_GRID (iface));
	grid = GNOME_DB_GRID (iface);
	g_return_if_fail (grid->priv);
	
	if (grid->priv->data_model) {
		editable = editable && !gnome_db_data_proxy_is_read_only (grid->priv->proxy);
		param = gnome_db_data_model_get_param_at_column (grid->priv->data_model, grid->priv->data_set, column);
		node = gnome_db_data_set_find_node_for_param (grid->priv->data_set, param);
		g_return_if_fail (node);
		column_data = get_column_data (grid, node);
		g_return_if_fail (column_data);
		
		if (editable && !gnome_db_data_proxy_is_read_only (grid->priv->proxy))
			column_data->data_locked = FALSE;
		else
			column_data->data_locked = TRUE;
		
		g_object_set (G_OBJECT (column_data->data_cell), "editable", !column_data->data_locked, NULL);
	}

	grid->priv->default_column_editable = editable;
}

static void
gnome_db_grid_show_column_actions (GnomeDbDataWidget *iface, gint column, gboolean show_actions)
{
	GnomeDbParameter *param;
	GnomeDbDataSetNode *cnode;
	ColumnData *cdata;
	GnomeDbGrid *grid;

	g_return_if_fail (iface && IS_GNOME_DB_GRID (iface));
	grid = GNOME_DB_GRID (iface);
	g_return_if_fail (grid->priv);

	if (column < 0) {
		/* apply to all the fields */
		gnome_db_grid_show_global_actions (iface, show_actions);
		return;
	}

	param = gnome_db_data_model_get_param_at_column (grid->priv->data_model, grid->priv->data_set, column);
	g_return_if_fail (param);
	
	cnode = gnome_db_data_set_find_node_for_param (grid->priv->data_set, param);
	g_return_if_fail (cnode);

	cdata = get_column_data (grid, cnode);
	g_return_if_fail (cdata);

	if (show_actions != cdata->info_shown) {
		cdata->info_shown = show_actions;
		g_object_set (G_OBJECT (cdata->info_cell), "visible", cdata->info_shown, NULL);
	}
}

static void
gnome_db_grid_col_set_show (GnomeDbDataWidget *iface, gint column, gboolean shown)
{
	GnomeDbGrid *grid;
	gint pos = -1;
	GtkTreeViewColumn *viewcol;
	GnomeDbDataSetNode *cnode;
	GnomeDbParameter *param;

	g_return_if_fail (iface && IS_GNOME_DB_GRID (iface));
	grid = GNOME_DB_GRID (iface);
	g_return_if_fail (grid->priv);

	param = gnome_db_data_model_get_param_at_column (grid->priv->data_model, grid->priv->data_set, column);
	g_return_if_fail (param);

	cnode = gnome_db_data_set_find_node_for_param (grid->priv->data_set, param);	
	pos = g_slist_index (grid->priv->data_set->nodes, cnode);	
	g_assert (pos >= 0);

	viewcol = gtk_tree_view_get_column (GTK_TREE_VIEW (grid->priv->treeview), pos);

	/* Sets the column's visibility */
	gtk_tree_view_column_set_visible (viewcol, shown);
}


static void
gnome_db_grid_show_global_actions (GnomeDbDataWidget *iface, gboolean show_actions)
{
	GnomeDbGrid *grid;

	g_return_if_fail (iface && IS_GNOME_DB_GRID (iface));
	grid = GNOME_DB_GRID (iface);
	g_return_if_fail (grid->priv);

	grid->priv->default_show_global_actions = show_actions;
	if (grid->priv->widget_group) {
		if (show_actions)
			gtk_widget_show (grid->priv->widget_group);
		else
			gtk_widget_hide (grid->priv->widget_group);
	}
}

static GtkActionGroup *
gnome_db_grid_get_actions_group (GnomeDbDataWidget *iface)
{
	GnomeDbGrid *grid;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_GRID (iface), NULL);
	grid = GNOME_DB_GRID (iface);
	g_return_val_if_fail (grid->priv, NULL);

	return grid->priv->actions_group;
}

static GnomeDbDataSet *
gnome_db_grid_widget_get_params_set (GnomeDbDataWidget *iface)
{
	GnomeDbGrid *grid;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_GRID (iface), NULL);
	grid = GNOME_DB_GRID (iface);
	g_return_val_if_fail (grid->priv, NULL);

	return gnome_db_data_model_get_params (grid->priv->data_model);
}

static GnomeDbDataSet *
gnome_db_grid_widget_get_data_set (GnomeDbDataWidget *iface)
{
	GnomeDbGrid *grid;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_GRID (iface), NULL);
	grid = GNOME_DB_GRID (iface);
	g_return_val_if_fail (grid->priv, NULL);

	return grid->priv->data_set;
}


static GdaDataModel *
gnome_db_grid_widget_get_gda_model (GnomeDbDataWidget *iface)
{
	GnomeDbGrid *grid;
	g_return_val_if_fail (iface && IS_GNOME_DB_GRID (iface), NULL);
	grid = GNOME_DB_GRID (iface);
	g_return_val_if_fail (grid->priv, NULL);

	/* warning: here we are not returning the *original* GdaDataModel, but one one encapsulated
	 * by the GnomeDbDataModel; maybe we'll have to change this */

	return GDA_DATA_MODEL (grid->priv->data_model);
}

static void
gnome_db_grid_widget_set_gda_model (GnomeDbDataWidget *iface, GnomeDbDict *dict, GdaDataModel *model)
{
	GnomeDbGrid *grid;
	GnomeDbDataModel *gmodel;
	GnomeDbDataProxy *proxy;

	g_return_if_fail (iface && IS_GNOME_DB_GRID (iface));
	grid = GNOME_DB_GRID (iface);
	g_return_if_fail (grid->priv);
	g_return_if_fail (!dict || IS_GNOME_DB_DICT (dict));
	
	gnome_db_grid_clean (grid);
	gmodel = GNOME_DB_DATA_MODEL (gnome_db_model_wrapper_new (ASSERT_DICT (dict), model));
	proxy = GNOME_DB_DATA_PROXY (gnome_db_data_proxy_new (gmodel));
	g_object_unref (gmodel);
	grid->priv->data_model = gmodel;
	grid->priv->proxy = proxy;
	grid->priv->data_set = gnome_db_data_model_get_new_data_set (gmodel);

	if (grid->priv->default_column_editable)
		grid->priv->default_column_editable = !gnome_db_data_proxy_is_read_only (grid->priv->proxy);
	gnome_db_grid_initialize (grid);
}

static void
gnome_db_grid_widget_set_title (GnomeDbDataWidget *iface, const gchar *title)
{
	GnomeDbGrid *grid;
	g_return_if_fail (iface && IS_GNOME_DB_GRID (iface));
	grid = GNOME_DB_GRID (iface);
	g_return_if_fail (grid->priv);

	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (grid->priv->title), title);
	if (title && *title)
		gtk_widget_show (grid->priv->title);
	else
		gtk_widget_hide (grid->priv->title);	
}

static void
gnome_db_grid_widget_set_show_title (GnomeDbDataWidget *iface, gboolean show)
{
	GnomeDbGrid *grid;
	g_return_if_fail (iface && IS_GNOME_DB_GRID (iface));
	grid = GNOME_DB_GRID (iface);
	g_return_if_fail (grid->priv);

	if (show)
		gtk_widget_show (grid->priv->title);
	else
		gtk_widget_hide (grid->priv->title);
}

static void
gnome_db_grid_widget_set_title_icon_from_stock (GnomeDbDataWidget *iface, const gchar *stock_id, GtkIconSize size)
{
	GnomeDbGrid *grid;
	g_return_if_fail (iface && IS_GNOME_DB_GRID (iface));
	grid = GNOME_DB_GRID (iface);
	g_return_if_fail (grid->priv);
	
	if (stock_id && *stock_id) {
		gnome_db_gray_bar_set_icon_from_stock (GNOME_DB_GRAY_BAR (grid->priv->title),
						       stock_id, size);
		gnome_db_gray_bar_set_show_icon (GNOME_DB_GRAY_BAR (grid->priv->title), TRUE);
	}
	else
		gnome_db_gray_bar_set_show_icon (GNOME_DB_GRAY_BAR (grid->priv->title), FALSE);
}
