/* gnome-db-selector.c
 *
 * Copyright (C) 2002 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include <libgnomedb/libgnomedb.h>
#include "gnome-db-selector.h"
#include "marshal.h"

#include "gnome-db-table.h"
#include "gnome-db-query.h"

#include "gnome-db-selector-private.h"

#include "sel-data-types.h"
#include "sel-functions.h"
#include "sel-aggregates.h"
#include "sel-tables.h"
#include "sel-onetable.h"
#include "sel-queries.h"
#include "sel-onequery.h"
#include "sel-graphs.h"
#include "sel-forms.h"

static void gnome_db_selector_class_init (GnomeDbSelectorClass * class);
static void gnome_db_selector_init (GnomeDbSelector * wid);
static void gnome_db_selector_dispose (GObject   * object);

static gboolean move_iter_to_next_leaf (GtkTreeModel *model, GtkTreeIter *iter);
static gboolean set_iter_position (GtkTreeModel *model, GSList *obj_list, gpointer object, gpointer missing, GtkTreeIter *iter);
static void     model_store_data (Module *module, GtkTreeIter *iter);


enum
{
	C_SELECTION_CHANGED,
	LAST_SIGNAL
};

static gint gnome_db_selector_signals[LAST_SIGNAL] = { 0 };

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;


GType
gnome_db_selector_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbSelectorClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_selector_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbSelector),
			0,
			(GInstanceInitFunc) gnome_db_selector_init
		};		
		
		type = g_type_register_static (GTK_TYPE_VBOX, "GnomeDbSelector", &info, 0);
	}

	return type;
}

static void
gnome_db_selector_class_init (GnomeDbSelectorClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	gnome_db_selector_signals[C_SELECTION_CHANGED] =
		g_signal_new ("selection_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbSelectorClass, selection_changed),
			      NULL, NULL,
			      marshal_VOID__OBJECT, G_TYPE_NONE, 1,
			      G_TYPE_OBJECT);

	class->selection_changed = NULL;
	object_class->dispose = gnome_db_selector_dispose;
}

static void
gnome_db_selector_init (GnomeDbSelector * wid)
{
	wid->priv = g_new0 (GnomeDbSelectorPriv, 1);
	wid->priv->dict = NULL;
	wid->priv->mode = 0;
	wid->priv->columns = 0;
	wid->priv->ref_object = NULL;
	wid->priv->treeview = NULL;
	wid->priv->headers_visible = TRUE;
	wid->priv->model = NULL;
	wid->priv->modules = NULL;
	wid->priv->selection = NULL;
}

static gboolean gnome_db_selector_initialize (GnomeDbSelector *mgsel, GObject *ref_object, gboolean keep_model, gboolean keep_columns);


static void object_weak_notify (GnomeDbSelector *mgsel, GObject *obj);
/**
 * gnome_db_selector_new
 * @dict: a #GnomeDbDict object
 * @ref_object: a #GObject object, or %NULL if none is required
 * @mode: an OR'ed value of the possible items to display in the widget
 * @columns: an OR'ed value describing which columns will be displayed
 *
 * Creates a new #GnomeDbSelector widget.
 *
 * @ref_object must be %NULL if @mode implies a list where the displayed items
 * are all fetched from within @dict (namely GNOME_DB_SELECTOR_DATA_TYPES, GNOME_DB_SELECTOR_FUNCTIONS, 
 * GNOME_DB_SELECTOR_AGGREGATES, GNOME_DB_SELECTOR_TABLES, GNOME_DB_SELECTOR_QUERIES). In this case @ref_object
 * will simply be ignored.
 *
 * @ref_object cannot be %NULL if @mode implies a list where the displayed items depend
 * on a specific object, namely GNOME_DB_SELECTOR_FIELDS (when not used in conjunction with GNOME_DB_SELECTOR_TABLES),
 * GNOME_DB_SELECTOR_TARGETS, GNOME_DB_SELECTOR_JOINS, GNOME_DB_SELECTOR_QVIS_FIELDS, GNOME_DB_SELECTOR_QALL_FIELDS and
 * GNOME_DB_SELECTOR_SUB_QUERIES (when not used in conjunction with GNOME_DB_SELECTOR_QUERIES).
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_selector_new (GnomeDbDict *dict, GObject *ref_object, gulong mode, gulong columns)
{
	GObject    *obj;
	GnomeDbSelector *mgsel;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);
		
	obj = g_object_new (GNOME_DB_SELECTOR_TYPE, NULL);
	mgsel = GNOME_DB_SELECTOR (obj);

	mgsel->priv->dict = ASSERT_DICT (dict);
	mgsel->priv->mode = mode;
	mgsel->priv->columns = columns;

	g_object_weak_ref (G_OBJECT (mgsel->priv->dict),
			   (GWeakNotify) object_weak_notify, mgsel);

	if (gnome_db_selector_initialize (mgsel, ref_object, FALSE, FALSE) && ref_object) {
		/* add a weak ref on that object */
		mgsel->priv->ref_object = ref_object;
		g_object_weak_ref (G_OBJECT (mgsel->priv->ref_object),
				   (GWeakNotify) object_weak_notify, mgsel);
	}

	return GTK_WIDGET (obj);
}

static void
object_weak_notify (GnomeDbSelector *mgsel, GObject *obj)
{
	GSList *list;

	/* modules */
	list = mgsel->priv->modules;
	while (list) {
		(MODULE (list->data)->free) (MODULE (list->data));
		g_free (list->data);
		list = g_slist_next (list);
	}
	g_slist_free (mgsel->priv->modules);
	mgsel->priv->modules = NULL;	

	/* Clear the model */
	if (mgsel->priv->model)
		gtk_tree_store_clear (GTK_TREE_STORE (mgsel->priv->model));
	
	if (obj == (GObject*) mgsel->priv->dict)
		/* Tell that we don't need to weak unref the GnomeDbDict */
		mgsel->priv->dict = NULL;
	if (obj == (GObject *) mgsel->priv->ref_object)
		/* Tell that we don't need to weak unref the Ref object */
		mgsel->priv->ref_object = NULL;
}

static void tree_selection_changed_cb (GtkTreeSelection *select, GnomeDbSelector *mgsel);
static void
gnome_db_selector_dispose (GObject *object)
{
	GnomeDbSelector *mgsel;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_SELECTOR (object));
	mgsel = GNOME_DB_SELECTOR (object);

	if (mgsel->priv) {
		GSList *list;
		GtkTreeSelection *select;

		/* signals, etc */
		if (mgsel->priv->model) {
			g_object_unref (G_OBJECT (mgsel->priv->model));
			mgsel->priv->model = NULL;
		}

		if (mgsel->priv->treeview) {
			select = gtk_tree_view_get_selection (GTK_TREE_VIEW (mgsel->priv->treeview));
			g_signal_handlers_disconnect_by_func (G_OBJECT (select),
							      G_CALLBACK (tree_selection_changed_cb), mgsel);
		}

		/* modules */
		list = mgsel->priv->modules;
		while (list) {
			(MODULE (list->data)->free) (MODULE (list->data));
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (mgsel->priv->modules);
		mgsel->priv->modules = NULL;

		/* Weak unref the GnomeDbDict if necessary */
		if (mgsel->priv->dict)
			g_object_weak_unref (G_OBJECT (mgsel->priv->dict),
					     (GWeakNotify) object_weak_notify, mgsel);
		/* Weak unref the Ref object if necessary */
		if (mgsel->priv->ref_object)
			g_object_weak_unref (G_OBJECT (mgsel->priv->ref_object),
					     (GWeakNotify) object_weak_notify, mgsel);
		

		/* the private area itself */
		g_free (mgsel->priv);
		mgsel->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void tree_value_set_func (GtkTreeViewColumn *tree_column,
				 GtkCellRenderer *cell,
				 GtkTreeModel *model,
				 GtkTreeIter *iter,
				 gpointer user_data);
/*
 * create the model, the columns and the treeview widget itself
 */
static gboolean
gnome_db_selector_initialize (GnomeDbSelector *mgsel, GObject *ref_object, gboolean keep_model, gboolean keep_columns)
{
	GtkTreeModel *model;
	guint nb_modules = 0;
	guint i;
	GSList *list;
	gulong mode = 1;
	gboolean need_header;

	gtk_container_set_border_width (GTK_CONTAINER (mgsel), 0);

	/*
	 * basic modes checks
	 */
	if (!(mgsel->priv->mode & GNOME_DB_SELECTOR_TABLES) && 
	    (mgsel->priv->mode & GNOME_DB_SELECTOR_FIELDS) && 
	    (!ref_object || !IS_GNOME_DB_TABLE (ref_object))) {
		return FALSE;
	}
	
	if (!(mgsel->priv->mode & GNOME_DB_SELECTOR_QUERIES) &&
	    ((mgsel->priv->mode & GNOME_DB_SELECTOR_TARGETS) ||
	     (mgsel->priv->mode & GNOME_DB_SELECTOR_TARGETS_CTS) ||
	     (mgsel->priv->mode & GNOME_DB_SELECTOR_JOINS) ||
	     (mgsel->priv->mode & GNOME_DB_SELECTOR_QVIS_FIELDS) ||
	     (mgsel->priv->mode & GNOME_DB_SELECTOR_QALL_FIELDS) ||
	     (mgsel->priv->mode & GNOME_DB_SELECTOR_SUB_QUERIES)) && 
	    (!ref_object || !IS_GNOME_DB_QUERY (ref_object))) {
		return FALSE;
	}


	/* 
	 * safe fallback. We set keep_columns to FALSE because otherwise the code needs to be modified to
	 * take into account the columns title which can change, and there is little to gain in doing so
	 * compared to create new coluns.
	 */
	if (!mgsel->priv->treeview) {
		keep_model = FALSE;
		keep_columns = FALSE;
	}
	keep_columns = FALSE;
	
	/*
	 * Global computations
	 */
	/* counting the number of top level modules */
	for (i = 0 ; i <= 4 ; i++) {
		if (mgsel->priv->mode & mode)
			nb_modules++;	
		mode <<= 1;
	}
	need_header = nb_modules > 1 ? TRUE : FALSE;


	/*
	 * GtkTreeStore model part
	 */
	if (!keep_model) {
		/* clean any previous model and associated data */
		if (mgsel->priv->model) {
			gtk_tree_store_clear (GTK_TREE_STORE (mgsel->priv->model));
			g_object_unref (G_OBJECT (mgsel->priv->model));
			mgsel->priv->model = NULL;

			if (mgsel->priv->treeview)
				gtk_tree_view_set_model (mgsel->priv->treeview, NULL);
		}
		
		if (mgsel->priv->modules) {
			list = mgsel->priv->modules;
			while (list) {
				(MODULE (list->data)->free) (MODULE (list->data));
				g_free (list->data);
				list = g_slist_next (list);
			}
			g_slist_free (mgsel->priv->modules);
			mgsel->priv->modules = NULL;
		}
		
		/* model creation */
		model = GTK_TREE_MODEL (gtk_tree_store_new (NUM_COLUMNS, 
							    G_TYPE_STRING,    /* NAME_COLUMN */
							    G_TYPE_STRING,    /* OWNER_COLUMN */
							    G_TYPE_STRING,    /* DESCR_COLUMN */
							    G_TYPE_STRING,    /* EXTRA1_COLUMN */
							    G_TYPE_STRING,    /* EXTRA2_COLUMN */
							    G_TYPE_BOOLEAN,   /* EXTRA3_COLUMN */
							    G_TYPE_BOOLEAN,   /* EXTRA4_COLUMN */
							    G_TYPE_STRING,    /* EXTRA5_COLUMN */
							    G_TYPE_STRING,    /* EXTRA6_COLUMN */
							    G_TYPE_STRING,    /* EXTRA7_COLUMN */
							    G_TYPE_BOOLEAN,   /* EXTRA_END_COLUMN */
							    G_TYPE_POINTER,   /* OBJ_COLUMN */
							    GDK_TYPE_PIXBUF,  /* PIXBUF_COLUMN */
							    G_TYPE_INT,       /* CONTENTS_COLUMN */
							    G_TYPE_POINTER,   /* SUB_MODULE_COLUMN */
							    G_TYPE_BOOLEAN)); /* ERROR_COLUMN marker column */
		mgsel->priv->model = model;

		/* 'top level' modules selection (not using ref_object) */
		if (mgsel->priv->mode & GNOME_DB_SELECTOR_DATA_TYPES) {
			Module *module = sel_module_data_types_new (mgsel, need_header, NULL, NULL);
			mgsel->priv->modules = g_slist_append (mgsel->priv->modules, module);
		}
		
		if (mgsel->priv->mode & GNOME_DB_SELECTOR_FUNCTIONS) {
			Module *module = sel_module_functions_new (mgsel, need_header, NULL, NULL);
			mgsel->priv->modules = g_slist_append (mgsel->priv->modules, module);
		}
		
		if (mgsel->priv->mode & GNOME_DB_SELECTOR_AGGREGATES) {
			Module *module = sel_module_aggregates_new (mgsel, need_header, NULL, NULL);
			mgsel->priv->modules = g_slist_append (mgsel->priv->modules, module);
		}
		
		if (mgsel->priv->mode & GNOME_DB_SELECTOR_TABLES) {
			Module *module = sel_module_tables_new (mgsel, need_header, NULL, NULL);
			mgsel->priv->modules = g_slist_append (mgsel->priv->modules, module);
		}
		
		if (mgsel->priv->mode & GNOME_DB_SELECTOR_QUERIES) {
			Module *module = sel_module_queries_new (mgsel, need_header, NULL, NULL);
			mgsel->priv->modules = g_slist_append (mgsel->priv->modules, module);
		}

		if (mgsel->priv->mode & GNOME_DB_SELECTOR_GRAPHS) {
			Module *module = sel_module_graphs_new (mgsel, need_header, NULL, NULL);
			mgsel->priv->modules = g_slist_append (mgsel->priv->modules, module);
		}

		if (mgsel->priv->mode & GNOME_DB_SELECTOR_FORMS) {
			Module *module = sel_module_forms_new (mgsel, need_header, NULL, NULL);
			mgsel->priv->modules = g_slist_append (mgsel->priv->modules, module);
		}

		if (!mgsel->priv->modules) {
			/* 'low level' modules selection (making use of ref_object) */
			if (mgsel->priv->mode & GNOME_DB_SELECTOR_FIELDS) {
				Module *module = sel_module_onetable_new (mgsel, need_header, NULL, 
									  (gpointer) ref_object);
				mgsel->priv->modules = g_slist_append (mgsel->priv->modules, module);
			}

			if ((mgsel->priv->mode & GNOME_DB_SELECTOR_TARGETS) ||
			    (mgsel->priv->mode & GNOME_DB_SELECTOR_TARGETS_CTS) ||
			    (mgsel->priv->mode & GNOME_DB_SELECTOR_JOINS) ||
			    (mgsel->priv->mode & GNOME_DB_SELECTOR_QVIS_FIELDS) ||
			    (mgsel->priv->mode & GNOME_DB_SELECTOR_QALL_FIELDS) ||
			    (mgsel->priv->mode & GNOME_DB_SELECTOR_SUB_QUERIES)) {
				Module *module = sel_module_onequery_new (mgsel, need_header, NULL, 
									  (gpointer) ref_object);
				mgsel->priv->modules = g_slist_append (mgsel->priv->modules, module);
			}
		}
	}
	else 
		model = mgsel->priv->model;

	/*
	 * TreeView Widget part
	 */
	if (!keep_columns) {
		GtkCellRenderer *renderer;
		GtkTreeViewColumn *column;
		GtkTreeSelection *select;
		GtkWidget *sw = NULL, *treeview;
		const gchar *str;

		/* get rid of any old tree view widget already here if necessary */
		if (mgsel->priv->treeview) {
			sw = g_object_get_data (G_OBJECT (mgsel->priv->treeview), "sw");
			gtk_widget_destroy (GTK_WIDGET (mgsel->priv->treeview));
			mgsel->priv->treeview = NULL;
		}

		/* widget */
		if (!sw) {
			sw = gtk_scrolled_window_new (NULL, NULL);
			gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
							GTK_POLICY_AUTOMATIC,
							GTK_POLICY_AUTOMATIC);
			gtk_box_pack_start (GTK_BOX (mgsel), sw, TRUE, TRUE, 0);
		}

		treeview = gtk_tree_view_new_with_model (model);
		gtk_container_add (GTK_CONTAINER (sw), treeview);
		mgsel->priv->treeview = GTK_TREE_VIEW (treeview);
		g_object_set_data (G_OBJECT (mgsel->priv->treeview), "sw", sw);
		gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (treeview), mgsel->priv->headers_visible);

		gtk_widget_set_size_request (sw, 100, 150);
		gtk_widget_show_all (sw); 


		/* adding columns */
		/* name column is always here */
		column = gtk_tree_view_column_new ();
		if (nb_modules == 1)
			str = (MODULE (mgsel->priv->modules->data)->col_name)
				(MODULE (mgsel->priv->modules->data), NAME_COLUMN);
		else
			str = _("Name");
		gtk_tree_view_column_set_title (column, str);
		renderer = gtk_cell_renderer_pixbuf_new ();
		gtk_tree_view_column_pack_start (column, renderer, FALSE);
		renderer = gtk_cell_renderer_text_new ();
		g_object_set (G_OBJECT (renderer), 
			      "strikethrough", TRUE,
			      "strikethrough-set", FALSE, NULL);
		gtk_tree_view_column_pack_start (column, renderer, TRUE);
		gtk_tree_view_column_set_cell_data_func (column,
							 renderer,
							 tree_value_set_func,
							 NULL, NULL);
		gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
		
		/* other columns */
		for (i = OWNER_COLUMN; i < EXTRA_END_COLUMN; i++) {
			GnomeDbSelectorColumn selcol = 0;
			switch (i) {
			case OWNER_COLUMN:
				selcol = GNOME_DB_SELECTOR_COLUMN_OWNER;
				str = _("Owner");
				break;
			case DESCR_COLUMN:
				selcol = GNOME_DB_SELECTOR_COLUMN_COMMENTS;
				str = _("Description");
				break;
			case EXTRA1_COLUMN:
				selcol = GNOME_DB_SELECTOR_COLUMN_TYPE;
				str = _("Type");
				break;
			case EXTRA2_COLUMN:
				selcol = GNOME_DB_SELECTOR_COLUMN_FIELD_LENGTH;
				str = _("Length");
				break;
			case EXTRA3_COLUMN:
				selcol = GNOME_DB_SELECTOR_COLUMN_FIELD_NNUL;
				str = _("Not NULL?");
				break;
			case EXTRA5_COLUMN:
				selcol = GNOME_DB_SELECTOR_COLUMN_FIELD_DEFAULT;
				str = _("Default value");
				break;
			case EXTRA6_COLUMN:
				selcol = GNOME_DB_SELECTOR_COLUMN_QFIELD_VALUE;
				str = _("Value");
				break;		
			case EXTRA7_COLUMN:
				selcol = GNOME_DB_SELECTOR_COLUMN_QFIELD_TYPE;
				str = _("Kind of field");
				break;		
			default:
				break;
			}

			if (mgsel->priv->columns & selcol) {
				if (nb_modules == 1) {
					const gchar *strtmp = (MODULE (mgsel->priv->modules->data)->col_name)
						(MODULE (mgsel->priv->modules->data), i);
					if (strtmp)
						str = strtmp;
				}
				
				switch (gtk_tree_model_get_column_type (model, i)) {
				case G_TYPE_STRING:
				default:
					renderer = gtk_cell_renderer_text_new ();
					g_object_set (G_OBJECT (renderer), 
						      "strikethrough", TRUE,
						      "strikethrough-set", FALSE, NULL);
					column = gtk_tree_view_column_new_with_attributes (str, renderer, 
											   "text", i, 
											   "strikethrough-set", ERROR_COLUMN, NULL);
					break;
				case G_TYPE_BOOLEAN:
					renderer = gtk_cell_renderer_toggle_new ();
					column = gtk_tree_view_column_new_with_attributes (str, renderer, 
											   "active", i, 
											   "visible", i+1, 
											   NULL);
					break;
				}
				
				gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
			}
		}

		/* selection signal */
		select = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
		gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
		g_signal_connect (G_OBJECT (select), "changed",
				  G_CALLBACK (tree_selection_changed_cb), mgsel);
	}
	else {
		gtk_tree_view_set_model (mgsel->priv->treeview, model);
		g_object_unref (G_OBJECT (model));
	}

	/* model population */
	list = mgsel->priv->modules;
	while (list) {
		(MODULE (list->data)->fill_model) (MODULE (list->data));
		list = g_slist_next (list);
	}

	return TRUE;
}

static void
tree_value_set_func (GtkTreeViewColumn *tree_column,
                     GtkCellRenderer *cell,
                     GtkTreeModel *model,
                     GtkTreeIter *iter,
                     gpointer user_data)
{
        GdkPixbuf *pixbuf;
        gchar *text;
        GList *renderers;
	gboolean error;

        gtk_tree_model_get (model, iter, PIXBUF_COLUMN, &pixbuf, NAME_COLUMN, &text, ERROR_COLUMN, &error, -1);
        renderers = gtk_tree_view_column_get_cell_renderers (tree_column);
        cell = renderers->data;
        g_object_set (G_OBJECT (cell), "pixbuf", pixbuf, NULL);
        cell = renderers->next->data;
        g_object_set (G_OBJECT (cell), "text", text, "strikethrough-set", error, NULL);
        g_free (text);
        g_list_free (renderers);
}

static void
tree_selection_changed_cb (GtkTreeSelection *select, GnomeDbSelector *mgsel) 
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	GObject *sel_obj = NULL;

	if (gtk_tree_selection_get_selected (select, &model, &iter)) {
		gtk_tree_model_get (model, &iter, OBJ_COLUMN, &sel_obj, -1);
		mgsel->priv->selection = sel_obj;
	}
	else 
		mgsel->priv->selection = NULL;

#ifdef debug_signal
	g_print (">> 'SELECTION_CHANGED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (mgsel), gnome_db_selector_signals[C_SELECTION_CHANGED], 0, sel_obj);
#ifdef debug_signal
	g_print ("<< 'SELECTION_CHANGED' from %s\n", __FUNCTION__);
#endif

}


/**
 * gnome_db_selector_set_mode_columns
 * @mgsel: a #GnomeDbSelector widget
 * @ref_object: a #GObject, or %NULL
 * @mode: an OR'ed value of the possible items to display in the widget
 * @columns: an OR'ed value describing which columns will be displayed
 * 
 * Changes what data gets displayed in @mgsel and which columns are displayed.
 * The @mode and @columns have the same meaning as for the gnome_db_selector_new()
 * function.
 *
 * The usage of the @ref_object parameter is the same as for gnome_db_selector_new().
 */
void
gnome_db_selector_set_mode_columns (GnomeDbSelector *mgsel, GObject *ref_object, gulong mode, gulong columns)
{
	gboolean keep_model, keep_columns;

	g_return_if_fail (mgsel && IS_GNOME_DB_SELECTOR (mgsel));
	g_return_if_fail (mgsel->priv);

	keep_model = (ref_object == mgsel->priv->ref_object) && (mgsel->priv->mode == mode);
	keep_columns = (mgsel->priv->columns == columns);
	if (keep_model && keep_columns)
		return;

	mgsel->priv->mode = mode;
	mgsel->priv->columns = columns;

	if (mgsel->priv->ref_object) {
		g_object_weak_unref (G_OBJECT (mgsel->priv->ref_object),
				     (GWeakNotify) object_weak_notify, mgsel);
		mgsel->priv->ref_object = NULL;
	}

	if (gnome_db_selector_initialize (mgsel, ref_object, keep_model, keep_columns) && ref_object) {
		/* add a weak ref on that object */
		mgsel->priv->ref_object = ref_object;
		g_object_weak_ref (G_OBJECT (mgsel->priv->ref_object),
				   (GWeakNotify) object_weak_notify, mgsel);		
	}

	/* TODO: save, for the current @mode, the currently selected object and the current vertical
	 * scroll of the treeview; and for the new @mode, if there are already such saved attributes,
	 * then restore them, all the while still emiting the right "selection_changed" signals */
}


static gchar *get_cut_path_depth (const gchar *path, guint depth);

/**
 * gnome_db_selector_set_selected_object
 * @mgsel:
 * @selection:
 *
 * Force the widget to select a given object, and to display it in its
 * visible area (unfolding nodes on the way if necessary)
 *
 * Returns: TRUE if the specified object was found, and FALSE otherwise
 */
gboolean
gnome_db_selector_set_selected_object (GnomeDbSelector *mgsel, GObject *selection)
{
	GtkTreeModel *model;
        GtkTreePath *path = NULL;
        GtkTreeIter iter;
	gpointer obj;
	
        g_return_val_if_fail (mgsel && IS_GNOME_DB_SELECTOR (mgsel), FALSE);
        g_return_val_if_fail (selection && G_IS_OBJECT (selection), FALSE);

        model = mgsel->priv->model;

	if (!gtk_tree_model_get_iter_first (model, &iter))
		return FALSE;
	
	gtk_tree_model_get (model, &iter, OBJ_COLUMN, &obj, -1);
	if (obj == selection) 
		path = gtk_tree_path_new_first ();
	
	while (!path && move_iter_to_next_leaf (model, &iter)) {
		gtk_tree_model_get (model, &iter, OBJ_COLUMN, &obj, -1);
		if (obj == selection) 
			path = gtk_tree_path_new_first ();
	}

        if (path) {
                gchar *strpath, *partpath;
                guint i = 1;
                GtkTreePath *ppath;
                GtkTreeSelection *tselect;

                strpath = gtk_tree_path_to_string (path);
                partpath = get_cut_path_depth (strpath, i);
                while (partpath) {
                        ppath = gtk_tree_path_new_from_string (partpath);
                        g_free (partpath);
                        gtk_tree_view_expand_row (mgsel->priv->treeview, ppath, FALSE);
                        gtk_tree_path_free (ppath);

                        i++;
                        partpath = get_cut_path_depth (strpath, i);
                }

                g_free (strpath);
                gtk_tree_view_scroll_to_cell (mgsel->priv->treeview, path, NULL, TRUE,
                                              0.5, 0.);
                tselect = gtk_tree_view_get_selection (mgsel->priv->treeview);

                gtk_tree_selection_unselect_all (tselect);
                gtk_tree_selection_select_path (tselect, path);
		
                gtk_tree_view_set_cursor (mgsel->priv->treeview, path, NULL, FALSE);
                gtk_tree_path_free (path);

		return TRUE;
        }
	else
		return FALSE;
}


/* returns an allocated string containing the first part of a path
 * representing a GtkTreePath, depth = 1 for the first ":" 
 */
static gchar *
get_cut_path_depth (const gchar *path, guint depth)
{
	gchar *str, *ptr;
        guint i = 0;

        str = g_strdup (path);
        ptr = str;
        while ((i < depth) && (*ptr != 0)) {
                if (*ptr == ':') 
                        i++;

                if (i == depth)
                        *ptr = 0;

                ptr++;
        }

        if (i != depth) {
                g_free (str);
                str = NULL;
        }

        return str;
}

/**
 * gnome_db_selector_set_headers_visible
 * @mgsel:
 * @visible:
 *
 * Show or hide the headers.
 */
void
gnome_db_selector_set_headers_visible (GnomeDbSelector *mgsel, gboolean visible)
{
	g_return_if_fail (mgsel && IS_GNOME_DB_SELECTOR (mgsel));
	g_return_if_fail (mgsel->priv);

	mgsel->priv->headers_visible = visible;
	if (mgsel->priv->treeview)
		gtk_tree_view_set_headers_visible (mgsel->priv->treeview, visible);
}

/**
 * gnome_db_selector_set_column_label
 * @mgsel:
 * @column:
 * @label:
 *
 * Sets the label of a column's header.
 */
void
gnome_db_selector_set_column_label (GnomeDbSelector *mgsel, guint column, const gchar *label)
{
	GtkTreeViewColumn *tcolumn;

	g_return_if_fail (mgsel && IS_GNOME_DB_SELECTOR (mgsel));
	g_return_if_fail (mgsel->priv);

	tcolumn = gtk_tree_view_get_column (mgsel->priv->treeview, column);
	gtk_tree_view_column_set_title (tcolumn, label);
}


/**
 * gnome_db_selector_get_selected_object
 * @mgsel:
 *
 * Get the currently selected object.
 *
 * Returns: the selected object or NULL if nothing is selected or the current selection is
 * on a "category" of objects (such as the "Functions" category for example).
 */
GObject *
gnome_db_selector_get_selected_object (GnomeDbSelector *mgsel)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *select = NULL;
	GObject *sel_obj = NULL;

	g_return_val_if_fail (mgsel && IS_GNOME_DB_SELECTOR (mgsel), NULL);
	g_return_val_if_fail (mgsel->priv, NULL);

	if (mgsel->priv->treeview)
		select = gtk_tree_view_get_selection (mgsel->priv->treeview);
	
	if (select && gtk_tree_selection_get_selected (select, &model, &iter)) 
		gtk_tree_model_get (model, &iter, OBJ_COLUMN, &sel_obj, -1);
	
	return sel_obj;
}


/**
 * gnome_db_selector_get_selected_object_parent
 * @mgsel:
 *
 * FIXME
 *
 * Returns:
 */
GObject *
gnome_db_selector_get_selected_object_parent (GnomeDbSelector *mgsel)
{
	GtkTreeIter iter, parent_iter;
	GtkTreeModel *model;
	GtkTreeSelection *select;
	GObject *sel_obj = NULL;

	select = gtk_tree_view_get_selection (mgsel->priv->treeview);
	
	if (gtk_tree_selection_get_selected (select, &model, &iter)) {
		if (gtk_tree_model_iter_parent (model, &parent_iter, &iter)) {
			gtk_tree_model_get (model, &parent_iter, OBJ_COLUMN, &sel_obj, -1);
		}
	}
	
	return sel_obj;
}

/*
 * Moves iter to the next LEAF of the model.
 * Returns: TRUE if successfull and FALSE otherwise (in this case iter is set to
 * invalid)
 */
static gboolean
move_iter_to_next_leaf (GtkTreeModel *model, GtkTreeIter *iter)
{
	GtkTreeIter tmp;
	gboolean down = FALSE;

	/* decide if we can go down in the tree hierarchy */
	if (gtk_tree_model_iter_children (model, &tmp, iter)) {
		gint contents_id;
		
		gtk_tree_model_get (model, iter, CONTENTS_COLUMN, &contents_id, -1);
		if (contents_id != CONTENTS_OBJECT) 
			down = TRUE;
	}
	
	/* action */
	if (down) {
		/* going down in the tree hierarchy */
		*iter = tmp;
		if (gtk_tree_model_iter_has_child (model, iter))
			return move_iter_to_next_leaf (model, iter);
		else
			return TRUE;
	}
	else {
		tmp = *iter;
		if (gtk_tree_model_iter_next (model, &tmp)) {
			/* we have a next sibling */
			*iter = tmp;

			/* can we go down? */
			if (gtk_tree_model_iter_has_child (model, iter)) {
				gint contents_id;
				
				gtk_tree_model_get (model, iter, CONTENTS_COLUMN, &contents_id, -1);
				if (contents_id != CONTENTS_OBJECT) 
					down = TRUE;
			}
			
			if (down)
			        return move_iter_to_next_leaf (model, iter);
			else
				return TRUE;
		}
		else {
			/* dead end, we need to go back in the tree hierarchy */
			tmp = *iter;

			while (gtk_tree_model_iter_parent (model, &tmp, iter)) {
				gint contents_id;
				*iter = tmp;
				
				gtk_tree_model_get (model, iter, CONTENTS_COLUMN, &contents_id, -1);
				if (contents_id == CONTENTS_TOP_CATEGORY)
					return FALSE;
				
				if (gtk_tree_model_iter_next (model, &tmp)) {
					*iter = tmp;
					if (gtk_tree_model_iter_has_child (model, iter))
						return move_iter_to_next_leaf (model, iter);
					else
						return TRUE;
				}
			}
			return FALSE;
		}
	}
}



/*
 * sets iter to the corresponding object. Iter MUST be initialized either at the begining
 * or to another object before calling.
 * The missing parameter must point to an object which appears _before_ 'object' in the list but not in the
 * model (this introduces a step to ignore in the indexes counting to reach 'object'). Usually 'missing' is NULL,
 * except when we need to add an object to the model and we only know its next sibling.
 */ 
static gboolean
set_iter_position (GtkTreeModel *model, GSList *obj_list, gpointer object, gpointer missing, GtkTreeIter *iter)
{
	gint i = 0, objpos;
	gboolean error = FALSE;
	gpointer obj;

	/* object and missing can't be equal */
	g_return_val_if_fail (object != missing, FALSE);

	/* going to the first leaf if we are on a node! */
	if (gtk_tree_model_iter_has_child (model, iter)) {
		gint contents_id;
					
		gtk_tree_model_get (model, iter, CONTENTS_COLUMN, &contents_id, -1);
		if (contents_id != CONTENTS_OBJECT) 
			error = !move_iter_to_next_leaf (model, iter);
	}

	if (error) 
		return FALSE;


	/* if iter already points to another object, we set it as the starting point */
	gtk_tree_model_get (model, iter, OBJ_COLUMN, &obj, -1);
	if (obj) {
		/*g_print ("\tobj %p of type %s\n", obj, G_OBJECT_TYPE_NAME (obj));*/
		i = g_slist_index (obj_list, obj);
		if (i < 0) {
			g_warning ("Iter is invalid!");
			i = 0;
		}
	}
	
	if (obj != object) {
		/* moving forward to the desired position */
		objpos = g_slist_index (obj_list, object) - (missing ? 1 : 0);
		while ((i != objpos) && !error) {
			error = !move_iter_to_next_leaf (model, iter);
			i++;
		}
		
		if (!error) {
			gtk_tree_model_get (model, iter, OBJ_COLUMN, &obj, -1);
			if (obj != object) {
				g_warning ("Obj found %p (%s=%s) != object requested %p (%s=%s)\n", 
					   obj, G_OBJECT_TYPE_NAME (obj), gnome_db_base_get_name (GNOME_DB_BASE (obj)),
					   object, G_OBJECT_TYPE_NAME (object), gnome_db_base_get_name (GNOME_DB_BASE (object)));
				error = TRUE;
			}
		}
	}
	
	return !error;
}

/*
 * Fills a model iterator with data, depending on the module
 */
static void
model_store_data (Module *module, GtkTreeIter *iter)
{
	if (module->model_store_data)
		(module->model_store_data) (module, iter);
}






/*
 * Generic module model where objects are grouped by their gnome_db_base_get_name() value
 * The text which gets displayed
 */

/*
 * Initial filling of the model
 */
void
name_group_init_model_fill (Module *module, GtkTreeModel *model)
{
	GSList *list, *ptl;
	const gchar *current_name = "";
	GtkTreeIter iter;
	GtkTreeIter cat_iter, *cat_iter_ptr = NULL;
	
	list = GROUP_DATA (module)->get_objects_list (module);
	ptl = list;
	while (ptl) {
		gchar *str1;
		const gchar *str2, *str3;

		if (strcmp (current_name, gnome_db_base_get_name (GNOME_DB_BASE (ptl->data)))) {
			GSList *next_obj = g_slist_next (ptl);
			/* function with a different name */
			current_name = gnome_db_base_get_name (GNOME_DB_BASE (ptl->data));
			
			if (next_obj &&
			    !strcmp (gnome_db_base_get_name (GNOME_DB_BASE (next_obj->data)), current_name)) {
				/* a new category is needed */
				gtk_tree_store_append (GTK_TREE_STORE (model), &cat_iter, module->iter);
				gtk_tree_store_set (GTK_TREE_STORE (model), &cat_iter, 
						    NAME_COLUMN, current_name, 
						    CONTENTS_COLUMN, CONTENTS_GROUP_CATEGORY, 
						    SUB_MODULE_COLUMN, NULL, -1);
				cat_iter_ptr = &cat_iter;
			}
			else
				cat_iter_ptr = NULL;
		}
		
		if (cat_iter_ptr)
			gtk_tree_store_append (GTK_TREE_STORE (model), &iter, cat_iter_ptr);
		else
			gtk_tree_store_append (GTK_TREE_STORE (model), &iter, module->iter);

		str1 = GROUP_DATA (module)->get_extended_name (G_OBJECT (ptl->data));
		str2 = gnome_db_base_get_owner (GNOME_DB_BASE (ptl->data));
		str3 = gnome_db_base_get_description (GNOME_DB_BASE (ptl->data));
		gtk_tree_store_set (GTK_TREE_STORE (model), &iter, 
				    NAME_COLUMN, str1,
				    OWNER_COLUMN, str2, 
				    DESCR_COLUMN, str3, 
				    PIXBUF_COLUMN, GROUP_DATA (module)->obj_pixbuf,
				    OBJ_COLUMN, ptl->data, 
				    CONTENTS_COLUMN, CONTENTS_OBJECT, 
				    SUB_MODULE_COLUMN, NULL, -1);
		model_store_data (module, &iter);
		g_free (str1);
		
		/* add sub modules if necessary */
		if (module->obj_manager) {
			Module *sub_module;
			
			sub_module = (module->obj_manager) (module, &iter, G_OBJECT (ptl->data));
			if (sub_module) {
				sub_module->parent_module = module;
				(sub_module->fill_model) (sub_module);
				module->sub_modules = g_slist_append (module->sub_modules, sub_module);
				gtk_tree_store_set (GTK_TREE_STORE (model), &iter, 
						    SUB_MODULE_COLUMN, sub_module, -1);
			}
		}

		ptl = g_slist_next (ptl);
	}

	GROUP_DATA (module)->objects = (gpointer) list;
	GROUP_DATA (module)->iter_obj = NULL;
	GROUP_DATA (module)->iter = NULL;

	/* manager "nullified" connect if possible, or otherwise only weak ref */
	if (g_signal_lookup ("nullified", G_OBJECT_TYPE (GROUP_DATA (module)->manager)))
		GROUP_DATA (module)->manager_nullified_sig_id = 
			gnome_db_base_connect_nullify (
				GROUP_DATA (module)->manager, 
				G_CALLBACK (name_group_manager_nullified_cb), module);
	else {
		g_object_weak_ref (G_OBJECT (GROUP_DATA (module)->manager),
				   (GWeakNotify) name_group_manager_weak_notify, module);
		GROUP_DATA (module)->manager_weak_refed = TRUE;
	}
}

/*
 * Remove the module
 */
void
name_group_manager_nullified_cb (GObject *manager_obj, Module *module)
{
	GtkTreeModel *model;
	
	model = module->selector->priv->model;
	g_signal_handler_disconnect (G_OBJECT (GROUP_DATA (module)->manager),
				     GROUP_DATA (module)->manager_nullified_sig_id);
	GROUP_DATA (module)->manager_nullified_sig_id = 0;
	
	if (module->iter) {
		gtk_tree_store_remove (GTK_TREE_STORE (model), module->iter);
		gtk_tree_iter_free (module->iter);
		module->iter = NULL;
	}
	else 
		gtk_tree_store_clear (GTK_TREE_STORE (model));
	
	(module->free) (module);
	
	if (module->parent_module)
		module->parent_module->sub_modules = g_slist_remove (module->parent_module->sub_modules, module);
	else
		module->selector->priv->modules = g_slist_remove (module->selector->priv->modules, module);
	g_free (module);
}

void
name_group_manager_weak_notify (Module *module, GObject *manager_obj)
{
	GtkTreeModel *model;

	model = module->selector->priv->model;
	GROUP_DATA (module)->manager_weak_refed = FALSE;

	if (module->iter) {
		gtk_tree_store_remove (GTK_TREE_STORE (model), module->iter);
		gtk_tree_iter_free (module->iter);
		module->iter = NULL;
	}
	else 
		gtk_tree_store_clear (GTK_TREE_STORE (model));

	(module->free) (module);

	if (module->parent_module)
		module->parent_module->sub_modules = g_slist_remove (module->parent_module->sub_modules, module);
	else
		module->selector->priv->modules = g_slist_remove (module->selector->priv->modules, module);
	g_free (module);
}

/*
 * To be connected to the object which signals the creation of a new object which is to be
 * added to the data model
 */
void
name_group_obj_added_cb (GObject *manager_obj, GObject *added_obj, Module *module)
{
	GSList *tmplist;
	gint objpos;

	/* g_print ("ADDING obj %p of type %s ", added_obj, G_OBJECT_TYPE_NAME (added_obj)); */
	/* 	g_print (" NAME: %s\n", gnome_db_base_get_name (GNOME_DB_BASE (added_obj))); */
	

	/* module's objects list sync. */
	tmplist = GROUP_DATA (module)->get_objects_list (module);
	objpos = g_slist_index (tmplist, added_obj);
	GROUP_DATA (module)->objects = g_slist_insert (GROUP_DATA (module)->objects, added_obj, objpos);
	g_slist_free (tmplist);

	if (GROUP_DATA (module)->iter) {
		/* need to reset iter ? */
		if (GROUP_DATA (module)->iter_obj) {
			gpointer obj;
			GtkTreeModel *model;
			
			model = module->selector->priv->model;
			gtk_tree_model_get (model, GROUP_DATA (module)->iter, OBJ_COLUMN, &obj, -1);
			if (g_slist_index (GROUP_DATA (module)->objects, obj) >
			    g_slist_index (GROUP_DATA (module)->objects, added_obj)) {
				gtk_tree_model_iter_children (model, GROUP_DATA (module)->iter, module->iter);
			}
		}
	}
	
	name_group_do_add_obj (module, added_obj);
}

/*
 * To be connected to the object which signals the removal of an object which is to be
 * removed the data model
 */
void
name_group_obj_removed_cb (GObject *manager_obj, GObject *removed_obj, Module *module)
{
	if (GROUP_DATA (module)->iter) {
		/* need to reset iter ? */
		if (GROUP_DATA (module)->iter_obj) {
			gpointer obj;
			GtkTreeModel *model;

			model = module->selector->priv->model;
			gtk_tree_model_get (model, GROUP_DATA (module)->iter, OBJ_COLUMN, &obj, -1);
			if (g_slist_index (GROUP_DATA (module)->objects, obj) > 
			    g_slist_index (GROUP_DATA (module)->objects, removed_obj)) {
				gtk_tree_model_iter_children (model, GROUP_DATA (module)->iter, module->iter);
			}
		}
	}
	
	name_group_do_remove_obj (module, removed_obj);

	/* module's objects list sync. */
	GROUP_DATA (module)->objects = g_slist_remove (GROUP_DATA (module)->objects, removed_obj);

}

/*
 * To be connected to the object which signals the update of an object
 */
void
name_group_obj_updated_cb (GObject *manager_obj, GObject *upd_obj, Module *module)
{
	if (GROUP_DATA (module)->iter) {
		/* need to reset iter ? */
		if (GROUP_DATA (module)->iter_obj) {
			gpointer obj;
			GtkTreeModel *model;

			model = module->selector->priv->model;
			gtk_tree_model_get (model, GROUP_DATA (module)->iter, OBJ_COLUMN, &obj, -1);
			if (g_slist_index (GROUP_DATA (module)->objects, obj) > 
			    g_slist_index (GROUP_DATA (module)->objects, upd_obj)) {
				gtk_tree_model_iter_children (model, GROUP_DATA (module)->iter, module->iter);
			}
		}
	}
	
	name_group_do_update_obj (module, upd_obj);
}



/*
 * To be used to declare a "Big update" has started. It is used for optimisation
 * and it is not necessary to use it.
 */
void
name_group_update_started_cb (GObject *manager_obj, Module *module)
{
	GtkTreeModel *model;
	
	model = module->selector->priv->model;
	GROUP_DATA (module)->iter = g_new0 (GtkTreeIter, 1);
	gtk_tree_model_iter_children (model, GROUP_DATA (module)->iter, module->iter);
}

/*
 * To be used to declare a "Big update" has started. It is used for optimisation
 * and it is not necessary to use it.
 */
void
name_group_update_finished_cb (GObject *manager_obj, Module *module)
{
	gtk_tree_iter_free (GROUP_DATA (module)->iter);
	GROUP_DATA (module)->iter = NULL;
	GROUP_DATA (module)->iter_obj = NULL;
}

/*
 * Real addition of the object to the model
 */
void
name_group_do_add_obj (Module *module, GObject *added_obj)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	GObject *sibling_before = NULL, *sibling_after = NULL;
	gint objpos;
	const gchar *objname;
	gboolean insert_ok = FALSE;

	objname = gnome_db_base_get_name (GNOME_DB_BASE (added_obj));
	model = module->selector->priv->model;


	/* finding the siblings of the new object */
	objpos = g_slist_index (GROUP_DATA (module)->objects, added_obj);
	if (objpos > 0) {
		sibling_before = g_slist_nth_data (GROUP_DATA (module)->objects, objpos - 1);
		if (strcmp (gnome_db_base_get_name (GNOME_DB_BASE (sibling_before)), objname))
			sibling_before = NULL;
	}

	sibling_after = g_slist_nth_data (GROUP_DATA (module)->objects, objpos + 1);
	if (sibling_after) {
		if (strcmp (gnome_db_base_get_name (GNOME_DB_BASE (sibling_after)), objname))
			sibling_after = NULL;
	}

	if (!sibling_before && !sibling_after) {
		/* NON "polymorphic" object */
		if (objpos == 0) {
			gtk_tree_store_prepend (GTK_TREE_STORE (model), &iter, module->iter);
			if (GROUP_DATA (module)->iter) {
				*GROUP_DATA (module)->iter = iter;
				GROUP_DATA (module)->iter_obj = G_OBJECT (added_obj);
			}
			insert_ok = TRUE;
		}
		else {
			GObject *obj_before;
			
			obj_before = g_slist_nth_data (GROUP_DATA (module)->objects, objpos - 1);
			if (!GROUP_DATA (module)->iter)
				gtk_tree_model_iter_children (model, &iter, module->iter);
			else
				iter = *GROUP_DATA (module)->iter;

			if (set_iter_position (model, GROUP_DATA (module)->objects, obj_before, NULL, &iter)) {
				GtkTreeIter tmpiter;

				/* if the previous object is a polymorphic object, then
				   we need to create a node after its parent and not after
				   the current iterator */
				if (gtk_tree_model_iter_parent (model, &tmpiter, &iter)) {
					gint contents_id;
					
					gtk_tree_model_get (model, &tmpiter, 
							    CONTENTS_COLUMN, &contents_id, -1);
					if (contents_id == CONTENTS_GROUP_CATEGORY) 
						iter = tmpiter;
				}

				gtk_tree_store_insert_after (GTK_TREE_STORE (model), &tmpiter, 
							     module->iter, &iter);
				iter = tmpiter;
				insert_ok = TRUE;
			}
			else
				g_error ("Can't set iter at %s, line %d\n", __FUNCTION__, __LINE__);
		}
	}
	else {
		/* "polymorphic" object */
		if (sibling_before && sibling_after) {
			if (!GROUP_DATA (module)->iter)
				gtk_tree_model_iter_children (model, &iter, module->iter);
			else
				iter = *GROUP_DATA (module)->iter;

			if (set_iter_position (model, GROUP_DATA (module)->objects, 
					       sibling_before, NULL, &iter)) {
				GtkTreeIter tmpiter;
				
				gtk_tree_store_insert_after (GTK_TREE_STORE (model), &tmpiter, 
							     NULL, &iter);
				iter = tmpiter;
				insert_ok = TRUE;
			}
			else
				g_error ("Can't set iter at %s, line %d\n", __FUNCTION__, __LINE__);
		}
		else {
			GObject *sibling_obj;
			gpointer missing = NULL;

			if (sibling_before) 
				sibling_obj = sibling_before;
			else {
				sibling_obj = sibling_after;
				missing = added_obj;
			}
			
			if (!GROUP_DATA (module)->iter)
				gtk_tree_model_iter_children (model, &iter, module->iter);
			else
				iter = *GROUP_DATA (module)->iter;

			/* sets iter to point to sibling_obj */
			if (set_iter_position (model, GROUP_DATA (module)->objects, 
					       sibling_obj, missing, &iter)) {
				GtkTreeIter tmpiter;
				gboolean make_new_cat = FALSE;
				
				/* do we alreday have a object category or do we need one? */
				if (gtk_tree_model_iter_parent (model, &tmpiter, &iter)) {
					gint contents_id;
					
					gtk_tree_model_get (model, &tmpiter, 
							    CONTENTS_COLUMN, &contents_id, -1);
					if (contents_id != CONTENTS_GROUP_CATEGORY)
						make_new_cat = TRUE;
				}
				else
					make_new_cat = TRUE;
				
				if (make_new_cat) { /* a new category is needed */
					GtkTreeIter cat_iter;
					gchar *str1;
					const gchar *str2, *str3;
					
					gtk_tree_store_insert_after (GTK_TREE_STORE (model), &cat_iter, 
								     NULL, &iter);
					tmpiter = cat_iter;
					
					gtk_tree_store_set (GTK_TREE_STORE (model), &cat_iter, NAME_COLUMN, 
							    gnome_db_base_get_name (GNOME_DB_BASE (added_obj)), 
							    CONTENTS_COLUMN, CONTENTS_GROUP_CATEGORY, 
							    SUB_MODULE_COLUMN, NULL, -1);
					gtk_tree_store_remove (GTK_TREE_STORE (model), &iter);
					gtk_tree_store_append (GTK_TREE_STORE (model), &iter, &cat_iter);
					
					str1 = GROUP_DATA (module)->get_extended_name (sibling_obj);
					str2 = gnome_db_base_get_owner (GNOME_DB_BASE (sibling_obj));
					str3 = gnome_db_base_get_description (GNOME_DB_BASE (sibling_obj));
					gtk_tree_store_set (GTK_TREE_STORE (model), &iter,
							    NAME_COLUMN, str1, 
							    OWNER_COLUMN, str2, 
							    DESCR_COLUMN, str3, 
							    PIXBUF_COLUMN, 
							    GROUP_DATA (module)->obj_pixbuf,
							    OBJ_COLUMN, sibling_obj, 
							    CONTENTS_COLUMN, CONTENTS_OBJECT, 
							    SUB_MODULE_COLUMN, NULL, -1);
					model_store_data (module, &iter);
					g_free (str1);
				}
				
				/* here iter points to the sibling and tmpiter to the parent category */
				if (sibling_before)
					gtk_tree_store_insert_after (GTK_TREE_STORE (model), &tmpiter, 
								     &tmpiter, &iter);
				else
					gtk_tree_store_insert_before (GTK_TREE_STORE (model), &tmpiter, 
								      &tmpiter, &iter);
				iter = tmpiter;
				insert_ok = TRUE;
			}
			else
				g_error ("Can't set iter at %s, line %d\n", __FUNCTION__, __LINE__);
		}
	}
	
	if (insert_ok) {
		/* set the columns values */
		gchar *str1;
		const gchar *str2, *str3;

		str1 = GROUP_DATA (module)->get_extended_name (added_obj);
		str2 = gnome_db_base_get_owner (GNOME_DB_BASE (added_obj));
		str3 = gnome_db_base_get_description (GNOME_DB_BASE (added_obj));
		gtk_tree_store_set (GTK_TREE_STORE (model), &iter, 
				    NAME_COLUMN, str1,
				    OWNER_COLUMN, str2,
				    DESCR_COLUMN, str3, 
				    PIXBUF_COLUMN, GROUP_DATA (module)->obj_pixbuf,
				    OBJ_COLUMN, added_obj, 
				    CONTENTS_COLUMN, CONTENTS_OBJECT, 
				    SUB_MODULE_COLUMN, NULL, -1);
		model_store_data (module, &iter);
		g_free (str1);
		if (GROUP_DATA (module)->iter) {
			*GROUP_DATA (module)->iter = iter;
			GROUP_DATA (module)->iter_obj = G_OBJECT (added_obj);
		}

		/* add sub modules if necessary */
		if (module->obj_manager) {
			Module *sub_module;

			sub_module = (module->obj_manager) (module, &iter, G_OBJECT (added_obj));
			if (sub_module) {
				sub_module->parent_module = module;
				(sub_module->fill_model) (sub_module);
				module->sub_modules = g_slist_append (module->sub_modules, sub_module);
				gtk_tree_store_set (GTK_TREE_STORE (model), &iter, 
						    SUB_MODULE_COLUMN, sub_module, -1);
			}
		}
	}

}

/*
 * Real deletion from the object to the model
 */
void
name_group_do_remove_obj (Module *module, GObject *removed_obj)
{
	GtkTreeModel *model;
	GtkTreeIter iter;

	model = module->selector->priv->model;

	if (!GROUP_DATA (module)->iter) 
		gtk_tree_model_iter_children (model, &iter, module->iter);
	else 
		iter = *GROUP_DATA (module)->iter;
	
	if (set_iter_position (model, GROUP_DATA (module)->objects, removed_obj, NULL, &iter)) {
		GtkTreeIter tmpiter;
		Module *sub_module;

		/* Remove associated sub module if necessary */
		gtk_tree_model_get (model, &iter, SUB_MODULE_COLUMN, &sub_module, -1);
		if (sub_module) {
			g_assert (g_slist_find (module->sub_modules, sub_module));
			(sub_module->free) (sub_module);
			module->sub_modules = g_slist_remove (module->sub_modules, sub_module);
			g_free (sub_module);
		}

		if (gtk_tree_model_iter_parent (model, &tmpiter, &iter)) {
			gint contents_id;
			gtk_tree_model_get (model, &tmpiter, CONTENTS_COLUMN, &contents_id, -1);
			if ((contents_id == CONTENTS_GROUP_CATEGORY) && 
			    (gtk_tree_model_iter_n_children (model, &tmpiter) == 1))
				iter = tmpiter;
		}

		if (gtk_tree_store_remove (GTK_TREE_STORE (model), &iter)) {
			if (GROUP_DATA (module)->iter) {
				gpointer obj;

				gtk_tree_model_get (model, &iter, OBJ_COLUMN, &obj, -1);
				if (!obj) {
					if (move_iter_to_next_leaf (model, &iter)) {
						gtk_tree_model_get (model, &iter, OBJ_COLUMN, &obj, -1);
						*GROUP_DATA (module)->iter = iter;
						GROUP_DATA (module)->iter_obj = G_OBJECT (obj);
					}
					else
						GROUP_DATA (module)->iter_obj = NULL;
				}
				else {
					*GROUP_DATA (module)->iter = iter;
					GROUP_DATA (module)->iter_obj = G_OBJECT (obj);
				}
			}
		}
		else {
			if (GROUP_DATA (module)->iter) 
				GROUP_DATA (module)->iter_obj = NULL;
		}
	}
	else
		g_warning ("Can't find right GtkTreeIter for object %p (%s)!",
			   removed_obj, 
			   removed_obj ? gnome_db_base_get_name (GNOME_DB_BASE (removed_obj)) : "NULL");
}

/*
 * Real update of the object to the model
 */
void
name_group_do_update_obj (Module *module, GObject *upd_obj)
{
	GtkTreeModel *model;
	GtkTreeIter iter;

	model = module->selector->priv->model;
	if (!GROUP_DATA (module)->iter)
		gtk_tree_model_iter_children (model, &iter, module->iter);
	else
		iter = *GROUP_DATA (module)->iter;
		
	if (set_iter_position (model, GROUP_DATA (module)->objects, upd_obj, NULL, &iter)) {
		/* set the columns values */
		gchar *str1;
		const gchar *str2, *str3;

		str1 = GROUP_DATA (module)->get_extended_name (upd_obj);
		str2 = gnome_db_base_get_owner (GNOME_DB_BASE (upd_obj));
		str3 = gnome_db_base_get_description (GNOME_DB_BASE (upd_obj));
		gtk_tree_store_set (GTK_TREE_STORE (model), &iter, 
				    NAME_COLUMN, str1,
				    OWNER_COLUMN, str2,
				    DESCR_COLUMN, str3, 
				    PIXBUF_COLUMN, GROUP_DATA (module)->obj_pixbuf,
				    OBJ_COLUMN, upd_obj, CONTENTS_COLUMN, CONTENTS_OBJECT, -1);
		model_store_data (module, &iter);
		g_free (str1);
		if (GROUP_DATA (module)->iter) {
			*GROUP_DATA (module)->iter = iter;
			GROUP_DATA (module)->iter_obj = G_OBJECT (upd_obj);
		}
	}

}


/*
 * Does memory deallocation for data specific to the group model.
 */
void
name_group_free_mod_data (Module *module)
{
	if (GROUP_DATA (module)->objects) {
		g_slist_free (GROUP_DATA (module)->objects);
		GROUP_DATA (module)->objects = NULL;
	}

	if (GROUP_DATA (module)->iter) {
		gtk_tree_iter_free (GROUP_DATA (module)->iter);
		GROUP_DATA (module)->iter = NULL;
	}

	if (GROUP_DATA (module)->obj_pixbuf) {
		g_object_unref (G_OBJECT (GROUP_DATA (module)->obj_pixbuf));
		GROUP_DATA (module)->obj_pixbuf = NULL;
	}

	/* manager weak unref */
	if (GROUP_DATA (module)->manager_weak_refed) {
		g_object_weak_unref (G_OBJECT (GROUP_DATA (module)->manager),
				     (GWeakNotify) name_group_manager_weak_notify, module);
		GROUP_DATA (module)->manager_weak_refed = FALSE;
	}

	/* manager "nullified" signal disconnect */
	if (GROUP_DATA (module)->manager_nullified_sig_id) {
		g_signal_handler_disconnect (G_OBJECT (GROUP_DATA (module)->manager),
					     GROUP_DATA (module)->manager_nullified_sig_id);
		GROUP_DATA (module)->manager_nullified_sig_id = 0;
	}
}












/*
 * Generic module model where objects are listed directly in the order of the provided
 * list of objects
 */

/*
 * Initial filling of the model
 */
void
flat_init_model_fill (Module *module, GtkTreeModel *model)
{
	GSList *list, *ptl;
	GtkTreeIter iter;
	GdkPixbuf *pixbuf = NULL;
	
	list = FLAT_DATA (module)->get_objects_list (module);
	ptl = list;
	while (ptl) {
		const gchar *str1, *str2, *str3;

		gtk_tree_store_append (GTK_TREE_STORE (model), &iter, module->iter);

		if (FLAT_DATA (module)->pixbuf_hash) 
			pixbuf = g_hash_table_lookup (FLAT_DATA (module)->pixbuf_hash, 
						      GUINT_TO_POINTER (G_OBJECT_TYPE (ptl->data)));
		if (! pixbuf)
			pixbuf = FLAT_DATA (module)->fallback_obj_pixbuf;
		
		str1 = gnome_db_base_get_name (GNOME_DB_BASE (ptl->data));
		str2 = gnome_db_base_get_owner (GNOME_DB_BASE (ptl->data));
		str3 = gnome_db_base_get_description (GNOME_DB_BASE (ptl->data));
		gtk_tree_store_set (GTK_TREE_STORE (model), &iter, 
				    NAME_COLUMN, str1,
				    OWNER_COLUMN, str2, 
				    DESCR_COLUMN, str3, 
				    PIXBUF_COLUMN, pixbuf,
				    OBJ_COLUMN, ptl->data, 
				    CONTENTS_COLUMN, CONTENTS_OBJECT, 
				    SUB_MODULE_COLUMN, NULL, -1);
		model_store_data (module, &iter);
		
		/* add sub modules if necessary */
		if (module->obj_manager) {
			Module *sub_module;
			
			sub_module = (module->obj_manager) (module, &iter, G_OBJECT (ptl->data));
			if (sub_module) {
				sub_module->parent_module = module;
				(sub_module->fill_model) (sub_module);
				module->sub_modules = g_slist_append (module->sub_modules, sub_module);
				gtk_tree_store_set (GTK_TREE_STORE (model), &iter, 
						    SUB_MODULE_COLUMN, sub_module, -1);
			}
		}

		ptl = g_slist_next (ptl);
	}

	FLAT_DATA (module)->objects = (gpointer) list;

	/* manager weak ref */
	if (FLAT_DATA (module)->manager) {
		g_object_weak_ref (G_OBJECT (FLAT_DATA (module)->manager),
				   (GWeakNotify) flat_manager_weak_notify, module);
		FLAT_DATA (module)->manager_weak_refed = TRUE;
	}
}

/*
 * Remove the module
 */
void
flat_manager_weak_notify (Module *module, GObject *manager_obj)
{
	GtkTreeModel *model;

	model = module->selector->priv->model;
	FLAT_DATA (module)->manager_weak_refed = FALSE;

	if (module->iter) {
		gtk_tree_store_remove (GTK_TREE_STORE (model), module->iter);
		gtk_tree_iter_free (module->iter);
		module->iter = NULL;
	}
	else 
		gtk_tree_store_clear (GTK_TREE_STORE (model));

	(module->free) (module);

	if (module->parent_module)
		module->parent_module->sub_modules = g_slist_remove (module->parent_module->sub_modules, module);
	else
		module->selector->priv->modules = g_slist_remove (module->selector->priv->modules, module);
	g_free (module);
}

/*
 * To be connected to the object which signals the creation of a new object which is to be
 * added to the data model
 */
void
flat_obj_added_cb (GObject *manager_obj, GObject *added_obj, Module *module)
{
	GSList *tmplist;
	gint objpos;

	/* module's objects list sync. */
	tmplist = FLAT_DATA (module)->get_objects_list (module);
	objpos = g_slist_index (tmplist, added_obj);
	if (objpos >= 0) {
		FLAT_DATA (module)->objects = g_slist_insert (FLAT_DATA (module)->objects, added_obj, objpos);
		flat_do_add_obj (module, added_obj);
	}
	g_slist_free (tmplist);
}

/*
 * To be connected to the object which signals the removal of an object which is to be
 * removed the data model
 */
void
flat_obj_removed_cb (GObject *manager_obj, GObject *removed_obj, Module *module)
{
	flat_do_remove_obj (module, removed_obj);

	/* module's objects list sync. */
	FLAT_DATA (module)->objects = g_slist_remove (FLAT_DATA (module)->objects, removed_obj);
}

/*
 * To be connected to the object which signals the update of an object
 */
void
flat_obj_updated_cb (GObject *manager_obj, GObject *upd_obj, Module *module)
{
	flat_do_update_obj (module, upd_obj);
}

/*
 * To be connected to the object which signals a change in the objects' order
 */
void
flat_objs_order_changed_cb (GObject *manager_obj, Module *module)
{
	GSList *new_list, *old_list;
	GSList *list;
	gint *new_order;
	gint old_pos, new_pos;
	gint nb_objects;

	new_list = FLAT_DATA (module)->get_objects_list (module);
	old_list = FLAT_DATA (module)->objects;
	nb_objects = g_slist_length (new_list);
	g_return_if_fail (nb_objects == g_slist_length (old_list));

	new_order = g_new0 (gint, nb_objects);
	list = old_list;
	old_pos = 0;
	while (list) {
		new_pos = g_slist_index (new_list, list->data);
		if (new_pos < 0) {
			g_warning ("Can't find object in new list");
			return;
		}
		new_order [new_pos] = old_pos;

		list = g_slist_next (list);
		old_pos ++;
	}

	FLAT_DATA (module)->objects = new_list;
	g_slist_free (old_list);

	gtk_tree_store_reorder (GTK_TREE_STORE (module->selector->priv->model), module->iter, new_order);
	g_free (new_order);
}


/*
 * Real addition of the object to the model
 */
void
flat_do_add_obj (Module *module, GObject *added_obj)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gint objpos;
	const gchar *str1, *str2, *str3;
	GdkPixbuf *pixbuf = NULL;

	model = module->selector->priv->model;
	objpos = g_slist_index (FLAT_DATA (module)->objects, added_obj);
	
	/* test if the added object has been found. This is not always the case since the provided list
	   of objects may have filtered out some object. */
	if (objpos < 0) 
		return;

	if (FLAT_DATA (module)->pixbuf_hash) 
			pixbuf = g_hash_table_lookup (FLAT_DATA (module)->pixbuf_hash, 
						      GUINT_TO_POINTER (G_OBJECT_TYPE (added_obj)));
	if (! pixbuf)
		pixbuf = FLAT_DATA (module)->fallback_obj_pixbuf;

	gtk_tree_store_insert (GTK_TREE_STORE (model), &iter, module->iter, objpos);
	str1 = gnome_db_base_get_name (GNOME_DB_BASE (added_obj));
	str2 = gnome_db_base_get_owner (GNOME_DB_BASE (added_obj));
	str3 = gnome_db_base_get_description (GNOME_DB_BASE (added_obj));
	gtk_tree_store_set (GTK_TREE_STORE (model), &iter, 
			    NAME_COLUMN, str1,
			    OWNER_COLUMN, str2,
			    DESCR_COLUMN, str3, 
			    PIXBUF_COLUMN, pixbuf,
			    OBJ_COLUMN, added_obj, 
			    CONTENTS_COLUMN, CONTENTS_OBJECT, 
			    SUB_MODULE_COLUMN, NULL, -1);
	model_store_data (module, &iter);

	/* add sub modules if necessary */
	if (module->obj_manager) {
		Module *sub_module;
		
		sub_module = (module->obj_manager) (module, &iter, G_OBJECT (added_obj));
		if (sub_module) {
			sub_module->parent_module = module;
			(sub_module->fill_model) (sub_module);
			module->sub_modules = g_slist_append (module->sub_modules, sub_module);
			gtk_tree_store_set (GTK_TREE_STORE (model), &iter, 
					    SUB_MODULE_COLUMN, sub_module, -1);
		}
	}
}

/*
 * Real deletion from the object to the model
 */
void
flat_do_remove_obj (Module *module, GObject *removed_obj)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gint objpos;

	model = module->selector->priv->model;
	objpos = g_slist_index (FLAT_DATA (module)->objects, removed_obj);

	/* test if the removed object has been found. This is not always the case since the provided list
	   of objects may have filtered out some object. */
	if (objpos < 0) 
		return;

	if (gtk_tree_model_iter_nth_child (model, &iter, module->iter, objpos)) {
		Module *sub_module;

		/* Remove associated sub module if necessary */
		gtk_tree_model_get (model, &iter, SUB_MODULE_COLUMN, &sub_module, -1);
		if (sub_module) {
			g_assert (g_slist_find (module->sub_modules, sub_module));
			(sub_module->free) (sub_module);
			module->sub_modules = g_slist_remove (module->sub_modules, sub_module);
			g_free (sub_module);
		}

		gtk_tree_store_remove (GTK_TREE_STORE (model), &iter);
	}
	else
		g_warning ("Can't find right GtkTreeIter for object %p (%s) at position %d!",
			   removed_obj, 
			   removed_obj ? gnome_db_base_get_name (GNOME_DB_BASE (removed_obj)) : "NULL", objpos);
}

/*
 * Real update of the object to the model
 */
void
flat_do_update_obj (Module *module, GObject *upd_obj)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gint objpos;

	model = module->selector->priv->model;
	objpos = g_slist_index (FLAT_DATA (module)->objects, upd_obj);

	/* test if the updated object has been found. This is not always the case since the provided list
	   of objects may have filtered out some object. */
	if (objpos < 0) 
		return;

	if (gtk_tree_model_iter_nth_child (model, &iter, module->iter, objpos)) {
		const gchar *str1, *str2, *str3;
		GdkPixbuf *pixbuf = NULL;

		if (FLAT_DATA (module)->pixbuf_hash) 
			pixbuf = g_hash_table_lookup (FLAT_DATA (module)->pixbuf_hash, 
						      GUINT_TO_POINTER (G_OBJECT_TYPE (upd_obj)));
		if (! pixbuf)
			pixbuf = FLAT_DATA (module)->fallback_obj_pixbuf;

		str1 = gnome_db_base_get_name (GNOME_DB_BASE (upd_obj));
		str2 = gnome_db_base_get_owner (GNOME_DB_BASE (upd_obj));
		str3 = gnome_db_base_get_description (GNOME_DB_BASE (upd_obj));
		gtk_tree_store_set (GTK_TREE_STORE (model), &iter, 
				    NAME_COLUMN, str1,
				    OWNER_COLUMN, str2,
				    DESCR_COLUMN, str3, 
				    PIXBUF_COLUMN, pixbuf,
				    OBJ_COLUMN, upd_obj, CONTENTS_COLUMN, CONTENTS_OBJECT, -1);
		model_store_data (module, &iter);
		
	}
	else
		g_warning ("Can't find right GtkTreeIter for object %p (%s) at position %d!",
			   upd_obj, 
			   upd_obj ? gnome_db_base_get_name (GNOME_DB_BASE (upd_obj)) : "NULL", objpos);
}


/*
 * Does memory deallocation for data specific to the flat model.
 */
void
flat_free_mod_data (Module *module)
{
	if (FLAT_DATA (module)->objects) {
		g_slist_free (FLAT_DATA (module)->objects);
		FLAT_DATA (module)->objects = NULL;
	}

	if (FLAT_DATA (module)->pixbuf_hash) {
		g_hash_table_destroy (FLAT_DATA (module)->pixbuf_hash);
		FLAT_DATA (module)->pixbuf_hash = NULL;
	}

	if (FLAT_DATA (module)->fallback_obj_pixbuf) {
		g_object_unref (G_OBJECT (FLAT_DATA (module)->fallback_obj_pixbuf));
		FLAT_DATA (module)->fallback_obj_pixbuf = NULL;
	}

	/* manager weak unref */
	if (FLAT_DATA (module)->manager_weak_refed) {
		g_object_weak_unref (G_OBJECT (FLAT_DATA (module)->manager),
				     (GWeakNotify) flat_manager_weak_notify, module);
		FLAT_DATA (module)->manager_weak_refed = FALSE;
	}
}

