/* sel-onequery.c
 *
 * Copyright (C) 2004 - 2006 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "sel-onequery.h"
#include <libgda/libgda.h>
#include <glib/gi18n-lib.h>
#include "gnome-db-stock.h"
#include "sel-onetarget.h"

/*
 *
 * Modules for the fields, targets, joins, etc of a given GdaQuery
 *
 */
typedef struct {
	ModFlatData  data;
	GdkPixbuf   *field_pixbuf;
} ModOneQueryData;
#define QUERY_DATA(x) ((ModOneQueryData *)(x->mod_data))

static void         module_onequery_fill_model (Module *module);
static void         module_onequery_free (Module *module);
static const gchar *module_onequery_col_name (Module *module, guint colno);
static Module      *module_onequery_obj_manager (Module *module, GtkTreeIter *iter, GObject *object);
static void         module_onequery_model_store_data (Module *module, GtkTreeIter *iter);
static GSList      *module_onequery_get_objects_list (Module *module);
static const gchar *module_onequery_render_qfield_type (GdaQueryField *field);

Module *
sel_module_onequery_new (GnomeDbSelector *mgsel, gboolean insert_header, 
			 GtkTreeIter *iter, gpointer data)
{
	Module *module;
	GdaQuery *query;
	GdkPixbuf *pixbuf_query = NULL;
	GdkPixbuf *pixbuf_field = NULL;
	GdkPixbuf *pixbuf_target = NULL;

	g_assert (data && GDA_IS_QUERY (data));
	pixbuf_query = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-query_16x16.png");
	pixbuf_field = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-field_16x16.png");
	pixbuf_target = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-table_16x16.png");

	/* module structure */
	query = GDA_QUERY (data);
	module = g_new0 (Module, 1);
	module->selector = mgsel;
	module->fill_model = module_onequery_fill_model;
	module->free = module_onequery_free;
	module->col_name = module_onequery_col_name;
	module->obj_manager = module_onequery_obj_manager;
	module->model_store_data = module_onequery_model_store_data;
	module->mod_data = NULL;
	module->iter = NULL;
	module->parent_module = NULL;
	module->sub_modules = NULL;

	/* Module's private data */
	module->mod_data = g_new0 (ModOneQueryData, 1);
	FLAT_DATA (module)->manager = G_OBJECT (query);
	FLAT_DATA (module)->manager_weak_refed = FALSE;
	FLAT_DATA (module)->pixbuf_hash = g_hash_table_new_full (NULL, NULL, NULL, 
								 (GDestroyNotify) g_object_unref);
	FLAT_DATA (module)->fallback_obj_pixbuf = pixbuf_query;
	FLAT_DATA (module)->get_objects_list = module_onequery_get_objects_list;
	QUERY_DATA (module)->field_pixbuf = pixbuf_field;

	g_hash_table_insert (FLAT_DATA (module)->pixbuf_hash, GUINT_TO_POINTER (GDA_TYPE_QUERY_TARGET), pixbuf_target);

	/* model settings */
	if (insert_header) {
		GdkPixbuf *pixbuf = NULL;
		GtkTreeModel *model = mgsel->priv->model;

		module->iter = g_new0 (GtkTreeIter, 1);
		gtk_tree_store_append (GTK_TREE_STORE (model), module->iter, iter);
		gtk_tree_store_set (GTK_TREE_STORE (model), module->iter, 
				    NAME_COLUMN, gda_object_get_name (GDA_OBJECT (query)), 
				    PIXBUF_COLUMN, pixbuf, 
				    CONTENTS_COLUMN, CONTENTS_TOP_CATEGORY, 
				    SUB_MODULE_COLUMN, NULL, -1);
	}
	else {
		if (iter)
			module->iter = gtk_tree_iter_copy (iter);
	}

	return module;	
}

static void
module_onequery_fill_model (Module *module)
{
	GObject *manager;
	GtkTreeModel *model;

	manager = FLAT_DATA (module)->manager;

	/* Initial model filling */
	model = module->selector->priv->model;
	flat_init_model_fill (module, model);

	/* Signals handlers */
	g_signal_connect (manager, "target_added",
			  G_CALLBACK (flat_obj_added_cb), module);
	g_signal_connect (manager, "target_removed",
			  G_CALLBACK (flat_obj_removed_cb), module);
	g_signal_connect (manager, "target_updated",
			  G_CALLBACK (flat_obj_updated_cb), module);

	g_signal_connect (manager, "join_added",
			  G_CALLBACK (flat_obj_added_cb), module);
	g_signal_connect (manager, "join_removed",
			  G_CALLBACK (flat_obj_removed_cb), module);
	g_signal_connect (manager, "join_updated",
			  G_CALLBACK (flat_obj_updated_cb), module);

	g_signal_connect (manager, "field_added",
			  G_CALLBACK (flat_obj_added_cb), module);
	g_signal_connect (manager, "field_removed",
			  G_CALLBACK (flat_obj_removed_cb), module);
	g_signal_connect (manager, "field_updated",
			  G_CALLBACK (flat_obj_updated_cb), module);
	g_signal_connect (manager, "fields_order_changed",
			  G_CALLBACK (flat_objs_order_changed_cb), module);

	g_signal_connect (manager, "sub_query_added",
			  G_CALLBACK (flat_obj_added_cb), module);
	g_signal_connect (manager, "sub_query_removed",
			  G_CALLBACK (flat_obj_removed_cb), module);
	g_signal_connect (manager, "sub_query_updated",
			  G_CALLBACK (flat_obj_updated_cb), module);
}

static GSList *
module_onequery_get_objects_list (Module *module)
{
	GSList *retval = NULL;
	GSList *tmplist;

	g_return_val_if_fail (FLAT_DATA (module)->manager, NULL);
	g_return_val_if_fail (GDA_IS_QUERY (FLAT_DATA (module)->manager), NULL);

	/* sub queries first, if applicable */
	if (module->selector->priv->mode & GNOME_DB_SELECTOR_SUB_QUERIES) 
		retval = g_slist_concat (retval, 
					 gda_query_get_sub_queries (GDA_QUERY (FLAT_DATA (module)->manager)));

	/* targets, if applicable */
	if ((module->selector->priv->mode & GNOME_DB_SELECTOR_TARGETS) ||
	    (module->selector->priv->mode & GNOME_DB_SELECTOR_TARGETS_CTS))
		retval = g_slist_concat (retval, 
					 gda_query_get_targets (GDA_QUERY (FLAT_DATA (module)->manager)));

	/* fields, if applicable.
	 * note: if targets are shown then the fields "belonging" to the targets
	 * will be shown as children of the targets and must not appear there */
	if (!(module->selector->priv->mode & GNOME_DB_SELECTOR_TARGETS_CTS)) {
		tmplist = NULL;
		if (module->selector->priv->mode & GNOME_DB_SELECTOR_QVIS_FIELDS)
			tmplist = gda_entity_get_fields (GDA_ENTITY (FLAT_DATA (module)->manager));
		else
			if (module->selector->priv->mode & GNOME_DB_SELECTOR_QALL_FIELDS) 
				tmplist = gda_query_get_all_fields (GDA_QUERY (FLAT_DATA (module)->manager));
		if (tmplist && (module->selector->priv->mode & GNOME_DB_SELECTOR_TARGETS)) {
			GSList *ptr, *newlist = NULL;
			ptr = tmplist;
			while (ptr) {
				if (!GDA_IS_QUERY_FIELD_FIELD (ptr->data) && !GDA_IS_QUERY_FIELD_ALL (ptr->data))
					newlist = g_slist_prepend (newlist, ptr->data);
				ptr = g_slist_next (ptr);
			}
			g_slist_free (tmplist);
			tmplist = g_slist_reverse (newlist);
		}
		if (tmplist)
			retval = g_slist_concat (retval, tmplist);
	}

	return retval;
}

static void
module_onequery_free (Module *module)
{
	GObject *manager = FLAT_DATA (module)->manager;
	GSList *list = module->sub_modules;

	g_assert (manager);

	/* free the extra pixbufs */
	if (QUERY_DATA (module)->field_pixbuf)
		g_object_unref (G_OBJECT (QUERY_DATA (module)->field_pixbuf));

	/* Free the sub modules */
	while (list) {
		(MODULE (list->data)->free) (MODULE (list->data));
		g_free (list->data);
		list = g_slist_next (list);
	}
	if (module->sub_modules) {
		g_slist_free (module->sub_modules);
		module->sub_modules = NULL;
	}


	/* free this module */
	g_signal_handlers_disconnect_by_func (manager,
					      G_CALLBACK (flat_obj_added_cb), module);
	g_signal_handlers_disconnect_by_func (manager,
					      G_CALLBACK (flat_obj_removed_cb), module);
	g_signal_handlers_disconnect_by_func (manager,
					      G_CALLBACK (flat_obj_updated_cb), module);

	if (module->iter)
		gtk_tree_iter_free (module->iter);
	flat_free_mod_data (module);
	g_free (module->mod_data);
	module->mod_data = NULL;
}


static const gchar *
module_onequery_col_name (Module *module, guint colno)
{
	switch (colno) {
	case 0:
		return _("Field");
		break;
	case EXTRA1_COLUMN:
		return _("Type");
		break;
	default:
		return NULL;
		break;
	}
}

/*
 * module_onequery_obj_manager
 *
 * This function is called when a new object is created (GdaQuery). It simply creates a new
 * Module (which will be appended to the list of modules of the module for the queries) for 
 * for the contents of that new query.
 *
 * The same applies for GdaQueryTarget objects, with the module for targets.
 */
static Module *
module_onequery_obj_manager (Module *module, GtkTreeIter *iter, GObject *object)
{
	Module *sub_module = NULL;

	g_assert (object);

	if (GDA_IS_QUERY (object)) {
		if (module->selector->priv->mode & GNOME_DB_SELECTOR_QVIS_FIELDS) 
			sub_module = sel_module_onequery_new (module->selector, FALSE, iter, object);
	}

	if (GDA_IS_QUERY_TARGET (object)) {
		if (module->selector->priv->mode & GNOME_DB_SELECTOR_TARGETS) 
			sub_module = sel_module_onetarget_new (module->selector, FALSE, iter, object);
		if (module->selector->priv->mode & GNOME_DB_SELECTOR_TARGETS_CTS) 
			sub_module = sel_module_onetarget_new_all (module->selector, FALSE, iter, object);
	}

	return sub_module;
}

static void
module_onequery_model_store_data (Module *module, GtkTreeIter *iter)
{
	GObject *obj;
	GtkTreeModel *model;

	model = module->selector->priv->model;
	gtk_tree_model_get (model, iter, OBJ_COLUMN, &obj, -1);

	if (obj && GDA_IS_QUERY_FIELD (obj)) {
		GdaQueryField *field;
		const gchar *str1, *str2;
		gchar *str3;
		GdaDictType *type;
		
		field = GDA_QUERY_FIELD (obj);

		/* data type */
		type = gda_entity_field_get_dict_type (GDA_ENTITY_FIELD (obj));
		if (type)
			str1 = gda_dict_type_get_sqlname (type);
		else
			str1 = _("-");
		/* other */
		str2 = module_onequery_render_qfield_type (GDA_QUERY_FIELD (obj));
		str3 = gda_renderer_render_as_sql (GDA_RENDERER (obj), NULL, 0, NULL);
		
		gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
				    NAME_COLUMN, str3 ? str3 : gda_object_get_name (GDA_OBJECT (obj)),
				    EXTRA1_COLUMN, str1,
				    EXTRA6_COLUMN, str3 ? str3 : gda_object_get_name (GDA_OBJECT (obj)),
				    EXTRA7_COLUMN, str2,
				    PIXBUF_COLUMN, QUERY_DATA (module)->field_pixbuf,
				    ERROR_COLUMN, str3 ? FALSE : TRUE,
				    -1);
		if (str3)
			g_free (str3);

		str1 = gda_object_get_name (GDA_OBJECT (field));
		if (GDA_IS_QUERY_FIELD_FIELD (obj) && (!str1 || !(*str1))) {
			gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
					    NAME_COLUMN, _("Field <no name>"),
					    -1);
		}
	}

	if (obj && GDA_IS_QUERY (obj)) {
		const gchar *str1;
		const gchar *color;

		if (gda_query_get_query_type (GDA_QUERY (obj)) == GDA_QUERY_TYPE_NON_PARSED_SQL)
			color = GNOME_DB_REFERER_UNKNOWN;
		else
			color = gda_referer_activate (GDA_REFERER (obj)) ? GNOME_DB_REFERER_ACTIVE : GNOME_DB_REFERER_INACTIVE;
		/* set color to NULL since libgda is currently broken regarding objects activation changes */
		color = NULL;

		gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
				    EXTRA1_COLUMN, gda_query_get_query_type_string (GDA_QUERY (obj)),
				    BG_COLOR_COLUMN, color,
				    -1);
		str1 = gda_object_get_name (GDA_OBJECT (obj));
		if (!str1 || !(*str1)) {
			gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
					    NAME_COLUMN, _("Query <no name>"),
					    -1);
		}
	}

	if (obj && GDA_IS_QUERY_TARGET (obj)) {
		const gchar *alias = gda_query_target_get_alias (GDA_QUERY_TARGET (obj));
		if (alias && *alias) {
			gchar *str = g_strdup_printf ("%s AS %s", gda_object_get_name (GDA_OBJECT (obj)), alias);
			gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
					    NAME_COLUMN, str, -1);
			g_free (str);
		}
		
	}
}

static const gchar *
module_onequery_render_qfield_type (GdaQueryField *field)
{
	GType ftype = G_OBJECT_TYPE (field);

	if (ftype == GDA_TYPE_QUERY_FIELD_ALL)
		return _("entity.*");
	if (ftype == GDA_TYPE_QUERY_FIELD_FIELD)
		return _("entity.field");
	if (ftype == GDA_TYPE_QUERY_FIELD_VALUE) {
		if (gda_query_field_value_is_parameter (GDA_QUERY_FIELD_VALUE (field)))
			return _("parameter");
		else
			return _("value");
	}
	if (ftype == GDA_TYPE_QUERY_FIELD_FUNC) 
		return _("function");

	/* for other types */
	TO_IMPLEMENT;
	return "???";
}
