/* gnome-db-goo-query-struct.c
 *
 * Copyright (C) 2002 - 2006 Vivien Malerba
 * Copyright (C) 2002 Fernando Martins
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gtk/gtk.h>
#include <libgnomedb/marshal.h>
#include <libgda/libgda.h>
#include <glib/gi18n-lib.h>
#include "gnome-db-goo-query-struct.h"
#include "gnome-db-goo-entity.h"
#include "gnome-db-goo-join.h"
#include "gnome-db-goo-field.h"

static void gnome_db_goo_query_struct_class_init (GnomeDbGooQueryStructClass * class);
static void gnome_db_goo_query_struct_init       (GnomeDbGooQueryStruct * canvas);
static void gnome_db_goo_query_struct_dispose   (GObject *object);
static void gnome_db_goo_query_struct_finalize   (GObject *object);

/* virtual functions */
static void       create_canvas_items (GnomeDbGoo *canvas);
static void       clean_canvas_items  (GnomeDbGoo *canvas);
static void       graph_item_added    (GnomeDbGoo *canvas, GdaGraphItem *item);
static void       graph_item_dropped  (GnomeDbGoo *canvas, GdaGraphItem *item);
static GtkWidget *build_context_menu  (GnomeDbGoo *canvas);

static void       query_destroyed_cb     (GdaQuery *query, GnomeDbGoo *canvas);
static void       query_join_added_cb (GdaQuery *query, GdaQueryJoin *join, GnomeDbGoo *canvas);

static void       drag_action_dcb     (GnomeDbGoo *canvas, GnomeDbGooItem *from_item, GnomeDbGooItem *to_item);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;


struct _GnomeDbGooQueryStructPrivate
{
	GdaQuery    *query;
	GSList      *ent_items;
	GHashTable  *hash_targets; /* key = GdaQueryTarget, value = GnomeDbGooEntity */
	GHashTable  *hash_joins; /* key = GdaQueryJoin, value = GnomeDbGooJoin */
};

GType
gnome_db_goo_query_struct_get_type (void)
{
	static GType type = 0;

	if (G_UNLIKELY (type == 0)) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGooQueryStructClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_goo_query_struct_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGooQueryStruct),
			0,
			(GInstanceInitFunc) gnome_db_goo_query_struct_init
		};		

		type = g_type_register_static (GNOME_DB_TYPE_GOO, "GnomeDbGooQueryStruct", &info, 0);
	}
	return type;
}

static void
gnome_db_goo_query_struct_init (GnomeDbGooQueryStruct * canvas)
{
	canvas->priv = g_new0 (GnomeDbGooQueryStructPrivate, 1);
	canvas->priv->hash_targets = g_hash_table_new (NULL, NULL);
	canvas->priv->hash_joins = g_hash_table_new (NULL, NULL);
	canvas->priv->query = NULL;
}

static void
gnome_db_goo_query_struct_class_init (GnomeDbGooQueryStructClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	parent_class = g_type_class_peek_parent (class);

	/* GnomeDbGoo virtual functions */
	GNOME_DB_GOO_CLASS (class)->create_canvas_items = create_canvas_items;
	GNOME_DB_GOO_CLASS (class)->clean_canvas_items = clean_canvas_items;
	GNOME_DB_GOO_CLASS (class)->graph_item_added = graph_item_added;
	GNOME_DB_GOO_CLASS (class)->graph_item_dropped = graph_item_dropped;
	GNOME_DB_GOO_CLASS (class)->build_context_menu = build_context_menu;

	GNOME_DB_GOO_CLASS (class)->drag_action = drag_action_dcb;
	
	object_class->dispose = gnome_db_goo_query_struct_dispose;
	object_class->finalize = gnome_db_goo_query_struct_finalize;
}

static void
gnome_db_goo_query_struct_dispose (GObject *object)
{
	GnomeDbGooQueryStruct *canvas;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_GOO_QUERY_STRUCT (object));

	canvas = GNOME_DB_GOO_QUERY_STRUCT (object);

	if (canvas->priv) {
		if (canvas->priv->query)
			query_destroyed_cb (canvas->priv->query, GNOME_DB_GOO (canvas));
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void
gnome_db_goo_query_struct_finalize (GObject *object)
{
	GnomeDbGooQueryStruct *canvas;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_GOO_QUERY_STRUCT (object));

	canvas = GNOME_DB_GOO_QUERY_STRUCT (object);

	if (canvas->priv) {
		g_hash_table_destroy (canvas->priv->hash_targets);
		g_hash_table_destroy (canvas->priv->hash_joins);

		g_free (canvas->priv);
		canvas->priv = NULL;
	}

	/* for the parent class */
	parent_class->finalize (object);
}

static void
drag_action_dcb (GnomeDbGoo *canvas, GnomeDbGooItem *from_item, GnomeDbGooItem *to_item)
{
	GdaEntityField *f_field = NULL, *t_field = NULL;
	GdaQueryTarget *f_target = NULL, *t_target = NULL;
	GdaGraphItem *gitem;
	GnomeDbGooItem *citem;
	GdaQueryJoin *join;
	GdaQuery *query;
	GdaQueryCondition *cond, *newcond;
	GdaQueryField *qfield;
	
	/* fields retreival */
	if (GNOME_DB_IS_GOO_FIELD (from_item))
		f_field = gnome_db_goo_field_get_field (GNOME_DB_GOO_FIELD (from_item));
	if (GNOME_DB_IS_GOO_FIELD (to_item))
		t_field = gnome_db_goo_field_get_field (GNOME_DB_GOO_FIELD (to_item));
	
	if (!f_field || !t_field)
		return;
	
	/* targets retreival */
	citem = (GnomeDbGooItem *) gnome_db_goo_field_get_parent_item (GNOME_DB_GOO_FIELD (from_item));
	gitem = gnome_db_goo_item_get_graph_item (citem);
	f_target = (GdaQueryTarget *) gda_graph_item_get_ref_object (gitem);

	citem = (GnomeDbGooItem *) gnome_db_goo_field_get_parent_item (GNOME_DB_GOO_FIELD (to_item));
	gitem = gnome_db_goo_item_get_graph_item (citem);
	t_target = (GdaQueryTarget *) gda_graph_item_get_ref_object (gitem);

	if (!f_target || !GDA_IS_QUERY_TARGET (f_target) ||
	    !t_target || !GDA_IS_QUERY_TARGET (t_target))
		return;

	if (f_target == t_target) {
		GtkWidget *dlg;
		gchar *msg = g_strdup_printf ("<big>%s</big>\n\n%s",
					      _("Can not create join:"),
					      _("A join must be between two different targets. If the "
						"same table or view must be joined to itself, then "
						"create another target for that table or view before "
						"creating the new join."));
		dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_INFO,
							  GTK_BUTTONS_CLOSE, msg);
		g_free (msg);
		gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);

		return;
	}

	query = GNOME_DB_GOO_QUERY_STRUCT (canvas)->priv->query;
	join = gda_query_get_join_by_targets (query, f_target, t_target);
	if (!join) {
		/* create a join */
		join = gda_query_join_new_with_targets (query, f_target, t_target);
		gda_query_join_set_join_type (join, GDA_QUERY_JOIN_TYPE_INNER);
		gda_query_add_join (query, join);
		g_object_unref (join);
	}

	/* add an EQUAL condition */
	newcond = gda_query_condition_new (query, GDA_QUERY_CONDITION_LEAF_EQUAL);
	qfield = gda_query_get_field_by_ref_field (query, f_target, f_field, GDA_ENTITY_FIELD_ANY);
	if (!qfield) {
		qfield = g_object_new (GDA_TYPE_QUERY_FIELD_FIELD, 
				       "dict", gda_object_get_dict (GDA_OBJECT (query)), 
				       "query", query, "target", f_target, "field", f_field, NULL);
		gda_query_field_set_visible (qfield, FALSE);
		gda_entity_add_field (GDA_ENTITY (query), GDA_ENTITY_FIELD (qfield));
		g_object_unref (qfield);
	}
	gda_query_condition_leaf_set_operator (newcond, GDA_QUERY_CONDITION_OP_LEFT, qfield);

	qfield = gda_query_get_field_by_ref_field (query, t_target, t_field, GDA_ENTITY_FIELD_ANY);
	if (!qfield) {
		qfield = g_object_new (GDA_TYPE_QUERY_FIELD_FIELD, 
				       "dict", gda_object_get_dict (GDA_OBJECT (query)), 
				       "query", query, "target", t_target, "field", t_field, NULL);
		gda_query_field_set_visible (qfield, FALSE);
		gda_entity_add_field (GDA_ENTITY (query), GDA_ENTITY_FIELD (qfield));
		g_object_unref (qfield);
	}
	gda_query_condition_leaf_set_operator (newcond, GDA_QUERY_CONDITION_OP_RIGHT, qfield);

	/* integrate newcond in the join's condition */
	cond = gda_query_join_get_condition (join);
	if (cond) {
		if (gda_query_condition_get_cond_type (cond) != GDA_QUERY_CONDITION_NODE_AND) {
			GdaQueryCondition *cond2;

			cond2 = gda_query_condition_new (query, GDA_QUERY_CONDITION_NODE_AND);
			g_return_if_fail (gda_query_condition_node_add_child (cond2, cond, NULL));
			gda_query_join_set_condition (join, cond2);
			g_object_unref (cond2);
			cond = cond2;
		}

		g_return_if_fail (gda_query_condition_node_add_child (cond, newcond, NULL));
		g_object_unref (newcond);
	}
	else {
		gda_query_join_set_condition (join, newcond);
		g_object_unref (newcond);
	}

#ifdef debug
	gda_object_dump (GDA_OBJECT (query), 0);
#endif
}

/**
 * gnome_db_goo_query_struct_new
 * @query: a #GdaQuery object
 *
 * Creates a new canvas widget to display the targets and joins of @query
 *
 * Returns: a new #GnomeDbGooQueryStruct widget
 */
GtkWidget *
gnome_db_goo_query_struct_new (GdaQuery *query, GdaGraphQuery *graph)
{
	GObject *obj;
	GnomeDbGooQueryStruct *canvas;
	GSList *joins, *list;
	gboolean do_layout = TRUE;
	
	g_return_val_if_fail (query && GDA_IS_QUERY (query), NULL);
        obj = g_object_new (GNOME_DB_TYPE_GOO_QUERY_STRUCT, NULL);
	canvas = GNOME_DB_GOO_QUERY_STRUCT (obj);

	/* connecting to the GdaQuery */
	canvas->priv->query = query;
	gda_object_connect_destroy (query, G_CALLBACK (query_destroyed_cb), obj);
	g_signal_connect (G_OBJECT (query), "join_added",
			  G_CALLBACK (query_join_added_cb), obj);

	/* populating the canvas */
	if (graph) {
		do_layout = FALSE;
		g_object_set (obj, "graph", graph, NULL);
	}
	else {
		graph = GDA_GRAPH_QUERY (gda_graph_query_new (query));
		gda_graph_query_sync_targets (GDA_GRAPH_QUERY (graph));
		g_object_set (obj, "graph", graph, NULL);
		g_object_unref (graph);
	}

	/* adding all the joins */
	joins = gda_query_get_joins (query);
	list = joins;
	while (list) {
		query_join_added_cb (query, (GdaQueryJoin *) (list->data), (GnomeDbGoo *) obj);
		list = g_slist_next (list);
	}
	g_slist_free (joins);

	if (do_layout && gnome_db_goo_auto_layout_enabled (GNOME_DB_GOO (canvas))) 
		gnome_db_goo_perform_auto_layout (GNOME_DB_GOO (canvas), FALSE, GNOME_DB_GOO_LAYOUT_DEFAULT);
	gnome_db_goo_center (GNOME_DB_GOO (canvas));

	return GTK_WIDGET (obj);
}

static void
query_destroyed_cb (GdaQuery *query, GnomeDbGoo *canvas)
{
	/* disconnecting signals */
	GNOME_DB_GOO_QUERY_STRUCT (canvas)->priv->query = NULL;
	g_signal_handlers_disconnect_by_func (G_OBJECT (query),
					      G_CALLBACK (query_destroyed_cb), canvas);
	g_signal_handlers_disconnect_by_func (G_OBJECT (query),
					      G_CALLBACK (query_join_added_cb), canvas);

	/* clean the canvas */
	clean_canvas_items (canvas);
}

static void
query_join_added_cb (GdaQuery *query, GdaQueryJoin *join, GnomeDbGoo *canvas)
{
	GnomeDbGooItem *canvas_join;
	GooCanvasItem *canvas_item;
	GnomeDbGooQueryStruct *cc = GNOME_DB_GOO_QUERY_STRUCT (canvas);

	canvas_join = g_hash_table_lookup (cc->priv->hash_joins, join);
	g_return_if_fail (!canvas_join);
		
	canvas_item = gnome_db_goo_join_new (goo_canvas_get_root_item (GOO_CANVAS (canvas)), join,
					     NULL);
		
	g_hash_table_insert (cc->priv->hash_joins, join, canvas_item);
}

/*
 * Add all the required GooCanvasItem objects for the associated #GdaGraph object 
 */
static void
create_canvas_items (GnomeDbGoo *canvas)
{
	GSList *list, *graph_items;
	GdaGraph *graph = gnome_db_goo_get_graph (canvas);

	graph_items = gda_graph_get_items (graph);
	list = graph_items;
	while (list) {
		graph_item_added (canvas, GDA_GRAPH_ITEM (list->data));
		list = g_slist_next (list);
	}
	g_slist_free (graph_items);
}

static void
clean_canvas_items (GnomeDbGoo *canvas)
{
	GSList *list;
	GnomeDbGooQueryStruct *cc = GNOME_DB_GOO_QUERY_STRUCT (canvas);

	/* clean memory */
	g_hash_table_destroy (cc->priv->hash_targets);
	g_hash_table_destroy (cc->priv->hash_joins);
	cc->priv->hash_targets = g_hash_table_new (NULL, NULL);
	cc->priv->hash_joins = g_hash_table_new (NULL, NULL);
}

/*
 * Add the GnomeDbGooEntity corresponding to the graph item
 */
static GtkWidget *canvas_entity_popup_func (GnomeDbGooEntity *ce);
static void
graph_item_added (GnomeDbGoo *canvas, GdaGraphItem *item)
{
	GooCanvasItem *canvas_item;
	GdaObject *ref_obj = gda_graph_item_get_ref_object (item);
	GnomeDbGooQueryStruct *cc = GNOME_DB_GOO_QUERY_STRUCT (canvas);

	if (GDA_IS_QUERY_TARGET (ref_obj)) {
		/* GnomeDbGooEntity for the table */
		canvas_item = gnome_db_goo_entity_new_target (goo_canvas_get_root_item (GOO_CANVAS (canvas)),
							      GDA_QUERY_TARGET (ref_obj), 50., 50.,
							      "popup_menu_func", canvas_entity_popup_func,
							      "graph_item", item,
							      NULL);
		gnome_db_goo_declare_item (canvas, GNOME_DB_GOO_ITEM (canvas_item));
		g_hash_table_insert (cc->priv->hash_targets, ref_obj, canvas_item);
	}
}

static void popup_func_delete_cb (GtkMenuItem *mitem, GnomeDbGooEntity *ce);
static GtkWidget *
canvas_entity_popup_func (GnomeDbGooEntity *ce)
{
	GtkWidget *menu, *entry;

	menu = gtk_menu_new ();
	entry = gtk_menu_item_new_with_label (_("Remove"));
	g_signal_connect (G_OBJECT (entry), "activate", G_CALLBACK (popup_func_delete_cb), ce);
	gtk_menu_append (GTK_MENU (menu), entry);
	gtk_widget_show (entry);

	return menu;
}

static void
popup_func_delete_cb (GtkMenuItem *mitem, GnomeDbGooEntity *ce)
{
	GdaGraphItem *gitem;
	GdaQueryTarget *target;
#ifdef debug
	GdaQuery *query;
#endif

	gitem = gnome_db_goo_item_get_graph_item (GNOME_DB_GOO_ITEM (ce));
	target = (GdaQueryTarget *) gda_graph_item_get_ref_object (gitem);

	g_assert (target && GDA_IS_QUERY_TARGET (target));
#ifdef debug
	query = gda_query_target_get_query (target);
#endif
	gda_object_destroy (GDA_OBJECT (target));

#ifdef debug
	gda_object_dump (GDA_OBJECT (query), 0);
#endif
}

/*
 * Remove the GnomeDbGooEntity corresponding to the graph item
 */
static void
graph_item_dropped (GnomeDbGoo *canvas, GdaGraphItem *item)
{
	GdaObject *ref_obj = gda_graph_item_get_ref_object (item);
	GnomeDbGooQueryStruct *qstruct = GNOME_DB_GOO_QUERY_STRUCT (canvas);
	
	if (GDA_IS_QUERY_TARGET (ref_obj)) {
		GooCanvasItem *canvas_item = g_hash_table_lookup (qstruct->priv->hash_targets,
								  ref_obj);
		if (canvas_item) {
			goo_canvas_item_remove (canvas_item);
			g_hash_table_remove (qstruct->priv->hash_targets, ref_obj);
		}
	}
}

static void popup_add_target_cb (GtkMenuItem *mitem, GnomeDbGooQueryStruct *canvas);
static GtkWidget *
build_context_menu (GnomeDbGoo *canvas)
{
	GtkWidget *menu, *mitem, *submenu, *submitem;
	GSList *tables, *list;
	GnomeDbGooQueryStruct *qstruct = GNOME_DB_GOO_QUERY_STRUCT (canvas);
	GdaObjectRef *refbase;
	GdaDict *dict = gda_object_get_dict (GDA_OBJECT (qstruct->priv->query));
	gboolean other_tables = FALSE;

	tables = gda_dict_database_get_tables (gda_dict_get_database (dict));

	menu = gtk_menu_new ();
	submitem = gtk_menu_item_new_with_label (_("New target from table"));
	gtk_widget_show (submitem);
	gtk_menu_append (menu, submitem);
	if (tables) {
		submenu = gtk_menu_new ();
		gtk_menu_item_set_submenu (GTK_MENU_ITEM (submitem), submenu);
		gtk_widget_show (submenu);
		
		/* add a menu item for each table not currently shown displayed.
		 * WARNING: if a GdaDictTable is detroyed while that menu is displayed, it can still be selected
		 * from within the menu even though it will not do anything.
		 */
		list = tables;
		while (list) {
			mitem = gtk_menu_item_new_with_label (gda_object_get_name (GDA_OBJECT (list->data)));
			gtk_widget_show (mitem);
			gtk_menu_append (submenu, mitem);
			
			refbase = GDA_OBJECT_REF (gda_object_ref_new (dict));
			gda_object_ref_set_ref_object (refbase, GDA_OBJECT (list->data));
			g_object_set_data_full (G_OBJECT (mitem), "table", refbase, g_object_unref);
			
			g_signal_connect (G_OBJECT (mitem), "activate", G_CALLBACK (popup_add_target_cb), canvas);
			other_tables = TRUE;

			list = g_slist_next (list);
		}
		g_slist_free (tables);
	}

	/* sub menu is incensitive if there are no more tables left to add */
	gtk_widget_set_sensitive (submitem, other_tables);

	return menu;
}

static void
popup_add_target_cb (GtkMenuItem *mitem, GnomeDbGooQueryStruct *canvas)
{
	GdaObjectRef *refbase;
	GdaObject *ref_obj;

	refbase = g_object_get_data (G_OBJECT (mitem), "table");
	ref_obj =  gda_object_ref_get_ref_object (refbase);

	if (ref_obj && GDA_IS_DICT_TABLE (ref_obj)) {
		GdaDictTable *table = GDA_DICT_TABLE (ref_obj);
		GdaQueryTarget *target;
		GdaGraphItem *gitem;

		GSList *constraints, *list;
		GSList *targets = gda_query_get_targets (canvas->priv->query), *tlist;

		/* actual target and corresponding GdaGraphItem creation */
		target = gda_query_target_new (canvas->priv->query, gda_object_get_name (GDA_OBJECT (table)));
		gitem = GDA_GRAPH_ITEM (gda_graph_item_new (gda_object_get_dict (GDA_OBJECT (canvas->priv->query)), GDA_OBJECT (target)));
		gda_graph_item_set_position (gitem, GNOME_DB_GOO (canvas)->xmouse, GNOME_DB_GOO (canvas)->ymouse);
		gda_graph_add_item (gnome_db_goo_get_graph (GNOME_DB_GOO (canvas)), gitem);
		g_object_unref (G_OBJECT (gitem));
		gda_query_add_target (canvas->priv->query, target, NULL);
		g_object_unref (target);
		
		/* using database FK constraints to create joins */
		constraints = gda_dict_database_get_tables_fk_constraints (gda_dict_get_database (gda_object_get_dict (GDA_OBJECT (table))),
								     table, NULL, FALSE);
		list = constraints;
		while (list) {
			GdaDictConstraint *cons = GDA_DICT_CONSTRAINT (list->data);
			GdaDictTable *otable;
			GdaQueryTarget *otarget = NULL;
			gboolean table_is_pkey = TRUE;
			
			otable = gda_dict_constraint_get_table (cons);
			if (otable == table) {
				table_is_pkey = FALSE;
				otable = gda_dict_constraint_fkey_get_ref_table (cons);
			}
			
			/* find a suitable target to make a join with */
			tlist = targets;
			while (tlist && !otarget) {
				if (gda_query_target_get_represented_entity (GDA_QUERY_TARGET (tlist->data)) == 
				    (GdaEntity *) otable)
					otarget = GDA_QUERY_TARGET (tlist->data);
				tlist = g_slist_next (tlist);
			}
			
			if (otarget) {
				GdaQueryJoin *join;
				
				/* actual join */
				join = gda_query_join_new_with_targets (canvas->priv->query, otarget, target);
				gda_query_join_set_join_type (join, GDA_QUERY_JOIN_TYPE_INNER);
				gda_query_add_join (canvas->priv->query, join);
				gda_query_join_set_condition_from_fkcons (join);
				g_object_unref (join);
			}
			
			list = g_slist_next (list);
		}
		g_slist_free (constraints);
		g_slist_free (targets);
	}

	/* REM: ref_obj could also be in the future a SELECT #GdaQuery */
}
