#define __TESTPRINT_C__

/*
 * libgnomeprint test program
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors:
 *   Unknown author
 *   Lauris Kaplinski <lauris@helixcode.com>
 *
 * Copyright 2000-2001 Ximian, Inc. and authors
 *
 */

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>
#include <popt.h>
#include <math.h>
#include <libbonobo.h>
#include <libgnomeprint/gnome-print-i18n.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-master.h>
#include <libgnomeprint/gnome-print-paper.h>
#include <libgnomeprint/gnome-print-meta.h>
#include <libgnomeprint/gnome-print-bonobo.h>
#include <libgnomeprint/gnome-print-bonobo-client.h>

#define WE_ARE_LIBGNOMEPRINT_UI
#include <libgnomeprint/gpa-private.h>

#define CM(v) ((v) * 72.0 / 2.54)
#define A4_WIDTH (210.0 * 72 / 25.4)
#define A4_HEIGHT (297.0 * 72 / 25.4)
#define LINESPACING 6.0
#define INDENT 20.0

static gchar *textfile = NULL;
static gchar *dumpfile = NULL;
static gboolean master = FALSE;
static gboolean bonobo = FALSE;
static gboolean papers = FALSE;
static gboolean gpa = FALSE;

static struct poptOption options[] = {
	{ "papers", 'p', POPT_ARG_NONE, &papers, 0,
	  N_("Print Paper database"), NULL},
	{ "gpa", 'g', POPT_ARG_NONE, &gpa, 0,
	  N_("Describe GPA Tree"), NULL},
	{ "text", 't', POPT_ARG_STRING, &textfile, 0,
	  N_("Print text file"), N_("PATH")},
	{ "master", 'm', POPT_ARG_NONE, &master, 0,
	  N_("Use GnomePrintMaster for printing"), NULL},
	{ "dump", 'd', POPT_ARG_STRING, &dumpfile, 0,
	  N_("Dump printout to metafile ('-' for stdout)"), N_("PATH")},
	{ "bonobo", 'b', POPT_ARG_NONE, &bonobo, 0,
	  N_("Split job between bonobo"), NULL},
	POPT_AUTOHELP
	{ NULL, '\0', 0, NULL, 0 }
};

static void do_print_text (GnomePrintConfig *config, GnomePrintContext *ctx, GnomeFont *font, const guchar *buf, gint buflen);

#define G_P_PIXELS 256

static void
print_line (GnomePrintContext *ctx, gdouble x, gdouble y)
{
	gdouble px, py;

	gnome_print_newpath (ctx);
	gnome_print_moveto (ctx, x, y);
	gnome_print_lineto (ctx, x, y + 200.0);
	gnome_print_arcto (ctx, x + 50, y + 200, 50, 180.0, 45.0, 1);

	px = 50 * cos (0.25);
	py = 50 * sin (0.25);

	gnome_print_lineto (ctx, x + 50 + px + 100, y + 100 + py);

	gnome_print_stroke (ctx);
}

static void
print_lines (GnomePrintContext *ctx)
{
	gdouble d0[] = {10.0, 10.0};

	gnome_print_scale (ctx, 0.2, 0.2);

	gnome_print_setlinewidth (ctx, 32.0);
	gnome_print_setdash (ctx, 0, NULL, 0);
	print_line (ctx, 0, 0);
	gnome_print_setdash (ctx, 2, d0, 0);
	print_line (ctx, 200, 0);
	gnome_print_setdash (ctx, 2, d0, 3);
	print_line (ctx, 400, 0);
	gnome_print_setdash (ctx, 2, d0, 7);
	print_line (ctx, 600, 0);
}

static void
print_page (GnomePrintContext *pc)
{
	GnomeGlyphList *gl;
	GnomeFont *font;
	ArtDRect bbox;

	double matrix_slanted[6] = {0.9, 0.1, -0.8, 0.9, 0, 0};
	double matrix2[6] = {1, 0, 0, 1, 0, 100};
	double matrix3[6] = {100, 0, 0, 100, 50, 300};
	double matrix4[6] = {100, 0, 0, 100, 50, 410};

	char img      [G_P_PIXELS] [G_P_PIXELS];
	char colorimg [G_P_PIXELS] [G_P_PIXELS] [3];
	int x, y;
	gint pixels;

	/* Rotated text (using glyphlist) */
	gnome_print_gsave (pc);
	gnome_print_translate (pc, 50, 600);
	gnome_print_rotate (pc, 25);
	font = gnome_font_find_closest_from_weight_slant ("Times", GNOME_FONT_BOLD, 1, 36);
	if (font == NULL) g_error ("Cannot find suitable font, exiting...");
	gnome_print_setfont (pc, font);
	gl = gnome_glyphlist_from_text_dumb (font, 0x000000ff, 0.0, 0.0, _("Gnome-print test page, rotated"));
	gnome_glyphlist_bbox (gl, NULL, 0, &bbox);
	gnome_print_setrgbcolor (pc, 0.9, 0.8, 0.6);
	g_print ("bbox: %g %g %g %g\n", bbox.x0, bbox.y0, bbox.x1, bbox.y1);
	gnome_print_rect_filled (pc, bbox.x0 - 2.0, bbox.y0 - 2.0, bbox.x1 + 2.0 - bbox.x0, bbox.y1 + 2.0 - bbox.y0);
	gnome_print_moveto (pc, 0.0, 0.0);
	gnome_print_glyphlist (pc, gl);
	gnome_glyphlist_unref (gl);
	gnome_print_grestore (pc);

	/* Slanted text (using show) */
	gnome_print_gsave (pc);
	gnome_print_setfont (pc, font);
	gnome_print_moveto (pc, 150, 650);
	gnome_print_concat (pc, matrix_slanted);
	gnome_print_show (pc, _("Gnome-print test page, slanted"));
	gnome_print_grestore (pc);

	gnome_print_gsave (pc);
	font = gnome_font_find_closest_from_weight_slant ("Times", GNOME_FONT_BOLD, 1, 10);
	gnome_print_setfont (pc, font);
	gnome_print_moveto (pc, 150, 700);
	gnome_print_concat (pc, matrix_slanted);
	gnome_print_show (pc, _("Gnome-print test page, slanted. Small"));
	gnome_print_grestore (pc);

	/* Lines */
	gnome_print_gsave (pc);
	gnome_print_translate (pc, 50, 700);
	print_lines (pc);
	gnome_print_grestore (pc);

#if 0
  font = gnome_font_new ("NimbusRomNo9L-Regu", 36);
  gnome_print_setfont (pc, font);
  gnome_print_show (pc, "Gnome-print test page");
  gnome_print_grestore (pc);

  gnome_print_setrgbcolor (pc, 1, 0, 0.5);
  gnome_print_moveto (pc, 50, 50);
  gnome_print_lineto (pc, 50, 90);
  gnome_print_curveto (pc, 70, 90, 90, 70, 90, 50);

  gnome_print_show (pc, "Gnome-print test page");
  gnome_print_grestore (pc);
#endif

  gnome_print_setrgbcolor (pc, 1, 0, 0.5);
  gnome_print_moveto (pc, 50, 50);
  gnome_print_lineto (pc, 50, 90);
  gnome_print_curveto (pc, 70, 90, 90, 70, 90, 50);
  gnome_print_closepath (pc);
  gnome_print_fill (pc);

  pixels = G_P_PIXELS;
  
  for (y = 0; y < pixels; y++)
      for (x = 0; x < pixels; x++)
	img[y][x] = ((x+y)*256/pixels)/2;

  gnome_print_gsave (pc);
    gnome_print_concat (pc, matrix3);
    gnome_print_moveto (pc, 0, 0);
    gnome_print_grayimage (pc, (char *)img, pixels, pixels, pixels);
  gnome_print_grestore (pc);
  
#if 0
  gnome_print_gsave (pc);
    g_print ("Image slanted\n");
    gnome_print_concat (pc, matrix_slanted);
    gnome_print_moveto (pc, 10, 10);
    gnome_print_grayimage (pc, (char *)img, pixels, pixels, pixels);
    gnome_print_moveto (pc, 10, 100);
    gnome_print_grayimage (pc, (char *)img, pixels, pixels, pixels);
    gnome_print_moveto (pc, 10, 100);
    gnome_print_grayimage (pc, (char *)img, pixels, pixels, pixels);
  gnome_print_grestore (pc);
#endif	

  for (y = 0; y < pixels; y++)
      for (x = 0; x < pixels; x++)
	{
	  colorimg[y][x][0] = (x + y) >> 1;
	  colorimg[y][x][1] = (x + (pixels - 1  - y)) >> 1;
	  colorimg[y][x][2] = ((pixels - 1 - x) + y) >> 1;
	}

  gnome_print_gsave (pc);
    gnome_print_concat (pc, matrix4);
    gnome_print_moveto (pc, 0, 0);
    gnome_print_rgbimage (pc, (char *)colorimg, pixels, pixels, pixels * 3);
  gnome_print_grestore (pc);

  gnome_print_concat (pc, matrix2);
  gnome_print_setrgbcolor (pc, 0, 0.5, 0);
  gnome_print_moveto (pc, 50, 50);
  gnome_print_lineto (pc, 50, 90);
  gnome_print_curveto (pc, 70, 90, 90, 70, 90, 50);
  gnome_print_closepath (pc);
  gnome_print_stroke (pc);

  gnome_print_concat (pc, matrix2);
  gnome_print_setrgbcolor (pc, 0, 0, 0.5);
  gnome_print_moveto (pc, 50, 50);
  gnome_print_lineto (pc, 50, 90);
  gnome_print_curveto (pc, 70, 90, 90, 70, 90, 50);
  gnome_print_closepath (pc);
  gnome_print_gsave (pc);
  gnome_print_stroke (pc);
  gnome_print_grestore (pc);
  gnome_print_clip (pc);

  gnome_print_moveto (pc, 50, 50);
  gnome_font_unref (font);
  font = gnome_font_find_closest ("Courier", 18);
  gnome_print_setfont (pc, font);
  gnome_print_show (pc, "clip!");

  gnome_font_unref (font);
}

static void
print_test_page (GnomePrintContext *ctx)
{
	gnome_print_beginpage (ctx, "testprint demo page");

	print_page (ctx);

	gnome_print_showpage (ctx);
}

static void
print_bonobo_page (GnomePrintBonobo *gpb, GnomePrintContext *ctx, gdouble width, gdouble height,
		   const Bonobo_PrintScissor *opt_scissor, gpointer data)
{
	g_print ("Printing bonobo page\n");

	print_page (ctx);
}

static void
gpa_node_describe (GPANode *node, GSList *nodes, gint level)
{
	GPANode *child;
	guchar c[32];
	gchar *id, *val;

	if (level > 0) {
		g_snprintf (c, 32, "%%%ds", level);
		g_print (c, " ");
	}

	id = gpa_node_id (node);
	val = gpa_node_get_value (node);
	g_print ("%s:%s\n", (id) ? id : "UNNAMED", (val) ? val : "(empty)");
	g_free (val);
	g_free (id);

	if (!g_slist_find (nodes, node)) {
		GPANode *prev;
		nodes = g_slist_prepend (nodes, node);
		prev = NULL;
		for (child = gpa_node_get_child (node, NULL); child != NULL; child = gpa_node_get_child (node, child)) {
			gpa_node_describe (child, nodes, level + 2);
			if (prev) gpa_node_unref (prev);
			prev = child;
		}
		if (prev) gpa_node_unref (prev);
	}
}

int
main (int argc, const char **argv)
{
	GnomePrintConfig *config;
	GnomePrintContext *pctx;
	GnomePrintMaster *gpm;
	poptContext ctx;
	char **args;
	int ret;

	if (!bonobo_init (&argc, (char **)argv)) {
		g_error ("Could not initialize bonobo");
	}

	/* Parse arguments */
	ctx = poptGetContext (NULL, argc, argv, options, 0);
	if (!ctx) {
		g_error ("Could not parse arguments");
	}

	ret = poptGetNextOpt (ctx);
	if (ret != -1) {
		fprintf (stderr, "%s: %s: %s\n",
			 "testprint",
			 poptBadOption (ctx, POPT_BADOPTION_NOALIAS),
			 poptStrerror (ret));
		exit (1);
	}
	args = (char **) poptGetArgs (ctx);

	/* Get valid GnomePrintConfig to start with */

	config = gnome_print_config_default ();
	if (!config) {
		g_error ("Could not get default print config");
	}

	/* Process command-line arguments */

	if (args && args[0]) {
		gint i;
		for (i = 0; args[i] != NULL; i ++) {
			if (strchr (args[i], '=')) {
				gchar *k, *v;
				k = g_strdup (args[i]);
				v = strchr (k, '=');
				*v++ = '\0';
				if (!gnome_print_config_set (config, k, v)) {
					g_warning ("Setting %s = %s failed", k, v);
				}
				g_free (k);
			} else {
				g_warning ("Argument is not key=value pair: %s", args[i]);
			}
		}
	}

	/* Arguments are processed, free poptContext */
	poptFreeContext (ctx);

	if (papers) {
		GList *plist, *l;
		/* Simply describe paper list */
		g_print ("Papers defined for gnome-print\n");
		plist = gnome_print_paper_get_list ();
		for (l = plist; l != NULL; l = l->next) {
			GnomePrintPaper *p;
			p = (GnomePrintPaper *) l->data;
			g_print ("Paper: [%5g mm x %5g mm] %s\n", p->width * 25.4 / 72, p->height * 25.4 / 72, p->name);
		}
		gnome_print_paper_free_list (plist);
		gnome_print_config_unref (config);
		exit (0);
	}

	if (gpa) {
		GPANode *root;
#if 0
		GSList *nodes, *n;
#endif
		/* Describe GPA tree */
		/* NB! GNOME_PRINT_CONFIG_NODE is not to be used in application now */
		root = GNOME_PRINT_CONFIG_NODE (config);
		gpa_node_describe (root, NULL, 0);
		gnome_print_config_unref (config);
#if 0
		nodes = gpa_list_nodes ();
		for (n = nodes; n != NULL; n = n->next) {
			GPANode *node;
			node = GPA_NODE (n->data);
			if (!node->parent) {
				g_print ("------------ STALE TREE -------------\n");
				gpa_node_describe (node, NULL, 0);
			} else {
				g_print ("Node: %p id:%s qid:%d parent:%p\n", node, GPA_NODE_ID (node), node->qid, node->parent);
			}
		}
#endif
		exit (0);
	}

	/* Now set up output target */

	if (master) {
		/* Create GnomePrintMaster */
		gpm = gnome_print_master_new_from_config (config);
		if (!gpm) {
			g_error ("Cannot create GnomePrintMaster");
		}
		pctx = gnome_print_master_get_context (gpm);
	} else if (dumpfile) {
		/* Create metafile to be dumped */
		pctx = gnome_print_meta_new ();
		g_object_ref (G_OBJECT (pctx));
	} else {
		/* Create just ordinary context */
		pctx = gnome_print_context_new (config);
	}

	if (!pctx) {
		g_error ("Cannot create GnomePrintContext");
	}

	/* Now pctx is final output context */
	if (bonobo) {
		CORBA_ORB orb;
		GnomePrintBonobo *gpb;
		Bonobo_Print bp;
		GnomePrintBonoboData *gpbd;
		CORBA_Environment ev;
		CORBA_char *ior;
		GnomePrintBonoboDimensions *dim;
		orb = bonobo_orb ();
		gpb = gnome_print_bonobo_new (print_bonobo_page, NULL);
		CORBA_exception_init (&ev);
		ior = CORBA_ORB_object_to_string (orb, BONOBO_OBJREF (gpb), &ev);
		CORBA_exception_init (&ev);
		bp = CORBA_ORB_string_to_object (orb, ior, &ev);
		CORBA_free (ior);
		dim = gnome_print_bonobo_dimensions_new (21.0 * 72.0 / 2.54, 29.7 * 72.0 / 2.54);
		gpbd = gnome_print_bonobo_client_remote_render (bp, dim, NULL);

		gnome_print_beginpage (pctx, "Testprint Bonobo demo page");
		gnome_print_bonobo_data_re_render (pctx, 0, 0, gpbd, 0, 0);
		gnome_print_showpage (pctx);
	} else {
		if (textfile) {
			struct stat s;
			gint fh, len;
			guchar *b;
			if (!stat (textfile, &s) && S_ISREG (s.st_mode)) {
				len = s.st_size;
				fh = open (textfile, O_RDONLY);
				if (fh >= 0) {
					b = g_new (guchar, len + 1);
					*(b + len) = '\0';
					read (fh, b, len);
					close (fh);
					do_print_text (config, pctx, NULL, b, len);
					g_free (b);
				}
			}
		} else {
			print_test_page (pctx);
		}
	}

	/* Close and unref output (if we have master/metafile, it is still referenced) */
	gnome_print_context_close (pctx);
	g_object_unref (G_OBJECT (pctx));

	if (master) {
		gnome_print_master_close (gpm);
		gnome_print_master_print (gpm);
		g_object_unref (G_OBJECT (gpm));
	} else if (dumpfile) {
		const guchar *buf;
		gint len;
		FILE *f;
		buf = gnome_print_meta_get_buffer (GNOME_PRINT_META (pctx));
		len = gnome_print_meta_get_length (GNOME_PRINT_META (pctx));
		if (strcmp (dumpfile, "-")) {
			f = fopen (dumpfile, "w");
		} else {
			f = stdout;
		}
		g_return_val_if_fail (f != NULL, 1);
		fwrite (buf, sizeof (guchar), len, f);
		fclose (f);

		g_object_unref (G_OBJECT (pctx));
	}

	gnome_print_config_unref (config);

	return 0;
}

static gint
do_print_paragraph (GnomePrintContext *pc, GnomeFont *font, const guchar *b, const guchar *e, gdouble x0, gdouble *y0, gdouble x1, gdouble y1)
{
	GSList *words;
	const guchar *p;
	gchar *ub, *u;
	gdouble fontheight;
	gint space;
	gdouble x, y;

	if (!font) return TRUE;
	g_print ("Ascender %g\n", gnome_font_get_ascender (font));
	g_print ("Descender %g\n", gnome_font_get_descender (font));
	fontheight = gnome_font_get_ascender (font) + gnome_font_get_descender (font);
	space = gnome_font_face_lookup_default (gnome_font_get_face (font), ' ');

	/* Test for line space */
	if (*y0 - y1 < fontheight) return TRUE;

	/* Split text into words & convert to utf-8 */
	ub = g_new (guchar, (e - b) * 2 + 1);
	u = ub;
	words = NULL;
	p = b;
	while (p < e) {
		while ((p < e) && (*p <= ' ')) p++;
		if (p < e) {
			words = g_slist_prepend (words, u);
			while ((p < e) && (*p > ' ')) {
#if 0
				u += g_unichar_to_utf8 (*p++, u);
#else
				*u++ = *p++;
#endif
			}
			*u++ = '\0';
		}
	}
	words = g_slist_reverse (words);

	x = x0 + INDENT;
	while ((*y0 - y1 >= fontheight) && (words != NULL)) {
		ArtPoint spadv;
		gdouble accwidth, spcwidth;
		gboolean stop;
		GSList *lw;
		gint numwords;
		GnomeGlyphList * gl;
		/* Find actual Y */
		y = *y0 - gnome_font_get_ascender (font);
		/* Find space advance */
		gnome_font_get_glyph_stdadvance (font, space, &spadv);
		accwidth = 0.0;
		spcwidth = 0.0;
		stop = FALSE;
		lw = NULL;
		while ((accwidth <= (x1 - x)) && !stop) {
			gdouble width;
			width = gnome_font_get_width_utf8 (font, (gchar *) words->data);
			if (accwidth > 0.0) spcwidth += spadv.x;
			if ((accwidth == 0.0) || (accwidth + spcwidth + width < (x1 - x))) {
				/* We have room */
				accwidth += width;
				lw = g_slist_prepend (lw, words->data);
				words = g_slist_remove (words, words->data);
				if (!words) stop = TRUE;
			} else {
				stop = TRUE;
			}
		}
		lw = g_slist_reverse (lw);
		/* Typeset */
		numwords = g_slist_length (lw);
		if ((numwords > 1) && (words != NULL)) {
			spcwidth = ((x1 - x) - accwidth) / (numwords - 1);
		} else {
			spcwidth = spadv.x;
		}
		gl = gnome_glyphlist_from_text_dumb (font, 0x000000ff, 0.0, 0.0, "");
		gnome_glyphlist_advance (gl, TRUE);
		while (lw) {
			guchar *str;
			str = (guchar *) lw->data;
			gnome_glyphlist_moveto (gl, x, y);
			if (!strcmp (str, "GNU")) {
				gint glyph;
				gnome_glyphlist_color (gl, 0xff0000ff);
				gnome_glyphlist_rmoveto (gl, 0.0, -fontheight / 3.0);
				glyph = gnome_font_face_lookup_default (gnome_font_get_face (font), 'G');
				gnome_glyphlist_glyph (gl, glyph);
				gnome_glyphlist_color (gl, 0x00ff00ff);
				gnome_glyphlist_rmoveto (gl, 0.0, fontheight / 3.0);
				glyph = gnome_font_face_lookup_default (gnome_font_get_face (font), 'N');
				gnome_glyphlist_glyph (gl, glyph);
				gnome_glyphlist_color (gl, 0x0000ffff);
				gnome_glyphlist_rmoveto (gl, 0.0, fontheight / 3.0);
				glyph = gnome_font_face_lookup_default (gnome_font_get_face (font), 'U');
				gnome_glyphlist_glyph (gl, glyph);
				gnome_glyphlist_color (gl, 0x000000ff);
				gnome_glyphlist_rmoveto (gl, 0.0, -fontheight / 3.0);
			} else {
				gnome_glyphlist_text_dumb (gl, str);
			}
			x += gnome_font_get_width_utf8 (font, str);
			x += spcwidth;
			lw = g_slist_remove (lw, str);
		}
		gnome_print_moveto (pc, 0.0, 0.0);
		gnome_print_glyphlist (pc, gl);
		gnome_glyphlist_unref (gl);
		*y0 = *y0 - gnome_font_get_size (font) - LINESPACING;
		x = x0;
	}

	if (words) g_slist_free (words);

	g_free (ub);

	return FALSE;
}

static void
do_print_text (GnomePrintConfig *config, GnomePrintContext *ctx, GnomeFont *font, const guchar *buf, gint buflen)
{
	gdouble pw, ph;
	gdouble ml, mr, mt, mb;
	gdouble x0, y0, x1, y1;
	gint pagenum;
	const guchar *bend, *b;

	if (!font) font = gnome_font_find_closest_from_weight_slant ("Times", GNOME_FONT_BOOK, 1, 12);
	if (!font) g_error ("Cannot find suitable font, exiting...");

	pw = A4_WIDTH;
	ph = A4_HEIGHT;

	gnome_print_master_get_page_size_from_config (config, &pw, &ph);

	ml = CM(1);
	mr = CM(1);
	mt = CM(1);
	mb = CM(1);
	/* fixme: Parse margins here, as soon as we have assigned correct keys */

	x0 = ml;
	y0 = mb;
	x1 = pw - mr;
	y1 = ph - mt;

	pagenum = 1;
	bend = buf + buflen;
	b = buf;
	while (b < bend) {
		gchar *str;
		gdouble y;
		gboolean stop;
		/* Start page */
		str = g_strdup_printf ("Test Page %d", pagenum++);
		gnome_print_beginpage (ctx, str);
		g_free (str);
		/* Draw margins */
		gnome_print_setrgbcolor (ctx, 0.0, 0.0, 1.0);
		gnome_print_setlinewidth (ctx, 1.0);
		gnome_print_moveto (ctx, x0, y0);
		gnome_print_lineto (ctx, x0, y1);
		gnome_print_lineto (ctx, x1, y1);
		gnome_print_lineto (ctx, x1, y0);
		gnome_print_closepath (ctx);
		gnome_print_stroke (ctx);
		/* Draw paragraphs */
		y = y1;
		stop = FALSE;
		while (!stop) {
			/* Find start of paragraph */
			while ((b < bend) && (isspace (*b) || iscntrl (*b))) b += 1;
			if (b < bend) {
				const guchar *e;
				/* Search paragraph end */
				for (e = b; e < (bend - 1); e++) {
					if ((*e == '\n') && (*(e + 1) == '\n')) break;
				}
				if (e == (bend - 1)) e += 1;
				stop = do_print_paragraph (ctx, font, b, e, x0, &y, x1, y0);
				b = e;
			} else {
				stop = TRUE;
			}
		}
		/* Finish page */
		gnome_print_showpage (ctx);
	}
}
