/* workspace object */

/*
 * Copyright (C) 2001 Havoc Pennington
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "workspace.h"
#include "xutils.h"

static GHashTable* workspace_hash = NULL;

struct _WnckWorkspacePrivate
{
  int number;
  
};

enum {
  
  LAST_SIGNAL
};

static void wnck_workspace_init        (WnckWorkspace      *workspace);
static void wnck_workspace_class_init  (WnckWorkspaceClass *klass);
static void wnck_workspace_finalize    (GObject        *object);


static gpointer parent_class;
static guint signals[LAST_SIGNAL] = { 0 };

GType
wnck_workspace_get_type (void)
{
  static GType object_type = 0;

  g_type_init ();
  
  if (!object_type)
    {
      static const GTypeInfo object_info =
      {
        sizeof (WnckWorkspaceClass),
        (GBaseInitFunc) NULL,
        (GBaseFinalizeFunc) NULL,
        (GClassInitFunc) wnck_workspace_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (WnckWorkspace),
        0,              /* n_preallocs */
        (GInstanceInitFunc) wnck_workspace_init,
      };
      
      object_type = g_type_register_static (G_TYPE_OBJECT,
                                            "WnckWorkspace",
                                            &object_info, 0);
    }
  
  return object_type;
}

static void
wnck_workspace_init (WnckWorkspace *workspace)
{
  workspace->priv = g_new0 (WnckWorkspacePrivate, 1);

  workspace->priv->number = -1;
}

static void
wnck_workspace_class_init (WnckWorkspaceClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  
  parent_class = g_type_class_peek_parent (klass);
  
  object_class->finalize = wnck_workspace_finalize;

}

static void
wnck_workspace_finalize (GObject *object)
{
  WnckWorkspace *workspace;

  workspace = WNCK_WORKSPACE (object);

  
  g_free (workspace->priv);
  
  G_OBJECT_CLASS (parent_class)->finalize (object);
}

/**
 * wnck_workspace_get:
 * @number: a workspace index
 * 
 * Gets the workspace numbered @number, or returns %NULL if
 * no such workspace exists.
 * 
 * Return value: the workspace, or %NULL
 **/
WnckWorkspace*
wnck_workspace_get (int number)
{
  /* We trust this function with property-provided numbers, it
   * must reliably return NULL on bad data
   */
  if (workspace_hash == NULL)
    return NULL;
  else
    return g_hash_table_lookup (workspace_hash, &number);
}

/**
 * wnck_workspace_get_number:
 * @space: a #WnckWorkspace
 * 
 * 
 * 
 * Return value: get the index of the workspace
 **/
int
wnck_workspace_get_number (WnckWorkspace *space)
{
  g_return_val_if_fail (WNCK_IS_WORKSPACE (space), 0);
  
  return space->priv->number;
}

/**
 * wnck_workspace_activate:
 * @space: a #WnckWorkspace
 * 
 * Ask window manager to make @space the active workspace.
 * 
 **/
void
wnck_workspace_activate (WnckWorkspace *space)
{
  g_return_if_fail (WNCK_IS_WORKSPACE (space));

  _wnck_activate_workspace (space->priv->number);
}

WnckWorkspace*
_wnck_workspace_create  (int number)
{
  WnckWorkspace *space;
  
  if (workspace_hash == NULL)
    workspace_hash = g_hash_table_new (g_int_hash, g_int_equal);

  g_return_val_if_fail (g_hash_table_lookup (workspace_hash, &number) == NULL,
                        NULL);
  
  space = g_object_new (WNCK_TYPE_WORKSPACE, NULL);
  space->priv->number = number;

  g_hash_table_insert (workspace_hash, &space->priv->number, space);

  /* Hash now owns one ref, caller gets none */
  
  return space;
}

void
_wnck_workspace_destroy (WnckWorkspace *space)
{
  g_return_if_fail (wnck_workspace_get (space->priv->number) == space);
  
  g_hash_table_remove (workspace_hash, &space->priv->number);

  g_return_if_fail (wnck_workspace_get (space->priv->number) == NULL);

  space->priv->number = -1;
  
  /* remove hash's ref on the workspace */  
  g_object_unref (G_OBJECT (space));
}
