'''
Defines the L{AccessEngine} class which runs the main program loop.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import EventManager, ViewManager, TierManager, DeviceManager, SettingsManager
import gobject, time, signal
from i18n import _

# a tiny delay in the main loop that lets other threads run and doesn't suck
# up all available CPU time
LOOP_INTERVAL = 1e-5

class AccessEngine(object):
  '''
  Runs the main bonobo loop so that AT-SPI events are received and GUI monitors
  function properly. Allows clients to register for callbacks from the main loop
  on a set interval. Maintains a dictionary of managers that can be loaned out
  to L{Task.Tools} objects
  
  @ivar callbacks: Functions to invoke on each event loop iteration
  @type callbacks: list
  @ivar vm: Reference to the L{ViewManager}
  @type vm: L{ViewManager}
  @ivar tm: Reference to the L{TierManager}
  @type tm: L{TierManager}
  @ivar em: Reference to the L{EventManager}
  @type em: L{EventManager}
  @ivar dm: Reference to the L{DeviceManager}
  @type dm: L{DeviceManager}
  @ivar sm: Reference to the L{SettingsManager}
  @type sm: L{SettingsManager}
  @ivar manager_dict: Dictionary of managers keyed by manager names (device_man,
    event_man, tier_man, and view_man)
  @type manager_dict: dictionary
  @ivar none_dict: Dictionary of None values with the same keys as 
    L{manager_dict}
  @type none_dict: dictionary
  @ivar profile: Name of the profile this instance of L{AccessEngine} was
      started under
  @type profile: string
  '''
  def __init__(self, profile):
    '''
    Creates the L{ViewManager}, L{TierManager}, and L{EventManager} objects.
    Calls the init method here to intialize the managers with references to
    other managers.

    @param profile: Name of the profile LSR was executed under
    @type profile: string
    '''
    self.profile = profile
    self.callbacks = []
    self.alive = False
    self.manager_dict = None
    self.none_dict = None
    
    # build all the managers (include reference to AccessEngine
    self.sm = SettingsManager.SettingsManager(self)
    self.em = EventManager.EventManager(self)
    self.vm = ViewManager.ViewManager(self)
    self.tm = TierManager.TierManager(self)
    self.dm = DeviceManager.DeviceManager(self)

  def _init(self, **managers):
    '''
    Calls init on each manager providing it with references to the other
    managers in case it wants to hold one or more references. Called implicitly
    by L{run}.
    
    @keyword event_man: Reference to the L{EventManager}
    @type event_man: L{EventManager}
    @keyword view_man: Reference to the L{ViewManager}
    @type view_man: L{ViewManager}
    @keyword tier_man: Reference to the L{TierManager}
    @type tier_man: L{TierManager}
    @keyword device_man: Reference to the L{DeviceManager}
    @type device_man: L{DeviceManager}
    '''
    # store the dictionary of all managers so we can quickly loan them out in 
    # the future to other objects that need them
    self.manager_dict = managers
    # create a dictionary with the same keys, but None values so we can quickly
    # unloan managers in the future
    self.none_dict = dict([(name, None) for name in managers.keys()])
    # init the managers, in correct dependency order
    self.sm.init(**managers)
    self.dm.init(**managers)
    self.em.init(**managers)
    self.vm.init(**managers)
    self.tm.init(**managers)
    
  def _close(self):
    '''
    Shutsdown the L{DeviceManager} and all its devices. Called implicitly by
    L{run} when the main loop ends.
    '''
    self.dm.close()
    self.tm.close()
    self.vm.close()
    self.sm.close()
    self.em.close()
    
  def quit(self, *args):
    '''
    Sets the alive flag to false so that the L{AccessEngine} loop may quit. 
    Extra parameters are ignored but captured by args so that this method may be
    reigstered as a signal handler callback.
    '''
    self.alive = False
      
  def loanManagers(self, obj):
    '''
    Updates the namespace dictionary of the given object with the names and 
    values of the managers stored in L{manager_dict}. These references can be
    wiped out by a later call to L{unloanManagers}.
    
    @param obj: An object with a __dict__ attribute
    @type obj: object
    '''
    obj.__dict__.update(self.manager_dict)
  
  def unloanManagers(self, obj):
    '''
    Updates the namespace dictionary of the given object with the names and 
    None values of stored in L{none_dict}. This effectively wipes out references
    to managers previously loaned out by L{loanManagers}.
    
    @param obj: An object with a __dict__ attribute
    @type obj: object
    '''
    obj.__dict__.update(self.none_dict)

  def addTimer(self, callback):
    '''
    Creates a timer that calls the given callback on each iteration of the main
    loop on approximately the interval specified by L{LOOP_INTERVAL}.

    @param callback: Function to call when the timer fires
    @type callback: callable
    '''
    self.callbacks.append(callback)

  def removeTimer(self, callback):
    '''
    Ceases calls to the given registered callback.

    @param callback: Function to stop calling
    @type callback: callable
    @raise ValueError: When the callback is not already registered
    '''
    self.callbacks.remove(callback)

  def run(self):
    '''
    Runs an event loop by creating a gobject.MainLoop object and calling the
    iteration method of its gobject.MainContext. Makes the timer callbacks
    registered via L{AccessEngine.addTimer} on each iteration of the loop.
    '''
    # set the alive flag to true
    self.alive = True
    
    # register a signal handlers for gracefully killing the loop
    signal.signal(signal.SIGTERM, self.quit)
    signal.signal(signal.SIGINT, self.quit)
    
    # initialize all managers
    self._init(event_man=self.em, view_man=self.vm, tier_man=self.tm,
              device_man=self.dm, sett_man=self.sm)
    
    # create a main loop object and get its context
    ml = gobject.MainLoop()
    context = ml.get_context()
    # loop until the alive flag is unset
    while self.alive:
      # sleep for a bit to release the GIL and avoid sucking up the CPU
      time.sleep(LOOP_INTERVAL)
      # handle all outstanding events
      while context.pending():
        context.iteration()
      # before making registered callbacks
      for cb in self.callbacks:
        cb()

    # uninitialize all managers
    self._close()
    
  def getProfile(self):
    '''
    Gets the name of the profile this L{AccessEngine} instance is running under.
    
    @return: Name of the profile
    @rtype: string
    '''
    return self.profile
