'''
Defines the L{UIRegistrar} class responsible for managing an on-disk repository 
of User Interface Elements (UIEs) that define the L{Perk}s, Devices, Choosers,
and Monitors available to the user and the system. Defines a helper class 
L{Profile} that is used internally by the registrar to represent collections of
UIEs that are to be loaded when LSR starts, when any L{Tier} starts, or when a
particular L{Tier} starts.

@var MODULE_FILE_TEMPLATE: Template for forming an module file name from a UIE
    name
@type MODULE_FILE_TEMPLATE: string
@var PROFILE_FILE_TEMPLATE: Template for forming an profile file name from a 
    profile name
@type PROFILE_FILE_TEMPLATE: string
@var ROOT_PATH: Root path of the UIE repository relative to the LSR lib folder
@type ROOT_PATH: string
@var INSTALL_PATH: Path where folders named after kinds of UIEs contain symlinks
    to real Python UIE modules
@type INSTALL_PATH: string
@var PROFILE_PATH: Path where serialized L{Profile}s live
@type PROFILE_PATH: string
@var STARTUP: Indicates UIEs to be loaded on LSR startup
@type STARTUP: string
@var ONE_TIER: Indicates UIEs to be loaded when a particular L{Tier} is created
@type ONE_TIER: string
@var ANY_TIER: Indicates UIEs to be loaded when any L{Tier} is created
@type ANY_TIER: string
@var PERK: Kind of UIE responsible for registering L{Task}s that handle 
  L{AEEvent}s and L{AEInput.Gesture}s
@type PERK: string
@var DEVICE: Kind of UIE responsible for managing a method of input 
    (e.g keyboard), managing a method of output (e.g speech), or both (e.g. 
    Braille device)
@type DEVICE: string
@var CHOOSER: Kind of UIE responsible for interacting with the user via more 
    than the usual LSR key combos (e.g. configuration, help, search)
@type CHOOSER: string
@var MONITOR: Kind of UIE responsible for creating a human-readable 
    representation (e.g. GUI dialog, log file) of some AccessEngine data (e.g. 
    input, output, events) for the purposes of development and debugging
@type MONITOR: string
@var ALL_KINDS: List of all known kinds of UIEs
@type ALL_KINDS: list

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import os, glob, imp, cPickle, sys, logging
import AEMonitor, AEChooser, AEOutput, AEInput, Perk
from i18n import _
from LSRConstants import HOME_USER

# create a logger for UIRegistrar
log = logging.getLogger('UIRegistrar')

# templates for module file names
MODULE_FILE_TEMPLATE = '%s'+os.extsep+'py'
PROFILE_FILE_TEMPLATE = '%s'+os.extsep+'profile'

# paths to various parts of the repository
ROOT_PATH = os.path.join(HOME_USER, 'ui')
INSTALL_PATH = os.path.join(ROOT_PATH, 'installed')
PROFILE_PATH = os.path.join(ROOT_PATH, 'profiles')
VERSION_PATH = os.path.join(ROOT_PATH, 'version')

# constants representing when associated UIEs are loaded
STARTUP = 'startup'
ONE_TIER = 'onetier'
ANY_TIER = 'anytier'

# constants representing kinds of UIEs and the folders where they are stored 
# under the INSTALL_PATH
PERK = 'perk'
DEVICE = 'device'
CHOOSER = 'chooser'
MONITOR = 'monitor'

# list of all supports types of UIEs
ALL_KINDS = [DEVICE, CHOOSER, MONITOR, PERK]
ALL_TIMES = [STARTUP, ONE_TIER, ANY_TIER]

class Profile(object):
  '''
  Associates the names of installed UIEs with times when they should be
  automatically loaded by LSR under a particular user profile. Supported times
  for automatically loading UIEs include when LSR starts (L{STARTUP}), when any
  new L{Tier} is created (L{ANY_TIER}), or when a L{Tier} with a particular name
  is created (L{ONE_TIER}).
  
  @ivar name: Name of this L{Profile}
  @type name: string
  @ivar on_startup: Lists of UIE names keyed by UIE kind
  @type on_startup: dictionary
  @ivar on_tier: Lists of UIE names keyed by UIE kind in dictionaries keyed by
      L{Tier} name
  @type on_tier: dictionary
  @ivar on_any_tier: Lists of UIE names keyed by UIE kind
  @type on_any_tier: dictionary
  '''
  def __init__(self, name):
    '''
    Stores the name of the L{Profile}. Initializes the instance dictionaries.
    
    @param name: Name of the L{Profile}
    @type name: string
    '''
    self.name = name
    self.on_startup = {}
    self.on_tier = {}
    self.on_any_tier = {}
    
  def addToStartup(self, kind, name, index):
    '''    
    Adds the name of a UIE of the given kind to the L{on_startup} dictionary so
    that it is loaded automatically when LSR starts. The index determines when
    in the list of all L{on_startup} UIEs of the given kind the named UIE is
    loaded.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @param index: Load order of the UIE when multiple UIEs are to be loaded at
        the given time. A value of None meaning after all other UIEs of the same
        kind.
    @type index: integer or None
    @raise ValueError: When the UIE with the given name is already in the 
      startup list
    '''
    uies = self.on_startup.setdefault(kind, [])
    if name in uies:
      raise ValueError(_('A %s named %s is already associated with profile %s' \
                         ' at startup') % (kind, name, self.name))
    if index is None: 
      index = len(uies)
    uies.insert(index, name)
  
  def removeFromStartup(self, kind, name):
    '''    
    Removes the name of a UIE of the given kind from the L{on_startup}
    dictionary so that it is no longer loaded automatically on LSR startup.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @raise ValueError: When the UIE of the given name and kind is not associated
        with this profile to be loaded on startup
    '''
    uies = self.on_startup.get(kind, [])
    try:
      uies.remove(name)
    except ValueError:
      raise ValueError(_('A %s named %s is not associated with profile %s ' \
                         'at startup') % (kind, name, self.name))
  
  def addToTier(self, kind, name, tier_name, index):
    '''
    Adds the name of a UIE of the given kind to the L{on_tier} dictionary so
    that it is loaded automatically when a L{Tier} with the given name is
    created. The index determines when in the list of all L{on_tier} UIEs of the
    given kind the named UIE is loaded.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @param tier_name: Name of the L{Tier}
    @type tier_name: string
    @param index: Load order of the UIE when multiple UIEs are to be loaded at
        the given time. A value of None meaning after all other UIEs of the same
        kind.
    @type index: integer or None
    @raise ValueError: When the UIE of the given name and kind is not associated
        with this profile to be loaded on the given L{Tier}
    '''
    kinds = self.on_tier.setdefault(tier_name, {})
    uies = kinds.setdefault(kind, [])
    if name in uies:
      raise ValueError(_('A %s named %s is already associated with profile %s' \
                       ' for tier %s') % (kind, name, self.name, tier_name))
    if index is None: 
      index = len(uies)
    uies.insert(index, name)
  
  def removeFromTier(self, kind, name, tier_name):
    '''    
    Removes the name of a UIE of the given kind from the L{on_tier}
    dictionary so that it is no longer loaded automatically on when a L{Tier}
    with the given name is created.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @param tier_name: Name of the L{Tier}
    @type tier_name: string
    @raise ValueError: When the UIE of the given name and kind is not associated
        with this profile to be loaded when the given L{Tier} starts
    '''
    kinds = self.on_tier.get(tier_name, {})
    uies = kinds.get(kind, [])
    try:
      uies.remove(name)
    except ValueError:
      raise ValueError(_('A %s named %s is not associated with profile %s ' \
                         'for tier %s') % (kind, name, self.name, tier_name))
  
  def addToAnyTier(self, kind, name, index):
    '''     
    Adds the name of a UIE of the given kind to the L{on_any_tier} dictionary
    so that it is loaded automatically when any L{Tier} starts. The index
    determines when in the list of all L{on_any_tier} UIEs of the given kind
    the named UIE is loaded.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @param index: Load order of the UIE when multiple UIEs are to be loaded at
        the given time. A value of None meaning after all other UIEs of the same
        kind.
    @type index: integer or None
    @raise ValueError: When the UIE of the given name and kind is not associated
        with this profile to be loaded on any L{Tier}
    '''
    uies = self.on_any_tier.setdefault(kind, [])
    if name in uies:
      raise ValueError(_('A %s named %s is already associated with profile %s' \
                       ' for any tier') % (kind, name, self.name))
    if index is None: 
      index = len(uies)
    uies.insert(index, name)
  
  def removeFromAnyTier(self, kind, name):
    '''  
    Removes the name of a UIE of the given kind from the L{on_any_tier}
    dictionary so that it is no longer loaded automatically when any L{Tier} is
    created.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @raise ValueError: When the UIE of the given name and kind is not associated
        with this profile to be loaded when any L{Tier} is created
    '''
    uies = self.on_any_tier.get(kind, [])
    try:
      uies.remove(name)
    except ValueError:
      raise ValueError(_('A %s named %s is not associated with profile %s ' \
                         'for any tier' % (kind, name, self.name)))
    
  def removeFromAll(self, kind, name):
    '''
    Removes all references to the UIE with the given name and kind from the
    L{on_startup}, L{on_tier}, and L{on_any_tier} dictionaries. Ignores all
    exceptions.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    '''
    try:
      self.removeFromStartup(kind, name)
    except ValueError:
      pass
    try:
      self.removeFromAnyTier(kind, name)
    except ValueError:
      pass
    for tier_name in self.on_tier:
      try:
        self.removeFromTier(kind, name, tier_name)
      except ValueError:
        pass
      
  def listStartup(self, kind):
    '''
    Gets a list of all UIE names of the given kind that are to be loaded at LSR
    startup.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @return: Names of all UIEs to be loaded at LSR startup
    @rtype: list of string
    '''
    return self.on_startup.get(kind, [])
  
  def listTier(self, kind, tier_name=None):
    '''
    Gets a list of all UIE names of the given kind that are to be loaded when 
    a L{Tier} with the given name is created. If the given L{Tier} name is None,
    then list all names and the UIEs associated with them.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param tier_name: Name of the L{Tier}
    @type tier_name: string
    @return: Names of all UIEs to be loaded at L{Tier} creation time
    @rtype: list of string or list of dictionaries
    '''
    # get all UIEs for all named tiers
    if tier_name is None:
      l = []
      for name, kinds in self.on_tier.items():
        try:
          l.append((name, kinds[kind]))
        except KeyError:
          pass
      return l
    else:
      # get UIEs for the named tier only
      kinds = self.on_tier.get(tier_name, {})
      return kinds.get(kind, [])
  
  def listAnyTier(self, kind):
    '''
    Gets a list of all UIE names of the given kind that are to be loaded when 
    any L{Tier} is created.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @return: Names of all UIEs to be loaded at L{Tier} creation time
    @rtype: list of string
    '''
    return self.on_any_tier.get(kind, [])
  
  def exists(self):
    '''
    Gets if this L{Profile} exists on disk.
    
    @return: Does the profile exist (True) or not (False)
    @rtype: boolean
    '''
    file_name = PROFILE_FILE_TEMPLATE % self.name
    return os.path.exists(os.path.join(PROFILE_PATH, file_name))
  
  def delete(self):
    '''
    Deletes this L{Profile} from disk.
    
    @raise OSError: When the profile file cannot be deleted
    '''
    fn = os.path.join(PROFILE_PATH, PROFILE_FILE_TEMPLATE % self.name)
    os.unlink(fn)
      
  def save(self):
    '''
    Serializes this L{Profile} instance to disk using the cPickle module. The
    highest pickle protocol available is used to write the data.
    
    @raise IOError: When the pickle file cannot be created
    '''
    file_name = PROFILE_FILE_TEMPLATE % self.name
    f = file(os.path.join(PROFILE_PATH, file_name), 'wb')
    profile = cPickle.dump(self, f, cPickle.HIGHEST_PROTOCOL)
    f.close()
  
  @classmethod
  def load(cls, name):
    '''
    Loads a serialized instance of this class from disk using the cPickle module.
    
    @note: This is a class method.
    @param name: Name of the L{Profile} to load
    @type name: string
    @return: Previously pickled instance
    @rtype: L{Profile}
    @raise ValueError: When a pickled L{Profile} with the give name cannot be
        read from disk.
    '''
    try:
      file_name = PROFILE_FILE_TEMPLATE % name
      f = file(os.path.join(PROFILE_PATH, file_name), 'rb')
      profile = cPickle.load(f)
      f.close()
    except IOError:
      raise ValueError(_('A profile with name %s does not exist') % name)
    return profile
  
class UIRegistrar(object):
  '''
  Manages a repository of User Interface Elements (UIE)s on disk and allows
  other objects to query the repository to:
  
      - install new UIEs
      - uninstall existing UIEs
      - list installed UIEs
      - associate installed UIEs with a particular profile
      - disassociate UIEs from a particular profile
      - list UIEs associated with a particular profile
      - create profiles
      - remove profiles
      - list profiles
      - load a UIE by name
      - load associated UIEs
      - list the contents of the repository
      
  Methods to install, uninstall, associate, disassociate, and list UIEs are
  primarily of interest to third-party UIE developers and expert users. These
  methods allow extensions to be added to LSR and associated with profiles so
  they are loaded automatically at startup or when L{Tier}s are created. The
  L{LSRMain} script defines a command line interface for accessing these
  methods. See the documentation in that module for details.
  
  Methods to load UIEs are of interest to LSR developers. Given a UIE kind and
  name, the L{loadOne} method will import the Python module containing the named
  UIE and return a reference to the class in that module of the same name (if
  the UIE module is properly installed). Given a UIE kind and name along with
  the name of a L{Profile} and a indicator of when the request is taking place
  (L{STARTUP}, L{ONE_TIER}, L{ANY_TIER}), the L{loadAssociated} method will
  import all Python modules containing the UIEs to be loaded at the indicated
  time and return a list of UIE objects sorted in the desired load order. 
  Objects, not classes, are returned by this method implying any initialization
  requiring data from an external source must be done outside the constructor.
  '''
  def _ensurePathsExist(self):
    '''
    Creates the necessary directory structure for the UIE repository on disk.
    
    @raise OSError: When at least one required directory cannot be created
    '''
    if not os.path.exists(PROFILE_PATH):
      os.makedirs(PROFILE_PATH)
    for kind in ALL_KINDS:
      path = os.path.join(INSTALL_PATH, kind)
      if not os.path.exists(path):
        os.makedirs(path)

  def _isInstalled(self, kind, name):
    '''
    Gets if a UIE of the given kind and name is installed (i.e. has a symlink
    in the installed directory).
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @return: Is the UIE installed (True) or not (False)?
    @rtype: boolean
    '''
    module_name = MODULE_FILE_TEMPLATE % name
    return os.path.lexists(os.path.join(INSTALL_PATH, kind, module_name))
  
  def _ensureInstalled(self, kind, name):
    '''
    Ensures the UIE with the given kind and name is installed.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @raise ValueError: When the UIE is not installed
    '''
    if not self._isInstalled(kind, name):
      raise ValueError(_('A %s with name %s is not installed') % (kind, name))
  
  def _ensureNotInstalled(self, kind, name):
    '''
    Ensures the UIE with the given kind and name is not installed.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @raise ValueError: When the UIE is already installed
    '''
    if self._isInstalled(kind, name):
      raise ValueError(_('A %s with name %s is already installed')%(kind, name))
    
  def _ensureAssociationValid(self, kind, when):
    '''
    Ensures the given kind of UIE can be loaded at the time given by when. At
    present, the following pairings are valid:
    
    Chooser: None
    Device: STARTUP
    Monitor: STARTUP
    Perk: ANY_TIER, ONE_TIER
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param when: Identifier of when the UIE will be loaded automatically: on
       LSR startup (L{STARTUP}), on creation of any L{Tier} (L{ANY_TIER}), or
       on creation of a particular L{Tier} (L{ONE_TIER})
    @type when: string
    @raise ValueError: When the pairing is invalid
    '''
    if kind == PERK and when in (ANY_TIER, ONE_TIER):
      return
    elif kind == CHOOSER or when != STARTUP:
      raise ValueError(_('A %s cannot be loaded on %s') % (kind, when))
  
  def _ensureUIEValid(self, kind, name, real_path):
    '''    
    Ensures the UIE module at the given path contains a class matching the
    stated name. 

    @note: No longer checks if a class fits the definition of the given kind
    of UIE. This validation is difficult when the module imports dependencies
    and actually provides little useful information about whether or not the UIE
    is implemented correctly. It can only tell us if it provides the proper 
    interface. We have gracefully degredation at runtime, so why bother 
    "almost ensuring" it's valid at install time?
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @param real_path: Real path to the UIE module located on disk
    @type real_path: string
    @raise NotImplementedError: When the UIE class does not fit the definition
      of the given kind
    @raise IOError: When the UIE module cannot be read
    @raise AttributeError: When the UIE module does not have a class of the 
      given name
    '''
    text = file(real_path).read()
    if text.find('class %s' % name) < 0:
      raise AttributeError
      
  def _getUIEInstance(self, kind, name, path=None):
    '''
    Gets an instance of the UIE class having the same name as the module 
    indicated by the given kind and name.
    
    Defaults to searching the L{INSTALL_PATH} for the given kind, but can be
    set to search any path. Searching another path can be useful for loading a
    UIE for validation, for instance, in the L{_ensureUIEValid} method.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @param path: Path where the UIE module resides or None to indicate the 
      install path
    @type path: string
    @return: UIE instance
    @rtype: object
    @raise AttributeError: When the module does not contain a class of the same
      name
    @raise ImportError: When the UIE module cannot be imported
    '''
    mod = self._getUIEModule(kind, name, path)
    # now get the class and instantiate it
    cls = getattr(mod, name)
    return cls()
  
  def _getUIEModule(self, kind, name, path=None):
    '''
    Loads the UIE module of the given kind and name. Returns a reference to the
    module for use by L{_getUIEInstance}.

    Defaults to searching the L{INSTALL_PATH} for the given kind, but can be
    set to search any path. Searching another path can be useful for loading a
    UIE for validation, for instance, in the L{_ensureUIEValid} method.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @param path: Path where the UIE module resides or None to indicate the 
      install path
    @type path: string
    @return: Reference to the UIE module
    @rtype: module
    @raise ImportError: When the UIE module cannot be imported
    '''
    if path is None:
      path = os.path.join(INSTALL_PATH, kind)
    # get the metadata for the module to import
    f, pathname, description = imp.find_module(name, [path])
    if os.path.islink(pathname):
      context = os.readlink(pathname)
    else:
      context = pathname
    # add the foldering containing the UIE to the Python search path temporarily
    sys.path.append(os.path.dirname(context))
    try:
      # load the module from disk
      m = imp.load_module(name, f, pathname, description)
    finally:
      # make sure we close the file
      f.close()
      # remove the last added search path
      sys.path.pop()
    # get the module
    return m
  
  def getVersion(self):
    '''
    Gets the repository version from disk.
    
    @return: Version string or None if not found
    @rtype: string
    '''
    try:
      f = file(VERSION_PATH)
      v = f.read()
      f.close()
      return v
    except IOError:
      return None
    
  def setVersion(self, v):
    '''
    Stores the repository version on disk.
    
    @param v: Version string
    @type v: string
    @raise IOError: When the version cannot be stored on disk
    '''
    self._ensurePathsExist()
    f = file(VERSION_PATH, 'w')
    f.write(v)
    f.close()
   
  def removeAll(self):
    '''    
    Wipes the existing repository entries completely from disk. All profiles and
    installed UIE references will be lost as well as any version information.
    
    @raise OSError: When any profile or symlink to a UIE cannot be removed
    '''
    for curr_dir, dns, fns in os.walk(ROOT_PATH):
      for fn in fns:
        os.unlink(os.path.join(ROOT_PATH, curr_dir, fn))

  def install(self, kind, name, real_path):
    '''
    Installs a new UIE module in the repository by creating a symlink to the
    Python module at INSTALL_PATH/kind/name. Checks if a symlink already exists
    at that location and raises ValueError if it does. Checks if the UIE to be
    installed is valid as determined by L{_ensureUIEValid}. Checks if the 
    repository directory structure is initialized properly on disk.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @param real_path: Location of the UIE module on disk
    @type real_path: string
    @raise ValueError: When a UIE with the given kind and name is already 
        installed
    @raise NotImplementedError: When the UIE does not conform to the required
        interface for its kind
    @raise OSError: When the registrar cannot create in the repository directory
    '''
    # make sure the repository is setup on disk
    self._ensurePathsExist()
    # make sure the UIE is not already installed
    self._ensureNotInstalled(kind, name)
    # make sure the UIE is valid
    self._ensureUIEValid(kind, name, real_path)
    # create a symlink to the UIE
    module_name = MODULE_FILE_TEMPLATE % name
    os.symlink(os.path.realpath(real_path), 
               os.path.join(INSTALL_PATH, kind, module_name))
  
  def uninstall(self, kind, name):
    '''
    Removes a UIE module from the repository by removing the symlink to the
    Python module at INSTALL_PATH/kind/name. Checks if a symlink exists at that
    location and raises ValueError if it does not. Removes all references to
    the uninstalled UIE from existing profiles.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @raise ValueError: When a UIE with the given kind and name is not installed
    @raise OSError: When the registrar cannot delete from the repository folder
    '''
    # make sure the item is installed
    self._ensureInstalled(kind, name)
    # remove the symlink entry and all compiled files
    mn = MODULE_FILE_TEMPLATE % name
    for module_name in (mn, mn+'c', mn+'o'):
      path = os.path.join(INSTALL_PATH, kind, module_name)
      if os.path.lexists(path):
        os.unlink(path)
    # remove all references in the available profiles
    for name in self.listProfiles():
      profile = Profile.load(name)
      profile.removeFromAll(kind, name)
      profile.save()
  
  def associate(self, kind, name, profile_name, when, tier_name=None, 
                index=None):
    '''
    Associates an installed UIE of the given kind and name with a profile so 
    that is loaded automatically by LSR at the time specified. Ensures the
    desired UIE is installed and that the when parameter is valid for the given
    UIE kind via the L{_ensureAssociationValid}.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @param profile_name: Name of the profile in which UIE of the given name and
      kind will be loaded automatically
    @type profile_name: string
    @param when: Identifier of when the UIE will be loaded automatically: on
     LSR startup (L{STARTUP}), on creation of any L{Tier} (L{ANY_TIER}), or on
     creation of a particular L{Tier} (L{ONE_TIER})
    @type when: string
    @param tier_name: Name of the L{Tier} which will cause the loading of the
      given UIE. Only used when the previous parameter is set to L{ONE_TIER}
    @type tier_name: string
    @param index: Load order of the UIE when multiple UIEs are to be loaded at
      the given time. Defaults to None meaning after all other associated UIEs
      of the same kind.
    @type index: integer or None
    @raise ValueError: When a UIE with the given kind and name is not installed
    @raise ValueError: When a UIE with the given kind and name is already
      associated with the given profile at the given time
    @raise IOError: When the registrar cannot write to the profiles folder
    '''
    # make sure the item is installed
    self._ensureInstalled(kind, name)
    # make sure the UIE can be loaded at the time given by when
    self._ensureAssociationValid(kind, when)
    # load the profile, will raise ValueError if not found
    profile = Profile.load(profile_name)
    # decide when to load the give UIE
    if when == STARTUP:
      profile.addToStartup(kind, name, index)
    elif when == ONE_TIER:
      profile.addToTier(kind, name, tier_name, index)
    elif when == ANY_TIER:
      profile.addToAnyTier(kind, name, index)
    # write the profile back to disk
    profile.save()
  
  def disassociate(self, kind, name, profile_name, when, tier_name=None):
    '''
    Disassociates an installed UIE of the given kind and name with a profile so 
    that it is no longer loaded automatically by LSR at the time specified.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @param profile_name: Name of the profile in which UIE of the given name and
        kind will no longer be loaded automatically
    @type profile_name: string
    @param when: Identifier of when the UIE will be disassociated: on LSR 
       startup (L{STARTUP}), on creation of any L{Tier} (L{ANY_TIER}), or on
       creation of a particular L{Tier} (L{ONE_TIER})
    @type when: string
    @param tier_name: Name of the L{Tier} which will no longer cause the loading
        of the given UIE. Only used when the previous parameter is set to 
        L{ONE_TIER}
    @type tier_name: string
    @raise ValueError: When a UIE with the given kind and name is not installed
    @raise IOError: When the registrar cannot write to the profiles folder
    '''
    # load the profile, will raise ValueError if not found
    profile = Profile.load(profile_name)
    # decide when not to load the give UIE
    if when == STARTUP:
      profile.removeFromStartup(kind, name)
    elif when == ONE_TIER:
      profile.removeFromTier(kind, name, tier_name)
    elif when == ANY_TIER:
      profile.removeFromAnyTier(kind, name)
    # write the profile back to disk
    profile.save()
  
  def listProfiles(self):
    '''
    Gets a list of all available L{Profile}s.
    
    @return: Names of all profiles
    @rtype: list of string
    '''
    return [name.split(os.extsep)[0] for name in os.listdir(PROFILE_PATH)]
  
  def listInstalled(self, kind):
    '''
    Gets a list of all installed UIEs of the given kind.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @return: Names of all installed UIEs
    @rtype: list of string
    '''
    # get all filenames in the profile path
    fns = os.listdir(os.path.join(INSTALL_PATH, kind))
    s = set()
    for fn in fns:
      name, ext = fn.split(os.extsep)
      # add names to a set to avoid duplicates from py, pyc, and pyo
      s.add(name)
    # convert the set to a list
    return list(s)
  
  def listAssociated(self, kind, profile_name, when, tier_name=None):
    '''
    Gets a list of the names of all UIEs of the given kind associated with the
    given profile to be loaded at the specified time.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param profile_name: Name of the profile
    @type profile_name: string
    @param when: Identifier of the time when the listed UIEs are loaded: on LSR 
       startup (L{STARTUP}), on creation of any L{Tier} (L{ANY_TIER}), or on
       creation of a particular L{Tier} (L{ONE_TIER})
    @type when: string
    @param tier_name: Name of the L{Tier} which causes the loading of the listed
        UIEs. Only used when the previous parameter is set to L{ONE_TIER}
    @type tier_name: string
    @return: Names of all UIEs of the given kind associated with the given 
        profile to be loaded at the given time
    @rtype: list of string
    '''
    # load the profile, will raise ValueError if not found
    profile = Profile.load(profile_name)
    # decide when not to load the give UIE
    if when == STARTUP:
      return profile.listStartup(kind)
    elif when == ONE_TIER:
      return profile.listTier(kind, tier_name)
    elif when == ANY_TIER:
      return profile.listAnyTier(kind)
  
  def loadOne(self, kind, name):
    '''    
    Gets the class representing the installed UIE of the given kind and name.
    Checks if the UIE is installed. Fails to load the UIE if any of the
    dependencies are missing.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param name: Name of the UIE, unique across all UIEs of the same kind
    @type name: string
    @return: Instantiated UIE object with dependencies resolved or None if the
      dependency check failed
    @rtype: object or None
    @raise ValueError: When a UIE with the given kind and name is not installed
    '''
    self._ensureInstalled(kind, name)
    try:
      return self._getUIEInstance(kind, name)
    except Exception, e:
      # on any exception, pass on loading this UIE, but log the error
      log.info('Error %s loading UIE: %s', str(e), name)
      return None
  
  def loadAssociated(self, kind, profile_name, when, tier_name=None):
    '''
    Gets all UIE classes in the given profile to be loaded at the given time.
    
    @param kind: Kind of UIE, one of L{ALL_KINDS}
    @type kind: string
    @param profile_name: Name of the profile
    @type profile_name: string
    @param when: Identifier of the time when the listed UIEs are loaded: on LSR 
       startup (L{STARTUP}), on creation of any L{Tier} (L{ANY_TIER}), or on
       creation of a particular L{Tier} (L{ONE_TIER})
    @type when: string
    @param tier_name: Name of the L{Tier} which causes the loading of the listed
        UIEs. Only used when the previous parameter is set to L{ONE_TIER}
    @type tier_name: string
    @return: All UIE classes to be loaded at this time sorted proper load order
    @rtype: list of class
    @raise ValueError: When any associated UIE with the given kind and name is 
        not installed
    '''
    names = self.listAssociated(kind, profile_name, when, tier_name)
    uies = []
    for name in names:
      obj = self.loadOne(kind, name)
      if obj is not None:
        uies.append(obj)
    return uies
  
  def createProfile(self, profile_name):
    '''
    Creates a new empty L{Profile} on disk. Ensure that the repository directory 
    structure is initialized properly on disk. Ensures the profile does not
    exist on disk already so it is not erased.
    
    @raise ValueError: When the profile exists on disk
    @raise IOError: When the registrar cannot write to the profiles folder
    '''
    self._ensurePathsExist()
    profile = Profile(profile_name)
    if profile.exists():
      raise ValueError(_('A profile with name %s already exists' %
                         profile_name))
    profile.save()
    
  def removeProfile(self, profile_name):
    '''
    Removes a L{Profile} from disk. Ensures the profile exists on disk before
    trying to remove it.
    
    @param profile_name: Name of the profile to remove
    @type profile_name: string
    @raise ValueError: When a profile with the given name does not exist
    '''
    profile = Profile(profile_name)
    if not profile.exists():
      raise ValueError(_('A profile with name %s does not exist' %
                         profile_name))
    profile.delete()
    
  def isProfile(self, profile_name):
    '''
    Checks if a L{Profile} with the given name exists on disk.
    
    @param profile_name: Name of the profile to check
    @type profile_name: string
    @return: Does the profile exist?
    @rtype: boolean
    '''
    profile = Profile(profile_name)
    return profile.exists()