'''
Defines a class responsible for managing all input and output devices and
monitors.

@author: Larry Weiss
@author: Peter Parente
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}

@todo: DBC: add methods to play sound, ie. play(dev, file)
'''
import logging, sys
import AEOutput, AEInput, AEEvent
from LSRInterfaces import implements
from UIRegistrar import UIRegistrar, DEVICE, MONITOR
from i18n import _
import LSRConstants

log = logging.getLogger('Device')

class DeviceManager(object):
  '''
  Creates and manages the devices and monitors use by LSR. Keeps a list of each
  and also defines a "default" for devices. Provides a common interface to all
  output and input devices and mirrors I/O to the monitors. Provides a mapping
  from output styles to semantic concepts in the user interface to allow
  customization of how information is presented.
  
  @ivar dirty_style: Tracks whether a style has been fetched or not for 
    modification. If it has, the L{sendString} should also send the style to
    a device.
  @type dirty_style: boolean
  @ivar last_style: The last L{AEOutput.Style} object sent to a device. Used
    to determine if the next L{sendString} should send a style or not. Not
    sending a style is an optimization.
  @type last_style: L{AEOutput.Style}
  @ivar event_manager: L{EventManager} to which this L{DeviceManager} can post 
    L{AEEvent}s 
  @type event_manager: L{EventManager}
  @ivar out_devs: All output devices registered and initialized
  @type out_devs: list
  @ivar in_devs: All input devices registered and initialized
  @type in_devs: list
  @ivar out_mons: All output monitors registered and initialized
  @type out_mons: list
  @ivar in_mons: All input monitors registered and initialized
  @type in_mons: list
  @ivar def_output: The "default" output device
  @type def_output: L{AEOutput}
  @ivar def_input: The "default" input device
  @type def_input: L{AEInput}
  @ivar acc_eng: The AccessEngine reference that created this class
  @type acc_eng: L{AccessEngine}
  @ivar styles: Dictionary of L{AEOutput.Style}s populated on device 
    registration in L{_registerOutputDevice}. Keys are a 3-tuple of device,
    semantic, and layer.
  @type styles: dictionary of L{AEOutput.Style}
  '''
  def __init__(self, acc_eng):
    '''
    Creates the empty lists for devices and monitors.

    @param acc_eng: The AccessEngine reference that created this class
    @type acc_eng: L{AccessEngine}
    '''
    self.out_devs = []
    self.in_devs = []
    self.out_mons = []
    self.in_mons = []
    self.def_output = None
    self.def_input = None
    self.acc_eng = acc_eng
    self.event_manager = None
    self.styles = {}
    self.dirty_style = True
    self.last_style = None

  def init(self, event_man, **kwargs):
    '''
    Called by L{AccessEngine} at startup.
    
    @todo: PP: how do we unload these devices when they're no longer desired?

    @param event_man: L{EventManager} to which this L{DeviceManager} can
      post L{AEEvent}s 
    @type event_man: L{EventManager}
    @param kwargs: References to managers not of interest here
    @type kwargs: dictionary
    '''
    self.event_manager = event_man
    reg = UIRegistrar()
    # load all device classes to be created at startup
    devs = reg.loadAssociated(DEVICE, self.acc_eng.getProfile())
    # register all devices, making the first input and output the defaults
    for d in devs:
      # call registerDevice which will automatically set defaults
      try:
        self.registerDevice(d)
      except (AEOutput.AEOutputError, AEInput.AEInputError):
        log.info('Could not initialize device: %s', str(d))
      except NotImplementedError:
        log.info('Device does not provide an I/O interface: %s', str(d))
    # load all I/O monitor classes to be created at startup
    mons = reg.loadAssociated(MONITOR, self.acc_eng.getProfile())
    for m in mons:
      # call registerDevice which will weed out non-IO monitors for us
      try:
        self.registerMonitor(m)
      except NotImplementedError:
        # ignore NotImplementedErrors if we get other monitor types in the mix
        pass
  
    # make sure we initialized at least one input and one output device
    if self.def_input is None:
      self.close()
      raise ValueError(_('No input device available'))
    if self.def_output is None:
      self.close()
      raise ValueError(_('No output device available'))
    
  def close(self):
    '''
    Shuts down this manager and all its registered L{AEOutput} and L{AEInput}
    devices.
    '''
    for d in self.out_devs:
      try:
        d.close()
      except:
        # ignore all exceptions
        pass
    for d in self.in_devs:
      try:
        d.close()
      except:
        # ignore all exceptions
        pass
      
  def sendStop(self, dev, sem, layer):
    '''
    Sends the stop command to the referenced output device. When the device has
    not been registered or is None, the stop command is sent to the default
    output device. After sending the stop, L{showStop} is called.

    @param dev: The device to send the stop command or None to send to L{the 
      default output device<setDefaultOutput>}.
    @type dev: L{AEOutput}
    @param sem: Semantic description of the information to stop None to indicate
      stopping all output
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    # when dev is None assume defaultOutput (which may also be None)
    dev = dev or self.def_output
    if sem is None and layer is None:
      self.showStop(dev, sem, layer, None)
      dev.sendStop(None)
    elif sem is None:
      # find all styles associated with the given layer
      all = set([self.styles[(sem, layer, dev)] for sem in
                 LSRConstants.SEMANTIC_STYLES.keys()])
      # send stops to all associated
      for s in all:
        self.showStop(dev, sem, layer, s)
        dev.sendStop(s)
    else:
      s = self.styles[(sem, layer, dev)]
      dev.sendStop(s)
      self.showStop(dev, sem, layer, s)
    # reset the last style flag so that it is resent on the next sendString,
    # otherwise, the stop my prempt the application of the last style
    self.last_style = None

  def showStop(self, dev, sem, layer, style):
    '''
    Calls all registered output monitors to show that a stop command has been
    sent to the referenced output device. This does nothing when no monitors
    are registered.

    @param dev: The device that was sent the stop command
    @type dev: L{AEOutput}
    @param sem: Semantic description of the output that was stopped
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @param style: Style associated with the semantic 
    @type style: L{AEOutput.Style}
    '''
    for mon in self.out_mons:
      mon.showStop(dev, sem, layer, style)
    # for demonstration purposes, just print if no monitors
    #if not self.out_mons:
    #  sys.stdout.write('{stop}')

  def sendTalk(self, dev, sem, layer):
    '''
    Tells the specified output device to send buffered data. When the device
    has not been registered or is None, the talk command is sent to the
    default output device. After sending the talk, L{showTalk} is called.

    @param dev: The device to send the talk command or None to send to L{the 
      default output device<setDefaultOutput>}
    @type dev: L{AEOutput}
    @param sem: Semantic information to start outputting or None to indicate 
      that all buffered information should be output
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    # when dev is None assume defaultOutput (which may also be None)
    dev = dev or self.def_output
    if sem is None:
      s = None
    else:
      s = self.styles[(sem, layer, dev)]
    dev.sendTalk(s)
    # always send the command to the registered monitors
    self.showTalk(dev, sem, layer, s)

  def showTalk(self, dev, sem, layer, style):
    '''
    Calls all registered output monitors to show the talk command sent to the
    referenced output device. This does nothing when no monitors are
    registered.

    @param dev: The device that was sent the talk command.
    @type dev: L{AEOutput}
    @param sem: Semantic information that was output
    @type sem: integer 
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @param style: Style associated with the semantic 
    @type style: L{AEOutput.Style}
    '''
    for mon in self.out_mons:
      mon.showTalk(dev, sem, layer, style)
    # for demonstration purposes, just print if no monitors
    #if not self.out_mons:
    #  print '{talk}'
    
  def sendString(self, dev, text, sem, layer, talk=True):
    '''
    Sends the string to the specified output device. When the device has not
    been registered or is None, the string is sent to the default output
    device. The string is sent with a L{AEOutput.Base.AEOutput.sendString} call
    followed by L{sendTalk} when the talk parameter is True.

    @param dev: The device to send the string or None to send to L{the default 
      output device<setDefaultOutput>}
    @type dev: L{AEOutput}
    @param text: The text to send to the referenced device
    @type text: string
    @param sem: Semantic description of the text to send or None to indicate the
      information is void of any known semantic
    @type sem: integer 
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @param talk: Defaults to True to indicate that L{sendTalk} should be called
      after all characters are sent
    @type talk: boolean
    '''
    # when dev is None assume defaultOutput (which may also be None)
    dev = dev or self.def_output
    s = self.styles[(sem, layer, dev)]
    # respect muting
    if s.Mute: return
    # if the style hasn't changed, and the last style is the same one as the
    # one we're trying to send now, don't resend it
    if not self.dirty_style and self.last_style == s:
      s = None
    else:
      # keep track of the last style sent and reset the dirty flag
      self.last_style = s
      self.dirty_style = False
    dev.sendString(text, s)
    if talk:
       self.sendTalk(dev, sem, layer)
    self.showString(dev, text, sem, layer, s, talk)

  def showString(self, dev, text, sem, layer, style, talk):
    '''
    Sends a string to all registered output monitors. This method should be
    used when only monitor output is desired without sending to any devices.

    This is also called by L{sendString} when no output devices have been
    registered to support testers with no other output.

    @param dev: The device that was sent the string.
    @type dev: L{AEOutput}
    @param text: The text that was sent to the referenced device
    @type text: string
    @param sem: Semantic description of the text sent
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @param style: Style associated with the semantic 
    @type style: L{AEOutput.Style}
    @param talk: Indicates that L{sendTalk} was called
    @type talk: boolean
    '''
    for mon in self.out_mons:
      mon.showString(dev, text, sem, layer, style, talk)
    # for demonstration purposes, just print if no monitors
    #if not self.out_mons:
      #if talk:
        #print text
      #else:
        #for ch in text:
          #sys.stdout.write(ch)

  def sendIndex(self, dev, sem, layer, por):
    '''
    Sends the referenced index marker to the referenced device. When the
    device has not been registered or is None, the index marker is sent to the
    default output device. After sending the command, L{showIndex} is called.

    @param dev: The device to send the index marker or None to send to L{the 
      default output device<setDefaultOutput>}
    @type dev: L{AEOutput}
    @param sem: Semantic description of the index text to send or None to 
      indicate the information is void of any known semantic
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @param por: Point of regard associated with the index marker
    @type por: L{POR}

    @todo: PP: rewrite for POR
    '''
    # when dev is None assume defaultOutput (which may also be None)
    if dev is None:
      dev = self.def_output
    try:
      if dev in self.out_devs:
        dev.sendIndex(marker)
      # always send the command to the registered monitors
      self.showIndex(dev, sem, layer, marker)
    except NotImplementedError:
      # do nothing when indexing not implemented
      pass

  def showIndex(self, dev, sem, layer, por):
    '''
    Calls all registered output monitors to show the index marker sent to the
    referenced output device. This does nothing when no output monitors are
    registered.

    @param dev: The device that was sent the index marker
    @type dev: L{AEOutput}
    @param sem: Semantic description of the index text to send or None to 
      indicate the information is void of any known semantic
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @param por: por at which to place the marker
    @type por: L{POR}

    @todo: PP: rewrite to handle POR
    '''
    for mon in self.out_mons:
      mon.showIndex(dev, sem, layer, por)
    # for demonstration purposes, just print if no monitors
    #if not self.out_mons:
      #sys.stdout.write('{I:%s:%d}' % (dev, marker))
      
  def _isDuplicateDevice(self, dev, output):
    '''
    Checks if the given device is a duplicate in that it provides no new 
    interfaces beyond those provided by the devices already registered.
    
    @param dev: A device class
    @type dev: L{AEInput} or L{AEOutput} class
    @param output: Is the device an output device (True) or input (False)?
    @type output: boolean
    '''
    if output:
      dt = AEOutput.DEVICE_TYPES
      reg_devs = self.out_devs
    else:
      dt = AEInput.DEVICE_TYPES
      reg_devs = self.in_devs
    # get interfaces device implements
    dev_int = (i for i in dt if implements(dev, i))
    # check if some registered device already implements all of these interfaces
    for reg_dev in reg_devs:
      reg_int = (i for i in dt if implements(reg_dev, i))
      if set(reg_int).issuperset(dev_int):
        return True
    return False

  def _registerInputDevice(self, dev):
    '''
    Registers the given device as an input device if the device implements the
    L{AEInput} base class and provides some subinterface not already provided
    by another registered output device. For instance, if a device implementing
    L{AEInput.SystemInput} is already registered, another device implementing 
    just L{AEInput.SystemInput} will not be registered by this method.
    
    @param dev: Device to attempt to register
    @type dev: L{AEInput.AEInput} class
    @return: True if the device was registered, False if it is not an 
      L{AEInput} device, and None if it provides no new interfaces
    @rtype: boolean or None
    '''
    if not implements(dev, AEInput.AEInput):
      # don't register something that isn't an input device
      return False
    if self._isDuplicateDevice(dev, False):
      # don't register a device that provides all the same interfaces as a
      # previous device
      return False
    if dev not in self.out_devs:
      # don't reinitialize a device that was added to out_devs
      dev.init()
    # add DeviceManager as a gesture listener
    dev.addInputListener(self._onGesture)
    # append to the in_devs list after successfully initialized
    self.in_devs.append(dev)
    log.info('Added input device %s', str(dev))
    # first input device registered is default
    if self.def_input is None:
      self.def_input = dev
    return True
  
  def _registerOutputDevice(self, dev, num_styles):
    '''
    Registers the given device as an output device if the device implements the
    L{AEOutput} base class and provides some subinterface not already provided
    by another registered output device. For instance, if a device implementing
    L{AEOutput.Audio} is already registered, another device implementing just
    L{AEOutput.Audio} will not be registered by this method.
    
    @param dev: Device to attempt to register
    @type dev: L{AEOutput.AEOutput} class
    @param num_styles: Number of distinct L{AEOutput.Style}s to requested from
      the device
    @type num_styles: integer 
    @return: True if the device was registered, False if it is not an 
      L{AEOutput} device, and None if it provides no new interfaces
    @rtype: boolean or None
    '''
    if not implements(dev, AEOutput.AEOutput):
      # don't register something that isn't an output device
      return False
    dev = dev.getProxy()  
    if self._isDuplicateDevice(dev, True):
      # don't register a device that provides all the same interfaces as a
      # previous device
      return None
    if dev not in self.in_devs:
      # don't reinitialize a device that was added to in_devs
      dev.init()
    # add DeviceManager as an index listener when implemented
    try:
      dev.addIndexListener(self._onIndex)
    except NotImplementedError:
      # do nothing when indexing not implemented
      pass
    # append to the out_devs list after successfully initialized
    self.out_devs.append(dev)
    log.info('Added output device %s', str(dev))
    # first output device registered is default
    if self.def_output is None:
      self.def_output = dev
   
    # create styles based on the abilities of this device
    styles = dev.createDistinctStyles(num_styles)
    # create semantics, layer to styles mapping for device
    # todo: DBC: later, check to see if previously user-configured
    for layer, i in enumerate(AEEvent.ALL_LAYERS):
      # copy the style so we can change the channel
      s = styles[0].copy()
      s.Channel = i
      # provide a default style for a semantic of None
      self.styles[(None, layer, dev)] = s
      for sem, grp in LSRConstants.SEMANTIC_STYLES.items():
        s = styles[grp % num_styles].copy()
        s.Channel = i
        # map the style for (semantic, layer, device) using default mapping
        self.styles[(sem, layer, dev)] = s
    return True

  def registerDevice(self, dev, num_styles=LSRConstants.DEFAULT_NUM_STYLES):
    '''    
    Registers the referenced device based on its one or more interfaces.
    When the interface is determined, the init() method is called on the device.
    Returns true when the reference is of a known type that initializes
    successfully.
    
    @param dev: The device to register
    @type dev: L{AEOutput} or L{AEInput} class
    @param num_styles: Number of distinct L{AEOutput.Style}s to requested from
      the device
    @type num_styles: integer 
    @raise NotImplementedError: When the device implements none of the required
      interfaces
    @raise AEOutputError: When output device initialization fails
    @raise AEInputError: When input device initialization fails
    '''
    is_out = self._registerOutputDevice(dev, num_styles)
    is_in = self._registerInputDevice(dev)

    if is_out == False and is_in == False:
      # raise an exception if none of the required interfaces was implemented
      raise NotImplementedError
  
  def registerMonitor(self, mon):
    '''
    @todo: PP: implement when we have input and output monitors
    '''
    pass
    #implemented = False
    #registered = False
    ## maybe register as output monitor
    #if isinstance(mon, OutputMonitor):
      #implemented = True
      #if mon not in self.out_mons:
        #if not registered:
          ## no need to multi-init
          #mon.init()
        ## append to the in_devs list after successfully initialized
        #self.out_mons.append(mon)
        #registered = True
      #else:
        #registered = True   # indicate already initialized

    ## maybe register as input monitor
    #if isinstance(mon, InputMonitor):
      #implemented = True
      #if mon not in self.in_mons:
        #if not registered:
          ## no need to multi-init
          #mon.init()
        ## append to the in_devs list after successfully initialized
        #self.in_mons.append(mon)
        #registered = True
      #else:
        #registered = True   # indicate already initialized
    #if not implemented:
      ## raise an exception if none of the required interfaces was implemented
      #raise NotImplementedError
    ## return whether the device/monitor was registered
    #return registered

  def setDefaultOutput(self, dev):
    '''
    Sets the referenced output device as the default output device. When the
    referenced device is not registered, or does not implement the
    L{AEOutput} interface, this does nothing. 

    @param dev: The input device
    @type dev: L{AEOutput}
    @raise NotImplementedError: When the provided device does not implement
        L{AEOutput.AEOutput}
    '''
    if dev in self.out_devs:
      self.def_output = dev
    elif implements(dev, AEOutput.AEOutput):
      if self.registerDevice(dev):
        self.def_output = dev
    else:
      raise NotImplementedError(_('Not an output device'))

  def getDefaultOutput(self):
    '''
    Returns the current default output device or None when no output devices
    have been registered.

    @return: The current default output device
    @rtype: L{AEOutput}
    '''
    return self.def_output
  
  def getOutputByName(self, name):
    '''
    Gets the L{AEOutput} device registered under the given name.
    
    @param name: Name of the output device
    @type name: string
    @return: Output device or None if not registered
    @rtype: L{AEOutput}
    '''
    for dev in self.out_devs:
      if dev.__class__.__name__ == name:
        return dev
    return None

  def setDefaultInput(self, dev):
    '''
    Sets the referenced input device as the default input device. When the
    referenced device is not registered, or does not implement the
    L{AEInput} interface, this does nothing. 

    @param dev: The input device
    @type dev: L{AEInput}
    @raise NotImplementedError: When the provided device does not implement
        L{AEInput.AEInput}
    '''
    if dev in self.in_devs:
      self.def_input = dev
    elif implements(dev, AEInput.AEInput):
      if self.register(dev):
        self.def_input = dev
    else:
      raise NotImplementedError(_('Not an input device'))

  def getDefaultInput(self):
    '''
    Returns the current default input device or None when no input devices
    have been registered.

    @return: The current default input device
    @rtype: L{AEInput}
    '''
    return self.def_input
  
  def getInputByName(self, name):
    '''
    Gets the L{AEInput} device registered under the given name.
    
    @param name: Name of the input device
    @type name: string
    @return: Input device or None if not registered
    @rtype: L{AEInput}
    '''
    for dev in self.in_devs:
      if dev.__class__.__name__ == name:
        return dev
    return None
  
  def _onGesture(self, gesture):
    '''
    Creates an L{AEEvent} indicating the given gesture was found on a registered
    input device. When executed, the L{AEEvent} will notify the L{TierManager}
    about the gesture and allow it to activate the appropriate L{Task}
    registered to respond to the gesture in the active L{Tier}.
    
    @param gesture: Gesture seen on an input device
    @type gesture: L{AEInput.Gesture}
    '''
    event = AEEvent.InputGesture(gesture)
    self.event_manager.postEvents(event)
    
  def _onIndex(self, index):
    '''
    @todo: PP: handle index callback markers
    '''
    pass
  
  def getStyle(self, dev, sem, layer):
    '''
    Gets the style for semantic/layer of a given device.
    
    @param dev: The device from which to get the style.
    @type dev: L{AEOutput}
    @param sem: The device semantic from which to get the style, or None to
      get the device's default style.
    @type sem: integer 
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the L{AEOutput.Style} for the given semantic of a device
    @rtype: L{AEOutput.Style}
    @raise KeyError: When the semantic is invalid
    '''
    self.dirty_style = True
    dev = dev or self.def_output
    if sem is None:
      return dev.getDefaultStyle() 
    else:
      return self.styles[(sem, layer, dev)]

