/* settings-utils.h
 *
 * Copyright (C) 2002 Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gtk/gtktogglebutton.h>
#include <libgnomedb/gnome-db-config.h>
#include <libgnomedb/gnome-db-data-source-selector.h>
#include <libgnomedb/gnome-db-util.h>
#include "settings-utils.h"

static void
widget_destroyed_cb (GtkWidget *w, gpointer user_data)
{
	gnome_db_config_remove_listener (GPOINTER_TO_INT (user_data));
}

static void
check_button_conf_changed (GConfClient *conf_client,
			   guint lid,
			   GConfEntry *entry,
			   gpointer user_data)
{
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (user_data),
				      gconf_value_get_bool (gconf_entry_get_value (entry)));
}

static void
check_button_toggled (GtkToggleButton *button, gpointer user_data)
{
	gnome_db_config_set_boolean ((const gchar *) user_data, gtk_toggle_button_get_active (button));
}

GtkWidget *
settings_utils_new_check_button (const gchar *label, const gchar *key)
{
	GtkWidget *w;
	guint lid;

	w = gnome_db_new_check_button_widget (label);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				      gnome_db_config_get_boolean (key));
	g_signal_connect (G_OBJECT (w), "toggled",
			  G_CALLBACK (check_button_toggled), (gpointer) key);

	lid = gnome_db_config_add_listener (key, (GConfClientNotifyFunc) check_button_conf_changed, w);
	g_signal_connect (G_OBJECT (w), "destroy",
			  G_CALLBACK (widget_destroyed_cb), GINT_TO_POINTER (lid));

	return w;
}

static void
dsn_selector_conf_changed (GConfClient *conf_client,
			   guint lid,
			   GConfEntry *entry,
			   gpointer user_data)
{
	gnome_db_option_menu_set_selection (GTK_OPTION_MENU (user_data),
					    gconf_value_get_string (gconf_entry_get_value (entry)));
}

static void
dsn_selector_changed_cb (GtkOptionMenu *option_menu, gpointer user_data)
{
	gnome_db_config_set_string ((const gchar *) user_data,
				    gnome_db_option_menu_get_selection (option_menu));
}

GtkWidget *
settings_utils_new_dsn_selector (const gchar *key)
{
	GtkWidget *w;
	guint lid;

	w = gnome_db_data_source_selector_new (NULL);
	gtk_widget_show (w);
	gnome_db_option_menu_set_selection (GTK_OPTION_MENU (w),
					    gnome_db_config_get_string (key));
	g_signal_connect (G_OBJECT (w), "changed",
			  G_CALLBACK (dsn_selector_changed_cb), (gpointer) key);

	lid = gnome_db_config_add_listener (key, (GConfClientNotifyFunc) dsn_selector_conf_changed, w);
	g_signal_connect (G_OBJECT (w), "destroy",
			  G_CALLBACK (widget_destroyed_cb), GINT_TO_POINTER (lid));

	return w;
}

static void
int_entry_changed_cb (GtkEditable *editable, gpointer user_data)
{
	gnome_db_config_set_int ((const gchar *) user_data, atoi (gtk_entry_get_text (GTK_ENTRY (editable))));
}

static void
int_entry_conf_changed_cb (GConfClient *conf_client,
			   guint lid,
			   GConfEntry *entry,
			   gpointer user_data)
{
	gchar *value;

	value = g_strdup_printf ("%d", gnome_db_config_get_int (entry->key));
	gtk_entry_set_text (GTK_ENTRY (user_data), value);
	g_free (value);
}

GtkWidget *
settings_utils_new_integer_entry (const gchar *key)
{
	GtkWidget *w;
	guint lid;
	gchar *value;

	w = gnome_db_new_entry_widget (0, TRUE);
	value = g_strdup_printf ("%d", gnome_db_config_get_int (key));
	gtk_entry_set_text (GTK_ENTRY (w), value);
	g_free (value);

	g_signal_connect (G_OBJECT (w), "changed", G_CALLBACK (int_entry_changed_cb), key);
	lid = gnome_db_config_add_listener (key, (GConfClientNotifyFunc) int_entry_conf_changed_cb, w);
	g_signal_connect (G_OBJECT (w), "destroy",
			  G_CALLBACK (widget_destroyed_cb), GINT_TO_POINTER (lid));

	return w;
}
