/* query-druid.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "query-druid.h"
#include <libgnomedb/gnome-db-editor.h>

static void query_druid_class_init (QueryDruidClass * class);
static void query_druid_init (QueryDruid * wid);
static void query_druid_dispose (GObject   * object);


struct _QueryDruidPriv
{
	MgConf      *conf;
	
	/* druid pages */
	GtkWidget   *d_start;
	GtkWidget   *d_query_type;
	GtkWidget   *d_sql;
	GtkWidget   *d_not_yet_impl;
	GtkWidget   *d_end;

	/* extra data for each page */
	GtkWidget   *query_name;
	GtkWidget   *query_descr;
	GtkWidget   *query_type_table;
	GtkWidget   *sql_editor;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

guint
query_druid_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (QueryDruidClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) query_druid_class_init,
			NULL,
			NULL,
			sizeof (QueryDruid),
			0,
			(GInstanceInitFunc) query_druid_init
		};		
		
		type = g_type_register_static (GNOME_TYPE_DRUID, "QueryDruid", &info, 0);
	}

	return type;
}

static void
query_druid_class_init (QueryDruidClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = query_druid_dispose;
}

static void
query_druid_init (QueryDruid * wid)
{
	wid->priv = g_new0 (QueryDruidPriv, 1);
	wid->priv->conf = NULL;
}

static void query_druid_initialize (QueryDruid *mgsel);


static void object_weak_notify (QueryDruid *mgsel, GObject *obj);
/**
 * query_druid_new
 * @conf: a #MgConf object
 *
 * Creates a new #QueryDruid widget.
 *
 * Returns: the new widget
 */
GtkWidget *
query_druid_new (MgConf *conf)
{
	GObject    *obj;
	QueryDruid *qdruid;

	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
		
	obj = g_object_new (QUERY_DRUID_TYPE, NULL);
	qdruid = QUERY_DRUID (obj);

	qdruid->priv->conf = conf;

	g_object_weak_ref (G_OBJECT (qdruid->priv->conf),
			   (GWeakNotify) object_weak_notify, qdruid);
	
	query_druid_initialize (qdruid);

	return GTK_WIDGET (obj);
}

static void
object_weak_notify (QueryDruid *qdruid, GObject *obj)
{
	if (obj == (GObject*) qdruid->priv->conf)
		/* Tell that we don't need to weak unref the MgConf */
		qdruid->priv->conf = NULL;
}

static void
query_druid_dispose (GObject *object)
{
	QueryDruid *qdruid;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_QUERY_DRUID (object));
	qdruid = QUERY_DRUID (object);

	if (qdruid->priv) {
		/* Weak unref the MgConf if necessary */
		if (qdruid->priv->conf)
			g_object_weak_unref (G_OBJECT (qdruid->priv->conf),
					     (GWeakNotify) object_weak_notify, qdruid);

		/* the private area itself */
		g_free (qdruid->priv);
		qdruid->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static gboolean prev_page_cb (GnomeDruidPage *page, GtkWidget *druid, QueryDruid *qdruid);
static gboolean next_page_cb (GnomeDruidPage *page, GtkWidget *druid, QueryDruid *qdruid);
static void finish_page_cb (GnomeDruidPage *page, GtkWidget *druid, QueryDruid *qdruid);
static void query_type_rb_toggled_cb (GtkToggleButton *btn, QueryDruid *qdruid);
static void sql_query_text_changed_cb (GnomeDbEditor *editor, QueryDruid *qdruid);

static void
query_druid_initialize (QueryDruid *qdruid)
{
	GtkWidget *wid, *vbox, *rb, *table;
	GSList *rb_group1 = NULL;
	GSList *rb_group2 = NULL;
	
	/* druid itself */
	gtk_container_set_border_width (GTK_CONTAINER (qdruid), 4);

	/* start page */
	wid = gnome_druid_page_edge_new (GNOME_EDGE_START);
	qdruid->priv->d_start = wid;
	gtk_widget_show (wid);
	g_signal_connect (G_OBJECT (wid), "next", G_CALLBACK (next_page_cb), qdruid);
	gnome_druid_append_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (wid));
	gnome_druid_page_edge_set_title (GNOME_DRUID_PAGE_EDGE (wid), _("New query"));
	gnome_druid_page_edge_set_text (GNOME_DRUID_PAGE_EDGE (wid),
					_("This druid allows you to create a new query.\n\n"
					  "The query can be a data manipulation query of any type "
					  "(SELECT, INSERT, UPDATE or DELETE) and can also be an aggregation "
					  "query (UNION, INTERSECT or EXCEPT) query.\n\n"
					  "The query can be created either with a point and click interface "
					  "or entering the corresponding SQL statement."));

	/* radio buttons page */
	wid = gnome_druid_page_standard_new ();
	qdruid->priv->d_query_type = wid;
	gtk_widget_show_all (wid);
	g_signal_connect (G_OBJECT (wid), "back", G_CALLBACK (prev_page_cb), qdruid);
	g_signal_connect (G_OBJECT (wid), "next", G_CALLBACK (next_page_cb), qdruid);
	gnome_druid_append_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (wid));
	gnome_druid_page_standard_set_title (GNOME_DRUID_PAGE_STANDARD (wid), _("Query attributes"));
	vbox = GNOME_DRUID_PAGE_STANDARD (wid)->vbox;
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 16);

	wid = gtk_label_new (_("<b>Name and description:</b>"));
	gtk_widget_show (wid);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_label_set_justify (GTK_LABEL (wid), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (wid), 0, 0.5);

	table = gtk_table_new (2, 2, FALSE);
	gtk_widget_show (table);
	gtk_box_pack_start (GTK_BOX (vbox), table, TRUE, TRUE, 0);
	
	wid = gtk_label_new (_("Name:"));
	gtk_widget_show (wid);
	gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 0, 1, GTK_FILL, 0, 0, 0);
	gtk_label_set_justify (GTK_LABEL (wid), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (wid), 0, 0.5);
	
	wid = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY (wid), _("Untitled"));
	gtk_widget_show (wid);
	gtk_table_attach (GTK_TABLE (table), wid, 1, 2, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
	qdruid->priv->query_name = wid;

	
	wid = gtk_label_new (_("Description:"));
	gtk_widget_show (wid);
	gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 1, 2, GTK_FILL, 0, 0, 0);
	gtk_label_set_justify (GTK_LABEL (wid), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (wid), 0, 0.5);
	
	wid = gtk_entry_new ();
	gtk_widget_show (wid);
	gtk_table_attach (GTK_TABLE (table), wid, 1, 2, 1, 2, GTK_EXPAND | GTK_FILL, 0, 0, 0);
	qdruid->priv->query_descr = wid;

	wid = gtk_label_new (_("<b>Type of query:</b>"));
	gtk_widget_show (wid);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_label_set_justify (GTK_LABEL (wid), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (wid), 0, 0.5);

	rb = gtk_radio_button_new_with_mnemonic (NULL, _("Point & click wizard"));
	gtk_widget_show (rb);
	gtk_box_pack_start (GTK_BOX (vbox), rb, FALSE, FALSE, 0);
	rb_group1 = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
	g_object_set_data (G_OBJECT (rb), "action", GINT_TO_POINTER (-1));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (query_type_rb_toggled_cb), qdruid);
	
	wid = gtk_table_new (5, 2, FALSE);
	gtk_widget_show (wid);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, FALSE, 0);
	gtk_table_set_col_spacings (GTK_TABLE (wid), 20);
	qdruid->priv->query_type_table = wid;
	gtk_widget_set_sensitive (qdruid->priv->query_type_table, FALSE);

	rb = gtk_radio_button_new_with_mnemonic (NULL, _("SELECT query"));
	gtk_widget_show (rb);
	gtk_table_attach (GTK_TABLE (wid), rb, 1, 2, 0, 1, GTK_FILL, 0, 0, 0);
	rb_group2 = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
	g_object_set_data (G_OBJECT (rb), "action", GINT_TO_POINTER (MG_QUERY_TYPE_SELECT));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (query_type_rb_toggled_cb), qdruid);
	g_object_set_data (G_OBJECT (qdruid->priv->d_query_type), "query_type_1", GINT_TO_POINTER (MG_QUERY_TYPE_SELECT));

	rb = gtk_radio_button_new_with_mnemonic (NULL, _("INSERT query"));
	gtk_widget_show (rb);
	gtk_table_attach (GTK_TABLE (wid), rb, 1, 2, 1, 2, GTK_FILL, 0, 0, 0);
	gtk_radio_button_set_group (GTK_RADIO_BUTTON (rb), rb_group2);
	rb_group2 = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
	g_object_set_data (G_OBJECT (rb), "action", GINT_TO_POINTER (MG_QUERY_TYPE_INSERT));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (query_type_rb_toggled_cb), qdruid);

	rb = gtk_radio_button_new_with_mnemonic (NULL, _("UPDATE query"));
	gtk_widget_show (rb);
	gtk_table_attach (GTK_TABLE (wid), rb, 1, 2, 2, 3, GTK_FILL, 0, 0, 0);
	gtk_radio_button_set_group (GTK_RADIO_BUTTON (rb), rb_group2);
	rb_group2 = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
	g_object_set_data (G_OBJECT (rb), "action", GINT_TO_POINTER (MG_QUERY_TYPE_UPDATE));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (query_type_rb_toggled_cb), qdruid);

	rb = gtk_radio_button_new_with_mnemonic (NULL, _("DELETE query"));
	gtk_widget_show (rb);
	gtk_table_attach (GTK_TABLE (wid), rb, 1, 2, 3, 4, GTK_FILL, 0, 0, 0);
	gtk_radio_button_set_group (GTK_RADIO_BUTTON (rb), rb_group2);
	rb_group2 = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
	g_object_set_data (G_OBJECT (rb), "action", GINT_TO_POINTER (MG_QUERY_TYPE_DELETE));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (query_type_rb_toggled_cb), qdruid);

	rb = gtk_radio_button_new_with_mnemonic (NULL, _("Aggreation query (union, intersect or except)"));
	gtk_widget_show (rb);
	gtk_table_attach (GTK_TABLE (wid), rb, 1, 2, 4, 5, GTK_FILL, 0, 0, 0);
	gtk_radio_button_set_group (GTK_RADIO_BUTTON (rb), rb_group2);
	rb_group2 = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
	g_object_set_data (G_OBJECT (rb), "action", GINT_TO_POINTER (MG_QUERY_TYPE_UNION));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (query_type_rb_toggled_cb), qdruid);

	rb = gtk_radio_button_new_with_mnemonic (NULL, _("SQL query (of any type)"));
	gtk_widget_show (rb);
	gtk_box_pack_start (GTK_BOX (vbox), rb, FALSE, FALSE, 0);
	gtk_radio_button_set_group (GTK_RADIO_BUTTON (rb), rb_group1);
	rb_group1 = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
	g_object_set_data (G_OBJECT (rb), "action", GINT_TO_POINTER (-2));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (query_type_rb_toggled_cb), qdruid);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rb), TRUE);
	g_object_set_data (G_OBJECT (qdruid->priv->d_query_type), "query_type", GINT_TO_POINTER (-2));

	/* Not yet implemented page */
	wid = gnome_druid_page_standard_new ();
	gtk_widget_show_all (wid);
	qdruid->priv->d_not_yet_impl = wid;
	g_signal_connect (G_OBJECT (wid), "next", G_CALLBACK (next_page_cb), qdruid);
	gnome_druid_append_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (wid));
	gnome_druid_page_standard_set_title (GNOME_DRUID_PAGE_STANDARD (wid), _("To be implemented"));
	vbox = GNOME_DRUID_PAGE_STANDARD (wid)->vbox;
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 16);

	wid = gtk_label_new (_("<b>This functionnality is not yet implemented</b>"));
	gtk_widget_show (wid);
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, FALSE, 0);

	/* SQL query page */
	wid = gnome_druid_page_standard_new ();
	qdruid->priv->d_sql = wid;
	gtk_widget_show_all (wid);
	g_signal_connect (G_OBJECT (wid), "back", G_CALLBACK (prev_page_cb), qdruid);
	g_signal_connect (G_OBJECT (wid), "next", G_CALLBACK (next_page_cb), qdruid);
	g_signal_connect (G_OBJECT (wid), "finish", G_CALLBACK (finish_page_cb), qdruid);
	gnome_druid_append_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (wid));
	gnome_druid_page_standard_set_title (GNOME_DRUID_PAGE_STANDARD (wid), _("SQL statement Query"));
	vbox = GNOME_DRUID_PAGE_STANDARD (wid)->vbox;
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 16);

	wid = gtk_label_new (_("<b>Enter the SQL statement representing the query:</b>"));
	gtk_widget_show (wid);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_label_set_justify (GTK_LABEL (wid), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (wid), 0, 0.5);

	wid = gnome_db_editor_new ();
	g_signal_connect (G_OBJECT (wid), "text_changed", G_CALLBACK (sql_query_text_changed_cb), qdruid);
	qdruid->priv->sql_editor = wid;
	gtk_widget_show (wid);
        gnome_db_editor_set_editable (GNOME_DB_EDITOR (wid), TRUE);
        gnome_db_editor_set_highlight (GNOME_DB_EDITOR (wid), TRUE);
        gtk_box_pack_start (GTK_BOX (vbox), wid, TRUE, TRUE, 0);
}

/* 
 * General callback for all the pages when the BACK button is pressed
 * Returns: TRUE if a page change has occured.
 */
static gboolean 
prev_page_cb (GnomeDruidPage *page, GtkWidget *druid, QueryDruid *qdruid)
{
	gboolean done = FALSE;
	gboolean retval = FALSE;


	/* page to select the query type */
	if (!done && (page == (GnomeDruidPage *) qdruid->priv->d_query_type)) {
		gnome_druid_set_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (qdruid->priv->d_start));
		gnome_druid_set_show_finish (GNOME_DRUID (qdruid), FALSE);
		retval = TRUE;
		done = TRUE;
	}

	/* page for SQL queries */
	if (!done && (page == (GnomeDruidPage *) qdruid->priv->d_sql)) {
		done = TRUE;
		gnome_druid_set_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (qdruid->priv->d_query_type));
		gnome_druid_set_show_finish (GNOME_DRUID (qdruid), FALSE);
		retval = TRUE;
	}	
	
	if (!done)
		g_warning ("Unhandled 'back' druid page signal");

	return retval;
}


/* 
 * General callback for all the pages when the NEXT button is pressed
 * Returns: TRUE if a page change has occured.
 */
static gboolean 
next_page_cb (GnomeDruidPage *page, GtkWidget *druid, QueryDruid *qdruid)
{
	gboolean done = FALSE;
	gboolean retval = FALSE;

	/* Start page */
	if (page == (GnomeDruidPage *) qdruid->priv->d_start) {
		/* nothing to be done here */
		done = TRUE;
	}

	/* page to select the query type */
	if (!done && (page == (GnomeDruidPage *) qdruid->priv->d_query_type)) {
		gint query_type = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (page), "query_type"));
	
		switch (query_type) {
		case -2:
			gnome_druid_set_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (qdruid->priv->d_sql));
			gnome_druid_set_show_finish (GNOME_DRUID (qdruid), TRUE);
			break;
		default:
			gnome_druid_set_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (qdruid->priv->d_not_yet_impl));
			gnome_druid_set_show_finish (GNOME_DRUID (qdruid), TRUE);
			gnome_druid_set_buttons_sensitive (GNOME_DRUID (qdruid), FALSE, TRUE, TRUE, FALSE);
			break;
		}
		retval = TRUE;
		done = TRUE;
	}

	/* page for SQL queries */
	if (!done && (page == (GnomeDruidPage *) qdruid->priv->d_sql)) {
		done = TRUE;
		g_assert_not_reached ();
	}	

	if (!done)
		g_warning ("Unhandled 'next' druid page signal");

	return retval;
}

static void
finish_page_cb (GnomeDruidPage *page, GtkWidget *druid, QueryDruid *qdruid)
{
	gboolean done = FALSE;

	/* page for SQL queries */
	if (!done && (page == (GnomeDruidPage *) qdruid->priv->d_sql)) {
		MgQuery *query;
		gchar *sql;
		gchar *str;

		sql = gnome_db_editor_get_all_text (GNOME_DB_EDITOR (qdruid->priv->sql_editor));
		query = MG_QUERY (mg_query_new_from_sql (qdruid->priv->conf, sql, NULL));
		str = gtk_entry_get_text (GTK_ENTRY (qdruid->priv->query_name));
		if (str && *str)
			mg_base_set_name (MG_BASE (query), str);
		str = gtk_entry_get_text (GTK_ENTRY (qdruid->priv->query_descr));
		if (str && *str)
			mg_base_set_description (MG_BASE (query), str);

		mg_conf_assume_query (qdruid->priv->conf, query);
		g_object_unref (G_OBJECT (query));
		/* saving of MgConf */
		mg_conf_save_xml (qdruid->priv->conf, NULL);
		gtk_widget_destroy (GTK_WIDGET (qdruid));
		done = TRUE;
	}

	if (!done)
		g_warning ("Unhandled 'finish' druid page signal");
}


static void
query_type_rb_toggled_cb (GtkToggleButton *btn, QueryDruid *qdruid)
{
	gint action = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (btn), "action"));

	switch (action) {
	case -1:
		gtk_widget_set_sensitive (qdruid->priv->query_type_table, TRUE);
		g_object_set_data (G_OBJECT (qdruid->priv->d_query_type), "query_type", 
				   g_object_get_data (G_OBJECT (qdruid->priv->d_query_type), "query_type_1"));
		break;
	case -2:
		gtk_widget_set_sensitive (qdruid->priv->query_type_table, FALSE);
		g_object_set_data (G_OBJECT (qdruid->priv->d_query_type), "query_type", GINT_TO_POINTER (action));
		break;
	default:
		g_object_set_data (G_OBJECT (qdruid->priv->d_query_type), "query_type_1", GINT_TO_POINTER (action));
		g_object_set_data (G_OBJECT (qdruid->priv->d_query_type), "query_type", GINT_TO_POINTER (action));
		break;
	}
}

static void
sql_query_text_changed_cb (GnomeDbEditor *editor, QueryDruid *qdruid)
{
	MgQuery *query;
	gchar *sql;
	GError *error = NULL;
	gchar *str = NULL;	

	sql = gnome_db_editor_get_all_text (GNOME_DB_EDITOR (qdruid->priv->sql_editor));
	query = MG_QUERY (mg_query_new_from_sql (qdruid->priv->conf, sql, &error));
	if (mg_query_get_query_type (query) ==  MG_QUERY_TYPE_NON_PARSED_SQL) {
		if (error) {
			str = g_strdup (error->message);
			g_error_free (error);
		}
		else
			str = g_strdup (_("Error (no details available)."));
	}

	g_object_unref (G_OBJECT (query));
}
