/* Mergeant
 *
 * Copyright (C) 1999 - 2003 Vivien Malerba
 * Copyright (C) 2002 - 2003 Rodrigo Moya
 *
 * Authors:
 *       Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <bonobo/bonobo-i18n.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktable.h>
#include <gtk/gtkvpaned.h>
#include <libgnomedb/gnome-db-gray-bar.h>
#include <libgnomedb/gnome-db-util.h>
#include <libgnomedb/gnome-db-editor.h>
#include <libmergeant/libmergeant.h>
#include "workspace.h"
#include <string.h>
#include "query-druid.h"
#include "query-editor.h"

enum {
	WORKSPACE_TABLES  = 0,
	WORKSPACE_DTYPES  = 1,
	WORKSPACE_QUERIES = 2,
	MAX_WORKSPACES
};

struct _WorkspacePrivate {
	MgServer  *server;

	/* main page */
	GtkWidget *main_page_pane;
	GtkWidget *main_page_object_type;
	GtkWidget *main_page_object_list;
	GtkWidget *main_page_detail_table;
	GtkWidget *main_page_detail_label;
	GtkWidget *main_page_detail_contents;
	guint      main_page_detail_contents_object_type;
	GtkWidget *main_page_details[MAX_WORKSPACES];
	void      *main_page_details_structs[MAX_WORKSPACES];
	GObject   *main_page_selected_object[MAX_WORKSPACES]; /* currently selected objects */

	GHashTable *detailed_viewers;
};

static void workspace_class_init (WorkspaceClass *klass);
static void workspace_init (Workspace *wk, WorkspaceClass *klass);
static void workspace_finalize (GObject *object);

static void create_widgets (Workspace *wk);
static void finish_widgets (Workspace *wk);


static void not_yet_implemented (Workspace *wk, const gchar *feature, 
				 const gchar *file, const gchar *function, gint lineno);
#define NOT_YET_IMPLEMENTED(wk,feature) not_yet_implemented(wk, feature, __FILE__, __FUNCTION__, __LINE__)

static GObjectClass *parent_class = NULL;

static void
workspace_class_init (WorkspaceClass *klass)
{
	GObjectClass *object_class;

	parent_class = g_type_class_peek_parent (klass);

	object_class = G_OBJECT_CLASS (klass);
	object_class->finalize = workspace_finalize;
}

static void
workspace_init (Workspace *wk, WorkspaceClass *klass)
{
	gint page;

	/* allocate private structure */
	wk->priv = g_new0 (WorkspacePrivate, 1);

	/* load the available detailed viewers */
	wk->priv->detailed_viewers = g_hash_table_new (g_int_hash, g_int_equal);

	/* set options on the notebook */
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (wk), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK (wk), FALSE);
	gtk_notebook_popup_disable (GTK_NOTEBOOK (wk));

	/* safe init values */
	for (page = 0; page < MAX_WORKSPACES; page++) {
		wk->priv->main_page_details [page] = NULL;
		wk->priv->main_page_details_structs [page] = NULL;
		wk->priv->main_page_selected_object [page] = NULL;
	}
}

static void
free_hash_viewer (gpointer key, gpointer value, gpointer user_data)
{
	g_object_unref (value);
}

static void
workspace_finalize (GObject *object)
{
	Workspace *wk = (Workspace *) object;

	/* free memory */
	if (wk->priv) {
		gint page;

		finish_widgets (wk);

		for (page = 0; page < MAX_WORKSPACES; page++) {
			if (wk->priv->main_page_details_structs [page])
				g_free (wk->priv->main_page_details_structs [page]);
		}

		if (wk->priv->detailed_viewers) {
			g_hash_table_foreach (wk->priv->detailed_viewers, (GHFunc) free_hash_viewer, NULL);
			g_hash_table_destroy (wk->priv->detailed_viewers);
			wk->priv->detailed_viewers = NULL;
		}

		wk->priv->server = NULL;
		g_free (wk->priv);
		wk->priv = NULL;
	}

	if (parent_class->finalize)
		parent_class->finalize (object);
}

GType
workspace_get_type (void)
{
        static GType type = 0;
                                                                                    
        if (!type) {
                static const GTypeInfo info = {
                        sizeof (WorkspaceClass),
                        (GBaseInitFunc) NULL,
                        (GBaseFinalizeFunc) NULL,
                        (GClassInitFunc) workspace_class_init,
                        NULL,
                        NULL,
                        sizeof (Workspace),
                        0,
                        (GInstanceInitFunc) workspace_init
                };
                type = g_type_register_static (GTK_TYPE_NOTEBOOK, "Workspace", &info, 0);
        }
        return type;
}

GtkWidget *
workspace_new (MgServer *server)
{
	Workspace *wk;

	wk = g_object_new (workspace_get_type (), NULL);
	workspace_set_server (wk, server);
	create_widgets (wk);

	return GTK_WIDGET (wk);
}

MgServer *
workspace_get_server (Workspace *wk)
{
	return wk->priv->server;
}

void
workspace_set_server (Workspace *wk, MgServer *server)
{
	wk->priv->server = server;
}


static void mitem_activate_cb (GtkMenuItem *mitem, Workspace *wk);
static GtkWidget *create_tables_workspace (Workspace *wk);
static GtkWidget *create_dtypes_workspace (Workspace *wk);
static GtkWidget *create_queries_workspace (Workspace *wk);


/*
 * Creates the widgets composing the different 'pages', and sets
 * up any required signal callback
 */
static void
create_widgets (Workspace *wk)
{
	GtkWidget *table, *mitem, *wid;
	gchar *title;

	/* add main page */
	wk->priv->main_page_pane = gnome_db_new_hpaned_widget ();

	gtk_notebook_append_page (GTK_NOTEBOOK (wk), wk->priv->main_page_pane, gtk_label_new (_("Database")));

	/* left part of main page */
	table = gnome_db_new_table_widget (2, 3, FALSE);
	wk->priv->main_page_object_type = gnome_db_new_option_menu_widget ();
	
	mitem = gnome_db_option_menu_add_item (GTK_OPTION_MENU (wk->priv->main_page_object_type), _("Tables & views"));
	g_signal_connect (G_OBJECT (mitem), "activate", G_CALLBACK (mitem_activate_cb), wk);
	g_object_set_data (G_OBJECT (mitem), "pageno", GINT_TO_POINTER (WORKSPACE_TABLES));
	gnome_db_option_menu_set_selection (GTK_OPTION_MENU (wk->priv->main_page_object_type), _("Tables & views"));

	mitem = gnome_db_option_menu_add_item (GTK_OPTION_MENU (wk->priv->main_page_object_type), _("Data types"));
	g_signal_connect (G_OBJECT (mitem), "activate", G_CALLBACK (mitem_activate_cb), wk);
	g_object_set_data (G_OBJECT (mitem), "pageno", GINT_TO_POINTER (WORKSPACE_DTYPES));

	mitem = gnome_db_option_menu_add_item (GTK_OPTION_MENU (wk->priv->main_page_object_type), _("Queries"));
	g_signal_connect (G_OBJECT (mitem), "activate", G_CALLBACK (mitem_activate_cb), wk);
	g_object_set_data (G_OBJECT (mitem), "pageno", GINT_TO_POINTER (WORKSPACE_QUERIES));

	gtk_table_attach (GTK_TABLE (table), wk->priv->main_page_object_type, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);

	wk->priv->main_page_object_list = mg_selector_new (mg_server_get_conf (wk->priv->server), NULL,
							   MG_SELECTOR_TABLES, MG_SELECTOR_COLUMN_TYPE);

	gtk_widget_show (wk->priv->main_page_object_list);
	gtk_table_attach (GTK_TABLE (table), wk->priv->main_page_object_list, 0, 3, 1, 2,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 0);

	gtk_paned_pack1 (GTK_PANED (wk->priv->main_page_pane), table, TRUE, FALSE);
	

	/* right part of  main page */
	wk->priv->main_page_detail_table = gnome_db_new_table_widget (2, 1, FALSE);

	title = g_strdup_printf ("<b>%s</b>", _("No table selected"));
	wk->priv->main_page_detail_label = gnome_db_gray_bar_new (title);
	g_free (title);
	gnome_db_gray_bar_set_show_icon (GNOME_DB_GRAY_BAR (wk->priv->main_page_detail_label), TRUE);
	gnome_db_gray_bar_set_icon_from_stock (GNOME_DB_GRAY_BAR (wk->priv->main_page_detail_label),
					       GTK_STOCK_NEW, GTK_ICON_SIZE_BUTTON);
	gnome_db_gray_bar_set_show_icon (GNOME_DB_GRAY_BAR (wk->priv->main_page_detail_label), FALSE);
	gtk_widget_show (wk->priv->main_page_detail_label);
	gtk_table_attach (GTK_TABLE (wk->priv->main_page_detail_table),
			  wk->priv->main_page_detail_label, 0, 1, 0, 1,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK, GTK_FILL, 0, 0);

	wk->priv->main_page_detail_contents = gtk_notebook_new ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (wk->priv->main_page_detail_contents), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (wk->priv->main_page_detail_contents), FALSE);
	gtk_widget_show (wk->priv->main_page_detail_contents);
	gtk_table_attach_defaults (GTK_TABLE (wk->priv->main_page_detail_table),
				   wk->priv->main_page_detail_contents, 0 ,1, 1, 2);

	gtk_paned_pack2 (GTK_PANED (wk->priv->main_page_pane), wk->priv->main_page_detail_table, TRUE, FALSE);

	/* tables page: page num. WORKSPACE_TABLES */
	wid = create_tables_workspace (wk);
	wk->priv->main_page_details[WORKSPACE_TABLES] = wid;
	gtk_widget_show (wid);
	gtk_notebook_append_page (GTK_NOTEBOOK (wk->priv->main_page_detail_contents), wid, NULL);
	wk->priv->main_page_detail_contents_object_type = MG_DB_TABLE_TYPE;

	/* tables page: page num. WORKSPACE_DTYPES */
	wid = create_dtypes_workspace (wk);
	wk->priv->main_page_details[WORKSPACE_DTYPES] = wid;
	gtk_widget_show (wid);
	gtk_notebook_append_page (GTK_NOTEBOOK (wk->priv->main_page_detail_contents), wid, NULL);

	/* queries pages: page num. WORKSPACE_QUERIES */
	wid = create_queries_workspace (wk);
	wk->priv->main_page_details[WORKSPACE_QUERIES] = wid;
	gtk_widget_show (wid);
	gtk_notebook_append_page (GTK_NOTEBOOK (wk->priv->main_page_detail_contents), wid, NULL);

}

static void       finish_tables_workspace (Workspace *wk);
static void       finish_dtypes_workspace (Workspace *wk);
static void       finish_queries_workspace (Workspace *wk);

/*
 * Don't destriy the widgets composing the different 'pages', but
 * disconnect any signal callback connected when create_widgets() was called.
 */
static void
finish_widgets (Workspace *wk)
{
	finish_tables_workspace (wk);
	finish_dtypes_workspace (wk);
	finish_queries_workspace (wk);
}




/*
 * Called when the type of object to display on the left part changes (from tables to queries, etc).
 */
static gchar *main_page_get_contents_title (Workspace *wk, GObject *sel_object, guint object_type);

static void
mitem_activate_cb (GtkMenuItem *mitem, Workspace *wk)
{
	const gchar *str;
	gboolean treated = FALSE;
	gchar *title;
	guint object_type = 0;

	/* set the correct contents in the MgSelector widget */
	str = g_object_get_data (G_OBJECT (mitem), "GNOME:Database:MenuItemLabel");
	if (!strcmp (str, _("Tables & views"))) {
		treated = TRUE;
		wk->priv->main_page_detail_contents_object_type = MG_DB_TABLE_TYPE;
		mg_selector_set_mode_columns (MG_SELECTOR (wk->priv->main_page_object_list), NULL,
					      MG_SELECTOR_TABLES, MG_SELECTOR_COLUMN_TYPE);
		object_type = MG_DB_TABLE_TYPE;
	}

	if (!treated && !strcmp (str, _("Data types"))) {
		treated = TRUE;
		wk->priv->main_page_detail_contents_object_type = MG_SERVER_DATA_TYPE_TYPE;
		mg_selector_set_mode_columns (MG_SELECTOR (wk->priv->main_page_object_list), NULL,
					      MG_SELECTOR_DATA_TYPES, 0);
		object_type = MG_SERVER_DATA_TYPE_TYPE;
	}

	if (!treated && !strcmp (str, _("Queries"))) {
		treated = TRUE;
		wk->priv->main_page_detail_contents_object_type = MG_QUERY_TYPE;
		mg_selector_set_mode_columns (MG_SELECTOR (wk->priv->main_page_object_list), NULL,
					      MG_SELECTOR_QUERIES, MG_SELECTOR_COLUMN_TYPE);
		object_type = MG_QUERY_TYPE;
	}
	g_assert (treated);

	/* set the contents title */
	title = main_page_get_contents_title (wk, mg_selector_get_selected_object 
					      (MG_SELECTOR (wk->priv->main_page_object_list)), object_type);
	if (title) {
		gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (wk->priv->main_page_detail_label), title);
		g_free (title);
	}

	/* display the correct notebook page */
	gtk_notebook_set_current_page (GTK_NOTEBOOK (wk->priv->main_page_detail_contents),
				       GPOINTER_TO_INT (g_object_get_data (G_OBJECT (mitem), "pageno")));
}

static gchar *
main_page_get_contents_title (Workspace *wk, GObject *sel_object, guint object_type)
{
	gchar *title = NULL;
	gboolean treated = FALSE;
	if (object_type == 0)
		object_type = G_OBJECT_TYPE (sel_object);

	if (object_type == MG_DB_TABLE_TYPE) {
		treated = TRUE;
		if (sel_object)
			title = g_strdup_printf ("Table: <b>%s</b>", mg_base_get_name (MG_BASE (sel_object)));
		else
			title = g_strdup_printf ("<b>%s</b>", _("No table selected"));
	}

	if (!treated && (object_type == MG_SERVER_DATA_TYPE_TYPE)) {
		treated = TRUE;
		if (sel_object)
			title = g_strdup_printf ("Data type: <b>%s</b>", mg_base_get_name (MG_BASE (sel_object)));
		else
			title = g_strdup_printf ("<b>%s</b>", _("No data type selected"));
	}

	if (!treated && (object_type == MG_QUERY_TYPE)) {
		treated = TRUE;
		if (sel_object)
			title = g_strdup_printf ("Query: <b>%s</b>", mg_base_get_name (MG_BASE (sel_object)));
		else
			title = g_strdup_printf ("<b>%s</b>", _("No query selected"));
	}

	return title;
}

static void
not_yet_implemented (Workspace *wk, const gchar *feature, const gchar *file, const gchar *function, gint lineno)
{
	GtkWidget *dlg;

	if (feature && *feature)
		dlg = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_INFO,
					      GTK_BUTTONS_CLOSE, 
					      _("This feature (%s)\nis not yet implemented\n\nIn %s, %s(), line %d"), 
					      feature, file, function, lineno);
	else
		dlg = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_INFO,
					      GTK_BUTTONS_CLOSE, 
					      _("This feature is not yet implemented\n\nIn %s, %s(), line %d"),
					      file, function, lineno);

	gtk_dialog_run (GTK_DIALOG (dlg));
	gtk_widget_destroy (dlg);
}





/*
 * Contents for tables
 */
typedef struct {
	GtkWidget     *notebook;
	GtkWidget     *description;
	GtkWidget     *fields;
	GtkWidget     *table_add;
	GtkWidget     *table_del;
	GtkWidget     *table_contents;
	GtkTextBuffer *integ_descr;
} ContentsTables;
#define CONTENTS_TABLES(x) ((ContentsTables*)(x))

static void tables_selector_selection_changed_cb (MgSelector *mgsel, GObject *sel_object, Workspace *wk);
static void table_add_clicked_cb (GtkButton *button, Workspace *wk);
static void table_del_clicked_cb (GtkButton *button, Workspace *wk);
static void table_contents_clicked_cb (GtkButton *button, Workspace *wk);
static void table_updated_cb (MgDatabase *db, MgDbTable *table, Workspace *wk);
static GtkWidget *
create_tables_workspace (Workspace *wk)
{
	GtkWidget *label, *vbox, *wid, *nb, *vp, *vbox2, *bbox, *button, *hbox, *sw;
	ContentsTables *ct;

	vbox = gtk_vbox_new (FALSE, 0);
	ct = g_new0 (ContentsTables, 1);
	wk->priv->main_page_details_structs[WORKSPACE_TABLES] = ct;
	
	nb = gtk_notebook_new ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (nb), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nb), FALSE);
	gtk_box_pack_start (GTK_BOX (vbox), nb, TRUE, TRUE, 0);
	gtk_widget_show (nb);
	ct->notebook = nb;
	
	label = gtk_label_new (_("Please select a table from the list on the left,\n"
				 "or create a new one using the 'Add' button below."));
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), label, NULL);
	gtk_widget_show (label);
	
	vp = gtk_vpaned_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), vp, NULL);
	gtk_widget_show (vp);

	vbox2 = gtk_vbox_new (FALSE, 5);
	gtk_paned_add1 (GTK_PANED (vp), vbox2);
	gtk_widget_show (vbox2);

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Description:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	label = gtk_label_new (NULL);
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_widget_show (label);
	ct->description = label;
	
	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Fields:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	wid = mg_selector_new (mg_server_get_conf (wk->priv->server), NULL,
			       MG_SELECTOR_FIELDS, 
			       MG_SELECTOR_COLUMN_TYPE | MG_SELECTOR_COLUMN_FIELD_LENGTH | MG_SELECTOR_COLUMN_FIELD_NNUL |
			       MG_SELECTOR_COLUMN_FIELD_DEFAULT | MG_SELECTOR_COLUMN_COMMENTS);
	gtk_box_pack_start (GTK_BOX (hbox), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	ct->fields = wid;

	vbox2 = gtk_vbox_new (FALSE, 5);
	gtk_paned_add2 (GTK_PANED (vp), vbox2);
	gtk_widget_show (vbox2);

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Integrity rules:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), sw, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_widget_show (sw);

	wid = gtk_text_view_new ();
	gtk_container_add (GTK_CONTAINER (sw), wid);
	gtk_text_view_set_left_margin (GTK_TEXT_VIEW (wid), 5);
	gtk_text_view_set_right_margin (GTK_TEXT_VIEW (wid), 5);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (wid), FALSE);
	ct->integ_descr = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wid));
	gtk_text_buffer_set_text (ct->integ_descr, "", -1);
	gtk_widget_show (wid);

	gtk_text_buffer_create_tag (ct->integ_descr, "header",
				    "weight", PANGO_WEIGHT_BOLD,
				    "foreground", "red", NULL);

	gtk_text_buffer_create_tag (ct->integ_descr, "section",
				    "weight", PANGO_WEIGHT_BOLD,
				    "foreground", "blue", NULL);

	bbox = gtk_hbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_SPREAD);
	gtk_box_pack_start (GTK_BOX (vbox), bbox, FALSE, FALSE, 0);
	gtk_widget_show (bbox);

	button = gtk_button_new_from_stock (GTK_STOCK_ADD);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	ct->table_add = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (table_add_clicked_cb), wk);

	button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	gtk_widget_set_sensitive (button, FALSE);
	ct->table_del = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (table_del_clicked_cb), wk);

	button = gtk_button_new_from_stock (GTK_STOCK_INDEX);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	gtk_widget_set_sensitive (button, FALSE);
	gtk_button_set_label (GTK_BUTTON (button), _("Contents"));
	ct->table_contents = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (table_contents_clicked_cb), wk);
	
	g_signal_connect (G_OBJECT (wk->priv->main_page_object_list), "selection_changed", 
			  G_CALLBACK (tables_selector_selection_changed_cb), wk);

	g_signal_connect (G_OBJECT (mg_conf_get_database (mg_server_get_conf (wk->priv->server))), "table_updated",
			  G_CALLBACK (table_updated_cb), wk);

	return vbox;
}

static void
finish_tables_workspace (Workspace *wk)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (mg_conf_get_database (mg_server_get_conf (wk->priv->server))),
					      G_CALLBACK (table_updated_cb), wk);
}

static void
table_info_display_update (MgDbTable *table, Workspace *wk)
{
	ContentsTables *ct = CONTENTS_TABLES (wk->priv->main_page_details_structs[WORKSPACE_TABLES]);
	const gchar *str = NULL;
	gchar *title = NULL;
	GtkTextIter start, end;

	if (table)
		str = mg_base_get_description (MG_BASE (table));
	if (str && *str) 
		gtk_label_set_text (GTK_LABEL (ct->description), str);
	else
		gtk_label_set_text (GTK_LABEL (ct->description), "---");

	mg_selector_set_mode_columns (MG_SELECTOR (ct->fields), (GObject*) table,
				      MG_SELECTOR_FIELDS, 
				      MG_SELECTOR_COLUMN_TYPE | MG_SELECTOR_COLUMN_FIELD_LENGTH | 
				      MG_SELECTOR_COLUMN_FIELD_NNUL |
				      MG_SELECTOR_COLUMN_FIELD_DEFAULT | MG_SELECTOR_COLUMN_COMMENTS);

	/* global title update */
	title = main_page_get_contents_title (wk, (GObject*) table, MG_DB_TABLE_TYPE);
	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (wk->priv->main_page_detail_label), title);
	g_free (title);

	/* integrity rules update */
	gtk_text_buffer_get_start_iter (ct->integ_descr, &start);
	gtk_text_buffer_get_end_iter (ct->integ_descr, &end);
	gtk_text_buffer_delete (ct->integ_descr, &start, &end);
	
	if (table) {
		GtkTextIter current;
		GSList *constraints, *list;

		gtk_text_buffer_get_start_iter (ct->integ_descr, &current);		
		/* constraints list */
		constraints = mg_db_table_get_constraints (table);
		
		/* PKey */
		list = constraints;
		while (list) {
			gboolean first = TRUE;
			if (mg_db_constraint_get_constraint_type (MG_DB_CONSTRAINT (list->data)) == 
			    CONSTRAINT_PRIMARY_KEY) {
				MgDbConstraint *cstr = MG_DB_CONSTRAINT (list->data);
				GSList *fields, *list2;
				gboolean header = FALSE;

				fields = mg_db_constraint_pkey_get_fields (cstr);
				list2 = fields;
				while (list2) {
					if (!header) {
						header = TRUE;
						gtk_text_buffer_insert_with_tags_by_name (ct->integ_descr,
											  &current, 
											  _("Primary key"), -1,
											  "section", NULL);
						gtk_text_buffer_insert (ct->integ_descr, &current, "\n", -1);
					}

					if (first) 
						first = FALSE;
					else
						gtk_text_buffer_insert (ct->integ_descr, &current, ", ", -1);

					gtk_text_buffer_insert (ct->integ_descr, &current, 
								mg_base_get_name (MG_BASE (list2->data)), -1);
					list2 = g_slist_next (list2);
				}
				g_slist_free (fields);
				gtk_text_buffer_insert (ct->integ_descr, &current, "\n\n", -1);
			}
			list = g_slist_next (list);
		}

		/* FKey */
		list = constraints;
		while (list) {
			if (mg_db_constraint_get_constraint_type (MG_DB_CONSTRAINT (list->data)) == 
			    CONSTRAINT_FOREIGN_KEY) {
				MgDbConstraint *cstr = MG_DB_CONSTRAINT (list->data);
				GSList *fields, *list2;
				gboolean header = FALSE;

				fields = mg_db_constraint_fkey_get_fields (cstr);
				list2 = fields;
				while (list2) {
					MgEntity *ent;
					if (!header) {
						header = TRUE;
						gtk_text_buffer_insert_with_tags_by_name (ct->integ_descr,
											  &current, 
											  _("Foreign key"), -1,
											  "section", NULL);
						gtk_text_buffer_insert (ct->integ_descr, &current, "\n", -1);
					}
						
					str = mg_base_get_name (MG_BASE (MG_DB_CONSTRAINT_FK_PAIR (list2->data)->fkey));
					gtk_text_buffer_insert (ct->integ_descr, &current, str, -1);
					gtk_text_buffer_insert (ct->integ_descr, &current, " --> ", -1);
					ent = mg_field_get_entity (MG_FIELD (MG_DB_CONSTRAINT_FK_PAIR (list2->data)->ref_pkey));
					str = mg_base_get_name (MG_BASE (ent));
					gtk_text_buffer_insert (ct->integ_descr, &current, str, -1);
					gtk_text_buffer_insert (ct->integ_descr, &current, ".", -1);
					str = mg_base_get_name (MG_BASE (MG_DB_CONSTRAINT_FK_PAIR (list2->data)->ref_pkey));
					gtk_text_buffer_insert (ct->integ_descr, &current, str, -1);
					gtk_text_buffer_insert (ct->integ_descr, &current, "\n", -1);
						
					list2 = g_slist_next (list2);
				}
				g_slist_free (fields);
				gtk_text_buffer_insert (ct->integ_descr, &current, "\n", -1);
			}
			list = g_slist_next (list);
		}

		/* Unique */
		list = constraints;
		while (list) {
			if (mg_db_constraint_get_constraint_type (MG_DB_CONSTRAINT (list->data)) == 
			    CONSTRAINT_UNIQUE) {
				MgDbConstraint *cstr = MG_DB_CONSTRAINT (list->data);
				GSList *fields, *list2;
				gboolean header = FALSE;

				fields = mg_db_constraint_unique_get_fields (cstr);
				list2 = fields;
				while (list2) {
					if (!header) {
						header = TRUE;
						gtk_text_buffer_insert_with_tags_by_name (ct->integ_descr,
											  &current, 
											  _("UNIQUE constraint"), -1,
											  "section", NULL);
						gtk_text_buffer_insert (ct->integ_descr, &current, "\n", -1);
					}
					else
						gtk_text_buffer_insert (ct->integ_descr, &current, ", ", -1);

					gtk_text_buffer_insert (ct->integ_descr, &current, 
								mg_base_get_name (MG_BASE (list2->data)), -1);
						
					list2 = g_slist_next (list2);
				}
				g_slist_free (fields);
				gtk_text_buffer_insert (ct->integ_descr, &current, "\n\n", -1);
			}
			list = g_slist_next (list);
		}

		/* check constraint: FIXME */

		g_slist_free (constraints);			
	}
}

static void
tables_selector_selection_changed_cb (MgSelector *mgsel, GObject *sel_object, Workspace *wk)
{
	ContentsTables *ct = CONTENTS_TABLES (wk->priv->main_page_details_structs[WORKSPACE_TABLES]);

	if (sel_object && !IS_MG_DB_TABLE (sel_object))
		return;

	wk->priv->main_page_selected_object [WORKSPACE_TABLES] = sel_object;
	gtk_notebook_set_current_page (GTK_NOTEBOOK (ct->notebook), sel_object ? 1 : 0);
	gtk_widget_set_sensitive (ct->table_del, sel_object ? TRUE : FALSE);
	gtk_widget_set_sensitive (ct->table_contents, sel_object ? TRUE : FALSE);
	table_info_display_update ((MgDbTable *) sel_object, wk);
}

static void
table_updated_cb (MgDatabase *db, MgDbTable *table, Workspace *wk)
{
	if ((GObject *)table == wk->priv->main_page_selected_object [WORKSPACE_TABLES])
		table_info_display_update (table, wk);
}

static void
table_add_clicked_cb (GtkButton *button, Workspace *wk)
{
	NOT_YET_IMPLEMENTED (wk, _("table creation"));
}

static void
table_del_clicked_cb (GtkButton *button, Workspace *wk)
{
	NOT_YET_IMPLEMENTED (wk, _("table removal"));
}

static void
table_contents_clicked_cb (GtkButton *button, Workspace *wk)
{
	MgQuery *query;
	gchar *sql;
	MgConf *conf = mg_server_get_conf (wk->priv->server);
	GError *error = NULL;
	
	sql = g_strdup_printf ("SELECT * FROM %s\n", 
			       mg_base_get_name (MG_BASE (wk->priv->main_page_selected_object [WORKSPACE_TABLES])));
	query = (MgQuery *) mg_query_new_from_sql (conf, sql, &error);
	g_free (sql);

	if (query) {
		GtkWidget *dlg, *wid;
		gchar *str;
		GSList *targets;

		str = g_strdup_printf (_("Contents of '%s'"), 
				       mg_base_get_name (MG_BASE (wk->priv->main_page_selected_object [WORKSPACE_TABLES])));
		dlg = gtk_dialog_new_with_buttons (str, NULL, GTK_DIALOG_NO_SEPARATOR, 
						   GTK_STOCK_CLOSE, GTK_RESPONSE_NONE, NULL);
		g_free (str);

		
		targets = mg_query_get_targets (query);
		wid = mg_work_grid_new (query, MG_TARGET (targets->data));
		g_slist_free (targets);
		
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), wid, TRUE, TRUE, 0);
		mg_work_widget_run (MG_WORK_WIDGET (wid), MG_ACTION_ASK_CONFIRM_UPDATE | MG_ACTION_ASK_CONFIRM_DELETE |
				    MG_ACTION_ASK_CONFIRM_INSERT | MG_ACTION_REPORT_ERROR);
		gtk_widget_show (wid);

		gtk_window_set_default_size (GTK_WINDOW (dlg), 400, 250);
		g_signal_connect_swapped (GTK_OBJECT (dlg), 
					  "response", 
					  G_CALLBACK (gtk_widget_destroy),
					  GTK_OBJECT (dlg));
				  
		gtk_widget_show (dlg);

		g_object_unref (G_OBJECT (query));
	}
	else {
		if (error) {
			g_print ("ERROR: %s\n", error->message);
			g_error_free (error);
		}
	}
}







/*
 * Contents for data types
 */
typedef struct {
	GtkWidget     *notebook;
	GtkWidget     *description;
	gint           filter_mode;
	GtkTextBuffer *textbuffer;
} ContentsDTypes;
#define CONTENTS_DTYPES(x) ((ContentsDTypes*)(x))

static void dtypes_selector_selection_changed_cb (MgSelector *mgsel, GObject *sel_object, Workspace *wk);
static void dtypes_filter_mode_changed_cb (GtkWidget *wid, Workspace *wk);
static void dtype_updated_cb (MgServer *srv, MgServerDataType *dtype, Workspace *wk);

static GtkWidget *
create_dtypes_workspace (Workspace *wk)
{
	GtkWidget *label, *vbox, *wid, *nb, *vbox2, *bbox, *hbox, *sw;
	GtkWidget *rd1, *rd2, *rd3;
	ContentsDTypes *ct;

	vbox = gtk_vbox_new (FALSE, 0);
	ct = g_new0 (ContentsDTypes, 1);
	wk->priv->main_page_details_structs[WORKSPACE_DTYPES] = ct;
	
	nb = gtk_notebook_new ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (nb), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nb), FALSE);
	gtk_box_pack_start (GTK_BOX (vbox), nb, TRUE, TRUE, 0);
	gtk_widget_show (nb);
	ct->notebook = nb;
	
	label = gtk_label_new (_("Please select a data type from the list on the left"));
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), label, NULL);
	gtk_widget_show (label);
	
	vbox2 = gtk_vbox_new (FALSE, 5);
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), vbox2, NULL);
	gtk_widget_show (vbox2);

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Description:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	label = gtk_label_new (NULL);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_widget_show (label);
	ct->description = label;
	
	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Select a filter option:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	bbox = gtk_vbutton_box_new ();
	gtk_box_pack_start (GTK_BOX (hbox), bbox, FALSE, FALSE, 0);
	gtk_widget_show (bbox);

	rd1 = gtk_radio_button_new_with_label (NULL, _("Functions returning this data type"));
	gtk_box_pack_start (GTK_BOX (bbox), rd1, FALSE, FALSE, 0);
	gtk_widget_show (rd1);
	g_object_set_data (G_OBJECT (rd1), "mode", GINT_TO_POINTER (0));
	g_signal_connect (G_OBJECT (rd1), "toggled",
			  G_CALLBACK (dtypes_filter_mode_changed_cb), wk);

	rd2 = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (rd1),
							   _("Functions using this data type"));
	gtk_box_pack_start (GTK_BOX (bbox), rd2, FALSE, FALSE, 0);
	gtk_widget_show (rd2);
	g_object_set_data (G_OBJECT (rd2), "mode", GINT_TO_POINTER (1));
	g_signal_connect (G_OBJECT (rd2), "toggled",
			  G_CALLBACK (dtypes_filter_mode_changed_cb), wk);

	rd3 = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (rd1),
							   _("Aggregates using this data type"));
	gtk_box_pack_start (GTK_BOX (bbox), rd3, FALSE, FALSE, 0);
	gtk_widget_show (rd3);
	g_object_set_data (G_OBJECT (rd3), "mode", GINT_TO_POINTER (2));
	g_signal_connect (G_OBJECT (rd3), "toggled",
			  G_CALLBACK (dtypes_filter_mode_changed_cb), wk);

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Result of filter:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), sw, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_widget_show (sw);

	wid = gtk_text_view_new ();
	gtk_container_add (GTK_CONTAINER (sw), wid);
	gtk_text_view_set_left_margin (GTK_TEXT_VIEW (wid), 5);
	gtk_text_view_set_right_margin (GTK_TEXT_VIEW (wid), 5);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (wid), FALSE);
	ct->textbuffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wid));
	gtk_text_buffer_set_text (ct->textbuffer, "", -1);
	gtk_widget_show (wid);
	gtk_text_buffer_create_tag (ct->textbuffer, "funcname",
				    "weight", PANGO_WEIGHT_BOLD,
				    "foreground", "blue", NULL);
	
	gtk_text_buffer_create_tag (ct->textbuffer, "descr",
				    "style", PANGO_STYLE_ITALIC, NULL);
	
	g_signal_connect (G_OBJECT (wk->priv->main_page_object_list), "selection_changed", 
			  G_CALLBACK (dtypes_selector_selection_changed_cb), wk);
	
	g_signal_connect (G_OBJECT (wk->priv->server), "data_type_updated",
			  G_CALLBACK (dtype_updated_cb), wk);
	
	return vbox;
}

static void
finish_dtypes_workspace (Workspace *wk)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (wk->priv->server), 
					      G_CALLBACK (dtype_updated_cb), wk);
}

static void 
dtypes_filter_mode_changed_cb (GtkWidget *wid, Workspace *wk)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wid))) {
		ContentsDTypes *ct;
		GObject *sel_obj;
		
		ct = CONTENTS_DTYPES (wk->priv->main_page_details_structs[WORKSPACE_DTYPES]);
		ct->filter_mode = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "mode"));
		sel_obj = mg_selector_get_selected_object (MG_SELECTOR (wk->priv->main_page_object_list));
		dtypes_selector_selection_changed_cb (MG_SELECTOR (wk->priv->main_page_object_list), sel_obj, wk);
	}

}

static gchar *function_get_args (MgServerFunction *func);
static void
dtype_info_display_update (MgServerDataType *dtype, Workspace *wk)
{
	ContentsDTypes *ct = CONTENTS_DTYPES (wk->priv->main_page_details_structs[WORKSPACE_DTYPES]);
	GtkTextIter start, end;
	const gchar *str = NULL;
	gchar *title = NULL;

	if (dtype)
		str = mg_base_get_description (MG_BASE (dtype));
	
	if (str && *str) 
		gtk_label_set_text (GTK_LABEL (ct->description), str);
	else
		gtk_label_set_text (GTK_LABEL (ct->description), "---");

	/* global title update */
	title = main_page_get_contents_title (wk, (GObject *) dtype, MG_SERVER_DATA_TYPE_TYPE);
	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (wk->priv->main_page_detail_label), title);
	g_free (title);

	/* apply filter */
	gtk_text_buffer_get_start_iter (ct->textbuffer, &start);
	gtk_text_buffer_get_end_iter (ct->textbuffer, &end);
	gtk_text_buffer_delete (ct->textbuffer, &start, &end);
	if (dtype) {
		GSList *list, *funcs, *args, *aggs;
		gboolean getfunc;
		MgServerDataType *ldt;
		
		funcs = mg_server_get_functions (wk->priv->server);
		aggs = mg_server_get_aggregates (wk->priv->server);
		list = g_slist_concat (funcs, aggs);
		funcs = list;
		while (list) {
			getfunc = FALSE;
			switch (ct->filter_mode) {
			case 0:
				if (IS_MG_SERVER_FUNCTION (list->data)) {
					ldt = mg_server_function_get_ret_type (MG_SERVER_FUNCTION (list->data));
					getfunc = (ldt == dtype) ? TRUE : FALSE;
				}
				break;
			case 1:
				if (IS_MG_SERVER_FUNCTION (list->data)) {
					args = mg_server_function_get_arg_types (MG_SERVER_FUNCTION (list->data));
					while (args && !getfunc) {
						if (args->data == (gpointer) dtype)
							getfunc = TRUE;
						args = g_slist_next (args);
					}
				}
				break;
			case 2:
				if (IS_MG_SERVER_AGGREGATE (list->data)) {
					ldt = mg_server_aggregate_get_arg_type (MG_SERVER_AGGREGATE (list->data));
					getfunc = !ldt || (ldt == dtype) ? TRUE : FALSE;
				}
				break;
			}

			if (getfunc) {
				gchar *str;

				str = mg_base_get_name (MG_BASE (list->data));
				gtk_text_buffer_get_end_iter (ct->textbuffer, &end);
				gtk_text_buffer_insert_with_tags_by_name (ct->textbuffer, 
									  &end, str, -1, 
									  "funcname", NULL);

				if (IS_MG_SERVER_FUNCTION (list->data))
					str = function_get_args (MG_SERVER_FUNCTION (list->data));
				else {
					if ((ldt = mg_server_aggregate_get_arg_type (MG_SERVER_AGGREGATE (list->data)))) {
						str = g_strdup_printf (" (%s)", mg_base_get_name (MG_BASE (ldt)));
					}
					else
						str = g_strdup (" (*)");
				}
				gtk_text_buffer_get_end_iter (ct->textbuffer, &end);
				gtk_text_buffer_insert (ct->textbuffer, &end, str, -1);
				g_free (str);

				str = mg_base_get_description (MG_BASE (list->data));
				if (str && *str) {
					gchar *str2 = g_strdup_printf (" -- %s\n", str);
					gtk_text_buffer_get_end_iter (ct->textbuffer, &end);
					gtk_text_buffer_insert_with_tags_by_name (ct->textbuffer, &end, str2, -1, 
										  "descr", NULL);
					g_free (str2);
				}
				else {
					gtk_text_buffer_get_end_iter (ct->textbuffer, &end);
					gtk_text_buffer_insert (ct->textbuffer, &end, "\n", -1);
				}
			}
			list = g_slist_next (list);
		}
		g_slist_free (funcs);
	}
}

static gchar *
function_get_args (MgServerFunction *func)
{
	GString *string;
	const GSList *args;
	gchar *retval;
	gboolean firstarg = TRUE;

	string = g_string_new ("");
	args = mg_server_function_get_arg_types (func);
	g_string_append (string, " (");
	while (args) {
		if (firstarg)
			firstarg = FALSE;
		else
			g_string_append (string, ", ");
		g_string_append (string,
				 mg_server_data_type_get_sqlname (MG_SERVER_DATA_TYPE (args->data)));
		args = g_slist_next (args);
	}
	g_string_append (string, ")");
	retval = string->str;
	g_string_free (string, FALSE);

	return retval;
}


static void
dtype_updated_cb (MgServer *srv, MgServerDataType *dtype, Workspace *wk)
{
	if ((GObject *)dtype == wk->priv->main_page_selected_object [WORKSPACE_DTYPES])
		dtype_info_display_update (dtype, wk);
}


static void
dtypes_selector_selection_changed_cb (MgSelector *mgsel, GObject *sel_object, Workspace *wk)
{
	ContentsDTypes *ct = CONTENTS_DTYPES (wk->priv->main_page_details_structs[WORKSPACE_DTYPES]);

	if (sel_object && !IS_MG_SERVER_DATA_TYPE (sel_object))
		return;

	wk->priv->main_page_selected_object [WORKSPACE_DTYPES] = sel_object;
	gtk_notebook_set_current_page (GTK_NOTEBOOK (ct->notebook), sel_object ? 1 : 0);
	dtype_info_display_update ((MgServerDataType *) sel_object, wk);
}





/*
 * Contents for queries
 */
typedef struct {
	GtkWidget *notebook;
	GtkWidget *description;
	GtkWidget *sql_editor;
	GtkWidget *query_add;
	GtkWidget *query_edit;
	GtkWidget *query_del;
	GtkWidget *query_exec;
} ContentsQueries;
#define CONTENTS_QUERIES(x) ((ContentsQueries*)(x))

static void queries_selector_selection_changed_cb (MgSelector *mgsel, GObject *sel_object, Workspace *wk);
static void query_add_clicked_cb (GtkButton *button, Workspace *wk);
static void query_edit_clicked_cb (GtkButton *button, Workspace *wk);
static void query_del_clicked_cb (GtkButton *button, Workspace *wk);
static void query_exec_clicked_cb (GtkButton *button, Workspace *wk);
static void query_updated_cb (MgConf *conf, MgQuery *query, Workspace *wk);

static GtkWidget *
create_queries_workspace (Workspace *wk)
{
	GtkWidget *label, *vbox, *nb, *vbox2, *bbox, *button, *wid, *hbox;
	ContentsQueries *ct;

	vbox = gtk_vbox_new (FALSE, 0);
	ct = g_new0 (ContentsQueries, 1);
	wk->priv->main_page_details_structs[WORKSPACE_QUERIES] = ct;
	
	nb = gtk_notebook_new ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (nb), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nb), FALSE);
	gtk_box_pack_start (GTK_BOX (vbox), nb, TRUE, TRUE, 0);
	gtk_widget_show (nb);
	ct->notebook = nb;
	
	label = gtk_label_new (_("Please select a query from the list on the left,\n"
				 "or create a new one using the 'Add' button below."));
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), label, NULL);
	gtk_widget_show (label);
	
	vbox2 = gtk_vbox_new (FALSE, 5);
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), vbox2, NULL);
	gtk_widget_show (vbox2);

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Description:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	label = gtk_label_new (NULL);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_widget_show (label);
	ct->description = label;

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>SQL statement:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	wid = gnome_db_editor_new ();
        gnome_db_editor_set_editable (GNOME_DB_EDITOR (wid), FALSE);
        gnome_db_editor_set_highlight (GNOME_DB_EDITOR (wid), FALSE); /* CHANGE ME */
        gtk_box_pack_start (GTK_BOX (hbox), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	ct->sql_editor = wid;

	bbox = gtk_hbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_SPREAD);
	gtk_box_pack_start (GTK_BOX (vbox), bbox, FALSE, FALSE, 0);
	gtk_widget_show (bbox);

	button = gtk_button_new_from_stock (GTK_STOCK_ADD);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	ct->query_add = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (query_add_clicked_cb), wk);

	button = gtk_button_new_with_label (_("Edit"));
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	gtk_widget_set_sensitive (button, FALSE);
	ct->query_edit = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (query_edit_clicked_cb), wk);

	button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	gtk_widget_set_sensitive (button, FALSE);
	ct->query_del = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (query_del_clicked_cb), wk);

	button = gtk_button_new_from_stock (GTK_STOCK_INDEX);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	gtk_widget_set_sensitive (button, FALSE);
	gtk_button_set_label (GTK_BUTTON (button), _("Execute"));
	ct->query_exec = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (query_exec_clicked_cb), wk);
	
	g_signal_connect (G_OBJECT (wk->priv->main_page_object_list), "selection_changed", 
			  G_CALLBACK (queries_selector_selection_changed_cb), wk);

	g_signal_connect (G_OBJECT (mg_server_get_conf (wk->priv->server)), "query_updated",
			  G_CALLBACK (query_updated_cb), wk);
	return vbox;
}

static void
finish_queries_workspace (Workspace *wk)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (mg_server_get_conf (wk->priv->server)),
					      G_CALLBACK (query_updated_cb), wk);
}

static void
query_info_display_update (MgQuery *query, Workspace *wk)
{
	ContentsQueries *ct = CONTENTS_QUERIES (wk->priv->main_page_details_structs[WORKSPACE_QUERIES]);
	const gchar *str = NULL;
	gchar *title = NULL;

	if (query)
		str = mg_base_get_description (MG_BASE (query));
	if (str && *str) 
		gtk_label_set_text (GTK_LABEL (ct->description), str);
	else
		gtk_label_set_text (GTK_LABEL (ct->description), "---");

	/* global title update */
	title = main_page_get_contents_title (wk, (GObject*) query, MG_QUERY_TYPE);
	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (wk->priv->main_page_detail_label), title);
	g_free (title);

	/* SQL version of the query */
	if (query) {
		gchar *sql;
		GError *error = NULL;
		sql = mg_renderer_render_as_sql (MG_RENDERER (query), NULL,
						 MG_RENDERER_EXTRA_PRETTY_SQL, &error);
		if (sql) {
			gnome_db_editor_set_text (GNOME_DB_EDITOR (ct->sql_editor), sql, -1);
			g_free (sql);
		}
		else {
			if (error) {
				str = error->message;
				gnome_db_editor_set_text (GNOME_DB_EDITOR (ct->sql_editor), str, -1);
				g_error_free (error);
			}
			else
				gnome_db_editor_set_text (GNOME_DB_EDITOR (ct->sql_editor),
							  _("Non reported error"), -1);
		}
	}
	else
		gnome_db_editor_set_text (GNOME_DB_EDITOR (ct->sql_editor), "", -1);	
}

static void
queries_selector_selection_changed_cb (MgSelector *mgsel, GObject *sel_object, Workspace *wk)
{
	ContentsQueries *ct = CONTENTS_QUERIES (wk->priv->main_page_details_structs[WORKSPACE_QUERIES]);

	if (sel_object && !IS_MG_QUERY (sel_object))
		return;

	wk->priv->main_page_selected_object [WORKSPACE_QUERIES] = sel_object;
	gtk_notebook_set_current_page (GTK_NOTEBOOK (ct->notebook), sel_object ? 1 : 0);
	gtk_widget_set_sensitive (ct->query_del, sel_object ? TRUE : FALSE);
	gtk_widget_set_sensitive (ct->query_edit, sel_object ? TRUE : FALSE);
	gtk_widget_set_sensitive (ct->query_exec, sel_object ? TRUE : FALSE);
	query_info_display_update ((MgQuery*) sel_object, wk);
}

static void
query_updated_cb (MgConf *conf, MgQuery *query, Workspace *wk)
{
	if ((GObject *)query == wk->priv->main_page_selected_object [WORKSPACE_QUERIES])
		query_info_display_update (query, wk);
}

static void
dialog_exec_response_cb (GtkDialog *dlg, gint response, GObject *obj)
{
	gtk_widget_destroy (GTK_WIDGET (dlg));
}

static void
query_add_clicked_cb (GtkButton *button, Workspace *wk)
{
	GtkWidget *druid;

	druid = query_druid_new (mg_server_get_conf (wk->priv->server));
	gtk_widget_show (druid);
	gnome_druid_construct_with_window (GNOME_DRUID (druid), _("New Query"), NULL, TRUE, NULL);
}

static void
query_edit_clicked_cb (GtkButton *button, Workspace *wk)
{
	GtkWidget *dlg, *editor;
	GObject *sel_query = wk->priv->main_page_selected_object [WORKSPACE_QUERIES];
	gchar *title;

	if (!sel_query)
		return;

	editor = query_editor_new (MG_QUERY (sel_query));
	gtk_widget_show (editor);
	title = g_strdup_printf (_("Edition of query '%s'"), mg_base_get_name (MG_BASE (sel_query)));
	dlg = gtk_dialog_new_with_buttons (title, NULL, 0, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
	g_free (title);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), editor, TRUE, TRUE, 6);
	gtk_widget_set_size_request (dlg, 550, 430);
	gtk_widget_show (dlg);
	g_signal_connect (G_OBJECT (dlg), "response",
			  G_CALLBACK (dialog_exec_response_cb), NULL);
}

static void
query_del_clicked_cb (GtkButton *button, Workspace *wk)
{
	GObject *sel_query = wk->priv->main_page_selected_object [WORKSPACE_QUERIES];
	GtkWidget *dlg;

	if (!sel_query)
		return;

	dlg = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
				      _("Do you want to delete the '%s' query?"), 
				      mg_base_get_name (MG_BASE (sel_query)));
	if (gtk_dialog_run (GTK_DIALOG (dlg)) == GTK_RESPONSE_YES) {
		mg_base_nullify (MG_BASE (sel_query));

		/* saving of MgConf */
		mg_conf_save_xml (mg_server_get_conf (wk->priv->server), NULL);
	}
	gtk_widget_destroy (dlg);
}


static void
query_exec_clicked_cb (GtkButton *button, Workspace *wk)
{
	GtkWidget *topdlg;
	GError *error = NULL;
	MgQuery *query;
	gint result = GTK_RESPONSE_ACCEPT;
	MgContext *context;
		
	query = (MgQuery *) wk->priv->main_page_selected_object [WORKSPACE_QUERIES];

	context = mg_entity_get_exec_context (MG_ENTITY (query));
	if (mg_context_needs_user_input (context)) {
		GtkWidget *dlg;
		MgForm *sform;
		
		dlg = mg_form_new_in_dialog (mg_server_get_conf (wk->priv->server), 
					     context, NULL,
					     _("Values to be filled"), 
					     _("<big><b>Required values:</b></big>\n"
					       "<small>The following values are required to "
					       "execute the query.</small>"));
		sform = g_object_get_data (G_OBJECT (dlg), "form");
		mg_form_set_entries_auto_default (sform, TRUE);
		
		gtk_widget_show (dlg);
		result = gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
	}
	
	if (result == GTK_RESPONSE_ACCEPT) {
		if (mg_query_is_modif_query (query)) { /* modification query */
			mg_util_query_execute_modif (query, context, TRUE, TRUE, TRUE, NULL, NULL, NULL);
		}
		else { /* non modification query */
			if (mg_query_is_select_query (query)) {
				GtkWidget *grid;
				MgTarget *target;
				guint mode = 0;
				gchar *str;

				/* FIXME: the first target is the one that gets modified */
				GSList *targets = mg_query_get_targets (query);
				target = targets ? (MgTarget *) targets->data : NULL;
				if (targets) g_slist_free (targets);

				grid = mg_work_grid_new (query, target);
				str = g_strdup_printf (_("Execution of query '%s'"), mg_base_get_name (MG_BASE (query)));
				g_object_set (G_OBJECT (grid), "title_string", str, NULL);
				topdlg = gtk_dialog_new_with_buttons (str, NULL,
								      0, GTK_STOCK_CLOSE,
								      GTK_RESPONSE_ACCEPT,
								      NULL);
				g_free (str);
				gtk_container_add (GTK_CONTAINER (GTK_DIALOG (topdlg)->vbox), grid);
				gtk_widget_set_size_request (grid, 400, 250);
				if (target)
					mode = MG_ACTION_ASK_CONFIRM_DELETE | 
						MG_ACTION_ASK_CONFIRM_UPDATE | 
						MG_ACTION_ASK_CONFIRM_INSERT;
				
				mg_work_widget_bind_to_context_all (MG_WORK_WIDGET (grid), context);
				mg_work_widget_run (MG_WORK_WIDGET (grid), mode);
				gtk_widget_show_all (topdlg);
				g_signal_connect (G_OBJECT (topdlg), "response",
						  G_CALLBACK (dialog_exec_response_cb), NULL);
			}
			else {
				NOT_YET_IMPLEMENTED (wk, _("execution of SQL specific only code"));
			}
		}
	}
	
	/* get rid of that context object */
	g_object_unref (G_OBJECT (context));
}
