/* query-editor-fields.c
 *
 * Copyright (C) 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "query-editor-fields.h"
#include "marshal.h"

/*
 *  
 *
 * Implementation of the QueryEditorFields Widget
 * 
 *
 */

struct _QueryEditorFieldsPrivate {
	Query         *query;
	QueryField    *selection;
	gboolean       query_in_destroy;

	/* Widgets */
	GtkWidget     *treeview;
	GtkWidget     *edit_btn;
	GtkWidget     *del_btn;

	QueryEditorFieldsShowType show_type;
};

enum {
	COLUMN_QUERY_FIELD,
	COLUMN_NAME,
	COLUMN_NAME_EDITABLE,
	COLUMN_PRINT,
	COLUMN_PRINTNAME,
	COLUMN_VALUE,
	COLUMN_GROUPING,
	COLUMN_ORDER,
	COLUMN_MISC,
	NUM_COLUMNS
};

static void query_editor_fields_class_init (QueryEditorFieldsClass * class);
static void query_editor_fields_init (QueryEditorFields * qef);
static void query_editor_fields_initialize (QueryEditorFields * qef);
static void query_editor_fields_finalize (GObject   *obj);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *qee_parent_class = NULL;

guint
query_editor_fields_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (QueryEditorFieldsClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) query_editor_fields_class_init,
			NULL,
			NULL,
			sizeof (QueryEditorFields),
			0,
			(GInstanceInitFunc) query_editor_fields_init
		};		

		type = g_type_register_static (GTK_TYPE_VBOX, "QueryEditorFields", &info, 0);
	}
	return type;
}

static void
query_editor_fields_class_init (QueryEditorFieldsClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	qee_parent_class = g_type_class_peek_parent (class);
	object_class->finalize = query_editor_fields_finalize;
}

static void
query_editor_fields_init (QueryEditorFields * qef)
{
	qef->priv = g_new0 (QueryEditorFieldsPrivate, 1);
	qef->priv->query = NULL;
	qef->priv->selection = NULL;

	qef->priv->treeview = NULL;
	qef->priv->query_in_destroy = FALSE;

	qef->priv->edit_btn = NULL;
	qef->priv->del_btn = NULL;

	qef->priv->show_type = QEF_EXPR;
}

static void query_destroy_cb (QueryEditorFields *qef, Query *q); /* GWeakNotify */
static void query_type_changed_cb (Query *q, QueryEditorFields *qef);
static void query_editor_fields_initialize (QueryEditorFields * qef);
static void query_field_created_cb (Query *q, QueryField *new_field, QueryEditorFields *qef);
static void query_field_dropped_cb (Query *q, QueryField *old_field, QueryEditorFields *qef);
static void query_field_name_alias_changed_cb (Query *q, QueryField *field, QueryEditorFields *qef);
GtkWidget *
query_editor_fields_new (Query * q)
{
	GObject   *obj;
	QueryEditorFields *qef;

	g_return_val_if_fail (q, NULL);
	g_return_val_if_fail (IS_QUERY (q), NULL);

	obj = g_object_new (QUERY_EDITOR_FIELDS_TYPE, NULL);
	qef = QUERY_EDITOR_FIELDS (obj);
	qef->priv->query = q;

	gtk_box_set_homogeneous (GTK_BOX (qef), FALSE);
	query_editor_fields_initialize (qef);

	/* signals */
	g_object_weak_ref (G_OBJECT (q), (GWeakNotify) query_destroy_cb, qef);

	g_signal_connect (G_OBJECT (qef->priv->query), "type_changed",
			  G_CALLBACK (query_type_changed_cb), qef);

	g_signal_connect (G_OBJECT (qef->priv->query), "field_created",
			  G_CALLBACK (query_field_created_cb), qef);
	
	g_signal_connect (G_OBJECT (qef->priv->query), "field_dropped",
			  G_CALLBACK (query_field_dropped_cb), qef);

	g_signal_connect (G_OBJECT (qef->priv->query), "field_modified",
			  G_CALLBACK (query_field_name_alias_changed_cb), qef);

	g_signal_connect (G_OBJECT (qef->priv->query), "field_name_modified",
			  G_CALLBACK (query_field_name_alias_changed_cb), qef);

	g_signal_connect (G_OBJECT (qef->priv->query), "field_alias_modified",
			  G_CALLBACK (query_field_name_alias_changed_cb), qef);


	return GTK_WIDGET (obj);
}

static void 
query_editor_fields_finalize (GObject   *object)
{
	QueryEditorFields *qef;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_QUERY_EDITOR_FIELDS (object));

	qef = QUERY_EDITOR_FIELDS (object);

	if (qef->priv) {
		/* signal handlers */
		g_signal_handlers_disconnect_by_func (G_OBJECT (qef->priv->query),
						      G_CALLBACK (query_type_changed_cb), qef);
		
		g_signal_handlers_disconnect_by_func (G_OBJECT (qef->priv->query), 
						      G_CALLBACK (query_field_created_cb), qef);
		
		g_signal_handlers_disconnect_by_func (G_OBJECT (qef->priv->query), 
						      G_CALLBACK (query_field_dropped_cb), qef);
		
		g_signal_handlers_disconnect_by_func (G_OBJECT (qef->priv->query),
						      G_CALLBACK (query_field_name_alias_changed_cb), qef);
		
		if (!qef->priv->query_in_destroy)
			g_object_weak_unref (G_OBJECT (qef->priv->query), (GWeakNotify) query_destroy_cb, qef);

		g_free (qef->priv);
		qef->priv = NULL;
	}
		
	/* parent class */
	qee_parent_class->finalize (object);
}

Query *
query_editor_fields_get_query (QueryEditorFields *qef)
{
	g_return_val_if_fail (qef && IS_QUERY_EDITOR_FIELDS (qef), NULL);

	return qef->priv->query;
}

static void 
query_destroy_cb (QueryEditorFields *qef, Query *q)
{
	qef->priv->query_in_destroy = TRUE;
	gtk_widget_destroy (GTK_WIDGET (qef));
}

/* Update the GUI because of Query changes */
static void 
query_type_changed_cb (Query *q, QueryEditorFields *qef)
{
	gboolean sensi;

	sensi = q->type == QUERY_TYPE_STD ? TRUE : FALSE;
	gtk_widget_set_sensitive (GTK_WIDGET (qef), sensi);
}

static gboolean query_field_to_be_displayed (QueryField *qf, QueryEditorFields *qef);
static gint find_qf_position (QueryField *qf, QueryEditorFields *qef);
static void store_qf_in_model (GtkListStore *model, GtkTreeIter *iter, QueryField *qf);
static void 
query_field_created_cb (Query *q, QueryField *new_field, QueryEditorFields *qef)
{
	if (query_field_to_be_displayed (new_field, qef)) {
		gint pos;
		GtkTreeIter iter;
		GtkTreeModel *model;

		model = gtk_tree_view_get_model (GTK_TREE_VIEW (qef->priv->treeview));
		pos = find_qf_position (new_field, qef);
		if (pos == 0) {
			gtk_list_store_prepend (GTK_LIST_STORE (model), &iter);
			store_qf_in_model (GTK_LIST_STORE (model), &iter, new_field);
		}
		else {
			GtkTreeIter sibling;
			if (gtk_tree_model_iter_nth_child (model, &sibling, NULL, pos-1)) {
				gtk_list_store_insert_after (GTK_LIST_STORE (model), &iter, &sibling);
				store_qf_in_model (GTK_LIST_STORE (model), &iter, new_field);
			}
			else
				g_warning ("query_field_created_cb(): can't find iterator at pos=%d\n", 
					   pos);
		}
	}
}

static gint
find_qf_position (QueryField *qf, QueryEditorFields *qef)
{
	gint i = 0;
	gboolean found = FALSE;
	GSList *list = qef->priv->query->fields;
	
	while (list && !found) {
		if (QUERY_FIELD (list->data) == qf)
			found = TRUE;
		else {
			if (query_field_to_be_displayed (QUERY_FIELD (list->data), qef))
				i++;
		}

		list = g_slist_next (list);
	}

	return i;
}


static void 
query_field_dropped_cb (Query *q, QueryField *old_field, QueryEditorFields *qef)
{
	gboolean found = FALSE;
	GtkTreeIter iter;
	GtkTreeModel *model;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (qef->priv->treeview));
	if (gtk_tree_model_get_iter_first (model, &iter)) {
		QueryField *qf;

		gtk_tree_model_get (model, &iter, COLUMN_QUERY_FIELD, &qf, -1);
		if (qf == old_field) 
			gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
		else {
			while (!found && gtk_tree_model_iter_next (model, &iter)) {
				gtk_tree_model_get (model, &iter, COLUMN_QUERY_FIELD, &qf, -1);
				if (qf == old_field) {
					gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
					found = TRUE;
				}
			}
		}
	}
}

static void 
query_field_name_alias_changed_cb (Query *q, QueryField *field, QueryEditorFields *qef)
{
	gint pos;
	GtkTreeIter iter;
	GtkTreeModel *model;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (qef->priv->treeview));
	pos = find_qf_position (field, qef);
	gtk_tree_model_iter_nth_child (model, &iter, NULL, pos);

	store_qf_in_model (GTK_LIST_STORE (model), &iter, field);
	/* if (!query_field_to_be_displayed (field, qef))  */
/* 		gtk_list_store_remove (GTK_LIST_STORE (model), &iter); */
}


static void
store_qf_in_model (GtkListStore *model, GtkTreeIter *iter, QueryField *qf)
{
	gchar *value;
	gchar *alias, *name;

	g_return_if_fail (qf && IS_QUERY_FIELD (qf));

	if (query_field_get_alias (qf))
		alias = g_strdup (query_field_get_alias (qf));
	else
		alias = g_strdup ("");

	if (query_field_get_name (qf))
		name = g_strdup (query_field_get_name (qf));
	else
		name = g_strdup ("");
	value = query_field_render_as_string (qf, NULL);
	
	gtk_list_store_set (model, iter,
			    COLUMN_QUERY_FIELD, qf,
			    COLUMN_NAME, name,
			    COLUMN_NAME_EDITABLE, TRUE,
			    COLUMN_PRINT, query_field_get_is_printed (qf),
			    COLUMN_PRINTNAME, alias,
			    COLUMN_VALUE, value,
			    COLUMN_GROUPING, FALSE,
			    COLUMN_ORDER, FALSE,
			    COLUMN_MISC, NULL,
			    -1);
	g_free (value);
	g_free (alias);
	g_free (name);
}

static void cell_toggled_cb (GtkCellRendererToggle *renderer, gchar *path, QueryEditorFields * qef);
static void cell_edited_cb (GtkCellRendererToggle *renderer, gchar *path, gchar *new_str, 
			    QueryEditorFields * qef);
static void query_editor_fields_update_model (QueryEditorFields *qef);
static void obj_mitem_display (GtkButton *button, QueryEditorFields * qef);
static void obj_select_changed_cb (GtkTreeSelection *select, QueryEditorFields * qef);
static void add_expression_cb (GtkWidget *button, QueryEditorFields * qef);
static void edit_expression_cb (GtkWidget *button, QueryEditorFields * qef);
static void del_expression_cb (GtkWidget *button, QueryEditorFields * qef);

static void
query_editor_fields_initialize (QueryEditorFields * qef)
{
	GtkWidget *treeview, *bb, *button, *wid, *sw, *hb, *align;
	GtkTreeModel *model;
	GtkCellRenderer *renderer;
	GtkTreeSelection *select;

	/* model creation */
	model = GTK_TREE_MODEL (gtk_list_store_new (NUM_COLUMNS, G_TYPE_POINTER, G_TYPE_STRING, G_TYPE_BOOLEAN, 
						    G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_STRING, 
						    G_TYPE_BOOLEAN, G_TYPE_BOOLEAN, G_TYPE_STRING));

	/* treeview itself */
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start (GTK_BOX (qef), sw, TRUE, TRUE, GNOME_PAD/2.);
	
	treeview = gtk_tree_view_new_with_model (model);
	g_object_unref (G_OBJECT (model));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);
	qef->priv->treeview = treeview;

	gtk_container_add (GTK_CONTAINER (sw), treeview);
	gtk_widget_show_all (sw);

	/* treeview selection */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	g_signal_connect (G_OBJECT (select), "changed",
			  G_CALLBACK (obj_select_changed_cb), qef);


	/* treeview columns */
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						     -1, _("Name"), renderer,
						     "text", COLUMN_NAME,
						     "editable", COLUMN_NAME_EDITABLE,
						     NULL);
	g_object_set_data (G_OBJECT (renderer), "column", GINT_TO_POINTER (COLUMN_NAME));
	g_signal_connect (G_OBJECT (renderer), "edited",
			  G_CALLBACK (cell_edited_cb), qef);
	
	renderer = gtk_cell_renderer_toggle_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						     -1, _("Printed"), renderer,
						     "active", COLUMN_PRINT,
						     "activatable", COLUMN_NAME_EDITABLE,
						     NULL);
	g_object_set_data (G_OBJECT (renderer), "column", GINT_TO_POINTER (COLUMN_PRINT));
	g_signal_connect (G_OBJECT (renderer), "toggled",
			  G_CALLBACK (cell_toggled_cb), qef);
	

	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						     -1, _("Print As"), renderer,
						     "text", COLUMN_PRINTNAME,
						     "editable", COLUMN_NAME_EDITABLE,
						     NULL);
	g_object_set_data (G_OBJECT (renderer), "column", GINT_TO_POINTER (COLUMN_PRINTNAME));
	g_signal_connect (G_OBJECT (renderer), "edited",
			  G_CALLBACK (cell_edited_cb), qef);


	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						     -1, _("Value"), renderer,
						     "text", COLUMN_VALUE,
						     NULL);

	renderer = gtk_cell_renderer_toggle_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						     -1, _("Grouping"), renderer,
						     "active", COLUMN_GROUPING,
						     "activatable", COLUMN_NAME_EDITABLE,
						     NULL);


	/* Action area */
	bb = gtk_hbutton_box_new ();
	gtk_container_set_border_width (GTK_CONTAINER (bb), 0);
	gtk_box_pack_start (GTK_BOX (qef), bb, FALSE, TRUE, GNOME_PAD/2.);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bb), GTK_BUTTONBOX_START);
	gtk_box_set_spacing (GTK_BOX (bb), GNOME_PAD);

	button = gtk_button_new (); 
	align = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);
	gtk_container_add (GTK_CONTAINER (button), align);
	hb = gtk_hbox_new (FALSE, GNOME_PAD/2.);
	gtk_container_add (GTK_CONTAINER (align), hb);
	wid = gtk_image_new_from_stock (GTK_STOCK_JUMP_TO, GTK_ICON_SIZE_BUTTON);
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	wid = gtk_label_new (_("Options"));
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (button), "clicked",
			    G_CALLBACK (obj_mitem_display), qef);
	gtk_container_add (GTK_CONTAINER (bb), button);

	button = gtk_button_new_from_stock (GTK_STOCK_ADD);
	gtk_container_add (GTK_CONTAINER (bb), button);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (add_expression_cb), qef);
	
	button = gtk_button_new_from_stock (GTK_STOCK_PROPERTIES);
	gtk_container_add (GTK_CONTAINER (bb), button);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (edit_expression_cb), qef);
	qef->priv->edit_btn = button;
	gtk_widget_set_sensitive (button, FALSE);

	button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
	gtk_container_add (GTK_CONTAINER (bb), button);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (del_expression_cb), qef);
	qef->priv->del_btn = button;
	gtk_widget_set_sensitive (button, FALSE);

	/* Show everything but the widget itself */
	gtk_widget_show_all (bb);

	/* Initial settings display */
	query_editor_fields_update_model (qef);	
	query_type_changed_cb (qef->priv->query, qef);
}

static void 
cell_toggled_cb (GtkCellRendererToggle *renderer, gchar *path, QueryEditorFields * qef)
{
	GtkTreeModel *model;
	GtkTreePath *tpath = gtk_tree_path_new_from_string (path);
	GtkTreeIter iter;
	gint *column;
	QueryField *qf;

	g_return_if_fail (qef && IS_QUERY_EDITOR_FIELDS (qef));
	
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (qef->priv->treeview));
	column = g_object_get_data (G_OBJECT (renderer), "column");
	gtk_tree_model_get_iter (model, &iter, tpath);
	gtk_tree_path_free (tpath);
	gtk_tree_model_get (model, &iter, COLUMN_QUERY_FIELD, &qf, -1);

	switch (GPOINTER_TO_INT (column)) {
	case COLUMN_PRINT:
		query_field_set_is_printed (qf, !gtk_cell_renderer_toggle_get_active (renderer));
		store_qf_in_model (GTK_LIST_STORE (model), &iter, qf);
		/* if (!query_field_to_be_displayed (qf, qef))  */
/* 			gtk_list_store_remove (GTK_LIST_STORE (model), &iter); */
		break;
	case COLUMN_GROUPING:
	default:
		break;
	}
}

static void 
cell_edited_cb (GtkCellRendererToggle *renderer, gchar *path, gchar *new_str, QueryEditorFields * qef)
{
	GtkTreeModel *model;
	GtkTreePath *tpath = gtk_tree_path_new_from_string (path);
	GtkTreeIter iter;
	gint *column;
	QueryField *qf;

	g_return_if_fail (qef && IS_QUERY_EDITOR_FIELDS (qef));

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (qef->priv->treeview));
	column = g_object_get_data (G_OBJECT (renderer), "column");
	gtk_tree_model_get_iter (model, &iter, tpath);
	gtk_tree_path_free (tpath);
	gtk_tree_model_get (model, &iter, COLUMN_QUERY_FIELD, &qf, -1);

	switch (GPOINTER_TO_INT (column)) {
	case COLUMN_NAME:
		query_field_set_name (qf, new_str);
		store_qf_in_model (GTK_LIST_STORE (model), &iter, qf);
		/* if (!query_field_to_be_displayed (qf, qef))  */
/* 			gtk_list_store_remove (GTK_LIST_STORE (model), &iter); */
		break;
	case COLUMN_PRINTNAME:
		query_field_set_alias (qf, new_str);
		store_qf_in_model (GTK_LIST_STORE (model), &iter, qf);
		/* if (!query_field_to_be_displayed (qf, qef))  */
/* 			gtk_list_store_remove (GTK_LIST_STORE (model), &iter); */
		break;
	default:
		break;
	}
}

static void obj_mitem_toggled_cb (GtkCheckMenuItem *mitem, QueryEditorFields * qef);
static void 
obj_mitem_display (GtkButton *button, QueryEditorFields * qef)
{
	GtkWidget *menu, *mitem;

	menu = gtk_menu_new ();

	mitem = gtk_check_menu_item_new_with_label (_("Query expressions"));
	g_object_set_data (G_OBJECT (mitem), "show_type", GINT_TO_POINTER (QEF_EXPR)); 
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), qef->priv->show_type & QEF_EXPR);
	g_signal_connect (G_OBJECT (mitem), "toggled",
			    G_CALLBACK (obj_mitem_toggled_cb), qef);
	gtk_menu_append (GTK_MENU (menu), mitem);
	gtk_widget_show (mitem);

	mitem = gtk_check_menu_item_new_with_label (_("Query conditions"));
	g_object_set_data (G_OBJECT (mitem), "show_type", GINT_TO_POINTER (QEF_COND)); 
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), qef->priv->show_type & QEF_COND);
	g_signal_connect (G_OBJECT (mitem), "toggled",
			    G_CALLBACK (obj_mitem_toggled_cb), qef);
	gtk_menu_append (GTK_MENU (menu), mitem);
	gtk_widget_show (mitem);

	mitem = gtk_check_menu_item_new_with_label (_("Parameters"));
	g_object_set_data (G_OBJECT (mitem), "show_type", GINT_TO_POINTER (QEF_PARAM)); 
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), qef->priv->show_type & QEF_PARAM);
	g_signal_connect (G_OBJECT (mitem), "toggled",
			    G_CALLBACK (obj_mitem_toggled_cb), qef);
	gtk_menu_append (GTK_MENU (menu), mitem);
	gtk_widget_show (mitem);

	mitem = gtk_check_menu_item_new_with_label (_("Unnamed expressions"));
	g_object_set_data (G_OBJECT (mitem), "show_type", GINT_TO_POINTER (QEF_NOTNAMED)); 
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), qef->priv->show_type & QEF_NOTNAMED);
	g_signal_connect (G_OBJECT (mitem), "toggled",
			    G_CALLBACK (obj_mitem_toggled_cb), qef);
	gtk_menu_append (GTK_MENU (menu), mitem);
	gtk_widget_show (mitem);

	gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
			NULL, NULL, 0, 0);
}


static void
query_editor_fields_update_model (QueryEditorFields *qef)
{
	GtkListStore *model;
	GSList *list;

	model = GTK_LIST_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (qef->priv->treeview)));
	g_assert (model);
	gtk_list_store_clear (GTK_LIST_STORE (model));
	
	list = qef->priv->query->fields;
	while (list) {
		QueryField *qf = QUERY_FIELD (list->data);

		if (query_field_to_be_displayed (qf, qef)) {
			GtkTreeIter iter;
			gtk_list_store_append (model, &iter);

			store_qf_in_model (model, &iter, qf);
		}

		list = g_slist_next (list);
	}
}

static gboolean
query_field_to_be_displayed (QueryField *qf, QueryEditorFields *qef)
{
	gboolean to_display = FALSE;


	if ((qef->priv->show_type & QEF_EXPR) &&
	    (query_field_get_name (qf) || query_field_get_is_printed (qf)))
		to_display = TRUE;

	if ((qef->priv->show_type & QEF_COND));
	/* FIXME */

	if ((qef->priv->show_type & QEF_PARAM) &&
	    (query_field_get_ftype (qf) == QUERY_FIELD_VALUE) && query_field_value_is_parameter (qf))
		to_display = TRUE;

	if ((qef->priv->show_type & QEF_NOTNAMED) &&
	    (!query_field_get_name (qf) || (!query_field_get_is_printed (qf))))
		to_display = TRUE;

	return to_display;
}



/* Changes what objects are being displayed */
static void
obj_mitem_toggled_cb (GtkCheckMenuItem *mitem, QueryEditorFields * qef)
{
	QueryEditorFieldsShowType type;

	type = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (mitem), "show_type"));
	if (mitem->active)
		qef->priv->show_type = qef->priv->show_type | type;
	else
		qef->priv->show_type = qef->priv->show_type & (~type);

	query_editor_fields_update_model (qef);
}

static void 
obj_select_changed_cb (GtkTreeSelection *select, QueryEditorFields * qef)
{
	GtkTreeIter iter;
	GtkTreeModel *model;

	if (gtk_tree_selection_get_selected (select, &model, &iter)) {
		QueryField *qf;

		gtk_tree_model_get (model, &iter, COLUMN_QUERY_FIELD, &qf, -1);
		qef->priv->selection = qf;
		gtk_widget_set_sensitive (qef->priv->edit_btn, TRUE);
		gtk_widget_set_sensitive (qef->priv->del_btn, TRUE);
	}
	else {
		qef->priv->selection = NULL;
		gtk_widget_set_sensitive (qef->priv->edit_btn, FALSE);
		gtk_widget_set_sensitive (qef->priv->del_btn, FALSE);
	}
}







/*
 * Addition of a QueryField 
 */

static void qee_status_changed_cb (QueryEditorExpr * qee, gboolean status, GtkDialog *dlg);
static void qee_dlg_destroy_cb    (GtkDialog *dialog, QueryEditorFields * qef);
static void qee_dlg_response_cb   (GtkDialog *dialog, gint btnno, QueryEditorFields * qef);
static void qee_destroy_cb        (QueryEditorExpr * qee, GtkDialog *dialog);
static void 
add_expression_cb (GtkWidget *button, QueryEditorFields * qef)
{
	GtkWidget *wid, *dlg;

	dlg = gtk_dialog_new_with_buttons (_("New expression"), NULL, 0, 
					   GTK_STOCK_OK, GTK_RESPONSE_OK,
					   GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
					   NULL);
	
	gtk_window_set_policy (GTK_WINDOW (dlg), TRUE, TRUE, FALSE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG (dlg), GTK_RESPONSE_OK, FALSE);

	wid = query_editor_expr_new (qef, NULL);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), wid, TRUE, TRUE, GNOME_PAD/2.);
	gtk_widget_show (wid);
	gtk_widget_show (dlg);

	g_signal_connect (G_OBJECT (wid), "status",
			  G_CALLBACK (qee_status_changed_cb), dlg);
	g_object_set_data (G_OBJECT (dlg), "qee", wid);

	g_signal_connect (G_OBJECT (dlg), "response",
			  G_CALLBACK (qee_dlg_response_cb), qef);
	g_signal_connect (G_OBJECT (dlg), "destroy",
			  G_CALLBACK (qee_dlg_destroy_cb), qef);

	g_signal_connect (G_OBJECT (wid), "destroy",
			  G_CALLBACK (qee_destroy_cb), dlg);
}

static void
qee_status_changed_cb (QueryEditorExpr * qee, gboolean status, GtkDialog *dlg)
{
	gtk_dialog_set_response_sensitive (GTK_DIALOG (dlg), GTK_RESPONSE_OK, status);
}

static void
qee_dlg_destroy_cb (GtkDialog *dialog, QueryEditorFields * qef)
{
	gpointer wid;

	wid = g_object_get_data (G_OBJECT (dialog), "qee");
	g_signal_handlers_disconnect_by_func (G_OBJECT (wid), 
					      G_CALLBACK (qee_status_changed_cb), dialog);
	g_object_set_data (G_OBJECT (dialog), "qee", NULL);
}

static void do_after_edition_merge_back (QueryEditorFields *qef, QueryEditorExpr * qee);
static void 
qee_dlg_response_cb    (GtkDialog *dialog, gint btnno, QueryEditorFields * qef)
{
	gpointer wid;

	switch (btnno) {
	case GTK_RESPONSE_OK:
		wid = g_object_get_data (G_OBJECT (dialog), "qee");
		do_after_edition_merge_back (qef, QUERY_EDITOR_EXPR (wid));
		gtk_widget_destroy (GTK_WIDGET (dialog));
		break;
	case GTK_RESPONSE_CANCEL:
		gtk_widget_destroy (GTK_WIDGET (dialog));
		break;
	}
}

static GSList *list_used_query_fields_recursive (GSList *fields, QueryField *qf, GSList *list);
static QueryField *find_similar_query_field (Query *q, GHashTable *equals, GHashTable *diffs, 
					     QueryField *qf);
static void 
do_after_edition_merge_back (QueryEditorFields *qef, QueryEditorExpr * qee)
{
	GSList *used_list, *list, *unused_list;
	GHashTable *equals, *diffs;
	gboolean orig_qf_replaced;

	QueryField *top_qf, *orig_qf;
	GSList *fields;

	gchar *top_name = NULL, *top_alias = NULL;

	g_return_if_fail (qee->top_qf);

	/* We now take ownership of the list of QueryField objects which were in the
	 * QueryEditorExpr object (=> we now have the reference on these objects)
	 */
	top_qf = qee->top_qf;
	orig_qf = qee->orig_qf;
	fields = query_editor_expr_fetch_fields (qee);

	/* Step 1, action 1: 
	 * fetch the list of QueryFields which are referenced by the
	 * top QueryField being edited 
	 */
	used_list = list_used_query_fields_recursive (fields, top_qf, NULL);
	
	/* Step 1, action 2: 
	 * remove QueryFields present in the 'fields' list and not in the
	 * list we just made ('used_list')
	 */

	/* From now on, each QF will have to look into the query's list of fields
	 * to activate itself if necessary, not in another list provided
	 * in the "qf_list" attribute */
	list = fields;
	while (list) {
		g_object_set_data (G_OBJECT (list->data), "qf_list", NULL);
		list = g_slist_next (list);	
	}

	/* remove unused QFs */
	list = fields;
	unused_list = NULL;
	while (list) {
		if (!g_slist_find (used_list, list->data)) {
			GSList *next;

			unused_list = g_slist_prepend (unused_list, list->data);

			g_object_unref (G_OBJECT (list->data));

			next = g_slist_next (list);
			fields = g_slist_delete_link (fields, list);
			list = next;
		}
		else
			list = g_slist_next (list);
	}
#ifdef debug
	g_print ("\n*Merging step 1: %d QF used and %d QF unused, now remaining %d\n", 
		 g_slist_length (used_list), g_slist_length (unused_list), g_slist_length (fields));
	g_print ("*----- USED QF -----\n");
	list = used_list;
	while (list) {
		g_print ("*\t QF %p\n", list->data);
		list = g_slist_next (list);
	}
	g_print ("*----- NON USED QF -----\n");
	list = unused_list;
	while (list) {
		g_print ("*\t QF %p\n", list->data);
		list = g_slist_next (list);
	}
#endif
	if (g_slist_length (used_list) != g_slist_length (fields))
		g_warning ("used_list != total_list!\n");
	g_slist_free (unused_list);
	g_slist_free (used_list);
	unused_list = NULL;
	used_list = NULL;


	/* Step 2:
	 * for each QF in the remaining list, try to find the corresponding one in
	 * the query->fields list (and when found, update the 'equals' hash table)
	 */

	/* Build the 'equals' and 'diffs' hash tables.
	 * --> The "equals" contains the QF for which we have found an equivalent QF in the Query
	 *     the QF in the hash will be replaced by the original ones in the Query.
	 * --> The "diffs" contains the QF for which there is no equivalent in the Query
	 *     the QF in this hash will be inserted in the Query.
	 * This does not take into account the printed, name and alias parameters.
	 *
	 * Equals and diffs hash tables MUST be disjoint and their union MUST make the whole fields list
	 */
	equals = g_hash_table_new (NULL, NULL);
	diffs = g_hash_table_new (NULL, NULL);

	list = fields;
	while (list) {
		find_similar_query_field (qee->qef->priv->query, equals, diffs, QUERY_FIELD (list->data));
		list = g_slist_next (list);
	}
#ifdef debug
	g_print ("\n*Hash tables sizes: Equals=%d, Diffs=%d, length(fields)=%d\n", 
		 g_hash_table_size (equals), 
		 g_hash_table_size (diffs), g_slist_length (fields));
#endif	

	if (g_hash_table_size (equals) + g_hash_table_size (diffs) != g_slist_length (fields))
		g_warning ("equals + differents != total_list!\n");


	/* Replacing the QFs referenced by a QF in the 'fields' list by its similar one
	 * in the q->fields list */
	list = fields;
	while (list) {
		GSList *olist, *iter, *ofields;
		gpointer data;

		ofields = fields;
		olist = query_field_get_monitored_objects (QUERY_FIELD (list->data));
		iter = olist;
		while (iter) {
			if (IS_QUERY_FIELD (iter->data)) {
				if ((data = g_hash_table_lookup (equals, iter->data))) {
					query_field_replace_ref_ptr (QUERY_FIELD (list->data),
								     G_OBJECT (iter->data),
								     G_OBJECT (data));
				}
			}
			iter = g_slist_next (iter);
		}
		g_slist_free (olist);

		/* the fields list may have changed because of the QF's replacement */
		if (ofields == fields)
			list = g_slist_next (list);
		else
			list = fields;
	}



	/* Step 3:
	 * Destroying the QueryFields in the equals hash, moving the other ones into
	 * the query itself, and resolving any naming issues that might have occured
	 */

	/* 3.1 */
	orig_qf_replaced = TRUE;
	list = fields;
	while (list) {
		if (g_hash_table_lookup (diffs, list->data)) {
			/* addition to the query */
			if (list->data == top_qf) {
				if (query_field_get_name (top_qf))
					top_name = g_strdup (query_field_get_name (top_qf));
				if (query_field_get_alias (top_qf))
					top_alias = g_strdup (query_field_get_alias (top_qf));

				if (orig_qf && g_slist_find (qef->priv->query->fields, orig_qf)) 
					query_add_field_before (qee->qef->priv->query, QUERY_FIELD (list->data),
								orig_qf);
				else
					query_add_field (qee->qef->priv->query, QUERY_FIELD (list->data));
			}
			else
				query_add_field (qee->qef->priv->query, QUERY_FIELD (list->data));

			/* lose the reference on this QueryField */
			g_object_unref (G_OBJECT (list->data));

			/* The name and alias may be changed during the insertion, but will be set
			   to the right value at the end of step 3.2 */
		}
		else {
			gpointer data;

			data = g_hash_table_lookup (equals, list->data);
			g_assert (data); 

			/* is it orig_qf ? */
			if (orig_qf && (QUERY_FIELD (data) == orig_qf))
				orig_qf_replaced = FALSE;

			/* replacing the name if necessary */
			if (query_field_get_name (QUERY_FIELD (list->data)))
				query_field_set_name (QUERY_FIELD (data), 
						      query_field_get_name (QUERY_FIELD (list->data)));

			/* replacing the alias if necessary */
			if (query_field_get_alias (QUERY_FIELD (list->data)))
				query_field_set_alias (QUERY_FIELD (data), 
						       query_field_get_alias (QUERY_FIELD (list->data)));

			/* setting the 'is_printed' attribute if necessary */
			query_field_set_is_printed (QUERY_FIELD (data), 
						    query_field_get_is_printed (QUERY_FIELD (list->data)));
		}

		list = g_slist_next (list);
	}
	
	/* 3.2 
	 * Each QueryField in the Query which referenced the edited QueryField MUST be told to 
	 * update its reference to the new QueryField (which has replaced the edited one),
	 * (if we can still find the original QF). */
	if (!g_slist_find (fields, top_qf)) 
		orig_qf_replaced = FALSE;

	if (orig_qf_replaced && orig_qf) {
		/* Replace any reference to orig_qf to top_qf */
		list = qee->qef->priv->query->fields;
		while (list) {
			GSList *olist;

			olist = query_field_get_monitored_objects (QUERY_FIELD (list->data));
			while (olist) {
				if (IS_QUERY_FIELD (olist->data) &&
				    (QUERY_FIELD (olist->data) == orig_qf))
					query_field_replace_ref_ptr (QUERY_FIELD (list->data), 
								     G_OBJECT (olist->data),
								     G_OBJECT (top_qf));
				olist = g_slist_next (olist);
			}			
			list = g_slist_next (list);
		}

		/* Remove the orig_qf QueryField */
		query_del_field (qee->qef->priv->query, orig_qf);

		/* Set back the right name and alias */
		if (top_name) {
			query_field_set_name (top_qf, top_name);
			g_free (top_name);
			top_name = NULL;
		}
		if (top_alias) {
			query_field_set_alias (top_qf, top_alias);
			g_free (top_alias);
			top_alias = NULL;
		}
	}

	/* Last step:
	   Freeing the fields list and the hash tables in the end 
	*/
	list = fields;
	while (list) {
		if (g_hash_table_lookup (equals, list->data)) 
			g_object_unref (G_OBJECT (list->data));

		list = g_slist_next (list);
	}
	
	g_slist_free (fields);

	g_hash_table_destroy (equals);
	g_hash_table_destroy (diffs);

	if (top_name)
		g_free (top_name);
	if (top_alias)
		g_free (top_alias);
}


/*
 * Build a list of QF used by the "qf" argument. The "list" argument is the list
 * which is actually "augmented" during the process, and the "fields" list is
 * the list where all the query fields are.
 */
static GSList *
list_used_query_fields_recursive (GSList *fields, QueryField *qf, GSList *list)
{
	GSList *retlist = list, *olist, *iter;

	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), list);

	/* This Query Field ? */
	if (g_slist_find (fields, qf)) {
		if (!g_slist_find (retlist, qf))
			retlist = g_slist_append (retlist, qf);
	}
	else
		g_warning ("Can't find referenced QueryField in QueryEditorFields's fields list!\n");

	olist = query_field_get_monitored_objects (qf);
	iter = olist;
	while (iter) {
		if (IS_QUERY_FIELD (iter->data)) {
			retlist = list_used_query_fields_recursive (fields, 
								    QUERY_FIELD (iter->data), retlist);
		}
		iter = g_slist_next (iter);
	}
	g_slist_free (olist);

	return retlist;
}


/* 
 * Finds a similar QueryField in the fields of the Query passed as argument
 * and update the 'equals' and 'diffs' hash tables
 * ('qf' does not need to be in the Query's fields list) 
 */
static QueryField *
find_similar_query_field (Query *q, GHashTable *equals, GHashTable *diffs, QueryField *qf)
{
	QueryField *similar = NULL;
	GSList *list;
	gpointer value;

	/* If we have already done the research, then it is easy */
	if ((value = g_hash_table_lookup (equals, qf)))
		return QUERY_FIELD (value);
	if (g_hash_table_lookup (diffs, qf))
		return NULL;

	/* Real search is necessary */
	list = q->fields;
	while (list && !similar) {
		if (query_field_is_equal (qf, QUERY_FIELD (list->data))) {
			QueryField *simqf = QUERY_FIELD (list->data);
			GSList *olist, *simolist, *iter, *simiter;
			gboolean subs_all_similars = TRUE;

			/* Make sure any monitored QF has a similar QF in the query */
			olist = query_field_get_monitored_objects (qf);
			simolist = query_field_get_monitored_objects (simqf);

			if (g_slist_length (olist) == g_slist_length (simolist)) {
				iter = olist;
				simiter = simolist;
				
				while (iter && subs_all_similars) {
					if (IS_QUERY_FIELD (iter->data) && IS_QUERY_FIELD (simiter->data)) {
						QueryField *sqf;
						
						if ((sqf = find_similar_query_field (q, equals, diffs,
										     QUERY_FIELD (iter->data)))) {
							if (sqf != QUERY_FIELD (simiter->data))
								subs_all_similars = FALSE;
						}
						else
							subs_all_similars = FALSE;	
					}
					else {
						if ((IS_QUERY_FIELD (iter->data) &&
						     !IS_QUERY_FIELD (simiter->data)) ||
						    (!IS_QUERY_FIELD (iter->data) &&
						     IS_QUERY_FIELD (simiter->data)))
							subs_all_similars = FALSE;
					}
					
					iter = g_slist_next (iter);
					simiter = g_slist_next (simiter);
				}
			}
			else
				subs_all_similars = FALSE;

			g_slist_free (olist);
			g_slist_free (simolist);


			if (subs_all_similars) {
				QueryField *oqf = g_object_get_data (G_OBJECT (qf), "orig_qf");
				if (oqf) {
					/* this QF is made from an alias => more tests:
					   if the name has been modified, then we consider we have a
					   new QF. */
					
					similar = QUERY_FIELD (list->data); /* FIXME */
				}
				else
					similar = QUERY_FIELD (list->data); 
			}
		}
		list = g_slist_next (list);
	}

	if (!similar) {
		g_hash_table_insert (diffs, qf, qf);
		g_print ("No Similar for QF: %p\n", qf);
	}
	else {
		g_hash_table_insert (equals, qf, similar);
		g_print ("Similar QF: %p <=> %p\n", qf, similar);
	}

	return similar;
}

static void
qee_destroy_cb        (QueryEditorExpr * qee, GtkDialog *dialog)
{
	/* close the dialog */
	gtk_widget_destroy (GTK_WIDGET (dialog));
}











/*
 * Edition of a QueryField 
 */

static void 
edit_expression_cb (GtkWidget *button, QueryEditorFields * qef)
{
	if (qef->priv->selection) {
		GtkWidget *wid, *dlg;
		dlg = gtk_dialog_new_with_buttons (_("Expression's edition"), NULL, 0, 
						   GTK_STOCK_OK, GTK_RESPONSE_OK,
						   GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
						   NULL);

		gtk_window_set_policy (GTK_WINDOW (dlg), TRUE, TRUE, FALSE);
		gtk_dialog_set_response_sensitive (GTK_DIALOG (dlg), GTK_RESPONSE_OK, 
						   query_field_get_activated (qef->priv->selection));		

		wid = query_editor_expr_new (qef, qef->priv->selection);
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), wid, TRUE, TRUE, GNOME_PAD/2.);
		gtk_widget_show (wid);
		gtk_widget_show (dlg);

		g_signal_connect (G_OBJECT (wid), "status",
				  G_CALLBACK (qee_status_changed_cb), dlg);
		g_object_set_data (G_OBJECT (dlg), "qee", wid);

		g_signal_connect (G_OBJECT (dlg), "response",
				  G_CALLBACK (qee_dlg_response_cb), qef);
		g_signal_connect (G_OBJECT (dlg), "destroy",
				  G_CALLBACK (qee_dlg_destroy_cb), qef);
		g_signal_connect (G_OBJECT (wid), "destroy",
				  G_CALLBACK (qee_destroy_cb), dlg);
	}
}


static void 
del_expression_cb (GtkWidget *button, QueryEditorFields * qef)
{
	if (qef->priv->selection) 
		query_del_field (qef->priv->query, qef->priv->selection);
}
