/*
 *
 * mtm-ext-handler :
 *
 * Copyright (C) 2000-2001 Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Richard Hestilow <hestgray@ionet.net>
 *
 */

#include <mtm-ext-handler.h>
#include "internals.h"

/* FIXME: eep */
#include <gdk/gdktypes.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

static MtmHandlerClass *mtm_ext_handler_parent_class;

/* Destroy handler for MtmExtHandler */
static void
mtm_ext_handler_destroy (GtkObject *object)
{
	MtmExtHandler *handler = MTM_EXT_HANDLER (object);

	if (handler->subclass_args)
		g_free (handler->subclass_args);

	if (GTK_OBJECT_CLASS (mtm_ext_handler_parent_class)->destroy)
		(*GTK_OBJECT_CLASS(mtm_ext_handler_parent_class)->destroy)(object);
}

/* Overridden MtmHandler::register. Registers the ExtHandler with the handler
 * manager. */
static void
mtm_ext_handler_register (MtmHandler *handler)
{
	handler_manager_add (MTM_STATEFUL (handler)->env->handler_manager,
		MTM_EXT_HANDLER_ID, handler);
}

/* Class initialization function for MtmExtHandler. */
static void
mtm_ext_handler_class_init (GtkObjectClass *object_class)
{
	mtm_ext_handler_parent_class = gtk_type_class (mtm_handler_get_type ());
	object_class->destroy = mtm_ext_handler_destroy;
	MTM_HANDLER_CLASS (object_class)->_register = mtm_ext_handler_register;
}

/* Object initialization function for MtmExtHandler. */
static void
mtm_ext_handler_init (GtkObject *object)
{
	MtmHandler *handler = MTM_HANDLER (object);
	handler->cleanup_handler = mtm_ext_handler_default_cleanup;
	MTM_EXT_HANDLER (handler)->save = mtm_ext_handler_default_save;
}

/**
 * mtm_ext_handler_get_type:
 * @void:
 *
 * Registers the #MtmExtHandler class if necessary, and returns the type ID
 * associated to it.
 * 
 * Return value: The type ID of the #MtmExtHandler class.
 */
GtkType
mtm_ext_handler_get_type (void)
{
	static GtkType type = 0;

	if (!type)
	{
		GtkTypeInfo info =
		{
			"MtmExtHandler",
			sizeof (MtmExtHandler),
			sizeof (MtmExtHandlerClass),
			(GtkClassInitFunc) mtm_ext_handler_class_init,
			(GtkObjectInitFunc) mtm_ext_handler_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (mtm_handler_get_type (), &info);
	}

	return type;
}

/**
 * mtm_ext_handler_new:
 * @env: The environment object.
 *
 * Creates a new extension handler.
 *
 * Return value: A newly-created extension handler.
 */
MtmExtHandler *
mtm_ext_handler_new (MtmEnv *env)
{
	MtmExtHandler *handler;

	g_return_val_if_fail (env != NULL, NULL);
	
	handler = gtk_type_new (mtm_ext_handler_get_type ());
	
	mtm_ext_handler_construct (handler, env);
	
	return handler;
}

/**
 * mtm_ext_handler_construct:
 * @handler: The uninitialized extension handler.
 * @env: The environment object.
 * 
 * Intializes the provided handler.
 */
void
mtm_ext_handler_construct (MtmExtHandler *handler, MtmEnv *env)
{
	g_return_if_fail (handler != NULL);
	g_return_if_fail (env != NULL);
	
	MTM_STATEFUL (handler)->env = env;
}

/**
 * mtm_ext_handler_default_cleanup:
 * @handler: The handler.
 *
 * This is what the "cleanup_handler" member method of all extension handlers
 * initially points to.
 * Frees editcmd and default_suffix if they exist, then calls
 * mtm_handler_default_cleanup.
 *
 * Return value: 1 if successful, -1 otherwise.
 */
int
mtm_ext_handler_default_cleanup (MtmHandler *handler)
{
	MtmExtHandler *eh;
	g_return_val_if_fail (handler != NULL, -1);

	eh = MTM_EXT_HANDLER (handler);
	if (eh->editcmd)
		g_free (eh->editcmd);
	if (eh->default_suffix)
		g_free (eh->default_suffix);

	return mtm_handler_default_cleanup (handler);
}

/**
 * mtm_env_get_ext_handler:
 * @env: The environment object.
 * @key: The key of the requested extension handler (corresponds to
 * MtmExt::type).
 *
 * Return value: The extension handler if found, NULL otherwise.
 */
MtmExtHandler*
mtm_env_get_ext_handler (MtmEnv *env, gchar *key)
{
	g_return_val_if_fail (env != NULL, NULL);
	g_return_val_if_fail (key != NULL, NULL);
	
	return MTM_EXT_HANDLER (handler_manager_get (env->handler_manager,
		MTM_EXT_HANDLER_ID, key)); 
}

/**
 * mtm_env_get_all_ext_handlers:
 * @env: The environment object.
 *
 * Return value: A pointer to the environment's list of known extension
 * handlers.
 **/
GList*
mtm_env_get_all_ext_handlers (MtmEnv *env)
{
	g_return_val_if_fail (env != NULL, NULL);

	return handler_manager_get_all (env->handler_manager,
		MTM_EXT_HANDLER_ID);
}

/**
 * mtm_ext_handler_check_args:
 * @handler: The extension handler.
 * 
 * Fills in the appropriate subclass argument information for the given
 * extension handler. This must be called before attempting to access
 * MtmExtHandler::subclass_args, or MtmExtHandler::n_subclass_args.
 */
void
mtm_ext_handler_check_args (MtmExtHandler *handler)
{
	GArray *arr;
	GtkArg *tmp;
	GtkType type;
	guint n, i, len = 0;
	
	g_return_if_fail (handler != NULL);
	
	if (handler->subclass_args)
		return;
	
	type = GTK_OBJECT_TYPE (handler);
	arr = g_array_new (FALSE, TRUE, sizeof (GtkArg));

	while (type != MTM_EXT_HANDLER_TYPE && type != GTK_TYPE_INVALID)
	{
		tmp = gtk_object_query_args (type, NULL, &n);
		for (i = 0; i < n; i++)
		{
			if (strcmp (mtm_ext_handler_get_arg_name (handler, tmp[i].name), "ext_context") == 0)
			{
				continue;
			}
			else
			{
				arr = g_array_append_val (arr, tmp[i]);
				len++;
			}
				
		}

		g_free (tmp);
		
		type = gtk_type_parent (type);
	}
	
	if (!len)
		return;

	handler->subclass_args = (GtkArg*) arr->data;
	handler->n_subclass_args = len; 
	g_array_free (arr, FALSE);
}

/**
 * mtm_ext_handler_get_arg_name:
 * @handler: The extension handler.
 * @fullname: The fully qualified argument name.
 *
 * For a string of the form "Class::Arg", returns "Arg".
 * Used for subclassed extension handlers.
 *
 * Return value: A pointer to the beginning of the argument name.
 */
gchar *
mtm_ext_handler_get_arg_name (MtmExtHandler *handler, gchar *fullname)
{
	int len;
	int i;

	g_return_val_if_fail (handler != NULL, NULL);
	g_return_val_if_fail (fullname != NULL, NULL);

	len = strlen (fullname);
	if (len <= 2)
		return NULL;

	for (i = len - 2; i >= 0; i--)
	{
		if (fullname[i] == ':')
			return fullname + i + 1;
	}

	return NULL;
}

/**
 * mtm_ext_handler_print_arg:
 * @handler: The extension handler.
 * @file: A pointer to a file stream.
 * @arg: A Gtk+ argument.
 *
 * Prints the given argument using the stream provided.
 * Used for subclassed extension handlers.
 */
void
mtm_ext_handler_print_arg (MtmExtHandler *handler, FILE *file,
					  GtkArg *arg)
{
	MtmExtHandlerClass *class;

	g_return_if_fail (handler != NULL); 
	g_return_if_fail (file != NULL); 
	g_return_if_fail (arg != NULL);

	class = MTM_EXT_HANDLER_CLASS (GTK_OBJECT (handler)->klass);

	fprintf (file, "type=\"%s\" value=\"", gtk_type_name (arg->type));
	/* FIXME: Add more types */
	if (arg->type == GTK_TYPE_STRING)
		fprintf (file, GTK_VALUE_STRING (*arg));
	else if (arg->type == GTK_TYPE_BOOL)
		fprintf (file, "%i", GTK_VALUE_BOOL (*arg));
	else if (arg->type == GTK_TYPE_GDK_COLOR)
	{
		GdkColor *color = GTK_VALUE_BOXED (*arg);
		if (!color)
			fprintf (file, "#005060");
		else
			fprintf (file, "#%02x%02x%02x",
				color->red >> 8,
				color->green >> 8,
				color->blue >> 8);
	}
	else /* We assume it's an enum */
		fprintf (file, "%i", GTK_VALUE_ENUM (*arg));

	fprintf (file, "\"");
}

MtmResult
mtm_ext_handler_default_save (MtmExtHandler *handler, MtmExt *ext, const gchar *dirname)
{
	gchar *file;
	struct stat buf;
	
	g_return_val_if_fail (MTM_IS_EXT_HANDLER (handler), MTM_GENERAL_ERROR);
	g_return_val_if_fail (MTM_IS_EXT (ext), MTM_GENERAL_ERROR);
	g_return_val_if_fail (ext->file != NULL, MTM_GENERAL_ERROR);
	g_return_val_if_fail (dirname != NULL, MTM_GENERAL_ERROR);

	if (mtm_check_dir (dirname) != MTM_OK)
		return MTM_NO_ACCESS;
	
	if (!strcmp (g_basename (ext->file), ext->file) && handler->find)
	{
		file = handler->find (handler, ext->file, FALSE);
		if (!file)
			file = handler->find (handler, ext->file, TRUE);
		if (!file)
			return MTM_NO_ACCESS;
	}
	else
		file = g_strdup (ext->file);
	
	if (stat (ext->file, &buf) != 0)
	{
		g_free (file);
		return MTM_NO_ACCESS;
	}
	
	if (mtm_file_is_targz (file))
	{
		gchar *tmp = mtm_file_untargz (file, dirname, ".tar.gz");
		g_free (tmp);
	}
	else 
	{
		gchar *tmp = g_strconcat (dirname, "/", g_basename (ext->file), NULL);

		if S_ISDIR (buf.st_mode)
			mtm_copy_directory (ext->file, tmp);
		else
			mtm_copy_file (ext->file, tmp);

		g_free (tmp);
	}

	g_free (file);

	return MTM_OK;
}
