/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */
/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *Copyright 2001-2003 Dodji SEKETELI, Gal CHAMOULAUD.
 *http://www.freespiders.org
 */

/**
 *@file
 *This widget is the visual tree editor.
 *It can load an xml tree, 
 *visualize it as a tree and allow 
 *the user to edit the tree.
 *Each time one of the nodes of the 
 *tree is selected, cut, copied etc, 
 *this widget emits signals
 *that can be caught and handler by the caller. 
 *See mlview-tree-editor.h for the declaration of the
 *pulic services offered by this widget.
 *
 */

#define NEW 1
#ifdef NEW

#include <string.h>
#include <stdlib.h>
#include <glade/glade.h>
#include "mlview-tree-editor2.h"
#include "mlview-node-type-picker.h"
#include "mlview-global-settings.h"
#include "mlview-marshal.h"
#include "mlview-utils.h"

#define SEARCH_IN_NAMES_CHECK_BUTTON "search-in-names-check-box"
#define SEARCH_IN_ATTRIBUTES_NAMES_CHECK_BUTTON "search-in-attributes-names-check-box"
#define SEARCH_IN_ATTRIBUTES_VALUES_CHECK_BUTTON "search-in-attributes-values-check-box"
#define SEARCH_IN_CONTENT_CHECK_BUTTON "search-in-content-check-box"
#define SEARCHED_TEXT_ENTRY "searched-text-entry"

#define IS_THE_FIND_DIALOG "is-the-find-dialog"

/*================================================================
 *The struct defining the private attributes of 
 *MlViewTreeEditor and it associated macros
 *=================================================================*/

#define PRIVATE(mlview_tree_editor2) ((mlview_tree_editor2)->priv)


struct _MlViewTreeEditor2Private {
        /*the xml document being edited. FIXME => handle it lifetime !!! */
        xmlDocPtr xml_doc;

        MlViewXMLDocument *mlview_xml_doc;

        /*the viewable tree that matches xml_doc */
        GtkTreeView *tree_view;
        /*
         *a cache that contains
         *a list of xmlNode/GtkTreeRowReference
         *pair. This is to speed up searches.
         */
        GHashTable *nodes_rows_hash;

        /*
         *the menu poped up when the user 
         *right clicks on an xml_node
         */
        GtkWidget *xml_node_popup_menu;

        /*a reference to the first selected row */
        GtkTreeRowReference *cur_sel_start;

        /*a reference to the last selected row... 
           not supported atm. */
        /*GtkTreeRowReference * cur_sel_end */

        /*the node type picker used when creating a new node. 
         *MlViewTreeEditor does not have to handle it lifetime
         */
        MlViewNodeTypePicker *node_type_picker;
        MlViewAppContext *app_context;
        /*the search dialog*/
        GtkDialog *search_dialog ;
        gboolean dispose_has_run ;
        gpointer backup_drag_data_delete ;
        gpointer backup_drag_data_received ;        
};

/**
 *The different signals emited
 *by #MlViewTreeEditor2.
 */
enum {
        TREE_CHANGED = 1,
        MARK_SET_TO_NODE,
        MARK_REMOVED_FROM_NODE,
        NODE_CUT,
        NODE_PASTED,
        NODE_ADDED,
        NODE_SELECTED,
        NODE_UNSELECTED,
        NUMBER_OF_SIGNALS
};


/*static const gchar * p_tree_editor_titles[]=
{
	N_("the xml document"),
	N_("the xml document tags")
} ;
*/
enum MLVIEW_TREE_EDITOR_COLUMNS {
        /*hidden column, where the xml node is stored.*/
        XML_NODE_COLUMN = 0,
        /*
         *contains a boolean that says
         *if the column is editable or not.
         */
        IS_EDITABLE_COLUMN,
        /*the first visible column*/
        FIRST_COLUMN, 
        /*the second visible column*/
        SECOND_COLUMN,
        /*
         *This must be the last element
         *of the enum.
         */
        NB_COLUMNS
};

enum MlViewTreeInsertType {
        INSERT_TYPE_ADD_CHILD,
        INSERT_TYPE_INSERT_BEFORE,
        INSERT_TYPE_INSERT_AFTER
};

#define MLVIEW_TREE_EDITOR2_NODE_CLIPBOARD_SIZE 128
static const gint TREE_EDITOR_NUMBER_OF_COLUMNS = 1;
static const gint TREE_EDITOR_TREE_COLUMN = 0;
static const gint TREE_EDITOR_SPACE_BETWEEN_PIXMAP_AND_TEXT = 2;
static const gint MLVIEW_CONTENT_MAX_LEN = 512;
static const gint MLVIEW_DEFAULT_EXPANSION_DEPTH = 2;
static guint gv_signals[NUMBER_OF_SIGNALS] = { 0 };
static GtkVBoxClass *gv_parent_class = NULL;
static GtkTargetEntry row_targets[] = {
        {(gchar *)"GTK_TREE_MODEL_ROW", GTK_TARGET_SAME_WIDGET, 0}
} ;
/****************************************************************
 *The edition popup menu definition array.
 *This menu is poped up on an xml visual node.
 *Note that this pop up menu must be dynamicaly build 
 *when this widget supports
 *Validation.
 *
 *FIXME: do not use this popup menu when validation is on.
 *Maybe build a dynamic menu using GtkMenu widgetware.
 ***************************************************************/


/********************************************
 *declaration of some of the private functions used *
 *and defined by this object.               *
 ********************************************/

static enum MlViewStatus
set_our_dnd_callbacks (MlViewTreeEditor2 *a_this) ;

static void
node_cell_edited_cb (GtkCellRendererText *a_renderer,
                     gchar *a_cell_path,
                     gchar *a_new_text,
                     gpointer a_data) ;
/***************************************************
 *private methods required by the GTK typing system
 ***************************************************/

/**
 *The dispose method is suppose to unref all the external
 *reference this object may hold, and free all the members
 *(if applicable) of this object.
 */
static void
mlview_tree_editor2_dispose (GObject *a_this)
{
        MlViewTreeEditor2 *ed=NULL ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)) ;
        ed = MLVIEW_TREE_EDITOR2 (a_this) ;
        g_return_if_fail (ed && PRIVATE (ed)) ;

        if (PRIVATE (ed)->dispose_has_run == TRUE) {
                return ;
        }
        if (PRIVATE (ed)->node_type_picker) {
                gtk_widget_destroy (GTK_WIDGET 
                                    (PRIVATE (ed)->node_type_picker));
                PRIVATE (ed)->node_type_picker = NULL ;
        }
        if (PRIVATE (ed)->search_dialog) {
                gtk_widget_destroy 
                        (GTK_WIDGET (PRIVATE (ed)->search_dialog)) ;
                PRIVATE (ed)->search_dialog = NULL ;
        }
        PRIVATE (ed)->dispose_has_run = TRUE ;
        if (gv_parent_class) {
                G_OBJECT_CLASS (gv_parent_class)->dispose (a_this) ;
        }
}

/**
 *Instane finalyzer, responsible of freeing the instance's memory.
 *Is called at the end of the object's destruction process.
 *@param a_this the current instance of #MlViewTreeEditor2
 */
static void
mlview_tree_editor2_finalize (GObject *a_this)
{
        MlViewTreeEditor2 *ed=NULL ;

        g_return_if_fail (a_this
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)) ;
        ed = MLVIEW_TREE_EDITOR2 (a_this) ;
        g_return_if_fail (ed && PRIVATE (ed)) ;
        g_free (PRIVATE (ed)) ;
        PRIVATE (ed) = NULL ;
        if (gv_parent_class) {
                G_OBJECT_CLASS (gv_parent_class)->finalize (a_this) ;
        }
}

/**
 *The Gtk standard class initialyzer of the 
 *MlViewTreeEditor2Class class. 
 *@param a_klass
 */
static void
mlview_tree_editor2_class_init (MlViewTreeEditor2Class * a_klass)
{
        GObjectClass *gobject_class = NULL ;

        g_return_if_fail (a_klass != NULL);

        gv_parent_class = g_type_class_peek_parent (a_klass);
        g_return_if_fail (gv_parent_class) ;
        gobject_class = G_OBJECT_CLASS (a_klass);
        g_return_if_fail (gobject_class);
        gobject_class->dispose = mlview_tree_editor2_dispose  ;
        gobject_class->finalize = mlview_tree_editor2_finalize ;
        /*object_class->destroy = NULL ;mlview_tree_editor2_destroy;*/

        gv_signals[TREE_CHANGED] =
                g_signal_new ("tree-changed",
                              G_TYPE_FROM_CLASS (gobject_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewTreeEditor2Class,
                               tree_changed), NULL, NULL,
                              mlview_marshal_VOID__VOID,
                              G_TYPE_NONE, 0, NULL);

        gv_signals[NODE_CUT] =
                g_signal_new ("node-cut",
                              G_TYPE_FROM_CLASS (gobject_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewTreeEditor2Class, node_cut),
                              NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);

        gv_signals[NODE_PASTED] =
                g_signal_new ("node-pasted",
                              G_TYPE_FROM_CLASS (gobject_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewTreeEditor2Class,
                               node_pasted), NULL, NULL,
                              gtk_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);

        gv_signals[NODE_ADDED] =
                g_signal_new ("node-added",
                              G_TYPE_FROM_CLASS (gobject_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewTreeEditor2Class,
                               node_added), NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);

        gv_signals[NODE_SELECTED] =
                g_signal_new ("node-selected",
                              G_TYPE_FROM_CLASS (gobject_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewTreeEditor2Class,
                               node_selected), NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);

        a_klass->tree_changed = NULL;
        a_klass->node_cut = NULL;
        a_klass->node_added = NULL;
        a_klass->node_pasted = NULL;
        a_klass->node_selected = NULL;
}

/**
 *The instance initialyzer of the MlViewTreeEditor2. 
 *
 */
static void
mlview_tree_editor2_init (MlViewTreeEditor2 * a_editor)
{
        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (PRIVATE (a_editor) == NULL);

        PRIVATE (a_editor) =
                g_try_malloc (sizeof (MlViewTreeEditor2Private));
        if (!PRIVATE (a_editor)) {
                mlview_utils_trace_info
                        ("malloc failed, system may be out of memory");
                return;
        }
        memset (PRIVATE (a_editor), 0,
                sizeof (MlViewTreeEditor2Private));
}

/*******************************
 *Private methods and callbacks.
 *******************************/
static void
nodeset_selected_cb (GtkTreeSelection * a_sel, gpointer * a_data)
{
        guint nb_row_selected = 0;
        GtkTreeView *tree_view = NULL;
        MlViewTreeEditor2 *tree_ed = NULL;
        GtkTreeRowReference *row_ref = NULL;
        GtkTreeModel *model = NULL;
        GList *row_sel = NULL;
        GtkTreeIter iter={0} ;
        gboolean is_ok=FALSE ;

        g_return_if_fail (a_sel
                          && GTK_IS_TREE_SELECTION (a_sel));
        g_return_if_fail (a_data
                          && MLVIEW_IS_TREE_EDITOR2 (a_data));

        tree_ed = MLVIEW_TREE_EDITOR2 (a_data);
        tree_view = gtk_tree_selection_get_tree_view (a_sel);
        g_return_if_fail (tree_view);
        model = gtk_tree_view_get_model (tree_view);
        g_return_if_fail (model);
        nb_row_selected = gtk_tree_selection_count_selected_rows
                (a_sel);
        /*we just support one row selected at a time now */
        g_return_if_fail (nb_row_selected <= 1);
        if (nb_row_selected == 0){
                /*no node is selected anymore*/
                PRIVATE (tree_ed)->cur_sel_start = NULL ;
                return ;
        }
        /*
         *Now, lets get an iterator on the row selected
         */
        row_sel = gtk_tree_selection_get_selected_rows
                (a_sel, &model);
        g_return_if_fail (row_sel && row_sel->data);
        is_ok = gtk_tree_model_get_iter 
                (model, &iter, row_sel->data) ;
        g_return_if_fail (is_ok == TRUE) ;
        row_ref = mlview_tree_editor2_iter_2_row_ref 
                (tree_ed, &iter) ;
        g_return_if_fail (row_ref) ;
        PRIVATE (tree_ed)->cur_sel_start = row_ref ;

        g_signal_emit (G_OBJECT (tree_ed),
                       gv_signals[NODE_SELECTED], 0, row_ref) ;
        g_list_foreach (row_sel, (GFunc)gtk_tree_path_free, NULL) ;
        g_list_free (row_sel) ;
}

/**
 *Helper function. Given an attributes node,
 * builds the string "attr1=val1 ... attrN=valN"
 *Note that this function is recursive. 
 *@param a_attr_node the instance of xmlAttr to consider.
 *@param a_result out parameter the resulting string.
 */
static void
xml_attr_to_string (void *a_attr_node, guchar ** a_result)
{
        xmlAttrPtr xml_attr = (xmlAttrPtr) a_attr_node;
        xmlNodePtr xml_node = (xmlNodePtr) a_attr_node;

        static int num_of_use = 0;

        if (num_of_use++ == 0)
                *a_result = NULL;

        if (a_attr_node == NULL)
                return;

        if (xml_attr->type == XML_ATTRIBUTE_NODE) {
                gchar *tmp_str = *a_result,
                        *name;

                if (xml_attr->ns != NULL
                    && xml_attr->ns->prefix != NULL) {
                        name = g_strconcat
                                (xml_attr->ns->prefix, ":",
                                 xml_attr->name, NULL);
                } else {
                        name = g_strdup (xml_attr->name);
                }
                if (tmp_str == NULL)
                        *a_result = g_strdup (name);
                else
                        *a_result = g_strconcat
                                (tmp_str, " ", name, NULL);
                if (tmp_str) {
                        g_free (tmp_str);
                        tmp_str = NULL;
                }
                if (name) {
                        g_free (name);
                        name = NULL;
                }
                if (xml_attr->children)
                        xml_attr_to_string
                                (xml_attr->children, a_result);

                if (xml_attr->next)
                        xml_attr_to_string
                                (xml_attr->next, a_result);
        } else if (xml_node->type == XML_TEXT_NODE) {
                gchar *tmp_str = *a_result;

                if (tmp_str) {
                        *a_result = g_strconcat
                                (tmp_str, "=\"",
                                 xml_node->content, "\"", NULL);
                        g_free (tmp_str);
                        tmp_str = NULL;
                }
        }
}

/**
 *Walks through the list of attributes of
 *the instance of xmlNode and
 *construct a string which looks like
 *'attrname0="attrval0" attrname1=attrvall'
 *where each attrnamei is
 *@param a_node the  node to build the attribute list from.
 *@return the newly built attribute list string, or NULL if something
 *bad happened.
 */
static guchar *
build_attrs_list_str (xmlNode * a_node)
{
        guchar *result = NULL;

        g_return_val_if_fail (a_node, NULL);

        xml_attr_to_string (a_node->properties, &result);
        return result;
}

/**
 *Builds a start tag string 
 *(e.g: <node-name attr0="val0" attr1="val1">)out of
 *an instance of xmlNode *
 *@param a_node the instance of xmlNode * to consider.
 *@return the newly built start tag string.
 */
static guchar *
node_to_string_tag (xmlNode * a_node)
{
        guchar *result = NULL,
                *content = NULL;

        g_return_val_if_fail (a_node != NULL, NULL);

        if (a_node->type == XML_ELEMENT_NODE) {
                guchar *utf8_ns_prefix = NULL,
                        *ns_prefix = NULL,
                        *attr_str = NULL,
                        *name = NULL,
                        *tmp_str = NULL;

                attr_str = build_attrs_list_str (a_node);
                if (a_node->ns != NULL && a_node->ns->prefix) {
                        enum MlViewStatus status = MLVIEW_OK;

                        utf8_ns_prefix =
                                g_strconcat (a_node->ns->prefix,
                                             ":", NULL);
                        status = mlview_utils_utf8_str_to_isolat1
                                (utf8_ns_prefix, &ns_prefix);
                        g_return_val_if_fail (status ==
                                              MLVIEW_OK, NULL);
                } else {
                        ns_prefix = NULL;
                }

                if (ns_prefix) {
                        enum MlViewStatus status = MLVIEW_OK;

                        status = mlview_utils_utf8_str_to_isolat1
                                ((guchar *) a_node->name,
                                 &tmp_str);
                        g_return_val_if_fail (status ==
                                              MLVIEW_OK, NULL);
                        g_return_val_if_fail (tmp_str != NULL,
                                              NULL);
                        name = g_strconcat (ns_prefix, tmp_str,
                                            NULL);
                } else {
                        enum MlViewStatus status = MLVIEW_OK;

                        status = mlview_utils_utf8_str_to_isolat1
                                ((guchar *) a_node->name, &name);
                        g_return_val_if_fail
                                (status == MLVIEW_OK, NULL);
                }
                if (ns_prefix) {
                        g_free (ns_prefix);
                        ns_prefix = NULL;
                }

                if (utf8_ns_prefix) {
                        g_free (utf8_ns_prefix);
                        utf8_ns_prefix = NULL;
                }
                if (tmp_str) {
                        g_free (tmp_str);
                        tmp_str = NULL;
                }
                if (a_node->children != NULL) {
                        if (attr_str)
                                result = g_strconcat
                                        ("<", name, " ",
                                         attr_str, ">", NULL);
                        else
                                result = g_strconcat
                                        ("<", name, ">", NULL);
                } else {        /*empty tag */
                        if (attr_str)
                                result = g_strconcat
                                        ("<", name, " ",
                                         attr_str, "/>", NULL);
                        else
                                result = g_strconcat
                                        ("<", name, "/>", NULL);
                }

                if (name) {
                        g_free (name);
                        name = NULL;
                }

        } else if (xmlNodeIsText (a_node)) {
                enum MlViewStatus status = MLVIEW_OK;
                guchar *utf8_content = NULL;

                utf8_content = xmlNodeGetContent (a_node);

                if (utf8_content) {
                        status = mlview_utils_utf8_str_to_isolat1
                                (utf8_content, &content);
                        g_return_val_if_fail
                                (status == MLVIEW_OK, NULL);
                        xmlFree (utf8_content);
                        utf8_content = NULL;
                }
                if (content == NULL) {
                        xmlNodeSetContent (a_node, "text");
                        content = xmlNodeGetContent (a_node);
                }
                if (content && strlen (content)
                    > MLVIEW_CONTENT_MAX_LEN) {
                        content[MLVIEW_CONTENT_MAX_LEN] = '\0';
                }
                if (content != NULL) {
                        result = g_strdup (content);
                        g_free (content);
                        content = NULL ;
                }
        } else if (a_node->type == XML_COMMENT_NODE) {
                enum MlViewStatus status = MLVIEW_OK;
                guchar *utf8_content = NULL;

                utf8_content = xmlNodeGetContent (a_node);
                if (utf8_content) {
                        status = mlview_utils_utf8_str_to_isolat1
                                (utf8_content, &content);
                        g_return_val_if_fail (status ==
                                              MLVIEW_OK, NULL);

                        xmlFree (utf8_content);
                        utf8_content = NULL;
                }
                if (content == NULL) {
                        xmlNodeSetContent (a_node,
                                           "<!--comment-->");
                        content = xmlNodeGetContent (a_node);
                }
                if (strlen (content) > MLVIEW_CONTENT_MAX_LEN)
                        content[MLVIEW_CONTENT_MAX_LEN] = '\0';

                result = g_strconcat ("<!--", content, "-->",
                                      NULL);
                if (content != NULL) {
                        g_free (content);
                        content = NULL;
                }
        } else if (a_node->type == XML_PI_NODE) {
                enum MlViewStatus status = MLVIEW_OK;
                guchar *utf8_content = NULL,
                        *name = NULL;

                utf8_content = xmlNodeGetContent (a_node);

                if (utf8_content) {
                        status = mlview_utils_utf8_str_to_isolat1
                                (utf8_content, &content);
                        g_return_val_if_fail
                                (status == MLVIEW_OK, NULL);

                        xmlFree (utf8_content);
                        utf8_content = NULL;
                }
                if (content == NULL) {
                        xmlNodeSetContent
                                (a_node,
                                 "<?processing instruction node>");
                }
                status = mlview_utils_utf8_str_to_isolat1
                        ((guchar *) a_node->name, &name);
                g_return_val_if_fail (status == MLVIEW_OK, NULL);

                if (strlen (content) > MLVIEW_CONTENT_MAX_LEN) {
                        content[MLVIEW_CONTENT_MAX_LEN] = '\0';
                }
                result = g_strconcat
                        ("<?", name, " ", content, ">", NULL);
                if (content != NULL) {
                        g_free (content);
                        content = NULL;
                }
                if (name != NULL) {
                        g_free (name);
                        name = NULL;
                }
        }
        return result;
}


/**
 *Helper function that builds a GtkTreeModel out of a
 *libxml2 infoset tree.
 *@param a_app_context the application context.
 *@param a_node the xml node to build the gtktreemodel from.
 *@param a_parent_iter an tree iterator that points to the 
 *current parent tree row
 *@param a_model in/out parameter. The built treemodel.
 *@return MLVIEW_OK upon successful completion, an error
 *code otherwise.
 */
static enum MlViewStatus
build_tree_model_from_xml_tree (MlViewTreeEditor2 * a_this,
                                const xmlNode * a_node,
                                GtkTreeIter * a_ref_iter,
                                enum MlViewTreeInsertType a_type,
                                GtkTreeModel ** a_model)
{
        GtkTreeStore *model = NULL;
        GtkTreeIter iter = {0} ;
        GtkTreeIter parent_iter = {0} ;
        GtkTreePath *tree_path = NULL;
        xmlNode *cur_node = NULL,
                *parent_node = NULL;
        gchar *start_tag = NULL;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_node && a_model && *a_model,
                              MLVIEW_BAD_PARAM_ERROR);

        model = GTK_TREE_STORE (*a_model);
        g_return_val_if_fail (model, MLVIEW_BAD_PARAM_ERROR);

        if (!PRIVATE (a_this)->nodes_rows_hash) {
                PRIVATE (a_this)->nodes_rows_hash =
                        g_hash_table_new (g_direct_hash,
                                          g_direct_equal);
                if (!PRIVATE (a_this)->nodes_rows_hash) {
                        mlview_utils_trace_info 
                                ("The system may be out of memory");
                        return MLVIEW_ERROR;
                }
        }

        for (cur_node = (xmlNode *) a_node;
             cur_node; cur_node = cur_node->next) {
                GtkTreeRowReference *row_ref = NULL;

                start_tag = node_to_string_tag (cur_node);
                switch (a_type) {
                case INSERT_TYPE_ADD_CHILD:
                        gtk_tree_store_append (model, &iter,
                                               a_ref_iter);
                        break;
                case INSERT_TYPE_INSERT_BEFORE:
                case INSERT_TYPE_INSERT_AFTER:
                        parent_node = cur_node->parent;
                        if (!parent_node) {
                                mlview_utils_trace_info 
                                        ("parent_node failed") ;
                                status = MLVIEW_ERROR ;
                                goto cleanup ;
                                
                        }
                        status = mlview_tree_editor2_get_iter
                                (a_this, parent_node,
                                 &parent_iter);
                        if (status != MLVIEW_OK) {
                                mlview_utils_trace_info 
                                        ("status == MLVIEW_OK failed") ;
                                status = MLVIEW_ERROR ;
                                goto cleanup ;
                        }
                        model = GTK_TREE_STORE
                                (mlview_tree_editor2_get_model
                                 (a_this));
                        if (!model) {
                                mlview_utils_trace_info 
                                        ("model failed") ;
                                status = MLVIEW_ERROR ;
                                goto cleanup ;
                        }
                        if (a_type == INSERT_TYPE_INSERT_BEFORE)
                                gtk_tree_store_insert_before
                                        (model, &iter,
                                         &parent_iter,
                                         a_ref_iter);
                        else
                                gtk_tree_store_insert_after
                                        (model, &iter,
                                         &parent_iter,
                                         a_ref_iter);
                        break;
                default:
                        break;
                }
                tree_path = gtk_tree_model_get_path
                        (GTK_TREE_MODEL (model), &iter);
                if (!tree_path) {
                        mlview_utils_trace_info ("tree_path failed") ;
                        status = MLVIEW_ERROR ;
                        goto cleanup ;
                }
                row_ref = gtk_tree_row_reference_new
                        (GTK_TREE_MODEL (model), tree_path);
                if (!row_ref) {
                        mlview_utils_trace_info ("row_ref failed") ;
                        status = MLVIEW_ERROR ;
                        goto cleanup ;
                }
                g_hash_table_insert
                        (PRIVATE (a_this)->nodes_rows_hash,
                         cur_node, row_ref);
                gtk_tree_store_set (model, &iter,
                                    XML_NODE_COLUMN, cur_node, -1);
                gtk_tree_store_set (model, &iter,
                                    FIRST_COLUMN, start_tag, -1);
                if (cur_node->type == XML_ELEMENT_NODE) {
                        gtk_tree_store_set (model, &iter,
                                            SECOND_COLUMN,
                                            "Element Node", 
                                            IS_EDITABLE_COLUMN,
                                            TRUE, -1);
                        if (cur_node->children) {
                                build_tree_model_from_xml_tree
                                        (a_this,
                                         cur_node->children,
                                         &iter,
                                         INSERT_TYPE_ADD_CHILD,
                                         a_model);
                        }
                } else if (cur_node->type == XML_TEXT_NODE) {
                        gtk_tree_store_set (model, &iter,
                                            SECOND_COLUMN,
                                            "Text Node",
                                            IS_EDITABLE_COLUMN,
                                            TRUE, -1);
                } else if (cur_node->type == XML_COMMENT_NODE
                           || cur_node->type == XML_PI_NODE) {
                        gtk_tree_store_set (model, &iter,
                                            SECOND_COLUMN,
                                            "Comment or PI Node", 
                                            IS_EDITABLE_COLUMN,
                                            FALSE,
                                            -1);
                } else {
                        mlview_utils_trace_info ("unknown node");
                }
                if (start_tag) {
                        g_free (start_tag) ;
                        start_tag = NULL ;
                }
                if (tree_path) {
                        gtk_tree_path_free (tree_path) ;
                        tree_path = NULL ;
                }
                if (a_type == INSERT_TYPE_INSERT_BEFORE
                    || a_type == INSERT_TYPE_INSERT_AFTER) {
                        /*
                         *we are building a tree model which root
                         *node is cur_node so we should not
                         *visit cur_node->next.
                         */
                        break ;
                }
        }
        if (*a_model) {
                g_object_set_data (G_OBJECT (*a_model), 
                                   "MlViewTreeEditor2",
                                   a_this) ;
        }
 cleanup:
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        if (start_tag) {
                g_free (start_tag) ;
                start_tag = NULL ;
        }
        return status ;
}

/**
 *Builds an instance of GtkTreeModel* out of
 *an instance of xmlDoc*.
 *@param a_context the application context.
 *@param a_doc the instance of xmlDoc* to consider.
 *@param a_model out parameter the resulting model.
 *@return MLVIEW_OK upon successful completion, an error
 *code otherwise.
 */
static enum MlViewStatus
build_tree_model_from_xml_doc (MlViewTreeEditor2 * a_this,
                               const xmlDoc * a_doc,
                               GtkTreeModel ** a_model)
{
        GtkTreeIter iter = { 0 };
        GtkTreeStore *model = NULL;
        GtkTreeRowReference *row_ref = NULL;
        GtkTreePath *tree_path = NULL;
        xmlNode *xml_tree = NULL;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_doc && a_model
                              && *a_model == NULL,
                              MLVIEW_BAD_PARAM_ERROR);

        if (!PRIVATE (a_this)->nodes_rows_hash) {
                PRIVATE (a_this)->nodes_rows_hash =
                        g_hash_table_new (g_direct_hash,
                                          g_direct_equal);
                if (!PRIVATE (a_this)->nodes_rows_hash) {
                        mlview_utils_trace_info 
                                ("The system may be out of memory");
                        return MLVIEW_ERROR;
                }
        }
        model = gtk_tree_store_new (NB_COLUMNS,
                                    G_TYPE_POINTER,
                                    G_TYPE_BOOLEAN,
                                    G_TYPE_STRING,
                                    G_TYPE_STRING);
        g_return_val_if_fail (model, MLVIEW_BAD_PARAM_ERROR);
        *a_model = GTK_TREE_MODEL (model);
        g_return_val_if_fail (model, MLVIEW_BAD_PARAM_ERROR);

        gtk_tree_store_append (model, &iter, NULL);
        tree_path =
                gtk_tree_model_get_path (GTK_TREE_MODEL (model),
                                         &iter);
        g_return_val_if_fail (tree_path, MLVIEW_BAD_PARAM_ERROR);
        row_ref = gtk_tree_row_reference_new
                (GTK_TREE_MODEL (model), tree_path);
        if (!row_ref) {
                mlview_utils_trace_info ("!row_ref failed") ;
                goto cleanup ;
        }
        g_hash_table_insert (PRIVATE (a_this)->nodes_rows_hash,
                             (gpointer) a_doc, row_ref);
        gtk_tree_store_set (model, &iter, XML_NODE_COLUMN, a_doc, -1);
        gtk_tree_store_set (model, &iter, FIRST_COLUMN,
                            "XML Document Root", -1);
        gtk_tree_store_set (model, &iter, SECOND_COLUMN, "", -1);
        xml_tree = a_doc->children;
        status = build_tree_model_from_xml_tree
                (a_this, xml_tree,
                 &iter, INSERT_TYPE_ADD_CHILD,
                 (GtkTreeModel **) & model);
 cleanup:
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        return status;
}

/**
 *Builds an instance of GtkTreeView from an instance
 *of xmlDoc .
 *@param the application context.
 *@param a_doc the instance of xmlDoc to consider.
 *@return the newly built instance of GtkTreeView, or NULL
 *if a probleme arose.
 */
static GtkTreeView *
build_tree_view_from_xml_doc (MlViewTreeEditor2 * a_this,
                              xmlDoc * a_doc)
{
        GtkTreeView *tree_view = NULL;
        GtkTreeModel *model = NULL;
        GtkCellRenderer *renderer = NULL;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this), NULL);

        status = build_tree_model_from_xml_doc
                (a_this, a_doc, &model);
        g_return_val_if_fail (model, NULL);
        tree_view = GTK_TREE_VIEW
                (gtk_tree_view_new_with_model (model));
        g_return_val_if_fail (tree_view, NULL);
        renderer = gtk_cell_renderer_text_new ();
        gtk_tree_view_insert_column_with_attributes
                (tree_view, FIRST_COLUMN, _("Element start tag"),
                 renderer, "text", FIRST_COLUMN, 
                 "editable", IS_EDITABLE_COLUMN, NULL);
         g_signal_connect (G_OBJECT (renderer),
                          "edited", G_CALLBACK (node_cell_edited_cb),
                          a_this) ;
        renderer = gtk_cell_renderer_text_new ();
        gtk_tree_view_insert_column_with_attributes
                (tree_view, SECOND_COLUMN, _("Element type"),
                 renderer, "text", SECOND_COLUMN, NULL);      
        return tree_view;
}


static gboolean
event_cb (GtkWidget * a_widget,
          GdkEvent * a_event, gpointer a_user_data)
{
        MlViewTreeEditor2 *tree_editor = NULL;

        g_return_val_if_fail (a_widget != NULL, FALSE);
        g_return_val_if_fail (GTK_IS_WIDGET (a_widget), FALSE);
        g_return_val_if_fail (a_user_data != NULL, FALSE);
        g_return_val_if_fail (MLVIEW_IS_TREE_EDITOR2
                              (a_user_data), FALSE);
        g_return_val_if_fail (a_event != NULL, FALSE);

        tree_editor = MLVIEW_TREE_EDITOR2 (a_user_data);
        g_return_val_if_fail (tree_editor != NULL, FALSE);
        g_return_val_if_fail (PRIVATE (tree_editor), FALSE);

        switch (a_event->type) {
        case GDK_BUTTON_PRESS:
                if (a_event->button.button == 3) {
                        /*user pressed the right mouse button */
                        GtkWidget *root_widget =
                                gtk_widget_get_toplevel
                                (GTK_WIDGET (tree_editor));

                        g_return_val_if_fail (root_widget !=
                                              NULL, FALSE);

                        gtk_propagate_event (root_widget,
                                             a_event);
                }
                break;
        default:
                break;
        }
        return TRUE;
}

static void
node_cell_edited_cb (GtkCellRendererText *a_renderer,
                     gchar *a_cell_path,
                     gchar *a_new_text,
                     gpointer a_data)
{
        MlViewTreeEditor2 *tree_editor = NULL ;
        GtkTreePath *tree_path = NULL ;
        GtkTreeIter iter = {0} ;
        GtkTreeModel *model = NULL ;
        MlViewXMLDocument *mlview_xml_doc = NULL ;
        enum MlViewStatus status = MLVIEW_OK ;
        GString *element_name = NULL ;
        GList *nv_pair_list = NULL ;
        xmlNode *cur_node = NULL ;
        guchar *start_tag = NULL ;

        g_return_if_fail (a_renderer && a_data && a_cell_path) ;
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR2 (a_data) 
                          && GTK_IS_CELL_RENDERER_TEXT (a_renderer)) ;
        tree_editor = a_data ;
        model = mlview_tree_editor2_get_model (tree_editor) ;
        g_return_if_fail (model) ;
        tree_path = gtk_tree_path_new_from_string (a_cell_path) ;
        g_return_if_fail (tree_path) ;
        status = mlview_tree_editor2_get_cur_sel_start_iter 
                (tree_editor, &iter) ;
        g_return_if_fail (status == MLVIEW_OK) ;
        cur_node = mlview_tree_editor2_get_cur_sel_xml_node (tree_editor) ;
        if (!cur_node) {
                mlview_utils_trace_info ("cur_node failed") ;
                goto cleanup ;
        }
        mlview_xml_doc = mlview_tree_editor2_get_mlview_xml_doc (tree_editor);
        if (!mlview_xml_doc) {
                mlview_utils_trace_info 
                        ("mlview_xml_doc failed") ;
                goto cleanup ;
        }
        start_tag = node_to_string_tag (cur_node) ;
        if (cur_node->type == XML_ELEMENT_NODE) {
                status = mlview_utils_parse_start_tag
                        (a_new_text, &element_name, &nv_pair_list) ;
                if (status != MLVIEW_OK) {
                        g_signal_handlers_block_by_func
                                (a_renderer,
                                 G_CALLBACK (node_cell_edited_cb),
                                 a_data) ;
                        gtk_tree_store_set (GTK_TREE_STORE (model), 
                                            &iter, FIRST_COLUMN,
                                            start_tag, -1) ;
                        g_signal_handlers_unblock_by_func 
                                (a_renderer,
                                 G_CALLBACK (node_cell_edited_cb),
                                 a_data) ;
                } else {                        
                        mlview_xml_document_set_node_name
                                (mlview_xml_doc, cur_node,
                                 element_name->str, UTF8, TRUE) ;
                        mlview_xml_document_synch_attributes 
                                (mlview_xml_doc, cur_node, nv_pair_list) ;
                }
        } else if (cur_node->type == XML_TEXT_NODE) {
                mlview_xml_document_set_node_content 
                        (mlview_xml_doc, cur_node, a_new_text, UTF8, TRUE) ;
        }

 cleanup:
        if (start_tag) {
                g_free (start_tag) ;
                start_tag = NULL ;
        }
        if (element_name) {
                g_string_free (element_name, TRUE) ;
                element_name = NULL ;
        }
        if (nv_pair_list) {
                GList *cur_item = NULL;
                for (cur_item = nv_pair_list; cur_item ;
                     cur_item = cur_item->next) {
                        if (cur_item->data) {
                                mlview_utils_name_value_pair_free 
                                        (cur_item->data, TRUE) ;
                        }
                }
                g_list_free (nv_pair_list) ;
                nv_pair_list = NULL ;
        }
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
}

/***************************
 *private helper functions
 ***************************/
/**
 *Builds a new xml node which type is given in argument.
 *@param a_node_type the node type.
 *@param the instance of #MlViewXMLDocument that holds the
 *newly created instance of xmlNode.
 *@return the newly created instance of xmlNode, or NULL if something
 *bad happened.
 */
static xmlNode *
new_xml_node (xmlElementType a_node_type,
              MlViewXMLDocument * a_xml_doc)
{
        xmlNodePtr result = NULL;
        xmlDoc *doc = NULL;

        if (a_xml_doc)
                doc = mlview_xml_document_get_xml_document
                        (a_xml_doc);
        switch (a_node_type) {
        case XML_ELEMENT_NODE:
                result = xmlNewNode (NULL, "");
                break;
        case XML_TEXT_NODE:
                result = xmlNewText ("");
                break;
        case XML_CDATA_SECTION_NODE:
                g_return_val_if_fail (doc != NULL, NULL);
                xmlNewCDataBlock (doc, "", 128);
                break;
        case XML_PI_NODE:
                result = xmlNewPI ("", "");
                break;
        case XML_COMMENT_NODE:
                result = xmlNewComment ("");
                break;
        case XML_DOCUMENT_NODE:
        case XML_DOCUMENT_TYPE_NODE:
        case XML_DOCUMENT_FRAG_NODE:
        case XML_NOTATION_NODE:
        case XML_DTD_NODE:
        case XML_ELEMENT_DECL:
        case XML_ATTRIBUTE_DECL:
        case XML_ENTITY_DECL:
        case XML_NAMESPACE_DECL:
        case XML_XINCLUDE_START:
        case XML_XINCLUDE_END:
        default:
                result = xmlNewNode (NULL, "");
                break;
        }
        return result;
}

/**
 *This function is to be called once the user
 *clicks the OK button of the node type picker,
 *which means she wants to add a child node to the
 *currently selected xml node.
 *@param a_this the current instance of #MlViewTreeEditor2 .
 */
static void
handle_nt_picker_ok_button_clicked_to_add_child (MlViewTreeEditor2 *a_this)
{
        guint selected_node_type=0,
		node_addion_status=0;
        MlViewNodeTypePicker *picker;
        xmlNodePtr xml_node = NULL;
        MlViewXMLDocument *xml_doc = NULL;
        xmlNs *ns = NULL;
        guchar *node_name_or_content = NULL,
                *local_name = NULL;
        GtkTreeIter iter={0} ;
        enum MlViewStatus status=MLVIEW_OK ;

        g_return_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)) ;
        picker =
                mlview_tree_editor2_get_node_type_picker
                (a_this);
        g_return_if_fail (picker != NULL);
        node_name_or_content =
                mlview_node_type_picker_get_node_name_or_content
                (picker);
        if (node_name_or_content != NULL
            && !mlview_utils_is_white_string (node_name_or_content)) {
                selected_node_type =
                        mlview_node_type_picker_get_selected_node_type
                        (picker);
                xml_doc =
                        mlview_tree_editor2_get_mlview_xml_doc
                        (a_this);
                xml_node =
                        new_xml_node (selected_node_type, xml_doc);
                switch (selected_node_type) {
                case XML_ELEMENT_NODE:
                case XML_PI_NODE:
                        mlview_utils_parse_full_name
                                (xml_node, node_name_or_content,
                                 &ns, &local_name);
                        if (local_name != NULL) {
                                mlview_xml_document_set_node_name
                                        (xml_doc, xml_node,
                                         local_name, ISO8859_1,
                                         TRUE);
                                g_free (local_name);
                                local_name = NULL;
                        }
                        break;
                default:
                        mlview_xml_document_set_node_content
                                (xml_doc, xml_node,
                                 node_name_or_content, ISO8859_1,
                                 FALSE);
                        break;
                }
                status = mlview_tree_editor2_get_cur_sel_start_iter 
                        (a_this, &iter) ;
                g_return_if_fail (status == MLVIEW_OK) ;
                node_addion_status =
                        mlview_tree_editor2_add_child_node (a_this,
                                                            &iter, 
                                                            xml_node);
                if (!node_addion_status
                    && (selected_node_type == XML_ELEMENT_NODE
                        || selected_node_type == XML_PI_NODE)) {
                        mlview_utils_parse_full_name
                                (xml_node, node_name_or_content,
                                 &ns, &local_name);
                        if (ns) {
                                xmlSetNs (xml_node, ns);
                        } else {
                                xml_node->ns = NULL;
                        }
                        if (local_name) {
                                g_free (local_name);
                                local_name = NULL;
                        }
                }
        }
}

/**
 *This function is to be called once the user
 *clicks the OK button of the node type picker,
 *which means she wants to insert a sibling node to
 *the currently selected xml node.
 *@param a_this the current instance of #MlViewTreeEditor2 .
 */
static void
handle_nt_picker_ok_button_clicked_to_insert_sibling_node (MlViewTreeEditor2 *a_this)
{
        guint selected_node_type=0 ;
        MlViewNodeTypePicker *picker;
        xmlNodePtr xml_node = NULL;
        MlViewXMLDocument *xml_doc = NULL;
        xmlNs *ns = NULL;
        guchar *node_name_or_content = NULL,
                *local_name = NULL;
        GtkTreeIter iter={0} ;
        gboolean *prev_ptr=FALSE ;
        enum MlViewStatus status=MLVIEW_OK ;

        g_return_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)) ;
        picker =
                mlview_tree_editor2_get_node_type_picker (a_this);
        g_return_if_fail (picker != NULL);
        node_name_or_content =
                mlview_node_type_picker_get_node_name_or_content
                (picker);
        if (node_name_or_content != NULL
            && !mlview_utils_is_white_string (node_name_or_content)) {
                selected_node_type =
                        mlview_node_type_picker_get_selected_node_type
                        (picker);
                xml_doc =
                        mlview_tree_editor2_get_mlview_xml_doc
                        (a_this);
                xml_node =
                        new_xml_node (selected_node_type, xml_doc);
                switch (selected_node_type) {
                case XML_ELEMENT_NODE:
                case XML_PI_NODE:
                        mlview_utils_parse_full_name
                                (xml_node, node_name_or_content,
                                 &ns, &local_name);
                        if (local_name != NULL) {
                                mlview_xml_document_set_node_name
                                        (xml_doc, xml_node,
                                         local_name, ISO8859_1,
                                         TRUE);
                                g_free (local_name);
                                local_name = NULL;
                        }
                        break;
                default:
                        mlview_xml_document_set_node_content
                                (xml_doc, xml_node,
                                 node_name_or_content, ISO8859_1,
                                 FALSE);
                        break;
                }
                /*
                 *retrieve a flag set to indicate 
                 *if the sibling has to be inserted 
                 *before or after the current xml node
                 */
                prev_ptr =
                        gtk_object_get_data (GTK_OBJECT
                                             (a_this),
                                             "prev");
                status = mlview_tree_editor2_get_cur_sel_start_iter 
                        (a_this, &iter) ;
                g_return_if_fail (status == MLVIEW_OK) ;
                status = mlview_tree_editor2_insert_sibling_node
                        (a_this, &iter, xml_node, 
                         GPOINTER_TO_INT (prev_ptr)) ;

                if (status == MLVIEW_OK
                    && (selected_node_type == XML_ELEMENT_NODE
                        || selected_node_type == XML_PI_NODE)) {
                        mlview_utils_parse_full_name
                                (xml_node, node_name_or_content,
                                 &ns, &local_name);
                        if (ns) {
                                xmlSetNs (xml_node, ns);
                        } else {
                                xml_node->ns = NULL;
                        }
                        if (local_name) {
                                g_free (local_name);
                                local_name = NULL;
                        }
                        mlview_tree_editor2_update_visual_node 
                                (a_this, &iter) ;
                }
        }
}

/**
 *Returns a copy of the search string typed by the user.
 *@param a GtkDialog that represents the "search" dialog. Must
 *have been constructed by a call to 
 *get_search_dialog().
 *@return a pointer to the search string typed by the user.
 *The return string must _NOT_ be freed by the user.
 */
static const guchar *
get_search_string (GtkDialog *a_search_dialog)
{
        GtkWidget *text_entry = NULL ;

        g_return_val_if_fail (a_search_dialog 
                              && GTK_IS_DIALOG (a_search_dialog),
                              NULL) ;
        
        text_entry = g_object_get_data (G_OBJECT (a_search_dialog),
                                        "SearchEntry") ;
        if (!text_entry || !GTK_IS_ENTRY (text_entry)) {
                mlview_utils_trace_info 
                        ("Retrieving data associated to "
                         "SearchEntry from the Search Dialog failed. "
                         "The Search dialog may not be a valid one.");
                return NULL ;
        }
        return gtk_entry_get_text (GTK_ENTRY (text_entry)) ;
}

/**
 *Gets the search configuration from what the
 *user has selected through the search dialog.
 *@param a_search_dialog the search dialog. Must have
 *been constructed by a call to get_search_dialog() .
 *@param a_config the search configuration (i.e: whether to
 *look into node names, content, attribute names etc ...)
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
static enum MlViewStatus
get_search_config (GtkDialog *a_search_dialog,
                   struct SearchConfig *a_config)
{       
        GtkWidget *widget = NULL ;

        g_return_val_if_fail (a_search_dialog
                          && GTK_IS_DIALOG (a_search_dialog)
                          && a_config, MLVIEW_BAD_PARAM_ERROR) ;

        widget = g_object_get_data (G_OBJECT (a_search_dialog),
                                    "MatchCaseButton") ;        
        g_return_val_if_fail (widget 
                              && GTK_IS_CHECK_BUTTON (widget), 
                              MLVIEW_ERROR) ;
        a_config->ignore_case = 
                gtk_toggle_button_get_active
                (GTK_TOGGLE_BUTTON (widget)) ;
        if (a_config->ignore_case == TRUE) {
                a_config->ignore_case = FALSE ;
        } else {
                a_config->ignore_case = TRUE ;
        }        
        widget = g_object_get_data (G_OBJECT (a_search_dialog),
                                    "SearchInNodeNamesButton") ;
        g_return_val_if_fail (widget 
                              && GTK_IS_CHECK_BUTTON (widget), 
                              MLVIEW_ERROR) ;
        if (gtk_toggle_button_get_active 
            (GTK_TOGGLE_BUTTON (widget)) == TRUE) {
                a_config->where |=  NODE_NAME;
        }
        widget = g_object_get_data (G_OBJECT (a_search_dialog),
                                    "SearchInAttrNamesButton") ;
        g_return_val_if_fail (widget
                              && GTK_IS_CHECK_BUTTON (widget), 
                              MLVIEW_ERROR) ;
        if (gtk_toggle_button_get_active 
            (GTK_TOGGLE_BUTTON (widget)) == TRUE) {
                a_config->where |= NODE_ATTRIBUTE_NAME ;
        }
        widget = g_object_get_data (G_OBJECT (a_search_dialog),
                                    "SearchInAttrValuesButton") ;
        g_return_val_if_fail (widget
                              && GTK_IS_CHECK_BUTTON (widget), 
                              MLVIEW_ERROR) ;
        if (gtk_toggle_button_get_active 
            (GTK_TOGGLE_BUTTON (widget)) == TRUE) {
                a_config->where |=  NODE_ATTRIBUTE_VALUE ;
        }        
        widget = g_object_get_data (G_OBJECT (a_search_dialog),
                                    "SearchInNodeContentButton") ;
        g_return_val_if_fail (widget
                              && GTK_IS_CHECK_BUTTON (widget), 
                              MLVIEW_ERROR) ;
        if (gtk_toggle_button_get_active 
            (GTK_TOGGLE_BUTTON (widget)) == TRUE) {
                a_config->where |=  NODE_CONTENT ;
        }
        a_config->search_string = (guchar*)
                get_search_string (a_search_dialog) ;
        return MLVIEW_OK ;
}

/**
 *Builds and returns the "search" dialog.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@return the dialog.
 */
static GtkWidget *
get_search_dialog (MlViewTreeEditor2 *a_this)
{
        MlViewAppContext *ctxt = NULL ;
        GladeXML *glade_xml = NULL ;
        GtkWidget *widget = NULL, *dialog_widget = NULL ;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              NULL) ;
        if (!PRIVATE (a_this)->search_dialog) {
                guchar *glade_file_path=NULL ;
                glade_file_path = g_strconcat
                        (MLVIEW_GLADE_DIR,
                         "/tree-view-find-dialog.glade",
                         NULL) ;
                glade_xml = glade_xml_new
                        (glade_file_path,
                         "MlViewTreeViewSearchDialog", NULL) ;
                if (!glade_xml) {
                        mlview_utils_trace_info 
                                ("glade xml file loading failed") ;
                        g_object_unref (glade_xml) ;
                        return NULL ;                        
                }
                dialog_widget =
                        glade_xml_get_widget 
                                    (glade_xml, 
                                     "MlViewTreeViewSearchDialog") ;
                if (!dialog_widget) {
                        mlview_utils_trace_info 
                                ("getting widget from glade failed") ;
                        goto cleanup ;
                }
                widget = glade_xml_get_widget (glade_xml,
                                               "SearchEntry") ;
                if (!widget) {
                        mlview_utils_trace_info 
                                ("getting SearchEntry " 
                                 "from glade file failed") ;
                        goto cleanup ;
                }
                g_object_set_data (G_OBJECT (dialog_widget), 
                                   "SearchEntry", widget) ;
                widget = glade_xml_get_widget 
                        (glade_xml,
                         "MatchCaseButton") ;
                if (!widget) {
                        mlview_utils_trace_info 
                                ("getting MatchCaseButton from "
                                 "glade file failed") ;
                        goto cleanup ;
                }
                g_object_set_data (G_OBJECT (dialog_widget),
                                   "MatchCaseButton",
                                   widget) ;
                widget = glade_xml_get_widget 
                        (glade_xml, 
                         "SearchInNodeNamesButton") ;
                if (!widget) {
                        mlview_utils_trace_info 
                                ("getting from SearchInNodeNamesButton "
                                 "glade file failed") ;
                        goto cleanup ;
                }
                g_object_set_data (G_OBJECT (dialog_widget),
                                   "SearchInNodeNamesButton",
                                   widget) ;
                widget = glade_xml_get_widget 
                        (glade_xml, 
                         "SearchInAttrNamesButton") ;
                if (!widget) {
                        mlview_utils_trace_info 
                                ("getting from SearchInAttrNamesButton "
                                 "glade file failed") ;
                        goto cleanup ;
                }
                g_object_set_data (G_OBJECT (dialog_widget),
                                   "SearchInAttrNamesButton",
                                   widget) ;
                widget = glade_xml_get_widget 
                        (glade_xml, 
                         "SearchInAttrValuesButton") ;
                if (!widget) {
                        mlview_utils_trace_info 
                                ("getting from SearchInAttrValuesButton "
                                 "glade file failed") ;
                        goto cleanup ;
                }
                g_object_set_data (G_OBJECT (dialog_widget),
                                   "SearchInAttrValuesButton", 
                                   widget) ;
                widget = glade_xml_get_widget 
                        (glade_xml, 
                         "SearchInNodeContentButton") ;
                if (!widget) {
                        mlview_utils_trace_info 
                                ("getting from SearchInNodeContentButton "
                                 "glade file failed") ;
                        goto cleanup ;
                }
                g_object_set_data (G_OBJECT (dialog_widget),
                                   "SearchInNodeContentButton", 
                                   widget) ;
                ctxt = mlview_tree_editor2_get_application_context
                        (a_this) ;
                if (ctxt) {
                        mlview_app_context_set_window_icon 
                                (ctxt, GTK_WINDOW (dialog_widget)) ;
                }
                PRIVATE (a_this)->search_dialog = 
                        GTK_DIALOG (dialog_widget) ;
                dialog_widget = NULL ;                
        }

 cleanup:
        if (dialog_widget) {
                gtk_widget_destroy (dialog_widget) ;
                dialog_widget = NULL ;
        }
        if (glade_xml) {
                g_object_unref (glade_xml) ;
                glade_xml = NULL ;
        }
        return  GTK_WIDGET (PRIVATE (a_this)->search_dialog) ;
}
typedef gboolean (*DragDataReceivedFunc) (GtkTreeDragDest *,
                                           GtkTreePath *,
                                           GtkSelectionData *) ;
static gboolean
drag_data_received (GtkTreeDragDest *a_drag_dest,
                    GtkTreePath *a_dest_path,
                    GtkSelectionData *a_sel_data)
{
        GtkTreeModel *src_model, *dest_model = NULL ;
        GtkTreePath *src_path = NULL ;
        MlViewTreeEditor2 *editor = NULL ;
        gboolean is_ok = TRUE, result = FALSE ;        
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_drag_dest && a_dest_path
                              && a_sel_data, FALSE) ;
        dest_model = GTK_TREE_MODEL (a_drag_dest) ;
        g_return_val_if_fail (dest_model, FALSE) ;
        is_ok = gtk_tree_get_row_drag_data 
                (a_sel_data, &src_model, &src_path) ;
        g_return_val_if_fail (is_ok == TRUE
                              && src_model == dest_model, 
                              FALSE);
        editor = g_object_get_data (G_OBJECT (a_drag_dest),
                                    "MlViewTreeEditor2") ;
        if (!editor) {
                mlview_utils_trace_info ("editor != NULL failed.") ;
                goto cleanup ;
        }
        /*copy the xml node pointed to by src_path to the clipboard.*/
        status = mlview_tree_editor2_copy_node2 (editor, src_path) ;
        if (status != MLVIEW_OK) {
                mlview_utils_trace_info 
                        ("status == MLVIEW_OK failed.") ;
                goto cleanup ;
        }
        /*
         *paste the newly copied node from the clipboard
         *to the new location
         */
        status = mlview_tree_editor2_paste_node_as_sibling2
                (editor, a_dest_path,
                 TRUE/*paste as previous*/);
        if (status != MLVIEW_OK) {
                mlview_utils_trace_info 
                        ("status == MLVIEW_OK failed") ;
                goto cleanup ;
        }
        result = TRUE ;
 cleanup:
        if (src_path) {
                gtk_tree_path_free (src_path) ;
                src_path = NULL ;
        }
        return result ;
}

typedef gboolean (*DragDataDeleteFunc) (GtkTreeDragSource *, 
                                        GtkTreePath *) ;

static gboolean
drag_data_delete (GtkTreeDragSource *a_drag_src,
                  GtkTreePath *a_path)
{
        GtkTreeModel *model = NULL ;
        MlViewTreeEditor2 *editor = NULL ;
        enum MlViewStatus status =  MLVIEW_OK ;

        editor = g_object_get_data (G_OBJECT (a_drag_src),
                                    "MlViewTreeEditor2") ;
        g_return_val_if_fail (editor, FALSE) ;
        model = GTK_TREE_MODEL (a_drag_src) ;
        g_return_val_if_fail (model, FALSE) ;
        status = mlview_tree_editor2_cut_node2 (editor, a_path) ;
        if (status == MLVIEW_OK) {
                return TRUE ;
        }
        return FALSE ;
}
static enum MlViewStatus
backup_original_dnd_callbacks (MlViewTreeEditor2 *a_this)
{
        GtkTreeDragSourceIface *drag_source_iface = NULL;
        GtkTreeDragDestIface *drag_dest_iface = NULL;
        GtkTreeModel *model = NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model && GTK_IS_TREE_STORE (model),
                              MLVIEW_ERROR) ;
        drag_source_iface = GTK_TREE_DRAG_SOURCE_GET_IFACE (model) ;
        g_return_val_if_fail (drag_source_iface, MLVIEW_ERROR) ;
        drag_dest_iface = GTK_TREE_DRAG_DEST_GET_IFACE (model) ;

        if (!PRIVATE (a_this)->backup_drag_data_delete) {
                PRIVATE (a_this)->backup_drag_data_delete = 
                        drag_source_iface->drag_data_delete ;
        }
        if (!PRIVATE (a_this)->backup_drag_data_received) {
                PRIVATE (a_this)->backup_drag_data_received =
                        drag_dest_iface->drag_data_received ;
        }
        return MLVIEW_OK ;
}

static enum MlViewStatus
restore_original_dnd_callbacks (MlViewTreeEditor2 *a_this)
{
        GtkTreeDragSourceIface *drag_source_iface ;
        GtkTreeDragDestIface *drag_dest_iface ;
        GtkTreeModel *model = NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model && GTK_IS_TREE_STORE (model),
                              MLVIEW_ERROR) ;
        drag_source_iface = GTK_TREE_DRAG_SOURCE_GET_IFACE (model) ;
        g_return_val_if_fail (drag_source_iface, MLVIEW_ERROR) ;
        drag_dest_iface = GTK_TREE_DRAG_DEST_GET_IFACE (model) ;

        drag_source_iface->drag_data_delete  =
                PRIVATE (a_this)->backup_drag_data_delete ;
        PRIVATE (a_this)->backup_drag_data_delete = NULL ;
        drag_dest_iface->drag_data_received =
                PRIVATE (a_this)->backup_drag_data_received ;
        PRIVATE (a_this)->backup_drag_data_received = NULL ;

        return MLVIEW_OK ;
}

/**
 *Enables the Drad and drop functionality
 *on the PRIVATE (a_this)->tree_view.
 *This function must be called once a
 *PRIVATE (a_this)->tree_view has been allocated.
 */
static enum MlViewStatus
set_our_dnd_callbacks (MlViewTreeEditor2 *a_this)
{
        GtkTreeDragSourceIface *drag_source_iface ;
        GtkTreeDragDestIface *drag_dest_iface ;
        GtkTreeModel *model = NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model && GTK_IS_TREE_STORE (model),
                              MLVIEW_ERROR) ;
        drag_source_iface = GTK_TREE_DRAG_SOURCE_GET_IFACE (model) ;
        g_return_val_if_fail (drag_source_iface, MLVIEW_ERROR) ;
        drag_dest_iface = GTK_TREE_DRAG_DEST_GET_IFACE (model) ;
        g_return_val_if_fail (drag_dest_iface, MLVIEW_ERROR) ;
        backup_original_dnd_callbacks (a_this) ;
        drag_source_iface->drag_data_delete  = drag_data_delete ;
        drag_dest_iface->drag_data_received = drag_data_received ;
        return MLVIEW_OK ;
}

/*********************
 *Public methods
 ********************/

/**
 *the standard type id builder of the MlViewTreeEditor2 object. 
 *
 *@return the type id of the MlViewTreeEditor2 object. 
 */
guint
mlview_tree_editor2_get_type (void)
{
        static guint type = 0;

        if (!type) {
                static const GTypeInfo type_info = {
                        sizeof (MlViewTreeEditor2Class),
                        NULL, NULL,
                        (GClassInitFunc)
                                mlview_tree_editor2_class_init,
                        NULL, NULL,
                        sizeof (MlViewTreeEditor2),
                        0,
                        (GInstanceInitFunc)
                        mlview_tree_editor2_init
                };
                type = g_type_register_static (GTK_TYPE_VBOX,
                                               "MlViewTreeEditor2",
                                               &type_info, 0);
        }
        return type;
}


/**
 *MlViewTreeEditor2 instance builder.
 *@param a_context the application context.
 *@return the newly built app context or NULL
 *if an error arises.
 */
GtkWidget *
mlview_tree_editor2_new (MlViewAppContext * a_context)
{
        MlViewTreeEditor2 *editor = NULL;

        editor = g_object_new (MLVIEW_TYPE_TREE_EDITOR2, NULL);
        PRIVATE (editor)->app_context = a_context;

        g_signal_connect (G_OBJECT (editor),
                          "button_press_event",
                          G_CALLBACK (event_cb), editor);       
        return GTK_WIDGET (editor);
}

/**
 *Setter of the application context
 *@param a_this the current instance
 *of MlViewTreeEditor2.
 *@param a_app_context the new application context.
 */
void
mlview_tree_editor2_set_application_context (MlViewTreeEditor2 * a_this,
                                              MlViewAppContext * a_app_context) 
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR2
                          (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        PRIVATE (a_this)->app_context = a_app_context;
}

/**
 *Gets the instance of GtkTreeModel (namely an instance of
 *GtkTreeStore) of the current xml document.
 *@param a_this the current instance of #MlViewEditor.
 *@return the instance of GtkTreeModel, or NULL.
 */
GtkTreeModel *
mlview_tree_editor2_get_model (MlViewTreeEditor2 * a_this)
{
        GtkTreeView *tree_view = NULL;
        GtkTreeModel *model = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this),
                              NULL);
        tree_view = mlview_tree_editor2_get_tree_view (a_this);
        g_return_val_if_fail (tree_view, NULL);
        model = gtk_tree_view_get_model (tree_view);
        g_return_val_if_fail (model, NULL);
        return model;
}

/**
 *Gets the instance of GtkTreeView that
 *features the xml document graphically.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@return the GtkTreeView graphical tree.
 */
GtkTreeView *
mlview_tree_editor2_get_tree_view (MlViewTreeEditor2 * a_this)
{
        g_return_val_if_fail (a_this && PRIVATE (a_this), NULL);
        g_return_val_if_fail
                (MLVIEW_IS_TREE_EDITOR2 (a_this), NULL);

        return PRIVATE (a_this)->tree_view;
}

/**
 *Getter of the application context.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@return the app context contained in this instance of MlViewTreeEditor2.
 */
MlViewAppContext *
mlview_tree_editor2_get_application_context (MlViewTreeEditor2 * a_this) 
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_TREE_EDITOR2 (a_this),
                              NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);

        return PRIVATE (a_this)->app_context;
}


/**
 *Sets the title of the xml DOM to @a_file_path.
 *Updates this information in the tree editor and in the
 *XML DOM. 
 *@param a_file_path the new file path of xml document.
 *@param a_this the current tree editor.
 */
void
mlview_tree_editor2_set_xml_document_path (MlViewTreeEditor2 * a_this, 
                                           gchar * a_file_path) 
{
        GtkTreeViewColumn *tree_column = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (a_file_path != NULL);

        if (!PRIVATE (a_this)->xml_doc)
                return;

        if (PRIVATE (a_this)->xml_doc->name) {
                g_free (PRIVATE (a_this)->xml_doc->name);
                PRIVATE (a_this)->xml_doc->name = NULL ;
        }

        PRIVATE (a_this)->xml_doc->name = g_strdup (a_file_path);

        tree_column = gtk_tree_view_get_column
                (PRIVATE (a_this)->tree_view, FIRST_COLUMN);
        g_return_if_fail (tree_column);

        gtk_tree_view_column_set_title
                (tree_column, PRIVATE (a_this)->xml_doc->name);
}

/**
 *Return the first row of the currently selected row.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@return the first row of the set of selected rows.
 */
GtkTreeRowReference *
mlview_tree_editor2_get_sel_start (MlViewTreeEditor2 * a_this)
{
        GtkTreeRowReference *row_ref = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this), NULL);
        return row_ref = PRIVATE (a_this)->cur_sel_start;
}

/**
 *Get the instance of #MlViewXMLDocument associated to this
 *instance of #MlViewTreeEditor2
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@return the instance of #MlViewXMLDocument associated to this
 *the tree editor or NULL.
 */
MlViewXMLDocument *
mlview_tree_editor2_get_mlview_xml_doc (MlViewTreeEditor2 *a_this)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              NULL) ;
        return PRIVATE (a_this)->mlview_xml_doc ;
}

/**
 *Gets the node type picker associated to this
 *instance of #MlViewTreeEditor2. If the picker doesn't
 *exists, this function creates it.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@return the node type picker, or NULL if something bad happened.
 */
MlViewNodeTypePicker *
mlview_tree_editor2_get_node_type_picker (MlViewTreeEditor2 *a_this)
{
        GtkWidget *res=NULL ;
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this), NULL) ;
        res = GTK_WIDGET (PRIVATE (a_this)->node_type_picker) ;
        if (! res) {
                res = mlview_node_type_picker_new 
                        (PRIVATE (a_this)->app_context) ;
                g_return_val_if_fail (res, NULL) ;
                if (! MLVIEW_IS_NODE_TYPE_PICKER (res))
                {
                        mlview_utils_trace_info 
                                ("Expected a Node type picker, found "
                                 "an unknown type" ) ;
                        return NULL ;
                }
                gtk_window_set_modal (GTK_WINDOW (res), TRUE) ;
                mlview_tree_editor2_set_node_type_picker 
                        (a_this, MLVIEW_NODE_TYPE_PICKER (res)) ;
        }
        return MLVIEW_NODE_TYPE_PICKER (res) ;
}

/**
 *Sets the node type picker associated to
 *the current instance of #MlViewTreeEditor2.
 *If a node type picker was already associated to
 *the current instance of #MlViewTreeEditor2, unref it
 *before associating the new one.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_set_node_type_picker (MlViewTreeEditor2 *a_this,
                                          MlViewNodeTypePicker *a_picker)

{
        g_return_val_if_fail (a_this 
                              && MLVIEW_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_picker 
                              && MLVIEW_IS_NODE_TYPE_PICKER (a_picker),
                              MLVIEW_BAD_PARAM_ERROR) ;

        if (PRIVATE (a_this)->node_type_picker) {
                g_object_unref 
                        (G_OBJECT (PRIVATE (a_this)->node_type_picker)) ;
        }
        PRIVATE (a_this)->node_type_picker = a_picker ;
        return MLVIEW_OK ;
}

/**
 *Gets the GtkTreeRowReference associated
 *to the node pointed to by a_iter. User doesn't
 *need to free the returned GtkTreeRowReference because
 *MlViewTreeEditor2 takes care of it.
 */
GtkTreeRowReference *
mlview_tree_editor2_iter_2_row_ref (MlViewTreeEditor2 *a_this,
                                    GtkTreeIter *a_iter)
{
        GtkTreeRowReference *result=NULL ;
        xmlNode *xml_node=NULL ;
        GtkTreeModel *model=NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->nodes_rows_hash
                              && a_iter, NULL) ;

        model = mlview_tree_editor2_get_model (a_this) ;
        gtk_tree_model_get (model, a_iter, XML_NODE_COLUMN,
                            &xml_node, -1) ;
        g_return_val_if_fail (xml_node, NULL) ;
        result = g_hash_table_lookup 
                (PRIVATE (a_this)->nodes_rows_hash,
                 xml_node) ;
        return result ;
}

/**
 *A very important method in this class 
 *(and even in the whole MlView software).
 *It takes an xml document in parameter, 
 *builds the graphical view that matches it and displays it. 
 *
 *@param a_xmldoc the xml document to edit.
 *@param a_editor the current instance of MlViewTreeEditor.
 *@return MLVIEW_OK upon successful completion,
 *an error code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_edit_xml_doc (MlViewTreeEditor2 * a_this,
                                  MlViewXMLDocument * a_doc,
                                  guchar * a_doc_name)
{
        GtkTreeView *tree_view = NULL;
        GtkTreeSelection *selection = NULL;
        GtkWidget *scr_win = NULL;
        glong expansion_depth = MLVIEW_DEFAULT_EXPANSION_DEPTH;
        gchar *expansion_depth_str = NULL;
        GHashTable *settings = NULL;
        xmlDoc *xml_doc = NULL;

        g_return_val_if_fail (a_this != NULL, MLVIEW_ERROR);
        g_return_val_if_fail (MLVIEW_IS_TREE_EDITOR2 (a_this),
                              MLVIEW_ERROR);
        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              MLVIEW_ERROR);
        g_return_val_if_fail (a_doc != NULL, MLVIEW_ERROR);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT
                              (a_doc), MLVIEW_ERROR);

        xml_doc =
                mlview_xml_document_get_xml_document
                ((MlViewXMLDocument *) a_doc);
        g_return_val_if_fail (xml_doc != NULL, -1);
        PRIVATE (a_this)->mlview_xml_doc = a_doc;
        if (PRIVATE (a_this)->app_context)
                settings =
                        mlview_app_context_get_settings_hash_table
                        (PRIVATE (a_this)->app_context);
        if (settings
            && (expansion_depth_str =
                g_hash_table_lookup
                (settings,
                 MLVIEW_STG_K_DEFAULT_TREE_EXPANSION_LEVEL))) {
                expansion_depth = strtol
                        (expansion_depth_str, NULL, 10);
        }
        if ((expansion_depth == LONG_MIN)
            || (expansion_depth == LONG_MAX)) {
                expansion_depth = MLVIEW_DEFAULT_EXPANSION_DEPTH;
        }
        tree_view = build_tree_view_from_xml_doc
                (a_this, xml_doc);
        g_assert (tree_view != NULL);
        if (PRIVATE (a_this)->tree_view) {
                gtk_widget_destroy
                        (GTK_WIDGET (PRIVATE
                                   (a_this)->tree_view));
        }
        PRIVATE (a_this)->tree_view = tree_view;
        selection = gtk_tree_view_get_selection (tree_view);
        g_return_val_if_fail (selection, MLVIEW_ERROR);
        gtk_tree_selection_set_mode
                (selection, GTK_SELECTION_SINGLE);
        g_signal_connect (G_OBJECT (selection),
                          "changed",
                          G_CALLBACK (nodeset_selected_cb),
                          a_this);        
        scr_win = gtk_scrolled_window_new (NULL, NULL);
        gtk_container_add (GTK_CONTAINER (scr_win),
                           GTK_WIDGET (tree_view));
        gtk_box_pack_start (GTK_BOX (a_this), scr_win,
                            TRUE, TRUE, 0);
        gtk_widget_show_all (GTK_WIDGET (a_this));
        PRIVATE (a_this)->xml_doc = xml_doc;
        set_our_dnd_callbacks (a_this) ;
        gtk_tree_view_enable_model_drag_source 
                (tree_view, GDK_BUTTON1_MASK|GDK_BUTTON2_MASK,
                 row_targets, G_N_ELEMENTS (row_targets),
                 GDK_ACTION_MOVE | GDK_ACTION_COPY) ;
        gtk_tree_view_enable_model_drag_dest
                (tree_view, row_targets, 
                 G_N_ELEMENTS (row_targets),
                 GDK_ACTION_MOVE|GDK_ACTION_COPY) ;
        return 0;
}

/**
 *Creates a new xml document inside the tree editor.
 *@param a_this the current instance of #MlViewTreeEditor.
 *@param a_doc the mlview xml document to edit.
 */
void
mlview_tree_editor2_create_new_xml_doc (MlViewTreeEditor2 *
                                        a_this,
                                        MlViewXMLDocument *
                                        a_doc)
{
        g_return_if_fail (a_this && a_doc);
        mlview_tree_editor2_edit_xml_doc (a_this, a_doc, NULL);
}

/**
 *Set the root element of the visual xml doc.
 *Note that the xml doc must be empty ,that is, it must
 *not have any root element.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_node the new document root.
 *@param a_emit_signals if is TRUE, this function emits the
 *"node-added" and the "tree-changed" signals in case of successful
 *completion.
 */
void
mlview_tree_editor2_set_root_element (MlViewTreeEditor2 *a_this,
                                      xmlNode *a_node, 
                                      gboolean a_emit_signals)
{
        GtkTreeIter iter={0} ;
        xmlNode *node=NULL ;
        GtkTreeModel *model=NULL ;
        gboolean is_ok=TRUE ;
        GtkTreeRowReference *row_ref=NULL ;
        enum MlViewStatus status=MLVIEW_OK ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)
                          && PRIVATE (a_this)
                          && PRIVATE (a_this)->xml_doc
                          && PRIVATE (a_this)->tree_view
                          && a_node) ;
        node = xmlDocGetRootElement (PRIVATE (a_this)->xml_doc) ;
        g_return_if_fail (node == NULL) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_if_fail (model) ;
        xmlDocSetRootElement (PRIVATE (a_this)->xml_doc,
                              a_node) ;        
        is_ok = gtk_tree_model_get_iter_first (model, &iter) ;
        g_return_if_fail (is_ok == TRUE) ;
        status = build_tree_model_from_xml_tree (a_this, a_node, &iter, 
                                                 INSERT_TYPE_ADD_CHILD,
                                                 &model) ;        
        g_return_if_fail (status == MLVIEW_OK) ;        
        if (a_emit_signals == TRUE) {
                row_ref = g_hash_table_lookup 
                        (PRIVATE (a_this)->nodes_rows_hash, a_node) ;
                g_return_if_fail (row_ref) ;
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_ADDED],0,
                               row_ref) ;
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[TREE_CHANGED], 0) ;
        }
}

/**
 *Gets a tree iterator that points to the tree row associated
 *to a given xml node.
 *An example of use of this function is:
 *GtkTreeIter iter ;
 *enum MlViewStatus status ;
 *status = mlview_tree_editor2_get_iter (a_this,
 *                                       an_xml_node,
 *                                       &iter) ;
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_node the xml node to consider.
 *@param a_iter out parameter the place where to copy the
 *iterator. Must have been allocated by the caller.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_get_iter (MlViewTreeEditor2 * a_this,
                              xmlNode * a_node,
                              GtkTreeIter * a_iter)
{
        GtkTreeModel *model = NULL;
        GtkTreeRowReference *row_ref = NULL;
        GtkTreePath *tree_path = NULL;
        gboolean is_ok = FALSE;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->
                              nodes_rows_hash
                              && a_iter, MLVIEW_BAD_PARAM_ERROR);

        model = mlview_tree_editor2_get_model (a_this);
        g_return_val_if_fail (model, MLVIEW_ERROR);

        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash, a_node);
        if (!row_ref)
                return MLVIEW_NODE_NOT_FOUND_ERROR;
        tree_path = gtk_tree_row_reference_get_path (row_ref);
        g_return_val_if_fail (tree_path, MLVIEW_ERROR);
        is_ok = gtk_tree_model_get_iter (model, a_iter,
                                         tree_path);
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        if (is_ok == TRUE)
                return MLVIEW_OK;
        return MLVIEW_ERROR;
}

/**
 *Gets a row reference pointer to the first node
 *a user selection.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@return the row reference pointer.
 */
GtkTreeRowReference* 
mlview_tree_editor2_get_cur_sel_start (MlViewTreeEditor2 *a_this)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              NULL) ;

        return PRIVATE (a_this)->cur_sel_start ;
}

/**
 *Gets an iterator on the first element of a selection.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_iter out parameter, the iterator to fill. Must be allocated
 *by the caller.
 *@return MLVIEW_OK upon successful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_get_cur_sel_start_iter (MlViewTreeEditor2 *a_this,
                                            GtkTreeIter *a_iter)
{       
        GtkTreeModel *model=NULL ;
        GtkTreePath *tree_path=NULL ;
        gboolean is_ok=TRUE ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_iter,
                              MLVIEW_BAD_PARAM_ERROR) ;

        if (!PRIVATE (a_this)->cur_sel_start) {
                return MLVIEW_NODE_NOT_FOUND_ERROR ;
        }
        tree_path = gtk_tree_row_reference_get_path 
                (PRIVATE (a_this)->cur_sel_start) ;
        g_return_val_if_fail (tree_path, MLVIEW_ERROR) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        if (!model) {
                mlview_utils_trace_info ("model failed") ;
                status = MLVIEW_OK ;
                goto cleanup ;
        }
        is_ok = gtk_tree_model_get_iter (model, a_iter, tree_path) ;
        if (is_ok != TRUE) {
                mlview_utils_trace_info ("is_ok == TRUE failed") ;
                status = MLVIEW_OK ;
                goto cleanup ;
        }

 cleanup:
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        return status ;
}

/**
 *Gets the xml node associated to the current selected
 *Row.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@return the xml node or NULL if something bad happened.
 */
xmlNode * 
mlview_tree_editor2_get_cur_sel_xml_node (MlViewTreeEditor2 *a_this)
{
        GtkTreeIter iter = {0} ;
        enum MlViewStatus status = MLVIEW_OK ;
        xmlNode *result ;
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this), NULL) ;
        status = mlview_tree_editor2_get_cur_sel_start_iter (a_this, &iter) ;
        g_return_val_if_fail (status == MLVIEW_OK, NULL) ;
        result = mlview_tree_editor2_get_xml_node (a_this, &iter) ;
        return result ;
}

/**
 *Gets the instance of xmlNode* associated to
 *an instance of GtkTreeIter*
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_iter the iterator to consider.
 *@return the xmlNode associated to the iterator or NULL.
 */
xmlNode *
mlview_tree_editor2_get_xml_node (MlViewTreeEditor2 * a_this,
                                  GtkTreeIter * a_iter)
{
        GtkTreeModel *model = NULL;
        xmlNode *result = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_iter, NULL);

        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model, NULL);
        gtk_tree_model_get (model, a_iter,
                            XML_NODE_COLUMN, &result, -1);
        return result;
}

/**
 *Gets the instance of xmlNode* associated to
 *an instance of GtkTreeRowReference.
 *@param a_this the current instance of #MlViewTreeEditor.
 *@param a_iter the instance of GtkTreeRowReference to consider.
 *@return the xmlNode associated to the iterator or NULL.
 */
xmlNode *
mlview_tree_editor2_get_xml_node2 (MlViewTreeEditor2 * a_this,
                                   GtkTreeRowReference * a_row_ref)
{
        GtkTreeModel *model=NULL;
        GtkTreePath *tree_path=NULL ;
        xmlNode *result=NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_row_ref, NULL);

        model = mlview_tree_editor2_get_model (a_this);
        g_return_val_if_fail (model, NULL);
        tree_path = gtk_tree_row_reference_get_path (a_row_ref) ;
        g_return_val_if_fail (tree_path, NULL) ;
        result = mlview_tree_editor2_get_xml_node3 (a_this, tree_path) ;
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        return result;
}

/**
 *Gets the instance of xmlNode associated to an instance
 *of GtkTreePath.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_path the path to a graphical GtkTreeModel node.
 *@return the xml node associated to a_path or NULL.
 */
xmlNode *
mlview_tree_editor2_get_xml_node3 (MlViewTreeEditor2 *a_this,
                                   GtkTreePath *a_path)
{
        xmlNode *result = NULL ;
        GtkTreeModel *model = NULL ;
        gboolean is_ok = TRUE ;
        GtkTreeIter iter = {0} ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_path,
                              NULL) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model, NULL) ;
        is_ok=gtk_tree_model_get_iter (model, &iter, a_path) ;
        if (is_ok != TRUE) {
                mlview_utils_trace_info ("is_ok == TRUE failed") ;
                return NULL ;
        }
        gtk_tree_model_get (model, &iter,
                            XML_NODE_COLUMN, &result, -1);
        return result ;
}

/**
 *Adds a child node to the node pointed to by a_parent_iter.
 *@param a_this the current instance of #MlViewTreeEditor
 *@param a_parent_iter an iterator to the parent of the node
 *to be added.
 *@param a_node the xml node to add.
 *@return MVLIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_add_child_node (MlViewTreeEditor2 * a_this,
                                    GtkTreeIter * a_parent_iter,
                                    xmlNode * a_node)
{
        xmlNode *parent_xml_node = NULL,
                *added_node = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_parent_iter
                              && a_node, MLVIEW_BAD_PARAM_ERROR);

        parent_xml_node = mlview_tree_editor2_get_xml_node
                (a_this, a_parent_iter);
        g_return_val_if_fail (parent_xml_node, MLVIEW_ERROR);
        added_node = mlview_xml_document_add_child_node
                (PRIVATE (a_this)->mlview_xml_doc,
                 parent_xml_node, a_node, TRUE,
                 TRUE /*emit signal */ );
        if (added_node)
                return MLVIEW_OK;
        return MLVIEW_ERROR;
}

/**
 *Asks the user for the type of node he wants to add and adds it.
 *@param a_this the current instance of #MlViewTreeEditor2
 */
void
mlview_tree_editor2_add_child_node_interactive (MlViewTreeEditor2 *a_this)
{
        MlViewNodeTypePicker *picker=NULL ;
        gint button=0 ;
        xmlNode *cur_node=NULL ;
        
        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)
                          && PRIVATE (a_this)->cur_sel_start) ;

        cur_node = mlview_tree_editor2_get_xml_node2 
                (a_this, PRIVATE (a_this)->cur_sel_start) ;
        g_return_if_fail (cur_node) ;
        /*only an element node should have children*/
        if (cur_node->type != XML_ELEMENT_NODE)
                return ;
        picker = mlview_tree_editor2_get_node_type_picker (a_this) ;
        g_return_if_fail (picker) ;
        mlview_node_type_picker_set_title (picker, 
                                           _("add a child node")) ;
        mlview_node_type_picker_build_element_name_choice_list
                (picker, ADD_CHILD, cur_node) ;
        mlview_node_type_picker_select_node_name_or_content_entry_text 
                (picker) ;
        mlview_app_context_set_window_icon 
                (PRIVATE (a_this)->app_context, GTK_WINDOW (picker)) ;
        button = gtk_dialog_run (GTK_DIALOG (picker)) ;
        switch (button) {
        case GTK_RESPONSE_ACCEPT:
                handle_nt_picker_ok_button_clicked_to_add_child 
                        (a_this) ;
                break ;
        default:
                break ;
        }
        gtk_widget_hide 
                (GTK_WIDGET (PRIVATE (a_this)->node_type_picker)) ;
}

/**
 *Asks the user for the type of nodes she wants
 *to insert an inserts it.
 *@param a_this the current instance of #MlViewTreeEditor2
 */
void
mlview_tree_editor2_insert_prev_sibling_node_interactive (MlViewTreeEditor2 *a_this)
{
        MlViewNodeTypePicker *picker=NULL ;
        gint button=0 ;
        xmlNode *cur_node=NULL ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)
                          && PRIVATE (a_this)
                          && PRIVATE (a_this)->cur_sel_start) ;
        
        picker = mlview_tree_editor2_get_node_type_picker (a_this) ;
        g_return_if_fail (picker) ;
        mlview_node_type_picker_set_title
                (picker, _("insert a previous sibling node")) ;
        mlview_node_type_picker_select_node_name_or_content_entry_text
                (picker) ;
        /*
         *insert a flag to indicate that the picker is used
         *to insert node as previous node.
         *this is used by function
         *mlview_tree_editor_insert_prev_sibling_node_interactive().
         */
        g_object_set_data (G_OBJECT (a_this), "prev",
                           GINT_TO_POINTER (TRUE)) ;
        cur_node = mlview_tree_editor2_get_xml_node2 
                (a_this, PRIVATE (a_this)->cur_sel_start) ;
        g_return_if_fail (cur_node) ;
        mlview_node_type_picker_build_element_name_choice_list 
                (picker, INSERT_BEFORE, cur_node) ;
        mlview_app_context_set_window_icon 
                (PRIVATE (a_this)->app_context, 
                 GTK_WINDOW (picker)) ;
        button = gtk_dialog_run (GTK_DIALOG (picker)) ;
        switch (button) {
        case GTK_RESPONSE_ACCEPT:/*OK button*/
                handle_nt_picker_ok_button_clicked_to_insert_sibling_node
                        (a_this) ;                
                break ;
        default:
                break ;
        }
        gtk_widget_hide 
                (GTK_WIDGET (PRIVATE (a_this)->node_type_picker)) ;
}

/**
 *Asks the user for the type of node she wants
 *to insert and inserts it as a next sibling of the
 *currently selected node.
 *@param a_this the current instance of #MlViewTreeEditor.
 */
void
mlview_tree_editor2_insert_next_sibling_node_interactive (MlViewTreeEditor2 *a_this)
{
        MlViewNodeTypePicker *picker=NULL ;
        gint button=0 ;
        xmlNode *cur_node=NULL ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)
                          && PRIVATE (a_this)
                          && PRIVATE (a_this)->cur_sel_start) ;
        
        picker = mlview_tree_editor2_get_node_type_picker (a_this) ;
        g_return_if_fail (picker) ;
        mlview_node_type_picker_set_title 
                (picker, _("insert a next sibling node")) ;
        mlview_node_type_picker_select_node_name_or_content_entry_text
                (picker) ;
        /*
         *insert a flag to indicate that the picker is used
         *to insert node as next sibling node.
         *this is used by function
         *mlview_tree_editor_insert_next_sibling_node_interactive().
         */
        g_object_set_data (G_OBJECT (a_this), "prev",
                           GINT_TO_POINTER (FALSE)) ;
        cur_node = mlview_tree_editor2_get_xml_node2 
                (a_this, PRIVATE (a_this)->cur_sel_start) ;
        g_return_if_fail (cur_node) ;
        mlview_node_type_picker_build_element_name_choice_list 
                (picker, INSERT_BEFORE, cur_node) ;
        mlview_app_context_set_window_icon 
                (PRIVATE (a_this)->app_context, 
                 GTK_WINDOW (picker)) ;
        button = gtk_dialog_run (GTK_DIALOG (picker)) ;
        switch (button) {
        case GTK_RESPONSE_ACCEPT:/*OK button*/
                handle_nt_picker_ok_button_clicked_to_insert_sibling_node
                        (a_this) ;                
                break ;
        default:
                break ;
        }
        gtk_widget_hide 
                (GTK_WIDGET (PRIVATE (a_this)->node_type_picker)) ;
}

/**
 *Inserts a sibling node to the xml document.
 *@param a_this the current instance of #MlViewTreeEditor
 *@param a_ref_iter an iterator to the reference row of the visual
 *tree.
 *@param a_node the node to add before/previous the node
 *pointed to by a_ref_iter.
 *@param a_previous if set to TRUE, a_node is inserted before
 *the node pointed to by a_ref_iter, otherwise a_node is inserted
 *after the node pointed to by a_ref_iter.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_insert_sibling_node (MlViewTreeEditor2 *a_this,
                                         GtkTreeIter * a_ref_iter,
                                         xmlNode * a_node,
                                         gboolean a_previous)
{
        GtkTreeRowReference *row_ref = NULL;
        xmlNode *ref_node = NULL,
                *tmp_node = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->
                              nodes_rows_hash && a_node
                              && a_ref_iter,
                              MLVIEW_BAD_PARAM_ERROR);

        /*make sure a_node hasn't be drawn already */
        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash, a_node);
        g_return_val_if_fail (row_ref == NULL,
                              MLVIEW_BAD_PARAM_ERROR);

        ref_node = mlview_tree_editor2_get_xml_node (a_this,
                                                     a_ref_iter);
        g_return_val_if_fail (ref_node, MLVIEW_BAD_PARAM_ERROR);
        /*
         *call mlview_xml_document_insert_x_sibling_node()
         *to actualy insert the node.
         */
        if (a_previous == TRUE) {
                tmp_node =
                        mlview_xml_document_insert_prev_sibling_node
                        (PRIVATE (a_this)->mlview_xml_doc,
                         ref_node, a_node, TRUE, TRUE);
        } else {
                tmp_node =
                        mlview_xml_document_insert_next_sibling_node
                        (PRIVATE (a_this)->mlview_xml_doc,
                         ref_node, a_node, TRUE, TRUE);
        }
        g_return_val_if_fail (tmp_node == a_node, MLVIEW_ERROR);
        return MLVIEW_OK;
}

/**
 *Cuts the xml node associated to a given tree iterator.
 *Basically, this just cut the underlying xml node.
 *@param a_this the current instance of #MlViewTreeEditor.
 *@param a_iter an iterator to the node to cut.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_cut_node (MlViewTreeEditor2 * a_this,
                              GtkTreeIter * a_iter)
{
        xmlNode *node = NULL,
                *tmp_node = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && a_iter, MLVIEW_BAD_PARAM_ERROR);

        node = mlview_tree_editor2_get_xml_node (a_this, a_iter);
        g_return_val_if_fail (node, MLVIEW_ERROR);
        tmp_node = mlview_xml_document_cut_node
                (PRIVATE (a_this)->mlview_xml_doc, node, TRUE);
        g_return_val_if_fail (tmp_node == node, MLVIEW_ERROR);
        return MLVIEW_OK;
}

/**
 *Cuts the xml node associated to a given tree iterator.
 *Basically, this just cut the underlying xml node.
 *@param a_this the current instance of #MlViewTreeEditor.
 *@param a_pat a path to the node to cut.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_cut_node2 (MlViewTreeEditor2 * a_this,
                              GtkTreePath * a_path)
{
        GtkTreeIter iter = {0} ;
        GtkTreeModel *model = NULL ;
        gboolean is_ok = TRUE ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model, MLVIEW_ERROR) ;
        is_ok = gtk_tree_model_get_iter (model, &iter, a_path) ;
        g_return_val_if_fail (is_ok == TRUE, MLVIEW_ERROR) ;
        return mlview_tree_editor2_cut_node (a_this, &iter) ;        
}

/**
 *Copy the node pointed by an iterator to the clipboard.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_iter an iterator to the node to copy.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_copy_node (MlViewTreeEditor2 * a_this,
                               GtkTreeIter * a_iter)
{
        xmlNode *xml_node = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR);

        xml_node = mlview_tree_editor2_get_xml_node (a_this,
                                                     a_iter);
        g_return_val_if_fail (xml_node, MLVIEW_ERROR);
        mlview_xml_document_copy_node_to_clipboard
                (xml_node, PRIVATE (a_this)->xml_doc);
        return MLVIEW_OK;
}

/**
 *Copy the node pointed by a_path to the clipboard.
 *@param a_this the current instance of #MlViewTreeEditor.
 *@param a_path the path to the xml node to copy to the clipboard.
 *@return MLVIEW_OK upon successful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_copy_node2 (MlViewTreeEditor2 *a_this,
                                GtkTreePath *a_path)
{
        GtkTreeModel *model = NULL ;
        GtkTreeIter iter = {0} ;
        gboolean is_ok = TRUE ;
        
        g_return_val_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)
                          && a_path,
                          MLVIEW_BAD_PARAM_ERROR) ;

        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model, MLVIEW_ERROR) ;
        is_ok = gtk_tree_model_get_iter (model, &iter, a_path) ;
        g_return_val_if_fail (is_ok == TRUE, MLVIEW_ERROR) ;
        return mlview_tree_editor2_copy_node (a_this, &iter) ;
}

/**
 *Gets the last node put into the clipboard and inserts it into
 *the visual tree.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_parent_iter an iterator to the parent node.
 *@return MLVIEW_OK upon successful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_paste_node_as_child (MlViewTreeEditor2 *a_this,
                                         GtkTreeIter *a_parent_iter)
{
        xmlNode *parent_node=NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->mlview_xml_doc
                              && a_parent_iter,
                              MLVIEW_BAD_PARAM_ERROR) ;
        
        parent_node = mlview_tree_editor2_get_xml_node (a_this,
                                                        a_parent_iter) ;
        g_return_val_if_fail (parent_node, MLVIEW_NODE_NOT_FOUND_ERROR) ;
        mlview_xml_document_paste_node_as_child 
                (PRIVATE (a_this)->mlview_xml_doc, parent_node, TRUE) ;
        return MLVIEW_OK ;
}

/**
 *Gets the last node put in the clipboard and pastes
 *it as a sibling of the node pointed to by a_ref_iter.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_ref_iter an iterator that points to the
 *reference node.
 *@param a_previous if set to TRUE, the node must be pasted before
 *the reference node otherwise it is to be pasted after the reference 
 *node.
 *@return MLVIEW_OK upon sucessful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_paste_node_as_sibling (MlViewTreeEditor2 *a_this,
                                           GtkTreeIter *a_ref_iter,
                                           gboolean a_previous)
{
        xmlNode *sibling_node=NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->mlview_xml_doc
                              && a_ref_iter,
                              MLVIEW_BAD_PARAM_ERROR) ;

        sibling_node = mlview_tree_editor2_get_xml_node 
                (a_this, a_ref_iter) ;
        g_return_val_if_fail (sibling_node
                              && sibling_node->parent, 
                              MLVIEW_ERROR) ;
        mlview_xml_document_paste_node_as_sibling 
                (PRIVATE (a_this)->mlview_xml_doc,
                 sibling_node->parent, sibling_node, 
                 a_previous, TRUE) ;
        return MLVIEW_OK ;
}

/**
 *Gets the last node put in the clipboard and pastes
 *it as a sibling of the node pointed to by a_ref_iter.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_ref_path the path that points to the reference node.
 *@param a_previous if set to TRUE, the node must be pasted before
 *the reference node otherwise it is to be pasted after the reference 
 *node.
 *@return MLVIEW_OK upon sucessful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_paste_node_as_sibling2 (MlViewTreeEditor2 *a_this,
                                            GtkTreePath *a_ref_path,
                                            gboolean a_previous)
{
        GtkTreeModel *model = NULL ;
        GtkTreeIter iter = {0} ;
        gboolean is_ok = TRUE ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_ref_path,
                              MLVIEW_BAD_PARAM_ERROR) ;
        
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_val_if_fail (model, MLVIEW_ERROR) ;
        is_ok = gtk_tree_model_get_iter (model, &iter, a_ref_path) ;
        g_return_val_if_fail (is_ok == TRUE, MLVIEW_ERROR) ;
        return mlview_tree_editor2_paste_node_as_sibling 
                (a_this, &iter, a_previous) ;        
}

/**
 *Visualy Updates the tree editor
 *to make the addition of child node visible.
 *If the node addition has been already "updated",
 *this method does nothing.
 *@param a_this the current instance of #MlViewEditor.
 *@param a_parent the parent node of the newly added
 *instance of xmlNode.
 *@param a_node the newly added instance of xmlNode.
 *@return MLVIEW_OK upon successful completion, an error
 *code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_update_child_node_added (MlViewTreeEditor2 * a_this,
                                             xmlNode * a_parent,
                                             xmlNode * a_node, 
                                             gboolean a_emit_signals) 
{
        GtkTreeView *tree_view = NULL;
        GtkTreeRowReference *parent_row_ref = NULL,
                *row_ref = NULL;
        GtkTreeIter iter = { 0 };
        GtkTreeModel *model = NULL;
        GtkTreePath *tree_path = NULL;
        gboolean is_ok = FALSE;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail
                (a_this && MLVIEW_IS_TREE_EDITOR2 (a_this)
                 && PRIVATE (a_this), MLVIEW_BAD_PARAM_ERROR);
        /*
         *make sure the a_node hasn't been added to
         *the visual tree yet. row_ref must be NULL.
         */
        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash, a_node);
        if (row_ref)
                return MLVIEW_OK;
        /*only an element node should have children */
        g_return_val_if_fail
                (a_parent->type == XML_ELEMENT_NODE,
                 MLVIEW_BAD_PARAM_ERROR);
        tree_view = mlview_tree_editor2_get_tree_view (a_this);
        g_return_val_if_fail (tree_view != NULL, MLVIEW_ERROR);
        model = gtk_tree_view_get_model (tree_view);
        g_return_val_if_fail (model, MLVIEW_ERROR);
        /*
         *get the visual node that matches the a_parent_node
         *It must have been drawn already.
         */
        parent_row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash, a_parent);
        g_return_val_if_fail (parent_row_ref,
                              MLVIEW_NODE_NOT_FOUND_ERROR);
        /*get an iterator on the parent node */
        tree_path = gtk_tree_row_reference_get_path
                (parent_row_ref);
        g_return_val_if_fail (tree_path, MLVIEW_ERROR);
        is_ok = gtk_tree_model_get_iter (model, &iter,
                                         tree_path);
        if (is_ok != TRUE) {
                mlview_utils_trace_info ("is_ok == TRUE failed") ;
                status = MLVIEW_OK ;
                goto cleanup ;
        }
        /*
         *build a GtkTreeModel subtree that matches the a_node
         *subtree.
         */
        status = build_tree_model_from_xml_tree
                (a_this, a_node,
                 &iter, INSERT_TYPE_ADD_CHILD, &model);
        if (status != MLVIEW_OK) {
                mlview_utils_trace_info 
                        ("status ==  MVIEW_OK failed") ;
                goto cleanup ;                
        }
        status = mlview_tree_editor2_update_visual_node
                (a_this, &iter);
        /*emit the appropriate signals */
        if (a_emit_signals == TRUE) {
                row_ref = g_hash_table_lookup
                        (PRIVATE (a_this)->nodes_rows_hash,
                         a_node);
                if (!row_ref) {
                        mlview_utils_trace_info ("row_ref failed") ;
                        status = MLVIEW_ERROR ;
                        goto cleanup ;
                }
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_ADDED], 0,
                               row_ref);
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[TREE_CHANGED], 0);
        }
 cleanup:
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        return status;
}

/**
 *Updates the graphical tree to reflect a
 *"node pasted" event.
 *@param a_this the current instance of #MlViewTreeEditor
 *@param a_parent_node the parent node of the pasted node.
 *@param a_node the pasted node.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_update_node_pasted (MlViewTreeEditor2 *
                                        a_this,
                                        xmlNode * a_parent_node,
                                        xmlNode * a_node,
                                        gboolean a_emit_signals)
{
        GtkTreeRowReference *row_ref = NULL;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_parent_node
                              && a_node, MLVIEW_BAD_PARAM_ERROR);
        /*make sure a_parent_node is drawn */
        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash,
                 a_parent_node);
        g_return_val_if_fail (row_ref, MLVIEW_BAD_PARAM_ERROR);
        /*make sure a_node is not drawn */
        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash, a_node);
        g_return_val_if_fail (row_ref == NULL,
                              MLVIEW_BAD_PARAM_ERROR);
        status = mlview_tree_editor2_update_child_node_added
                (a_this, a_parent_node, a_node, FALSE);
        g_return_val_if_fail (status == MLVIEW_OK, status);
        if (a_emit_signals == TRUE) {
                row_ref = g_hash_table_lookup
                        (PRIVATE (a_this)->nodes_rows_hash,
                         a_node);
                g_return_val_if_fail (row_ref, MLVIEW_ERROR);
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_PASTED], 0,
                               row_ref);
        }
        return MLVIEW_OK;
}

/**
 *updates the visual tree node to reflect the
 *"sibling node inserted event."
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_ref_node the reference node.
 *@param a_inserted the node that has been inserted after
 *or before ref_node.
 *@param a_previous if set to TRUE, a_inserted_node
 *is to be inserted before a_ref_node, otherwise a_inserted_node
 *is to be inserted after a_ref_node.
 *@return MLVIEW_OK upon sucessfull completion, an error
 *code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_update_sibling_node_inserted (MlViewTreeEditor2 * a_this, 
                                                  xmlNode * a_ref_node, 
                                                  xmlNode * a_inserted_node, 
                                                  gboolean a_previous,
                                                  gboolean a_emit_signals) 
{
        GtkTreeRowReference *row_ref = NULL;
        GtkTreeModel *model = NULL;
        GtkTreeIter iter = { 0 };
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->
                              nodes_rows_hash && a_ref_node
                              && a_inserted_node,
                              MLVIEW_BAD_PARAM_ERROR);

        /*make sure ref_node has been drawn already */
        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash, a_ref_node);
        g_return_val_if_fail (row_ref, MLVIEW_BAD_PARAM_ERROR);
        /*make sure a_inserted_node hasn't been drawn already */
        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash,
                 a_inserted_node);
        g_return_val_if_fail (row_ref == NULL,
                              MLVIEW_BAD_PARAM_ERROR);
        status = mlview_tree_editor2_get_iter
                (a_this, a_ref_node, &iter);
        g_return_val_if_fail (status == MLVIEW_OK, status);
        model = mlview_tree_editor2_get_model (a_this);
        g_return_val_if_fail (model, MLVIEW_ERROR);
        if (a_previous == TRUE) {
                status = build_tree_model_from_xml_tree
                        (a_this, a_inserted_node,
                         &iter, INSERT_TYPE_INSERT_BEFORE,
                         &model);
                g_return_val_if_fail (status == MLVIEW_OK,
                                      status);
        } else {
                status = build_tree_model_from_xml_tree
                        (a_this, a_inserted_node,
                         &iter, INSERT_TYPE_INSERT_AFTER,
                         &model);
                g_return_val_if_fail (status == MLVIEW_OK,
                                      status);
        }
        status = mlview_tree_editor2_update_visual_node
                (a_this, &iter);
        if (status == MLVIEW_OK && a_emit_signals == TRUE) {
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_ADDED], 0,
                               row_ref);
        }
        return status;
}

/**
 *Updates the visual tree so that it reflects
 *a node cut event.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_parent_node the parent node of the xml node that
 *has been cut.
 *@param a_node_cut the xml node that has been cut.
 *@return MLVIEW_OK upon successful completion, an error
 *code otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_update_node_cut (MlViewTreeEditor2 * a_this,
                                     xmlNode * a_parent_node,
                                     xmlNode * a_node_cut)
{
        GtkTreeIter iter = { 0 };
        GtkTreeModel *model = NULL;
        GtkTreeRowReference *row_ref=NULL ;
        gboolean is_ok = TRUE;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_node_cut
                              && a_parent_node,
                              MLVIEW_BAD_PARAM_ERROR);
        g_return_val_if_fail (a_node_cut->parent == NULL
                              && a_parent_node,
                              MLVIEW_BAD_PARAM_ERROR);

        g_return_val_if_fail (a_node_cut, MLVIEW_ERROR);
        /*
         *make sure the a_parent_node and a_cut_node are still
         *referenced.
         */
        status = mlview_tree_editor2_get_iter
                (a_this, a_parent_node, &iter);
        if (status != MLVIEW_OK)
                return status;
        row_ref = g_hash_table_lookup 
                (PRIVATE (a_this)->nodes_rows_hash, a_node_cut);
        g_return_val_if_fail (row_ref, MLVIEW_ERROR) ;        
        status = mlview_tree_editor2_get_iter
                (a_this, a_node_cut, &iter);
        if (status != MLVIEW_OK)
                return status;
        model = mlview_tree_editor2_get_model (a_this);
        g_return_val_if_fail (model, MLVIEW_ERROR);
        is_ok = gtk_tree_store_remove
                (GTK_TREE_STORE (model), &iter);
        g_return_val_if_fail (is_ok == TRUE, MLVIEW_ERROR);        
        g_hash_table_remove
                (PRIVATE (a_this)->nodes_rows_hash, a_node_cut);
        gtk_tree_row_reference_free (row_ref) ;
        row_ref=NULL ;
        g_signal_emit (G_OBJECT (a_this),
                       gv_signals[NODE_CUT], 0, a_node_cut);
        g_signal_emit (G_OBJECT (a_this),
                       gv_signals[TREE_CHANGED], 0);
        return MLVIEW_OK;
}

/**
 *Update the tag string of a given visual node.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_iter an iterator that points to the node to be updated.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_update_visual_node (MlViewTreeEditor2 *a_this,
                                        GtkTreeIter * a_iter)
{
        guchar *start_tag_str = NULL;
        GtkTreeModel *model = NULL;
        xmlNode *xml_node = NULL;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && a_iter, MLVIEW_BAD_PARAM_ERROR);
        
        model = mlview_tree_editor2_get_model (a_this);
        g_return_val_if_fail (model, MLVIEW_ERROR);
        gtk_tree_model_get (model, a_iter, XML_NODE_COLUMN,
                            &xml_node, -1);
        start_tag_str = node_to_string_tag (xml_node);
        g_return_val_if_fail (start_tag_str, MLVIEW_ERROR);
        gtk_tree_store_set (GTK_TREE_STORE (model),
                            a_iter, FIRST_COLUMN, start_tag_str, -1);
        if (start_tag_str) {
                g_free (start_tag_str) ;
                start_tag_str = NULL ;
        }

        return MLVIEW_OK;
}

/**
 *Update the tag string of a given visual node.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_iter an iterator that points to the node to be updated.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_update_visual_node2 (MlViewTreeEditor2 *a_this,
                                         xmlNode * a_node)
{
        GtkTreeRowReference *row_ref = NULL;
        GtkTreePath *tree_path = NULL;
        GtkTreeIter iter = {0};
        GtkTreeModel *model = NULL;
        gboolean is_ok = FALSE;
        enum MlViewStatus status = MLVIEW_ERROR;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && a_node, MLVIEW_BAD_PARAM_ERROR);

        row_ref = g_hash_table_lookup
                (PRIVATE (a_this)->nodes_rows_hash, a_node);
        if (!row_ref) {
                return MLVIEW_NODE_NOT_FOUND_ERROR;
        }
        tree_path = gtk_tree_row_reference_get_path (row_ref);
        g_return_val_if_fail (tree_path, MLVIEW_ERROR);
        model = mlview_tree_editor2_get_model (a_this);
        if (!model) {
                mlview_utils_trace_info ("model failed") ;
                status = MLVIEW_ERROR ;
                goto cleanup ;
        }
        is_ok = gtk_tree_model_get_iter (model, &iter,
                                         tree_path);
        if (is_ok != TRUE) {
                mlview_utils_trace_info ("is_ok == TRUE failed") ;
                status = MLVIEW_ERROR ;
                goto cleanup ;
        }
        status = mlview_tree_editor2_update_visual_node
                (a_this, &iter);
 cleanup:
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
        return status;
}

/**
 *Displays a dialog box to let the user
 *type in search info and triggers the
 *search.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_search_interactive (MlViewTreeEditor2 *a_this)
{
        GtkWidget *find_dialog = NULL ;
        gint button = 0, loop = 1 ;

        struct SearchConfig search_config = {0} ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_this, MLVIEW_BAD_PARAM_ERROR) ;

        find_dialog = get_search_dialog (a_this) ;
        g_return_val_if_fail (find_dialog
                              && GTK_IS_DIALOG (find_dialog),
                              MLVIEW_ERROR) ;
        while (loop) {
                button = gtk_dialog_run (GTK_DIALOG (find_dialog)) ;
                switch (button) {
                case 0:/*next button*/
                case 1:/*prev button*/
                        status = get_search_config
                                (GTK_DIALOG (find_dialog),
                                 &search_config) ;
                        g_return_val_if_fail (status == MLVIEW_OK,
                                              status) ;
                        if (button == 0) {
                                search_config.downward = TRUE ;
                        } else {
                                search_config.downward = FALSE ;
                        }
                        mlview_tree_editor2_search
                                (a_this, 
                                 PRIVATE (a_this)->cur_sel_start,
                                 &search_config) ;
                        break ;
                        
                case 2:/*cancel button*/
                default:
                        loop = 0 ;/*get out of the loop*/
                        break ;
                }
        }
        gtk_widget_hide (GTK_WIDGET (find_dialog)) ;
        return MLVIEW_OK ;
}

/**
 *Calls the search backend on the current instance
 *of #MlViewXMLDocument.
 *@param a_this the current instance of #MlViewTreeEditor2
 *@param a_from the row to start the search from.
 *@param a_config the search configuration .
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_editor2_search (MlViewTreeEditor2 *a_this,
                            GtkTreeRowReference *a_from,
                            struct SearchConfig *a_config)
{
        xmlNode *xml_node = NULL, *node_found = NULL ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_EDITOR2 (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->mlview_xml_doc,
                              MLVIEW_OK) ;

        if (PRIVATE (a_this)->cur_sel_start) {
                xml_node = mlview_tree_editor2_get_xml_node2 
                        (a_this, a_from) ;
                g_return_val_if_fail (xml_node, MLVIEW_ERROR) ;
        }
        status = mlview_xml_document_search
                (PRIVATE (a_this)->mlview_xml_doc, a_config,
                 xml_node, &node_found, TRUE) ;
        return status ;
}

/**
 *Selects the visual node that matches the xml node.
 *@param a_this the current instance of #MlViewTreeEditor2.
 *@param a_node the node to select.
 */
void
mlview_tree_editor2_select_node (MlViewTreeEditor2 *a_this,
                                 xmlNode *a_node)
{
        enum MlViewStatus status = MLVIEW_OK ;
        GtkTreePath *tree_path = NULL ;
        GtkTreeIter iter = {0} ;
        GtkTreeModel * model = NULL ;
        GtkTreeView *tree_view = NULL ;
        GtkTreeSelection *tree_sel = NULL ;

        status = mlview_tree_editor2_get_iter (a_this, a_node,
                                               &iter) ;
        g_return_if_fail (status == MLVIEW_OK) ;
        model = mlview_tree_editor2_get_model (a_this) ;
        g_return_if_fail (model) ;
        tree_view = mlview_tree_editor2_get_tree_view (a_this) ;
        g_return_if_fail (tree_view) ;
        tree_path = gtk_tree_model_get_path (model, &iter) ;
        g_return_if_fail (tree_path) ;
        tree_sel = gtk_tree_view_get_selection (tree_view) ;
        g_return_if_fail (tree_sel) ;
        gtk_tree_view_expand_to_path (tree_view, tree_path) ;
        gtk_tree_selection_select_path (tree_sel, tree_path);
        if (tree_path) {
                gtk_tree_path_free (tree_path) ;
                tree_path = NULL ;
        }
}

/**
 *Expands the current selected visual row to the depth
 *a_depth. If a_depth is set to -1 the tree is expanded to the
 *leaves.
 *@param a_this the current instance of #MlViewTreeEditor.
 *@param a_depth the expansion depth.
 */
void
mlview_tree_editor2_expand_tree_to_depth (MlViewTreeEditor2 * a_this,
                                         gint a_depth)
{
        GtkTreeRowReference *cur_row_ref=NULL ;
        GtkTreePath *cur_path=NULL ;
        GtkTreeView *tree_view=NULL ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_EDITOR2 (a_this)) ;
        cur_row_ref = PRIVATE (a_this)->cur_sel_start ;
        g_return_if_fail (cur_row_ref) ;
        cur_path = gtk_tree_row_reference_get_path (cur_row_ref) ;
        g_return_if_fail (cur_path) ;
        tree_view = mlview_tree_editor2_get_tree_view (a_this) ;
        if (!tree_view) {
                mlview_utils_trace_info ("tree_view failed") ;
                goto cleanup ;
        }
        status = mlview_utils_gtk_tree_view_expand_row_to_depth 
                        (tree_view, cur_path, a_depth) ;
        if (status != MLVIEW_OK) {
                mlview_utils_trace_info 
                        ("status == MLVIEW_OK failed.") ;
        }
 cleanup:
        if (cur_path) {
                gtk_tree_path_free (cur_path) ;
                  cur_path = NULL ;
        }
}

#endif
