import sys, logging, urlparse, gnome

import gobject, gtk, dbus

import gutil
from ddm import DataModel

_logger = logging.getLogger("weblogindriver.AccountsDialog")

class Dialog(gtk.Window):      
    def __init__(self, account_types, online_accounts_service, *args, **kwargs):
        super(Dialog, self).__init__(*args, **kwargs)        
        
        self.__online_accounts_service = online_accounts_service

        # we need the proxy as well to connect to the signals
        try:
            self.__onlineaccounts_proxy = dbus.SessionBus().get_object('org.gnome.OnlineAccounts', '/onlineaccounts')
        except dbus.DBusException, e:
            _logger.error("onlineaccounts DBus service not available, can't manage accounts")
            return

        self.__account_types = account_types
        self.set_title('Accounts')
        self.set_position(gtk.WIN_POS_CENTER)

        self.__notebook = gtk.Notebook()
        self.__notebook.set_border_width(5)

        self.__outer_existing_accounts_vbox = gtk.VBox() 
        self.__outer_existing_accounts_vbox.set_border_width(5)

        self.__outer_new_account_vbox = gtk.VBox() 
        self.__outer_new_account_vbox.set_border_width(5)

        # make tabs take an equal amount of space
        self.__notebook.set_property("homogeneous", True)
        self.__notebook.append_page(self.__outer_existing_accounts_vbox, gtk.Label("Existing Accounts"))
        self.__notebook.append_page(self.__outer_new_account_vbox, gtk.Label("Add Account"))
        # make tabs expand horizontally
        self.__notebook.set_tab_label_packing(self.__outer_new_account_vbox, True, True, gtk.PACK_START)

        self.__existing_accounts_vbox = gtk.VBox()
        self.__outer_existing_accounts_vbox.pack_start(self.__existing_accounts_vbox, False, False) 
        self.__new_account_vbox = gtk.VBox() 
        self.__outer_new_account_vbox.pack_start(self.__new_account_vbox, False, False) 

        self.add(self.__notebook)

        self.connect('delete-event', self.__on_delete_event)
        self.connect('destroy', self.__on_destroy)

        # stuff below gets added to the "Existing Accounts" tab

        # self.__no_accounts_label = gtk.Label("No existing accounts")
        # self.__no_accounts_label.size_allocate(gtk.gdk.Rectangle(width=0, height=0)) 
        # self.__existing_accounts_vbox.pack_start(self.__no_accounts_label)

        self.__model = gtk.ListStore(gobject.TYPE_STRING)
        self.__accounts_combo = gtk.ComboBox(self.__model)
        self.__existing_accounts_vbox.pack_start(self.__accounts_combo, True, False)
        textrender = gtk.CellRendererText()
        self.__accounts_combo.pack_start(textrender, True)
        self.__accounts_combo.add_attribute(textrender, 'text', 0)
        self.__accounts_combo.connect('notify::active', self.__on_edited_account_changed)

        # TODO: don't display the dropdown and password entry box if there are no accounts to edit 
        self.__password_entry = gtk.Entry()
        self.__password_entry.set_visibility(False) # this sets a password mode
        self.__password_entry.connect('changed',
                                      self.__on_account_settings_changed)
        self.__password_entry.connect('activate',
                                      self.__on_account_settings_applied)

        password_vbox = gtk.VBox(spacing=2)
        self.__existing_accounts_label = gtk.Label()
        password_vbox.pack_start(self.__existing_accounts_label, True, False)    
        password_vbox.pack_end(self.__password_entry, False, False)
        self.__existing_accounts_vbox.pack_start(password_vbox, padding=5)

        box_with_buttons = gtk.HBox()         
        self.__check_box = gtk.CheckButton(label="Enabled")
        box_with_buttons.pack_start(self.__check_box, False, False, padding=5)  
        self.__existing_accounts_vbox.pack_start(box_with_buttons, False, False, padding=5)

        self.__check_box.connect('toggled',
                                 self.__on_account_settings_changed)

        self.__undo_button = gtk.Button(stock=gtk.STOCK_UNDO)
        self.__undo_button.connect('clicked',
                                   self.__on_account_settings_reset)
        self.__undo_button.set_sensitive(False)                     
        self.__undo_button.size_allocate(gtk.gdk.Rectangle(width=100))               

        self.__apply_button = gtk.Button(stock=gtk.STOCK_APPLY)
        self.__apply_button.connect('clicked',
                                   self.__on_account_settings_applied)
        self.__apply_button.set_sensitive(False)

        box_with_buttons.set_spacing(5)
        box_with_buttons.set_homogeneous(False)
        box_with_buttons.pack_end(self.__apply_button, expand=False, fill=False)
        box_with_buttons.pack_end(self.__undo_button, expand=False, fill=False)

        remove_button_box = gtk.HBox()
        self.__remove_button = gtk.Button(label="Remove Account")
        remove_image = gtk.Image()
        remove_image.set_from_stock(gtk.STOCK_REMOVE, gtk.ICON_SIZE_BUTTON)
        self.__remove_button.set_image(remove_image) 
        self.__remove_button.set_sensitive(False)
        self.__remove_button.connect('clicked',
                                   self.__on_account_remove_clicked)
        remove_button_box.pack_end(self.__remove_button, expand=False, fill=False)

        self.__remove_link_box = gtk.HBox()
        self.__remove_link = Link()
        self.__remove_link.set_text("Remove account online")
        self.__remove_link.set_enabled(False)
        self.__remove_link.connect("clicked", self.__open_account_page)
        self.__remove_link_box.pack_end(self.__remove_link, expand=False, fill=False)

        self.__existing_accounts_vbox.pack_start(remove_button_box, False, False, padding=5)
        self.__existing_accounts_vbox.pack_start(self.__remove_link_box, False, False)   

        # stuff below gets added to the "Add Account" tab

        self.__account_types_model = gtk.ListStore(gobject.TYPE_STRING)
        self.__account_types_combo = gtk.ComboBox(self.__account_types_model)
        self.__new_account_vbox.pack_start(self.__account_types_combo, True, False)
        account_types_textrender = gtk.CellRendererText()
        self.__account_types_combo.pack_start(account_types_textrender, True)
        self.__account_types_combo.add_attribute(account_types_textrender, 'text', 0)
        
        _logger.debug("will get accounts types")
        self.__all_account_types = self.__online_accounts_service.GetAllAccountTypes()
        _logger.debug("will get accounts types 2")
     
        # TODO: once we save accounts added locally on server, we will allow adding all account types
        # in the dialog, but for now we don't allow adding any
        # for type_full_name in self.__all_account_types.values():
        #     self.__account_types_model.append([type_full_name])
            
        # Setting padding to 1 here is a hack to get the content of both tabs to be aligned,
        # we'll need to change this when it will become possible to add new account types to
        # the existing dialog, for example, by adding stocks that require these account types
        padding = 1  
        if len(self.__account_types_model) > 0:
            self.__account_types_combo.set_active(0) 
            padding = 5

        self.__username_entry = gtk.Entry()
        self.__username_entry.connect('changed',
                                      self.__on_new_username_changed)
        self.__username_entry.connect('activate',
                                      self.__on_new_username_added)

        username_vbox = gtk.VBox(spacing=2)
        self.__username_label = gtk.Label("Username:")
        self.__username_label.set_alignment(0.0, 0.5)
        username_vbox.pack_start(self.__username_label, False, False)    
        username_vbox.pack_end(self.__username_entry, False, False)
        self.__new_account_vbox.pack_start(username_vbox, padding=padding)

        self.__add_button_box = gtk.HBox()
        self.__add_button = gtk.Button(label="Add Account")
        add_image = gtk.Image()
        add_image.set_from_stock(gtk.STOCK_ADD, gtk.ICON_SIZE_BUTTON)
        self.__add_button.set_image(add_image) 
        self.__add_button.set_sensitive(False)
        self.__add_button.connect('clicked',
                                   self.__on_new_username_added)
        self.__add_button_box.pack_end(self.__add_button, expand=False, fill=False)
        self.__new_account_vbox.pack_start(self.__add_button_box, False, False, padding=padding)

        self.__add_link_box = gtk.HBox()
        self.__add_link = Link()
        self.__add_link.connect("clicked", self.__open_account_page)
        if len(self.__account_types_model) > 0:
            self.__account_types_combo.set_active(0)
            self.__add_link.set_text("Add other accounts online")
            self.__add_link_box.pack_end(self.__add_link, expand=False, fill=False)
            self.__outer_new_account_vbox.pack_end(self.__add_link_box, False, False)
        else:  
            if self.__account_types and len(self.__account_types) == 1 and \
               self.__account_types[0] in self.__all_account_types.keys():           
                self.__add_link.set_text("Add " + self.__all_account_types[self.__account_types[0]] + " accounts online")
            else:
                self.__add_link.set_text("Add accounts online")
            self.__add_link_box.pack_start(self.__add_link)
            self.__new_account_vbox.pack_start(self.__add_link_box)
   
        self.__model_tree_iter_by_account = {}
        self.__current_account = None

        self.show_all()
        self.__set_no_accounts_state(True)
        
        if len(self.__account_types_model) == 0:
            self.__account_types_combo.hide()
            self.__username_label.hide()
            self.__username_entry.hide()
            self.__add_button.hide()
 
        self.__notebook.set_current_page(1)

        # google_accounts = self.__online_accounts_service.GetAllAccountsWithTypes(["google"])
        all_account_paths = self.__online_accounts_service.GetAllAccounts()
        if len(all_account_paths) == 0:
            # TODO: do something else
            pass
        else:
            for a_path in all_account_paths:
                self.__on_account_added(a_path)

        self.__connections = gutil.DisconnectSet()

        id = self.__onlineaccounts_proxy.connect_to_signal('AccountAdded',
                                                           self.__on_account_added)
        self.__connections.add(self.__onlineaccounts_proxy, id)
        id = self.__onlineaccounts_proxy.connect_to_signal('AccountRemoved',
                                                           self.__on_account_removed)
        self.__connections.add(self.__onlineaccounts_proxy, id)

    def __open_account_page(self, l):
        gnome.url_show(urlparse.urljoin(get_baseurl(), "/account"))
      
    def __on_account_added(self, a_path):
        # TODO: check for account type when we have a per-account dialog
        if a_path not in self.__model_tree_iter_by_account:
            a = self.__online_accounts_service.get_existing_account(a_path)
            if a is None:
                _logger.error("onlineaccount for path %s was not found" % a_path)
                return

            if self.__account_types and a.GetType() not in self.__account_types:
                return 

            _logger.debug("account added %s" % a) 
            tree_iter = self.__model.append([self.__all_account_types[a.GetType()] + ': ' + a.GetUsername()])
                
            self.__model_tree_iter_by_account[a_path] = tree_iter

            if len(self.__model_tree_iter_by_account) == 1:
                self.__set_no_accounts_state(False)
                self.__current_account = a
                self.__accounts_combo.set_active(0) 
                self.__notebook.set_current_page(0)     
     
    def __on_account_removed(self, a_path):
        # we don't want to remove accounts that were disabled, but are still present in GConf here
        # if accts.has_account(a):
        #     return 
 
        _logger.debug("account removed %s" % a_path) 
        if a_path in self.__model_tree_iter_by_account:
            _logger.debug("will remove")
            self.__model.remove(self.__model_tree_iter_by_account[a_path])
            del self.__model_tree_iter_by_account[a_path]
            # TODO: possibly can get the index of the deleted selection, and move to the next one 
            if self.__current_account.GetObjectPath() == a_path and len(self.__model_tree_iter_by_account) > 0:  
                new_current_account_path = self.__model_tree_iter_by_account.keys()[0]
                self.__current_account = self.__online_accounts_service.get_existing_account(new_current_account_path)
                if self.__current_account is not None:   
                    self.__accounts_combo.set_active_iter(self.__model_tree_iter_by_account.values()[0]) 
                else:
                    _logger.error("onlineaccount for path %s was not found" % new_current_account_path)
   
        if len(self.__model_tree_iter_by_account) == 0:   
            self.__set_no_accounts_state(True)

    def __set_no_accounts_state(self, no_accounts):
        if no_accounts:
            self.__accounts_combo.hide()
            self.__password_entry.hide()
            self.__check_box.hide()
            self.__undo_button.hide()
            self.__apply_button.hide()
            self.__remove_button.hide()
            self.__remove_link_box.hide()
            self.__existing_accounts_label.set_text("No existing accounts") 
            self.__existing_accounts_label.set_alignment(0.5, 0.0)
        else:
            self.__existing_accounts_vbox.show_all()
            self.__existing_accounts_label.set_text("Password:")
            self.__existing_accounts_label.set_alignment(0.0, 0.5)    

    def __on_edited_account_changed(self, *args):
        new_iter = self.__accounts_combo.get_active_iter()
        _logger.debug("account changed to %s" % self.__model.get_value(new_iter, 0))        
        for (a_path, tree_iter) in self.__model_tree_iter_by_account.items():
            # this assumes that text for each entry in the drop down box is different, which it should be       
            if self.__model.get_value(tree_iter, 0) == self.__model.get_value(new_iter, 0):              
                self.__current_account =  self.__online_accounts_service.get_existing_account(a_path)
                if self.__current_account is None:
                    _logger.error("onlineaccount for path %s was not found" % a_path)
                    return
                # TODO: this will trigger __on_password_entry_changed, make sure that is harmless
                self.__password_entry.set_text(self.__current_account.GetPassword())
                self.__check_box.set_active(self.__current_account.GetEnabled())
                # TODO: set the remove button to be sensitive once we enable removing accounts on server
                self.__remove_button.set_sensitive(False) 
                self.__remove_link.set_enabled(True)
                return
        _logger.error("new edited account was not found in self.__model_tree_iter_by_account")

    def __on_account_settings_changed(self, widget):
        if self.__password_entry.get_text().strip() != self.__current_account.GetPassword() or \
           self.__check_box.get_active() != self.__current_account.GetEnabled():
            self.__undo_button.set_sensitive(True)  
            self.__apply_button.set_sensitive(True)  
        else:
            self.__undo_button.set_sensitive(False)  
            self.__apply_button.set_sensitive(False)  

    def __on_account_settings_applied(self, widget):
        text = self.__password_entry.get_text()
        enabled = self.__check_box.get_active()
        self.__online_accounts_service.SaveAccountChanges(self.__current_account.GetObjectPath(),
                                                       { 'password' : str(text), 'enabled' : str(enabled) })
        self.__undo_button.set_sensitive(False)  
        self.__apply_button.set_sensitive(False)  

    def __on_account_settings_reset(self, widget):
        self.__password_entry.set_text(self.__current_account.GetPassword())
        self.__check_box.set_active(self.__current_account.GetEnabled())
  
    def __on_account_remove_clicked(self, widget):
        self.__online_accounts_service.RemoveAccount(self.__current_account.GetObjectPath())

    def __on_new_username_changed(self, widget):
        # in the future can check if the input is of the desired form here
        username_entered = (len(self.__username_entry.get_text().strip()) > 0)  
        self.__add_button.set_sensitive(username_entered)        

    def __on_new_username_added(self, widget):
        text = self.__username_entry.get_text()
        username_entered = (len(text.strip()) > 0)
        # don't do anything if the user pressed enter in the username textbox while it is empty 
        if not username_entered:
            return
        
        current_iter = self.__account_types_combo.get_active_iter()
        for (account_type, type_full_name) in self.__all_account_types:
            if self.__account_types_model.get_value(current_iter, 0) == type_full_name:
                account_tuple = self.__online_accounts_service.GetOrCreateAccount(account_type, text)
                self.__username_entry.set_text("")
                self.__notebook.set_current_page(0)
                # TODO: Display a special message if the account aready existed (account_tuple[1] is False), also
                # the key should be self.__model_tree_iter_by_account only in that case
                if self.__model_tree_iter_by_account.has_key(account_tuple[0]):
                    account_iter = self.__model_tree_iter_by_account[account_tuple[0]]                
                    self.__accounts_combo.set_active_iter(account_iter)
                else:
                    # acounts system will emit a signal that will cause __on_account_added to be called again, 
                    # but it's ok to call this from here so that we can switch to the new account in the combo box right away
                    self.__on_account_added(account_tuple[0])
                    account_iter = self.__model_tree_iter_by_account[account_tuple[0]]   
                    self.__accounts_combo.set_active_iter(account_iter)
                return 
                
        _logger.warn("Did not find an account type that matched the account type in the dropdown %s" % self.__account_types_model.get_value(current_iter, 0))

    def __on_delete_event(self, dialog, event):
        self.hide()
        return True

    def __on_destroy(self, dialog):
        self.hide()
        self.__connections.disconnect_all()
        
class Link(gtk.EventBox):
    __gsignals__ = {
        "clicked" : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, ()),
    }
    
    def __init__(self,**kwargs):
        super(Link, self).__init__(**kwargs)
        self.__text = None
        self.__enabled = True
        self.__label = gtk.Label()
        self.add(self.__label)
        self.connect("button-press-event", self.__on_button_press)
        self.connect("enter_notify_event", self.__on_enter) 
        self.connect("leave_notify_event", self.__on_leave) 
        self.add_events(gtk.gdk.BUTTON_PRESS_MASK
                        & gtk.gdk.ENTER_NOTIFY_MASK
                        & gtk.gdk.LEAVE_NOTIFY_MASK)

    def set_alignment(self, x, y):
        self.__label.set_alignment(x, y)
        
    def set_ellipsize(self, do_ellipsize):
        self.__label.set_ellipsize(do_ellipsize)

    def __on_button_press(self, self2, e):
        if e.button == 1 and self.__enabled:
            self.emit("clicked")
            return True
        return False

    def get_text(self):
        return self.__text
    
    def set_text(self, text):
        self.__text = text
        self.set_markup(gobject.markup_escape_text(text))
    
    def set_enabled(self, enabled):        
        self.__enabled = enabled
        self.set_markup(gobject.markup_escape_text(self.__text))

    def set_markup(self, text):
        if  self.__enabled: 
            self.__label.set_markup('<span foreground="blue">%s</span>' % (text,))
        else:
            self.__label.set_markup('<span foreground="#666666">%s</span>' % (text,))

    def __on_enter(self, w, c):
        self.__talk_to_the_hand(True)

    def __on_leave(self, w, c):
        self.__talk_to_the_hand(False)

    def __talk_to_the_hand(self, hand):
        display = self.get_display()
        cursor = None
        if hand and self.__enabled:
            cursor = gtk.gdk.Cursor(display, gtk.gdk.HAND2)
        self.window.set_cursor(cursor)

def get_baseurl():
    ## first we prefer the base url from the model we're actually using.
    ## this happens when we just connect to "org.freedesktop.od.Engine"
    ## and don't know in advance whether a dogfood or production or whatever
    ## server instance owns that bus name.
    ## Note that this is _supposed_ to work offline as well - the od.Engine
    ## is supposed to have an offline mode.
    model = DataModel()
    if model.global_resource:
        try:
            return model.global_resource.webBaseUrl
        except AttributeError:
            pass

    ## we fall back to a hardcoded URL, since it's probably better
    ## than crashing and would normally be right in production, but never
    ## right when testing on dogfood.
    return "http://online.gnome.org"

