/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <libgnomeui/libgnomeui.h>

#include <pan/base/debug.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/gui.h>
#include <pan/prefs.h>
#include <pan/queue.h>
#include <pan/task-manager.h>
#include <pan/util.h>

#include <pan/xpm/delete_row.xpm>
#include <pan/xpm/delete_sheet.xpm>
#include <pan/xpm/server.xpm>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

typedef struct
{
	GtkWidget    * window;
	GtkWidget    * top;
	GtkWidget    * list;

	GtkWidget    * pause_pixmap;
	GtkWidget    * ready_pixmap;

	guint          timeout_id;
	guint          moved_signal_id;

	gboolean       titlebar_needs_refresh;
	gboolean       dampen_move_feedback_loop;
}
ManagerUI;

/*********************
**********************  Variables
*********************/

extern GtkTooltips * ttips;

static ManagerUI * manager_ui = NULL;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE ROUTINES
************/

/**
***  Internal Utility
**/

static GPtrArray*
get_selected_tasks (ManagerUI * ui)
{
	GPtrArray * a = g_ptr_array_new ();
	GtkCList *clist = GTK_CLIST(ui->list);
        GList * l;
	for (l=clist->selection; l!=NULL; l=l->next)
	{
		const int index = GPOINTER_TO_INT (l->data);
		Task * const task = gtk_clist_get_row_data (clist, index);
		g_ptr_array_add (a, task);
	}
	return a;
}

/**
*** Helper functions for user interactions
**/

static void
move_selected_tasks_to_top (ManagerUI * ui)
{
	GtkCList * clist;
	GList * l;
	gint new_row;
	debug_enter ("move_selected_tasks_to_top");

	/* sanity clause */
	g_return_if_fail (ui!=NULL);

	clist = GTK_CLIST(ui->list);
	new_row = -1;
	for (l=clist->selection; l!=NULL; l=l->next)
	{
		const int old_row = GPOINTER_TO_INT (l->data);
		Task * const task = gtk_clist_get_row_data (clist, old_row);
		queue_task_move (task, ++new_row);
	}

	debug_exit ("move_selected_tasks_to_top");
}

static void
move_selected_tasks_to_bottom (ManagerUI * ui)
{
	GtkCList * clist;
	GList * l;
	gint new_row;
	debug_enter ("move_selected_tasks_to_bottom");

	/* sanity clause */
	g_return_if_fail (ui!=NULL);

	clist = GTK_CLIST(ui->list);
	new_row = 0;
	for (l=clist->selection; l!=NULL; l=l->next)
	{
		const int old_row = GPOINTER_TO_INT (l->data);
		Task * const task = gtk_clist_get_row_data (clist, old_row);
		queue_task_move (task, clist->rows - ++new_row);
	}

	debug_exit ("move_selected_tasks_to_bottom");
}

static void
cancel_selected_tasks (ManagerUI * ui)
{
	GPtrArray * a;
        debug_enter ("cancel_selected_tasks");

	/* sanity clause */
	g_return_if_fail (ui!=NULL);

	a = get_selected_tasks (ui);
	pan_g_ptr_array_foreach (a, (GFunc)queue_task_remove, NULL);
	g_ptr_array_free (a, TRUE);

        debug_exit ("cancel_selected_tasks");
}

static void
select_task (ManagerUI * ui, gboolean first)
{
	GtkCList * clist;
	gint       row;
	debug_enter ("select_task");

	/* sanity clause */
	g_return_if_fail (ui!=NULL);

	clist = GTK_CLIST(ui->list);
	row = first ? 0 : clist->rows-1;
	
	gtk_clist_freeze(clist);
	gtk_clist_unselect_all (clist);
	gtk_clist_select_row (clist, row, 0);
	clist->focus_row = row;
	gtk_clist_thaw(clist);

	debug_exit("select_task");
}


/**
***  User Interactions
**/

static void
decorate_pause_button (ManagerUI * ui, GtkToggleButton * tb, gboolean paused)
{
	GtkWidget * w;

	/* update pixmap */
	w = GTK_BIN(tb)->child;
	if (w != NULL)
		gtk_container_remove (GTK_CONTAINER(tb), w);
	gtk_container_add (GTK_CONTAINER(tb), paused ? ui->pause_pixmap : ui->ready_pixmap);
	gtk_widget_show_all (GTK_WIDGET(tb));

	/* update tooltip */
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), GTK_WIDGET(tb), paused
		? _("The Queue is Paused")
		: _("The Queue is not Paused"), "");
}
static void
pause_toggled_cb (GtkToggleButton  * tb,
                  gpointer           user_data)
{
	ManagerUI * ui = (ManagerUI*) user_data;
	gboolean paused = gtk_toggle_button_get_active (tb);

	if (paused != queue_is_paused())
		queue_set_paused (paused);

	decorate_pause_button (ui, tb, paused);
}

static void
up_clicked_cb (GtkButton   * button,
               gpointer      user_data)
{
        ManagerUI * ui;
	GtkCList * clist;
	GList * l;
	gint last_index;
	debug_enter ("up_clicked_cb");

        ui = (ManagerUI*) user_data;
	clist = GTK_CLIST(ui->list);
	last_index = -1;
	for (l=clist->selection; l!=NULL; l=l->next)
	{
		const int index = GPOINTER_TO_INT (l->data);
		Task * const task = gtk_clist_get_row_data (clist, index);
		int new_index = MAX (last_index+1, index-1);
		queue_task_move (task, new_index);
		last_index = new_index;
	}

	debug_exit ("up_clicked_cb");
}

static void
top_clicked_cb (GtkButton    * button,
                gpointer       user_data)
{
	debug_enter ("top_clicked_cb");

	move_selected_tasks_to_top ((ManagerUI *) user_data);

	debug_exit ("top_clicked_cb");
}

static void
down_clicked_cb (GtkButton   * button,
                 gpointer      user_data)
{
	ManagerUI * ui;
	GtkCList * clist;
	GList * l;
	debug_enter ("down_clicked_cb");

	ui = (ManagerUI*) user_data;
	clist = GTK_CLIST(ui->list);
	for (l=g_list_last(clist->selection); l!=NULL; l=l->prev)
	{
		const int index = GPOINTER_TO_INT (l->data);
		Task * const task = gtk_clist_get_row_data (clist, index);
		queue_task_move (task, index+1);
	}

	debug_exit ("down_clicked_cb");
}

static void
bottom_clicked_cb (GtkButton   * button,
                   gpointer      user_data)
{
	debug_enter ("bottom_clicked_cb");

	move_selected_tasks_to_bottom ((ManagerUI *) user_data);

	debug_exit ("bottom_clicked_cb");
}

static void
cancel_clicked_cb (GtkButton   * button,
                   gpointer      user_data)
{
	debug_enter ("cancel_clicked");

	cancel_selected_tasks ((ManagerUI *) user_data);

	debug_exit ("cancel_clicked");
}

static void
clear_queue_clicked_cb (GtkButton   * b,
                        gpointer      user_data)
{
	guint i;
	GPtrArray * tasks;
	debug_enter ("clear_queue_clicked_cb");

	tasks = queue_get_tasks ();
	for (i=0; i<tasks->len; ++i) {
		Task * task = TASK(g_ptr_array_index (tasks,i));
		if (queue_get_task_status(task)==QUEUE_TASK_STATUS_QUEUED)
			queue_task_remove(task);
	}

	/* cleanup */
	pan_g_ptr_array_foreach (tasks, (GFunc)pan_object_unref, NULL);
	g_ptr_array_free (tasks, TRUE);

	debug_exit ("clear_queue_clicked_cb");
}

static void
requeue_cb (GtkButton   * button,
            gpointer      user_data)
{
	guint i;
	ManagerUI * ui;
	GPtrArray * a;
	debug_enter ("requeue_cb");

	ui = (ManagerUI*) user_data;
	a = get_selected_tasks (ui);
	for (i=0; i!=a->len; ++i) {
		Task * task = (Task*) g_ptr_array_index (a, i);
		if (queue_get_task_status(task) == QUEUE_TASK_STATUS_FAILED)
			queue_task_requeue_failed (task, 0);
	}
	g_ptr_array_free (a, TRUE);

	debug_exit ("requeue_cb");
}


/**
***  Display
**/

static int
get_secs_left (const Task * task)
{
	int retval = 0;

	if (task->sock)
	{
		double elapsed_time = time(0) - task->sock->byte_count_start_time;
		double progress = status_item_get_progress_of_100 (STATUS_ITEM(task)) / 100.0;
		double safe_progress = MAX(progress, 0.01); /* don't divide by 0 */
		const double estimated_total_time = elapsed_time/safe_progress;
		retval = estimated_total_time - elapsed_time;
	}

	return retval;
}

static void
get_percent_str (const Task   * task,
                 gchar        * buf)
{
	int percent = 0;

	if (task != NULL)
		percent = status_item_get_progress_of_100 (STATUS_ITEM(task));

	if (percent == 0)
		*buf = '\0';
	else
		sprintf (buf, "%d%%", percent);
}

static void
get_xfer_str (const Task    * task,
              gchar         * buf)
{
	if (!task->sock)
	{
		*buf = '\0';
	}
	else
	{
		int secs_left = get_secs_left (task);
		const int h = secs_left / 3600;
		const int m = (secs_left % 3600) / 60;
		const int s = secs_left % 60;
		const double KBps = pan_socket_get_xfer_rate_KBps (task->sock);
		sprintf (buf, _("%.2f KB/s, %d:%02d:%02d remaining"), KBps, h, m, s);
	}
}

static void
get_status_str (const Task     * task,
                gchar          * buf)
{
	const char* status_str;
	QueueTaskStatus status = queue_get_task_status(task);

	switch (status)
	{
		case QUEUE_TASK_STATUS_NOT_QUEUED:
			status_str = _("Not Queued");
			break;
		case QUEUE_TASK_STATUS_QUEUED:
			status_str = _("Queued");
			break;
		case QUEUE_TASK_STATUS_RUNNING:
			status_str = _("Running");
			break;
		case QUEUE_TASK_STATUS_FAILED:
			status_str = _("Failed");
			break;
		case QUEUE_TASK_STATUS_ABORTING:
			status_str = _("Aborting");
			break;
		case QUEUE_TASK_STATUS_CONNECTING:
			status_str = _("Connecting");
			break;
		default:
			status_str = _("???");
			break;
	}

	if (!task->server->online && status==QUEUE_TASK_STATUS_QUEUED)
		sprintf (buf, _("Offline"));
	else if (!task->tries)
		sprintf (buf, "%s", status_str);
	else
		sprintf (buf, _("%s (%d tries)"), status_str, task->tries);
}


static void
add_task_to_list (ManagerUI     * ui,
                  Task          * task,
                  int             index)
{
	int row;
	char statusbuf[64];
	char percentbuf[8];
	char xferbuf[64];
	GtkCList * list = GTK_CLIST(ui->list);
	char * description = status_item_describe(STATUS_ITEM(task));
	char * text [5];
       
	text[0] = statusbuf;
	text[1] = percentbuf;
	text[2] = task->server->name;
	text[3] = xferbuf;
	text[4] = description;

	get_xfer_str (task, xferbuf);
	get_status_str (task, statusbuf);
	get_percent_str (task, percentbuf);

	pan_lock ();
	row = gtk_clist_insert (list, index, text);
	gtk_clist_set_row_data (list, row, task);
	pan_unlock ();

	g_free (description);
}


static void
ui_populate (ManagerUI *ui)
{
	guint i;
	GPtrArray * tasks;

	tasks = queue_get_tasks ();
	for (i=0; i<tasks->len; ++i)
		add_task_to_list (ui, TASK(g_ptr_array_index(tasks,i)), i);

	/* cleanup */
	pan_g_ptr_array_foreach (tasks, (GFunc)pan_object_unref, NULL);
	g_ptr_array_free (tasks, TRUE);
}

static void
queue_row_move_cb (GtkCList   * clist,
                   int          old_index,
                   int          new_index,  
                   gpointer     user_data)
{
	ManagerUI * ui;
	Task * task;
	debug_enter ("queue_row_move_cb");

	ui = (ManagerUI*) user_data;
	task = TASK(gtk_clist_get_row_data (clist, old_index));
	if (!ui->dampen_move_feedback_loop)
		queue_task_move (task, new_index);

	debug_exit ("queue_row_move_cb");
}


/**
***  Queue Changes
**/

static void
task_added_cb (gpointer call_obj, gpointer call_arg, gpointer client_arg)
{
	Task * task = TASK(call_obj);
	int index = GPOINTER_TO_INT(call_arg);
	ManagerUI * ui = (ManagerUI*) client_arg;

	add_task_to_list (ui, task, index);
}

static void
task_removed_cb (gpointer call_obj, gpointer call_arg, gpointer client_arg)
{
	Task * task = TASK(call_obj);
	int index = GPOINTER_TO_INT(call_arg);
	ManagerUI * ui = (ManagerUI*) client_arg;
	GtkCList * list = GTK_CLIST(ui->list);

	g_return_if_fail (gtk_clist_get_row_data(list,index)==task);
	pan_lock ();
	gtk_clist_remove (list, index);
	pan_unlock ();
}

static void
task_moved_cb (gpointer call_obj, gpointer call_arg, gpointer client_arg)
{
	ManagerUI * ui = (ManagerUI*) client_arg;
	int old_row = ((int*)call_arg)[0];
	int new_row = ((int*)call_arg)[1];

	pan_lock ();
	gtk_signal_disconnect (GTK_OBJECT(ui->list), ui->moved_signal_id);
	gtk_clist_row_move (GTK_CLIST(ui->list), old_row, new_row);
	ui->moved_signal_id = gtk_signal_connect (GTK_OBJECT(ui->list), "row_move", queue_row_move_cb, ui);
	pan_unlock ();
}

static gboolean
gui_key_press_cb (GtkWidget      * widget,
                  GdkEventKey    * event,
                  gpointer         data)
{
	ManagerUI * ui;
	GtkCList  * clist;
	debug_enter ("gui_key_press_cb");

	ui    = (ManagerUI*) data;
	clist = GTK_CLIST (ui->list);

        switch (event->keyval)
        {
                case GDK_Delete:
			cancel_selected_tasks (ui);
			break;
                case GDK_Home:
                        if (event->state & GDK_SHIFT_MASK)
			{
				move_selected_tasks_to_top (ui);
			}
			else
			{
				select_task (ui, TRUE);
			}
			break;
                case GDK_End:
                        if (event->state & GDK_SHIFT_MASK)
			{
				move_selected_tasks_to_bottom (ui);
			}
			else
			{
				select_task (ui, FALSE);
			}
			break;
                default:
			break;
        }

	debug_exit ("gui_key_press_cb");
        return TRUE;
}

static void
update_titlebar_lock_unconditional (ManagerUI * ui)
{
	int running, queued, failed;
	guint i;
	GPtrArray * tasks;
	gchar* pch;

	g_return_if_fail (ui!=NULL);

	/* get information about the tasks */
	running = queued = failed = 0;
	tasks = queue_get_tasks ();
	for (i=0; i<tasks->len; ++i) {
		Task * t = TASK(g_ptr_array_index(tasks,i));
		QueueTaskStatus status = queue_get_task_status (t);
	
		if (status==QUEUE_TASK_STATUS_QUEUED)
			++queued;
		else if (status==QUEUE_TASK_STATUS_RUNNING)
			++running;
		else if (status==QUEUE_TASK_STATUS_ABORTING || status==QUEUE_TASK_STATUS_FAILED)
			++failed;
	}

	/* create a titlebar */
	if (failed)
		pch = g_strdup_printf (_("Pan %s Task Manager (%d Queued, %d Running, %d Failed)"), VERSION, queued, running, failed);
	else if (queued || running)
		pch = g_strdup_printf (_("Pan %s Task Manager (%d Queued, %d Running)"), VERSION, queued, running);
	else 
		pch = g_strdup_printf (_("Pan %s Task Manager"), VERSION);

	/* show the titlebar */
	pan_lock_unconditional ();
	gtk_window_set_title (GTK_WINDOW(ui->window), pch);
	pan_unlock_unconditional ();
	g_free (pch);

	/* cleanup */
	pan_g_ptr_array_foreach (tasks, (GFunc)pan_object_unref, NULL);
	g_ptr_array_free (tasks, TRUE);
}

static int
periodic_update_timer (gpointer user_data)
{
	ManagerUI * ui = (ManagerUI*) user_data;
	GtkCList * clist = GTK_CLIST(ui->list);
	gint row;

	debug0 (DEBUG_QUEUE, "task manager periodic ui refresh");

	/* update the ui */
	pan_lock_unconditional ();
	gtk_clist_freeze (clist);
	for (row=0; row<clist->rows; ++row)
	{
		if (gtk_clist_row_is_visible (clist, row))
		{
			char buf[256];
			char * pch;
			Task * task;

			task = gtk_clist_get_row_data(clist, row);
			pan_object_ref (PAN_OBJECT(task));

			/* have the index, now update it */
			get_status_str (task, buf);
			gtk_clist_set_text (clist, row, 0, buf);
			get_percent_str (task, buf);
			gtk_clist_set_text (clist, row, 1, buf);
			gtk_clist_set_text (clist, row, 2, task->server->name);
			get_xfer_str (task, buf);
			gtk_clist_set_text (clist, row, 3, buf);
			pch = status_item_describe(STATUS_ITEM(task));
			gtk_clist_set_text (clist, row, 4, pch);

			/* cleanup */
			g_free (pch);
			pan_object_unref (PAN_OBJECT(task));
		}
	}

	gtk_clist_thaw (clist);
	pan_unlock_unconditional ();

	if (ui->titlebar_needs_refresh) {
		update_titlebar_lock_unconditional (ui);
		ui->titlebar_needs_refresh = FALSE;
	}

	return 1;
}

static gint
window_delete_event_cb (GtkWidget * w, GdkEvent * e, gpointer data)
{
	ManagerUI * ui = (ManagerUI*) data;
	gui_save_window_size (ui->window, "task_manager_2");
	gui_save_column_widths (ui->list, "task_manager_2");
	return FALSE;
} 
static void
ui_destroy_cb (GtkObject * o, gpointer user_data)
{
	ManagerUI *ui = (ManagerUI*) user_data;

	pan_callback_remove (queue_task_added, task_added_cb, ui);
	pan_callback_remove (queue_task_removed, task_removed_cb, ui);
	pan_callback_remove (queue_task_moved, task_moved_cb, ui);

	g_source_remove (ui->timeout_id);

	pan_warn_if_fail (ui == manager_ui);
	g_free (manager_ui);
	manager_ui = NULL;
}

/**
***
**/

static ManagerUI*
task_manager_build_ui (GtkWidget * window)
{
	gboolean b;
	ManagerUI * ui = g_new0 (ManagerUI, 1);
	GtkWidget * w;
	GtkWidget * p;
	GtkWidget * hbox;
	GtkWidget * vbox;
	char* list_titles [5];

	ui->window = window;
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	/* create the ready/pause pixmaps */
	ui->ready_pixmap = gnome_stock_pixmap_widget_new (ui->window, GNOME_STOCK_PIXMAP_TIMER);
	ui->pause_pixmap = gnome_stock_pixmap_widget_new (ui->window, GNOME_STOCK_PIXMAP_TIMER_STOP);
	gtk_object_ref (GTK_OBJECT(ui->ready_pixmap));
	gtk_object_ref (GTK_OBJECT(ui->pause_pixmap));

	/* button bar */
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER(hbox), GNOME_PAD_SMALL);

	/* pause button */
	b = queue_is_paused ();
	w = gtk_toggle_button_new ();
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_signal_connect (GTK_OBJECT(w), "toggled", pause_toggled_cb, ui);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, GNOME_PAD);
	decorate_pause_button (ui, GTK_TOGGLE_BUTTON(w), b);

	/* separator */
	w = gtk_vseparator_new ();
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, GNOME_PAD);

	/* up button */
	w = gtk_button_new ();
	p = gnome_stock_pixmap_widget_new (window, GNOME_STOCK_PIXMAP_UP);
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", up_clicked_cb, ui);
	gtk_container_add (GTK_CONTAINER(w), p);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Move Selected Task(s) Up"), "");
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	/* top button */
	w = gtk_button_new ();
	p = gnome_stock_pixmap_widget_new (window, GNOME_STOCK_PIXMAP_TOP);
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", top_clicked_cb, ui);
	gtk_container_add (GTK_CONTAINER(w), p);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Move Selected Task(s) To Top"), "");
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	/* separator */
	w = gtk_vseparator_new ();
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, GNOME_PAD);

	/* down button */
	w = gtk_button_new ();
	p = gnome_stock_pixmap_widget_new (window, GNOME_STOCK_PIXMAP_DOWN);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", down_clicked_cb, ui);
	gtk_container_add (GTK_CONTAINER(w), p);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Move Selected Task(s) Down"), "");
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	/* bottom button */
	w = gtk_button_new ();
	p = gnome_stock_pixmap_widget_new (window, GNOME_STOCK_PIXMAP_BOTTOM);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", bottom_clicked_cb, ui);
	gtk_container_add (GTK_CONTAINER(w), p);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Move Selected Task(s) To Bottom"), "");
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	/* separator */
	w = gtk_vseparator_new ();
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, GNOME_PAD);

	/* requeue button */
	w = gtk_button_new ();
	p = gnome_stock_pixmap_widget_new (window, GNOME_STOCK_PIXMAP_REDO);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", requeue_cb, ui);
	gtk_container_add (GTK_CONTAINER(w), p);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Requeue selected failed Task(s)"), "");
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	/* separator */
	w = gtk_vseparator_new ();
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, GNOME_PAD_BIG);

        /* edit per-server maximum */
	w = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (server_xpm));
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Set Per-Server Connection Limits"), "");
	gtk_signal_connect (GTK_OBJECT(w), "clicked", prefs_spawn_to_news_connections, NULL);
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	/* clear queue */
	w = gtk_button_new ();
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Clear Queue"), "");
	gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (delete_sheet_xpm));
	gtk_signal_connect (GTK_OBJECT(w), "clicked", clear_queue_clicked_cb, ui);
	gtk_box_pack_end (GTK_BOX(hbox), w, FALSE, FALSE, GNOME_PAD_BIG);

	/* cancel button */
	w = gtk_button_new ();
	gtk_signal_connect (GTK_OBJECT(w), "clicked", cancel_clicked_cb, ui);
	gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (delete_row_xpm));
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Cancel Selected Task(s)"), "");
	gtk_button_set_relief (GTK_BUTTON(w),GTK_RELIEF_NONE);
	gtk_box_pack_end (GTK_BOX(hbox), w, FALSE, FALSE, GNOME_PAD_BIG);

	/* add button bar to the queued window */
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

        /* queue list */
        list_titles[0] = _("Status");
	list_titles[1] = _("% Done");
	list_titles[2] = _("Server");
        list_titles[3] = _("Transfer Rate");
	list_titles[4] = _("Description");
	ui->list = w = gtk_clist_new_with_titles (5, list_titles);
	gtk_clist_set_column_width (GTK_CLIST(w), 0, 200);
	gtk_clist_set_column_width (GTK_CLIST(w), 4, 1500);
	gui_restore_column_widths (ui->list, "task_manager_2");

	gtk_clist_set_selection_mode (GTK_CLIST(w), GTK_SELECTION_EXTENDED);
	ui->moved_signal_id = gtk_signal_connect (GTK_OBJECT(w), "row_move", queue_row_move_cb, ui);
	w = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(w), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (w), ui->list);
	gtk_box_pack_start (GTK_BOX(vbox), w, TRUE, TRUE, 0);

	pan_callback_add (queue_task_added, task_added_cb, ui);
	pan_callback_add (queue_task_removed, task_removed_cb, ui);
	pan_callback_add (queue_task_moved, task_moved_cb, ui);

	ui->titlebar_needs_refresh = TRUE;
	ui->timeout_id = gtk_timeout_add (2000, periodic_update_timer, ui);
	ui->top = vbox;
        gtk_signal_connect (GTK_OBJECT(ui->top), "destroy", ui_destroy_cb, ui);
	return ui;
}

/**
***  Titlebar dirty
**/

static void
update_titlebar_cb (gpointer call_obj, gpointer call_arg, gpointer user_data)
{
	((ManagerUI*)(user_data))->titlebar_needs_refresh = TRUE;
}
static void
remove_update_titlebar_callbacks_cb (GtkObject * o, gpointer user_data)
{
	pan_callback_remove (queue_task_added, update_titlebar_cb, user_data);
	pan_callback_remove (queue_task_removed, update_titlebar_cb, user_data);
	pan_callback_remove (queue_task_status_changed, update_titlebar_cb, user_data);
}

/************
*************  PUBLIC ROUTINES
************/

void
task_manager_spawn (void)
{
	/* There can be only one */
	if (manager_ui != NULL)
	{
		pan_lock();
		if (!GTK_WIDGET_MAPPED(GTK_WIDGET(manager_ui->window)))
			gtk_widget_map ( GTK_WIDGET(manager_ui->window));
		gdk_window_raise (manager_ui->window->window);
		gdk_window_show (manager_ui->window->window);
		pan_unlock();
		return;
	}

	manager_ui = task_manager_build_ui (gtk_window_new(GTK_WINDOW_TOPLEVEL));
	ui_populate (manager_ui);

	pan_callback_add (queue_task_added, update_titlebar_cb, manager_ui);
	pan_callback_add (queue_task_removed, update_titlebar_cb, manager_ui);
	pan_callback_add (queue_task_status_changed, update_titlebar_cb, manager_ui);
	gtk_signal_connect (GTK_OBJECT(manager_ui->window), "delete_event",
	                    GTK_SIGNAL_FUNC(window_delete_event_cb), manager_ui);
	gtk_signal_connect (GTK_OBJECT(manager_ui->window), "destroy",
	                    GTK_SIGNAL_FUNC(remove_update_titlebar_callbacks_cb), manager_ui);
	gtk_signal_connect (GTK_OBJECT (manager_ui->window), "key_press_event",
	                    GTK_SIGNAL_FUNC(gui_key_press_cb), manager_ui);
	gtk_window_set_title (GTK_WINDOW(manager_ui->window), _("Pan - Task Manager"));
	gtk_container_add (GTK_CONTAINER(manager_ui->window), manager_ui->top);

	gtk_window_set_default_size (GTK_WINDOW (manager_ui->window), 600, 400);
	gui_restore_window_size (manager_ui->window, "task_manager_2");
	gtk_widget_show_all (manager_ui->window);
}

