// -*- Mode: C++; indent-tabs-mode: nil; c-basic-offset: 4 -*-

/*
 *  PaperBox - tracker-client.cc
 *
 *  Copyright (C) 2007 Marko Anastasov
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <sys/time.h>
#include <iostream>
#include <vector>
#include <glibmm/fileutils.h>
#include <glibmm-utils/log-stream-utils.h>
#include "file-utils.hh"
#include "tag-request.hh"
#include "tracker-phone.hh"

namespace {

    static const int DOC_RETRIEVAL_LIMIT = 100000;

    static char* doc_types[] =
    {
        "application/pdf",
        "application/vnd.oasis.opendocument.text",
        NULL
    };

    static char* doc_keys[] =
    {
	"File:Name",
	"Doc:Subject",
	"Doc:Author",
	"Doc:Comments",
	"Doc:PageCount",
	"Doc:WordCount",
	"Doc:Created",
	NULL
    };

    enum {        // keep in sync with doc_keys
        FILE_NAME,
        SUBJECT,
        AUTHOR,
        COMMENTS,
        PAGE_COUNT,
        WORD_COUNT,
        DATE_CREATED
    };

    static inline bool
    is_empty_string(const char *s)
    {
	return s == 0 || s[0] == '\0';
    }

    static char**
    ustring_vector_to_array(const std::vector<Glib::ustring>& vec)
    {
        char** array = g_new (char*, vec.size() + 1);
        std::vector<Glib::ustring>::const_iterator it(vec.begin());
        std::vector<Glib::ustring>::const_iterator end(vec.end());

        for (int i = 0; it != end; ++it) {
            array[i++] = g_strdup(it->c_str());
        }

        array[vec.size()] = 0;

        return array;
    }

} // anonymous namespace

namespace paperbox {

    using std::vector;
    using boost::shared_ptr;

    TrackerPhone::TrackerPhone() throw (std::runtime_error)
        :
        tracker_connection_(0)
    {
        gboolean enable_warnings = TRUE;
        tracker_connection_ = tracker_connect(enable_warnings);

        if (! tracker_connection_)
            throw std::runtime_error("Unable to connect to Tracker daemon.");
    }

    TrackerPhone::~TrackerPhone()
    {
        tracker_disconnect(tracker_connection_);
    }

    bool
    TrackerPhone::ping_tracker()
    {
        TrackerClient* tc = 0;
        gboolean enable_warnings = TRUE;
        tc = tracker_connect(enable_warnings);
        bool result = (tc);
        tracker_disconnect(tc);
        return result;
    }

    void
    TrackerPhone::get_all_document_uris(std::queue<Glib::ustring>& uris)
    {
        LOG_FUNCTION_SCOPE_NORMAL_DD;

        int live_query_id = -1;
        int offset = 0;
        int max_hits = DOC_RETRIEVAL_LIMIT;

        char** results = 0;
        GError* error = 0;

        LOG_DD("sending request for document URIs");

        results = tracker_files_get_by_mime_type(tracker_connection_,
                                                 live_query_id,
                                                 doc_types,
                                                 offset,
                                                 max_hits,
                                                 &error);

        if (error) {
            LOG_ERROR("Ha! No documents for you: "
                      << error->message);
            g_clear_error(&error);
            return;
        }

        if (! results) {
            LOG_DD("Tracker returned *0* documents");
            return;
        }

        char** res_p = 0;

        LOG_DD("iterating over results");

        for (res_p = results; *res_p; res_p++) {
            Glib::ustring found_uri(*res_p);
            uris.push(found_uri);
        }

        g_strfreev (results);
    }

    void
    TrackerPhone::get_document(const Glib::ustring& uri,
                               shared_ptr<Document>& document)
    {
        LOG_FUNCTION_SCOPE_NORMAL_DD;

        char** results = 0;
        GError* error = 0;

        Document* null_doc = 0;
        document.reset(null_doc);

        if (! Glib::file_test(uri.raw(), Glib::FILE_TEST_EXISTS)) {
            throw Glib::FileError(Glib::FileError::NO_SUCH_ENTITY,
                                  "document does not exist");
        }

        LOG_DD("sending request for metadata for " << uri);

        results = tracker_metadata_get(tracker_connection_,
                                       SERVICE_DOCUMENTS,
                                       uri.c_str(),
                                       doc_keys,
                                       &error);

        if (error) {
            LOG_ERROR("Ha! No metadata for you: " << error->message);
            g_clear_error(&error);
            return;
        }

        if (! results) {
            LOG_DD("Tracker returned no metadata.");
            return;
        }

        prepare_document(uri, results, document);

        attach_tags(document);

        g_strfreev (results);
    }

    void
    TrackerPhone::prepare_document(const Glib::ustring& uri,
                                   char** keys,
                                   shared_ptr<Document>& document)
    {
        LOG_FUNCTION_SCOPE_NORMAL_DD;

        LOG_DD("preparing a document object with metadata for uri " << uri);

        document.reset(new Document(uri));

        if (! is_empty_string(keys[FILE_NAME])) {
            Glib::ustring file_name(keys[FILE_NAME]);
            document->set_file_name(file_name);
        }

        Glib::ustring file_modified = get_file_modification_time (uri);
        document->set_modification_time(file_modified);

        if (! is_empty_string(keys[SUBJECT])) {
            Glib::ustring subject(keys[SUBJECT]);
            document->set_subject(subject);
        }

        if (! is_empty_string(keys[AUTHOR])) {
            Glib::ustring author(keys[AUTHOR]);
            document->set_author(author);
        }

        if (! is_empty_string(keys[COMMENTS])) {
            Glib::ustring comments(keys[COMMENTS]);
            document->set_comments(comments);
        }

        if (! is_empty_string(keys[PAGE_COUNT])) {
            Glib::ustring page_count(keys[PAGE_COUNT]);
            document->set_page_count(page_count);
        }

        if (! is_empty_string(keys[WORD_COUNT])) {
            Glib::ustring word_count(keys[WORD_COUNT]);
            document->set_word_count(word_count);
        }

        if (! is_empty_string(keys[DATE_CREATED])) {
            Glib::ustring date_created(keys[DATE_CREATED]);
            document->set_date_created(date_created);
        }
    }

    // document information retrieval helper
    void
    TrackerPhone::attach_tags(shared_ptr<Document>& document)
    {
        vector<Glib::ustring> tags;
        TagRequest request(tracker_connection_, document->get_uri());

        request.get_results(tags);

        if (! tags.empty())
            document->set_tags(tags);
    }

    void
    TrackerPhone::add_tags(const Glib::ustring& uri,
                           const vector<Glib::ustring> tags)
    {
        GError* error = 0;

        char** ctags = ustring_vector_to_array(tags);

        tracker_keywords_add (tracker_connection_,
                              SERVICE_DOCUMENTS,
                              uri.c_str(),
                              ctags,
                              &error);

        g_strfreev (ctags);

        if (error) {
            LOG_ERROR("Ha! Adding tags failed: " << error->message);
            throw std::runtime_error(error->message);
        }
    }

    void
    TrackerPhone::remove_tags(const Glib::ustring& uri,
                              const std::vector<Glib::ustring> tags)
    {
        GError* error = 0;

        char** ctags = ustring_vector_to_array(tags);

        tracker_keywords_remove (tracker_connection_,
                                 SERVICE_DOCUMENTS,
                                 uri.c_str(),
                                 ctags,
                                 &error);

        g_strfreev (ctags);

        if (error) {
            LOG_ERROR("Ha! Removing tags failed: " << error->message);
            throw std::runtime_error(error->message);
        }
    }

} // namespace paperbox
