///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "basicframe.h"
#include "util/rectboundary.h"
#include <util/stringutil.h>
#include <util/warning.h>
#include <sigc++/sigc++.h>
#include <libxml++/libxml++.h>
#include <gdkmm.h>
#include <fstream>
#include <algorithm>

BasicFrame::BasicFrame(Group *parent, float w, float h, 
			 const std::string& name)
  :Pagent(parent, name)
{
  set_size(w, h);
}

BasicFrame::BasicFrame(Group *parent, const xmlpp::Element& node)
  :Pagent(parent, "unnamed")
{
  if(const xmlpp::Attribute* name = node.get_attribute("name"))
    set_name(name->get_value());

  if(const xmlpp::Attribute* locked = node.get_attribute("lock"))
    set_lock(to<bool>(locked->get_value()));

  if(const xmlpp::Attribute* flow = node.get_attribute("flowaround"))
    flow_around = to<bool>(flow->get_value());

  if(const xmlpp::Attribute* margin = node.get_attribute("obstaclemargin")) {
    try{
      obstacle_margin = to<float>(margin->get_value());
    } catch(const std::runtime_error& err){
      throw Error::Read(std::string("Bad obstaclemargin: \"") + err.what());
    }
  }
  
  if(const xmlpp::Attribute* w = node.get_attribute("width"))
    width = to<float>(w->get_value());
//   else
//     throw Error::Read("No \"width\" attribute found in frame");
  if(const xmlpp::Attribute* h = node.get_attribute("height"))
    height = to<float>(h->get_value());
//   else
//     throw Error::Read("No \"height\" attribute found in frame");

  if(const xmlpp::Attribute* matrix = node.get_attribute("matrix")) {
    set_matrix(to<Matrix>(matrix->get_value()));
  } else
    throw Error::Read("No \"matrix\" attribute found in <frame>");
}

xmlpp::Element *BasicFrame::save(xmlpp::Element& parent_node,
				 const FileContext &context) const 
{
  xmlpp::Element *node = parent_node.add_child("frame");
  if(!name.empty())
    node->set_attribute("name", name);

  if(resizable) {
    node->set_attribute("width", tostr(width));
    node->set_attribute("height", tostr(height));
  }

  node->set_attribute("matrix", tostr(get_matrix()));
  node->set_attribute("lock", tostr(get_lock()));
  node->set_attribute("flowaround", tostr(flow_around));
  node->set_attribute("obstaclemargin", tostr(obstacle_margin));

  return node;
}

BasicFrame::~BasicFrame() {
}

void BasicFrame::print(std::ostream &out, bool grayscale) const {
}

Boundary BasicFrame::get_box() const {
  return RectBoundary::create(get_matrix(), width, height);
}

Boundary BasicFrame::get_obstacle_boundary() const {
  if(flow_around) {
    /// \todo Isn't this bogus? width / height is before matrix, but the
    /// addition for the obstacle should be after.
    return RectBoundary::create(Matrix::translation(-obstacle_margin, 
						    -obstacle_margin)
				* get_matrix(),
				width + 2 * obstacle_margin, 
				height + 2 * obstacle_margin);
    
  } else
    return Boundary();
};
