# This file is part of pybliographer
# 
# Copyright (C) 1998 Frederic GOBRY
# Email : gobry@idiap.ch
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# $Id: BibTeX.py,v 1.17 1999/07/30 12:34:00 gobry Exp $

# Extension module for BibTeX files

import Pyblio.Base, bibtex
import os,sys, tempfile, pwd, time, traceback, re, string
import Pyblio.Open

from types import *
from Pyblio.Fields import *


def my_write (entry, output):
	""" Print an entry as BiBTeX """

	native = (entry.id == 'BibTeX')

	# get the entry description
	tp = entry.type
	
	output.write ('@%s{%s,\n' % (tp.name, entry.name))
	dico = entry.keys ()
	
	for f in tp.mandatory:
		field = string.lower (f.name)
		
		if entry.has_key (field):
			if native:
				text = entry.native (field)
			else:
				text = entry [field]
				
			output.write ('  ' + Pyblio.Base.format (
				'%-14s = %s' % (f.name, text),
                                75, 0, 19) + ',\n')
			dico.remove (field)

	for f in dico:
		if native:
			text = entry.native (f)
		else:
			text = entry [f]
			
		output.write ('  ' + Pyblio.Base.format (
			'%-14s = %s' % (f, text), 75, 0, 19) +
                              ',\n')
		
        output.write ('}\n\n')


# --------------------------------------------------
# Register a method to open BibTeX files
# --------------------------------------------------

def my_open (entity, check):
	
	method, address, file, p, q, f = entity
	base = None
	
	if (not check) or (method == 'file' and file [-4:] == '.bib'):
		base = DataBase (file)
		
	return base



Pyblio.Open.register ('BibTeX', my_open, my_write)


# --------------------------------------------------
#  Entry inherits from pyb
# --------------------------------------------------

class Entry (Pyblio.Base.Entry):

	id = 'BibTeX'
	
	def __init__ (self, key, name, type, content):
		Pyblio.Base.Entry.__init__ (self, key, name, type, content)

		self.__text   = {}
		return
	
	
	def __delitem__ (self, key):
		# First, eventually remove from cache
		if self.__text.has_key (key):
			del self.__text [key]

		del self.__dict [key]
		return
	
	def native (self, key):
		""" Return object in its native format """
		obj = self.__dict [key]
		
		return bibtex.native (obj)

	def text (self, key):
		# look in the cache first
		if self.__text.has_key (key):
			return self.__text [key]
		
		obj = self.__dict [key]

		# search its declared type

		type = Pyblio.Base.gettype (self.type, key)
		
		ret = bibtex.expand (self.key.base.get_parser (),
				     obj, type)
		
		if ret [0] == Pyblio.Fields.TypeAuthor:
			# Author
			val = AuthorGroup ()
			for aut in ret [3]: val.append (Author (aut))
			
		elif ret [0] == Pyblio.Fields.TypeDate:
			# Date
			val = Date (ret [3])
			
		else:
			# Any other text
			val = Text (ret [2])

		self.__text [key] = (val, ret [1])
		return (val, ret [1])



# --------------------------------------------------
# Une base de references bibliographiques,
# comme un dictionnaire avec des extensions...
# --------------------------------------------------

class DataBase (Pyblio.Base.DataBase):

	id = 'BibTeXDB'
	
	def __parsefile__ (self):
		self.__entry = {}
		self.__parser = 0
		self.__recursive = 0
		self.__current = 0
		
		# Stockage temporaire avant un "update"
		self.__to_delete = {}
		self.__to_modify = {}
		
		# Ouvrir le fichier associe
		self.__parser = bibtex.open (self.name)
		
		finished = 0
		errors = []
		
		# Creer la base de cles
		while 1:
			try:
				retval = bibtex.next (self.__parser)
			except IOError, err:
				errors.append (str (err))
				continue
			
			if retval == None: break

			name, type, offset, line, object = retval

			if not self.__entry.has_key (name):
				self.__entry [name] = offset
				continue
				
			errors.append ("%s:%d: key `%s' already defined" % (
				self.name, line, name))

		if len (errors) > 0:
			raise IOError, string.join (errors, "\n")

		
	def __init__ (self, basename):
		"Initialisation"
		Pyblio.Base.DataBase.__init__ (self, basename)
		self.__parsefile__ ()
		return

	
	def __del__ (self):
		pass

	def get_parser (self):
		return self.__parser
	
	def keys (self):
		""" Return the key sequence """
		
		return map (lambda k, self=self:
			    Pyblio.Base.Key (self, k), self.__entry.keys ())


	def __len__ (self):
		""
		return len (self.__entry)

	def __repr__ (self):
		""
		return "<BibTeX database `%s' (%d entries)>" % \
		       (self.name, len (self))
	
	def has_key (self, key):
		"Presence d'une cle"
		return self.__entry.has_key (key.key)

	def __getitem__ (self, name):
		"Retourne une entree en fonction de la cle"
		name = name.key
		
		type = __builtins__['type']

		# If we get a modified instance, return it directly
		ref = self.__entry [name]

		if type (ref) is InstanceType:
			return ref
		else:
			# Else read from file
			bibtex.set_offset (self.__parser, ref)

			name, type, offset, line, content = \
			      bibtex.next (self.__parser)

			type = Pyblio.Base.getentry (type)
			
			# If in recursive mode, go back to previous position
			if self.__recursive > 0:
				bibtex.set_offset (self.__parser,
						   self.__current)
				
			return Entry (Pyblio.Base.Key (self, name),
				      name, type, content);

	
	def __setitem__ (self, key, value):
		"Modifier ou ajouter une entree"

		value.name = key.key
		value.key  = key
		
		self.__entry [key.key] = value

		# Sauver la version modifiee aussi
		self.__to_modify [key.key] = value

		
	def __delitem__ (self, key):
		"Detruire une entree"

		if self.has_key (key):
			# Noter la destruction
			self.__to_delete [key.key] = 1
			# Enlever du dictionnaire
			del self.__entry [key.key]
			
		else:
			raise KeyError, "no such entry"


	# ==================================================

	def foreach (self, function, args = None):
		"Parcours de toutes les entrees"
		bibtex.first (self.__parser)
		finished = 0

		# If we are in a recursive state
		if self.__recursive > 0:
			previous_position = self.__current
			
		# On est susceptible d'avoir un comportement recursif
		self.__recursive = self.__recursive + 1
		
		# Parcourir les entrees du fichier
		while 1:
			entry = bibtex.next (self.__parser)
			
			if entry == None: break

			name, type, offset, line, content = entry

			type = Pyblio.Base.getentry (type)

			if self.__to_delete.has_key (name):
				continue
				
			if self.__to_modify.has_key (name):
				continue

			# Save the current position
			self.__current =  bibtex.get_offset (self.__parser)
				
			function (Entry (Pyblio.Base.Key (self, name),
					 name, type, content), args)

		# Les nouvelles ou les modifiees
		for entry in self.__to_modify.keys ():
			function (self.__to_modify [entry], args)
			
		# Leave recursive state
		self.__recursive = self.__recursive - 1
		if self.__recursive > 0:
			bibtex.set_offset (self.__parser, previous_position)
			self.__current = previous_position
			

	def update (self):
		""" save the database """

		# backup file
		os.rename (self.name, self.name + '.bak')
		
		tmpfile = open (self.name, 'w')
		self.foreach (my_write, tmpfile)
		tmpfile.close ()

		self.__parsefile__ ()
		return

	
	def where (self, key):
		"Chercher une entree selon des criteres"
		result = []

		def search_function (entry, param):
			"the real search function "
			key, result = param

			if key.match (entry):
				result.append (entry.key)
						
		self.foreach (search_function, (key, result))

		# Creer la base de references
		refs = Pyblio.Base.Reference ()

		if len (result) > 0:
			refs.add (self, result)

		return refs
