# This file is part of pybliographer
# 
# Copyright (C) 1998 Frederic GOBRY
# Email : gobry@idiap.ch
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# $Id: TextUI.py,v 1.14 1999/07/30 12:32:13 gobry Exp $

# --------------------------------------------------
# Basical `high level' functions
# --------------------------------------------------

from types import *

import os, sys, traceback, tempfile

import Pyblio.Base, Pyblio.Help
from Pyblio.Open import bibopen, bibwrite

# ----- Create elementary Test -----

Pyblio.Help.register ('searching', """
The general syntax of a search is :

 dbase.where ( constraints )

...where `constraints' is an arbitrary construction of :

  -  has (field, value) : match if `field' matches `value' (as a REGEXP)
  -  &, | : boolean operators (& = and, | = or)

Negation of a constraint is noted `- constraint'.

Examples :

dbase.where (has ('author', 'me'))
dbase.where (has ('author', 'me') & - has ('title', 'failed'))

A search returns references on the entries that were found. It is then
possible to search these references to restrain the result.

See also: references
""")

def has (field, value):
	return Pyblio.Base.Tester (field, value)


# ----- Generic reference holder -----

Pyblio.Help.register ('ref',"""
Syntax : reference = ref (base, [entries])

This function returns a reference object (like .where () method) on
the specified database (eventually restricted to a list of entry
names).
""")

def ref (base, list = None):
	ref = Pyblio.Base.Reference ()

	if list == None:
		ref.add (base)
	else:
		if type (list) is ListType:
			ref.add (base, list)
		else:
			ref.add (base, [ list ])
	return ref


# ----- Display -----

def pager_handle (inpager):
	if inpager:
		pagername = 'more'
	
		if os.environ.has_key ('PAGER'):
			pagername = os.environ ['PAGER']

		pager = os.popen (pagername, 'w')
	else:
		pager = sys.stdout

	return pager


Pyblio.Help.register ('more', """
Syntax: more (references, [nopager])

Display references in BibTeX format. If nopager is 1, the entries are
sent to stdout instead of a pager.
""")

def more (refs, inpager = 1):
	"Output entries"

	try:
		bibwrite (refs, pager_handle (inpager))
		
	except IOError:
		print "warning: broken pipe"
		

Pyblio.Help.register ('ls', """
Syntax: ls (references, [nopager])

Display only title/author and identifier for each entry.

See also : more, less
""")

def ls (refs, inpager = 1):
	"Output entries"

	def printer (entry, arg):
		title  = 'no title'
		author = 'no author'

		if entry.has_key ('title'):  title  = str (entry ['title'])
		if entry.has_key ('author'): author = str (entry ['author'])

		title  = title [0:34]
		author = author [0:24]
		name   = str (entry.name) [0:15]
		
		arg.write ('%-35s %-25s [%-16s]\n' % (title, author, name))


	try:
		refs.foreach (printer, pager_handle (inpager))
	except IOError:
		print "warning: broken pipe"

