/* -*- mode: C c-basic-offset: 4 -*- */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include "pyorbit-private.h"

static void
pycorba_struct_dealloc(PyObject *self)
{
    if (self->ob_type->tp_free)
	self->ob_type->tp_free((PyObject *)self);
    else
	PyObject_DEL(self);
}

static int
pycorba_struct_init(PyObject *self, PyObject *args, PyObject *kwargs)
{
    gint len, i;
    PyObject *pytc;
    CORBA_TypeCode tc;

    /* for the zero argument case, don't do anything */
    len = PyTuple_Size(args);
    if (len == 0 && kwargs == NULL) {
	return 0;
    }

    pytc = PyObject_GetAttrString(self, "__typecode__");
    if (!pytc)
	return -1;
    if (!PyObject_TypeCheck(pytc, &PyCORBA_TypeCode_Type)) {
	Py_DECREF(pytc);
	PyErr_SetString(PyExc_TypeError,
			"__typecode__ attribute not a typecode");
	return -1;
    }
    tc = ((PyCORBA_TypeCode *)pytc)->tc;
    Py_DECREF(pytc);

    if (tc->sub_parts != len) {
	PyErr_Format(PyExc_TypeError, "expected %d arguments, got %d",
		     tc->sub_parts, len);
	return -1;
    }

    for (i = 0; i < len; i++) {
	PyObject *item  = PyTuple_GetItem(args, i);

	PyObject_SetAttrString(self, tc->subnames[i], item);
    }
    return 0;
}

PyTypeObject PyCORBA_Struct_Type = {
    PyObject_HEAD_INIT(NULL)
    0,                                  /* ob_size */
    "CORBA.Struct",                     /* tp_name */
    sizeof(PyObject),                   /* tp_basicsize */
    0,                                  /* tp_itemsize */
    /* methods */
    (destructor)pycorba_struct_dealloc, /* tp_dealloc */
    (printfunc)0,                       /* tp_print */
    (getattrfunc)0,                     /* tp_getattr */
    (setattrfunc)0,                     /* tp_setattr */
    (cmpfunc)0,                         /* tp_compare */
    (reprfunc)0,                        /* tp_repr */
    0,                                  /* tp_as_number */
    0,                                  /* tp_as_sequence */
    0,                                  /* tp_as_mapping */
    (hashfunc)0,                        /* tp_hash */
    (ternaryfunc)0,                     /* tp_call */
    (reprfunc)0,                        /* tp_str */
    (getattrofunc)0,                    /* tp_getattro */
    (setattrofunc)0,                    /* tp_setattro */
    0,                                  /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,   /* tp_flags */
    NULL, /* Documentation string */
    (traverseproc)0,                    /* tp_traverse */
    (inquiry)0,                         /* tp_clear */
    (richcmpfunc)0,                     /* tp_richcompare */
    0,                                  /* tp_weaklistoffset */
    (getiterfunc)0,                     /* tp_iter */
    (iternextfunc)0,                    /* tp_iternext */
    0,                                  /* tp_methods */
    0,                                  /* tp_members */
    0,                                  /* tp_getset */
    (PyTypeObject *)0,                  /* tp_base */
    (PyObject *)0,                      /* tp_dict */
    0,                                  /* tp_descr_get */
    0,                                  /* tp_descr_set */
    0,                                  /* tp_dictoffset */
    (initproc)pycorba_struct_init,      /* tp_init */
    0,                                  /* tp_alloc */
    0,                                  /* tp_new */
    0,                                  /* tp_free */
    (inquiry)0,                         /* tp_is_gc */
    (PyObject *)0,                      /* tp_bases */
};

static int
pycorba_union_init(PyObject *self, PyObject *args, PyObject *kwargs)
{
    gint len;
    PyObject *pytc;
    CORBA_TypeCode tc;

    /* for the zero argument case, don't do anything */
    len = PyTuple_Size(args);
    if (len == 0 && kwargs == NULL) {
	return 0;
    }

    pytc = PyObject_GetAttrString(self, "__typecode__");
    if (!pytc)
	return -1;
    if (!PyObject_TypeCheck(pytc, &PyCORBA_TypeCode_Type)) {
	Py_DECREF(pytc);
	PyErr_SetString(PyExc_TypeError,
			"__typecode__ attribute not a typecode");
	return -1;
    }
    tc = ((PyCORBA_TypeCode *)pytc)->tc;
    Py_DECREF(pytc);

    if (len > 0 && kwargs == NULL) {
	PyObject *discriminator, *value;

	if (!PyArg_ParseTuple(args, "OO", &discriminator, &value))
	    return -1;
	PyObject_SetAttrString(self, "_d", discriminator);
	PyObject_SetAttrString(self, "_v", value);
    } else if (len == 0 && PyDict_Size(kwargs) == 1) {
	int pos = 0;
	PyObject *key, *val, *discriminator;
	const gchar *keyname;

	/* we know there is one value in the dict ... */
	PyDict_Next(kwargs, &pos, &key, &val);
	keyname = PyString_AsString(key);

	for (pos = 0; pos < tc->sub_parts; pos++) {
	    if (!strcmp(keyname, tc->subnames[pos]))
		break;
	}
	if (pos == tc->sub_parts) {
	    PyErr_Format(PyExc_TypeError, "union does not have member '%s'",
			 keyname);
	    return -1;
	}
	if (pos == tc->default_index) {
	    PyErr_SetString(PyExc_TypeError,
			    "can not deduce discriminator for default case");
	    return -1;
	}

	switch (tc->discriminator->kind) {
	case CORBA_tk_boolean:
	    discriminator = tc->sublabels[pos] ? Py_True : Py_False;
	    Py_INCREF(discriminator);
	    break;
	case CORBA_tk_char: {
	    char s[2] = { 0, '\0' };

	    s[0] = (CORBA_char)tc->sublabels[pos];
	    discriminator = PyString_FromString(s);
	    break;
	}
	case CORBA_tk_octet:
	case CORBA_tk_short:
	case CORBA_tk_long:
	case CORBA_tk_longlong:
	case CORBA_tk_ushort:
	case CORBA_tk_ulong:
	case CORBA_tk_ulonglong:
	    discriminator = PyInt_FromLong(tc->sublabels[pos]);
	    break;
	default:
	    PyErr_SetString(PyExc_TypeError, "unhandled discriminator type");
	    return -1;
	}

	PyObject_SetAttrString(self, "_d", discriminator);
	Py_DECREF(discriminator);
	PyObject_SetAttrString(self, "_v", val);
    } else {
	PyErr_SetString(PyExc_TypeError, "expected two arguments, or one keyword argument");
	return -1;
    }

    return 0;
}

PyTypeObject PyCORBA_Union_Type = {
    PyObject_HEAD_INIT(NULL)
    0,                                  /* ob_size */
    "CORBA.Union",                      /* tp_name */
    sizeof(PyObject),                   /* tp_basicsize */
    0,                                  /* tp_itemsize */
    /* methods */
    (destructor)pycorba_struct_dealloc, /* tp_dealloc */
    (printfunc)0,                       /* tp_print */
    (getattrfunc)0,                     /* tp_getattr */
    (setattrfunc)0,                     /* tp_setattr */
    (cmpfunc)0,                         /* tp_compare */
    (reprfunc)0,                        /* tp_repr */
    0,                                  /* tp_as_number */
    0,                                  /* tp_as_sequence */
    0,                                  /* tp_as_mapping */
    (hashfunc)0,                        /* tp_hash */
    (ternaryfunc)0,                     /* tp_call */
    (reprfunc)0,                        /* tp_str */
    (getattrofunc)0,                    /* tp_getattro */
    (setattrofunc)0,                    /* tp_setattro */
    0,                                  /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,   /* tp_flags */
    NULL, /* Documentation string */
    (traverseproc)0,                    /* tp_traverse */
    (inquiry)0,                         /* tp_clear */
    (richcmpfunc)0,                     /* tp_richcompare */
    0,                                  /* tp_weaklistoffset */
    (getiterfunc)0,                     /* tp_iter */
    (iternextfunc)0,                    /* tp_iternext */
    0,                                  /* tp_methods */
    0,                                  /* tp_members */
    0,                                  /* tp_getset */
    (PyTypeObject *)0,                  /* tp_base */
    (PyObject *)0,                      /* tp_dict */
    0,                                  /* tp_descr_get */
    0,                                  /* tp_descr_set */
    0,                                  /* tp_dictoffset */
    (initproc)pycorba_union_init,       /* tp_init */
    0,                                  /* tp_alloc */
    0,                                  /* tp_new */
    0,                                  /* tp_free */
    (inquiry)0,                         /* tp_is_gc */
    (PyObject *)0,                      /* tp_bases */
};

typedef struct {
    PyObject_HEAD
    const char *subname;
} PyCORBA_UnionMember;

static gboolean
branch_matches(PyCORBA_UnionMember *self, PyObject *obj) {
    PyObject *pytc;
    CORBA_TypeCode tc;
    PyObject *py_discrim;
    CORBA_long discrim, pos;

    pytc = PyObject_GetAttrString(obj, "__typecode__");
    if (!pytc)
	return FALSE;
    if (!PyObject_TypeCheck(pytc, &PyCORBA_TypeCode_Type)) {
	Py_DECREF(pytc);
	PyErr_SetString(PyExc_TypeError,
			"__typecode__ attribute not a typecode");
	return FALSE;
    }
    tc = ((PyCORBA_TypeCode *)pytc)->tc;
    Py_DECREF(pytc);

    /* get discriminator */
    py_discrim = PyObject_GetAttrString(obj, "_d");
    if (!py_discrim) {
	PyErr_Clear();
	PyErr_SetString(PyExc_AttributeError, "could not read discriminator");
	return FALSE;
    }
    if (PyString_Check(py_discrim)) {
	if (PyString_Size(py_discrim) != 1) {
	    Py_DECREF(py_discrim);
	    PyErr_SetString(PyExc_ValueError,
			"string discriminators must be one character long");
	    return FALSE;
	}
	discrim = *(CORBA_octet *)PyString_AsString(py_discrim);
    } else {
	discrim = PyInt_AsLong(py_discrim);
	if (PyErr_Occurred()) {
	    PyErr_Clear();
	    Py_DECREF(py_discrim);
	    PyErr_SetString(PyExc_ValueError,
			    "could not read discriminator as an integer");
	    return FALSE;
	}
    }
    Py_DECREF(py_discrim);

    for (pos = 0; pos < tc->sub_parts; pos++) {
	if (pos == tc->default_index) continue;
	if (tc->sublabels[pos] == discrim)
	    break;
    }
    if (pos == tc->sub_parts && tc->default_index < 0) {
	PyErr_SetString(PyExc_ValueError,
		"discriminator value doesn't match any union branches");
	return FALSE;
    }

    if (strcmp(self->subname, tc->subnames[pos]) != 0) {
	PyErr_Format(PyExc_ValueError, "union branch %s is not active",
		     self->subname);
	return FALSE;
    }
    return TRUE;
}

static PyObject *
pycorba_union_member_descr_get(PyCORBA_UnionMember *self, PyObject *obj,
			       PyObject *type)
{
    if (!obj) {
	Py_INCREF(self);
	return (PyObject *)self;
    }
    if (!PyObject_TypeCheck(obj, &PyCORBA_Union_Type)) {
	PyErr_SetString(PyExc_TypeError,
			"this descriptor can only be used with union objects");
	return NULL;
    }

    if (!branch_matches(self, obj))
	return NULL;

    return PyObject_GetAttrString(obj, "_v");
}

static int
pycorba_union_member_descr_set(PyCORBA_UnionMember *self, PyObject *obj,
			       PyObject *value)
{
    if (!PyObject_TypeCheck(obj, &PyCORBA_Union_Type)) {
	PyErr_SetString(PyExc_TypeError,
			"this descriptor can only be used with union objects");
	return -1;
    }

    if (!branch_matches(self, obj))
	return -1;

    return PyObject_SetAttrString(obj, "_v", value);
}

PyTypeObject PyCORBA_UnionMember_Type = {
    PyObject_HEAD_INIT(NULL)
    0,                                  /* ob_size */
    "CORBA.UnionMember",                /* tp_name */
    sizeof(PyCORBA_UnionMember),        /* tp_basicsize */
    0,                                  /* tp_itemsize */
    /* methods */
    (destructor)pycorba_struct_dealloc, /* tp_dealloc */
    (printfunc)0,                       /* tp_print */
    (getattrfunc)0,                     /* tp_getattr */
    (setattrfunc)0,                     /* tp_setattr */
    (cmpfunc)0,                         /* tp_compare */
    (reprfunc)0,                        /* tp_repr */
    0,                                  /* tp_as_number */
    0,                                  /* tp_as_sequence */
    0,                                  /* tp_as_mapping */
    (hashfunc)0,                        /* tp_hash */
    (ternaryfunc)0,                     /* tp_call */
    (reprfunc)0,                        /* tp_str */
    (getattrofunc)0,                    /* tp_getattro */
    (setattrofunc)0,                    /* tp_setattro */
    0,                                  /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,   /* tp_flags */
    NULL, /* Documentation string */
    (traverseproc)0,                    /* tp_traverse */
    (inquiry)0,                         /* tp_clear */
    (richcmpfunc)0,                     /* tp_richcompare */
    0,                                  /* tp_weaklistoffset */
    (getiterfunc)0,                     /* tp_iter */
    (iternextfunc)0,                    /* tp_iternext */
    0,                                  /* tp_methods */
    0,                                  /* tp_members */
    0,                                  /* tp_getset */
    (PyTypeObject *)0,                  /* tp_base */
    (PyObject *)0,                      /* tp_dict */
    (descrgetfunc)pycorba_union_member_descr_get,  /* tp_descr_get */
    (descrsetfunc)pycorba_union_member_descr_set,  /* tp_descr_set */
    0,                                  /* tp_dictoffset */
    (initproc)0,                        /* tp_init */
    0,                                  /* tp_alloc */
    0,                                  /* tp_new */
    0,                                  /* tp_free */
    (inquiry)0,                         /* tp_is_gc */
    (PyObject *)0,                      /* tp_bases */
};

void
pyorbit_add_union_members_to_stub(PyObject *stub, CORBA_TypeCode tc)
{
    PyObject *tp_dict;
    int i;

    g_return_if_fail(PyType_Check(stub) && PyType_IsSubtype((PyTypeObject *)stub, &PyCORBA_Union_Type));

    tp_dict = ((PyTypeObject *)stub)->tp_dict;
    for (i = 0; i < tc->sub_parts; i++) {
	PyCORBA_UnionMember *member;

	member = PyObject_NEW(PyCORBA_UnionMember, &PyCORBA_UnionMember_Type);
	if (!member)
	    return;
	member->subname = tc->subnames[i];
	PyDict_SetItemString(tp_dict, tc->subnames[i], (PyObject *)member);
	Py_DECREF(member);
    }
}
