/* Siobhan Audio Player
 *
 * (c) 2005 Ariel Rios <ariel@gnu.org>
 *
 * main.c: main program
 *
 * This file is part of Siobhan Audio Player.
 *
 *   Siobhan is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   Siobhan is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Foobar; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */


#ifdef HAVE_GPE
#include <gpe/init.h>
#endif

#include <gst/gst.h>
#include <gtk/gtk.h>

#include "timer.h"
#include "properties.h"
#include "callback.h"

/*
 * gcc main.c timer.c properties.c `pkg-config --cflags --libs gstreamer-0.8 gtk+-2.0` -o siobhan
 */


GstElement *filesrc, *decoder, *audiosink;
GstElement * my_thread;

#define GST_PLAYER_ERROR (gst_player_error_quark ())

static GQuark 
gst_player_error_quark ()
{
  static GQuark qk = 0;
  if (!qk)
    qk = g_quark_from_static_string ("gst-player-error-quark");

  return qk;
}

void 
eos (GstElement *src, gpointer data) 
{
  GstThread *thread = GST_THREAD (data);

  g_print ("have eos, quitting\n");

  /* stop the bin */

  gst_element_set_state (GST_ELEMENT (thread), GST_STATE_NULL);
}


static void
cb_play (GtkWidget *widget, gpointer data)
{
	gchar * playfile;
	g_object_get (G_OBJECT(filesrc), "location", &playfile, NULL);
	if (playfile == NULL)
		return;
	if (gst_element_get_state (my_thread)!= GST_STATE_PLAYING)
		gst_element_set_state (my_thread, GST_STATE_PLAYING);
}

static void
cb_stop (GtkWidget * widget, gpointer data)
{
	gst_element_set_state (my_thread, GST_STATE_NULL);
}


static void 
cb_pause (GtkWidget *widget, gpointer data)
{
	switch (gst_element_get_state(my_thread))
	{
		case GST_STATE_PLAYING:
			gst_element_set_state (my_thread, GST_STATE_PAUSED);
			break;
		case GST_STATE_PAUSED:
			gst_element_set_state (my_thread, GST_STATE_PLAYING);
			break;
		default:
			g_print ("other \n");
			break;
	}
}

static void
cb_properties (GtkWidget *widget, gpointer data)
{
  GtkWidget *properties = gst_player_properties_new ();

  gst_player_properties_update (GST_PLAYER_PROPERTIES (properties), filesrc, NULL);

  gtk_widget_show (properties);



}

static void 
select_song (const gchar * filename)
{
	g_object_set (G_OBJECT (filesrc), "location", filename, NULL);
}


static void 
cb_choose (GtkWidget * widget, gpointer data)
{
	GtkWidget * selection;
	const gchar *filename;
	
	
	selection = gtk_file_selection_new ("Open File");
	
	
	if (gtk_dialog_run (GTK_DIALOG (selection)) == GTK_RESPONSE_OK)
	{
		filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (selection));
		select_song (filename);
	}

	gtk_widget_destroy (selection);
	
	return;
	
}

int 
main (int argc, char *argv [])
{
	GtkWidget *app;
	GtkWidget *play, *stop, *pause, *properties;
	GtkWidget *hbox, *vbox;
	GtkWidget *filesel;
	GtkWidget *slider;
	GError *err = NULL;

	gst_init (&argc, &argv);

#ifndef HAVE_GPE
	gtk_init (&argc, &argv);
#else
	gpe_init (&argc, &argv);
#endif

	my_thread = gst_thread_new ("my_thread");
	filesrc = gst_element_factory_make ("filesrc", "disk_source");
	decoder = gst_element_factory_make ("mad", "decoder");
	audiosink = gst_element_factory_make ("osssink", "play_audio");
	

	gst_bin_add_many (GST_BIN (my_thread), filesrc, decoder, audiosink,NULL);

	if (!gst_element_link_many (filesrc, decoder, audiosink, NULL)){
	  g_set_error (&err, GST_PLAYER_ERROR, 1, "Failed to start plugins");
	  //return 0;
	}

	g_signal_connect (G_OBJECT (filesrc), "eos", G_CALLBACK (eos), my_thread);
	
	app = gtk_window_new (GTK_WINDOW_TOPLEVEL);

#ifdef HAVE_GPE
	play = gpe_button_new_from_stock (GTK_STOCK_MEDIA_PLAY, GPE_BUTTON_TYPE_ICON);
	pause = gpe_button_new_from_stock (GTK_STOCK_MEDIA_PAUSE, GPE_BUTTON_TYPE_ICON);
	stop = gpe_button_new_from_stock (GTK_STOCK_MEDIA_STOP, GPE_BUTTON_TYPE_ICON);
#else					  
#ifdef HAVE_GTK_2_6
	play = gtk_button_new_from_stock (GTK_STOCK_MEDIA_PLAY);
	pause = gtk_button_new_from_stock (GTK_STOCK_MEDIA_PAUSE);
	stop = gtk_button_new_from_stock (GTK_STOCK_MEDIA_STOP);
#else
	play = gtk_button_new_with_label (">");
	pause = gtk_button_new_with_label ("||");
	stop = gtk_button_new_with_label ("[]");
#endif
#endif 					  
	filesel = gtk_button_new_from_stock (GTK_STOCK_OPEN);
	properties = gtk_button_new_from_stock (GTK_STOCK_PROPERTIES);
	slider = gst_player_timer_new (filesrc);
	
	hbox = gtk_hbox_new (TRUE, 0);
	vbox = gtk_vbox_new (TRUE, 0);


	gtk_window_set_title (GTK_WINDOW (app), "Siobhan Audio Player");

	gtk_box_pack_start (GTK_BOX (hbox), play, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), pause, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), stop, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), filesel, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), properties, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), slider, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX  (vbox), hbox, TRUE, TRUE, 0);

	gtk_container_add (GTK_CONTAINER (app), vbox);
	
	g_signal_connect (G_OBJECT (app), "delete_event", G_CALLBACK (gtk_main_quit),NULL);
	g_signal_connect (G_OBJECT (play), "clicked", G_CALLBACK (cb_play),NULL);
	g_signal_connect (G_OBJECT (pause), "clicked", G_CALLBACK (cb_pause),NULL);
	g_signal_connect (G_OBJECT (stop), "clicked", G_CALLBACK (cb_stop),NULL);
	g_signal_connect (G_OBJECT (filesel), "clicked", G_CALLBACK (cb_choose),NULL);
	g_signal_connect (G_OBJECT (properties), "clicked", G_CALLBACK (cb_properties), NULL);

	gtk_widget_show_all (app);

	gtk_main ();
}
