/*
 * wsdl-parse.h: Parse a WSDL file, building up a tree of structs
 *
 * Authors:
 *	Dick Porter (dick@ximian.com)
 *
 * Copyright (C) 2001, Ximian, Inc.
 */

#ifndef _WSDL_PARSE_H_
#define _WSDL_PARSE_H_

#include <glib.h>
#include <tree.h>

#include "wsdl-typecodes.h"

/*
 * State diagram:
 *
 * WSDL_DEFINITIONS
 *	IMPORT
 *	WSDL_DOCUMENTATION
 *	WSDL_TYPES
 *		WSDL_DOCUMENTATION
 *		XSD_SCHEMA
 *		!!!extensibility element
 *	WSDL_MESSAGE
 *		WSDL_DOCUMENTATION
 *		PART
 *	WSDL_PORTTYPE
 *		WSDL_DOCUMENTATION
 *		WSDL_OPERATION
 *			WSDL_DOCUMENTATION
 *			WSDL_INPUT
 *				WSDL_DOCUMENTATION
 *			WSDL_OUTPUT
 *				WSDL_DOCUMENTATION
 *			WSDL_FAULT
 *				WSDL_DOCUMENTATION
 *	WSDL_BINDING
 *		WSDL_DOCUMENTATION
 *		!!!extensibility element
 *		WSDL_OPERATION
 *			WSDL_DOCUMENTATION
 *			!!!extensibility element
 *			WSDL_INPUT
 *				WSDL_DOCUMENTATION
 *				!!!extensibility element
 *			WSDL_OUTPUT
 *				WSDL_DOCUMENTATION
 *				!!!extensibility element
 *			WSDL_FAULT
 *				WSDL_DOCUMENTATION
 *				!!!extensibility element
 *	WSDL_SERVICE
 *		WSDL_DOCUMENTATION
 *		WSDL_PORT
 *			WSDL_DOCUMENTATION
 *			!!!extensibility element
 *		!!!extensibility element
 *	!!!extensibility element
 */

typedef enum {
	WSDL_START,
	WSDL_DEFINITIONS,
	WSDL_DOCUMENTATION,
	WSDL_TYPES,
	WSDL_TYPES_SCHEMA,
	WSDL_MESSAGE,
	WSDL_MESSAGE_PART,
	WSDL_PORTTYPE,
	WSDL_PORTTYPE_OPERATION,
	WSDL_PORTTYPE_OPERATION_INPUT,
	WSDL_PORTTYPE_OPERATION_OUTPUT,
	WSDL_PORTTYPE_OPERATION_FAULT,
	WSDL_BINDING,
	WSDL_BINDING_OPERATION,
	WSDL_BINDING_OPERATION_INPUT,
	WSDL_BINDING_OPERATION_OUTPUT,
	WSDL_SOAP_BODY,
	WSDL_SOAP_HEADER,
	WSDL_SOAP_FAULT,
	WSDL_BINDING_OPERATION_FAULT,
	WSDL_SOAP_OPERATION,
	WSDL_SOAP_BINDING,
	WSDL_SERVICE,
	WSDL_SERVICE_PORT,
	WSDL_SOAP_ADDRESS,
	WSDL_FINISH,
	WSDL_UNKNOWN,
	WSDL_STATE_MAX,
} wsdl_state_t;

typedef struct
{
	GString *documentation;

	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this types
					 * element
					 */
} wsdl_types;

typedef struct
{
	guchar *name;
	guchar *targetNamespace;
	GString *documentation;

	wsdl_types *types;
	GSList *messages;		/* a list of wsdl_message */
	GSList *porttypes;		/* a list of wsdl_porttype */
	GSList *bindings;		/* a list of wsdl_binding */
	GSList *services;		/* a list of wsdl_service */

	/* Threaded structs: */
	GSList *thread_soap_services;	/* a list of wsdl_service */
	
	xmlDocPtr xml;			/* the entire XML document */
} wsdl_definitions;

typedef struct
{
	guchar *name;
	GString *documentation;
	GSList *parts;			/* a list of wsdl_message_part */

	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this message
					 * element
					 */
} wsdl_message;

typedef struct
{
	guchar *name;
	const wsdl_typecode *typecode;

	/* Threaded structs: */

	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this message part
					 * element
					 */
} wsdl_message_part;

typedef struct 
{
	guchar *name;
	GString *documentation;
	GSList *operations;		/* a list of wsdl_porttype_operation */

	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this portType
					 * element
					 */
} wsdl_porttype;

typedef struct
{
	guchar *name;
	guchar *message;
	GString *documentation;

	/* Threaded structs: */
	wsdl_message *thread_soap_message;
	GSList *thread_soap_parts;	/* a list of wsdl_message_part */
	
	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this portType
					 * operation input, output or
					 * fault element
					 */
} wsdl_porttype_operation_inoutfault;

typedef struct 
{
	guchar *name;
	GString *documentation;
	gboolean solicit;
	wsdl_porttype_operation_inoutfault *input;
	wsdl_porttype_operation_inoutfault *output;
	GSList *faults;	/* a list of wsdl_porttype_operation_inoutfault */

	/* Threaded structs */
	GSList thread_soap_parts;	/* a list of wsdl_message_part */

	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this portType
					 * operation element
					 */
} wsdl_porttype_operation;

typedef struct
{
	guchar *style;
	guchar *transport;

	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this soap
					 * binding element
					 */
} wsdl_soap_binding;

typedef struct
{
	guchar *name;
	guchar *type;
	GString *documentation;
	wsdl_soap_binding *soap_binding;
	GSList *operations;		/* a list of wsdl_binding_operation */

	/* Threaded structs: */
	wsdl_porttype *thread_soap_porttype;
	
	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this binding
					 * element
					 */
} wsdl_binding;

typedef struct 
{
	guchar *parts;
	guchar *use;
	guchar *encodingStyle;
	guchar *namespace;

	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this soap
					 * body element
					 */
} wsdl_soap_body;

typedef struct 
{
	guchar *name;
	guchar *use;
	guchar *encodingStyle;
	guchar *namespace;

	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this soap
					 * fault element
					 */
} wsdl_soap_fault;

typedef struct
{
	guchar *name;
	GString *documentation;
	wsdl_soap_body *soap_body;
	wsdl_soap_fault *soap_fault;
	GSList *soap_headers;		/* a list of wsdl_soap_header */
	
	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this binding
					 * operation input, output or
					 * fault element
					 */
} wsdl_binding_operation_inoutfault;

typedef struct 
{
	guchar *element;
	guchar *fault;

	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this soap
					 * header element
					 */
} wsdl_soap_header;

typedef struct 
{
	guchar *soapAction;
	guchar *style;

	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this soap
					 * operation element
					 */
} wsdl_soap_operation;

typedef struct 
{
	guchar *name;
	GString *documentation;
	gboolean solicit;
	wsdl_soap_operation *soap_operation;
	wsdl_binding_operation_inoutfault *input;
	wsdl_binding_operation_inoutfault *output;
	GSList *faults;	/* a list of wsdl_binding_operation_inoutfault */

	/* Threaded structs: */
	wsdl_porttype_operation *thread_soap_porttype_operation;
	
	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this binding
					 * operation element
					 */
} wsdl_binding_operation;

typedef struct
{
	guchar *name;
	GString *documentation;
	GSList *ports;			/* a list of wsdl_service_port */

	/* Threaded structs: */
	GSList *thread_soap_ports;	/* a list of wsdl_service_port */
	
	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this service
					 * element
					 */
} wsdl_service;

typedef struct 
{
	guchar *location;

	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this soap
					 * address element
					 */
} wsdl_soap_address;

typedef struct
{
	guchar *name;
	guchar *binding;
	GString *documentation;
	wsdl_soap_address *soap_address;

	/* Threaded structs: */
	wsdl_binding *thread_soap_binding;
	
	xmlNodePtr xml_node;		/* the fragment of the xml doc
					 * that contains this service
					 * port element
					 */
} wsdl_service_port;

typedef struct 
{
	wsdl_state_t state, last_known_state, doc_prev_state, binding_operation_state;
	guint unknown_depth, types_depth;
	GString **documentation;
	wsdl_definitions *definitions;
	wsdl_types *types;
	wsdl_message *message;
	wsdl_message_part *message_part;
	wsdl_porttype *porttype;
	wsdl_porttype_operation *porttype_operation;
	wsdl_porttype_operation_inoutfault *porttype_operation_iof;
	wsdl_binding *binding;
	wsdl_binding_operation *binding_operation;
	wsdl_binding_operation_inoutfault *binding_operation_iof;
	wsdl_soap_operation *soap_operation;
	wsdl_soap_body *soap_body;
	wsdl_soap_header *soap_header;
	wsdl_soap_fault *soap_fault;
	wsdl_soap_binding *soap_binding;
	wsdl_service *service;
	wsdl_service_port *service_port;
	wsdl_soap_address *soap_address;
} wsdl_state;

extern wsdl_definitions *wsdl_parse(const char *file);

extern void wsdl_parse_warning(void *, const char *, ...);
extern void wsdl_parse_error(void *, const char *, ...);
extern void wsdl_parse_fatal(void *, const char *, ...);

#endif /* _WSDL_PARSE_H_ */
