/* valasignal.vala
 *
 * Copyright (C) 2006  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valasignal.h"
#include <vala/valatypereference.h>
#include <vala/valacallback.h>
#include <vala/valasourcereference.h>
#include <vala/valasignal.h>
#include <vala/valaformalparameter.h>
#include <vala/valasymbol.h>
#include <vala/valadatatype.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>

struct _ValaSignalPrivate {
	char* _name;
	ValaTypeReference* _return_type;
	ValaMemberAccessibility _access;
	gboolean _has_emitter;
	GList* parameters;
	ValaCallback* generated_callback;
	char* cname;
	gboolean lock_used;
};
#define VALA_SIGNAL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SIGNAL, ValaSignalPrivate))
enum  {
	VALA_SIGNAL_DUMMY_PROPERTY,
	VALA_SIGNAL_NAME,
	VALA_SIGNAL_RETURN_TYPE,
	VALA_SIGNAL_ACCESS,
	VALA_SIGNAL_HAS_EMITTER,
};
static GList* vala_signal_real_get_parameters (ValaInvokable* base);
static ValaTypeReference* vala_signal_real_get_return_type (ValaInvokable* base);
static gboolean vala_signal_real_is_invokable (ValaInvokable* base);
static void vala_signal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_signal_real_get_lock_used (ValaLockable* base);
static void vala_signal_real_set_lock_used (ValaLockable* base, gboolean used);
static gpointer vala_signal_parent_class = NULL;
static ValaInvokableIface* vala_signal_vala_invokable_parent_iface = NULL;
static ValaLockableIface* vala_signal_vala_lockable_parent_iface = NULL;
static void vala_signal_dispose (GObject * obj);


/**
 * Creates a new signal.
 *
 * @param name        signal name
 * @param return_type signal return type
 * @param source      reference to source code
 * @return            newly created signal
 */
ValaSignal* vala_signal_new (const char* _name, ValaTypeReference* _return_type, ValaSourceReference* source)
{
	g_return_val_if_fail (_name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (_return_type), NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	GParameter * __params = g_new0 (GParameter, 3);
	GParameter * __params_it = __params;
	(__params_it->name = "name", g_value_init (&__params_it->value, G_TYPE_STRING), g_value_set_string (&__params_it->value, _name), __params_it++);
	(__params_it->name = "return-type", g_value_init (&__params_it->value, VALA_TYPE_TYPE_REFERENCE), g_value_set_object (&__params_it->value, _return_type), __params_it++);
	(__params_it->name = "source-reference", g_value_init (&__params_it->value, VALA_TYPE_SOURCE_REFERENCE), g_value_set_object (&__params_it->value, source), __params_it++);
	ValaSignal * self = g_object_newv (VALA_TYPE_SIGNAL, __params_it - __params, __params);
	return self;
}


/**
 * Appends parameter to signal handler.
 *
 * @param param a formal parameter
 */
void vala_signal_add_parameter (ValaSignal* self, ValaFormalParameter* param)
{
	g_return_if_fail (VALA_IS_SIGNAL (self));
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (param));
	self->priv->parameters = g_list_append (self->priv->parameters, g_object_ref (param));
}


static GList* vala_signal_real_get_parameters (ValaInvokable* base)
{
	ValaSignal * self = VALA_SIGNAL (base);
	return g_list_copy (self->priv->parameters);
}


static ValaTypeReference* vala_signal_real_get_return_type (ValaInvokable* base)
{
	ValaSignal * self = VALA_SIGNAL (base);
	return g_object_ref (vala_signal_get_return_type (self));
}


static gboolean vala_signal_real_is_invokable (ValaInvokable* base)
{
	ValaSignal * self = VALA_SIGNAL (base);
	return TRUE;
}


/**
 * Returns generated callback to be used for signal handlers.
 *
 * @return callback
 */
ValaCallback* vala_signal_get_callback (ValaSignal* self)
{
	g_return_val_if_fail (VALA_IS_SIGNAL (self), NULL);
	if (self->priv->generated_callback == NULL) {
		ValaCallback* __temp3 = NULL;
		self->priv->generated_callback = (__temp3 = vala_callback_new (NULL, vala_signal_get_return_type (self), NULL), (self->priv->generated_callback == NULL ? NULL : (self->priv->generated_callback = (g_object_unref (self->priv->generated_callback), NULL))), __temp3);
		(vala_callback_set_instance (self->priv->generated_callback, TRUE), vala_callback_get_instance (self->priv->generated_callback));
		ValaTypeReference* sender_type = vala_type_reference_new ();
		(vala_type_reference_set_data_type (sender_type, VALA_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (self)))))), vala_type_reference_get_data_type (sender_type));
		ValaFormalParameter* sender_param = vala_formal_parameter_new ("sender", sender_type, NULL);
		vala_callback_add_parameter (self->priv->generated_callback, sender_param);
		{
			GList* __temp4 = NULL;
			__temp4 = self->priv->parameters;
			GList* param_it;
			for (param_it = __temp4; param_it != NULL; param_it = param_it->next) {
				ValaFormalParameter* param = param_it->data;
				{
					vala_callback_add_parameter (self->priv->generated_callback, param);
				}
			}
		}
		(sender_type == NULL ? NULL : (sender_type = (g_object_unref (sender_type), NULL)));
		(sender_param == NULL ? NULL : (sender_param = (g_object_unref (sender_param), NULL)));
	}
	ValaCallback* __temp5 = NULL;
	return (__temp5 = self->priv->generated_callback, (__temp5 == NULL ? NULL : g_object_ref (__temp5)));
}


/**
 * Returns the name of this signal as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_signal_get_cname (ValaSignal* self)
{
	g_return_val_if_fail (VALA_IS_SIGNAL (self), NULL);
	if (self->priv->cname == NULL) {
		char* __temp7 = NULL;
		self->priv->cname = (__temp7 = g_strdup (vala_signal_get_name (self)), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp7);
	}
	const char* __temp8 = NULL;
	return (__temp8 = self->priv->cname, (__temp8 == NULL ? NULL : g_strdup (__temp8)));
}


void vala_signal_set_cname (ValaSignal* self, const char* cname)
{
	g_return_if_fail (VALA_IS_SIGNAL (self));
	char* __temp11 = NULL;
	const char* __temp10 = NULL;
	self->priv->cname = (__temp11 = (__temp10 = cname, (__temp10 == NULL ? NULL : g_strdup (__temp10))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp11);
}


/**
 * Returns the string literal of this signal to be used in C code.
 *
 * @return string literal to be used in C code
 */
ValaCCodeConstant* vala_signal_get_canonical_cconstant (ValaSignal* self)
{
	g_return_val_if_fail (VALA_IS_SIGNAL (self), NULL);
	GString* str = g_string_new ("\"");
	char* i = g_strdup (vala_signal_get_name (self));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c = g_utf8_get_char (i);
		if (c == '_') {
			g_string_append_c (str, '-');
		} else {
			g_string_append_unichar (str, c);
		}
		char* __temp13 = NULL;
		const char* __temp12 = NULL;
		i = (__temp13 = (__temp12 = g_utf8_next_char (i), (__temp12 == NULL ? NULL : g_strdup (__temp12))), (i = (g_free (i), NULL)), __temp13);
	}
	g_string_append_c (str, '"');
	ValaCCodeConstant* __temp14 = NULL;
	return (__temp14 = vala_ccode_constant_new (str->str), (str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL))), (i = (g_free (i), NULL)), __temp14);
	(str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL)));
	(i = (g_free (i), NULL));
}


static void vala_signal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaSignal * self = VALA_SIGNAL (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_member (visitor, VALA_MEMBER (self));
	vala_code_visitor_visit_begin_signal (visitor, self);
	vala_code_node_accept (VALA_CODE_NODE (vala_signal_get_return_type (self)), visitor);
	{
		GList* __temp15 = NULL;
		__temp15 = self->priv->parameters;
		GList* param_it;
		for (param_it = __temp15; param_it != NULL; param_it = param_it->next) {
			ValaFormalParameter* param = param_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (param), visitor);
			}
		}
	}
	vala_code_visitor_visit_end_signal (visitor, self);
}


/**
 * Process all associated attributes.
 */
void vala_signal_process_attributes (ValaSignal* self)
{
	g_return_if_fail (VALA_IS_SIGNAL (self));
	{
		GList* __temp16 = NULL;
		__temp16 = VALA_CODE_NODE (self)->attributes;
		GList* a_it;
		for (a_it = __temp16; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* a = a_it->data;
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "HasEmitter") == 0) {
					(vala_signal_set_has_emitter (self, TRUE), vala_signal_get_has_emitter (self));
				}
			}
		}
	}
}


static gboolean vala_signal_real_get_lock_used (ValaLockable* base)
{
	ValaSignal * self = VALA_SIGNAL (base);
	return self->priv->lock_used;
}


static void vala_signal_real_set_lock_used (ValaLockable* base, gboolean used)
{
	ValaSignal * self = VALA_SIGNAL (base);
	self->priv->lock_used = used;
}


char* vala_signal_get_name (ValaSignal* self)
{
	g_return_val_if_fail (VALA_IS_SIGNAL (self), NULL);
	return self->priv->_name;
}


void vala_signal_set_name (ValaSignal* self, const char* value)
{
	g_return_if_fail (VALA_IS_SIGNAL (self));
	char* __temp19 = NULL;
	self->priv->_name = (__temp19 = g_strdup (value), (self->priv->_name = (g_free (self->priv->_name), NULL)), __temp19);
}


ValaTypeReference* vala_signal_get_return_type (ValaSignal* self)
{
	g_return_val_if_fail (VALA_IS_SIGNAL (self), NULL);
	return self->priv->_return_type;
}


void vala_signal_set_return_type (ValaSignal* self, ValaTypeReference* value)
{
	g_return_if_fail (VALA_IS_SIGNAL (self));
	ValaTypeReference* __temp21 = NULL;
	self->priv->_return_type = (__temp21 = g_object_ref (value), (self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL))), __temp21);
}


ValaMemberAccessibility vala_signal_get_access (ValaSignal* self)
{
	g_return_val_if_fail (VALA_IS_SIGNAL (self), 0);
	return self->priv->_access;
}


void vala_signal_set_access (ValaSignal* self, ValaMemberAccessibility value)
{
	g_return_if_fail (VALA_IS_SIGNAL (self));
	self->priv->_access = value;
}


gboolean vala_signal_get_has_emitter (ValaSignal* self)
{
	g_return_val_if_fail (VALA_IS_SIGNAL (self), FALSE);
	return self->priv->_has_emitter;
}


void vala_signal_set_has_emitter (ValaSignal* self, gboolean value)
{
	g_return_if_fail (VALA_IS_SIGNAL (self));
	self->priv->_has_emitter = value;
}


static void vala_signal_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaSignal * self = VALA_SIGNAL (object);
	switch (property_id) {
		case VALA_SIGNAL_NAME:
		g_value_set_string (value, vala_signal_get_name (self));
		break;
		case VALA_SIGNAL_RETURN_TYPE:
		g_value_set_object (value, vala_signal_get_return_type (self));
		break;
		case VALA_SIGNAL_ACCESS:
		g_value_set_int (value, vala_signal_get_access (self));
		break;
		case VALA_SIGNAL_HAS_EMITTER:
		g_value_set_boolean (value, vala_signal_get_has_emitter (self));
		break;
	}
}


static void vala_signal_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaSignal * self = VALA_SIGNAL (object);
	switch (property_id) {
		case VALA_SIGNAL_NAME:
		vala_signal_set_name (self, g_value_get_string (value));
		break;
		case VALA_SIGNAL_RETURN_TYPE:
		vala_signal_set_return_type (self, g_value_get_object (value));
		break;
		case VALA_SIGNAL_ACCESS:
		vala_signal_set_access (self, g_value_get_int (value));
		break;
		case VALA_SIGNAL_HAS_EMITTER:
		vala_signal_set_has_emitter (self, g_value_get_boolean (value));
		break;
	}
}


static void vala_signal_class_init (ValaSignalClass * klass)
{
	vala_signal_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSignalPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_signal_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_signal_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_signal_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_signal_real_accept;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SIGNAL_NAME, g_param_spec_string ("name", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SIGNAL_RETURN_TYPE, g_param_spec_object ("return-type", "foo", "bar", VALA_TYPE_TYPE_REFERENCE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SIGNAL_ACCESS, g_param_spec_int ("access", "foo", "bar", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SIGNAL_HAS_EMITTER, g_param_spec_boolean ("has-emitter", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_signal_vala_invokable_interface_init (ValaInvokableIface * iface)
{
	vala_signal_vala_invokable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_parameters = vala_signal_real_get_parameters;
	iface->get_return_type = vala_signal_real_get_return_type;
	iface->is_invokable = vala_signal_real_is_invokable;
}


static void vala_signal_vala_lockable_interface_init (ValaLockableIface * iface)
{
	vala_signal_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_signal_real_get_lock_used;
	iface->set_lock_used = vala_signal_real_set_lock_used;
}


static void vala_signal_init (ValaSignal * self)
{
	self->priv = VALA_SIGNAL_GET_PRIVATE (self);
	self->priv->lock_used = FALSE;
}


static void vala_signal_dispose (GObject * obj)
{
	ValaSignal * self = VALA_SIGNAL (obj);
	(self->priv->_name = (g_free (self->priv->_name), NULL));
	(self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL)));
	(self->priv->parameters == NULL ? NULL : (self->priv->parameters = (g_list_foreach (self->priv->parameters, (GFunc) g_object_unref, NULL), g_list_free (self->priv->parameters), NULL)));
	(self->priv->generated_callback == NULL ? NULL : (self->priv->generated_callback = (g_object_unref (self->priv->generated_callback), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	ValaSignalClass * klass;
	GObjectClass * parent_class;
	klass = VALA_SIGNAL_CLASS (g_type_class_peek (VALA_TYPE_SIGNAL));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_signal_get_type ()
{
	static GType vala_signal_type_id = 0;
	if (G_UNLIKELY (vala_signal_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSignalClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_signal_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSignal), 0, (GInstanceInitFunc) vala_signal_init };
		vala_signal_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaSignal", &g_define_type_info, 0);
		static const GInterfaceInfo vala_invokable_info = { (GInterfaceInitFunc) vala_signal_vala_invokable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		g_type_add_interface_static (vala_signal_type_id, VALA_TYPE_INVOKABLE, &vala_invokable_info);
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_signal_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		g_type_add_interface_static (vala_signal_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_signal_type_id;
}




