/* valaclass.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaclass.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valatypeparameter.h>
#include <vala/valatypereference.h>
#include <vala/valaconstant.h>
#include <vala/valafield.h>
#include <vala/valamethod.h>
#include <vala/valaproperty.h>
#include <vala/valasignal.h>
#include <vala/valastruct.h>
#include <vala/valaconstructor.h>
#include <vala/valadestructor.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valacreationmethod.h>
#include <vala/valaformalparameter.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valasourcefile.h>
#include <vala/valaexpression.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>

struct _ValaClassPrivate {
	ValaClass* _base_class;
	gboolean _is_abstract;
	gboolean _is_static;
	char* cname;
	char* const_cname;
	char* lower_case_cprefix;
	char* lower_case_csuffix;
	gboolean has_type_id;
	char* type_id;
	char* ref_function;
	char* unref_function;
	char* copy_function;
	char* free_function;
	char* marshaller_type_name;
	char* get_value_function;
	char* set_value_function;
	gboolean _has_private_fields;
	GeeList* type_parameters;
	GeeList* base_types;
	GeeList* constants;
	GeeList* fields;
	GeeList* methods;
	GeeList* properties;
	GeeList* signals;
	GeeList* classes;
	GeeList* structs;
	ValaMethod* _default_construction_method;
	ValaConstructor* _constructor;
	ValaDestructor* _destructor;
};
#define VALA_CLASS_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CLASS, ValaClassPrivate))
enum  {
	VALA_CLASS_DUMMY_PROPERTY,
	VALA_CLASS_BASE_CLASS,
	VALA_CLASS_IS_ABSTRACT,
	VALA_CLASS_IS_STATIC,
	VALA_CLASS_HAS_PRIVATE_FIELDS,
	VALA_CLASS_DEFAULT_CONSTRUCTION_METHOD,
	VALA_CLASS_CONSTRUCTOR,
	VALA_CLASS_DESTRUCTOR
};
static void vala_class_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_class_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_class_real_get_cprefix (ValaSymbol* base);
static char* vala_class_real_get_cname (ValaDataType* base, gboolean const_type);
static char* vala_class_get_lower_case_csuffix (ValaClass* self);
static char* vala_class_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
static char* vala_class_real_get_lower_case_cprefix (ValaSymbol* base);
static char* vala_class_real_get_upper_case_cname (ValaDataType* base, const char* infix);
static gboolean vala_class_real_is_reference_type (ValaDataType* base);
static void vala_class_process_ccode_attribute (ValaClass* self, ValaAttribute* a);
static gboolean vala_class_get_has_type_id (ValaClass* self);
static char* vala_class_real_get_type_id (ValaDataType* base);
static char* vala_class_real_get_marshaller_type_name (ValaDataType* base);
static char* vala_class_real_get_get_value_function (ValaDataType* base);
static char* vala_class_real_get_set_value_function (ValaDataType* base);
static gboolean vala_class_real_is_reference_counting (ValaDataType* base);
static char* vala_class_real_get_ref_function (ValaDataType* base);
static char* vala_class_real_get_unref_function (ValaDataType* base);
static char* vala_class_real_get_dup_function (ValaDataType* base);
static char* vala_class_real_get_free_function (ValaDataType* base);
static gboolean vala_class_real_is_subtype_of (ValaDataType* base, ValaDataType* t);
static gint vala_class_real_get_type_parameter_index (ValaDataType* base, const char* name);
static gpointer vala_class_parent_class = NULL;
static void vala_class_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);


/**
 * Creates a new class.
 *
 * @param name   type name
 * @param source reference to source code
 * @return       newly created class
 */
ValaClass* vala_class_new (const char* name, ValaSourceReference* source_reference) {
	ValaClass * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_CLASS, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Adds the specified class or interface to the list of base types of
 * this class.
 *
 * @param type a class or interface reference
 */
void vala_class_add_base_type (ValaClass* self, ValaTypeReference* type) {
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
	gee_collection_add (GEE_COLLECTION (self->priv->base_types), type);
}


/**
 * Returns a copy of the base type list.
 *
 * @return list of base types
 */
GeeCollection* vala_class_get_base_types (ValaClass* self) {
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_TYPE_REFERENCE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->base_types)));
}


/**
 * Appends the specified parameter to the list of type parameters.
 *
 * @param p a type parameter
 */
void vala_class_add_type_parameter (ValaClass* self, ValaTypeParameter* p) {
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_TYPE_PARAMETER (p));
	gee_collection_add (GEE_COLLECTION (self->priv->type_parameters), p);
	p->type = VALA_DATA_TYPE (self);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (p)), VALA_SYMBOL (p));
}


/**
 * Returns a copy of the type parameter list.
 *
 * @return list of type parameters
 */
GeeCollection* vala_class_get_type_parameters (ValaClass* self) {
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_TYPE_PARAMETER, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->type_parameters)));
}


/**
 * Adds the specified constant as a member to this class.
 *
 * @param c a constant
 */
void vala_class_add_constant (ValaClass* self, ValaConstant* c) {
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_CONSTANT (c));
	gee_collection_add (GEE_COLLECTION (self->priv->constants), c);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (c)), VALA_SYMBOL (c));
}


/**
 * Adds the specified field as a member to this class.
 *
 * @param f a field
 */
void vala_class_add_field (ValaClass* self, ValaField* f) {
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_FIELD (f));
	/* non_null fields not yet supported due to initialization issues*/
	vala_type_reference_set_non_null (vala_field_get_type_reference (f), FALSE);
	gee_collection_add (GEE_COLLECTION (self->priv->fields), f);
	if (vala_symbol_get_access (VALA_SYMBOL (f)) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE && vala_field_get_instance (f)) {
		self->priv->_has_private_fields = TRUE;
	}
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (f)), VALA_SYMBOL (f));
}


/**
 * Returns a copy of the list of fields.
 *
 * @return list of fields
 */
GeeCollection* vala_class_get_fields (ValaClass* self) {
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_FIELD, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->fields)));
}


/**
 * Adds the specified method as a member to this class.
 *
 * @param m a method
 */
void vala_class_add_method (ValaClass* self, ValaMethod* m) {
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	if (vala_method_get_instance (m) || VALA_IS_CREATION_METHOD (m)) {
		ValaFormalParameter* _tmp1;
		ValaTypeReference* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_method_set_this_parameter (m, (_tmp1 = vala_formal_parameter_new ("this", (_tmp0 = vala_type_reference_new ()), NULL)));
		(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
		(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
		vala_type_reference_set_data_type (vala_formal_parameter_get_type_reference (vala_method_get_this_parameter (m)), VALA_DATA_TYPE (self));
		vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (m)), vala_symbol_get_name (VALA_SYMBOL (vala_method_get_this_parameter (m))), VALA_SYMBOL (vala_method_get_this_parameter (m)));
	}
	if (VALA_IS_CREATION_METHOD (m) && vala_symbol_get_name (VALA_SYMBOL (m)) == NULL) {
		vala_class_set_default_construction_method (self, m);
	}
	gee_collection_add (GEE_COLLECTION (self->priv->methods), m);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (m)), VALA_SYMBOL (m));
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
GeeCollection* vala_class_get_methods (ValaClass* self) {
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->methods)));
}


/**
 * Adds the specified property as a member to this class.
 *
 * @param prop a property
 */
void vala_class_add_property (ValaClass* self, ValaProperty* prop, gboolean no_field) {
	ValaFormalParameter* _tmp1;
	ValaTypeReference* _tmp0;
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	gee_collection_add (GEE_COLLECTION (self->priv->properties), prop);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (prop)), VALA_SYMBOL (prop));
	_tmp1 = NULL;
	_tmp0 = NULL;
	vala_property_set_this_parameter (prop, (_tmp1 = vala_formal_parameter_new ("this", (_tmp0 = vala_type_reference_new ()), NULL)));
	(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	vala_type_reference_set_data_type (vala_formal_parameter_get_type_reference (vala_property_get_this_parameter (prop)), VALA_DATA_TYPE (self));
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (prop)), vala_symbol_get_name (VALA_SYMBOL (vala_property_get_this_parameter (prop))), VALA_SYMBOL (vala_property_get_this_parameter (prop)));
	if (!no_field && vala_property_get_set_accessor (prop) != NULL && vala_property_accessor_get_body (vala_property_get_set_accessor (prop)) == NULL && vala_code_node_get_source_reference (VALA_CODE_NODE (self)) != NULL && !vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (self))))) {
		ValaTypeReference* field_type;
		char* _tmp2;
		ValaField* _tmp3;
		ValaField* f;
		/* automatic property accessor body generation */
		field_type = vala_type_reference_copy (vala_property_get_type_reference (prop));
		/* non_null fields not yet supported due to initialization issues*/
		vala_type_reference_set_non_null (field_type, FALSE);
		_tmp2 = NULL;
		_tmp3 = NULL;
		f = (_tmp3 = vala_field_new ((_tmp2 = g_strdup_printf ("_%s", vala_symbol_get_name (VALA_SYMBOL (prop)))), field_type, NULL, vala_code_node_get_source_reference (VALA_CODE_NODE (prop))), (_tmp2 = (g_free (_tmp2), NULL)), _tmp3);
		vala_symbol_set_access (VALA_SYMBOL (f), VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
		vala_class_add_field (self, f);
		(field_type = (g_object_unref (field_type), NULL));
		(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
	}
}


/**
 * Returns a copy of the list of properties.
 *
 * @return list of properties
 */
GeeCollection* vala_class_get_properties (ValaClass* self) {
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_PROPERTY, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->properties)));
}


/**
 * Adds the specified signal as a member to this class.
 *
 * @param sig a signal
 */
void vala_class_add_signal (ValaClass* self, ValaSignal* sig) {
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_SIGNAL (sig));
	gee_collection_add (GEE_COLLECTION (self->priv->signals), sig);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (sig)), VALA_SYMBOL (sig));
}


/**
 * Returns a copy of the list of signals.
 *
 * @return list of signals
 */
GeeCollection* vala_class_get_signals (ValaClass* self) {
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_SIGNAL, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->signals)));
}


/**
 * Adds the specified class as an inner class.
 *
 * @param cl a class
 */
void vala_class_add_class (ValaClass* self, ValaClass* cl) {
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_CLASS (cl));
	gee_collection_add (GEE_COLLECTION (self->priv->classes), cl);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (cl)), VALA_SYMBOL (cl));
}


/**
 * Adds the specified struct as an inner struct.
 *
 * @param st a struct
 */
void vala_class_add_struct (ValaClass* self, ValaStruct* st) {
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_STRUCT (st));
	gee_collection_add (GEE_COLLECTION (self->priv->structs), st);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (st)), VALA_SYMBOL (st));
}


static void vala_class_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaClass * self;
	self = VALA_CLASS (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_class (visitor, self);
}


static void vala_class_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaClass * self;
	self = VALA_CLASS (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* type_collection;
		GeeIterator* type_it;
		type_collection = self->priv->base_types;
		type_it = gee_iterable_iterator (GEE_ITERABLE (type_collection));
		while (gee_iterator_next (type_it)) {
			ValaTypeReference* type;
			type = gee_iterator_get (type_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (type), visitor);
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
	}
	{
		GeeList* p_collection;
		GeeIterator* p_it;
		p_collection = self->priv->type_parameters;
		p_it = gee_iterable_iterator (GEE_ITERABLE (p_collection));
		while (gee_iterator_next (p_it)) {
			ValaTypeParameter* p;
			p = gee_iterator_get (p_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (p), visitor);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
		(p_it == NULL ? NULL : (p_it = (g_object_unref (p_it), NULL)));
	}
	{
		GeeList* f_collection;
		GeeIterator* f_it;
		f_collection = self->priv->fields;
		f_it = gee_iterable_iterator (GEE_ITERABLE (f_collection));
		while (gee_iterator_next (f_it)) {
			ValaField* f;
			f = gee_iterator_get (f_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (f), visitor);
				(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
			}
		}
		(f_it == NULL ? NULL : (f_it = (g_object_unref (f_it), NULL)));
	}
	{
		GeeList* c_collection;
		GeeIterator* c_it;
		c_collection = self->priv->constants;
		c_it = gee_iterable_iterator (GEE_ITERABLE (c_collection));
		while (gee_iterator_next (c_it)) {
			ValaConstant* c;
			c = gee_iterator_get (c_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (c), visitor);
				(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
			}
		}
		(c_it == NULL ? NULL : (c_it = (g_object_unref (c_it), NULL)));
	}
	{
		GeeList* m_collection;
		GeeIterator* m_it;
		m_collection = self->priv->methods;
		m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
		while (gee_iterator_next (m_it)) {
			ValaMethod* m;
			m = gee_iterator_get (m_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (m), visitor);
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
			}
		}
		(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
	}
	{
		GeeList* prop_collection;
		GeeIterator* prop_it;
		prop_collection = self->priv->properties;
		prop_it = gee_iterable_iterator (GEE_ITERABLE (prop_collection));
		while (gee_iterator_next (prop_it)) {
			ValaProperty* prop;
			prop = gee_iterator_get (prop_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (prop), visitor);
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
			}
		}
		(prop_it == NULL ? NULL : (prop_it = (g_object_unref (prop_it), NULL)));
	}
	{
		GeeList* sig_collection;
		GeeIterator* sig_it;
		sig_collection = self->priv->signals;
		sig_it = gee_iterable_iterator (GEE_ITERABLE (sig_collection));
		while (gee_iterator_next (sig_it)) {
			ValaSignal* sig;
			sig = gee_iterator_get (sig_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (sig), visitor);
				(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
			}
		}
		(sig_it == NULL ? NULL : (sig_it = (g_object_unref (sig_it), NULL)));
	}
	if (vala_class_get_constructor (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_class_get_constructor (self)), visitor);
	}
	if (vala_class_get_destructor (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_class_get_destructor (self)), visitor);
	}
	{
		GeeList* cl_collection;
		GeeIterator* cl_it;
		cl_collection = self->priv->classes;
		cl_it = gee_iterable_iterator (GEE_ITERABLE (cl_collection));
		while (gee_iterator_next (cl_it)) {
			ValaClass* cl;
			cl = gee_iterator_get (cl_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (cl), visitor);
				(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
			}
		}
		(cl_it == NULL ? NULL : (cl_it = (g_object_unref (cl_it), NULL)));
	}
	{
		GeeList* st_collection;
		GeeIterator* st_it;
		st_collection = self->priv->structs;
		st_it = gee_iterable_iterator (GEE_ITERABLE (st_collection));
		while (gee_iterator_next (st_it)) {
			ValaStruct* st;
			st = gee_iterator_get (st_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (st), visitor);
				(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
			}
		}
		(st_it == NULL ? NULL : (st_it = (g_object_unref (st_it), NULL)));
	}
}


static char* vala_class_real_get_cprefix (ValaSymbol* base) {
	ValaClass * self;
	self = VALA_CLASS (base);
	return vala_data_type_get_cname (VALA_DATA_TYPE (self), FALSE);
}


static char* vala_class_real_get_cname (ValaDataType* base, gboolean const_type) {
	ValaClass * self;
	const char* _tmp3;
	self = VALA_CLASS (base);
	if (const_type && self->priv->const_cname != NULL) {
		const char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = self->priv->const_cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
	}
	if (self->priv->cname == NULL) {
		char* _tmp2;
		_tmp2 = NULL;
		self->priv->cname = (_tmp2 = vala_class_get_default_cname (self), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp2);
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->cname, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
}


/**
 * Returns the default name of this class as it is used in C code.
 *
 * @return the name to be used in C code by default
 */
char* vala_class_get_default_cname (ValaClass* self) {
	char* _tmp0;
	char* _tmp1;
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_strdup_printf ("%s%s", (_tmp0 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), vala_symbol_get_name (VALA_SYMBOL (self))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


/**
 * Sets the name of this class as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_class_set_cname (ValaClass* self, const char* cname) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (cname != NULL);
	_tmp0 = NULL;
	self->priv->cname = (_tmp0 = g_strdup (cname), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp0);
}


static char* vala_class_get_lower_case_csuffix (ValaClass* self) {
	const char* _tmp1;
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->lower_case_csuffix = (_tmp0 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name (VALA_SYMBOL (self))), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->lower_case_csuffix, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


static char* vala_class_real_get_lower_case_cname (ValaSymbol* base, const char* infix) {
	ValaClass * self;
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	self = VALA_CLASS (base);
	if (infix == NULL) {
		infix = "";
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strdup_printf ("%s%s%s", (_tmp0 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), infix, (_tmp1 = vala_class_get_lower_case_csuffix (self))), (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2);
}


static char* vala_class_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaClass * self;
	const char* _tmp2;
	self = VALA_CLASS (base);
	if (self->priv->lower_case_cprefix == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->lower_case_cprefix = (_tmp1 = g_strdup_printf ("%s_", (_tmp0 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (self), NULL))), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), _tmp1);
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->lower_case_cprefix, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


static char* vala_class_real_get_upper_case_cname (ValaDataType* base, const char* infix) {
	ValaClass * self;
	char* _tmp0;
	char* _tmp1;
	self = VALA_CLASS (base);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_utf8_strup ((_tmp0 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (self), infix)), -1), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


static gboolean vala_class_real_is_reference_type (ValaDataType* base) {
	ValaClass * self;
	self = VALA_CLASS (base);
	return TRUE;
}


static void vala_class_process_ccode_attribute (ValaClass* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "ref_function")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_class_set_ref_function (self, (_tmp0 = vala_attribute_get_string (a, "ref_function")));
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	if (vala_attribute_has_argument (a, "unref_function")) {
		char* _tmp1;
		_tmp1 = NULL;
		vala_class_set_unref_function (self, (_tmp1 = vala_attribute_get_string (a, "unref_function")));
		(_tmp1 = (g_free (_tmp1), NULL));
	}
	if (vala_attribute_has_argument (a, "copy_function")) {
		char* _tmp2;
		_tmp2 = NULL;
		vala_class_set_dup_function (self, (_tmp2 = vala_attribute_get_string (a, "copy_function")));
		(_tmp2 = (g_free (_tmp2), NULL));
	}
	if (vala_attribute_has_argument (a, "free_function")) {
		char* _tmp3;
		_tmp3 = NULL;
		vala_class_set_free_function (self, (_tmp3 = vala_attribute_get_string (a, "free_function")));
		(_tmp3 = (g_free (_tmp3), NULL));
	}
	if (vala_attribute_has_argument (a, "has_type_id")) {
		self->priv->has_type_id = vala_attribute_get_bool (a, "has_type_id");
	}
	if (vala_attribute_has_argument (a, "type_id")) {
		char* _tmp4;
		_tmp4 = NULL;
		self->priv->type_id = (_tmp4 = vala_attribute_get_string (a, "type_id"), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), _tmp4);
	}
	if (vala_attribute_has_argument (a, "marshaller_type_name")) {
		char* _tmp5;
		_tmp5 = NULL;
		self->priv->marshaller_type_name = (_tmp5 = vala_attribute_get_string (a, "marshaller_type_name"), (self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL)), _tmp5);
	}
	if (vala_attribute_has_argument (a, "get_value_function")) {
		char* _tmp6;
		_tmp6 = NULL;
		self->priv->get_value_function = (_tmp6 = vala_attribute_get_string (a, "get_value_function"), (self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL)), _tmp6);
	}
	if (vala_attribute_has_argument (a, "set_value_function")) {
		char* _tmp7;
		_tmp7 = NULL;
		self->priv->set_value_function = (_tmp7 = vala_attribute_get_string (a, "set_value_function"), (self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL)), _tmp7);
	}
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp8;
		_tmp8 = NULL;
		vala_class_set_cname (self, (_tmp8 = vala_attribute_get_string (a, "cname")));
		(_tmp8 = (g_free (_tmp8), NULL));
	}
	if (vala_attribute_has_argument (a, "const_cname")) {
		char* _tmp9;
		_tmp9 = NULL;
		self->priv->const_cname = (_tmp9 = vala_attribute_get_string (a, "const_cname"), (self->priv->const_cname = (g_free (self->priv->const_cname), NULL)), _tmp9);
	}
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* _tmp10;
		_tmp10 = NULL;
		self->priv->lower_case_cprefix = (_tmp10 = vala_attribute_get_string (a, "cprefix"), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), _tmp10);
	}
	if (vala_attribute_has_argument (a, "lower_case_csuffix")) {
		char* _tmp11;
		_tmp11 = NULL;
		self->priv->lower_case_csuffix = (_tmp11 = vala_attribute_get_string (a, "lower_case_csuffix"), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), _tmp11);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			char** filename_it;
			int filename_collection_length1;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp12;
				char* filename;
				_tmp12 = NULL;
				filename = (_tmp12 = *filename_it, (_tmp12 == NULL ? NULL : g_strdup (_tmp12)));
				{
					vala_data_type_add_cheader_filename (VALA_DATA_TYPE (self), filename);
					(filename = (g_free (filename), NULL));
				}
			}
			(filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, ((GDestroyNotify) g_free)), NULL));
		}
		(val = (g_free (val), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_class_process_attributes (ValaClass* self) {
	g_return_if_fail (VALA_IS_CLASS (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = a_it->data, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_class_process_ccode_attribute (self, a);
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


static gboolean vala_class_get_has_type_id (ValaClass* self) {
	g_return_val_if_fail (VALA_IS_CLASS (self), FALSE);
	return self->priv->has_type_id || (vala_class_get_base_class (self) != NULL && vala_class_get_has_type_id (vala_class_get_base_class (self)));
}


static char* vala_class_real_get_type_id (ValaDataType* base) {
	ValaClass * self;
	const char* _tmp2;
	self = VALA_CLASS (base);
	if (self->priv->type_id == NULL) {
		if (vala_class_get_has_type_id (self)) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->type_id = (_tmp0 = vala_data_type_get_upper_case_cname (VALA_DATA_TYPE (self), "TYPE_"), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), _tmp0);
		} else {
			char* _tmp1;
			_tmp1 = NULL;
			self->priv->type_id = (_tmp1 = g_strdup ("G_TYPE_POINTER"), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), _tmp1);
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->type_id, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


void vala_class_set_type_id (ValaClass* self, const char* type_id) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (type_id != NULL);
	_tmp0 = NULL;
	self->priv->type_id = (_tmp0 = g_strdup (type_id), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), _tmp0);
}


static char* vala_class_real_get_marshaller_type_name (ValaDataType* base) {
	ValaClass * self;
	const char* _tmp2;
	self = VALA_CLASS (base);
	if (self->priv->marshaller_type_name == NULL) {
		if (vala_class_get_base_class (self) != NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->marshaller_type_name = (_tmp0 = vala_data_type_get_marshaller_type_name (VALA_DATA_TYPE (vala_class_get_base_class (self))), (self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL)), _tmp0);
		} else {
			char* _tmp1;
			_tmp1 = NULL;
			self->priv->marshaller_type_name = (_tmp1 = g_strdup ("POINTER"), (self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL)), _tmp1);
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->marshaller_type_name, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


static char* vala_class_real_get_get_value_function (ValaDataType* base) {
	ValaClass * self;
	const char* _tmp2;
	self = VALA_CLASS (base);
	if (self->priv->get_value_function == NULL) {
		if (vala_class_get_base_class (self) != NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->get_value_function = (_tmp0 = vala_data_type_get_get_value_function (VALA_DATA_TYPE (vala_class_get_base_class (self))), (self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL)), _tmp0);
		} else {
			char* _tmp1;
			_tmp1 = NULL;
			self->priv->get_value_function = (_tmp1 = g_strdup ("g_value_get_pointer"), (self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL)), _tmp1);
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->get_value_function, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


static char* vala_class_real_get_set_value_function (ValaDataType* base) {
	ValaClass * self;
	const char* _tmp2;
	self = VALA_CLASS (base);
	if (self->priv->set_value_function == NULL) {
		if (vala_class_get_base_class (self) != NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->set_value_function = (_tmp0 = vala_data_type_get_set_value_function (VALA_DATA_TYPE (vala_class_get_base_class (self))), (self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL)), _tmp0);
		} else {
			char* _tmp1;
			_tmp1 = NULL;
			self->priv->set_value_function = (_tmp1 = g_strdup ("g_value_set_pointer"), (self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL)), _tmp1);
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->set_value_function, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


static gboolean vala_class_real_is_reference_counting (ValaDataType* base) {
	ValaClass * self;
	char* _tmp0;
	gboolean _tmp1;
	self = VALA_CLASS (base);
	_tmp0 = NULL;
	return (_tmp1 = (_tmp0 = vala_data_type_get_ref_function (VALA_DATA_TYPE (self))) != NULL, (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


static char* vala_class_real_get_ref_function (ValaDataType* base) {
	ValaClass * self;
	self = VALA_CLASS (base);
	if (self->priv->ref_function == NULL && vala_class_get_base_class (self) != NULL) {
		return vala_data_type_get_ref_function (VALA_DATA_TYPE (vala_class_get_base_class (self)));
	} else {
		const char* _tmp1;
		_tmp1 = NULL;
		return (_tmp1 = self->priv->ref_function, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
	}
}


void vala_class_set_ref_function (ValaClass* self, const char* name) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (name != NULL);
	_tmp0 = NULL;
	self->priv->ref_function = (_tmp0 = g_strdup (name), (self->priv->ref_function = (g_free (self->priv->ref_function), NULL)), _tmp0);
}


static char* vala_class_real_get_unref_function (ValaDataType* base) {
	ValaClass * self;
	self = VALA_CLASS (base);
	if (self->priv->unref_function == NULL && vala_class_get_base_class (self) != NULL) {
		return vala_data_type_get_unref_function (VALA_DATA_TYPE (vala_class_get_base_class (self)));
	} else {
		const char* _tmp1;
		_tmp1 = NULL;
		return (_tmp1 = self->priv->unref_function, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
	}
}


void vala_class_set_unref_function (ValaClass* self, const char* name) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (name != NULL);
	_tmp0 = NULL;
	self->priv->unref_function = (_tmp0 = g_strdup (name), (self->priv->unref_function = (g_free (self->priv->unref_function), NULL)), _tmp0);
}


static char* vala_class_real_get_dup_function (ValaDataType* base) {
	ValaClass * self;
	const char* _tmp0;
	self = VALA_CLASS (base);
	_tmp0 = NULL;
	return (_tmp0 = self->priv->copy_function, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
}


void vala_class_set_dup_function (ValaClass* self, const char* name) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (name != NULL);
	_tmp0 = NULL;
	self->priv->copy_function = (_tmp0 = g_strdup (name), (self->priv->copy_function = (g_free (self->priv->copy_function), NULL)), _tmp0);
}


char* vala_class_get_default_free_function (ValaClass* self) {
	char* _tmp0;
	char* _tmp1;
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_strconcat ((_tmp0 = vala_symbol_get_lower_case_cprefix (VALA_SYMBOL (self))), "free", NULL), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


static char* vala_class_real_get_free_function (ValaDataType* base) {
	ValaClass * self;
	const char* _tmp1;
	self = VALA_CLASS (base);
	if (self->priv->free_function == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->free_function = (_tmp0 = vala_class_get_default_free_function (self), (self->priv->free_function = (g_free (self->priv->free_function), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->free_function, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


void vala_class_set_free_function (ValaClass* self, const char* name) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (name != NULL);
	_tmp0 = NULL;
	self->priv->free_function = (_tmp0 = g_strdup (name), (self->priv->free_function = (g_free (self->priv->free_function), NULL)), _tmp0);
}


static gboolean vala_class_real_is_subtype_of (ValaDataType* base, ValaDataType* t) {
	ValaClass * self;
	self = VALA_CLASS (base);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (t), FALSE);
	if (VALA_DATA_TYPE (self) == t) {
		return TRUE;
	}
	{
		GeeList* base_type_collection;
		GeeIterator* base_type_it;
		base_type_collection = self->priv->base_types;
		base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
		while (gee_iterator_next (base_type_it)) {
			ValaTypeReference* base_type;
			base_type = gee_iterator_get (base_type_it);
			{
				if (vala_type_reference_get_data_type (base_type) != NULL && vala_data_type_is_subtype_of (vala_type_reference_get_data_type (base_type), t)) {
					gboolean _tmp1;
					return (_tmp1 = TRUE, (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), _tmp1);
				}
				(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
			}
		}
		(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
	}
	return FALSE;
}


static gint vala_class_real_get_type_parameter_index (ValaDataType* base, const char* name) {
	ValaClass * self;
	gint i;
	self = VALA_CLASS (base);
	g_return_val_if_fail (name != NULL, 0);
	i = 0;
	{
		GeeList* parameter_collection;
		GeeIterator* parameter_it;
		parameter_collection = self->priv->type_parameters;
		parameter_it = gee_iterable_iterator (GEE_ITERABLE (parameter_collection));
		while (gee_iterator_next (parameter_it)) {
			ValaTypeParameter* parameter;
			parameter = gee_iterator_get (parameter_it);
			{
				if (g_utf8_collate (vala_symbol_get_name (VALA_SYMBOL (parameter)), name) == 0) {
					gint _tmp0;
					return (_tmp0 = i, (parameter == NULL ? NULL : (parameter = (g_object_unref (parameter), NULL))), (parameter_it == NULL ? NULL : (parameter_it = (g_object_unref (parameter_it), NULL))), _tmp0);
				}
				i++;
				(parameter == NULL ? NULL : (parameter = (g_object_unref (parameter), NULL)));
			}
		}
		(parameter_it == NULL ? NULL : (parameter_it = (g_object_unref (parameter_it), NULL)));
	}
	return -1;
}


ValaClass* vala_class_get_base_class (ValaClass* self) {
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return self->priv->_base_class;
}


void vala_class_set_base_class (ValaClass* self, ValaClass* value) {
	ValaClass* _tmp2;
	ValaClass* _tmp1;
	g_return_if_fail (VALA_IS_CLASS (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_base_class = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_base_class == NULL ? NULL : (self->priv->_base_class = (g_object_unref (self->priv->_base_class), NULL))), _tmp2);
}


gboolean vala_class_get_is_abstract (ValaClass* self) {
	g_return_val_if_fail (VALA_IS_CLASS (self), FALSE);
	return self->priv->_is_abstract;
}


void vala_class_set_is_abstract (ValaClass* self, gboolean value) {
	g_return_if_fail (VALA_IS_CLASS (self));
	self->priv->_is_abstract = value;
}


gboolean vala_class_get_is_static (ValaClass* self) {
	g_return_val_if_fail (VALA_IS_CLASS (self), FALSE);
	return self->priv->_is_static;
}


void vala_class_set_is_static (ValaClass* self, gboolean value) {
	g_return_if_fail (VALA_IS_CLASS (self));
	self->priv->_is_static = value;
}


gboolean vala_class_get_has_private_fields (ValaClass* self) {
	g_return_val_if_fail (VALA_IS_CLASS (self), FALSE);
	return self->priv->_has_private_fields;
}


ValaMethod* vala_class_get_default_construction_method (ValaClass* self) {
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return self->priv->_default_construction_method;
}


void vala_class_set_default_construction_method (ValaClass* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (VALA_IS_CLASS (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_default_construction_method = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_default_construction_method == NULL ? NULL : (self->priv->_default_construction_method = (g_object_unref (self->priv->_default_construction_method), NULL))), _tmp2);
}


ValaConstructor* vala_class_get_constructor (ValaClass* self) {
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return self->priv->_constructor;
}


void vala_class_set_constructor (ValaClass* self, ValaConstructor* value) {
	ValaConstructor* _tmp2;
	ValaConstructor* _tmp1;
	g_return_if_fail (VALA_IS_CLASS (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_constructor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_constructor == NULL ? NULL : (self->priv->_constructor = (g_object_unref (self->priv->_constructor), NULL))), _tmp2);
}


ValaDestructor* vala_class_get_destructor (ValaClass* self) {
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return self->priv->_destructor;
}


void vala_class_set_destructor (ValaClass* self, ValaDestructor* value) {
	ValaDestructor* _tmp2;
	ValaDestructor* _tmp1;
	g_return_if_fail (VALA_IS_CLASS (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_destructor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_destructor == NULL ? NULL : (self->priv->_destructor = (g_object_unref (self->priv->_destructor), NULL))), _tmp2);
}


static void vala_class_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaClass * self;
	self = VALA_CLASS (object);
	switch (property_id) {
		case VALA_CLASS_BASE_CLASS:
		g_value_set_object (value, vala_class_get_base_class (self));
		break;
		case VALA_CLASS_IS_ABSTRACT:
		g_value_set_boolean (value, vala_class_get_is_abstract (self));
		break;
		case VALA_CLASS_IS_STATIC:
		g_value_set_boolean (value, vala_class_get_is_static (self));
		break;
		case VALA_CLASS_HAS_PRIVATE_FIELDS:
		g_value_set_boolean (value, vala_class_get_has_private_fields (self));
		break;
		case VALA_CLASS_DEFAULT_CONSTRUCTION_METHOD:
		g_value_set_object (value, vala_class_get_default_construction_method (self));
		break;
		case VALA_CLASS_CONSTRUCTOR:
		g_value_set_object (value, vala_class_get_constructor (self));
		break;
		case VALA_CLASS_DESTRUCTOR:
		g_value_set_object (value, vala_class_get_destructor (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_class_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaClass * self;
	self = VALA_CLASS (object);
	switch (property_id) {
		case VALA_CLASS_BASE_CLASS:
		vala_class_set_base_class (self, g_value_get_object (value));
		break;
		case VALA_CLASS_IS_ABSTRACT:
		vala_class_set_is_abstract (self, g_value_get_boolean (value));
		break;
		case VALA_CLASS_IS_STATIC:
		vala_class_set_is_static (self, g_value_get_boolean (value));
		break;
		case VALA_CLASS_DEFAULT_CONSTRUCTION_METHOD:
		vala_class_set_default_construction_method (self, g_value_get_object (value));
		break;
		case VALA_CLASS_CONSTRUCTOR:
		vala_class_set_constructor (self, g_value_get_object (value));
		break;
		case VALA_CLASS_DESTRUCTOR:
		vala_class_set_destructor (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_class_class_init (ValaClassClass * klass) {
	vala_class_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaClassPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_class_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_class_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_class_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_class_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_class_real_accept_children;
	VALA_SYMBOL_CLASS (klass)->get_cprefix = vala_class_real_get_cprefix;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_class_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_class_real_get_lower_case_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_class_real_get_lower_case_cprefix;
	VALA_DATA_TYPE_CLASS (klass)->get_upper_case_cname = vala_class_real_get_upper_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type = vala_class_real_is_reference_type;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_class_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->get_marshaller_type_name = vala_class_real_get_marshaller_type_name;
	VALA_DATA_TYPE_CLASS (klass)->get_get_value_function = vala_class_real_get_get_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_set_value_function = vala_class_real_get_set_value_function;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_counting = vala_class_real_is_reference_counting;
	VALA_DATA_TYPE_CLASS (klass)->get_ref_function = vala_class_real_get_ref_function;
	VALA_DATA_TYPE_CLASS (klass)->get_unref_function = vala_class_real_get_unref_function;
	VALA_DATA_TYPE_CLASS (klass)->get_dup_function = vala_class_real_get_dup_function;
	VALA_DATA_TYPE_CLASS (klass)->get_free_function = vala_class_real_get_free_function;
	VALA_DATA_TYPE_CLASS (klass)->is_subtype_of = vala_class_real_is_subtype_of;
	VALA_DATA_TYPE_CLASS (klass)->get_type_parameter_index = vala_class_real_get_type_parameter_index;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_BASE_CLASS, g_param_spec_object ("base-class", "base-class", "base-class", VALA_TYPE_CLASS, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_IS_ABSTRACT, g_param_spec_boolean ("is-abstract", "is-abstract", "is-abstract", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_IS_STATIC, g_param_spec_boolean ("is-static", "is-static", "is-static", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_HAS_PRIVATE_FIELDS, g_param_spec_boolean ("has-private-fields", "has-private-fields", "has-private-fields", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_DEFAULT_CONSTRUCTION_METHOD, g_param_spec_object ("default-construction-method", "default-construction-method", "default-construction-method", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_CONSTRUCTOR, g_param_spec_object ("constructor", "constructor", "constructor", VALA_TYPE_CONSTRUCTOR, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_DESTRUCTOR, g_param_spec_object ("destructor", "destructor", "destructor", VALA_TYPE_DESTRUCTOR, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_class_init (ValaClass * self) {
	self->priv = VALA_CLASS_GET_PRIVATE (self);
	self->priv->type_parameters = GEE_LIST (gee_array_list_new (VALA_TYPE_TYPE_PARAMETER, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->base_types = GEE_LIST (gee_array_list_new (VALA_TYPE_TYPE_REFERENCE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->constants = GEE_LIST (gee_array_list_new (VALA_TYPE_CONSTANT, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->fields = GEE_LIST (gee_array_list_new (VALA_TYPE_FIELD, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->methods = GEE_LIST (gee_array_list_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->properties = GEE_LIST (gee_array_list_new (VALA_TYPE_PROPERTY, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->signals = GEE_LIST (gee_array_list_new (VALA_TYPE_SIGNAL, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->classes = GEE_LIST (gee_array_list_new (VALA_TYPE_CLASS, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->structs = GEE_LIST (gee_array_list_new (VALA_TYPE_STRUCT, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
}


static void vala_class_dispose (GObject * obj) {
	ValaClass * self;
	self = VALA_CLASS (obj);
	(self->priv->_base_class == NULL ? NULL : (self->priv->_base_class = (g_object_unref (self->priv->_base_class), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->const_cname = (g_free (self->priv->const_cname), NULL));
	(self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL));
	(self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL));
	(self->priv->type_id = (g_free (self->priv->type_id), NULL));
	(self->priv->ref_function = (g_free (self->priv->ref_function), NULL));
	(self->priv->unref_function = (g_free (self->priv->unref_function), NULL));
	(self->priv->copy_function = (g_free (self->priv->copy_function), NULL));
	(self->priv->free_function = (g_free (self->priv->free_function), NULL));
	(self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL));
	(self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL));
	(self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL));
	(self->priv->type_parameters == NULL ? NULL : (self->priv->type_parameters = (g_object_unref (self->priv->type_parameters), NULL)));
	(self->priv->base_types == NULL ? NULL : (self->priv->base_types = (g_object_unref (self->priv->base_types), NULL)));
	(self->priv->constants == NULL ? NULL : (self->priv->constants = (g_object_unref (self->priv->constants), NULL)));
	(self->priv->fields == NULL ? NULL : (self->priv->fields = (g_object_unref (self->priv->fields), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (g_object_unref (self->priv->methods), NULL)));
	(self->priv->properties == NULL ? NULL : (self->priv->properties = (g_object_unref (self->priv->properties), NULL)));
	(self->priv->signals == NULL ? NULL : (self->priv->signals = (g_object_unref (self->priv->signals), NULL)));
	(self->priv->classes == NULL ? NULL : (self->priv->classes = (g_object_unref (self->priv->classes), NULL)));
	(self->priv->structs == NULL ? NULL : (self->priv->structs = (g_object_unref (self->priv->structs), NULL)));
	(self->priv->_default_construction_method == NULL ? NULL : (self->priv->_default_construction_method = (g_object_unref (self->priv->_default_construction_method), NULL)));
	(self->priv->_constructor == NULL ? NULL : (self->priv->_constructor = (g_object_unref (self->priv->_constructor), NULL)));
	(self->priv->_destructor == NULL ? NULL : (self->priv->_destructor = (g_object_unref (self->priv->_destructor), NULL)));
	G_OBJECT_CLASS (vala_class_parent_class)->dispose (obj);
}


GType vala_class_get_type (void) {
	static GType vala_class_type_id = 0;
	if (G_UNLIKELY (vala_class_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaClassClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_class_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaClass), 0, (GInstanceInitFunc) vala_class_init };
		vala_class_type_id = g_type_register_static (VALA_TYPE_DATA_TYPE, "ValaClass", &g_define_type_info, 0);
	}
	return vala_class_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}




