/* valasignal.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valasignal.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <vala/valaformalparameter.h>
#include <vala/valadelegate.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valareferencetype.h>
#include <vala/valaclass.h>
#include <vala/valaclasstype.h>
#include <vala/valainterfacetype.h>
#include <vala/valainterface.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>




struct _ValaSignalPrivate {
	gboolean _has_emitter;
	GeeList* parameters;
	ValaDelegate* generated_delegate;
	char* cname;
	gboolean lock_used;
	ValaDataType* _return_type;
};
#define VALA_SIGNAL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SIGNAL, ValaSignalPrivate))
enum  {
	VALA_SIGNAL_DUMMY_PROPERTY,
	VALA_SIGNAL_RETURN_TYPE,
	VALA_SIGNAL_HAS_EMITTER
};
static void vala_signal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_signal_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_signal_real_get_lock_used (ValaLockable* base);
static void vala_signal_real_set_lock_used (ValaLockable* base, gboolean used);
static void vala_signal_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gpointer vala_signal_parent_class = NULL;
static ValaLockableIface* vala_signal_vala_lockable_parent_iface = NULL;
static void vala_signal_dispose (GObject * obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new signal.
 *
 * @param name        signal name
 * @param return_type signal return type
 * @param source      reference to source code
 * @return            newly created signal
 */
ValaSignal* vala_signal_new (const char* name, ValaDataType* return_type, ValaSourceReference* source_reference) {
	ValaSignal * self;
	g_return_val_if_fail (return_type == NULL || VALA_IS_DATA_TYPE (return_type), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_SIGNAL, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_signal_set_return_type (self, return_type);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends parameter to signal handler.
 *
 * @param param a formal parameter
 */
void vala_signal_add_parameter (ValaSignal* self, ValaFormalParameter* param) {
	g_return_if_fail (VALA_IS_SIGNAL (self));
	g_return_if_fail (param == NULL || VALA_IS_FORMAL_PARAMETER (param));
	/* default C parameter position*/
	vala_formal_parameter_set_cparameter_position (param, ((double) gee_collection_get_size (GEE_COLLECTION (self->priv->parameters)) + 1));
	vala_formal_parameter_set_carray_length_parameter_position (param, vala_formal_parameter_get_cparameter_position (param) + 0.1);
	vala_formal_parameter_set_cdelegate_target_parameter_position (param, vala_formal_parameter_get_cparameter_position (param) + 0.1);
	gee_collection_add (GEE_COLLECTION (self->priv->parameters), param);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (param)), VALA_SYMBOL (param));
}


GeeList* vala_signal_get_parameters (ValaSignal* self) {
	g_return_val_if_fail (VALA_IS_SIGNAL (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_FORMAL_PARAMETER, ((GBoxedCopyFunc) g_object_ref), g_object_unref, self->priv->parameters));
}


/**
 * Returns generated delegate to be used for signal handlers.
 *
 * @return delegate
 */
ValaDelegate* vala_signal_get_delegate (ValaSignal* self) {
	ValaDelegate* _tmp3;
	g_return_val_if_fail (VALA_IS_SIGNAL (self), NULL);
	if (self->priv->generated_delegate == NULL) {
		ValaDelegate* _tmp0;
		ValaReferenceType* sender_type;
		ValaFormalParameter* sender_param;
		_tmp0 = NULL;
		self->priv->generated_delegate = (_tmp0 = vala_delegate_new (NULL, vala_signal_get_return_type (self), NULL), (self->priv->generated_delegate == NULL ? NULL : (self->priv->generated_delegate = (g_object_unref (self->priv->generated_delegate), NULL))), _tmp0);
		vala_delegate_set_instance (self->priv->generated_delegate, TRUE);
		sender_type = NULL;
		if (VALA_IS_CLASS (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))) {
			ValaReferenceType* _tmp1;
			_tmp1 = NULL;
			sender_type = (_tmp1 = VALA_REFERENCE_TYPE (vala_class_type_new (VALA_CLASS (vala_symbol_get_parent_symbol (VALA_SYMBOL (self))))), (sender_type == NULL ? NULL : (sender_type = (g_object_unref (sender_type), NULL))), _tmp1);
		} else {
			ValaReferenceType* _tmp2;
			_tmp2 = NULL;
			sender_type = (_tmp2 = VALA_REFERENCE_TYPE (vala_interface_type_new (VALA_INTERFACE (vala_symbol_get_parent_symbol (VALA_SYMBOL (self))))), (sender_type == NULL ? NULL : (sender_type = (g_object_unref (sender_type), NULL))), _tmp2);
		}
		sender_param = vala_formal_parameter_new ("sender", VALA_DATA_TYPE (sender_type), NULL);
		vala_delegate_add_parameter (self->priv->generated_delegate, sender_param);
		{
			GeeList* param_collection;
			GeeIterator* param_it;
			param_collection = self->priv->parameters;
			param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
			while (gee_iterator_next (param_it)) {
				ValaFormalParameter* param;
				param = ((ValaFormalParameter*) gee_iterator_get (param_it));
				{
					vala_delegate_add_parameter (self->priv->generated_delegate, param);
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
				}
			}
			(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
		}
		(sender_type == NULL ? NULL : (sender_type = (g_object_unref (sender_type), NULL)));
		(sender_param == NULL ? NULL : (sender_param = (g_object_unref (sender_param), NULL)));
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->generated_delegate, (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
}


/**
 * Returns the name of this signal as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_signal_get_cname (ValaSignal* self) {
	const char* _tmp2;
	g_return_val_if_fail (VALA_IS_SIGNAL (self), NULL);
	if (self->priv->cname == NULL) {
		char* _tmp1;
		const char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cname = (_tmp1 = (_tmp0 = vala_symbol_get_name (VALA_SYMBOL (self)), (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->cname, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


void vala_signal_set_cname (ValaSignal* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (VALA_IS_SIGNAL (self));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
}


/**
 * Returns the string literal of this signal to be used in C code.
 *
 * @return string literal to be used in C code
 */
ValaCCodeConstant* vala_signal_get_canonical_cconstant (ValaSignal* self) {
	GString* str;
	const char* _tmp0;
	char* i;
	ValaCCodeConstant* _tmp3;
	g_return_val_if_fail (VALA_IS_SIGNAL (self), NULL);
	str = g_string_new ("\"");
	_tmp0 = NULL;
	i = (_tmp0 = vala_symbol_get_name (VALA_SYMBOL (self)), (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* _tmp2;
		const char* _tmp1;
		c = g_utf8_get_char (i);
		if (c == '_') {
			g_string_append_c (str, '-');
		} else {
			g_string_append_unichar (str, c);
		}
		_tmp2 = NULL;
		_tmp1 = NULL;
		i = (_tmp2 = (_tmp1 = g_utf8_next_char (i), (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (i = (g_free (i), NULL)), _tmp2);
	}
	g_string_append_c (str, '"');
	_tmp3 = NULL;
	return (_tmp3 = vala_ccode_constant_new (str->str), (str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL))), (i = (g_free (i), NULL)), _tmp3);
}


static void vala_signal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaSignal * self;
	self = VALA_SIGNAL (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_member (visitor, VALA_MEMBER (self));
	vala_code_visitor_visit_signal (visitor, self);
}


static void vala_signal_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaSignal * self;
	self = VALA_SIGNAL (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_signal_get_return_type (self)), visitor);
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = ((ValaFormalParameter*) gee_iterator_get (param_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (param), visitor);
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
}


/**
 * Process all associated attributes.
 */
void vala_signal_process_attributes (ValaSignal* self) {
	g_return_if_fail (VALA_IS_SIGNAL (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = ((ValaAttribute*) a_it->data), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "HasEmitter") == 0) {
					vala_signal_set_has_emitter (self, TRUE);
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


static gboolean vala_signal_real_get_lock_used (ValaLockable* base) {
	ValaSignal * self;
	self = VALA_SIGNAL (base);
	return self->priv->lock_used;
}


static void vala_signal_real_set_lock_used (ValaLockable* base, gboolean used) {
	ValaSignal * self;
	self = VALA_SIGNAL (base);
	self->priv->lock_used = used;
}


static void vala_signal_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaSignal * self;
	self = VALA_SIGNAL (base);
	g_return_if_fail (old_type == NULL || VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (new_type == NULL || VALA_IS_DATA_TYPE (new_type));
	if (vala_signal_get_return_type (self) == old_type) {
		vala_signal_set_return_type (self, new_type);
	}
}


ValaDataType* vala_signal_get_return_type (ValaSignal* self) {
	g_return_val_if_fail (VALA_IS_SIGNAL (self), NULL);
	return self->priv->_return_type;
}


void vala_signal_set_return_type (ValaSignal* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_SIGNAL (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_return_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_return_type), VALA_CODE_NODE (self));
}


gboolean vala_signal_get_has_emitter (ValaSignal* self) {
	g_return_val_if_fail (VALA_IS_SIGNAL (self), FALSE);
	return self->priv->_has_emitter;
}


void vala_signal_set_has_emitter (ValaSignal* self, gboolean value) {
	g_return_if_fail (VALA_IS_SIGNAL (self));
	self->priv->_has_emitter = value;
}


static void vala_signal_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaSignal * self;
	self = VALA_SIGNAL (object);
	switch (property_id) {
		case VALA_SIGNAL_RETURN_TYPE:
		g_value_set_object (value, vala_signal_get_return_type (self));
		break;
		case VALA_SIGNAL_HAS_EMITTER:
		g_value_set_boolean (value, vala_signal_get_has_emitter (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_signal_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaSignal * self;
	self = VALA_SIGNAL (object);
	switch (property_id) {
		case VALA_SIGNAL_RETURN_TYPE:
		vala_signal_set_return_type (self, g_value_get_object (value));
		break;
		case VALA_SIGNAL_HAS_EMITTER:
		vala_signal_set_has_emitter (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_signal_class_init (ValaSignalClass * klass) {
	vala_signal_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSignalPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_signal_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_signal_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_signal_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_signal_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_signal_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_signal_real_replace_type;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SIGNAL_RETURN_TYPE, g_param_spec_object ("return-type", "return-type", "return-type", VALA_TYPE_DATA_TYPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SIGNAL_HAS_EMITTER, g_param_spec_boolean ("has-emitter", "has-emitter", "has-emitter", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_signal_vala_lockable_interface_init (ValaLockableIface * iface) {
	vala_signal_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_signal_real_get_lock_used;
	iface->set_lock_used = vala_signal_real_set_lock_used;
}


static void vala_signal_init (ValaSignal * self) {
	self->priv = VALA_SIGNAL_GET_PRIVATE (self);
	self->priv->parameters = GEE_LIST (gee_array_list_new (VALA_TYPE_FORMAL_PARAMETER, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->lock_used = FALSE;
}


static void vala_signal_dispose (GObject * obj) {
	ValaSignal * self;
	self = VALA_SIGNAL (obj);
	(self->priv->parameters == NULL ? NULL : (self->priv->parameters = (g_object_unref (self->priv->parameters), NULL)));
	(self->priv->generated_delegate == NULL ? NULL : (self->priv->generated_delegate = (g_object_unref (self->priv->generated_delegate), NULL)));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	(self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL)));
	G_OBJECT_CLASS (vala_signal_parent_class)->dispose (obj);
}


GType vala_signal_get_type (void) {
	static GType vala_signal_type_id = 0;
	if (G_UNLIKELY (vala_signal_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSignalClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_signal_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSignal), 0, (GInstanceInitFunc) vala_signal_init };
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_signal_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_signal_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaSignal", &g_define_type_info, 0);
		g_type_add_interface_static (vala_signal_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_signal_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




