/* valamethod.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valamethod.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valabasicblock.h>
#include <vala/valaformalparameter.h>
#include <vala/valalocalvariable.h>
#include <vala/valadatatype.h>
#include <vala/valaexpression.h>
#include <vala/valablock.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>
#include <vala/valacodegenerator.h>




struct _ValaMethodPrivate {
	ValaBasicBlock* _entry_block;
	ValaBasicBlock* _exit_block;
	MemberBinding _binding;
	gboolean _is_abstract;
	gboolean _is_virtual;
	gboolean _overrides;
	gboolean _is_inline;
	gboolean _returns_modified_pointer;
	ValaMethod* _base_method;
	ValaMethod* _base_interface_method;
	ValaFormalParameter* _this_parameter;
	ValaLocalVariable* _result_var;
	double _cinstance_parameter_position;
	double _carray_length_parameter_position;
	double _cdelegate_target_parameter_position;
	gboolean _printf_format;
	GeeList* parameters;
	char* cname;
	char* _vfunc_name;
	char* _sentinel;
	gboolean _no_array_length;
	GeeList* error_domains;
	GeeList* preconditions;
	GeeList* postconditions;
	ValaDataType* _return_type;
	ValaBlock* _body;
};

#define VALA_METHOD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_METHOD, ValaMethodPrivate))
enum  {
	VALA_METHOD_DUMMY_PROPERTY,
	VALA_METHOD_RETURN_TYPE,
	VALA_METHOD_BODY,
	VALA_METHOD_ENTRY_BLOCK,
	VALA_METHOD_EXIT_BLOCK,
	VALA_METHOD_BINDING,
	VALA_METHOD_VFUNC_NAME,
	VALA_METHOD_SENTINEL,
	VALA_METHOD_IS_ABSTRACT,
	VALA_METHOD_IS_VIRTUAL,
	VALA_METHOD_OVERRIDES,
	VALA_METHOD_IS_INLINE,
	VALA_METHOD_RETURNS_MODIFIED_POINTER,
	VALA_METHOD_BASE_METHOD,
	VALA_METHOD_BASE_INTERFACE_METHOD,
	VALA_METHOD_THIS_PARAMETER,
	VALA_METHOD_RESULT_VAR,
	VALA_METHOD_CINSTANCE_PARAMETER_POSITION,
	VALA_METHOD_CARRAY_LENGTH_PARAMETER_POSITION,
	VALA_METHOD_CDELEGATE_TARGET_PARAMETER_POSITION,
	VALA_METHOD_NO_ARRAY_LENGTH,
	VALA_METHOD_PRINTF_FORMAT
};
static void vala_method_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_method_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_method_real_get_default_cname (ValaMethod* self);
static void vala_method_process_ccode_attribute (ValaMethod* self, ValaAttribute* a);
static void vala_method_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static ValaCodeBinding* vala_method_real_create_code_binding (ValaCodeNode* base, ValaCodeGenerator* codegen);
static GObject * vala_method_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer vala_method_parent_class = NULL;
static void vala_method_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new method.
 *
 * @param name        method name
 * @param return_type method return type
 * @param source      reference to source code
 * @return            newly created method
 */
ValaMethod* vala_method_new (const char* name, ValaDataType* return_type, ValaSourceReference* source_reference) {
	ValaMethod * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (return_type), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_METHOD, 0, NULL);
	vala_method_set_return_type (self, return_type);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	return self;
}


/**
 * Appends parameter to this method.
 *
 * @param param a formal parameter
 */
void vala_method_add_parameter (ValaMethod* self, ValaFormalParameter* param) {
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (param));
	if (vala_method_get_no_array_length (self)) {
		vala_formal_parameter_set_no_array_length (param, TRUE);
	}
	/* default C parameter position*/
	vala_formal_parameter_set_cparameter_position (param, ((double) gee_collection_get_size (GEE_COLLECTION (self->priv->parameters)) + 1));
	vala_formal_parameter_set_carray_length_parameter_position (param, vala_formal_parameter_get_cparameter_position (param) + 0.1);
	vala_formal_parameter_set_cdelegate_target_parameter_position (param, vala_formal_parameter_get_cparameter_position (param) + 0.1);
	gee_collection_add (GEE_COLLECTION (self->priv->parameters), param);
	if (!vala_formal_parameter_get_ellipsis (param)) {
		vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (param)), VALA_SYMBOL (param));
	}
}


GeeCollection* vala_method_get_parameters (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_FORMAL_PARAMETER, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->parameters)));
}


/**
 * Remove all parameters from this method.
 */
void vala_method_clear_parameters (ValaMethod* self) {
	g_return_if_fail (VALA_IS_METHOD (self));
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = ((ValaFormalParameter*) gee_iterator_get (param_it));
			{
				if (!vala_formal_parameter_get_ellipsis (param)) {
					vala_scope_remove (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (param)));
				}
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	gee_collection_clear (GEE_COLLECTION (self->priv->parameters));
}


static void vala_method_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMethod * self;
	self = VALA_METHOD (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_method (visitor, self);
}


static void vala_method_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMethod * self;
	self = VALA_METHOD (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (vala_method_get_return_type (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_method_get_return_type (self)), visitor);
	}
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = ((ValaFormalParameter*) gee_iterator_get (param_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (param), visitor);
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	{
		GeeList* error_domain_collection;
		GeeIterator* error_domain_it;
		error_domain_collection = self->priv->error_domains;
		error_domain_it = gee_iterable_iterator (GEE_ITERABLE (error_domain_collection));
		while (gee_iterator_next (error_domain_it)) {
			ValaDataType* error_domain;
			error_domain = ((ValaDataType*) gee_iterator_get (error_domain_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (error_domain), visitor);
				(error_domain == NULL ? NULL : (error_domain = (g_object_unref (error_domain), NULL)));
			}
		}
		(error_domain_it == NULL ? NULL : (error_domain_it = (g_object_unref (error_domain_it), NULL)));
	}
	if (self->priv->_result_var != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_local_variable_get_variable_type (self->priv->_result_var)), visitor);
	}
	{
		GeeList* precondition_collection;
		GeeIterator* precondition_it;
		precondition_collection = self->priv->preconditions;
		precondition_it = gee_iterable_iterator (GEE_ITERABLE (precondition_collection));
		while (gee_iterator_next (precondition_it)) {
			ValaExpression* precondition;
			precondition = ((ValaExpression*) gee_iterator_get (precondition_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (precondition), visitor);
				(precondition == NULL ? NULL : (precondition = (g_object_unref (precondition), NULL)));
			}
		}
		(precondition_it == NULL ? NULL : (precondition_it = (g_object_unref (precondition_it), NULL)));
	}
	{
		GeeList* postcondition_collection;
		GeeIterator* postcondition_it;
		postcondition_collection = self->priv->postconditions;
		postcondition_it = gee_iterable_iterator (GEE_ITERABLE (postcondition_collection));
		while (gee_iterator_next (postcondition_it)) {
			ValaExpression* postcondition;
			postcondition = ((ValaExpression*) gee_iterator_get (postcondition_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (postcondition), visitor);
				(postcondition == NULL ? NULL : (postcondition = (g_object_unref (postcondition), NULL)));
			}
		}
		(postcondition_it == NULL ? NULL : (postcondition_it = (g_object_unref (postcondition_it), NULL)));
	}
	if (vala_method_get_body (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_method_get_body (self)), visitor);
	}
}


/**
 * Returns the interface name of this method as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_method_get_cname (ValaMethod* self) {
	const char* _tmp1;
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (self->priv->cname == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->cname = (_tmp0 = vala_method_get_default_cname (self), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->cname, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


/**
 * Returns the default interface name of this method as it is used in C
 * code.
 *
 * @return the name to be used in C code by default
 */
static char* vala_method_real_get_default_cname (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (_vala_strcmp0 (vala_symbol_get_name (VALA_SYMBOL (self)), "main") == 0 && vala_symbol_get_name (vala_symbol_get_parent_symbol (VALA_SYMBOL (self))) == NULL) {
		/* avoid conflict with generated main function*/
		return g_strdup ("_main");
	} else {
		if (g_str_has_prefix (vala_symbol_get_name (VALA_SYMBOL (self)), "_")) {
			char* _tmp1;
			char* _tmp2;
			_tmp1 = NULL;
			_tmp2 = NULL;
			return (_tmp2 = g_strdup_printf ("_%s%s", (_tmp1 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), g_utf8_offset_to_pointer (vala_symbol_get_name (VALA_SYMBOL (self)), ((glong) 1))), (_tmp1 = (g_free (_tmp1), NULL)), _tmp2);
		} else {
			char* _tmp4;
			char* _tmp5;
			_tmp4 = NULL;
			_tmp5 = NULL;
			return (_tmp5 = g_strdup_printf ("%s%s", (_tmp4 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), vala_symbol_get_name (VALA_SYMBOL (self))), (_tmp4 = (g_free (_tmp4), NULL)), _tmp5);
		}
	}
}


char* vala_method_get_default_cname (ValaMethod* self) {
	return VALA_METHOD_GET_CLASS (self)->get_default_cname (self);
}


/**
 * Returns the implementation name of this data type as it is used in C
 * code.
 *
 * @return the name to be used in C code
 */
char* vala_method_get_real_cname (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (self->priv->_base_method != NULL || self->priv->_base_interface_method != NULL) {
		char* _tmp0;
		char* _tmp1;
		_tmp0 = NULL;
		_tmp1 = NULL;
		return (_tmp1 = g_strdup_printf ("%s_real_%s", (_tmp0 = vala_symbol_get_lower_case_cname (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)), NULL)), vala_symbol_get_name (VALA_SYMBOL (self))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	} else {
		return vala_method_get_cname (self);
	}
}


/**
 * Sets the name of this method as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_method_set_cname (ValaMethod* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (cname != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
}


static void vala_method_process_ccode_attribute (ValaMethod* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_method_set_cname (self, (_tmp0 = vala_attribute_get_string (a, "cname")));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			char** filename_it;
			int filename_collection_length1;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp1;
				char* filename;
				_tmp1 = NULL;
				filename = (_tmp1 = *filename_it, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
				{
					vala_member_add_cheader_filename (VALA_MEMBER (self), filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, ((GDestroyNotify) g_free)), NULL);
		}
		val = (g_free (val), NULL);
	}
	if (vala_attribute_has_argument (a, "vfunc_name")) {
		char* _tmp2;
		_tmp2 = NULL;
		vala_method_set_vfunc_name (self, (_tmp2 = vala_attribute_get_string (a, "vfunc_name")));
		_tmp2 = (g_free (_tmp2), NULL);
	}
	if (vala_attribute_has_argument (a, "sentinel")) {
		char* _tmp3;
		_tmp3 = NULL;
		vala_method_set_sentinel (self, (_tmp3 = vala_attribute_get_string (a, "sentinel")));
		_tmp3 = (g_free (_tmp3), NULL);
	}
	if (vala_attribute_has_argument (a, "instance_pos")) {
		vala_method_set_cinstance_parameter_position (self, vala_attribute_get_double (a, "instance_pos"));
	}
	if (vala_attribute_has_argument (a, "array_length_pos")) {
		vala_method_set_carray_length_parameter_position (self, vala_attribute_get_double (a, "array_length_pos"));
	}
	if (vala_attribute_has_argument (a, "delegate_target_pos")) {
		vala_method_set_cdelegate_target_parameter_position (self, vala_attribute_get_double (a, "delegate_target_pos"));
	}
}


/**
 * Process all associated attributes.
 */
void vala_method_process_attributes (ValaMethod* self) {
	g_return_if_fail (VALA_IS_METHOD (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = ((ValaAttribute*) a_it->data), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_method_process_ccode_attribute (self, a);
				} else {
					if (_vala_strcmp0 (vala_attribute_get_name (a), "ReturnsModifiedPointer") == 0) {
						vala_method_set_returns_modified_pointer (self, TRUE);
					} else {
						if (_vala_strcmp0 (vala_attribute_get_name (a), "FloatingReference") == 0) {
							vala_data_type_set_floating_reference (vala_method_get_return_type (self), TRUE);
						} else {
							if (_vala_strcmp0 (vala_attribute_get_name (a), "NoArrayLength") == 0) {
								vala_method_set_no_array_length (self, TRUE);
							} else {
								if (_vala_strcmp0 (vala_attribute_get_name (a), "PrintfFormat") == 0) {
									vala_method_set_printf_format (self, TRUE);
								}
							}
						}
					}
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


/**
 * Checks whether the parameters and return type of this method are
 * compatible with the specified method
 *
 * @param base_method a method
 * @param invalid_match error string about which check failed
 * @return true if the specified method is compatible to this method
 */
gboolean vala_method_compatible (ValaMethod* self, ValaMethod* base_method, char** invalid_match) {
	GeeIterator* method_params_it;
	gint param_index;
	gboolean _tmp13;
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	g_return_val_if_fail (VALA_IS_METHOD (base_method), FALSE);
	*invalid_match = NULL;
	if (!vala_data_type_equals (vala_method_get_return_type (self), vala_method_get_return_type (base_method))) {
		char* _tmp0;
		_tmp0 = NULL;
		(*invalid_match) = (_tmp0 = g_strdup ("incompatible return type"), ((*invalid_match) = (g_free ((*invalid_match)), NULL)), _tmp0);
		return FALSE;
	}
	method_params_it = gee_iterable_iterator (GEE_ITERABLE (self->priv->parameters));
	param_index = 1;
	{
		GeeList* base_param_collection;
		GeeIterator* base_param_it;
		base_param_collection = base_method->priv->parameters;
		base_param_it = gee_iterable_iterator (GEE_ITERABLE (base_param_collection));
		while (gee_iterator_next (base_param_it)) {
			ValaFormalParameter* base_param;
			base_param = ((ValaFormalParameter*) gee_iterator_get (base_param_it));
			{
				ValaFormalParameter* _tmp4;
				gboolean _tmp5;
				/* this method may not expect less arguments */
				if (!gee_iterator_next (method_params_it)) {
					char* _tmp2;
					gboolean _tmp3;
					_tmp2 = NULL;
					(*invalid_match) = (_tmp2 = g_strdup ("too few parameters"), ((*invalid_match) = (g_free ((*invalid_match)), NULL)), _tmp2);
					return (_tmp3 = FALSE, (base_param == NULL ? NULL : (base_param = (g_object_unref (base_param), NULL))), (base_param_it == NULL ? NULL : (base_param_it = (g_object_unref (base_param_it), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), _tmp3);
				}
				_tmp4 = NULL;
				if ((_tmp5 = !vala_data_type_equals (vala_formal_parameter_get_type_reference (base_param), vala_formal_parameter_get_type_reference ((_tmp4 = ((ValaFormalParameter*) gee_iterator_get (method_params_it))))), (_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL))), _tmp5)) {
					char* _tmp6;
					gboolean _tmp7;
					_tmp6 = NULL;
					(*invalid_match) = (_tmp6 = g_strdup_printf ("incompatible type of parameter %d", param_index), ((*invalid_match) = (g_free ((*invalid_match)), NULL)), _tmp6);
					return (_tmp7 = FALSE, (base_param == NULL ? NULL : (base_param = (g_object_unref (base_param), NULL))), (base_param_it == NULL ? NULL : (base_param_it = (g_object_unref (base_param_it), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), _tmp7);
				}
				param_index++;
				(base_param == NULL ? NULL : (base_param = (g_object_unref (base_param), NULL)));
			}
		}
		(base_param_it == NULL ? NULL : (base_param_it = (g_object_unref (base_param_it), NULL)));
	}
	/* this method may not expect more arguments */
	if (gee_iterator_next (method_params_it)) {
		char* _tmp8;
		gboolean _tmp9;
		_tmp8 = NULL;
		(*invalid_match) = (_tmp8 = g_strdup ("too many parameters"), ((*invalid_match) = (g_free ((*invalid_match)), NULL)), _tmp8);
		return (_tmp9 = FALSE, (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), _tmp9);
	}
	/* this method may throw more but not less errors than the base method */
	{
		GeeList* method_error_domain_collection;
		GeeIterator* method_error_domain_it;
		method_error_domain_collection = self->priv->error_domains;
		method_error_domain_it = gee_iterable_iterator (GEE_ITERABLE (method_error_domain_collection));
		while (gee_iterator_next (method_error_domain_it)) {
			ValaDataType* method_error_domain;
			method_error_domain = ((ValaDataType*) gee_iterator_get (method_error_domain_it));
			{
				gboolean match;
				match = FALSE;
				{
					GeeList* base_method_error_domain_collection;
					GeeIterator* base_method_error_domain_it;
					base_method_error_domain_collection = base_method->priv->error_domains;
					base_method_error_domain_it = gee_iterable_iterator (GEE_ITERABLE (base_method_error_domain_collection));
					while (gee_iterator_next (base_method_error_domain_it)) {
						ValaDataType* base_method_error_domain;
						base_method_error_domain = ((ValaDataType*) gee_iterator_get (base_method_error_domain_it));
						{
							if (vala_data_type_compatible (method_error_domain, base_method_error_domain)) {
								match = TRUE;
								(base_method_error_domain == NULL ? NULL : (base_method_error_domain = (g_object_unref (base_method_error_domain), NULL)));
								break;
							}
							(base_method_error_domain == NULL ? NULL : (base_method_error_domain = (g_object_unref (base_method_error_domain), NULL)));
						}
					}
					(base_method_error_domain_it == NULL ? NULL : (base_method_error_domain_it = (g_object_unref (base_method_error_domain_it), NULL)));
				}
				if (!match) {
					char* _tmp11;
					char* _tmp10;
					gboolean _tmp12;
					_tmp11 = NULL;
					_tmp10 = NULL;
					(*invalid_match) = (_tmp11 = g_strdup_printf ("incompatible error domain `%s'", (_tmp10 = vala_code_node_to_string (VALA_CODE_NODE (method_error_domain)))), ((*invalid_match) = (g_free ((*invalid_match)), NULL)), _tmp11);
					_tmp10 = (g_free (_tmp10), NULL);
					return (_tmp12 = FALSE, (method_error_domain == NULL ? NULL : (method_error_domain = (g_object_unref (method_error_domain), NULL))), (method_error_domain_it == NULL ? NULL : (method_error_domain_it = (g_object_unref (method_error_domain_it), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), _tmp12);
				}
				(method_error_domain == NULL ? NULL : (method_error_domain = (g_object_unref (method_error_domain), NULL)));
			}
		}
		(method_error_domain_it == NULL ? NULL : (method_error_domain_it = (g_object_unref (method_error_domain_it), NULL)));
	}
	return (_tmp13 = TRUE, (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), _tmp13);
}


/**
 * Adds an error domain to this method.
 *
 * @param error_domain an error domain
 */
void vala_method_add_error_domain (ValaMethod* self, ValaDataType* error_domain) {
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_DATA_TYPE (error_domain));
	gee_collection_add (GEE_COLLECTION (self->priv->error_domains), error_domain);
	vala_code_node_set_parent_node (VALA_CODE_NODE (error_domain), VALA_CODE_NODE (self));
}


/**
 * Returns a copy of the list of error domains of this method.
 *
 * @return list of error domains
 */
GeeCollection* vala_method_get_error_domains (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_DATA_TYPE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->error_domains)));
}


/**
 * Adds a precondition to this method.
 *
 * @param precondition a boolean precondition expression
 */
void vala_method_add_precondition (ValaMethod* self, ValaExpression* precondition) {
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_EXPRESSION (precondition));
	gee_collection_add (GEE_COLLECTION (self->priv->preconditions), precondition);
	vala_code_node_set_parent_node (VALA_CODE_NODE (precondition), VALA_CODE_NODE (self));
}


/**
 * Returns a copy of the list of preconditions of this method.
 *
 * @return list of preconditions
 */
GeeCollection* vala_method_get_preconditions (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->preconditions)));
}


/**
 * Adds a postcondition to this method.
 *
 * @param postcondition a boolean postcondition expression
 */
void vala_method_add_postcondition (ValaMethod* self, ValaExpression* postcondition) {
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_EXPRESSION (postcondition));
	gee_collection_add (GEE_COLLECTION (self->priv->postconditions), postcondition);
	vala_code_node_set_parent_node (VALA_CODE_NODE (postcondition), VALA_CODE_NODE (self));
}


/**
 * Returns a copy of the list of postconditions of this method.
 *
 * @return list of postconditions
 */
GeeCollection* vala_method_get_postconditions (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->postconditions)));
}


static void vala_method_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaMethod * self;
	self = VALA_METHOD (base);
	g_return_if_fail (VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (VALA_IS_DATA_TYPE (new_type));
	if (vala_method_get_return_type (self) == old_type) {
		vala_method_set_return_type (self, new_type);
		return;
	}
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size (GEE_COLLECTION (self->priv->error_domains)); i++) {
			ValaDataType* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = ((ValaDataType*) gee_list_get (((GeeList*) self->priv->error_domains), i))) == old_type, (_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL))), _tmp1)) {
				gee_list_set (((GeeList*) self->priv->error_domains), i, new_type);
				return;
			}
		}
	}
}


static ValaCodeBinding* vala_method_real_create_code_binding (ValaCodeNode* base, ValaCodeGenerator* codegen) {
	ValaMethod * self;
	self = VALA_METHOD (base);
	g_return_val_if_fail (VALA_IS_CODE_GENERATOR (codegen), NULL);
	return vala_code_generator_create_method_binding (codegen, self);
}


ValaDataType* vala_method_get_return_type (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_return_type;
}


void vala_method_set_return_type (ValaMethod* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_return_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_return_type), VALA_CODE_NODE (self));
}


ValaBlock* vala_method_get_body (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_body;
}


void vala_method_set_body (ValaMethod* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_body == NULL ? NULL : (self->priv->_body = (g_object_unref (self->priv->_body), NULL))), _tmp2);
	if (self->priv->_body != NULL) {
		vala_symbol_set_owner (VALA_SYMBOL (self->priv->_body), vala_symbol_get_scope (VALA_SYMBOL (self)));
	}
}


ValaBasicBlock* vala_method_get_entry_block (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_entry_block;
}


void vala_method_set_entry_block (ValaMethod* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_entry_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_entry_block == NULL ? NULL : (self->priv->_entry_block = (g_object_unref (self->priv->_entry_block), NULL))), _tmp2);
}


ValaBasicBlock* vala_method_get_exit_block (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_exit_block;
}


void vala_method_set_exit_block (ValaMethod* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_exit_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_exit_block == NULL ? NULL : (self->priv->_exit_block = (g_object_unref (self->priv->_exit_block), NULL))), _tmp2);
}


MemberBinding vala_method_get_binding (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), 0);
	return self->priv->_binding;
}


void vala_method_set_binding (ValaMethod* self, MemberBinding value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_binding = value;
}


char* vala_method_get_vfunc_name (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (self->priv->_vfunc_name == NULL) {
		char* _tmp1;
		const char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->_vfunc_name = (_tmp1 = (_tmp0 = vala_symbol_get_name (VALA_SYMBOL (self)), (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->_vfunc_name = (g_free (self->priv->_vfunc_name), NULL)), _tmp1);
	}
	return self->priv->_vfunc_name;
}


void vala_method_set_vfunc_name (ValaMethod* self, const char* value) {
	char* _tmp4;
	const char* _tmp3;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->_vfunc_name = (_tmp4 = (_tmp3 = value, (_tmp3 == NULL ? NULL : g_strdup (_tmp3))), (self->priv->_vfunc_name = (g_free (self->priv->_vfunc_name), NULL)), _tmp4);
}


char* vala_method_get_sentinel (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (self->priv->_sentinel == NULL) {
		return VALA_METHOD_DEFAULT_SENTINEL;
	}
	return self->priv->_sentinel;
}


void vala_method_set_sentinel (ValaMethod* self, const char* value) {
	char* _tmp3;
	const char* _tmp2;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->_sentinel = (_tmp3 = (_tmp2 = value, (_tmp2 == NULL ? NULL : g_strdup (_tmp2))), (self->priv->_sentinel = (g_free (self->priv->_sentinel), NULL)), _tmp3);
}


gboolean vala_method_get_is_abstract (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_is_abstract;
}


void vala_method_set_is_abstract (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_is_abstract = value;
}


gboolean vala_method_get_is_virtual (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_is_virtual;
}


void vala_method_set_is_virtual (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_is_virtual = value;
}


gboolean vala_method_get_overrides (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_overrides;
}


void vala_method_set_overrides (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_overrides = value;
}


gboolean vala_method_get_is_inline (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_is_inline;
}


void vala_method_set_is_inline (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_is_inline = value;
}


gboolean vala_method_get_returns_modified_pointer (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_returns_modified_pointer;
}


void vala_method_set_returns_modified_pointer (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_returns_modified_pointer = value;
}


ValaMethod* vala_method_get_base_method (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_base_method;
}


void vala_method_set_base_method (ValaMethod* self, ValaMethod* value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_base_method = value;
}


ValaMethod* vala_method_get_base_interface_method (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_base_interface_method;
}


void vala_method_set_base_interface_method (ValaMethod* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_base_interface_method = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_base_interface_method == NULL ? NULL : (self->priv->_base_interface_method = (g_object_unref (self->priv->_base_interface_method), NULL))), _tmp2);
}


ValaFormalParameter* vala_method_get_this_parameter (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_this_parameter;
}


void vala_method_set_this_parameter (ValaMethod* self, ValaFormalParameter* value) {
	ValaFormalParameter* _tmp2;
	ValaFormalParameter* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_this_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (g_object_unref (self->priv->_this_parameter), NULL))), _tmp2);
}


ValaLocalVariable* vala_method_get_result_var (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_result_var;
}


void vala_method_set_result_var (ValaMethod* self, ValaLocalVariable* value) {
	ValaLocalVariable* _tmp2;
	ValaLocalVariable* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_result_var = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_result_var == NULL ? NULL : (self->priv->_result_var = (g_object_unref (self->priv->_result_var), NULL))), _tmp2);
}


double vala_method_get_cinstance_parameter_position (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), 0.0);
	return self->priv->_cinstance_parameter_position;
}


void vala_method_set_cinstance_parameter_position (ValaMethod* self, double value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_cinstance_parameter_position = value;
}


double vala_method_get_carray_length_parameter_position (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), 0.0);
	return self->priv->_carray_length_parameter_position;
}


void vala_method_set_carray_length_parameter_position (ValaMethod* self, double value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_carray_length_parameter_position = value;
}


double vala_method_get_cdelegate_target_parameter_position (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), 0.0);
	return self->priv->_cdelegate_target_parameter_position;
}


void vala_method_set_cdelegate_target_parameter_position (ValaMethod* self, double value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_cdelegate_target_parameter_position = value;
}


gboolean vala_method_get_no_array_length (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_no_array_length;
}


void vala_method_set_no_array_length (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_no_array_length = value;
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = ((ValaFormalParameter*) gee_iterator_get (param_it));
			{
				vala_formal_parameter_set_no_array_length (param, value);
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
}


gboolean vala_method_get_printf_format (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_printf_format;
}


void vala_method_set_printf_format (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_printf_format = value;
}


static GObject * vala_method_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	ValaMethodClass * klass;
	GObjectClass * parent_class;
	ValaMethod * self;
	klass = VALA_METHOD_CLASS (g_type_class_peek (VALA_TYPE_METHOD));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = VALA_METHOD (obj);
	{
		vala_method_set_carray_length_parameter_position (self, ((double) -3));
		vala_method_set_cdelegate_target_parameter_position (self, ((double) -3));
	}
	return obj;
}


static void vala_method_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaMethod * self;
	self = VALA_METHOD (object);
	switch (property_id) {
		case VALA_METHOD_RETURN_TYPE:
		g_value_set_object (value, vala_method_get_return_type (self));
		break;
		case VALA_METHOD_BODY:
		g_value_set_object (value, vala_method_get_body (self));
		break;
		case VALA_METHOD_ENTRY_BLOCK:
		g_value_set_object (value, vala_method_get_entry_block (self));
		break;
		case VALA_METHOD_EXIT_BLOCK:
		g_value_set_object (value, vala_method_get_exit_block (self));
		break;
		case VALA_METHOD_BINDING:
		g_value_set_enum (value, vala_method_get_binding (self));
		break;
		case VALA_METHOD_VFUNC_NAME:
		g_value_set_string (value, vala_method_get_vfunc_name (self));
		break;
		case VALA_METHOD_SENTINEL:
		g_value_set_string (value, vala_method_get_sentinel (self));
		break;
		case VALA_METHOD_IS_ABSTRACT:
		g_value_set_boolean (value, vala_method_get_is_abstract (self));
		break;
		case VALA_METHOD_IS_VIRTUAL:
		g_value_set_boolean (value, vala_method_get_is_virtual (self));
		break;
		case VALA_METHOD_OVERRIDES:
		g_value_set_boolean (value, vala_method_get_overrides (self));
		break;
		case VALA_METHOD_IS_INLINE:
		g_value_set_boolean (value, vala_method_get_is_inline (self));
		break;
		case VALA_METHOD_RETURNS_MODIFIED_POINTER:
		g_value_set_boolean (value, vala_method_get_returns_modified_pointer (self));
		break;
		case VALA_METHOD_BASE_METHOD:
		g_value_set_object (value, vala_method_get_base_method (self));
		break;
		case VALA_METHOD_BASE_INTERFACE_METHOD:
		g_value_set_object (value, vala_method_get_base_interface_method (self));
		break;
		case VALA_METHOD_THIS_PARAMETER:
		g_value_set_object (value, vala_method_get_this_parameter (self));
		break;
		case VALA_METHOD_RESULT_VAR:
		g_value_set_object (value, vala_method_get_result_var (self));
		break;
		case VALA_METHOD_CINSTANCE_PARAMETER_POSITION:
		g_value_set_double (value, vala_method_get_cinstance_parameter_position (self));
		break;
		case VALA_METHOD_CARRAY_LENGTH_PARAMETER_POSITION:
		g_value_set_double (value, vala_method_get_carray_length_parameter_position (self));
		break;
		case VALA_METHOD_CDELEGATE_TARGET_PARAMETER_POSITION:
		g_value_set_double (value, vala_method_get_cdelegate_target_parameter_position (self));
		break;
		case VALA_METHOD_NO_ARRAY_LENGTH:
		g_value_set_boolean (value, vala_method_get_no_array_length (self));
		break;
		case VALA_METHOD_PRINTF_FORMAT:
		g_value_set_boolean (value, vala_method_get_printf_format (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_method_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaMethod * self;
	self = VALA_METHOD (object);
	switch (property_id) {
		case VALA_METHOD_RETURN_TYPE:
		vala_method_set_return_type (self, g_value_get_object (value));
		break;
		case VALA_METHOD_BODY:
		vala_method_set_body (self, g_value_get_object (value));
		break;
		case VALA_METHOD_ENTRY_BLOCK:
		vala_method_set_entry_block (self, g_value_get_object (value));
		break;
		case VALA_METHOD_EXIT_BLOCK:
		vala_method_set_exit_block (self, g_value_get_object (value));
		break;
		case VALA_METHOD_BINDING:
		vala_method_set_binding (self, g_value_get_enum (value));
		break;
		case VALA_METHOD_VFUNC_NAME:
		vala_method_set_vfunc_name (self, g_value_get_string (value));
		break;
		case VALA_METHOD_SENTINEL:
		vala_method_set_sentinel (self, g_value_get_string (value));
		break;
		case VALA_METHOD_IS_ABSTRACT:
		vala_method_set_is_abstract (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_IS_VIRTUAL:
		vala_method_set_is_virtual (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_OVERRIDES:
		vala_method_set_overrides (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_IS_INLINE:
		vala_method_set_is_inline (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_RETURNS_MODIFIED_POINTER:
		vala_method_set_returns_modified_pointer (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_BASE_METHOD:
		vala_method_set_base_method (self, g_value_get_object (value));
		break;
		case VALA_METHOD_BASE_INTERFACE_METHOD:
		vala_method_set_base_interface_method (self, g_value_get_object (value));
		break;
		case VALA_METHOD_THIS_PARAMETER:
		vala_method_set_this_parameter (self, g_value_get_object (value));
		break;
		case VALA_METHOD_RESULT_VAR:
		vala_method_set_result_var (self, g_value_get_object (value));
		break;
		case VALA_METHOD_CINSTANCE_PARAMETER_POSITION:
		vala_method_set_cinstance_parameter_position (self, g_value_get_double (value));
		break;
		case VALA_METHOD_CARRAY_LENGTH_PARAMETER_POSITION:
		vala_method_set_carray_length_parameter_position (self, g_value_get_double (value));
		break;
		case VALA_METHOD_CDELEGATE_TARGET_PARAMETER_POSITION:
		vala_method_set_cdelegate_target_parameter_position (self, g_value_get_double (value));
		break;
		case VALA_METHOD_NO_ARRAY_LENGTH:
		vala_method_set_no_array_length (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_PRINTF_FORMAT:
		vala_method_set_printf_format (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_method_class_init (ValaMethodClass * klass) {
	vala_method_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaMethodPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_method_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_method_set_property;
	G_OBJECT_CLASS (klass)->constructor = vala_method_constructor;
	G_OBJECT_CLASS (klass)->dispose = vala_method_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_method_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_method_real_accept_children;
	VALA_METHOD_CLASS (klass)->get_default_cname = vala_method_real_get_default_cname;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_method_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->create_code_binding = vala_method_real_create_code_binding;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_RETURN_TYPE, g_param_spec_object ("return-type", "return-type", "return-type", VALA_TYPE_DATA_TYPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_BODY, g_param_spec_object ("body", "body", "body", VALA_TYPE_BLOCK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_ENTRY_BLOCK, g_param_spec_object ("entry-block", "entry-block", "entry-block", VALA_TYPE_BASIC_BLOCK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_EXIT_BLOCK, g_param_spec_object ("exit-block", "exit-block", "exit-block", VALA_TYPE_BASIC_BLOCK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_BINDING, g_param_spec_enum ("binding", "binding", "binding", TYPE_MEMBER_BINDING, MEMBER_BINDING_INSTANCE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_VFUNC_NAME, g_param_spec_string ("vfunc-name", "vfunc-name", "vfunc-name", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_SENTINEL, g_param_spec_string ("sentinel", "sentinel", "sentinel", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_IS_ABSTRACT, g_param_spec_boolean ("is-abstract", "is-abstract", "is-abstract", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_IS_VIRTUAL, g_param_spec_boolean ("is-virtual", "is-virtual", "is-virtual", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_OVERRIDES, g_param_spec_boolean ("overrides", "overrides", "overrides", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_IS_INLINE, g_param_spec_boolean ("is-inline", "is-inline", "is-inline", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_RETURNS_MODIFIED_POINTER, g_param_spec_boolean ("returns-modified-pointer", "returns-modified-pointer", "returns-modified-pointer", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_BASE_METHOD, g_param_spec_object ("base-method", "base-method", "base-method", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_BASE_INTERFACE_METHOD, g_param_spec_object ("base-interface-method", "base-interface-method", "base-interface-method", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_THIS_PARAMETER, g_param_spec_object ("this-parameter", "this-parameter", "this-parameter", VALA_TYPE_FORMAL_PARAMETER, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_RESULT_VAR, g_param_spec_object ("result-var", "result-var", "result-var", VALA_TYPE_LOCAL_VARIABLE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_CINSTANCE_PARAMETER_POSITION, g_param_spec_double ("cinstance-parameter-position", "cinstance-parameter-position", "cinstance-parameter-position", -G_MAXDOUBLE, G_MAXDOUBLE, 0.0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_CARRAY_LENGTH_PARAMETER_POSITION, g_param_spec_double ("carray-length-parameter-position", "carray-length-parameter-position", "carray-length-parameter-position", -G_MAXDOUBLE, G_MAXDOUBLE, 0.0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_CDELEGATE_TARGET_PARAMETER_POSITION, g_param_spec_double ("cdelegate-target-parameter-position", "cdelegate-target-parameter-position", "cdelegate-target-parameter-position", -G_MAXDOUBLE, G_MAXDOUBLE, 0.0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_NO_ARRAY_LENGTH, g_param_spec_boolean ("no-array-length", "no-array-length", "no-array-length", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_PRINTF_FORMAT, g_param_spec_boolean ("printf-format", "printf-format", "printf-format", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_method_init (ValaMethod * self) {
	self->priv = VALA_METHOD_GET_PRIVATE (self);
	self->priv->_binding = MEMBER_BINDING_INSTANCE;
	self->priv->parameters = GEE_LIST (gee_array_list_new (VALA_TYPE_FORMAL_PARAMETER, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->error_domains = GEE_LIST (gee_array_list_new (VALA_TYPE_DATA_TYPE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->preconditions = GEE_LIST (gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->postconditions = GEE_LIST (gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
}


static void vala_method_dispose (GObject * obj) {
	ValaMethod * self;
	self = VALA_METHOD (obj);
	(self->priv->_entry_block == NULL ? NULL : (self->priv->_entry_block = (g_object_unref (self->priv->_entry_block), NULL)));
	(self->priv->_exit_block == NULL ? NULL : (self->priv->_exit_block = (g_object_unref (self->priv->_exit_block), NULL)));
	(self->priv->_base_interface_method == NULL ? NULL : (self->priv->_base_interface_method = (g_object_unref (self->priv->_base_interface_method), NULL)));
	(self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (g_object_unref (self->priv->_this_parameter), NULL)));
	(self->priv->_result_var == NULL ? NULL : (self->priv->_result_var = (g_object_unref (self->priv->_result_var), NULL)));
	(self->priv->parameters == NULL ? NULL : (self->priv->parameters = (g_object_unref (self->priv->parameters), NULL)));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	self->priv->_vfunc_name = (g_free (self->priv->_vfunc_name), NULL);
	self->priv->_sentinel = (g_free (self->priv->_sentinel), NULL);
	(self->priv->error_domains == NULL ? NULL : (self->priv->error_domains = (g_object_unref (self->priv->error_domains), NULL)));
	(self->priv->preconditions == NULL ? NULL : (self->priv->preconditions = (g_object_unref (self->priv->preconditions), NULL)));
	(self->priv->postconditions == NULL ? NULL : (self->priv->postconditions = (g_object_unref (self->priv->postconditions), NULL)));
	(self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL)));
	(self->priv->_body == NULL ? NULL : (self->priv->_body = (g_object_unref (self->priv->_body), NULL)));
	G_OBJECT_CLASS (vala_method_parent_class)->dispose (obj);
}


GType vala_method_get_type (void) {
	static GType vala_method_type_id = 0;
	if (G_UNLIKELY (vala_method_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMethodClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_method_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMethod), 0, (GInstanceInitFunc) vala_method_init };
		vala_method_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaMethod", &g_define_type_info, 0);
	}
	return vala_method_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




