/* valaparser.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaparser.h>
#include <stdlib.h>
#include <string.h>
#include <gee/list.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <vala/valascanner.h>
#include <vala/valasourcelocation.h>
#include <vala/valatokentype.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valaexpression.h>
#include <vala/valabooleanliteral.h>
#include <vala/valaintegerliteral.h>
#include <vala/valarealliteral.h>
#include <vala/valacharacterliteral.h>
#include <vala/valacodenode.h>
#include <vala/valastringliteral.h>
#include <vala/valanullliteral.h>
#include <vala/valasymbol.h>
#include <vala/valaunresolvedsymbol.h>
#include <vala/valadatatype.h>
#include <vala/valavoidtype.h>
#include <vala/valapointertype.h>
#include <vala/valaunresolvedtype.h>
#include <vala/valaarraytype.h>
#include <vala/valamemberaccess.h>
#include <vala/valatuple.h>
#include <vala/valaparenthesizedexpression.h>
#include <vala/valaobjectcreationexpression.h>
#include <vala/valamemberinitializer.h>
#include <vala/valainvocationexpression.h>
#include <vala/valaelementaccess.h>
#include <vala/valabaseaccess.h>
#include <vala/valapostfixexpression.h>
#include <vala/valainitializerlist.h>
#include <vala/valaarraycreationexpression.h>
#include <vala/valasizeofexpression.h>
#include <vala/valatypeofexpression.h>
#include <vala/valaunaryexpression.h>
#include <vala/valareferencetransferexpression.h>
#include <vala/valacastexpression.h>
#include <vala/valapointerindirection.h>
#include <vala/valaaddressofexpression.h>
#include <vala/valabinaryexpression.h>
#include <vala/valatypecheck.h>
#include <vala/valaconditionalexpression.h>
#include <vala/valalambdaexpression.h>
#include <vala/valablock.h>
#include <vala/valaassignment.h>
#include <vala/valastatement.h>
#include <vala/valaemptystatement.h>
#include <vala/valalocalvariable.h>
#include <vala/valadeclarationstatement.h>
#include <vala/valaexpressionstatement.h>
#include <vala/valaifstatement.h>
#include <vala/valaswitchstatement.h>
#include <vala/valaswitchsection.h>
#include <vala/valaswitchlabel.h>
#include <vala/valawhilestatement.h>
#include <vala/valadostatement.h>
#include <vala/valaforstatement.h>
#include <vala/valaforeachstatement.h>
#include <vala/valabreakstatement.h>
#include <vala/valacontinuestatement.h>
#include <vala/valareturnstatement.h>
#include <vala/valathrowstatement.h>
#include <vala/valacatchclause.h>
#include <vala/valatrystatement.h>
#include <vala/valalockstatement.h>
#include <vala/valadeletestatement.h>
#include <vala/valaattribute.h>
#include <vala/valanamedargument.h>
#include <vala/valanamespace.h>
#include <vala/valaclass.h>
#include <vala/valastruct.h>
#include <vala/valainterface.h>
#include <vala/valaenum.h>
#include <vala/valaerrordomain.h>
#include <vala/valadelegate.h>
#include <vala/valamethod.h>
#include <vala/valamember.h>
#include <vala/valafield.h>
#include <vala/valaconstant.h>
#include <vala/valausingdirective.h>
#include <vala/valatypeparameter.h>
#include <vala/valacreationmethod.h>
#include <vala/valasignal.h>
#include <vala/valaproperty.h>
#include <vala/valaconstructor.h>
#include <vala/valadestructor.h>
#include <vala/valaformalparameter.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valaenumvalue.h>
#include <vala/valaerrorcode.h>


#define VALA_PARSER_TYPE_MODIFIER_FLAGS (vala_parser_modifier_flags_get_type ())

#define VALA_PARSER_TYPE_RECOVERY_STATE (vala_parser_recovery_state_get_type ())
typedef struct _ValaParserTokenInfo ValaParserTokenInfo;

typedef enum  {
	VALA_PARSER_MODIFIER_FLAGS_NONE,
	VALA_PARSER_MODIFIER_FLAGS_ABSTRACT = 1 << 0,
	VALA_PARSER_MODIFIER_FLAGS_CLASS = 1 << 1,
	VALA_PARSER_MODIFIER_FLAGS_EXTERN = 1 << 2,
	VALA_PARSER_MODIFIER_FLAGS_INLINE = 1 << 3,
	VALA_PARSER_MODIFIER_FLAGS_OVERRIDE = 1 << 4,
	VALA_PARSER_MODIFIER_FLAGS_STATIC = 1 << 5,
	VALA_PARSER_MODIFIER_FLAGS_VIRTUAL = 1 << 6
} ValaParserModifierFlags;

typedef enum  {
	VALA_PARSER_RECOVERY_STATE_EOF,
	VALA_PARSER_RECOVERY_STATE_DECLARATION_BEGIN,
	VALA_PARSER_RECOVERY_STATE_STATEMENT_BEGIN
} ValaParserRecoveryState;

struct _ValaParserTokenInfo {
	ValaTokenType type;
	ValaSourceLocation begin;
	ValaSourceLocation end;
};



struct _ValaParserPrivate {
	ValaScanner* scanner;
	ValaCodeContext* context;
	ValaParserTokenInfo* tokens;
	gint tokens_length1;
	gint index;
	gint size;
	char* comment;
};

#define VALA_PARSER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_PARSER, ValaParserPrivate))
enum  {
	VALA_PARSER_DUMMY_PROPERTY
};
GType vala_parser_modifier_flags_get_type (void);
GType vala_parser_recovery_state_get_type (void);
#define VALA_PARSER_BUFFER_SIZE 32
static void vala_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static inline gboolean vala_parser_next (ValaParser* self);
static inline void vala_parser_prev (ValaParser* self);
static inline ValaTokenType vala_parser_current (ValaParser* self);
static inline gboolean vala_parser_accept (ValaParser* self, ValaTokenType type);
static char* vala_parser_get_error (ValaParser* self, const char* msg);
static inline gboolean vala_parser_expect (ValaParser* self, ValaTokenType type, GError** error);
static inline ValaSourceLocation vala_parser_get_location (ValaParser* self);
static char* vala_parser_get_last_string (ValaParser* self);
static ValaSourceReference* vala_parser_get_src (ValaParser* self, ValaSourceLocation* begin);
static ValaSourceReference* vala_parser_get_src_com (ValaParser* self, ValaSourceLocation* begin);
static ValaSourceReference* vala_parser_get_current_src (ValaParser* self);
static ValaSourceReference* vala_parser_get_last_src (ValaParser* self);
static void vala_parser_rollback (ValaParser* self, ValaSourceLocation* location);
static void vala_parser_skip_identifier (ValaParser* self, GError** error);
static char* vala_parser_parse_identifier (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_literal (ValaParser* self, GError** error);
static void vala_parser_skip_symbol_name (ValaParser* self, GError** error);
static ValaUnresolvedSymbol* vala_parser_parse_symbol_name (ValaParser* self, GError** error);
static void vala_parser_skip_type (ValaParser* self, GError** error);
static ValaDataType* vala_parser_parse_type (ValaParser* self, gboolean owned_by_default, GError** error);
static GeeList* vala_parser_parse_argument_list (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_primary_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_simple_name (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_tuple (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_member_access (ValaParser* self, ValaSourceLocation* begin, ValaExpression* inner, GError** error);
static ValaExpression* vala_parser_parse_pointer_member_access (ValaParser* self, ValaSourceLocation* begin, ValaExpression* inner, GError** error);
static ValaExpression* vala_parser_parse_invocation_expression (ValaParser* self, ValaSourceLocation* begin, ValaExpression* inner, GError** error);
static ValaExpression* vala_parser_parse_element_access (ValaParser* self, ValaSourceLocation* begin, ValaExpression* inner, GError** error);
static GeeList* vala_parser_parse_expression_list (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_this_access (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_base_access (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_post_increment_expression (ValaParser* self, ValaSourceLocation* begin, ValaExpression* inner, GError** error);
static ValaExpression* vala_parser_parse_post_decrement_expression (ValaParser* self, ValaSourceLocation* begin, ValaExpression* inner, GError** error);
static ValaExpression* vala_parser_parse_object_or_array_creation_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_object_creation_expression (ValaParser* self, ValaSourceLocation* begin, ValaMemberAccess* member, GError** error);
static ValaExpression* vala_parser_parse_array_creation_expression (ValaParser* self, ValaSourceLocation* begin, ValaMemberAccess* member, GError** error);
static GeeList* vala_parser_parse_object_initializer (ValaParser* self, GError** error);
static ValaMemberInitializer* vala_parser_parse_member_initializer (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_sizeof_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_typeof_expression (ValaParser* self, GError** error);
static ValaUnaryOperator vala_parser_get_unary_operator (ValaParser* self, ValaTokenType token_type);
static ValaExpression* vala_parser_parse_unary_expression (ValaParser* self, GError** error);
static ValaBinaryOperator vala_parser_get_binary_operator (ValaParser* self, ValaTokenType token_type);
static ValaExpression* vala_parser_parse_multiplicative_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_additive_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_shift_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_relational_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_equality_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_and_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_exclusive_or_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_inclusive_or_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_in_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_conditional_and_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_conditional_or_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_conditional_expression (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_lambda_expression (ValaParser* self, GError** error);
static ValaAssignmentOperator vala_parser_get_assignment_operator (ValaParser* self, ValaTokenType token_type);
static ValaExpression* vala_parser_parse_expression (ValaParser* self, GError** error);
static void vala_parser_parse_statements (ValaParser* self, ValaBlock* block, GError** error);
static gboolean vala_parser_is_expression (ValaParser* self, GError** error);
static ValaBlock* vala_parser_parse_embedded_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_embedded_statement_without_block (ValaParser* self, GError** error);
static ValaBlock* vala_parser_parse_block (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_empty_statement (ValaParser* self, GError** error);
static void vala_parser_parse_local_variable_declarations (ValaParser* self, ValaBlock* block, GError** error);
static ValaLocalVariable* vala_parser_parse_local_variable (ValaParser* self, ValaDataType* variable_type, GError** error);
static ValaStatement* vala_parser_parse_expression_statement (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_statement_expression (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_if_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_switch_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_while_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_do_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_for_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_foreach_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_break_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_continue_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_return_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_throw_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_try_statement (ValaParser* self, GError** error);
static void vala_parser_parse_catch_clauses (ValaParser* self, GeeList* catch_clauses, GError** error);
static ValaBlock* vala_parser_parse_finally_clause (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_lock_statement (ValaParser* self, GError** error);
static ValaStatement* vala_parser_parse_delete_statement (ValaParser* self, GError** error);
static GeeList* vala_parser_parse_attributes (ValaParser* self, GError** error);
static void vala_parser_set_attributes (ValaParser* self, ValaCodeNode* node, GeeList* attributes);
static ValaSymbol* vala_parser_parse_declaration (ValaParser* self, GError** error);
static void vala_parser_parse_declarations (ValaParser* self, ValaSymbol* parent, gboolean root, GError** error);
static ValaParserRecoveryState vala_parser_recover (ValaParser* self);
static ValaNamespace* vala_parser_parse_namespace_declaration (ValaParser* self, GeeList* attrs, GError** error);
static void vala_parser_parse_namespace_member (ValaParser* self, ValaNamespace* ns, GError** error);
static void vala_parser_parse_using_directives (ValaParser* self, GError** error);
static ValaSymbol* vala_parser_parse_class_declaration (ValaParser* self, GeeList* attrs, GError** error);
static void vala_parser_parse_class_member (ValaParser* self, ValaClass* cl, GError** error);
static ValaConstant* vala_parser_parse_constant_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaField* vala_parser_parse_field_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaInitializerList* vala_parser_parse_initializer (ValaParser* self, GError** error);
static ValaExpression* vala_parser_parse_variable_initializer (ValaParser* self, GError** error);
static ValaMethod* vala_parser_parse_method_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaProperty* vala_parser_parse_property_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaSignal* vala_parser_parse_signal_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaConstructor* vala_parser_parse_constructor_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaDestructor* vala_parser_parse_destructor_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaSymbol* vala_parser_parse_struct_declaration (ValaParser* self, GeeList* attrs, GError** error);
static void vala_parser_parse_struct_member (ValaParser* self, ValaStruct* st, GError** error);
static ValaSymbol* vala_parser_parse_interface_declaration (ValaParser* self, GeeList* attrs, GError** error);
static void vala_parser_parse_interface_member (ValaParser* self, ValaInterface* iface, GError** error);
static ValaSymbol* vala_parser_parse_enum_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaSymbol* vala_parser_parse_errordomain_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaSymbolAccessibility vala_parser_parse_access_modifier (ValaParser* self, ValaSymbolAccessibility default_access);
static ValaParserModifierFlags vala_parser_parse_type_declaration_modifiers (ValaParser* self);
static ValaParserModifierFlags vala_parser_parse_member_declaration_modifiers (ValaParser* self);
static ValaFormalParameter* vala_parser_parse_parameter (ValaParser* self, GError** error);
static ValaCreationMethod* vala_parser_parse_creation_method_declaration (ValaParser* self, GeeList* attrs, GError** error);
static ValaSymbol* vala_parser_parse_delegate_declaration (ValaParser* self, GeeList* attrs, GError** error);
static GeeList* vala_parser_parse_type_parameter_list (ValaParser* self, GError** error);
static void vala_parser_skip_type_argument_list (ValaParser* self, GError** error);
static GeeList* vala_parser_parse_type_argument_list (ValaParser* self, gboolean maybe_expression, GError** error);
static ValaMemberAccess* vala_parser_parse_member_name (ValaParser* self, GError** error);
static gboolean vala_parser_is_declaration_keyword (ValaParser* self, ValaTokenType type);
static gpointer vala_parser_parent_class = NULL;
static void vala_parser_finalize (ValaCodeVisitor* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);




GType vala_parser_modifier_flags_get_type (void) {
	static GType vala_parser_modifier_flags_type_id = 0;
	if (G_UNLIKELY (vala_parser_modifier_flags_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_PARSER_MODIFIER_FLAGS_NONE, "VALA_PARSER_MODIFIER_FLAGS_NONE", "none"}, {VALA_PARSER_MODIFIER_FLAGS_ABSTRACT, "VALA_PARSER_MODIFIER_FLAGS_ABSTRACT", "abstract"}, {VALA_PARSER_MODIFIER_FLAGS_CLASS, "VALA_PARSER_MODIFIER_FLAGS_CLASS", "class"}, {VALA_PARSER_MODIFIER_FLAGS_EXTERN, "VALA_PARSER_MODIFIER_FLAGS_EXTERN", "extern"}, {VALA_PARSER_MODIFIER_FLAGS_INLINE, "VALA_PARSER_MODIFIER_FLAGS_INLINE", "inline"}, {VALA_PARSER_MODIFIER_FLAGS_OVERRIDE, "VALA_PARSER_MODIFIER_FLAGS_OVERRIDE", "override"}, {VALA_PARSER_MODIFIER_FLAGS_STATIC, "VALA_PARSER_MODIFIER_FLAGS_STATIC", "static"}, {VALA_PARSER_MODIFIER_FLAGS_VIRTUAL, "VALA_PARSER_MODIFIER_FLAGS_VIRTUAL", "virtual"}, {0, NULL, NULL}};
		vala_parser_modifier_flags_type_id = g_enum_register_static ("ValaParserModifierFlags", values);
	}
	return vala_parser_modifier_flags_type_id;
}



GType vala_parser_recovery_state_get_type (void) {
	static GType vala_parser_recovery_state_type_id = 0;
	if (G_UNLIKELY (vala_parser_recovery_state_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_PARSER_RECOVERY_STATE_EOF, "VALA_PARSER_RECOVERY_STATE_EOF", "eof"}, {VALA_PARSER_RECOVERY_STATE_DECLARATION_BEGIN, "VALA_PARSER_RECOVERY_STATE_DECLARATION_BEGIN", "declaration-begin"}, {VALA_PARSER_RECOVERY_STATE_STATEMENT_BEGIN, "VALA_PARSER_RECOVERY_STATE_STATEMENT_BEGIN", "statement-begin"}, {0, NULL, NULL}};
		vala_parser_recovery_state_type_id = g_enum_register_static ("ValaParserRecoveryState", values);
	}
	return vala_parser_recovery_state_type_id;
}


ValaParser* vala_parser_construct (GType object_type) {
	ValaParser* self;
	ValaParserTokenInfo* _tmp0;
	self = ((ValaParser*) (g_type_create_instance (object_type)));
	_tmp0 = NULL;
	self->priv->tokens = (_tmp0 = g_new0 (ValaParserTokenInfo, VALA_PARSER_BUFFER_SIZE), (self->priv->tokens = (g_free (self->priv->tokens), NULL)), self->priv->tokens_length1 = VALA_PARSER_BUFFER_SIZE, _tmp0);
	return self;
}


ValaParser* vala_parser_new (void) {
	return vala_parser_construct (VALA_TYPE_PARSER);
}


/**
 * Parses all .vala and .vapi source files in the specified code
 * context and builds a code tree.
 *
 * @param context a code context
 */
void vala_parser_parse (ValaParser* self, ValaCodeContext* context) {
	ValaCodeContext* _tmp1;
	ValaCodeContext* _tmp0;
	g_return_if_fail (VALA_IS_PARSER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->context = (_tmp1 = (_tmp0 = context, (_tmp0 == NULL ? NULL : vala_code_context_ref (_tmp0))), (self->priv->context == NULL ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL))), _tmp1);
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
}


static void vala_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file) {
	ValaParser * self;
	self = VALA_PARSER (base);
	g_return_if_fail (VALA_IS_SOURCE_FILE (source_file));
	if (g_str_has_suffix (vala_source_file_get_filename (source_file), ".vala") || g_str_has_suffix (vala_source_file_get_filename (source_file), ".vapi")) {
		vala_parser_parse_file (self, source_file);
	}
}


static inline gboolean vala_parser_next (ValaParser* self) {
	g_return_val_if_fail (VALA_IS_PARSER (self), FALSE);
	self->priv->index = (self->priv->index + 1) % VALA_PARSER_BUFFER_SIZE;
	self->priv->size--;
	if (self->priv->size <= 0) {
		ValaSourceLocation begin = {0};
		ValaSourceLocation end = {0};
		ValaTokenType type;
		type = vala_scanner_read_token (self->priv->scanner, &begin, &end);
		self->priv->tokens[self->priv->index].type = type;
		self->priv->tokens[self->priv->index].begin = begin;
		self->priv->tokens[self->priv->index].end = end;
		self->priv->size = 1;
	}
	return (self->priv->tokens[self->priv->index].type != VALA_TOKEN_TYPE_EOF);
}


static inline void vala_parser_prev (ValaParser* self) {
	g_return_if_fail (VALA_IS_PARSER (self));
	self->priv->index = (self->priv->index - 1 + VALA_PARSER_BUFFER_SIZE) % VALA_PARSER_BUFFER_SIZE;
	self->priv->size++;
	g_assert (self->priv->size <= VALA_PARSER_BUFFER_SIZE);
}


static inline ValaTokenType vala_parser_current (ValaParser* self) {
	g_return_val_if_fail (VALA_IS_PARSER (self), 0);
	return self->priv->tokens[self->priv->index].type;
}


static inline gboolean vala_parser_accept (ValaParser* self, ValaTokenType type) {
	g_return_val_if_fail (VALA_IS_PARSER (self), FALSE);
	if (vala_parser_current (self) == type) {
		vala_parser_next (self);
		return TRUE;
	}
	return FALSE;
}


static char* vala_parser_get_error (ValaParser* self, const char* msg) {
	ValaSourceLocation begin;
	char* _tmp1;
	ValaSourceReference* _tmp0;
	const char* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (msg != NULL, NULL);
	begin = vala_parser_get_location (self);
	vala_parser_next (self);
	_tmp1 = NULL;
	_tmp0 = NULL;
	vala_report_error ((_tmp0 = vala_parser_get_src (self, &begin)), (_tmp1 = g_strconcat ("syntax error, ", msg, NULL)));
	_tmp1 = (g_free (_tmp1), NULL);
	(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
	_tmp2 = NULL;
	return (_tmp2 = msg, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


static inline gboolean vala_parser_expect (ValaParser* self, ValaTokenType type, GError** error) {
	GError * inner_error;
	char* _tmp2;
	char* _tmp1;
	GError* _tmp3;
	g_return_val_if_fail (VALA_IS_PARSER (self), FALSE);
	inner_error = NULL;
	if (vala_parser_accept (self, type)) {
		return TRUE;
	}
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp3 = NULL;
	inner_error = (_tmp3 = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, (_tmp2 = vala_parser_get_error (self, (_tmp1 = g_strdup_printf ("expected %s", vala_token_type_to_string (type)))))), (_tmp2 = (g_free (_tmp2), NULL)), (_tmp1 = (g_free (_tmp1), NULL)), _tmp3);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return FALSE;
	}
}


static inline ValaSourceLocation vala_parser_get_location (ValaParser* self) {
	0;
	return self->priv->tokens[self->priv->index].begin;
}


static char* vala_parser_get_last_string (ValaParser* self) {
	gint last_index;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	last_index = (self->priv->index + VALA_PARSER_BUFFER_SIZE - 1) % VALA_PARSER_BUFFER_SIZE;
	return g_strndup ((((const char*) (self->priv->tokens[last_index].begin.pos))), ((gulong) ((self->priv->tokens[last_index].end.pos - self->priv->tokens[last_index].begin.pos))));
}


static ValaSourceReference* vala_parser_get_src (ValaParser* self, ValaSourceLocation* begin) {
	gint last_index;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	last_index = (self->priv->index + VALA_PARSER_BUFFER_SIZE - 1) % VALA_PARSER_BUFFER_SIZE;
	return vala_source_reference_new (vala_scanner_get_source_file (self->priv->scanner), (*begin).line, (*begin).column, self->priv->tokens[last_index].end.line, self->priv->tokens[last_index].end.column);
}


static ValaSourceReference* vala_parser_get_src_com (ValaParser* self, ValaSourceLocation* begin) {
	gint last_index;
	ValaSourceReference* src;
	char* _tmp0;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	last_index = (self->priv->index + VALA_PARSER_BUFFER_SIZE - 1) % VALA_PARSER_BUFFER_SIZE;
	src = vala_source_reference_new_with_comment (vala_scanner_get_source_file (self->priv->scanner), (*begin).line, (*begin).column, self->priv->tokens[last_index].end.line, self->priv->tokens[last_index].end.column, self->priv->comment);
	_tmp0 = NULL;
	self->priv->comment = (_tmp0 = NULL, (self->priv->comment = (g_free (self->priv->comment), NULL)), _tmp0);
	return src;
}


static ValaSourceReference* vala_parser_get_current_src (ValaParser* self) {
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	return vala_source_reference_new (vala_scanner_get_source_file (self->priv->scanner), self->priv->tokens[self->priv->index].begin.line, self->priv->tokens[self->priv->index].begin.column, self->priv->tokens[self->priv->index].end.line, self->priv->tokens[self->priv->index].end.column);
}


static ValaSourceReference* vala_parser_get_last_src (ValaParser* self) {
	gint last_index;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	last_index = (self->priv->index + VALA_PARSER_BUFFER_SIZE - 1) % VALA_PARSER_BUFFER_SIZE;
	return vala_source_reference_new (vala_scanner_get_source_file (self->priv->scanner), self->priv->tokens[last_index].begin.line, self->priv->tokens[last_index].begin.column, self->priv->tokens[last_index].end.line, self->priv->tokens[last_index].end.column);
}


static void vala_parser_rollback (ValaParser* self, ValaSourceLocation* location) {
	g_return_if_fail (VALA_IS_PARSER (self));
	while (self->priv->tokens[self->priv->index].begin.pos != (*location).pos) {
		vala_parser_prev (self);
	}
}


static void vala_parser_skip_identifier (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaTokenType _tmp0;
	char* _tmp1;
	GError* _tmp2;
	g_return_if_fail (VALA_IS_PARSER (self));
	inner_error = NULL;
	/* also accept keywords as identifiers where there is no conflict*/
	_tmp0 = vala_parser_current (self);
	if (_tmp0 == VALA_TOKEN_TYPE_ABSTRACT || _tmp0 == VALA_TOKEN_TYPE_AS || _tmp0 == VALA_TOKEN_TYPE_BASE || _tmp0 == VALA_TOKEN_TYPE_BREAK || _tmp0 == VALA_TOKEN_TYPE_CASE || _tmp0 == VALA_TOKEN_TYPE_CATCH || _tmp0 == VALA_TOKEN_TYPE_CLASS || _tmp0 == VALA_TOKEN_TYPE_CONST || _tmp0 == VALA_TOKEN_TYPE_CONSTRUCT || _tmp0 == VALA_TOKEN_TYPE_CONTINUE || _tmp0 == VALA_TOKEN_TYPE_DEFAULT || _tmp0 == VALA_TOKEN_TYPE_DELEGATE || _tmp0 == VALA_TOKEN_TYPE_DELETE || _tmp0 == VALA_TOKEN_TYPE_DO || _tmp0 == VALA_TOKEN_TYPE_DYNAMIC || _tmp0 == VALA_TOKEN_TYPE_ELSE || _tmp0 == VALA_TOKEN_TYPE_ENUM || _tmp0 == VALA_TOKEN_TYPE_ENSURES || _tmp0 == VALA_TOKEN_TYPE_ERRORDOMAIN || _tmp0 == VALA_TOKEN_TYPE_EXTERN || _tmp0 == VALA_TOKEN_TYPE_FALSE || _tmp0 == VALA_TOKEN_TYPE_FINALLY || _tmp0 == VALA_TOKEN_TYPE_FOR || _tmp0 == VALA_TOKEN_TYPE_FOREACH || _tmp0 == VALA_TOKEN_TYPE_GET || _tmp0 == VALA_TOKEN_TYPE_IDENTIFIER || _tmp0 == VALA_TOKEN_TYPE_IF || _tmp0 == VALA_TOKEN_TYPE_IN || _tmp0 == VALA_TOKEN_TYPE_INLINE || _tmp0 == VALA_TOKEN_TYPE_INTERFACE || _tmp0 == VALA_TOKEN_TYPE_INTERNAL || _tmp0 == VALA_TOKEN_TYPE_IS || _tmp0 == VALA_TOKEN_TYPE_LOCK || _tmp0 == VALA_TOKEN_TYPE_NAMESPACE || _tmp0 == VALA_TOKEN_TYPE_NEW || _tmp0 == VALA_TOKEN_TYPE_NULL || _tmp0 == VALA_TOKEN_TYPE_OUT || _tmp0 == VALA_TOKEN_TYPE_OVERRIDE || _tmp0 == VALA_TOKEN_TYPE_PRIVATE || _tmp0 == VALA_TOKEN_TYPE_PROTECTED || _tmp0 == VALA_TOKEN_TYPE_PUBLIC || _tmp0 == VALA_TOKEN_TYPE_REF || _tmp0 == VALA_TOKEN_TYPE_REQUIRES || _tmp0 == VALA_TOKEN_TYPE_RETURN || _tmp0 == VALA_TOKEN_TYPE_SET || _tmp0 == VALA_TOKEN_TYPE_SIGNAL || _tmp0 == VALA_TOKEN_TYPE_SIZEOF || _tmp0 == VALA_TOKEN_TYPE_STATIC || _tmp0 == VALA_TOKEN_TYPE_STRUCT || _tmp0 == VALA_TOKEN_TYPE_SWITCH || _tmp0 == VALA_TOKEN_TYPE_THIS || _tmp0 == VALA_TOKEN_TYPE_THROW || _tmp0 == VALA_TOKEN_TYPE_THROWS || _tmp0 == VALA_TOKEN_TYPE_TRUE || _tmp0 == VALA_TOKEN_TYPE_TRY || _tmp0 == VALA_TOKEN_TYPE_TYPEOF || _tmp0 == VALA_TOKEN_TYPE_USING || _tmp0 == VALA_TOKEN_TYPE_VAR || _tmp0 == VALA_TOKEN_TYPE_VIRTUAL || _tmp0 == VALA_TOKEN_TYPE_VOID || _tmp0 == VALA_TOKEN_TYPE_VOLATILE || _tmp0 == VALA_TOKEN_TYPE_WEAK || _tmp0 == VALA_TOKEN_TYPE_WHILE)
	do {
		vala_parser_next (self);
		return;
	} while (0);
	_tmp1 = NULL;
	_tmp2 = NULL;
	inner_error = (_tmp2 = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, (_tmp1 = vala_parser_get_error (self, "expected identifier"))), (_tmp1 = (g_free (_tmp1), NULL)), _tmp2);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return;
	}
}


static char* vala_parser_parse_identifier (ValaParser* self, GError** error) {
	GError * inner_error;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	vala_parser_skip_identifier (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	return vala_parser_get_last_string (self);
}


static ValaExpression* vala_parser_parse_literal (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaTokenType _tmp32;
	char* _tmp33;
	GError* _tmp34;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	_tmp32 = vala_parser_current (self);
	if (_tmp32 == VALA_TOKEN_TYPE_TRUE)
	do {
		ValaSourceReference* _tmp0;
		ValaExpression* _tmp1;
		vala_parser_next (self);
		_tmp0 = NULL;
		_tmp1 = NULL;
		return (_tmp1 = VALA_EXPRESSION (vala_boolean_literal_new (TRUE, (_tmp0 = vala_parser_get_src (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	} while (0); else if (_tmp32 == VALA_TOKEN_TYPE_FALSE)
	do {
		ValaSourceReference* _tmp3;
		ValaExpression* _tmp4;
		vala_parser_next (self);
		_tmp3 = NULL;
		_tmp4 = NULL;
		return (_tmp4 = VALA_EXPRESSION (vala_boolean_literal_new (FALSE, (_tmp3 = vala_parser_get_src (self, &begin)))), (_tmp3 == NULL ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL))), _tmp4);
	} while (0); else if (_tmp32 == VALA_TOKEN_TYPE_INTEGER_LITERAL)
	do {
		ValaSourceReference* _tmp7;
		char* _tmp6;
		ValaExpression* _tmp8;
		vala_parser_next (self);
		_tmp7 = NULL;
		_tmp6 = NULL;
		_tmp8 = NULL;
		return (_tmp8 = VALA_EXPRESSION (vala_integer_literal_new ((_tmp6 = vala_parser_get_last_string (self)), (_tmp7 = vala_parser_get_src (self, &begin)))), (_tmp7 == NULL ? NULL : (_tmp7 = (vala_source_reference_unref (_tmp7), NULL))), (_tmp6 = (g_free (_tmp6), NULL)), _tmp8);
	} while (0); else if (_tmp32 == VALA_TOKEN_TYPE_REAL_LITERAL)
	do {
		ValaSourceReference* _tmp11;
		char* _tmp10;
		ValaExpression* _tmp12;
		vala_parser_next (self);
		_tmp11 = NULL;
		_tmp10 = NULL;
		_tmp12 = NULL;
		return (_tmp12 = VALA_EXPRESSION (vala_real_literal_new ((_tmp10 = vala_parser_get_last_string (self)), (_tmp11 = vala_parser_get_src (self, &begin)))), (_tmp11 == NULL ? NULL : (_tmp11 = (vala_source_reference_unref (_tmp11), NULL))), (_tmp10 = (g_free (_tmp10), NULL)), _tmp12);
	} while (0); else if (_tmp32 == VALA_TOKEN_TYPE_CHARACTER_LITERAL)
	do {
		ValaSourceReference* _tmp15;
		char* _tmp14;
		ValaCharacterLiteral* _tmp16;
		ValaCharacterLiteral* lit;
		vala_parser_next (self);
		_tmp15 = NULL;
		_tmp14 = NULL;
		_tmp16 = NULL;
		lit = (_tmp16 = vala_character_literal_new ((_tmp14 = vala_parser_get_last_string (self)), (_tmp15 = vala_parser_get_src (self, &begin))), (_tmp15 == NULL ? NULL : (_tmp15 = (vala_source_reference_unref (_tmp15), NULL))), (_tmp14 = (g_free (_tmp14), NULL)), _tmp16);
		if (vala_code_node_get_error (VALA_CODE_NODE (lit))) {
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (lit)), "invalid character literal");
		}
		return VALA_EXPRESSION (lit);
	} while (0); else if (_tmp32 == VALA_TOKEN_TYPE_STRING_LITERAL)
	do {
		ValaSourceReference* _tmp19;
		char* _tmp18;
		ValaExpression* _tmp20;
		vala_parser_next (self);
		_tmp19 = NULL;
		_tmp18 = NULL;
		_tmp20 = NULL;
		return (_tmp20 = VALA_EXPRESSION (vala_string_literal_new ((_tmp18 = vala_parser_get_last_string (self)), (_tmp19 = vala_parser_get_src (self, &begin)))), (_tmp19 == NULL ? NULL : (_tmp19 = (vala_source_reference_unref (_tmp19), NULL))), (_tmp18 = (g_free (_tmp18), NULL)), _tmp20);
	} while (0); else if (_tmp32 == VALA_TOKEN_TYPE_VERBATIM_STRING_LITERAL)
	do {
		char* raw_string;
		char* _tmp23;
		char* _tmp22;
		char* _tmp24;
		char* escaped_string;
		ValaSourceReference* _tmp26;
		char* _tmp25;
		ValaExpression* _tmp27;
		ValaExpression* _tmp28;
		vala_parser_next (self);
		raw_string = vala_parser_get_last_string (self);
		_tmp23 = NULL;
		_tmp22 = NULL;
		_tmp24 = NULL;
		escaped_string = (_tmp24 = g_strescape ((_tmp23 = (_tmp22 = g_utf8_offset_to_pointer (raw_string, ((glong) (3))), g_strndup (_tmp22, g_utf8_offset_to_pointer (_tmp22, g_utf8_strlen (raw_string, -1) - 6) - _tmp22))), ""), (_tmp23 = (g_free (_tmp23), NULL)), _tmp24);
		_tmp26 = NULL;
		_tmp25 = NULL;
		_tmp27 = NULL;
		_tmp28 = NULL;
		return (_tmp28 = (_tmp27 = VALA_EXPRESSION (vala_string_literal_new ((_tmp25 = g_strdup_printf ("\"%s\"", escaped_string)), (_tmp26 = vala_parser_get_src (self, &begin)))), (_tmp26 == NULL ? NULL : (_tmp26 = (vala_source_reference_unref (_tmp26), NULL))), (_tmp25 = (g_free (_tmp25), NULL)), _tmp27), (raw_string = (g_free (raw_string), NULL)), (escaped_string = (g_free (escaped_string), NULL)), _tmp28);
	} while (0); else if (_tmp32 == VALA_TOKEN_TYPE_NULL)
	do {
		ValaSourceReference* _tmp29;
		ValaExpression* _tmp30;
		vala_parser_next (self);
		_tmp29 = NULL;
		_tmp30 = NULL;
		return (_tmp30 = VALA_EXPRESSION (vala_null_literal_new ((_tmp29 = vala_parser_get_src (self, &begin)))), (_tmp29 == NULL ? NULL : (_tmp29 = (vala_source_reference_unref (_tmp29), NULL))), _tmp30);
	} while (0);
	_tmp33 = NULL;
	_tmp34 = NULL;
	inner_error = (_tmp34 = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, (_tmp33 = vala_parser_get_error (self, "expected literal"))), (_tmp33 = (g_free (_tmp33), NULL)), _tmp34);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
}


void vala_parser_parse_file (ValaParser* self, ValaSourceFile* source_file) {
	GError * inner_error;
	ValaScanner* _tmp0;
	ValaScanner* _tmp1;
	g_return_if_fail (VALA_IS_PARSER (self));
	g_return_if_fail (VALA_IS_SOURCE_FILE (source_file));
	inner_error = NULL;
	_tmp0 = NULL;
	self->priv->scanner = (_tmp0 = vala_scanner_new (source_file), (self->priv->scanner == NULL ? NULL : (self->priv->scanner = (vala_scanner_unref (self->priv->scanner), NULL))), _tmp0);
	self->priv->index = -1;
	self->priv->size = 0;
	vala_parser_next (self);
	{
		vala_parser_parse_using_directives (self, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				goto __catch3_vala_parse_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
		vala_parser_parse_declarations (self, VALA_SYMBOL (vala_code_context_get_root (self->priv->context)), TRUE, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == VALA_PARSE_ERROR) {
				goto __catch3_vala_parse_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
	}
	goto __finally3;
	__catch3_vala_parse_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			(e == NULL ? NULL : (e = (g_error_free (e), NULL)));
		}
	}
	__finally3:
	;
	/* already reported*/
	_tmp1 = NULL;
	self->priv->scanner = (_tmp1 = NULL, (self->priv->scanner == NULL ? NULL : (self->priv->scanner = (vala_scanner_unref (self->priv->scanner), NULL))), _tmp1);
}


static void vala_parser_skip_symbol_name (ValaParser* self, GError** error) {
	GError * inner_error;
	g_return_if_fail (VALA_IS_PARSER (self));
	inner_error = NULL;
	do {
		vala_parser_skip_identifier (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			return;
		}
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_DOT) || vala_parser_accept (self, VALA_TOKEN_TYPE_DOUBLE_COLON));
}


static ValaUnresolvedSymbol* vala_parser_parse_symbol_name (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaUnresolvedSymbol* sym;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	sym = NULL;
	do {
		char* name;
		ValaUnresolvedSymbol* _tmp4;
		ValaSourceReference* _tmp3;
		name = vala_parser_parse_identifier (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
			return NULL;
		}
		if (_vala_strcmp0 (name, "global") == 0 && vala_parser_accept (self, VALA_TOKEN_TYPE_DOUBLE_COLON)) {
			char* _tmp0;
			ValaUnresolvedSymbol* _tmp2;
			ValaSourceReference* _tmp1;
			/* global::Name
			 qualified access to global symbol*/
			_tmp0 = NULL;
			name = (_tmp0 = vala_parser_parse_identifier (self, &inner_error), (name = (g_free (name), NULL)), _tmp0);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				name = (g_free (name), NULL);
				(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
				return NULL;
			}
			_tmp2 = NULL;
			_tmp1 = NULL;
			sym = (_tmp2 = vala_unresolved_symbol_new (sym, name, (_tmp1 = vala_parser_get_src (self, &begin))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp2);
			(_tmp1 == NULL ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL)));
			vala_unresolved_symbol_set_qualified (sym, TRUE);
			name = (g_free (name), NULL);
			continue;
		}
		_tmp4 = NULL;
		_tmp3 = NULL;
		sym = (_tmp4 = vala_unresolved_symbol_new (sym, name, (_tmp3 = vala_parser_get_src (self, &begin))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp4);
		(_tmp3 == NULL ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL)));
		name = (g_free (name), NULL);
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_DOT));
	return sym;
}


static void vala_parser_skip_type (ValaParser* self, GError** error) {
	GError * inner_error;
	g_return_if_fail (VALA_IS_PARSER (self));
	inner_error = NULL;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_VOID)) {
		while (vala_parser_accept (self, VALA_TOKEN_TYPE_STAR)) {
		}
		return;
	}
	vala_parser_accept (self, VALA_TOKEN_TYPE_DYNAMIC);
	vala_parser_accept (self, VALA_TOKEN_TYPE_WEAK);
	vala_parser_skip_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return;
	}
	vala_parser_skip_type_argument_list (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return;
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_STAR)) {
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_BRACKET)) {
		do {
			if (vala_parser_current (self) != VALA_TOKEN_TYPE_COMMA && vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACKET) {
				ValaExpression* _tmp0;
				_tmp0 = NULL;
				_tmp0 = vala_parser_parse_expression (self, &inner_error);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					return;
				}
				(_tmp0 == NULL ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)));
			}
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACKET, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			return;
		}
	}
	vala_parser_accept (self, VALA_TOKEN_TYPE_OP_NEG);
	vala_parser_accept (self, VALA_TOKEN_TYPE_INTERR);
	vala_parser_accept (self, VALA_TOKEN_TYPE_HASH);
}


static ValaDataType* vala_parser_parse_type (ValaParser* self, gboolean owned_by_default, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	gboolean is_dynamic;
	gboolean value_owned;
	ValaUnresolvedSymbol* sym;
	GeeList* type_arg_list;
	ValaSourceReference* _tmp2;
	ValaDataType* _tmp3;
	ValaDataType* type;
	ValaDataType* _tmp10;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_VOID)) {
		ValaDataType* type;
		type = VALA_DATA_TYPE (vala_void_type_new ());
		while (vala_parser_accept (self, VALA_TOKEN_TYPE_STAR)) {
			ValaDataType* _tmp0;
			_tmp0 = NULL;
			type = (_tmp0 = VALA_DATA_TYPE (vala_pointer_type_new (type, NULL)), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp0);
		}
		return type;
	}
	is_dynamic = vala_parser_accept (self, VALA_TOKEN_TYPE_DYNAMIC);
	value_owned = owned_by_default;
	if (owned_by_default) {
		value_owned = !vala_parser_accept (self, VALA_TOKEN_TYPE_WEAK);
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	type_arg_list = vala_parser_parse_type_argument_list (self, FALSE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		return NULL;
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	type = (_tmp3 = VALA_DATA_TYPE (vala_unresolved_type_new_from_symbol (sym, (_tmp2 = vala_parser_get_src (self, &begin)))), (_tmp2 == NULL ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL))), _tmp3);
	if (type_arg_list != NULL) {
		{
			GeeList* type_arg_collection;
			int type_arg_it;
			type_arg_collection = type_arg_list;
			for (type_arg_it = 0; type_arg_it < gee_collection_get_size (GEE_COLLECTION (type_arg_collection)); type_arg_it = type_arg_it + 1) {
				ValaDataType* type_arg;
				type_arg = ((ValaDataType*) (gee_list_get (GEE_LIST (type_arg_collection), type_arg_it)));
				{
					vala_data_type_add_type_argument (type, type_arg);
					(type_arg == NULL ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL)));
				}
			}
		}
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_STAR)) {
		ValaDataType* _tmp5;
		ValaSourceReference* _tmp4;
		_tmp5 = NULL;
		_tmp4 = NULL;
		type = (_tmp5 = VALA_DATA_TYPE (vala_pointer_type_new (type, (_tmp4 = vala_parser_get_src (self, &begin)))), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp5);
		(_tmp4 == NULL ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL)));
	}
	if (!(VALA_IS_POINTER_TYPE (type))) {
		vala_data_type_set_nullable (type, vala_parser_accept (self, VALA_TOKEN_TYPE_INTERR));
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_BRACKET)) {
		gint array_rank;
		ValaDataType* _tmp8;
		ValaSourceReference* _tmp7;
		array_rank = 0;
		do {
			array_rank++;
			/* support for stack-allocated arrays
			 also required for decision between expression and declaration statement*/
			if (vala_parser_current (self) != VALA_TOKEN_TYPE_COMMA && vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACKET) {
				ValaExpression* _tmp6;
				_tmp6 = NULL;
				_tmp6 = vala_parser_parse_expression (self, &inner_error);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
					(type_arg_list == NULL ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL)));
					(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
					return NULL;
				}
				(_tmp6 == NULL ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL)));
			}
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACKET, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
			(type_arg_list == NULL ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL)));
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			return NULL;
		}
		/* arrays contain strong references by default*/
		vala_data_type_set_value_owned (type, TRUE);
		_tmp8 = NULL;
		_tmp7 = NULL;
		type = (_tmp8 = VALA_DATA_TYPE (vala_array_type_new (type, array_rank, (_tmp7 = vala_parser_get_src (self, &begin)))), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp8);
		(_tmp7 == NULL ? NULL : (_tmp7 = (vala_source_reference_unref (_tmp7), NULL)));
		vala_data_type_set_nullable (type, vala_parser_accept (self, VALA_TOKEN_TYPE_INTERR));
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OP_NEG)) {
		ValaSourceReference* _tmp9;
		_tmp9 = NULL;
		vala_report_warning ((_tmp9 = vala_parser_get_last_src (self)), "obsolete syntax, types are non-null by default");
		(_tmp9 == NULL ? NULL : (_tmp9 = (vala_source_reference_unref (_tmp9), NULL)));
	}
	if (!owned_by_default) {
		value_owned = vala_parser_accept (self, VALA_TOKEN_TYPE_HASH);
	}
	vala_data_type_set_is_dynamic (type, is_dynamic);
	vala_data_type_set_value_owned (type, value_owned);
	_tmp10 = NULL;
	return (_tmp10 = type, (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), (type_arg_list == NULL ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL))), _tmp10);
}


static GeeList* vala_parser_parse_argument_list (ValaParser* self, GError** error) {
	GError * inner_error;
	GeeArrayList* list;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	list = gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
		do {
			ValaExpression* _tmp0;
			_tmp0 = NULL;
			gee_collection_add (GEE_COLLECTION (list), (_tmp0 = vala_parser_parse_expression (self, &inner_error)));
			(_tmp0 == NULL ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	return GEE_LIST (list);
}


static ValaExpression* vala_parser_parse_primary_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	ValaTokenType _tmp8;
	gboolean found;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	expr = NULL;
	_tmp8 = vala_parser_current (self);
	if (_tmp8 == VALA_TOKEN_TYPE_TRUE || _tmp8 == VALA_TOKEN_TYPE_FALSE || _tmp8 == VALA_TOKEN_TYPE_INTEGER_LITERAL || _tmp8 == VALA_TOKEN_TYPE_REAL_LITERAL || _tmp8 == VALA_TOKEN_TYPE_CHARACTER_LITERAL || _tmp8 == VALA_TOKEN_TYPE_STRING_LITERAL || _tmp8 == VALA_TOKEN_TYPE_VERBATIM_STRING_LITERAL || _tmp8 == VALA_TOKEN_TYPE_NULL)
	do {
		ValaExpression* _tmp0;
		_tmp0 = NULL;
		expr = (_tmp0 = vala_parser_parse_literal (self, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp0);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			return NULL;
		}
		break;
	} while (0); else if (_tmp8 == VALA_TOKEN_TYPE_OPEN_PARENS)
	do {
		ValaExpression* _tmp1;
		_tmp1 = NULL;
		expr = (_tmp1 = vala_parser_parse_tuple (self, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp1);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			return NULL;
		}
		break;
	} while (0); else if (_tmp8 == VALA_TOKEN_TYPE_THIS)
	do {
		ValaExpression* _tmp2;
		_tmp2 = NULL;
		expr = (_tmp2 = vala_parser_parse_this_access (self, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp2);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			return NULL;
		}
		break;
	} while (0); else if (_tmp8 == VALA_TOKEN_TYPE_BASE)
	do {
		ValaExpression* _tmp3;
		_tmp3 = NULL;
		expr = (_tmp3 = vala_parser_parse_base_access (self, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp3);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			return NULL;
		}
		break;
	} while (0); else if (_tmp8 == VALA_TOKEN_TYPE_NEW)
	do {
		ValaExpression* _tmp4;
		_tmp4 = NULL;
		expr = (_tmp4 = vala_parser_parse_object_or_array_creation_expression (self, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp4);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			return NULL;
		}
		break;
	} while (0); else if (_tmp8 == VALA_TOKEN_TYPE_SIZEOF)
	do {
		ValaExpression* _tmp5;
		_tmp5 = NULL;
		expr = (_tmp5 = vala_parser_parse_sizeof_expression (self, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp5);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			return NULL;
		}
		break;
	} while (0); else if (_tmp8 == VALA_TOKEN_TYPE_TYPEOF)
	do {
		ValaExpression* _tmp6;
		_tmp6 = NULL;
		expr = (_tmp6 = vala_parser_parse_typeof_expression (self, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp6);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			return NULL;
		}
		break;
	} while (0); else
	do {
		ValaExpression* _tmp7;
		_tmp7 = NULL;
		expr = (_tmp7 = vala_parser_parse_simple_name (self, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp7);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			return NULL;
		}
		break;
	} while (0);
	if (expr == NULL) {
		/* workaround for current limitation of exception handling*/
		inner_error = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, "syntax error in primary expression");
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			return NULL;
		}
	}
	/* process primary expressions that start with an inner primary expression*/
	found = TRUE;
	while (found) {
		ValaTokenType _tmp15;
		_tmp15 = vala_parser_current (self);
		if (_tmp15 == VALA_TOKEN_TYPE_DOT)
		do {
			ValaExpression* _tmp9;
			_tmp9 = NULL;
			expr = (_tmp9 = vala_parser_parse_member_access (self, &begin, expr, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp9);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
				return NULL;
			}
			break;
		} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_OP_PTR)
		do {
			ValaExpression* _tmp10;
			_tmp10 = NULL;
			expr = (_tmp10 = vala_parser_parse_pointer_member_access (self, &begin, expr, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp10);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
				return NULL;
			}
			break;
		} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_OPEN_PARENS)
		do {
			ValaExpression* _tmp11;
			_tmp11 = NULL;
			expr = (_tmp11 = vala_parser_parse_invocation_expression (self, &begin, expr, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp11);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
				return NULL;
			}
			break;
		} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_OPEN_BRACKET)
		do {
			ValaExpression* _tmp12;
			_tmp12 = NULL;
			expr = (_tmp12 = vala_parser_parse_element_access (self, &begin, expr, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp12);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
				return NULL;
			}
			break;
		} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_OP_INC)
		do {
			ValaExpression* _tmp13;
			_tmp13 = NULL;
			expr = (_tmp13 = vala_parser_parse_post_increment_expression (self, &begin, expr, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp13);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
				return NULL;
			}
			break;
		} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_OP_DEC)
		do {
			ValaExpression* _tmp14;
			_tmp14 = NULL;
			expr = (_tmp14 = vala_parser_parse_post_decrement_expression (self, &begin, expr, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp14);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
				return NULL;
			}
			break;
		} while (0); else
		do {
			found = FALSE;
			break;
		} while (0);
		if (expr == NULL) {
			/* workaround for current limitation of exception handling*/
			inner_error = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, "syntax error in primary expression");
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
				return NULL;
			}
		}
	}
	return expr;
}


static ValaExpression* vala_parser_parse_simple_name (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	char* id;
	gboolean qualified;
	GeeList* type_arg_list;
	ValaSourceReference* _tmp1;
	ValaMemberAccess* _tmp2;
	ValaMemberAccess* expr;
	ValaExpression* _tmp3;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	qualified = FALSE;
	if (_vala_strcmp0 (id, "global") == 0 && vala_parser_accept (self, VALA_TOKEN_TYPE_DOUBLE_COLON)) {
		char* _tmp0;
		_tmp0 = NULL;
		id = (_tmp0 = vala_parser_parse_identifier (self, &inner_error), (id = (g_free (id), NULL)), _tmp0);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			id = (g_free (id), NULL);
			return NULL;
		}
		qualified = TRUE;
	}
	type_arg_list = vala_parser_parse_type_argument_list (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		id = (g_free (id), NULL);
		return NULL;
	}
	_tmp1 = NULL;
	_tmp2 = NULL;
	expr = (_tmp2 = vala_member_access_new (NULL, id, (_tmp1 = vala_parser_get_src (self, &begin))), (_tmp1 == NULL ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL))), _tmp2);
	vala_member_access_set_qualified (expr, qualified);
	if (type_arg_list != NULL) {
		{
			GeeList* type_arg_collection;
			int type_arg_it;
			type_arg_collection = type_arg_list;
			for (type_arg_it = 0; type_arg_it < gee_collection_get_size (GEE_COLLECTION (type_arg_collection)); type_arg_it = type_arg_it + 1) {
				ValaDataType* type_arg;
				type_arg = ((ValaDataType*) (gee_list_get (GEE_LIST (type_arg_collection), type_arg_it)));
				{
					vala_member_access_add_type_argument (expr, type_arg);
					(type_arg == NULL ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL)));
				}
			}
		}
	}
	_tmp3 = NULL;
	return (_tmp3 = VALA_EXPRESSION (expr), (id = (g_free (id), NULL)), (type_arg_list == NULL ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL))), _tmp3);
}


static ValaExpression* vala_parser_parse_tuple (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	GeeArrayList* expr_list;
	ValaSourceReference* _tmp3;
	ValaExpression* _tmp2;
	ValaExpression* _tmp4;
	ValaExpression* _tmp5;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	expr_list = gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
		do {
			ValaExpression* _tmp0;
			_tmp0 = NULL;
			gee_collection_add (GEE_COLLECTION (expr_list), (_tmp0 = vala_parser_parse_expression (self, &inner_error)));
			(_tmp0 == NULL ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(expr_list == NULL ? NULL : (expr_list = (gee_collection_object_unref (expr_list), NULL)));
		return NULL;
	}
	if (gee_collection_get_size (GEE_COLLECTION (expr_list)) != 1) {
		ValaTuple* tuple;
		ValaExpression* _tmp1;
		tuple = vala_tuple_new ();
		{
			GeeArrayList* expr_collection;
			int expr_it;
			expr_collection = expr_list;
			for (expr_it = 0; expr_it < gee_collection_get_size (GEE_COLLECTION (expr_collection)); expr_it = expr_it + 1) {
				ValaExpression* expr;
				expr = ((ValaExpression*) (gee_list_get (GEE_LIST (expr_collection), expr_it)));
				{
					vala_tuple_add_expression (tuple, expr);
					(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
				}
			}
		}
		_tmp1 = NULL;
		return (_tmp1 = VALA_EXPRESSION (tuple), (expr_list == NULL ? NULL : (expr_list = (gee_collection_object_unref (expr_list), NULL))), _tmp1);
	}
	_tmp3 = NULL;
	_tmp2 = NULL;
	_tmp4 = NULL;
	_tmp5 = NULL;
	return (_tmp5 = (_tmp4 = VALA_EXPRESSION (vala_parenthesized_expression_new ((_tmp2 = ((ValaExpression*) (gee_list_get (GEE_LIST (expr_list), 0)))), (_tmp3 = vala_parser_get_src (self, &begin)))), (_tmp3 == NULL ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL))), (_tmp2 == NULL ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL))), _tmp4), (expr_list == NULL ? NULL : (expr_list = (gee_collection_object_unref (expr_list), NULL))), _tmp5);
}


static ValaExpression* vala_parser_parse_member_access (ValaParser* self, ValaSourceLocation* begin, ValaExpression* inner, GError** error) {
	GError * inner_error;
	char* id;
	GeeList* type_arg_list;
	ValaSourceReference* _tmp0;
	ValaMemberAccess* _tmp1;
	ValaMemberAccess* expr;
	ValaExpression* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (inner), NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_DOT, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	type_arg_list = vala_parser_parse_type_argument_list (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		id = (g_free (id), NULL);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	expr = (_tmp1 = vala_member_access_new (inner, id, (_tmp0 = vala_parser_get_src (self, &(*begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	if (type_arg_list != NULL) {
		{
			GeeList* type_arg_collection;
			int type_arg_it;
			type_arg_collection = type_arg_list;
			for (type_arg_it = 0; type_arg_it < gee_collection_get_size (GEE_COLLECTION (type_arg_collection)); type_arg_it = type_arg_it + 1) {
				ValaDataType* type_arg;
				type_arg = ((ValaDataType*) (gee_list_get (GEE_LIST (type_arg_collection), type_arg_it)));
				{
					vala_member_access_add_type_argument (expr, type_arg);
					(type_arg == NULL ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL)));
				}
			}
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = VALA_EXPRESSION (expr), (id = (g_free (id), NULL)), (type_arg_list == NULL ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL))), _tmp2);
}


static ValaExpression* vala_parser_parse_pointer_member_access (ValaParser* self, ValaSourceLocation* begin, ValaExpression* inner, GError** error) {
	GError * inner_error;
	char* id;
	GeeList* type_arg_list;
	ValaSourceReference* _tmp0;
	ValaMemberAccess* _tmp1;
	ValaMemberAccess* expr;
	ValaExpression* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (inner), NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_OP_PTR, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	type_arg_list = vala_parser_parse_type_argument_list (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		id = (g_free (id), NULL);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	expr = (_tmp1 = vala_member_access_new_pointer (inner, id, (_tmp0 = vala_parser_get_src (self, &(*begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	if (type_arg_list != NULL) {
		{
			GeeList* type_arg_collection;
			int type_arg_it;
			type_arg_collection = type_arg_list;
			for (type_arg_it = 0; type_arg_it < gee_collection_get_size (GEE_COLLECTION (type_arg_collection)); type_arg_it = type_arg_it + 1) {
				ValaDataType* type_arg;
				type_arg = ((ValaDataType*) (gee_list_get (GEE_LIST (type_arg_collection), type_arg_it)));
				{
					vala_member_access_add_type_argument (expr, type_arg);
					(type_arg == NULL ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL)));
				}
			}
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = VALA_EXPRESSION (expr), (id = (g_free (id), NULL)), (type_arg_list == NULL ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL))), _tmp2);
}


static ValaExpression* vala_parser_parse_invocation_expression (ValaParser* self, ValaSourceLocation* begin, ValaExpression* inner, GError** error) {
	GError * inner_error;
	GeeList* arg_list;
	GeeList* init_list;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (inner), NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	arg_list = vala_parser_parse_argument_list (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(arg_list == NULL ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL)));
		return NULL;
	}
	init_list = vala_parser_parse_object_initializer (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(arg_list == NULL ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL)));
		return NULL;
	}
	if (gee_collection_get_size (GEE_COLLECTION (init_list)) > 0 && VALA_IS_MEMBER_ACCESS (inner)) {
		ValaMemberAccess* _tmp0;
		ValaMemberAccess* member;
		ValaSourceReference* _tmp1;
		ValaObjectCreationExpression* _tmp2;
		ValaObjectCreationExpression* expr;
		ValaExpression* _tmp3;
		/* struct creation expression*/
		_tmp0 = NULL;
		member = (_tmp0 = VALA_MEMBER_ACCESS (inner), (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0)));
		vala_member_access_set_creation_member (member, TRUE);
		_tmp1 = NULL;
		_tmp2 = NULL;
		expr = (_tmp2 = vala_object_creation_expression_new (member, (_tmp1 = vala_parser_get_src (self, &(*begin)))), (_tmp1 == NULL ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL))), _tmp2);
		vala_object_creation_expression_set_struct_creation (expr, TRUE);
		{
			GeeList* arg_collection;
			int arg_it;
			arg_collection = arg_list;
			for (arg_it = 0; arg_it < gee_collection_get_size (GEE_COLLECTION (arg_collection)); arg_it = arg_it + 1) {
				ValaExpression* arg;
				arg = ((ValaExpression*) (gee_list_get (GEE_LIST (arg_collection), arg_it)));
				{
					vala_object_creation_expression_add_argument (expr, arg);
					(arg == NULL ? NULL : (arg = (vala_code_node_unref (arg), NULL)));
				}
			}
		}
		{
			GeeList* initializer_collection;
			int initializer_it;
			initializer_collection = init_list;
			for (initializer_it = 0; initializer_it < gee_collection_get_size (GEE_COLLECTION (initializer_collection)); initializer_it = initializer_it + 1) {
				ValaMemberInitializer* initializer;
				initializer = ((ValaMemberInitializer*) (gee_list_get (GEE_LIST (initializer_collection), initializer_it)));
				{
					vala_object_creation_expression_add_member_initializer (expr, initializer);
					(initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)));
				}
			}
		}
		_tmp3 = NULL;
		return (_tmp3 = VALA_EXPRESSION (expr), (member == NULL ? NULL : (member = (vala_code_node_unref (member), NULL))), (arg_list == NULL ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL))), (init_list == NULL ? NULL : (init_list = (gee_collection_object_unref (init_list), NULL))), _tmp3);
	} else {
		ValaSourceReference* _tmp4;
		ValaInvocationExpression* _tmp5;
		ValaInvocationExpression* expr;
		ValaExpression* _tmp6;
		_tmp4 = NULL;
		_tmp5 = NULL;
		expr = (_tmp5 = vala_invocation_expression_new (inner, (_tmp4 = vala_parser_get_src (self, &(*begin)))), (_tmp4 == NULL ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL))), _tmp5);
		{
			GeeList* arg_collection;
			int arg_it;
			arg_collection = arg_list;
			for (arg_it = 0; arg_it < gee_collection_get_size (GEE_COLLECTION (arg_collection)); arg_it = arg_it + 1) {
				ValaExpression* arg;
				arg = ((ValaExpression*) (gee_list_get (GEE_LIST (arg_collection), arg_it)));
				{
					vala_invocation_expression_add_argument (expr, arg);
					(arg == NULL ? NULL : (arg = (vala_code_node_unref (arg), NULL)));
				}
			}
		}
		_tmp6 = NULL;
		return (_tmp6 = VALA_EXPRESSION (expr), (arg_list == NULL ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL))), (init_list == NULL ? NULL : (init_list = (gee_collection_object_unref (init_list), NULL))), _tmp6);
	}
	(arg_list == NULL ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL)));
	(init_list == NULL ? NULL : (init_list = (gee_collection_object_unref (init_list), NULL)));
}


static ValaExpression* vala_parser_parse_element_access (ValaParser* self, ValaSourceLocation* begin, ValaExpression* inner, GError** error) {
	GError * inner_error;
	GeeList* index_list;
	ValaSourceReference* _tmp0;
	ValaElementAccess* _tmp1;
	ValaElementAccess* expr;
	ValaExpression* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (inner), NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACKET, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	index_list = vala_parser_parse_expression_list (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACKET, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(index_list == NULL ? NULL : (index_list = (gee_collection_object_unref (index_list), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	expr = (_tmp1 = vala_element_access_new (inner, (_tmp0 = vala_parser_get_src (self, &(*begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	{
		GeeList* index_collection;
		int index_it;
		index_collection = index_list;
		for (index_it = 0; index_it < gee_collection_get_size (GEE_COLLECTION (index_collection)); index_it = index_it + 1) {
			ValaExpression* index;
			index = ((ValaExpression*) (gee_list_get (GEE_LIST (index_collection), index_it)));
			{
				vala_element_access_append_index (expr, index);
				(index == NULL ? NULL : (index = (vala_code_node_unref (index), NULL)));
			}
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = VALA_EXPRESSION (expr), (index_list == NULL ? NULL : (index_list = (gee_collection_object_unref (index_list), NULL))), _tmp2);
}


static GeeList* vala_parser_parse_expression_list (ValaParser* self, GError** error) {
	GError * inner_error;
	GeeArrayList* list;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	list = gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	do {
		ValaExpression* _tmp0;
		_tmp0 = NULL;
		gee_collection_add (GEE_COLLECTION (list), (_tmp0 = vala_parser_parse_expression (self, &inner_error)));
		(_tmp0 == NULL ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)));
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	return GEE_LIST (list);
}


static ValaExpression* vala_parser_parse_this_access (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSourceReference* _tmp0;
	ValaExpression* _tmp1;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_THIS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = VALA_EXPRESSION (vala_member_access_new (NULL, "this", (_tmp0 = vala_parser_get_src (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
}


static ValaExpression* vala_parser_parse_base_access (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSourceReference* _tmp0;
	ValaExpression* _tmp1;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_BASE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = VALA_EXPRESSION (vala_base_access_new ((_tmp0 = vala_parser_get_src (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
}


static ValaExpression* vala_parser_parse_post_increment_expression (ValaParser* self, ValaSourceLocation* begin, ValaExpression* inner, GError** error) {
	GError * inner_error;
	ValaSourceReference* _tmp0;
	ValaExpression* _tmp1;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (inner), NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_OP_INC, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = VALA_EXPRESSION (vala_postfix_expression_new (inner, TRUE, (_tmp0 = vala_parser_get_src (self, &(*begin))))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
}


static ValaExpression* vala_parser_parse_post_decrement_expression (ValaParser* self, ValaSourceLocation* begin, ValaExpression* inner, GError** error) {
	GError * inner_error;
	ValaSourceReference* _tmp0;
	ValaExpression* _tmp1;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (inner), NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_OP_DEC, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = VALA_EXPRESSION (vala_postfix_expression_new (inner, FALSE, (_tmp0 = vala_parser_get_src (self, &(*begin))))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
}


static ValaExpression* vala_parser_parse_object_or_array_creation_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaMemberAccess* member;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_NEW, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	member = vala_parser_parse_member_name (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_PARENS)) {
		ValaExpression* expr;
		ValaExpression* _tmp0;
		expr = vala_parser_parse_object_creation_expression (self, &begin, member, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(member == NULL ? NULL : (member = (vala_code_node_unref (member), NULL)));
			return NULL;
		}
		_tmp0 = NULL;
		return (_tmp0 = expr, (member == NULL ? NULL : (member = (vala_code_node_unref (member), NULL))), _tmp0);
	} else {
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_BRACKET)) {
			ValaExpression* expr;
			ValaExpression* _tmp1;
			expr = vala_parser_parse_array_creation_expression (self, &begin, member, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(member == NULL ? NULL : (member = (vala_code_node_unref (member), NULL)));
				return NULL;
			}
			_tmp1 = NULL;
			return (_tmp1 = expr, (member == NULL ? NULL : (member = (vala_code_node_unref (member), NULL))), _tmp1);
		} else {
			char* _tmp2;
			GError* _tmp3;
			_tmp2 = NULL;
			_tmp3 = NULL;
			inner_error = (_tmp3 = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, (_tmp2 = vala_parser_get_error (self, "expected ( or ["))), (_tmp2 = (g_free (_tmp2), NULL)), _tmp3);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(member == NULL ? NULL : (member = (vala_code_node_unref (member), NULL)));
				return NULL;
			}
		}
	}
	(member == NULL ? NULL : (member = (vala_code_node_unref (member), NULL)));
}


static ValaExpression* vala_parser_parse_object_creation_expression (ValaParser* self, ValaSourceLocation* begin, ValaMemberAccess* member, GError** error) {
	GError * inner_error;
	GeeList* arg_list;
	GeeList* init_list;
	ValaSourceReference* _tmp0;
	ValaObjectCreationExpression* _tmp1;
	ValaObjectCreationExpression* expr;
	ValaExpression* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (member), NULL);
	inner_error = NULL;
	vala_member_access_set_creation_member (member, TRUE);
	arg_list = vala_parser_parse_argument_list (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(arg_list == NULL ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL)));
		return NULL;
	}
	init_list = vala_parser_parse_object_initializer (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(arg_list == NULL ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	expr = (_tmp1 = vala_object_creation_expression_new (member, (_tmp0 = vala_parser_get_src (self, &(*begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	{
		GeeList* arg_collection;
		int arg_it;
		arg_collection = arg_list;
		for (arg_it = 0; arg_it < gee_collection_get_size (GEE_COLLECTION (arg_collection)); arg_it = arg_it + 1) {
			ValaExpression* arg;
			arg = ((ValaExpression*) (gee_list_get (GEE_LIST (arg_collection), arg_it)));
			{
				vala_object_creation_expression_add_argument (expr, arg);
				(arg == NULL ? NULL : (arg = (vala_code_node_unref (arg), NULL)));
			}
		}
	}
	{
		GeeList* initializer_collection;
		int initializer_it;
		initializer_collection = init_list;
		for (initializer_it = 0; initializer_it < gee_collection_get_size (GEE_COLLECTION (initializer_collection)); initializer_it = initializer_it + 1) {
			ValaMemberInitializer* initializer;
			initializer = ((ValaMemberInitializer*) (gee_list_get (GEE_LIST (initializer_collection), initializer_it)));
			{
				vala_object_creation_expression_add_member_initializer (expr, initializer);
				(initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)));
			}
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = VALA_EXPRESSION (expr), (arg_list == NULL ? NULL : (arg_list = (gee_collection_object_unref (arg_list), NULL))), (init_list == NULL ? NULL : (init_list = (gee_collection_object_unref (init_list), NULL))), _tmp2);
}


static ValaExpression* vala_parser_parse_array_creation_expression (ValaParser* self, ValaSourceLocation* begin, ValaMemberAccess* member, GError** error) {
	GError * inner_error;
	gboolean size_specified;
	GeeList* size_specifier_list;
	gboolean first;
	ValaDataType* element_type;
	ValaInitializerList* initializer;
	ValaSourceReference* _tmp4;
	ValaArrayCreationExpression* _tmp5;
	ValaArrayCreationExpression* expr;
	ValaExpression* _tmp6;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (member), NULL);
	inner_error = NULL;
	size_specified = FALSE;
	size_specifier_list = NULL;
	first = TRUE;
	element_type = VALA_DATA_TYPE (vala_unresolved_type_new_from_expression (VALA_EXPRESSION (member)));
	do {
		GeeList* _tmp1;
		if (!first) {
			ValaDataType* _tmp0;
			/* array of arrays: new T[][42]*/
			_tmp0 = NULL;
			element_type = (_tmp0 = VALA_DATA_TYPE (vala_array_type_new (element_type, gee_collection_get_size (GEE_COLLECTION (size_specifier_list)), vala_code_node_get_source_reference (VALA_CODE_NODE (element_type)))), (element_type == NULL ? NULL : (element_type = (vala_code_node_unref (element_type), NULL))), _tmp0);
		} else {
			first = FALSE;
		}
		_tmp1 = NULL;
		size_specifier_list = (_tmp1 = GEE_LIST (gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)), (size_specifier_list == NULL ? NULL : (size_specifier_list = (gee_collection_object_unref (size_specifier_list), NULL))), _tmp1);
		do {
			ValaExpression* size;
			size = NULL;
			if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACKET && vala_parser_current (self) != VALA_TOKEN_TYPE_COMMA) {
				ValaExpression* _tmp2;
				_tmp2 = NULL;
				size = (_tmp2 = vala_parser_parse_expression (self, &inner_error), (size == NULL ? NULL : (size = (vala_code_node_unref (size), NULL))), _tmp2);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(size == NULL ? NULL : (size = (vala_code_node_unref (size), NULL)));
					(size_specifier_list == NULL ? NULL : (size_specifier_list = (gee_collection_object_unref (size_specifier_list), NULL)));
					(element_type == NULL ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)));
					return NULL;
				}
				size_specified = TRUE;
			}
			gee_collection_add (GEE_COLLECTION (size_specifier_list), size);
			(size == NULL ? NULL : (size = (vala_code_node_unref (size), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACKET, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(size_specifier_list == NULL ? NULL : (size_specifier_list = (gee_collection_object_unref (size_specifier_list), NULL)));
			(element_type == NULL ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)));
			return NULL;
		}
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_BRACKET));
	initializer = NULL;
	if (vala_parser_current (self) == VALA_TOKEN_TYPE_OPEN_BRACE) {
		ValaInitializerList* _tmp3;
		_tmp3 = NULL;
		initializer = (_tmp3 = vala_parser_parse_initializer (self, &inner_error), (initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL))), _tmp3);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(size_specifier_list == NULL ? NULL : (size_specifier_list = (gee_collection_object_unref (size_specifier_list), NULL)));
			(element_type == NULL ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)));
			(initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)));
			return NULL;
		}
	}
	_tmp4 = NULL;
	_tmp5 = NULL;
	expr = (_tmp5 = vala_array_creation_expression_new (element_type, gee_collection_get_size (GEE_COLLECTION (size_specifier_list)), initializer, (_tmp4 = vala_parser_get_src (self, &(*begin)))), (_tmp4 == NULL ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL))), _tmp5);
	if (size_specified) {
		{
			GeeList* size_collection;
			int size_it;
			size_collection = size_specifier_list;
			for (size_it = 0; size_it < gee_collection_get_size (GEE_COLLECTION (size_collection)); size_it = size_it + 1) {
				ValaExpression* size;
				size = ((ValaExpression*) (gee_list_get (GEE_LIST (size_collection), size_it)));
				{
					vala_array_creation_expression_append_size (expr, size);
					(size == NULL ? NULL : (size = (vala_code_node_unref (size), NULL)));
				}
			}
		}
	}
	_tmp6 = NULL;
	return (_tmp6 = VALA_EXPRESSION (expr), (size_specifier_list == NULL ? NULL : (size_specifier_list = (gee_collection_object_unref (size_specifier_list), NULL))), (element_type == NULL ? NULL : (element_type = (vala_code_node_unref (element_type), NULL))), (initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL))), _tmp6);
}


static GeeList* vala_parser_parse_object_initializer (ValaParser* self, GError** error) {
	GError * inner_error;
	GeeArrayList* list;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	list = gee_array_list_new (VALA_TYPE_MEMBER_INITIALIZER, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_BRACE)) {
		do {
			ValaMemberInitializer* _tmp0;
			_tmp0 = NULL;
			gee_collection_add (GEE_COLLECTION (list), (_tmp0 = vala_parser_parse_member_initializer (self, &inner_error)));
			(_tmp0 == NULL ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACE, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(list == NULL ? NULL : (list = (gee_collection_object_unref (list), NULL)));
			return NULL;
		}
	}
	return GEE_LIST (list);
}


static ValaMemberInitializer* vala_parser_parse_member_initializer (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	char* id;
	ValaExpression* expr;
	ValaSourceReference* _tmp0;
	ValaMemberInitializer* _tmp1;
	ValaMemberInitializer* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_ASSIGN, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		id = (g_free (id), NULL);
		return NULL;
	}
	expr = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		id = (g_free (id), NULL);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = vala_member_initializer_new (id, expr, (_tmp0 = vala_parser_get_src (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1), (id = (g_free (id), NULL)), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp2);
}


static ValaExpression* vala_parser_parse_sizeof_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaDataType* type;
	ValaSourceReference* _tmp0;
	ValaExpression* _tmp1;
	ValaExpression* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_SIZEOF, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	type = vala_parser_parse_type (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = VALA_EXPRESSION (vala_sizeof_expression_new (type, (_tmp0 = vala_parser_get_src (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp2);
}


static ValaExpression* vala_parser_parse_typeof_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaDataType* type;
	ValaSourceReference* _tmp0;
	ValaExpression* _tmp1;
	ValaExpression* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_TYPEOF, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	type = vala_parser_parse_type (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = VALA_EXPRESSION (vala_typeof_expression_new (type, (_tmp0 = vala_parser_get_src (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp2);
}


static ValaUnaryOperator vala_parser_get_unary_operator (ValaParser* self, ValaTokenType token_type) {
	ValaTokenType _tmp9;
	g_return_val_if_fail (VALA_IS_PARSER (self), 0);
	_tmp9 = token_type;
	if (_tmp9 == VALA_TOKEN_TYPE_PLUS)
	do {
		return VALA_UNARY_OPERATOR_PLUS;
	} while (0); else if (_tmp9 == VALA_TOKEN_TYPE_MINUS)
	do {
		return VALA_UNARY_OPERATOR_MINUS;
	} while (0); else if (_tmp9 == VALA_TOKEN_TYPE_OP_NEG)
	do {
		return VALA_UNARY_OPERATOR_LOGICAL_NEGATION;
	} while (0); else if (_tmp9 == VALA_TOKEN_TYPE_TILDE)
	do {
		return VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT;
	} while (0); else if (_tmp9 == VALA_TOKEN_TYPE_OP_INC)
	do {
		return VALA_UNARY_OPERATOR_INCREMENT;
	} while (0); else if (_tmp9 == VALA_TOKEN_TYPE_OP_DEC)
	do {
		return VALA_UNARY_OPERATOR_DECREMENT;
	} while (0); else if (_tmp9 == VALA_TOKEN_TYPE_REF)
	do {
		return VALA_UNARY_OPERATOR_REF;
	} while (0); else if (_tmp9 == VALA_TOKEN_TYPE_OUT)
	do {
		return VALA_UNARY_OPERATOR_OUT;
	} while (0); else
	do {
		return VALA_UNARY_OPERATOR_NONE;
	} while (0);
}


static ValaExpression* vala_parser_parse_unary_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaUnaryOperator operator;
	ValaTokenType _tmp18;
	ValaExpression* expr;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	operator = vala_parser_get_unary_operator (self, vala_parser_current (self));
	if (operator != VALA_UNARY_OPERATOR_NONE) {
		ValaExpression* op;
		ValaSourceReference* _tmp0;
		ValaExpression* _tmp1;
		ValaExpression* _tmp2;
		vala_parser_next (self);
		op = vala_parser_parse_unary_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			return NULL;
		}
		_tmp0 = NULL;
		_tmp1 = NULL;
		_tmp2 = NULL;
		return (_tmp2 = (_tmp1 = VALA_EXPRESSION (vala_unary_expression_new (operator, op, (_tmp0 = vala_parser_get_src (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1), (op == NULL ? NULL : (op = (vala_code_node_unref (op), NULL))), _tmp2);
	}
	_tmp18 = vala_parser_current (self);
	if (_tmp18 == VALA_TOKEN_TYPE_HASH)
	do {
		ValaExpression* op;
		ValaSourceReference* _tmp3;
		ValaExpression* _tmp4;
		ValaExpression* _tmp5;
		vala_parser_next (self);
		op = vala_parser_parse_unary_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			return NULL;
		}
		_tmp3 = NULL;
		_tmp4 = NULL;
		_tmp5 = NULL;
		return (_tmp5 = (_tmp4 = VALA_EXPRESSION (vala_reference_transfer_expression_new (op, (_tmp3 = vala_parser_get_src (self, &begin)))), (_tmp3 == NULL ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL))), _tmp4), (op == NULL ? NULL : (op = (vala_code_node_unref (op), NULL))), _tmp5);
	} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_OPEN_PARENS)
	do {
		ValaTokenType _tmp11;
		vala_parser_next (self);
		_tmp11 = vala_parser_current (self);
		if (_tmp11 == VALA_TOKEN_TYPE_VOID || _tmp11 == VALA_TOKEN_TYPE_DYNAMIC || _tmp11 == VALA_TOKEN_TYPE_WEAK || _tmp11 == VALA_TOKEN_TYPE_IDENTIFIER)
		do {
			ValaDataType* type;
			type = vala_parser_parse_type (self, TRUE, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				return NULL;
			}
			if (vala_parser_accept (self, VALA_TOKEN_TYPE_CLOSE_PARENS)) {
				ValaTokenType _tmp10;
				/* check follower to decide whether to create cast expression*/
				_tmp10 = vala_parser_current (self);
				if (_tmp10 == VALA_TOKEN_TYPE_OP_NEG || _tmp10 == VALA_TOKEN_TYPE_TILDE || _tmp10 == VALA_TOKEN_TYPE_OPEN_PARENS || _tmp10 == VALA_TOKEN_TYPE_TRUE || _tmp10 == VALA_TOKEN_TYPE_FALSE || _tmp10 == VALA_TOKEN_TYPE_INTEGER_LITERAL || _tmp10 == VALA_TOKEN_TYPE_REAL_LITERAL || _tmp10 == VALA_TOKEN_TYPE_CHARACTER_LITERAL || _tmp10 == VALA_TOKEN_TYPE_STRING_LITERAL || _tmp10 == VALA_TOKEN_TYPE_VERBATIM_STRING_LITERAL || _tmp10 == VALA_TOKEN_TYPE_NULL || _tmp10 == VALA_TOKEN_TYPE_THIS || _tmp10 == VALA_TOKEN_TYPE_BASE || _tmp10 == VALA_TOKEN_TYPE_NEW || _tmp10 == VALA_TOKEN_TYPE_SIZEOF || _tmp10 == VALA_TOKEN_TYPE_TYPEOF || _tmp10 == VALA_TOKEN_TYPE_IDENTIFIER)
				do {
					ValaExpression* inner;
					ValaSourceReference* _tmp7;
					ValaExpression* _tmp8;
					ValaExpression* _tmp9;
					if (!(VALA_IS_POINTER_TYPE (type)) && !vala_data_type_get_value_owned (type)) {
						ValaSourceReference* _tmp6;
						_tmp6 = NULL;
						vala_report_warning ((_tmp6 = vala_parser_get_src (self, &begin)), "obsolete syntax, weak type modifier unused in cast expressions");
						(_tmp6 == NULL ? NULL : (_tmp6 = (vala_source_reference_unref (_tmp6), NULL)));
					}
					inner = vala_parser_parse_unary_expression (self, &inner_error);
					if (inner_error != NULL) {
						g_propagate_error (error, inner_error);
						(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
						return NULL;
					}
					_tmp7 = NULL;
					_tmp8 = NULL;
					_tmp9 = NULL;
					return (_tmp9 = (_tmp8 = VALA_EXPRESSION (vala_cast_expression_new (inner, type, (_tmp7 = vala_parser_get_src (self, &begin)), FALSE)), (_tmp7 == NULL ? NULL : (_tmp7 = (vala_source_reference_unref (_tmp7), NULL))), _tmp8), (inner == NULL ? NULL : (inner = (vala_code_node_unref (inner), NULL))), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp9);
				} while (0);
			}
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			break;
		} while (0);
		vala_parser_rollback (self, &begin);
		break;
	} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_STAR)
	do {
		ValaExpression* op;
		ValaSourceReference* _tmp12;
		ValaExpression* _tmp13;
		ValaExpression* _tmp14;
		vala_parser_next (self);
		op = vala_parser_parse_unary_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			return NULL;
		}
		_tmp12 = NULL;
		_tmp13 = NULL;
		_tmp14 = NULL;
		return (_tmp14 = (_tmp13 = VALA_EXPRESSION (vala_pointer_indirection_new (op, (_tmp12 = vala_parser_get_src (self, &begin)))), (_tmp12 == NULL ? NULL : (_tmp12 = (vala_source_reference_unref (_tmp12), NULL))), _tmp13), (op == NULL ? NULL : (op = (vala_code_node_unref (op), NULL))), _tmp14);
	} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_BITWISE_AND)
	do {
		ValaExpression* op;
		ValaSourceReference* _tmp15;
		ValaExpression* _tmp16;
		ValaExpression* _tmp17;
		vala_parser_next (self);
		op = vala_parser_parse_unary_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			return NULL;
		}
		_tmp15 = NULL;
		_tmp16 = NULL;
		_tmp17 = NULL;
		return (_tmp17 = (_tmp16 = VALA_EXPRESSION (vala_addressof_expression_new (op, (_tmp15 = vala_parser_get_src (self, &begin)))), (_tmp15 == NULL ? NULL : (_tmp15 = (vala_source_reference_unref (_tmp15), NULL))), _tmp16), (op == NULL ? NULL : (op = (vala_code_node_unref (op), NULL))), _tmp17);
	} while (0);
	expr = vala_parser_parse_primary_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	return expr;
}


static ValaBinaryOperator vala_parser_get_binary_operator (ValaParser* self, ValaTokenType token_type) {
	ValaTokenType _tmp12;
	g_return_val_if_fail (VALA_IS_PARSER (self), 0);
	_tmp12 = token_type;
	if (_tmp12 == VALA_TOKEN_TYPE_STAR)
	do {
		return VALA_BINARY_OPERATOR_MUL;
	} while (0); else if (_tmp12 == VALA_TOKEN_TYPE_DIV)
	do {
		return VALA_BINARY_OPERATOR_DIV;
	} while (0); else if (_tmp12 == VALA_TOKEN_TYPE_PERCENT)
	do {
		return VALA_BINARY_OPERATOR_MOD;
	} while (0); else if (_tmp12 == VALA_TOKEN_TYPE_PLUS)
	do {
		return VALA_BINARY_OPERATOR_PLUS;
	} while (0); else if (_tmp12 == VALA_TOKEN_TYPE_MINUS)
	do {
		return VALA_BINARY_OPERATOR_MINUS;
	} while (0); else if (_tmp12 == VALA_TOKEN_TYPE_OP_LT)
	do {
		return VALA_BINARY_OPERATOR_LESS_THAN;
	} while (0); else if (_tmp12 == VALA_TOKEN_TYPE_OP_GT)
	do {
		return VALA_BINARY_OPERATOR_GREATER_THAN;
	} while (0); else if (_tmp12 == VALA_TOKEN_TYPE_OP_LE)
	do {
		return VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL;
	} while (0); else if (_tmp12 == VALA_TOKEN_TYPE_OP_GE)
	do {
		return VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL;
	} while (0); else if (_tmp12 == VALA_TOKEN_TYPE_OP_EQ)
	do {
		return VALA_BINARY_OPERATOR_EQUALITY;
	} while (0); else if (_tmp12 == VALA_TOKEN_TYPE_OP_NE)
	do {
		return VALA_BINARY_OPERATOR_INEQUALITY;
	} while (0); else
	do {
		return VALA_BINARY_OPERATOR_NONE;
	} while (0);
}


static ValaExpression* vala_parser_parse_multiplicative_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	gboolean found;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_unary_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	found = TRUE;
	while (found) {
		ValaBinaryOperator operator;
		ValaBinaryOperator _tmp2;
		operator = vala_parser_get_binary_operator (self, vala_parser_current (self));
		_tmp2 = operator;
		if (_tmp2 == VALA_BINARY_OPERATOR_MUL || _tmp2 == VALA_BINARY_OPERATOR_DIV || _tmp2 == VALA_BINARY_OPERATOR_MOD)
		do {
			ValaExpression* right;
			ValaExpression* _tmp1;
			ValaSourceReference* _tmp0;
			vala_parser_next (self);
			right = vala_parser_parse_unary_expression (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
				return NULL;
			}
			_tmp1 = NULL;
			_tmp0 = NULL;
			left = (_tmp1 = VALA_EXPRESSION (vala_binary_expression_new (operator, left, right, (_tmp0 = vala_parser_get_src (self, &begin)))), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp1);
			(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
			(right == NULL ? NULL : (right = (vala_code_node_unref (right), NULL)));
			break;
		} while (0); else
		do {
			found = FALSE;
			break;
		} while (0);
	}
	return left;
}


static ValaExpression* vala_parser_parse_additive_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	gboolean found;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_multiplicative_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	found = TRUE;
	while (found) {
		ValaBinaryOperator operator;
		ValaBinaryOperator _tmp2;
		operator = vala_parser_get_binary_operator (self, vala_parser_current (self));
		_tmp2 = operator;
		if (_tmp2 == VALA_BINARY_OPERATOR_PLUS || _tmp2 == VALA_BINARY_OPERATOR_MINUS)
		do {
			ValaExpression* right;
			ValaExpression* _tmp1;
			ValaSourceReference* _tmp0;
			vala_parser_next (self);
			right = vala_parser_parse_multiplicative_expression (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
				return NULL;
			}
			_tmp1 = NULL;
			_tmp0 = NULL;
			left = (_tmp1 = VALA_EXPRESSION (vala_binary_expression_new (operator, left, right, (_tmp0 = vala_parser_get_src (self, &begin)))), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp1);
			(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
			(right == NULL ? NULL : (right = (vala_code_node_unref (right), NULL)));
			break;
		} while (0); else
		do {
			found = FALSE;
			break;
		} while (0);
	}
	return left;
}


static ValaExpression* vala_parser_parse_shift_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	gboolean found;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_additive_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	found = TRUE;
	while (found) {
		ValaTokenType _tmp4;
		_tmp4 = vala_parser_current (self);
		if (_tmp4 == VALA_TOKEN_TYPE_OP_SHIFT_LEFT)
		do {
			ValaExpression* right;
			ValaExpression* _tmp1;
			ValaSourceReference* _tmp0;
			vala_parser_next (self);
			right = vala_parser_parse_additive_expression (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
				return NULL;
			}
			_tmp1 = NULL;
			_tmp0 = NULL;
			left = (_tmp1 = VALA_EXPRESSION (vala_binary_expression_new (VALA_BINARY_OPERATOR_SHIFT_LEFT, left, right, (_tmp0 = vala_parser_get_src (self, &begin)))), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp1);
			(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
			(right == NULL ? NULL : (right = (vala_code_node_unref (right), NULL)));
			break;
		} while (0); else if (_tmp4 == VALA_TOKEN_TYPE_OP_GT)
		do {
			gchar* first_gt_pos;
			first_gt_pos = self->priv->tokens[self->priv->index].begin.pos;
			vala_parser_next (self);
			if (vala_parser_current (self) == VALA_TOKEN_TYPE_OP_GT && self->priv->tokens[self->priv->index].begin.pos == first_gt_pos + 1) {
				ValaExpression* right;
				ValaExpression* _tmp3;
				ValaSourceReference* _tmp2;
				vala_parser_next (self);
				right = vala_parser_parse_additive_expression (self, &inner_error);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
					return NULL;
				}
				_tmp3 = NULL;
				_tmp2 = NULL;
				left = (_tmp3 = VALA_EXPRESSION (vala_binary_expression_new (VALA_BINARY_OPERATOR_SHIFT_RIGHT, left, right, (_tmp2 = vala_parser_get_src (self, &begin)))), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp3);
				(_tmp2 == NULL ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)));
				(right == NULL ? NULL : (right = (vala_code_node_unref (right), NULL)));
			} else {
				vala_parser_prev (self);
				found = FALSE;
			}
			break;
		} while (0); else
		do {
			found = FALSE;
			break;
		} while (0);
	}
	return left;
}


static ValaExpression* vala_parser_parse_relational_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	gboolean found;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_shift_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	found = TRUE;
	while (found) {
		ValaBinaryOperator operator;
		ValaBinaryOperator _tmp9;
		operator = vala_parser_get_binary_operator (self, vala_parser_current (self));
		_tmp9 = operator;
		if (_tmp9 == VALA_BINARY_OPERATOR_LESS_THAN || _tmp9 == VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL || _tmp9 == VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL)
		do {
			ValaExpression* right;
			ValaExpression* _tmp1;
			ValaSourceReference* _tmp0;
			vala_parser_next (self);
			right = vala_parser_parse_shift_expression (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
				return NULL;
			}
			_tmp1 = NULL;
			_tmp0 = NULL;
			left = (_tmp1 = VALA_EXPRESSION (vala_binary_expression_new (operator, left, right, (_tmp0 = vala_parser_get_src (self, &begin)))), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp1);
			(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
			(right == NULL ? NULL : (right = (vala_code_node_unref (right), NULL)));
			break;
		} while (0); else if (_tmp9 == VALA_BINARY_OPERATOR_GREATER_THAN)
		do {
			vala_parser_next (self);
			if (vala_parser_current (self) != VALA_TOKEN_TYPE_OP_GT && vala_parser_current (self) != VALA_TOKEN_TYPE_OP_GE) {
				ValaExpression* right;
				ValaExpression* _tmp3;
				ValaSourceReference* _tmp2;
				right = vala_parser_parse_shift_expression (self, &inner_error);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
					return NULL;
				}
				_tmp3 = NULL;
				_tmp2 = NULL;
				left = (_tmp3 = VALA_EXPRESSION (vala_binary_expression_new (operator, left, right, (_tmp2 = vala_parser_get_src (self, &begin)))), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp3);
				(_tmp2 == NULL ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)));
				(right == NULL ? NULL : (right = (vala_code_node_unref (right), NULL)));
			} else {
				vala_parser_prev (self);
				found = FALSE;
			}
			break;
		} while (0); else
		do {
			ValaTokenType _tmp8;
			_tmp8 = vala_parser_current (self);
			if (_tmp8 == VALA_TOKEN_TYPE_IS)
			do {
				ValaDataType* type;
				ValaExpression* _tmp5;
				ValaSourceReference* _tmp4;
				vala_parser_next (self);
				type = vala_parser_parse_type (self, TRUE, &inner_error);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
					return NULL;
				}
				_tmp5 = NULL;
				_tmp4 = NULL;
				left = (_tmp5 = VALA_EXPRESSION (vala_typecheck_new (left, type, (_tmp4 = vala_parser_get_src (self, &begin)))), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp5);
				(_tmp4 == NULL ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL)));
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
				break;
			} while (0); else if (_tmp8 == VALA_TOKEN_TYPE_AS)
			do {
				ValaDataType* type;
				ValaExpression* _tmp7;
				ValaSourceReference* _tmp6;
				vala_parser_next (self);
				type = vala_parser_parse_type (self, TRUE, &inner_error);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
					return NULL;
				}
				_tmp7 = NULL;
				_tmp6 = NULL;
				left = (_tmp7 = VALA_EXPRESSION (vala_cast_expression_new (left, type, (_tmp6 = vala_parser_get_src (self, &begin)), TRUE)), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp7);
				(_tmp6 == NULL ? NULL : (_tmp6 = (vala_source_reference_unref (_tmp6), NULL)));
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
				break;
			} while (0); else
			do {
				found = FALSE;
				break;
			} while (0);
			break;
		} while (0);
	}
	return left;
}


static ValaExpression* vala_parser_parse_equality_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	gboolean found;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_relational_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	found = TRUE;
	while (found) {
		ValaBinaryOperator operator;
		ValaBinaryOperator _tmp2;
		operator = vala_parser_get_binary_operator (self, vala_parser_current (self));
		_tmp2 = operator;
		if (_tmp2 == VALA_BINARY_OPERATOR_EQUALITY || _tmp2 == VALA_BINARY_OPERATOR_INEQUALITY)
		do {
			ValaExpression* right;
			ValaExpression* _tmp1;
			ValaSourceReference* _tmp0;
			vala_parser_next (self);
			right = vala_parser_parse_relational_expression (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
				return NULL;
			}
			_tmp1 = NULL;
			_tmp0 = NULL;
			left = (_tmp1 = VALA_EXPRESSION (vala_binary_expression_new (operator, left, right, (_tmp0 = vala_parser_get_src (self, &begin)))), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp1);
			(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
			(right == NULL ? NULL : (right = (vala_code_node_unref (right), NULL)));
			break;
		} while (0); else
		do {
			found = FALSE;
			break;
		} while (0);
	}
	return left;
}


static ValaExpression* vala_parser_parse_and_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_equality_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_BITWISE_AND)) {
		ValaExpression* right;
		ValaExpression* _tmp1;
		ValaSourceReference* _tmp0;
		right = vala_parser_parse_equality_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
			return NULL;
		}
		_tmp1 = NULL;
		_tmp0 = NULL;
		left = (_tmp1 = VALA_EXPRESSION (vala_binary_expression_new (VALA_BINARY_OPERATOR_BITWISE_AND, left, right, (_tmp0 = vala_parser_get_src (self, &begin)))), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp1);
		(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
		(right == NULL ? NULL : (right = (vala_code_node_unref (right), NULL)));
	}
	return left;
}


static ValaExpression* vala_parser_parse_exclusive_or_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_and_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_CARRET)) {
		ValaExpression* right;
		ValaExpression* _tmp1;
		ValaSourceReference* _tmp0;
		right = vala_parser_parse_and_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
			return NULL;
		}
		_tmp1 = NULL;
		_tmp0 = NULL;
		left = (_tmp1 = VALA_EXPRESSION (vala_binary_expression_new (VALA_BINARY_OPERATOR_BITWISE_XOR, left, right, (_tmp0 = vala_parser_get_src (self, &begin)))), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp1);
		(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
		(right == NULL ? NULL : (right = (vala_code_node_unref (right), NULL)));
	}
	return left;
}


static ValaExpression* vala_parser_parse_inclusive_or_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_exclusive_or_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_BITWISE_OR)) {
		ValaExpression* right;
		ValaExpression* _tmp1;
		ValaSourceReference* _tmp0;
		right = vala_parser_parse_exclusive_or_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
			return NULL;
		}
		_tmp1 = NULL;
		_tmp0 = NULL;
		left = (_tmp1 = VALA_EXPRESSION (vala_binary_expression_new (VALA_BINARY_OPERATOR_BITWISE_OR, left, right, (_tmp0 = vala_parser_get_src (self, &begin)))), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp1);
		(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
		(right == NULL ? NULL : (right = (vala_code_node_unref (right), NULL)));
	}
	return left;
}


static ValaExpression* vala_parser_parse_in_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_inclusive_or_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_IN)) {
		ValaExpression* right;
		ValaExpression* _tmp1;
		ValaSourceReference* _tmp0;
		right = vala_parser_parse_inclusive_or_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
			return NULL;
		}
		_tmp1 = NULL;
		_tmp0 = NULL;
		left = (_tmp1 = VALA_EXPRESSION (vala_binary_expression_new (VALA_BINARY_OPERATOR_IN, left, right, (_tmp0 = vala_parser_get_src (self, &begin)))), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp1);
		(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
		(right == NULL ? NULL : (right = (vala_code_node_unref (right), NULL)));
	}
	return left;
}


static ValaExpression* vala_parser_parse_conditional_and_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_in_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_OP_AND)) {
		ValaExpression* right;
		ValaExpression* _tmp1;
		ValaSourceReference* _tmp0;
		right = vala_parser_parse_in_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
			return NULL;
		}
		_tmp1 = NULL;
		_tmp0 = NULL;
		left = (_tmp1 = VALA_EXPRESSION (vala_binary_expression_new (VALA_BINARY_OPERATOR_AND, left, right, (_tmp0 = vala_parser_get_src (self, &begin)))), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp1);
		(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
		(right == NULL ? NULL : (right = (vala_code_node_unref (right), NULL)));
	}
	return left;
}


static ValaExpression* vala_parser_parse_conditional_or_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* left;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	left = vala_parser_parse_conditional_and_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_OP_OR)) {
		ValaExpression* right;
		ValaExpression* _tmp1;
		ValaSourceReference* _tmp0;
		right = vala_parser_parse_conditional_and_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL)));
			return NULL;
		}
		_tmp1 = NULL;
		_tmp0 = NULL;
		left = (_tmp1 = VALA_EXPRESSION (vala_binary_expression_new (VALA_BINARY_OPERATOR_OR, left, right, (_tmp0 = vala_parser_get_src (self, &begin)))), (left == NULL ? NULL : (left = (vala_code_node_unref (left), NULL))), _tmp1);
		(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
		(right == NULL ? NULL : (right = (vala_code_node_unref (right), NULL)));
	}
	return left;
}


static ValaExpression* vala_parser_parse_conditional_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* condition;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	condition = vala_parser_parse_conditional_or_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_INTERR)) {
		ValaExpression* true_expr;
		ValaExpression* false_expr;
		ValaSourceReference* _tmp0;
		ValaExpression* _tmp1;
		ValaExpression* _tmp2;
		true_expr = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
			return NULL;
		}
		vala_parser_expect (self, VALA_TOKEN_TYPE_COLON, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(true_expr == NULL ? NULL : (true_expr = (vala_code_node_unref (true_expr), NULL)));
			(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
			return NULL;
		}
		false_expr = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(true_expr == NULL ? NULL : (true_expr = (vala_code_node_unref (true_expr), NULL)));
			(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
			return NULL;
		}
		_tmp0 = NULL;
		_tmp1 = NULL;
		_tmp2 = NULL;
		return (_tmp2 = (_tmp1 = VALA_EXPRESSION (vala_conditional_expression_new (condition, true_expr, false_expr, (_tmp0 = vala_parser_get_src (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1), (true_expr == NULL ? NULL : (true_expr = (vala_code_node_unref (true_expr), NULL))), (false_expr == NULL ? NULL : (false_expr = (vala_code_node_unref (false_expr), NULL))), (condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL))), _tmp2);
	} else {
		return condition;
	}
	(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
}


static ValaExpression* vala_parser_parse_lambda_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	GeeList* params;
	ValaLambdaExpression* lambda;
	ValaExpression* _tmp6;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	params = GEE_LIST (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_direct_equal));
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_PARENS)) {
		if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
			do {
				char* _tmp0;
				_tmp0 = NULL;
				gee_collection_add (GEE_COLLECTION (params), (_tmp0 = vala_parser_parse_identifier (self, &inner_error)));
				_tmp0 = (g_free (_tmp0), NULL);
			} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		}
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(params == NULL ? NULL : (params = (gee_collection_object_unref (params), NULL)));
			return NULL;
		}
	} else {
		char* _tmp1;
		_tmp1 = NULL;
		gee_collection_add (GEE_COLLECTION (params), (_tmp1 = vala_parser_parse_identifier (self, &inner_error)));
		_tmp1 = (g_free (_tmp1), NULL);
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_LAMBDA, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(params == NULL ? NULL : (params = (gee_collection_object_unref (params), NULL)));
		return NULL;
	}
	lambda = NULL;
	if (vala_parser_current (self) == VALA_TOKEN_TYPE_OPEN_BRACE) {
		ValaBlock* block;
		ValaLambdaExpression* _tmp3;
		ValaSourceReference* _tmp2;
		block = vala_parser_parse_block (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(params == NULL ? NULL : (params = (gee_collection_object_unref (params), NULL)));
			(lambda == NULL ? NULL : (lambda = (vala_code_node_unref (lambda), NULL)));
			return NULL;
		}
		_tmp3 = NULL;
		_tmp2 = NULL;
		lambda = (_tmp3 = vala_lambda_expression_new_with_statement_body (block, (_tmp2 = vala_parser_get_src (self, &begin))), (lambda == NULL ? NULL : (lambda = (vala_code_node_unref (lambda), NULL))), _tmp3);
		(_tmp2 == NULL ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)));
		(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
	} else {
		ValaExpression* expr;
		ValaLambdaExpression* _tmp5;
		ValaSourceReference* _tmp4;
		expr = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(params == NULL ? NULL : (params = (gee_collection_object_unref (params), NULL)));
			(lambda == NULL ? NULL : (lambda = (vala_code_node_unref (lambda), NULL)));
			return NULL;
		}
		_tmp5 = NULL;
		_tmp4 = NULL;
		lambda = (_tmp5 = vala_lambda_expression_new (expr, (_tmp4 = vala_parser_get_src (self, &begin))), (lambda == NULL ? NULL : (lambda = (vala_code_node_unref (lambda), NULL))), _tmp5);
		(_tmp4 == NULL ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL)));
		(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
	}
	{
		GeeList* param_collection;
		int param_it;
		param_collection = params;
		for (param_it = 0; param_it < gee_collection_get_size (GEE_COLLECTION (param_collection)); param_it = param_it + 1) {
			char* param;
			param = ((char*) (gee_list_get (GEE_LIST (param_collection), param_it)));
			{
				vala_lambda_expression_add_parameter (lambda, param);
				param = (g_free (param), NULL);
			}
		}
	}
	_tmp6 = NULL;
	return (_tmp6 = VALA_EXPRESSION (lambda), (params == NULL ? NULL : (params = (gee_collection_object_unref (params), NULL))), _tmp6);
}


static ValaAssignmentOperator vala_parser_get_assignment_operator (ValaParser* self, ValaTokenType token_type) {
	ValaTokenType _tmp11;
	g_return_val_if_fail (VALA_IS_PARSER (self), 0);
	_tmp11 = token_type;
	if (_tmp11 == VALA_TOKEN_TYPE_ASSIGN)
	do {
		return VALA_ASSIGNMENT_OPERATOR_SIMPLE;
	} while (0); else if (_tmp11 == VALA_TOKEN_TYPE_ASSIGN_ADD)
	do {
		return VALA_ASSIGNMENT_OPERATOR_ADD;
	} while (0); else if (_tmp11 == VALA_TOKEN_TYPE_ASSIGN_SUB)
	do {
		return VALA_ASSIGNMENT_OPERATOR_SUB;
	} while (0); else if (_tmp11 == VALA_TOKEN_TYPE_ASSIGN_BITWISE_OR)
	do {
		return VALA_ASSIGNMENT_OPERATOR_BITWISE_OR;
	} while (0); else if (_tmp11 == VALA_TOKEN_TYPE_ASSIGN_BITWISE_AND)
	do {
		return VALA_ASSIGNMENT_OPERATOR_BITWISE_AND;
	} while (0); else if (_tmp11 == VALA_TOKEN_TYPE_ASSIGN_BITWISE_XOR)
	do {
		return VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR;
	} while (0); else if (_tmp11 == VALA_TOKEN_TYPE_ASSIGN_DIV)
	do {
		return VALA_ASSIGNMENT_OPERATOR_DIV;
	} while (0); else if (_tmp11 == VALA_TOKEN_TYPE_ASSIGN_MUL)
	do {
		return VALA_ASSIGNMENT_OPERATOR_MUL;
	} while (0); else if (_tmp11 == VALA_TOKEN_TYPE_ASSIGN_PERCENT)
	do {
		return VALA_ASSIGNMENT_OPERATOR_PERCENT;
	} while (0); else if (_tmp11 == VALA_TOKEN_TYPE_ASSIGN_SHIFT_LEFT)
	do {
		return VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT;
	} while (0); else
	do {
		return VALA_ASSIGNMENT_OPERATOR_NONE;
	} while (0);
}


static ValaExpression* vala_parser_parse_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	expr = vala_parser_parse_conditional_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	if (vala_parser_current (self) == VALA_TOKEN_TYPE_LAMBDA) {
		ValaExpression* lambda;
		ValaExpression* _tmp0;
		vala_parser_rollback (self, &begin);
		lambda = vala_parser_parse_lambda_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			return NULL;
		}
		_tmp0 = NULL;
		return (_tmp0 = lambda, (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp0);
	}
	while (TRUE) {
		ValaAssignmentOperator operator;
		operator = vala_parser_get_assignment_operator (self, vala_parser_current (self));
		if (operator != VALA_ASSIGNMENT_OPERATOR_NONE) {
			ValaExpression* rhs;
			ValaExpression* _tmp2;
			ValaSourceReference* _tmp1;
			vala_parser_next (self);
			rhs = vala_parser_parse_expression (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
				return NULL;
			}
			_tmp2 = NULL;
			_tmp1 = NULL;
			expr = (_tmp2 = VALA_EXPRESSION (vala_assignment_new (expr, rhs, operator, (_tmp1 = vala_parser_get_src (self, &begin)))), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp2);
			(_tmp1 == NULL ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL)));
			if (expr == NULL) {
				/* workaround for current limitation of exception handling*/
				inner_error = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, "syntax error in assignment");
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(rhs == NULL ? NULL : (rhs = (vala_code_node_unref (rhs), NULL)));
					(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
					return NULL;
				}
			}
			(rhs == NULL ? NULL : (rhs = (vala_code_node_unref (rhs), NULL)));
		} else {
			if (vala_parser_current (self) == VALA_TOKEN_TYPE_OP_GT) {
				gchar* first_gt_pos;
				/* >>=*/
				first_gt_pos = self->priv->tokens[self->priv->index].begin.pos;
				vala_parser_next (self);
				/* only accept >>= when there is no space between the two > signs*/
				if (vala_parser_current (self) == VALA_TOKEN_TYPE_OP_GE && self->priv->tokens[self->priv->index].begin.pos == first_gt_pos + 1) {
					ValaExpression* rhs;
					ValaExpression* _tmp4;
					ValaSourceReference* _tmp3;
					vala_parser_next (self);
					rhs = vala_parser_parse_expression (self, &inner_error);
					if (inner_error != NULL) {
						g_propagate_error (error, inner_error);
						(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
						return NULL;
					}
					_tmp4 = NULL;
					_tmp3 = NULL;
					expr = (_tmp4 = VALA_EXPRESSION (vala_assignment_new (expr, rhs, VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT, (_tmp3 = vala_parser_get_src (self, &begin)))), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp4);
					(_tmp3 == NULL ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL)));
					if (expr == NULL) {
						/* workaround for current limitation of exception handling*/
						inner_error = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, "syntax error in assignment");
						if (inner_error != NULL) {
							g_propagate_error (error, inner_error);
							(rhs == NULL ? NULL : (rhs = (vala_code_node_unref (rhs), NULL)));
							(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
							return NULL;
						}
					}
					(rhs == NULL ? NULL : (rhs = (vala_code_node_unref (rhs), NULL)));
				} else {
					vala_parser_prev (self);
					break;
				}
			} else {
				break;
			}
		}
	}
	return expr;
}


static void vala_parser_parse_statements (ValaParser* self, ValaBlock* block, GError** error) {
	GError * inner_error;
	g_return_if_fail (VALA_IS_PARSER (self));
	g_return_if_fail (VALA_IS_BLOCK (block));
	inner_error = NULL;
	while (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACE && vala_parser_current (self) != VALA_TOKEN_TYPE_CASE && vala_parser_current (self) != VALA_TOKEN_TYPE_DEFAULT) {
		{
			ValaStatement* stmt;
			gboolean is_decl;
			char* _tmp0;
			ValaTokenType _tmp18;
			stmt = NULL;
			is_decl = FALSE;
			_tmp0 = NULL;
			self->priv->comment = (_tmp0 = vala_scanner_pop_comment (self->priv->scanner), (self->priv->comment = (g_free (self->priv->comment), NULL)), _tmp0);
			_tmp18 = vala_parser_current (self);
			if (_tmp18 == VALA_TOKEN_TYPE_OPEN_BRACE)
			do {
				ValaStatement* _tmp1;
				_tmp1 = NULL;
				stmt = (_tmp1 = VALA_STATEMENT (vala_parser_parse_block (self, &inner_error)), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp1);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_SEMICOLON)
			do {
				ValaStatement* _tmp2;
				_tmp2 = NULL;
				stmt = (_tmp2 = vala_parser_parse_empty_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp2);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_IF)
			do {
				ValaStatement* _tmp3;
				_tmp3 = NULL;
				stmt = (_tmp3 = vala_parser_parse_if_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp3);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_SWITCH)
			do {
				ValaStatement* _tmp4;
				_tmp4 = NULL;
				stmt = (_tmp4 = vala_parser_parse_switch_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp4);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_WHILE)
			do {
				ValaStatement* _tmp5;
				_tmp5 = NULL;
				stmt = (_tmp5 = vala_parser_parse_while_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp5);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_DO)
			do {
				ValaStatement* _tmp6;
				_tmp6 = NULL;
				stmt = (_tmp6 = vala_parser_parse_do_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp6);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_FOR)
			do {
				ValaStatement* _tmp7;
				_tmp7 = NULL;
				stmt = (_tmp7 = vala_parser_parse_for_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp7);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_FOREACH)
			do {
				ValaStatement* _tmp8;
				_tmp8 = NULL;
				stmt = (_tmp8 = vala_parser_parse_foreach_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp8);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_BREAK)
			do {
				ValaStatement* _tmp9;
				_tmp9 = NULL;
				stmt = (_tmp9 = vala_parser_parse_break_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp9);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_CONTINUE)
			do {
				ValaStatement* _tmp10;
				_tmp10 = NULL;
				stmt = (_tmp10 = vala_parser_parse_continue_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp10);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_RETURN)
			do {
				ValaStatement* _tmp11;
				_tmp11 = NULL;
				stmt = (_tmp11 = vala_parser_parse_return_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp11);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_THROW)
			do {
				ValaStatement* _tmp12;
				_tmp12 = NULL;
				stmt = (_tmp12 = vala_parser_parse_throw_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp12);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_TRY)
			do {
				ValaStatement* _tmp13;
				_tmp13 = NULL;
				stmt = (_tmp13 = vala_parser_parse_try_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp13);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_LOCK)
			do {
				ValaStatement* _tmp14;
				_tmp14 = NULL;
				stmt = (_tmp14 = vala_parser_parse_lock_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp14);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_DELETE)
			do {
				ValaStatement* _tmp15;
				_tmp15 = NULL;
				stmt = (_tmp15 = vala_parser_parse_delete_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp15);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_VAR)
			do {
				is_decl = TRUE;
				vala_parser_parse_local_variable_declarations (self, block, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else if (_tmp18 == VALA_TOKEN_TYPE_OP_INC || _tmp18 == VALA_TOKEN_TYPE_OP_DEC || _tmp18 == VALA_TOKEN_TYPE_BASE || _tmp18 == VALA_TOKEN_TYPE_THIS || _tmp18 == VALA_TOKEN_TYPE_OPEN_PARENS || _tmp18 == VALA_TOKEN_TYPE_STAR || _tmp18 == VALA_TOKEN_TYPE_NEW)
			do {
				ValaStatement* _tmp16;
				_tmp16 = NULL;
				stmt = (_tmp16 = vala_parser_parse_expression_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp16);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				break;
			} while (0); else
			do {
				gboolean is_expr;
				is_expr = vala_parser_is_expression (self, &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch4_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
				if (is_expr) {
					ValaStatement* _tmp17;
					_tmp17 = NULL;
					stmt = (_tmp17 = vala_parser_parse_expression_statement (self, &inner_error), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp17);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
					}
				} else {
					is_decl = TRUE;
					vala_parser_parse_local_variable_declarations (self, block, &inner_error);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
					}
				}
				break;
			} while (0);
			if (!is_decl) {
				if (stmt == NULL) {
					/* workaround for current limitation of exception handling*/
					inner_error = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, "syntax error in statement");
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch4_vala_parse_error;
						}
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
					}
				}
				vala_block_add_statement (block, stmt);
			}
			(stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)));
		}
		goto __finally4;
		__catch4_vala_parse_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				if (vala_parser_recover (self) != VALA_PARSER_RECOVERY_STATE_STATEMENT_BEGIN) {
					/* beginning of next declaration or end of file reached
					 return what we have so far*/
					(e == NULL ? NULL : (e = (g_error_free (e), NULL)));
					break;
				}
				(e == NULL ? NULL : (e = (g_error_free (e), NULL)));
			}
		}
		__finally4:
		;
	}
}


static gboolean vala_parser_is_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaTokenType _tmp1;
	g_return_val_if_fail (VALA_IS_PARSER (self), FALSE);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	/* decide between declaration and expression statement*/
	vala_parser_skip_type (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return FALSE;
	}
	_tmp1 = vala_parser_current (self);
	if (_tmp1 == VALA_TOKEN_TYPE_OPEN_PARENS || _tmp1 == VALA_TOKEN_TYPE_OP_INC || _tmp1 == VALA_TOKEN_TYPE_OP_DEC || _tmp1 == VALA_TOKEN_TYPE_ASSIGN || _tmp1 == VALA_TOKEN_TYPE_ASSIGN_ADD || _tmp1 == VALA_TOKEN_TYPE_ASSIGN_BITWISE_AND || _tmp1 == VALA_TOKEN_TYPE_ASSIGN_BITWISE_OR || _tmp1 == VALA_TOKEN_TYPE_ASSIGN_BITWISE_XOR || _tmp1 == VALA_TOKEN_TYPE_ASSIGN_DIV || _tmp1 == VALA_TOKEN_TYPE_ASSIGN_MUL || _tmp1 == VALA_TOKEN_TYPE_ASSIGN_PERCENT || _tmp1 == VALA_TOKEN_TYPE_ASSIGN_SHIFT_LEFT || _tmp1 == VALA_TOKEN_TYPE_ASSIGN_SUB || _tmp1 == VALA_TOKEN_TYPE_OP_GT || _tmp1 == VALA_TOKEN_TYPE_DOT || _tmp1 == VALA_TOKEN_TYPE_OP_PTR)
	do {
		vala_parser_rollback (self, &begin);
		return TRUE;
	} while (0);
	vala_parser_rollback (self, &begin);
	return FALSE;
}


static ValaBlock* vala_parser_parse_embedded_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	char* _tmp1;
	ValaSourceReference* _tmp3;
	ValaSourceLocation _tmp2 = {0};
	ValaBlock* _tmp4;
	ValaBlock* block;
	ValaStatement* stmt;
	ValaBlock* _tmp5;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	if (vala_parser_current (self) == VALA_TOKEN_TYPE_OPEN_BRACE) {
		ValaBlock* block;
		block = vala_parser_parse_block (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			return NULL;
		}
		return block;
	}
	_tmp1 = NULL;
	self->priv->comment = (_tmp1 = vala_scanner_pop_comment (self->priv->scanner), (self->priv->comment = (g_free (self->priv->comment), NULL)), _tmp1);
	_tmp3 = NULL;
	_tmp4 = NULL;
	block = (_tmp4 = vala_block_new ((_tmp3 = vala_parser_get_src_com (self, (_tmp2 = vala_parser_get_location (self), &_tmp2)))), (_tmp3 == NULL ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL))), _tmp4);
	stmt = vala_parser_parse_embedded_statement_without_block (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
		return NULL;
	}
	if (stmt == NULL) {
		/* workaround for current limitation of exception handling*/
		inner_error = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, "syntax error in embedded statement");
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
			(stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)));
			return NULL;
		}
	}
	vala_block_add_statement (block, stmt);
	_tmp5 = NULL;
	return (_tmp5 = block, (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp5);
}


static ValaStatement* vala_parser_parse_embedded_statement_without_block (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaTokenType _tmp15;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	_tmp15 = vala_parser_current (self);
	if (_tmp15 == VALA_TOKEN_TYPE_SEMICOLON)
	do {
		return vala_parser_parse_empty_statement (self, &inner_error);
	} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_IF)
	do {
		return vala_parser_parse_if_statement (self, &inner_error);
	} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_SWITCH)
	do {
		return vala_parser_parse_switch_statement (self, &inner_error);
	} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_WHILE)
	do {
		return vala_parser_parse_while_statement (self, &inner_error);
	} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_DO)
	do {
		return vala_parser_parse_do_statement (self, &inner_error);
	} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_FOR)
	do {
		return vala_parser_parse_for_statement (self, &inner_error);
	} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_FOREACH)
	do {
		return vala_parser_parse_foreach_statement (self, &inner_error);
	} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_BREAK)
	do {
		return vala_parser_parse_break_statement (self, &inner_error);
	} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_CONTINUE)
	do {
		return vala_parser_parse_continue_statement (self, &inner_error);
	} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_RETURN)
	do {
		return vala_parser_parse_return_statement (self, &inner_error);
	} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_THROW)
	do {
		return vala_parser_parse_throw_statement (self, &inner_error);
	} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_TRY)
	do {
		return vala_parser_parse_try_statement (self, &inner_error);
	} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_LOCK)
	do {
		return vala_parser_parse_lock_statement (self, &inner_error);
	} while (0); else if (_tmp15 == VALA_TOKEN_TYPE_DELETE)
	do {
		return vala_parser_parse_delete_statement (self, &inner_error);
	} while (0); else
	do {
		return vala_parser_parse_expression_statement (self, &inner_error);
	} while (0);
}


static ValaBlock* vala_parser_parse_block (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	GeeList* list;
	ValaSourceReference* _tmp0;
	ValaBlock* _tmp1;
	ValaBlock* block;
	ValaSourceReference* _tmp3;
	ValaSourceReference* _tmp4;
	ValaBlock* _tmp5;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	list = GEE_LIST (gee_array_list_new (VALA_TYPE_STATEMENT, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal));
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(list == NULL ? NULL : (list = (gee_collection_object_unref (list), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	block = (_tmp1 = vala_block_new ((_tmp0 = vala_parser_get_src_com (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	vala_parser_parse_statements (self, block, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(list == NULL ? NULL : (list = (gee_collection_object_unref (list), NULL)));
		(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
		return NULL;
	}
	if (!vala_parser_accept (self, VALA_TOKEN_TYPE_CLOSE_BRACE)) {
		/* only report error if it's not a secondary error*/
		if (vala_report_get_errors () == 0) {
			ValaSourceReference* _tmp2;
			_tmp2 = NULL;
			vala_report_error ((_tmp2 = vala_parser_get_current_src (self)), "expected `}'");
			(_tmp2 == NULL ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)));
		}
	}
	_tmp3 = NULL;
	vala_source_reference_set_last_line (vala_code_node_get_source_reference (VALA_CODE_NODE (block)), vala_source_reference_get_last_line ((_tmp3 = vala_parser_get_current_src (self))));
	(_tmp3 == NULL ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL)));
	_tmp4 = NULL;
	vala_source_reference_set_last_column (vala_code_node_get_source_reference (VALA_CODE_NODE (block)), vala_source_reference_get_last_column ((_tmp4 = vala_parser_get_current_src (self))));
	(_tmp4 == NULL ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL)));
	_tmp5 = NULL;
	return (_tmp5 = block, (list == NULL ? NULL : (list = (gee_collection_object_unref (list), NULL))), _tmp5);
}


static ValaStatement* vala_parser_parse_empty_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = VALA_STATEMENT (vala_empty_statement_new ((_tmp0 = vala_parser_get_src_com (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
}


static void vala_parser_parse_local_variable_declarations (ValaParser* self, ValaBlock* block, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaDataType* variable_type;
	g_return_if_fail (VALA_IS_PARSER (self));
	g_return_if_fail (VALA_IS_BLOCK (block));
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	variable_type = NULL;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_VAR)) {
		ValaDataType* _tmp0;
		_tmp0 = NULL;
		variable_type = (_tmp0 = NULL, (variable_type == NULL ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL))), _tmp0);
	} else {
		ValaDataType* _tmp1;
		_tmp1 = NULL;
		variable_type = (_tmp1 = vala_parser_parse_type (self, TRUE, &inner_error), (variable_type == NULL ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL))), _tmp1);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(variable_type == NULL ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL)));
			return;
		}
	}
	do {
		ValaDataType* type_copy;
		ValaLocalVariable* local;
		ValaDeclarationStatement* _tmp3;
		type_copy = NULL;
		if (variable_type != NULL) {
			ValaDataType* _tmp2;
			_tmp2 = NULL;
			type_copy = (_tmp2 = vala_data_type_copy (variable_type), (type_copy == NULL ? NULL : (type_copy = (vala_code_node_unref (type_copy), NULL))), _tmp2);
		}
		local = vala_parser_parse_local_variable (self, type_copy, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(type_copy == NULL ? NULL : (type_copy = (vala_code_node_unref (type_copy), NULL)));
			(variable_type == NULL ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL)));
			return;
		}
		_tmp3 = NULL;
		vala_block_add_statement (block, VALA_STATEMENT ((_tmp3 = vala_declaration_statement_new (VALA_SYMBOL (local), vala_code_node_get_source_reference (VALA_CODE_NODE (local))))));
		(_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)));
		(type_copy == NULL ? NULL : (type_copy = (vala_code_node_unref (type_copy), NULL)));
		(local == NULL ? NULL : (local = (vala_code_node_unref (local), NULL)));
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(variable_type == NULL ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL)));
		return;
	}
	(variable_type == NULL ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL)));
}


static ValaLocalVariable* vala_parser_parse_local_variable (ValaParser* self, ValaDataType* variable_type, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	char* id;
	ValaExpression* initializer;
	ValaSourceReference* _tmp5;
	ValaLocalVariable* _tmp6;
	ValaLocalVariable* _tmp7;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (variable_type == NULL || VALA_IS_DATA_TYPE (variable_type), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	initializer = NULL;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_ASSIGN)) {
		ValaExpression* _tmp0;
		ValaArrayType* _tmp2;
		ValaDataType* _tmp1;
		ValaArrayType* array_type;
		_tmp0 = NULL;
		initializer = (_tmp0 = vala_parser_parse_variable_initializer (self, &inner_error), (initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL))), _tmp0);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			id = (g_free (id), NULL);
			(initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)));
			return NULL;
		}
		/* transform shorthand form
		     int[] array = { 42 };
		 into
		     int[] array = new int[] { 42 };*/
		_tmp2 = NULL;
		_tmp1 = NULL;
		array_type = (_tmp2 = (_tmp1 = variable_type, (VALA_IS_ARRAY_TYPE (_tmp1) ? ((ValaArrayType*) (_tmp1)) : NULL)), (_tmp2 == NULL ? NULL : vala_code_node_ref (_tmp2)));
		if (array_type != NULL && VALA_IS_INITIALIZER_LIST (initializer)) {
			ValaExpression* _tmp4;
			ValaDataType* _tmp3;
			_tmp4 = NULL;
			_tmp3 = NULL;
			initializer = (_tmp4 = VALA_EXPRESSION (vala_array_creation_expression_new ((_tmp3 = vala_data_type_copy (vala_array_type_get_element_type (array_type))), vala_array_type_get_rank (array_type), VALA_INITIALIZER_LIST (initializer), vala_code_node_get_source_reference (VALA_CODE_NODE (initializer)))), (initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL))), _tmp4);
			(_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)));
		}
		(array_type == NULL ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)));
	}
	_tmp5 = NULL;
	_tmp6 = NULL;
	_tmp7 = NULL;
	return (_tmp7 = (_tmp6 = vala_local_variable_new (variable_type, id, initializer, (_tmp5 = vala_parser_get_src_com (self, &begin))), (_tmp5 == NULL ? NULL : (_tmp5 = (vala_source_reference_unref (_tmp5), NULL))), _tmp6), (id = (g_free (id), NULL)), (initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL))), _tmp7);
}


static ValaStatement* vala_parser_parse_expression_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	ValaStatement* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	expr = vala_parser_parse_statement_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = VALA_STATEMENT (vala_expression_statement_new (expr, (_tmp0 = vala_parser_get_src_com (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp2);
}


static ValaExpression* vala_parser_parse_statement_expression (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaExpression* expr;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	/* invocation expression, assignment,
	 or pre/post increment/decrement expression*/
	expr = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	return expr;
}


static ValaStatement* vala_parser_parse_if_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* condition;
	ValaSourceReference* src;
	ValaBlock* true_stmt;
	ValaBlock* false_stmt;
	ValaStatement* _tmp1;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_IF, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	condition = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
		return NULL;
	}
	src = vala_parser_get_src_com (self, &begin);
	true_stmt = vala_parser_parse_embedded_statement (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
		(src == NULL ? NULL : (src = (vala_source_reference_unref (src), NULL)));
		return NULL;
	}
	false_stmt = NULL;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_ELSE)) {
		ValaBlock* _tmp0;
		_tmp0 = NULL;
		false_stmt = (_tmp0 = vala_parser_parse_embedded_statement (self, &inner_error), (false_stmt == NULL ? NULL : (false_stmt = (vala_code_node_unref (false_stmt), NULL))), _tmp0);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
			(src == NULL ? NULL : (src = (vala_source_reference_unref (src), NULL)));
			(true_stmt == NULL ? NULL : (true_stmt = (vala_code_node_unref (true_stmt), NULL)));
			(false_stmt == NULL ? NULL : (false_stmt = (vala_code_node_unref (false_stmt), NULL)));
			return NULL;
		}
	}
	_tmp1 = NULL;
	return (_tmp1 = VALA_STATEMENT (vala_if_statement_new (condition, true_stmt, false_stmt, src)), (condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL))), (src == NULL ? NULL : (src = (vala_source_reference_unref (src), NULL))), (true_stmt == NULL ? NULL : (true_stmt = (vala_code_node_unref (true_stmt), NULL))), (false_stmt == NULL ? NULL : (false_stmt = (vala_code_node_unref (false_stmt), NULL))), _tmp1);
}


static ValaStatement* vala_parser_parse_switch_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* condition;
	ValaSourceReference* _tmp0;
	ValaSwitchStatement* _tmp1;
	ValaSwitchStatement* stmt;
	ValaStatement* _tmp9;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_SWITCH, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	condition = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	stmt = (_tmp1 = vala_switch_statement_new (condition, (_tmp0 = vala_parser_get_src_com (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
		(stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)));
		return NULL;
	}
	while (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACE) {
		ValaSourceReference* _tmp2;
		ValaSwitchSection* _tmp3;
		ValaSwitchSection* section;
		_tmp2 = NULL;
		_tmp3 = NULL;
		section = (_tmp3 = vala_switch_section_new ((_tmp2 = vala_parser_get_src_com (self, &begin))), (_tmp2 == NULL ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL))), _tmp3);
		do {
			if (vala_parser_accept (self, VALA_TOKEN_TYPE_CASE)) {
				ValaSwitchLabel* _tmp6;
				ValaSourceReference* _tmp5;
				ValaExpression* _tmp4;
				_tmp6 = NULL;
				_tmp5 = NULL;
				_tmp4 = NULL;
				vala_switch_section_add_label (section, (_tmp6 = vala_switch_label_new ((_tmp4 = vala_parser_parse_expression (self, &inner_error)), (_tmp5 = vala_parser_get_src_com (self, &begin)))));
				(_tmp6 == NULL ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL)));
				(_tmp5 == NULL ? NULL : (_tmp5 = (vala_source_reference_unref (_tmp5), NULL)));
				(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
			} else {
				ValaSwitchLabel* _tmp8;
				ValaSourceReference* _tmp7;
				vala_parser_expect (self, VALA_TOKEN_TYPE_DEFAULT, &inner_error);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(section == NULL ? NULL : (section = (vala_code_node_unref (section), NULL)));
					(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
					(stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)));
					return NULL;
				}
				_tmp8 = NULL;
				_tmp7 = NULL;
				vala_switch_section_add_label (section, (_tmp8 = vala_switch_label_new_with_default ((_tmp7 = vala_parser_get_src_com (self, &begin)))));
				(_tmp8 == NULL ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL)));
				(_tmp7 == NULL ? NULL : (_tmp7 = (vala_source_reference_unref (_tmp7), NULL)));
			}
			vala_parser_expect (self, VALA_TOKEN_TYPE_COLON, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(section == NULL ? NULL : (section = (vala_code_node_unref (section), NULL)));
				(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
				(stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)));
				return NULL;
			}
		} while (vala_parser_current (self) == VALA_TOKEN_TYPE_CASE || vala_parser_current (self) == VALA_TOKEN_TYPE_DEFAULT);
		vala_parser_parse_statements (self, VALA_BLOCK (section), &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(section == NULL ? NULL : (section = (vala_code_node_unref (section), NULL)));
			(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
			(stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)));
			return NULL;
		}
		vala_switch_statement_add_section (stmt, section);
		(section == NULL ? NULL : (section = (vala_code_node_unref (section), NULL)));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
		(stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)));
		return NULL;
	}
	_tmp9 = NULL;
	return (_tmp9 = VALA_STATEMENT (stmt), (condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL))), _tmp9);
}


static ValaStatement* vala_parser_parse_while_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* condition;
	ValaBlock* body;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	ValaStatement* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_WHILE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	condition = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
		return NULL;
	}
	body = vala_parser_parse_embedded_statement (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = VALA_STATEMENT (vala_while_statement_new (condition, body, (_tmp0 = vala_parser_get_src_com (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1), (condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL))), (body == NULL ? NULL : (body = (vala_code_node_unref (body), NULL))), _tmp2);
}


static ValaStatement* vala_parser_parse_do_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaBlock* body;
	ValaExpression* condition;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	ValaStatement* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_DO, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	body = vala_parser_parse_embedded_statement (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_WHILE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(body == NULL ? NULL : (body = (vala_code_node_unref (body), NULL)));
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(body == NULL ? NULL : (body = (vala_code_node_unref (body), NULL)));
		return NULL;
	}
	condition = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(body == NULL ? NULL : (body = (vala_code_node_unref (body), NULL)));
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(body == NULL ? NULL : (body = (vala_code_node_unref (body), NULL)));
		(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(body == NULL ? NULL : (body = (vala_code_node_unref (body), NULL)));
		(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = VALA_STATEMENT (vala_do_statement_new (body, condition, (_tmp0 = vala_parser_get_src_com (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1), (body == NULL ? NULL : (body = (vala_code_node_unref (body), NULL))), (condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL))), _tmp2);
}


static ValaStatement* vala_parser_parse_for_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaBlock* block;
	GeeArrayList* initializer_list;
	ValaExpression* condition;
	GeeArrayList* iterator_list;
	ValaSourceReference* src;
	ValaBlock* body;
	ValaForStatement* stmt;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	block = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_FOR, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
		return NULL;
	}
	initializer_list = gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	if (!vala_parser_accept (self, VALA_TOKEN_TYPE_SEMICOLON)) {
		gboolean is_expr;
		ValaTokenType _tmp0;
		is_expr = FALSE;
		_tmp0 = vala_parser_current (self);
		if (_tmp0 == VALA_TOKEN_TYPE_VAR)
		do {
			is_expr = FALSE;
			break;
		} while (0); else if (_tmp0 == VALA_TOKEN_TYPE_OP_INC || _tmp0 == VALA_TOKEN_TYPE_OP_DEC)
		do {
			is_expr = TRUE;
			break;
		} while (0); else
		do {
			gboolean local_is_expr;
			local_is_expr = vala_parser_is_expression (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
				(initializer_list == NULL ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL)));
				return NULL;
			}
			is_expr = local_is_expr;
			break;
		} while (0);
		if (is_expr) {
			do {
				ValaExpression* _tmp1;
				_tmp1 = NULL;
				gee_collection_add (GEE_COLLECTION (initializer_list), (_tmp1 = vala_parser_parse_statement_expression (self, &inner_error)));
				(_tmp1 == NULL ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL)));
			} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		} else {
			ValaBlock* _tmp3;
			ValaSourceReference* _tmp2;
			ValaDataType* variable_type;
			ValaLocalVariable* local;
			ValaDeclarationStatement* _tmp6;
			_tmp3 = NULL;
			_tmp2 = NULL;
			block = (_tmp3 = vala_block_new ((_tmp2 = vala_parser_get_src (self, &begin))), (block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL))), _tmp3);
			(_tmp2 == NULL ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)));
			variable_type = NULL;
			if (vala_parser_accept (self, VALA_TOKEN_TYPE_VAR)) {
				ValaDataType* _tmp4;
				_tmp4 = NULL;
				variable_type = (_tmp4 = NULL, (variable_type == NULL ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL))), _tmp4);
			} else {
				ValaDataType* _tmp5;
				_tmp5 = NULL;
				variable_type = (_tmp5 = vala_parser_parse_type (self, TRUE, &inner_error), (variable_type == NULL ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL))), _tmp5);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(variable_type == NULL ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL)));
					(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
					(initializer_list == NULL ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL)));
					return NULL;
				}
			}
			local = vala_parser_parse_local_variable (self, variable_type, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(variable_type == NULL ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL)));
				(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
				(initializer_list == NULL ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL)));
				return NULL;
			}
			_tmp6 = NULL;
			vala_block_add_statement (block, VALA_STATEMENT ((_tmp6 = vala_declaration_statement_new (VALA_SYMBOL (local), vala_code_node_get_source_reference (VALA_CODE_NODE (local))))));
			(_tmp6 == NULL ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL)));
			(variable_type == NULL ? NULL : (variable_type = (vala_code_node_unref (variable_type), NULL)));
			(local == NULL ? NULL : (local = (vala_code_node_unref (local), NULL)));
		}
		vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
			(initializer_list == NULL ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL)));
			return NULL;
		}
	}
	condition = NULL;
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_SEMICOLON) {
		ValaExpression* _tmp7;
		_tmp7 = NULL;
		condition = (_tmp7 = vala_parser_parse_expression (self, &inner_error), (condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL))), _tmp7);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
			(initializer_list == NULL ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL)));
			(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
		(initializer_list == NULL ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL)));
		(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
		return NULL;
	}
	iterator_list = gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
		do {
			ValaExpression* _tmp8;
			_tmp8 = NULL;
			gee_collection_add (GEE_COLLECTION (iterator_list), (_tmp8 = vala_parser_parse_statement_expression (self, &inner_error)));
			(_tmp8 == NULL ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
		(initializer_list == NULL ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL)));
		(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
		(iterator_list == NULL ? NULL : (iterator_list = (gee_collection_object_unref (iterator_list), NULL)));
		return NULL;
	}
	src = vala_parser_get_src_com (self, &begin);
	body = vala_parser_parse_embedded_statement (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
		(initializer_list == NULL ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL)));
		(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
		(iterator_list == NULL ? NULL : (iterator_list = (gee_collection_object_unref (iterator_list), NULL)));
		(src == NULL ? NULL : (src = (vala_source_reference_unref (src), NULL)));
		return NULL;
	}
	stmt = vala_for_statement_new (condition, body, src);
	{
		GeeArrayList* init_collection;
		int init_it;
		init_collection = initializer_list;
		for (init_it = 0; init_it < gee_collection_get_size (GEE_COLLECTION (init_collection)); init_it = init_it + 1) {
			ValaExpression* init;
			init = ((ValaExpression*) (gee_list_get (GEE_LIST (init_collection), init_it)));
			{
				vala_for_statement_add_initializer (stmt, init);
				(init == NULL ? NULL : (init = (vala_code_node_unref (init), NULL)));
			}
		}
	}
	{
		GeeArrayList* iter_collection;
		int iter_it;
		iter_collection = iterator_list;
		for (iter_it = 0; iter_it < gee_collection_get_size (GEE_COLLECTION (iter_collection)); iter_it = iter_it + 1) {
			ValaExpression* iter;
			iter = ((ValaExpression*) (gee_list_get (GEE_LIST (iter_collection), iter_it)));
			{
				vala_for_statement_add_iterator (stmt, iter);
				(iter == NULL ? NULL : (iter = (vala_code_node_unref (iter), NULL)));
			}
		}
	}
	if (block != NULL) {
		ValaStatement* _tmp9;
		vala_block_add_statement (block, VALA_STATEMENT (stmt));
		_tmp9 = NULL;
		return (_tmp9 = VALA_STATEMENT (block), (initializer_list == NULL ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL))), (condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL))), (iterator_list == NULL ? NULL : (iterator_list = (gee_collection_object_unref (iterator_list), NULL))), (src == NULL ? NULL : (src = (vala_source_reference_unref (src), NULL))), (body == NULL ? NULL : (body = (vala_code_node_unref (body), NULL))), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp9);
	} else {
		ValaStatement* _tmp10;
		_tmp10 = NULL;
		return (_tmp10 = VALA_STATEMENT (stmt), (block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL))), (initializer_list == NULL ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL))), (condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL))), (iterator_list == NULL ? NULL : (iterator_list = (gee_collection_object_unref (iterator_list), NULL))), (src == NULL ? NULL : (src = (vala_source_reference_unref (src), NULL))), (body == NULL ? NULL : (body = (vala_code_node_unref (body), NULL))), _tmp10);
	}
	(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
	(initializer_list == NULL ? NULL : (initializer_list = (gee_collection_object_unref (initializer_list), NULL)));
	(condition == NULL ? NULL : (condition = (vala_code_node_unref (condition), NULL)));
	(iterator_list == NULL ? NULL : (iterator_list = (gee_collection_object_unref (iterator_list), NULL)));
	(src == NULL ? NULL : (src = (vala_source_reference_unref (src), NULL)));
	(body == NULL ? NULL : (body = (vala_code_node_unref (body), NULL)));
	(stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)));
}


static ValaStatement* vala_parser_parse_foreach_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaDataType* type;
	char* id;
	ValaExpression* collection;
	ValaSourceReference* src;
	ValaBlock* body;
	ValaStatement* _tmp1;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_FOREACH, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	type = NULL;
	if (!vala_parser_accept (self, VALA_TOKEN_TYPE_VAR)) {
		ValaDataType* _tmp0;
		_tmp0 = NULL;
		type = (_tmp0 = vala_parser_parse_type (self, TRUE, &inner_error), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp0);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			return NULL;
		}
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_IN, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		return NULL;
	}
	collection = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		(collection == NULL ? NULL : (collection = (vala_code_node_unref (collection), NULL)));
		return NULL;
	}
	src = vala_parser_get_src_com (self, &begin);
	body = vala_parser_parse_embedded_statement (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		(collection == NULL ? NULL : (collection = (vala_code_node_unref (collection), NULL)));
		(src == NULL ? NULL : (src = (vala_source_reference_unref (src), NULL)));
		return NULL;
	}
	_tmp1 = NULL;
	return (_tmp1 = VALA_STATEMENT (vala_foreach_statement_new (type, id, collection, body, src)), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), (id = (g_free (id), NULL)), (collection == NULL ? NULL : (collection = (vala_code_node_unref (collection), NULL))), (src == NULL ? NULL : (src = (vala_source_reference_unref (src), NULL))), (body == NULL ? NULL : (body = (vala_code_node_unref (body), NULL))), _tmp1);
}


static ValaStatement* vala_parser_parse_break_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_BREAK, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = VALA_STATEMENT (vala_break_statement_new ((_tmp0 = vala_parser_get_src_com (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
}


static ValaStatement* vala_parser_parse_continue_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_CONTINUE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = VALA_STATEMENT (vala_continue_statement_new ((_tmp0 = vala_parser_get_src_com (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
}


static ValaStatement* vala_parser_parse_return_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	ValaSourceReference* _tmp1;
	ValaStatement* _tmp2;
	ValaStatement* _tmp3;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_RETURN, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	expr = NULL;
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_SEMICOLON) {
		ValaExpression* _tmp0;
		_tmp0 = NULL;
		expr = (_tmp0 = vala_parser_parse_expression (self, &inner_error), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp0);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
		return NULL;
	}
	_tmp1 = NULL;
	_tmp2 = NULL;
	_tmp3 = NULL;
	return (_tmp3 = (_tmp2 = VALA_STATEMENT (vala_return_statement_new (expr, (_tmp1 = vala_parser_get_src_com (self, &begin)))), (_tmp1 == NULL ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL))), _tmp2), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp3);
}


static ValaStatement* vala_parser_parse_throw_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	ValaStatement* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_THROW, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	expr = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = VALA_STATEMENT (vala_throw_statement_new (expr, (_tmp0 = vala_parser_get_src_com (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp2);
}


static ValaStatement* vala_parser_parse_try_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaBlock* try_block;
	ValaBlock* finally_clause;
	GeeArrayList* catch_clauses;
	ValaSourceReference* _tmp2;
	ValaTryStatement* _tmp3;
	ValaTryStatement* stmt;
	ValaStatement* _tmp4;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_TRY, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	try_block = vala_parser_parse_block (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	finally_clause = NULL;
	catch_clauses = gee_array_list_new (VALA_TYPE_CATCH_CLAUSE, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	if (vala_parser_current (self) == VALA_TOKEN_TYPE_CATCH) {
		vala_parser_parse_catch_clauses (self, GEE_LIST (catch_clauses), &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(try_block == NULL ? NULL : (try_block = (vala_code_node_unref (try_block), NULL)));
			(finally_clause == NULL ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL)));
			(catch_clauses == NULL ? NULL : (catch_clauses = (gee_collection_object_unref (catch_clauses), NULL)));
			return NULL;
		}
		if (vala_parser_current (self) == VALA_TOKEN_TYPE_FINALLY) {
			ValaBlock* _tmp0;
			_tmp0 = NULL;
			finally_clause = (_tmp0 = vala_parser_parse_finally_clause (self, &inner_error), (finally_clause == NULL ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL))), _tmp0);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(try_block == NULL ? NULL : (try_block = (vala_code_node_unref (try_block), NULL)));
				(finally_clause == NULL ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL)));
				(catch_clauses == NULL ? NULL : (catch_clauses = (gee_collection_object_unref (catch_clauses), NULL)));
				return NULL;
			}
		}
	} else {
		ValaBlock* _tmp1;
		_tmp1 = NULL;
		finally_clause = (_tmp1 = vala_parser_parse_finally_clause (self, &inner_error), (finally_clause == NULL ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL))), _tmp1);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(try_block == NULL ? NULL : (try_block = (vala_code_node_unref (try_block), NULL)));
			(finally_clause == NULL ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL)));
			(catch_clauses == NULL ? NULL : (catch_clauses = (gee_collection_object_unref (catch_clauses), NULL)));
			return NULL;
		}
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	stmt = (_tmp3 = vala_try_statement_new (try_block, finally_clause, (_tmp2 = vala_parser_get_src_com (self, &begin))), (_tmp2 == NULL ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL))), _tmp3);
	{
		GeeArrayList* clause_collection;
		int clause_it;
		clause_collection = catch_clauses;
		for (clause_it = 0; clause_it < gee_collection_get_size (GEE_COLLECTION (clause_collection)); clause_it = clause_it + 1) {
			ValaCatchClause* clause;
			clause = ((ValaCatchClause*) (gee_list_get (GEE_LIST (clause_collection), clause_it)));
			{
				vala_try_statement_add_catch_clause (stmt, clause);
				(clause == NULL ? NULL : (clause = (vala_code_node_unref (clause), NULL)));
			}
		}
	}
	_tmp4 = NULL;
	return (_tmp4 = VALA_STATEMENT (stmt), (try_block == NULL ? NULL : (try_block = (vala_code_node_unref (try_block), NULL))), (finally_clause == NULL ? NULL : (finally_clause = (vala_code_node_unref (finally_clause), NULL))), (catch_clauses == NULL ? NULL : (catch_clauses = (gee_collection_object_unref (catch_clauses), NULL))), _tmp4);
}


static void vala_parser_parse_catch_clauses (ValaParser* self, GeeList* catch_clauses, GError** error) {
	GError * inner_error;
	g_return_if_fail (VALA_IS_PARSER (self));
	g_return_if_fail (GEE_IS_LIST (catch_clauses));
	inner_error = NULL;
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_CATCH)) {
		ValaSourceLocation begin;
		ValaDataType* type;
		char* id;
		ValaBlock* block;
		ValaCatchClause* _tmp3;
		ValaSourceReference* _tmp2;
		begin = vala_parser_get_location (self);
		type = NULL;
		id = NULL;
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_PARENS)) {
			ValaDataType* _tmp0;
			char* _tmp1;
			_tmp0 = NULL;
			type = (_tmp0 = vala_parser_parse_type (self, TRUE, &inner_error), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp0);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
				id = (g_free (id), NULL);
				return;
			}
			_tmp1 = NULL;
			id = (_tmp1 = vala_parser_parse_identifier (self, &inner_error), (id = (g_free (id), NULL)), _tmp1);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
				id = (g_free (id), NULL);
				return;
			}
			vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
				id = (g_free (id), NULL);
				return;
			}
		}
		block = vala_parser_parse_block (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			id = (g_free (id), NULL);
			return;
		}
		_tmp3 = NULL;
		_tmp2 = NULL;
		gee_collection_add (GEE_COLLECTION (catch_clauses), (_tmp3 = vala_catch_clause_new (type, id, block, (_tmp2 = vala_parser_get_src (self, &begin)))));
		(_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)));
		(_tmp2 == NULL ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)));
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
	}
}


static ValaBlock* vala_parser_parse_finally_clause (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaBlock* block;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	vala_parser_expect (self, VALA_TOKEN_TYPE_FINALLY, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	block = vala_parser_parse_block (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	return block;
}


static ValaStatement* vala_parser_parse_lock_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	ValaBlock* stmt;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	ValaStatement* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_LOCK, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	expr = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
		return NULL;
	}
	stmt = vala_parser_parse_embedded_statement (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = VALA_STATEMENT (vala_lock_statement_new (expr, stmt, (_tmp0 = vala_parser_get_src_com (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), (stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL))), _tmp2);
}


static ValaStatement* vala_parser_parse_delete_statement (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaExpression* expr;
	ValaSourceReference* _tmp0;
	ValaStatement* _tmp1;
	ValaStatement* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_DELETE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	expr = vala_parser_parse_expression (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = VALA_STATEMENT (vala_delete_statement_new (expr, (_tmp0 = vala_parser_get_src_com (self, &begin)))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp2);
}


static GeeList* vala_parser_parse_attributes (ValaParser* self, GError** error) {
	GError * inner_error;
	GeeArrayList* attrs;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_OPEN_BRACKET) {
		return NULL;
	}
	attrs = gee_array_list_new (VALA_TYPE_ATTRIBUTE, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_BRACKET)) {
		do {
			ValaSourceLocation begin;
			char* id;
			ValaSourceReference* _tmp1;
			ValaAttribute* _tmp2;
			ValaAttribute* attr;
			begin = vala_parser_get_location (self);
			id = vala_parser_parse_identifier (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
				return NULL;
			}
			_tmp1 = NULL;
			_tmp2 = NULL;
			attr = (_tmp2 = vala_attribute_new (id, (_tmp1 = vala_parser_get_src (self, &begin))), (_tmp1 == NULL ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL))), _tmp2);
			if (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_PARENS)) {
				if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
					do {
						char* id;
						ValaExpression* expr;
						ValaNamedArgument* _tmp4;
						ValaSourceReference* _tmp3;
						begin = vala_parser_get_location (self);
						id = vala_parser_parse_identifier (self, &inner_error);
						if (inner_error != NULL) {
							g_propagate_error (error, inner_error);
							id = (g_free (id), NULL);
							(attr == NULL ? NULL : (attr = (vala_code_node_unref (attr), NULL)));
							(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
							return NULL;
						}
						vala_parser_expect (self, VALA_TOKEN_TYPE_ASSIGN, &inner_error);
						if (inner_error != NULL) {
							g_propagate_error (error, inner_error);
							id = (g_free (id), NULL);
							id = (g_free (id), NULL);
							(attr == NULL ? NULL : (attr = (vala_code_node_unref (attr), NULL)));
							(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
							return NULL;
						}
						expr = vala_parser_parse_expression (self, &inner_error);
						if (inner_error != NULL) {
							g_propagate_error (error, inner_error);
							id = (g_free (id), NULL);
							id = (g_free (id), NULL);
							(attr == NULL ? NULL : (attr = (vala_code_node_unref (attr), NULL)));
							(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
							return NULL;
						}
						_tmp4 = NULL;
						_tmp3 = NULL;
						vala_attribute_add_argument (attr, (_tmp4 = vala_named_argument_new (id, expr, (_tmp3 = vala_parser_get_src (self, &begin)))));
						(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
						(_tmp3 == NULL ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL)));
						id = (g_free (id), NULL);
						(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
					} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
				}
				vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					id = (g_free (id), NULL);
					(attr == NULL ? NULL : (attr = (vala_code_node_unref (attr), NULL)));
					(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
					return NULL;
				}
			}
			gee_collection_add (GEE_COLLECTION (attrs), attr);
			id = (g_free (id), NULL);
			(attr == NULL ? NULL : (attr = (vala_code_node_unref (attr), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACKET, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
			return NULL;
		}
	}
	return GEE_LIST (attrs);
}


static void vala_parser_set_attributes (ValaParser* self, ValaCodeNode* node, GeeList* attributes) {
	g_return_if_fail (VALA_IS_PARSER (self));
	g_return_if_fail (VALA_IS_CODE_NODE (node));
	g_return_if_fail (attributes == NULL || GEE_IS_LIST (attributes));
	if (attributes != NULL) {
		{
			GeeList* attr_collection;
			int attr_it;
			attr_collection = GEE_LIST (attributes);
			for (attr_it = 0; attr_it < gee_collection_get_size (GEE_COLLECTION (attr_collection)); attr_it = attr_it + 1) {
				ValaAttribute* attr;
				attr = ((ValaAttribute*) (gee_list_get (GEE_LIST (attr_collection), attr_it)));
				{
					ValaAttribute* _tmp0;
					_tmp0 = NULL;
					node->attributes = g_list_append (node->attributes, (_tmp0 = attr, (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0))));
					(attr == NULL ? NULL : (attr = (vala_code_node_unref (attr), NULL)));
				}
			}
		}
	}
}


static ValaSymbol* vala_parser_parse_declaration (ValaParser* self, GError** error) {
	GError * inner_error;
	char* _tmp0;
	GeeList* attrs;
	ValaSourceLocation begin;
	ValaTokenType last_keyword;
	ValaTokenType _tmp21;
	char* _tmp22;
	GError* _tmp23;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	_tmp0 = NULL;
	self->priv->comment = (_tmp0 = vala_scanner_pop_comment (self->priv->scanner), (self->priv->comment = (g_free (self->priv->comment), NULL)), _tmp0);
	attrs = vala_parser_parse_attributes (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	begin = vala_parser_get_location (self);
	last_keyword = vala_parser_current (self);
	while (vala_parser_is_declaration_keyword (self, vala_parser_current (self))) {
		last_keyword = vala_parser_current (self);
		vala_parser_next (self);
	}
	_tmp21 = vala_parser_current (self);
	if (_tmp21 == VALA_TOKEN_TYPE_CONSTRUCT)
	do {
		ValaSymbol* _tmp1;
		vala_parser_rollback (self, &begin);
		_tmp1 = NULL;
		return (_tmp1 = VALA_SYMBOL (vala_parser_parse_constructor_declaration (self, attrs, &inner_error)), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp1);
	} while (0); else if (_tmp21 == VALA_TOKEN_TYPE_TILDE)
	do {
		ValaSymbol* _tmp2;
		vala_parser_rollback (self, &begin);
		_tmp2 = NULL;
		return (_tmp2 = VALA_SYMBOL (vala_parser_parse_destructor_declaration (self, attrs, &inner_error)), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp2);
	} while (0); else
	do {
		ValaTokenType _tmp20;
		vala_parser_skip_type (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
			return NULL;
		}
		_tmp20 = vala_parser_current (self);
		if (_tmp20 == VALA_TOKEN_TYPE_OPEN_BRACE || _tmp20 == VALA_TOKEN_TYPE_SEMICOLON || _tmp20 == VALA_TOKEN_TYPE_COLON)
		do {
			ValaTokenType _tmp9;
			vala_parser_rollback (self, &begin);
			_tmp9 = last_keyword;
			if (_tmp9 == VALA_TOKEN_TYPE_CLASS)
			do {
				ValaSymbol* _tmp3;
				_tmp3 = NULL;
				return (_tmp3 = vala_parser_parse_class_declaration (self, attrs, &inner_error), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp3);
			} while (0); else if (_tmp9 == VALA_TOKEN_TYPE_ENUM)
			do {
				ValaSymbol* _tmp4;
				_tmp4 = NULL;
				return (_tmp4 = vala_parser_parse_enum_declaration (self, attrs, &inner_error), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp4);
			} while (0); else if (_tmp9 == VALA_TOKEN_TYPE_ERRORDOMAIN)
			do {
				ValaSymbol* _tmp5;
				_tmp5 = NULL;
				return (_tmp5 = vala_parser_parse_errordomain_declaration (self, attrs, &inner_error), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp5);
			} while (0); else if (_tmp9 == VALA_TOKEN_TYPE_INTERFACE)
			do {
				ValaSymbol* _tmp6;
				_tmp6 = NULL;
				return (_tmp6 = vala_parser_parse_interface_declaration (self, attrs, &inner_error), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp6);
			} while (0); else if (_tmp9 == VALA_TOKEN_TYPE_NAMESPACE)
			do {
				ValaSymbol* _tmp7;
				_tmp7 = NULL;
				return (_tmp7 = VALA_SYMBOL (vala_parser_parse_namespace_declaration (self, attrs, &inner_error)), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp7);
			} while (0); else if (_tmp9 == VALA_TOKEN_TYPE_STRUCT)
			do {
				ValaSymbol* _tmp8;
				_tmp8 = NULL;
				return (_tmp8 = vala_parser_parse_struct_declaration (self, attrs, &inner_error), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp8);
			} while (0);
			break;
		} while (0); else if (_tmp20 == VALA_TOKEN_TYPE_OPEN_PARENS)
		do {
			ValaSymbol* _tmp10;
			vala_parser_rollback (self, &begin);
			_tmp10 = NULL;
			return (_tmp10 = VALA_SYMBOL (vala_parser_parse_creation_method_declaration (self, attrs, &inner_error)), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp10);
		} while (0); else
		do {
			ValaTokenType _tmp19;
			vala_parser_skip_type (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
				return NULL;
			}
			_tmp19 = vala_parser_current (self);
			if (_tmp19 == VALA_TOKEN_TYPE_OPEN_PARENS)
			do {
				ValaTokenType _tmp14;
				vala_parser_rollback (self, &begin);
				_tmp14 = last_keyword;
				if (_tmp14 == VALA_TOKEN_TYPE_DELEGATE)
				do {
					ValaSymbol* _tmp11;
					_tmp11 = NULL;
					return (_tmp11 = vala_parser_parse_delegate_declaration (self, attrs, &inner_error), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp11);
				} while (0); else if (_tmp14 == VALA_TOKEN_TYPE_SIGNAL)
				do {
					ValaSymbol* _tmp12;
					_tmp12 = NULL;
					return (_tmp12 = VALA_SYMBOL (vala_parser_parse_signal_declaration (self, attrs, &inner_error)), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp12);
				} while (0); else
				do {
					ValaSymbol* _tmp13;
					_tmp13 = NULL;
					return (_tmp13 = VALA_SYMBOL (vala_parser_parse_method_declaration (self, attrs, &inner_error)), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp13);
				} while (0);
			} while (0); else if (_tmp19 == VALA_TOKEN_TYPE_ASSIGN || _tmp19 == VALA_TOKEN_TYPE_SEMICOLON)
			do {
				ValaTokenType _tmp17;
				vala_parser_rollback (self, &begin);
				_tmp17 = last_keyword;
				if (_tmp17 == VALA_TOKEN_TYPE_CONST)
				do {
					ValaSymbol* _tmp15;
					_tmp15 = NULL;
					return (_tmp15 = VALA_SYMBOL (vala_parser_parse_constant_declaration (self, attrs, &inner_error)), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp15);
				} while (0); else
				do {
					ValaSymbol* _tmp16;
					_tmp16 = NULL;
					return (_tmp16 = VALA_SYMBOL (vala_parser_parse_field_declaration (self, attrs, &inner_error)), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp16);
				} while (0);
			} while (0); else if (_tmp19 == VALA_TOKEN_TYPE_OPEN_BRACE)
			do {
				ValaSymbol* _tmp18;
				vala_parser_rollback (self, &begin);
				_tmp18 = NULL;
				return (_tmp18 = VALA_SYMBOL (vala_parser_parse_property_declaration (self, attrs, &inner_error)), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp18);
			} while (0);
			break;
		} while (0);
		break;
	} while (0);
	_tmp22 = NULL;
	_tmp23 = NULL;
	inner_error = (_tmp23 = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, (_tmp22 = vala_parser_get_error (self, "expected declaration"))), (_tmp22 = (g_free (_tmp22), NULL)), _tmp23);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
		return NULL;
	}
	(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
}


static void vala_parser_parse_declarations (ValaParser* self, ValaSymbol* parent, gboolean root, GError** error) {
	GError * inner_error;
	g_return_if_fail (VALA_IS_PARSER (self));
	g_return_if_fail (VALA_IS_SYMBOL (parent));
	inner_error = NULL;
	if (!root) {
		vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACE, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			return;
		}
	}
	while (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACE && vala_parser_current (self) != VALA_TOKEN_TYPE_EOF) {
		{
			if (VALA_IS_NAMESPACE (parent)) {
				vala_parser_parse_namespace_member (self, VALA_NAMESPACE (parent), &inner_error);
				if (inner_error != NULL) {
					if (inner_error->domain == VALA_PARSE_ERROR) {
						goto __catch5_vala_parse_error;
					}
					g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
					g_clear_error (&inner_error);
				}
			} else {
				if (VALA_IS_CLASS (parent)) {
					vala_parser_parse_class_member (self, VALA_CLASS (parent), &inner_error);
					if (inner_error != NULL) {
						if (inner_error->domain == VALA_PARSE_ERROR) {
							goto __catch5_vala_parse_error;
						}
						g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
						g_clear_error (&inner_error);
					}
				} else {
					if (VALA_IS_STRUCT (parent)) {
						vala_parser_parse_struct_member (self, VALA_STRUCT (parent), &inner_error);
						if (inner_error != NULL) {
							if (inner_error->domain == VALA_PARSE_ERROR) {
								goto __catch5_vala_parse_error;
							}
							g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
							g_clear_error (&inner_error);
						}
					} else {
						if (VALA_IS_INTERFACE (parent)) {
							vala_parser_parse_interface_member (self, VALA_INTERFACE (parent), &inner_error);
							if (inner_error != NULL) {
								if (inner_error->domain == VALA_PARSE_ERROR) {
									goto __catch5_vala_parse_error;
								}
								g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
								g_clear_error (&inner_error);
							}
						}
					}
				}
			}
		}
		goto __finally5;
		__catch5_vala_parse_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				gint r;
				r = 0;
				while (TRUE) {
					r = ((gint) (vala_parser_recover (self)));
					if (r == VALA_PARSER_RECOVERY_STATE_STATEMENT_BEGIN) {
						vala_parser_next (self);
					} else {
						break;
					}
				}
				if (r == VALA_PARSER_RECOVERY_STATE_EOF) {
					(e == NULL ? NULL : (e = (g_error_free (e), NULL)));
					return;
				}
				(e == NULL ? NULL : (e = (g_error_free (e), NULL)));
			}
		}
		__finally5:
		;
	}
	if (!root) {
		if (!vala_parser_accept (self, VALA_TOKEN_TYPE_CLOSE_BRACE)) {
			/* only report error if it's not a secondary error*/
			if (vala_report_get_errors () == 0) {
				ValaSourceReference* _tmp0;
				_tmp0 = NULL;
				vala_report_error ((_tmp0 = vala_parser_get_current_src (self)), "expected `}'");
				(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
			}
		}
	}
}


static ValaParserRecoveryState vala_parser_recover (ValaParser* self) {
	g_return_val_if_fail (VALA_IS_PARSER (self), 0);
	while (vala_parser_current (self) != VALA_TOKEN_TYPE_EOF) {
		ValaTokenType _tmp2;
		_tmp2 = vala_parser_current (self);
		if (_tmp2 == VALA_TOKEN_TYPE_ABSTRACT || _tmp2 == VALA_TOKEN_TYPE_CLASS || _tmp2 == VALA_TOKEN_TYPE_CONST || _tmp2 == VALA_TOKEN_TYPE_CONSTRUCT || _tmp2 == VALA_TOKEN_TYPE_DELEGATE || _tmp2 == VALA_TOKEN_TYPE_ENUM || _tmp2 == VALA_TOKEN_TYPE_ERRORDOMAIN || _tmp2 == VALA_TOKEN_TYPE_EXTERN || _tmp2 == VALA_TOKEN_TYPE_INLINE || _tmp2 == VALA_TOKEN_TYPE_INTERFACE || _tmp2 == VALA_TOKEN_TYPE_INTERNAL || _tmp2 == VALA_TOKEN_TYPE_NAMESPACE || _tmp2 == VALA_TOKEN_TYPE_OVERRIDE || _tmp2 == VALA_TOKEN_TYPE_PRIVATE || _tmp2 == VALA_TOKEN_TYPE_PROTECTED || _tmp2 == VALA_TOKEN_TYPE_PUBLIC || _tmp2 == VALA_TOKEN_TYPE_SIGNAL || _tmp2 == VALA_TOKEN_TYPE_STATIC || _tmp2 == VALA_TOKEN_TYPE_STRUCT || _tmp2 == VALA_TOKEN_TYPE_VIRTUAL || _tmp2 == VALA_TOKEN_TYPE_VOLATILE)
		do {
			return VALA_PARSER_RECOVERY_STATE_DECLARATION_BEGIN;
		} while (0); else if (_tmp2 == VALA_TOKEN_TYPE_BREAK || _tmp2 == VALA_TOKEN_TYPE_CONTINUE || _tmp2 == VALA_TOKEN_TYPE_DELETE || _tmp2 == VALA_TOKEN_TYPE_DO || _tmp2 == VALA_TOKEN_TYPE_FOR || _tmp2 == VALA_TOKEN_TYPE_FOREACH || _tmp2 == VALA_TOKEN_TYPE_IF || _tmp2 == VALA_TOKEN_TYPE_LOCK || _tmp2 == VALA_TOKEN_TYPE_RETURN || _tmp2 == VALA_TOKEN_TYPE_SWITCH || _tmp2 == VALA_TOKEN_TYPE_THROW || _tmp2 == VALA_TOKEN_TYPE_TRY || _tmp2 == VALA_TOKEN_TYPE_VAR || _tmp2 == VALA_TOKEN_TYPE_WHILE)
		do {
			return VALA_PARSER_RECOVERY_STATE_STATEMENT_BEGIN;
		} while (0); else
		do {
			vala_parser_next (self);
			break;
		} while (0);
	}
	return VALA_PARSER_RECOVERY_STATE_EOF;
}


static ValaNamespace* vala_parser_parse_namespace_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaUnresolvedSymbol* sym;
	ValaSourceReference* _tmp0;
	ValaNamespace* _tmp1;
	ValaNamespace* ns;
	ValaNamespace* _tmp2;
	ValaNamespace* result;
	ValaNamespace* _tmp8;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_NAMESPACE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	ns = (_tmp1 = vala_namespace_new (vala_symbol_get_name (VALA_SYMBOL (sym)), (_tmp0 = vala_parser_get_src_com (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	vala_parser_set_attributes (self, VALA_CODE_NODE (ns), attrs);
	vala_parser_parse_declarations (self, VALA_SYMBOL (ns), FALSE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		(ns == NULL ? NULL : (ns = (vala_code_node_unref (ns), NULL)));
		return NULL;
	}
	_tmp2 = NULL;
	result = (_tmp2 = ns, (_tmp2 == NULL ? NULL : vala_code_node_ref (_tmp2)));
	while (vala_unresolved_symbol_get_inner (sym) != NULL) {
		ValaUnresolvedSymbol* _tmp4;
		ValaUnresolvedSymbol* _tmp3;
		ValaNamespace* _tmp5;
		ValaNamespace* _tmp7;
		ValaNamespace* _tmp6;
		_tmp4 = NULL;
		_tmp3 = NULL;
		sym = (_tmp4 = (_tmp3 = vala_unresolved_symbol_get_inner (sym), (_tmp3 == NULL ? NULL : vala_code_node_ref (_tmp3))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp4);
		_tmp5 = NULL;
		ns = (_tmp5 = vala_namespace_new (vala_symbol_get_name (VALA_SYMBOL (sym)), vala_code_node_get_source_reference (VALA_CODE_NODE (result))), (ns == NULL ? NULL : (ns = (vala_code_node_unref (ns), NULL))), _tmp5);
		vala_namespace_add_namespace (ns, VALA_NAMESPACE (result));
		_tmp7 = NULL;
		_tmp6 = NULL;
		result = (_tmp7 = (_tmp6 = ns, (_tmp6 == NULL ? NULL : vala_code_node_ref (_tmp6))), (result == NULL ? NULL : (result = (vala_code_node_unref (result), NULL))), _tmp7);
	}
	_tmp8 = NULL;
	return (_tmp8 = result, (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), (ns == NULL ? NULL : (ns = (vala_code_node_unref (ns), NULL))), _tmp8);
}


static void vala_parser_parse_namespace_member (ValaParser* self, ValaNamespace* ns, GError** error) {
	GError * inner_error;
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_PARSER (self));
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	inner_error = NULL;
	sym = vala_parser_parse_declaration (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return;
	}
	if (VALA_IS_NAMESPACE (sym)) {
		vala_namespace_add_namespace (ns, VALA_NAMESPACE (sym));
	} else {
		if (VALA_IS_CLASS (sym)) {
			vala_namespace_add_class (ns, VALA_CLASS (sym));
		} else {
			if (VALA_IS_INTERFACE (sym)) {
				vala_namespace_add_interface (ns, VALA_INTERFACE (sym));
			} else {
				if (VALA_IS_STRUCT (sym)) {
					vala_namespace_add_struct (ns, VALA_STRUCT (sym));
				} else {
					if (VALA_IS_ENUM (sym)) {
						vala_namespace_add_enum (ns, VALA_ENUM (sym));
					} else {
						if (VALA_IS_ERROR_DOMAIN (sym)) {
							vala_namespace_add_error_domain (ns, VALA_ERROR_DOMAIN (sym));
						} else {
							if (VALA_IS_DELEGATE (sym)) {
								vala_namespace_add_delegate (ns, VALA_DELEGATE (sym));
							} else {
								if (VALA_IS_METHOD (sym)) {
									ValaMethod* _tmp0;
									ValaMethod* method;
									_tmp0 = NULL;
									method = (_tmp0 = VALA_METHOD (sym), (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0)));
									if (vala_method_get_binding (method) == MEMBER_BINDING_INSTANCE) {
										/* default to static member binding*/
										vala_method_set_binding (method, MEMBER_BINDING_STATIC);
									}
									vala_namespace_add_method (ns, method);
									(method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL)));
								} else {
									if (VALA_IS_FIELD (sym)) {
										ValaField* _tmp1;
										ValaField* field;
										_tmp1 = NULL;
										field = (_tmp1 = VALA_FIELD (sym), (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1)));
										if (vala_field_get_binding (field) == MEMBER_BINDING_INSTANCE) {
											/* default to static member binding*/
											vala_field_set_binding (field, MEMBER_BINDING_STATIC);
										}
										vala_namespace_add_field (ns, field);
										(field == NULL ? NULL : (field = (vala_code_node_unref (field), NULL)));
									} else {
										if (VALA_IS_CONSTANT (sym)) {
											vala_namespace_add_constant (ns, VALA_CONSTANT (sym));
										} else {
											if (sym == NULL) {
												/* workaround for current limitation of exception handling*/
												inner_error = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, "syntax error in declaration");
												if (inner_error != NULL) {
													g_propagate_error (error, inner_error);
													(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
													return;
												}
											} else {
												vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (sym)), "unexpected declaration in namespace");
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	vala_source_file_add_node (vala_scanner_get_source_file (self->priv->scanner), VALA_CODE_NODE (sym));
	(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
}


static void vala_parser_parse_using_directives (ValaParser* self, GError** error) {
	GError * inner_error;
	g_return_if_fail (VALA_IS_PARSER (self));
	inner_error = NULL;
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_USING)) {
		ValaSourceLocation begin;
		begin = vala_parser_get_location (self);
		do {
			ValaSourceLocation begin;
			ValaUnresolvedSymbol* sym;
			ValaSourceReference* _tmp0;
			ValaUsingDirective* _tmp1;
			ValaUsingDirective* ns_ref;
			begin = vala_parser_get_location (self);
			sym = vala_parser_parse_symbol_name (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				return;
			}
			_tmp0 = NULL;
			_tmp1 = NULL;
			ns_ref = (_tmp1 = vala_using_directive_new (VALA_SYMBOL (sym), (_tmp0 = vala_parser_get_src (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
			vala_source_file_add_using_directive (vala_scanner_get_source_file (self->priv->scanner), ns_ref);
			(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
			(ns_ref == NULL ? NULL : (ns_ref = (vala_code_node_unref (ns_ref), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			return;
		}
	}
}


static ValaSymbol* vala_parser_parse_class_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaUnresolvedSymbol* sym;
	GeeList* type_param_list;
	GeeArrayList* base_types;
	ValaSourceReference* _tmp1;
	ValaClass* _tmp2;
	ValaClass* cl;
	ValaSymbol* _tmp5;
	ValaSymbol* result;
	ValaSymbol* _tmp10;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_type_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLASS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	type_param_list = vala_parser_parse_type_parameter_list (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		return NULL;
	}
	base_types = gee_array_list_new (VALA_TYPE_DATA_TYPE, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_COLON)) {
		do {
			ValaDataType* _tmp0;
			_tmp0 = NULL;
			gee_collection_add (GEE_COLLECTION (base_types), (_tmp0 = vala_parser_parse_type (self, TRUE, &inner_error)));
			(_tmp0 == NULL ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	_tmp1 = NULL;
	_tmp2 = NULL;
	cl = (_tmp2 = vala_class_new (vala_symbol_get_name (VALA_SYMBOL (sym)), (_tmp1 = vala_parser_get_src_com (self, &begin))), (_tmp1 == NULL ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL))), _tmp2);
	vala_symbol_set_access (VALA_SYMBOL (cl), access);
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_ABSTRACT)) == (VALA_PARSER_MODIFIER_FLAGS_ABSTRACT)) {
		vala_class_set_is_abstract (cl, TRUE);
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_STATIC)) == (VALA_PARSER_MODIFIER_FLAGS_STATIC)) {
		ValaSourceReference* _tmp3;
		vala_class_set_is_static (cl, TRUE);
		_tmp3 = NULL;
		vala_report_warning ((_tmp3 = vala_parser_get_last_src (self)), "static classes are deprecated, use namespaces");
		(_tmp3 == NULL ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL)));
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_EXTERN)) == (VALA_PARSER_MODIFIER_FLAGS_EXTERN) || vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
		vala_symbol_set_external (VALA_SYMBOL (cl), TRUE);
	}
	vala_parser_set_attributes (self, VALA_CODE_NODE (cl), attrs);
	{
		GeeList* type_param_collection;
		int type_param_it;
		type_param_collection = type_param_list;
		for (type_param_it = 0; type_param_it < gee_collection_get_size (GEE_COLLECTION (type_param_collection)); type_param_it = type_param_it + 1) {
			ValaTypeParameter* type_param;
			type_param = ((ValaTypeParameter*) (gee_list_get (GEE_LIST (type_param_collection), type_param_it)));
			{
				vala_class_add_type_parameter (cl, type_param);
				(type_param == NULL ? NULL : (type_param = (vala_code_node_unref (type_param), NULL)));
			}
		}
	}
	{
		GeeArrayList* base_type_collection;
		int base_type_it;
		base_type_collection = base_types;
		for (base_type_it = 0; base_type_it < gee_collection_get_size (GEE_COLLECTION (base_type_collection)); base_type_it = base_type_it + 1) {
			ValaDataType* base_type;
			base_type = ((ValaDataType*) (gee_list_get (GEE_LIST (base_type_collection), base_type_it)));
			{
				vala_class_add_base_type (cl, base_type);
				(base_type == NULL ? NULL : (base_type = (vala_code_node_unref (base_type), NULL)));
			}
		}
	}
	vala_parser_parse_declarations (self, VALA_SYMBOL (cl), FALSE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		(type_param_list == NULL ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL)));
		(base_types == NULL ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL)));
		(cl == NULL ? NULL : (cl = (vala_code_node_unref (cl), NULL)));
		return NULL;
	}
	/* ensure there is always a default construction method*/
	if (!vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner)) && !vala_class_get_is_static (cl) && !vala_class_get_is_abstract (cl) && vala_class_get_default_construction_method (cl) == NULL) {
		ValaCreationMethod* m;
		ValaBlock* _tmp4;
		m = vala_creation_method_new (vala_symbol_get_name (VALA_SYMBOL (cl)), NULL, vala_code_node_get_source_reference (VALA_CODE_NODE (cl)));
		vala_method_set_binding (VALA_METHOD (m), MEMBER_BINDING_STATIC);
		vala_symbol_set_access (VALA_SYMBOL (m), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
		_tmp4 = NULL;
		vala_method_set_body (VALA_METHOD (m), (_tmp4 = vala_block_new (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)))));
		(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
		vala_class_add_method (cl, VALA_METHOD (m));
		(m == NULL ? NULL : (m = (vala_code_node_unref (m), NULL)));
	}
	_tmp5 = NULL;
	result = (_tmp5 = VALA_SYMBOL (cl), (_tmp5 == NULL ? NULL : vala_code_node_ref (_tmp5)));
	while (vala_unresolved_symbol_get_inner (sym) != NULL) {
		ValaUnresolvedSymbol* _tmp7;
		ValaUnresolvedSymbol* _tmp6;
		ValaNamespace* ns;
		ValaSymbol* _tmp9;
		ValaSymbol* _tmp8;
		_tmp7 = NULL;
		_tmp6 = NULL;
		sym = (_tmp7 = (_tmp6 = vala_unresolved_symbol_get_inner (sym), (_tmp6 == NULL ? NULL : vala_code_node_ref (_tmp6))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp7);
		ns = vala_namespace_new (vala_symbol_get_name (VALA_SYMBOL (sym)), vala_code_node_get_source_reference (VALA_CODE_NODE (cl)));
		if (VALA_IS_NAMESPACE (result)) {
			vala_namespace_add_namespace (ns, VALA_NAMESPACE (result));
		} else {
			vala_namespace_add_class (ns, VALA_CLASS (result));
			vala_source_file_add_node (vala_scanner_get_source_file (self->priv->scanner), VALA_CODE_NODE (result));
		}
		_tmp9 = NULL;
		_tmp8 = NULL;
		result = (_tmp9 = (_tmp8 = VALA_SYMBOL (ns), (_tmp8 == NULL ? NULL : vala_code_node_ref (_tmp8))), (result == NULL ? NULL : (result = (vala_code_node_unref (result), NULL))), _tmp9);
		(ns == NULL ? NULL : (ns = (vala_code_node_unref (ns), NULL)));
	}
	_tmp10 = NULL;
	return (_tmp10 = result, (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), (type_param_list == NULL ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL))), (base_types == NULL ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL))), (cl == NULL ? NULL : (cl = (vala_code_node_unref (cl), NULL))), _tmp10);
}


static void vala_parser_parse_class_member (ValaParser* self, ValaClass* cl, GError** error) {
	GError * inner_error;
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_PARSER (self));
	g_return_if_fail (VALA_IS_CLASS (cl));
	inner_error = NULL;
	sym = vala_parser_parse_declaration (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return;
	}
	if (VALA_IS_CLASS (sym)) {
		vala_class_add_class (cl, VALA_CLASS (sym));
	} else {
		if (VALA_IS_STRUCT (sym)) {
			vala_class_add_struct (cl, VALA_STRUCT (sym));
		} else {
			if (VALA_IS_ENUM (sym)) {
				vala_class_add_enum (cl, VALA_ENUM (sym));
			} else {
				if (VALA_IS_DELEGATE (sym)) {
					vala_class_add_delegate (cl, VALA_DELEGATE (sym));
				} else {
					if (VALA_IS_METHOD (sym)) {
						vala_class_add_method (cl, VALA_METHOD (sym));
					} else {
						if (VALA_IS_SIGNAL (sym)) {
							vala_class_add_signal (cl, VALA_SIGNAL (sym));
						} else {
							if (VALA_IS_FIELD (sym)) {
								vala_class_add_field (cl, VALA_FIELD (sym));
							} else {
								if (VALA_IS_CONSTANT (sym)) {
									vala_class_add_constant (cl, VALA_CONSTANT (sym));
								} else {
									if (VALA_IS_PROPERTY (sym)) {
										vala_class_add_property (cl, VALA_PROPERTY (sym));
									} else {
										if (VALA_IS_CONSTRUCTOR (sym)) {
											ValaConstructor* _tmp0;
											ValaConstructor* c;
											_tmp0 = NULL;
											c = (_tmp0 = VALA_CONSTRUCTOR (sym), (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0)));
											if (vala_constructor_get_binding (c) == MEMBER_BINDING_INSTANCE) {
												vala_class_set_constructor (cl, c);
											} else {
												if (vala_constructor_get_binding (c) == MEMBER_BINDING_CLASS) {
													vala_class_set_class_constructor (cl, c);
												} else {
													vala_class_set_static_constructor (cl, c);
												}
											}
											(c == NULL ? NULL : (c = (vala_code_node_unref (c), NULL)));
										} else {
											if (VALA_IS_DESTRUCTOR (sym)) {
												vala_class_set_destructor (cl, VALA_DESTRUCTOR (sym));
											} else {
												if (sym == NULL) {
													/* workaround for current limitation of exception handling*/
													inner_error = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, "syntax error in declaration");
													if (inner_error != NULL) {
														g_propagate_error (error, inner_error);
														(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
														return;
													}
												} else {
													vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (sym)), "unexpected declaration in class");
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
}


static ValaConstant* vala_parser_parse_constant_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaDataType* type;
	char* id;
	ValaExpression* initializer;
	ValaArrayType* _tmp2;
	ValaDataType* _tmp1;
	ValaArrayType* array_type;
	ValaSourceReference* _tmp3;
	ValaConstant* _tmp4;
	ValaConstant* c;
	ValaConstant* _tmp5;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_CONST, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	type = vala_parser_parse_type (self, FALSE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		return NULL;
	}
	initializer = NULL;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_ASSIGN)) {
		ValaExpression* _tmp0;
		_tmp0 = NULL;
		initializer = (_tmp0 = vala_parser_parse_variable_initializer (self, &inner_error), (initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL))), _tmp0);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			id = (g_free (id), NULL);
			(initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)));
			return NULL;
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		(initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)));
		return NULL;
	}
	/* constant arrays don't own their element*/
	_tmp2 = NULL;
	_tmp1 = NULL;
	array_type = (_tmp2 = (_tmp1 = type, (VALA_IS_ARRAY_TYPE (_tmp1) ? ((ValaArrayType*) (_tmp1)) : NULL)), (_tmp2 == NULL ? NULL : vala_code_node_ref (_tmp2)));
	if (array_type != NULL) {
		vala_data_type_set_value_owned (vala_array_type_get_element_type (array_type), FALSE);
	}
	_tmp3 = NULL;
	_tmp4 = NULL;
	c = (_tmp4 = vala_constant_new (id, type, initializer, (_tmp3 = vala_parser_get_src_com (self, &begin))), (_tmp3 == NULL ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL))), _tmp4);
	vala_symbol_set_access (VALA_SYMBOL (c), access);
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_EXTERN)) == (VALA_PARSER_MODIFIER_FLAGS_EXTERN) || vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
		vala_symbol_set_external (VALA_SYMBOL (c), TRUE);
	}
	vala_parser_set_attributes (self, VALA_CODE_NODE (c), attrs);
	_tmp5 = NULL;
	return (_tmp5 = c, (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), (id = (g_free (id), NULL)), (initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL))), (array_type == NULL ? NULL : (array_type = (vala_code_node_unref (array_type), NULL))), _tmp5);
}


static ValaField* vala_parser_parse_field_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaDataType* type;
	char* id;
	ValaSourceReference* _tmp0;
	ValaField* _tmp1;
	ValaField* f;
	ValaField* _tmp3;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	type = vala_parser_parse_type (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	f = (_tmp1 = vala_field_new (id, type, NULL, (_tmp0 = vala_parser_get_src_com (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	vala_symbol_set_access (VALA_SYMBOL (f), access);
	vala_parser_set_attributes (self, VALA_CODE_NODE (f), attrs);
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_STATIC)) == (VALA_PARSER_MODIFIER_FLAGS_STATIC)) {
		vala_field_set_binding (f, MEMBER_BINDING_STATIC);
	} else {
		if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_CLASS)) == (VALA_PARSER_MODIFIER_FLAGS_CLASS)) {
			vala_field_set_binding (f, MEMBER_BINDING_CLASS);
		}
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_ABSTRACT)) == (VALA_PARSER_MODIFIER_FLAGS_ABSTRACT) || ((flags) & (VALA_PARSER_MODIFIER_FLAGS_VIRTUAL)) == (VALA_PARSER_MODIFIER_FLAGS_VIRTUAL) || ((flags) & (VALA_PARSER_MODIFIER_FLAGS_OVERRIDE)) == (VALA_PARSER_MODIFIER_FLAGS_OVERRIDE)) {
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (f)), "abstract, virtual, and override modifiers are not applicable to fields");
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_EXTERN)) == (VALA_PARSER_MODIFIER_FLAGS_EXTERN) || vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
		vala_symbol_set_external (VALA_SYMBOL (f), TRUE);
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_ASSIGN)) {
		ValaExpression* _tmp2;
		_tmp2 = NULL;
		vala_field_set_initializer (f, (_tmp2 = vala_parser_parse_expression (self, &inner_error)));
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			id = (g_free (id), NULL);
			(f == NULL ? NULL : (f = (vala_code_node_unref (f), NULL)));
			return NULL;
		}
		(_tmp2 == NULL ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		(f == NULL ? NULL : (f = (vala_code_node_unref (f), NULL)));
		return NULL;
	}
	_tmp3 = NULL;
	return (_tmp3 = f, (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), (id = (g_free (id), NULL)), _tmp3);
}


static ValaInitializerList* vala_parser_parse_initializer (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSourceReference* _tmp0;
	ValaInitializerList* _tmp1;
	ValaInitializerList* initializer;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	initializer = (_tmp1 = vala_initializer_list_new ((_tmp0 = vala_parser_get_src (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACE) {
		do {
			ValaExpression* init;
			init = vala_parser_parse_variable_initializer (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)));
				return NULL;
			}
			vala_initializer_list_append (initializer, init);
			(init == NULL ? NULL : (init = (vala_code_node_unref (init), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(initializer == NULL ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)));
		return NULL;
	}
	return initializer;
}


static ValaExpression* vala_parser_parse_variable_initializer (ValaParser* self, GError** error) {
	GError * inner_error;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	if (vala_parser_current (self) == VALA_TOKEN_TYPE_OPEN_BRACE) {
		ValaInitializerList* expr;
		expr = vala_parser_parse_initializer (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			return NULL;
		}
		return VALA_EXPRESSION (expr);
	} else {
		ValaExpression* expr;
		expr = vala_parser_parse_expression (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			return NULL;
		}
		return expr;
	}
}


static ValaMethod* vala_parser_parse_method_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaDataType* type;
	char* id;
	GeeList* _tmp0;
	ValaSourceReference* _tmp1;
	ValaMethod* _tmp2;
	ValaMethod* method;
	ValaMethod* _tmp7;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	type = vala_parser_parse_type (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp0 = vala_parser_parse_type_parameter_list (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		return NULL;
	}
	(_tmp0 == NULL ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)));
	_tmp1 = NULL;
	_tmp2 = NULL;
	method = (_tmp2 = vala_method_new (id, type, (_tmp1 = vala_parser_get_src_com (self, &begin))), (_tmp1 == NULL ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL))), _tmp2);
	vala_symbol_set_access (VALA_SYMBOL (method), access);
	vala_parser_set_attributes (self, VALA_CODE_NODE (method), attrs);
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_STATIC)) == (VALA_PARSER_MODIFIER_FLAGS_STATIC)) {
		vala_method_set_binding (method, MEMBER_BINDING_STATIC);
	} else {
		if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_CLASS)) == (VALA_PARSER_MODIFIER_FLAGS_CLASS)) {
			vala_method_set_binding (method, MEMBER_BINDING_CLASS);
		}
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_ABSTRACT)) == (VALA_PARSER_MODIFIER_FLAGS_ABSTRACT)) {
		vala_method_set_is_abstract (method, TRUE);
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_VIRTUAL)) == (VALA_PARSER_MODIFIER_FLAGS_VIRTUAL)) {
		vala_method_set_is_virtual (method, TRUE);
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_OVERRIDE)) == (VALA_PARSER_MODIFIER_FLAGS_OVERRIDE)) {
		vala_method_set_overrides (method, TRUE);
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_INLINE)) == (VALA_PARSER_MODIFIER_FLAGS_INLINE)) {
		vala_method_set_is_inline (method, TRUE);
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_EXTERN)) == (VALA_PARSER_MODIFIER_FLAGS_EXTERN)) {
		vala_symbol_set_external (VALA_SYMBOL (method), TRUE);
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		(method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL)));
		return NULL;
	}
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
		do {
			ValaFormalParameter* param;
			param = vala_parser_parse_parameter (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
				id = (g_free (id), NULL);
				(method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL)));
				return NULL;
			}
			vala_method_add_parameter (method, param);
			(param == NULL ? NULL : (param = (vala_code_node_unref (param), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		(method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL)));
		return NULL;
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_THROWS)) {
		do {
			ValaDataType* _tmp3;
			_tmp3 = NULL;
			vala_code_node_add_error_type (VALA_CODE_NODE (method), (_tmp3 = vala_parser_parse_type (self, TRUE, &inner_error)));
			(_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_REQUIRES)) {
		ValaExpression* _tmp4;
		vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			id = (g_free (id), NULL);
			(method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL)));
			return NULL;
		}
		_tmp4 = NULL;
		vala_method_add_precondition (method, (_tmp4 = vala_parser_parse_expression (self, &inner_error)));
		(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			id = (g_free (id), NULL);
			(method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL)));
			return NULL;
		}
	}
	while (vala_parser_accept (self, VALA_TOKEN_TYPE_ENSURES)) {
		ValaExpression* _tmp5;
		vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			id = (g_free (id), NULL);
			(method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL)));
			return NULL;
		}
		_tmp5 = NULL;
		vala_method_add_postcondition (method, (_tmp5 = vala_parser_parse_expression (self, &inner_error)));
		(_tmp5 == NULL ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL)));
		vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			id = (g_free (id), NULL);
			(method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL)));
			return NULL;
		}
	}
	if (!vala_parser_accept (self, VALA_TOKEN_TYPE_SEMICOLON)) {
		ValaBlock* _tmp6;
		_tmp6 = NULL;
		vala_method_set_body (method, (_tmp6 = vala_parser_parse_block (self, &inner_error)));
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			id = (g_free (id), NULL);
			(method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL)));
			return NULL;
		}
		(_tmp6 == NULL ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL)));
	} else {
		if (vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
			vala_symbol_set_external (VALA_SYMBOL (method), TRUE);
		}
	}
	_tmp7 = NULL;
	return (_tmp7 = method, (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), (id = (g_free (id), NULL)), _tmp7);
}


static ValaProperty* vala_parser_parse_property_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	gboolean is_weak;
	ValaDataType* type;
	char* id;
	ValaSourceReference* _tmp0;
	ValaProperty* _tmp1;
	ValaProperty* prop;
	ValaProperty* _tmp21;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	is_weak = vala_parser_accept (self, VALA_TOKEN_TYPE_WEAK);
	type = vala_parser_parse_type (self, FALSE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	prop = (_tmp1 = vala_property_new (id, type, NULL, NULL, (_tmp0 = vala_parser_get_src_com (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	vala_symbol_set_access (VALA_SYMBOL (prop), access);
	vala_parser_set_attributes (self, VALA_CODE_NODE (prop), attrs);
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_STATIC)) == (VALA_PARSER_MODIFIER_FLAGS_STATIC)) {
		vala_property_set_binding (prop, MEMBER_BINDING_STATIC);
	} else {
		if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_CLASS)) == (VALA_PARSER_MODIFIER_FLAGS_CLASS)) {
			vala_property_set_binding (prop, MEMBER_BINDING_CLASS);
		}
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_ABSTRACT)) == (VALA_PARSER_MODIFIER_FLAGS_ABSTRACT)) {
		vala_property_set_is_abstract (prop, TRUE);
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_VIRTUAL)) == (VALA_PARSER_MODIFIER_FLAGS_VIRTUAL)) {
		vala_property_set_is_virtual (prop, TRUE);
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_OVERRIDE)) == (VALA_PARSER_MODIFIER_FLAGS_OVERRIDE)) {
		vala_property_set_overrides (prop, TRUE);
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_EXTERN)) == (VALA_PARSER_MODIFIER_FLAGS_EXTERN) || vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
		vala_symbol_set_external (VALA_SYMBOL (prop), TRUE);
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
		return NULL;
	}
	while (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACE) {
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_DEFAULT)) {
			if (vala_property_get_default_expression (prop) != NULL) {
				char* _tmp2;
				GError* _tmp3;
				_tmp2 = NULL;
				_tmp3 = NULL;
				inner_error = (_tmp3 = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, (_tmp2 = vala_parser_get_error (self, "property default value already defined"))), (_tmp2 = (g_free (_tmp2), NULL)), _tmp3);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
					id = (g_free (id), NULL);
					(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
					return NULL;
				}
			}
			if (vala_parser_accept (self, VALA_TOKEN_TYPE_OPEN_PARENS)) {
				ValaSourceReference* _tmp4;
				ValaExpression* _tmp5;
				_tmp4 = NULL;
				vala_report_warning ((_tmp4 = vala_parser_get_last_src (self)), "deprecated syntax, use `default = value;`");
				(_tmp4 == NULL ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL)));
				_tmp5 = NULL;
				vala_property_set_default_expression (prop, (_tmp5 = vala_parser_parse_expression (self, &inner_error)));
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
					id = (g_free (id), NULL);
					(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
					return NULL;
				}
				(_tmp5 == NULL ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL)));
				vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
					id = (g_free (id), NULL);
					(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
					return NULL;
				}
			} else {
				ValaExpression* _tmp6;
				vala_parser_expect (self, VALA_TOKEN_TYPE_ASSIGN, &inner_error);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
					id = (g_free (id), NULL);
					(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
					return NULL;
				}
				_tmp6 = NULL;
				vala_property_set_default_expression (prop, (_tmp6 = vala_parser_parse_expression (self, &inner_error)));
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
					id = (g_free (id), NULL);
					(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
					return NULL;
				}
				(_tmp6 == NULL ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL)));
			}
			vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
				id = (g_free (id), NULL);
				(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
				return NULL;
			}
		} else {
			ValaSourceLocation accessor_begin;
			GeeList* attrs;
			ValaSymbolAccessibility accessor_access;
			accessor_begin = vala_parser_get_location (self);
			attrs = vala_parser_parse_attributes (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
				id = (g_free (id), NULL);
				(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
				return NULL;
			}
			accessor_access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			if (vala_parser_accept (self, VALA_TOKEN_TYPE_GET)) {
				ValaBlock* block;
				ValaPropertyAccessor* _tmp11;
				ValaSourceReference* _tmp10;
				if (vala_property_get_get_accessor (prop) != NULL) {
					char* _tmp7;
					GError* _tmp8;
					_tmp7 = NULL;
					_tmp8 = NULL;
					inner_error = (_tmp8 = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, (_tmp7 = vala_parser_get_error (self, "property get accessor already defined"))), (_tmp7 = (g_free (_tmp7), NULL)), _tmp8);
					if (inner_error != NULL) {
						g_propagate_error (error, inner_error);
						(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
						(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
						id = (g_free (id), NULL);
						(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
						return NULL;
					}
				}
				block = NULL;
				if (!vala_parser_accept (self, VALA_TOKEN_TYPE_SEMICOLON)) {
					ValaBlock* _tmp9;
					_tmp9 = NULL;
					block = (_tmp9 = vala_parser_parse_block (self, &inner_error), (block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL))), _tmp9);
					if (inner_error != NULL) {
						g_propagate_error (error, inner_error);
						(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
						(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
						(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
						id = (g_free (id), NULL);
						(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
						return NULL;
					}
					vala_symbol_set_external (VALA_SYMBOL (prop), FALSE);
				}
				_tmp11 = NULL;
				_tmp10 = NULL;
				vala_property_set_get_accessor (prop, (_tmp11 = vala_property_accessor_new (TRUE, FALSE, FALSE, block, (_tmp10 = vala_parser_get_src (self, &accessor_begin)))));
				(_tmp11 == NULL ? NULL : (_tmp11 = (vala_code_node_unref (_tmp11), NULL)));
				(_tmp10 == NULL ? NULL : (_tmp10 = (vala_source_reference_unref (_tmp10), NULL)));
				vala_parser_set_attributes (self, VALA_CODE_NODE (vala_property_get_get_accessor (prop)), attrs);
				vala_property_accessor_set_access (vala_property_get_get_accessor (prop), accessor_access);
				(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
			} else {
				gboolean writable;
				gboolean _construct;
				ValaBlock* block;
				ValaPropertyAccessor* _tmp18;
				ValaSourceReference* _tmp17;
				writable = FALSE;
				_construct = FALSE;
				if (vala_parser_accept (self, VALA_TOKEN_TYPE_SET)) {
					writable = TRUE;
					_construct = vala_parser_accept (self, VALA_TOKEN_TYPE_CONSTRUCT);
				} else {
					if (vala_parser_accept (self, VALA_TOKEN_TYPE_CONSTRUCT)) {
						_construct = TRUE;
						writable = vala_parser_accept (self, VALA_TOKEN_TYPE_SET);
					} else {
						char* _tmp12;
						GError* _tmp13;
						_tmp12 = NULL;
						_tmp13 = NULL;
						inner_error = (_tmp13 = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, (_tmp12 = vala_parser_get_error (self, "expected get, set, or construct"))), (_tmp12 = (g_free (_tmp12), NULL)), _tmp13);
						if (inner_error != NULL) {
							g_propagate_error (error, inner_error);
							(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
							(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
							id = (g_free (id), NULL);
							(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
							return NULL;
						}
					}
				}
				if (vala_property_get_set_accessor (prop) != NULL) {
					char* _tmp14;
					GError* _tmp15;
					_tmp14 = NULL;
					_tmp15 = NULL;
					inner_error = (_tmp15 = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, (_tmp14 = vala_parser_get_error (self, "property set accessor already defined"))), (_tmp14 = (g_free (_tmp14), NULL)), _tmp15);
					if (inner_error != NULL) {
						g_propagate_error (error, inner_error);
						(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
						(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
						id = (g_free (id), NULL);
						(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
						return NULL;
					}
				}
				block = NULL;
				if (!vala_parser_accept (self, VALA_TOKEN_TYPE_SEMICOLON)) {
					ValaBlock* _tmp16;
					_tmp16 = NULL;
					block = (_tmp16 = vala_parser_parse_block (self, &inner_error), (block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL))), _tmp16);
					if (inner_error != NULL) {
						g_propagate_error (error, inner_error);
						(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
						(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
						(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
						id = (g_free (id), NULL);
						(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
						return NULL;
					}
					vala_symbol_set_external (VALA_SYMBOL (prop), FALSE);
				}
				_tmp18 = NULL;
				_tmp17 = NULL;
				vala_property_set_set_accessor (prop, (_tmp18 = vala_property_accessor_new (FALSE, writable, _construct, block, (_tmp17 = vala_parser_get_src (self, &accessor_begin)))));
				(_tmp18 == NULL ? NULL : (_tmp18 = (vala_code_node_unref (_tmp18), NULL)));
				(_tmp17 == NULL ? NULL : (_tmp17 = (vala_source_reference_unref (_tmp17), NULL)));
				vala_parser_set_attributes (self, VALA_CODE_NODE (vala_property_get_set_accessor (prop)), attrs);
				vala_property_accessor_set_access (vala_property_get_set_accessor (prop), accessor_access);
				(block == NULL ? NULL : (block = (vala_code_node_unref (block), NULL)));
			}
			(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		(prop == NULL ? NULL : (prop = (vala_code_node_unref (prop), NULL)));
		return NULL;
	}
	if (!vala_property_get_is_abstract (prop) && !vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
		gboolean empty_get;
		gboolean empty_set;
		empty_get = (vala_property_get_get_accessor (prop) != NULL && vala_property_accessor_get_body (vala_property_get_get_accessor (prop)) == NULL);
		empty_set = (vala_property_get_set_accessor (prop) != NULL && vala_property_accessor_get_body (vala_property_get_set_accessor (prop)) == NULL);
		if (empty_get != empty_set) {
			if (empty_get) {
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), "property getter must have a body");
			} else {
				if (empty_set) {
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), "property setter must have a body");
				}
			}
			vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
		}
		if (empty_get && empty_set) {
			ValaDataType* field_type;
			ValaField* _tmp20;
			char* _tmp19;
			/* automatic property accessor body generation */
			field_type = vala_data_type_copy (vala_property_get_property_type (prop));
			vala_data_type_set_value_owned (field_type, !is_weak);
			_tmp20 = NULL;
			_tmp19 = NULL;
			vala_property_set_field (prop, (_tmp20 = vala_field_new ((_tmp19 = g_strdup_printf ("_%s", vala_symbol_get_name (VALA_SYMBOL (prop)))), field_type, vala_property_get_default_expression (prop), vala_code_node_get_source_reference (VALA_CODE_NODE (prop)))));
			(_tmp20 == NULL ? NULL : (_tmp20 = (vala_code_node_unref (_tmp20), NULL)));
			_tmp19 = (g_free (_tmp19), NULL);
			vala_symbol_set_access (VALA_SYMBOL (vala_property_get_field (prop)), VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
			(field_type == NULL ? NULL : (field_type = (vala_code_node_unref (field_type), NULL)));
		}
	}
	_tmp21 = NULL;
	return (_tmp21 = prop, (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), (id = (g_free (id), NULL)), _tmp21);
}


static ValaSignal* vala_parser_parse_signal_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaDataType* type;
	char* id;
	ValaSourceReference* _tmp0;
	ValaSignal* _tmp1;
	ValaSignal* sig;
	ValaSignal* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_SIGNAL, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	type = vala_parser_parse_type (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	sig = (_tmp1 = vala_signal_new (id, type, (_tmp0 = vala_parser_get_src_com (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	vala_symbol_set_access (VALA_SYMBOL (sig), access);
	vala_parser_set_attributes (self, VALA_CODE_NODE (sig), attrs);
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_VIRTUAL)) == (VALA_PARSER_MODIFIER_FLAGS_VIRTUAL)) {
		vala_signal_set_is_virtual (sig, TRUE);
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		(sig == NULL ? NULL : (sig = (vala_code_node_unref (sig), NULL)));
		return NULL;
	}
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
		do {
			ValaFormalParameter* param;
			param = vala_parser_parse_parameter (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
				id = (g_free (id), NULL);
				(sig == NULL ? NULL : (sig = (vala_code_node_unref (sig), NULL)));
				return NULL;
			}
			vala_signal_add_parameter (sig, param);
			(param == NULL ? NULL : (param = (vala_code_node_unref (param), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		(sig == NULL ? NULL : (sig = (vala_code_node_unref (sig), NULL)));
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		id = (g_free (id), NULL);
		(sig == NULL ? NULL : (sig = (vala_code_node_unref (sig), NULL)));
		return NULL;
	}
	_tmp2 = NULL;
	return (_tmp2 = sig, (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), (id = (g_free (id), NULL)), _tmp2);
}


static ValaConstructor* vala_parser_parse_constructor_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaParserModifierFlags flags;
	ValaSourceReference* _tmp0;
	ValaConstructor* _tmp1;
	ValaConstructor* c;
	ValaBlock* _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_CONSTRUCT, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	c = (_tmp1 = vala_constructor_new ((_tmp0 = vala_parser_get_src_com (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_STATIC)) == (VALA_PARSER_MODIFIER_FLAGS_STATIC)) {
		vala_constructor_set_binding (c, MEMBER_BINDING_STATIC);
	} else {
		if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_CLASS)) == (VALA_PARSER_MODIFIER_FLAGS_CLASS)) {
			vala_constructor_set_binding (c, MEMBER_BINDING_CLASS);
		}
	}
	_tmp2 = NULL;
	vala_constructor_set_body (c, (_tmp2 = vala_parser_parse_block (self, &inner_error)));
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(c == NULL ? NULL : (c = (vala_code_node_unref (c), NULL)));
		return NULL;
	}
	(_tmp2 == NULL ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)));
	return c;
}


static ValaDestructor* vala_parser_parse_destructor_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	char* _tmp0;
	ValaSourceReference* _tmp1;
	ValaDestructor* _tmp2;
	ValaDestructor* d;
	ValaBlock* _tmp3;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_TILDE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp0 = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp0 = (g_free (_tmp0), NULL);
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp1 = NULL;
	_tmp2 = NULL;
	d = (_tmp2 = vala_destructor_new ((_tmp1 = vala_parser_get_src_com (self, &begin))), (_tmp1 == NULL ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL))), _tmp2);
	_tmp3 = NULL;
	vala_destructor_set_body (d, (_tmp3 = vala_parser_parse_block (self, &inner_error)));
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(d == NULL ? NULL : (d = (vala_code_node_unref (d), NULL)));
		return NULL;
	}
	(_tmp3 == NULL ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)));
	return d;
}


static ValaSymbol* vala_parser_parse_struct_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaUnresolvedSymbol* sym;
	GeeList* type_param_list;
	GeeArrayList* base_types;
	ValaSourceReference* _tmp1;
	ValaStruct* _tmp2;
	ValaStruct* st;
	ValaSymbol* _tmp3;
	ValaSymbol* result;
	ValaSymbol* _tmp8;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_type_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_STRUCT, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	type_param_list = vala_parser_parse_type_parameter_list (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		return NULL;
	}
	base_types = gee_array_list_new (VALA_TYPE_DATA_TYPE, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_COLON)) {
		do {
			ValaDataType* _tmp0;
			_tmp0 = NULL;
			gee_collection_add (GEE_COLLECTION (base_types), (_tmp0 = vala_parser_parse_type (self, TRUE, &inner_error)));
			(_tmp0 == NULL ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	_tmp1 = NULL;
	_tmp2 = NULL;
	st = (_tmp2 = vala_struct_new (vala_symbol_get_name (VALA_SYMBOL (sym)), (_tmp1 = vala_parser_get_src_com (self, &begin))), (_tmp1 == NULL ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL))), _tmp2);
	vala_symbol_set_access (VALA_SYMBOL (st), access);
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_EXTERN)) == (VALA_PARSER_MODIFIER_FLAGS_EXTERN) || vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
		vala_symbol_set_external (VALA_SYMBOL (st), TRUE);
	}
	vala_parser_set_attributes (self, VALA_CODE_NODE (st), attrs);
	{
		GeeList* type_param_collection;
		int type_param_it;
		type_param_collection = type_param_list;
		for (type_param_it = 0; type_param_it < gee_collection_get_size (GEE_COLLECTION (type_param_collection)); type_param_it = type_param_it + 1) {
			ValaTypeParameter* type_param;
			type_param = ((ValaTypeParameter*) (gee_list_get (GEE_LIST (type_param_collection), type_param_it)));
			{
				vala_struct_add_type_parameter (st, type_param);
				(type_param == NULL ? NULL : (type_param = (vala_code_node_unref (type_param), NULL)));
			}
		}
	}
	{
		GeeArrayList* base_type_collection;
		int base_type_it;
		base_type_collection = base_types;
		for (base_type_it = 0; base_type_it < gee_collection_get_size (GEE_COLLECTION (base_type_collection)); base_type_it = base_type_it + 1) {
			ValaDataType* base_type;
			base_type = ((ValaDataType*) (gee_list_get (GEE_LIST (base_type_collection), base_type_it)));
			{
				vala_struct_add_base_type (st, base_type);
				(base_type == NULL ? NULL : (base_type = (vala_code_node_unref (base_type), NULL)));
			}
		}
	}
	vala_parser_parse_declarations (self, VALA_SYMBOL (st), FALSE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		(type_param_list == NULL ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL)));
		(base_types == NULL ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL)));
		(st == NULL ? NULL : (st = (vala_code_node_unref (st), NULL)));
		return NULL;
	}
	_tmp3 = NULL;
	result = (_tmp3 = VALA_SYMBOL (st), (_tmp3 == NULL ? NULL : vala_code_node_ref (_tmp3)));
	while (vala_unresolved_symbol_get_inner (sym) != NULL) {
		ValaUnresolvedSymbol* _tmp5;
		ValaUnresolvedSymbol* _tmp4;
		ValaNamespace* ns;
		ValaSymbol* _tmp7;
		ValaSymbol* _tmp6;
		_tmp5 = NULL;
		_tmp4 = NULL;
		sym = (_tmp5 = (_tmp4 = vala_unresolved_symbol_get_inner (sym), (_tmp4 == NULL ? NULL : vala_code_node_ref (_tmp4))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp5);
		ns = vala_namespace_new (vala_symbol_get_name (VALA_SYMBOL (sym)), vala_code_node_get_source_reference (VALA_CODE_NODE (st)));
		if (VALA_IS_NAMESPACE (result)) {
			vala_namespace_add_namespace (ns, VALA_NAMESPACE (result));
		} else {
			vala_namespace_add_struct (ns, VALA_STRUCT (result));
			vala_source_file_add_node (vala_scanner_get_source_file (self->priv->scanner), VALA_CODE_NODE (result));
		}
		_tmp7 = NULL;
		_tmp6 = NULL;
		result = (_tmp7 = (_tmp6 = VALA_SYMBOL (ns), (_tmp6 == NULL ? NULL : vala_code_node_ref (_tmp6))), (result == NULL ? NULL : (result = (vala_code_node_unref (result), NULL))), _tmp7);
		(ns == NULL ? NULL : (ns = (vala_code_node_unref (ns), NULL)));
	}
	_tmp8 = NULL;
	return (_tmp8 = result, (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), (type_param_list == NULL ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL))), (base_types == NULL ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL))), (st == NULL ? NULL : (st = (vala_code_node_unref (st), NULL))), _tmp8);
}


static void vala_parser_parse_struct_member (ValaParser* self, ValaStruct* st, GError** error) {
	GError * inner_error;
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_PARSER (self));
	g_return_if_fail (VALA_IS_STRUCT (st));
	inner_error = NULL;
	sym = vala_parser_parse_declaration (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return;
	}
	if (VALA_IS_METHOD (sym)) {
		vala_struct_add_method (st, VALA_METHOD (sym));
	} else {
		if (VALA_IS_FIELD (sym)) {
			vala_struct_add_field (st, VALA_FIELD (sym));
		} else {
			if (VALA_IS_CONSTANT (sym)) {
				vala_struct_add_constant (st, VALA_CONSTANT (sym));
			} else {
				if (sym == NULL) {
					/* workaround for current limitation of exception handling*/
					inner_error = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, "syntax error in declaration");
					if (inner_error != NULL) {
						g_propagate_error (error, inner_error);
						(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
						return;
					}
				} else {
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (sym)), "unexpected declaration in struct");
				}
			}
		}
	}
	(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
}


static ValaSymbol* vala_parser_parse_interface_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaUnresolvedSymbol* sym;
	GeeList* type_param_list;
	GeeArrayList* base_types;
	ValaSourceReference* _tmp0;
	ValaInterface* _tmp1;
	ValaInterface* iface;
	ValaSymbol* _tmp2;
	ValaSymbol* result;
	ValaSymbol* _tmp7;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_type_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_INTERFACE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	type_param_list = vala_parser_parse_type_parameter_list (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		return NULL;
	}
	base_types = gee_array_list_new (VALA_TYPE_DATA_TYPE, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_COLON)) {
		do {
			ValaDataType* type;
			type = vala_parser_parse_type (self, TRUE, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
				(type_param_list == NULL ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL)));
				(base_types == NULL ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL)));
				return NULL;
			}
			gee_collection_add (GEE_COLLECTION (base_types), type);
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	iface = (_tmp1 = vala_interface_new (vala_symbol_get_name (VALA_SYMBOL (sym)), (_tmp0 = vala_parser_get_src_com (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	vala_symbol_set_access (VALA_SYMBOL (iface), access);
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_EXTERN)) == (VALA_PARSER_MODIFIER_FLAGS_EXTERN) || vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
		vala_symbol_set_external (VALA_SYMBOL (iface), TRUE);
	}
	vala_parser_set_attributes (self, VALA_CODE_NODE (iface), attrs);
	{
		GeeList* type_param_collection;
		int type_param_it;
		type_param_collection = type_param_list;
		for (type_param_it = 0; type_param_it < gee_collection_get_size (GEE_COLLECTION (type_param_collection)); type_param_it = type_param_it + 1) {
			ValaTypeParameter* type_param;
			type_param = ((ValaTypeParameter*) (gee_list_get (GEE_LIST (type_param_collection), type_param_it)));
			{
				vala_interface_add_type_parameter (iface, type_param);
				(type_param == NULL ? NULL : (type_param = (vala_code_node_unref (type_param), NULL)));
			}
		}
	}
	{
		GeeArrayList* base_type_collection;
		int base_type_it;
		base_type_collection = base_types;
		for (base_type_it = 0; base_type_it < gee_collection_get_size (GEE_COLLECTION (base_type_collection)); base_type_it = base_type_it + 1) {
			ValaDataType* base_type;
			base_type = ((ValaDataType*) (gee_list_get (GEE_LIST (base_type_collection), base_type_it)));
			{
				vala_interface_add_prerequisite (iface, base_type);
				(base_type == NULL ? NULL : (base_type = (vala_code_node_unref (base_type), NULL)));
			}
		}
	}
	vala_parser_parse_declarations (self, VALA_SYMBOL (iface), FALSE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		(type_param_list == NULL ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL)));
		(base_types == NULL ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL)));
		(iface == NULL ? NULL : (iface = (vala_code_node_unref (iface), NULL)));
		return NULL;
	}
	_tmp2 = NULL;
	result = (_tmp2 = VALA_SYMBOL (iface), (_tmp2 == NULL ? NULL : vala_code_node_ref (_tmp2)));
	while (vala_unresolved_symbol_get_inner (sym) != NULL) {
		ValaUnresolvedSymbol* _tmp4;
		ValaUnresolvedSymbol* _tmp3;
		ValaNamespace* ns;
		ValaSymbol* _tmp6;
		ValaSymbol* _tmp5;
		_tmp4 = NULL;
		_tmp3 = NULL;
		sym = (_tmp4 = (_tmp3 = vala_unresolved_symbol_get_inner (sym), (_tmp3 == NULL ? NULL : vala_code_node_ref (_tmp3))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp4);
		ns = vala_namespace_new (vala_symbol_get_name (VALA_SYMBOL (sym)), vala_code_node_get_source_reference (VALA_CODE_NODE (iface)));
		if (VALA_IS_NAMESPACE (result)) {
			vala_namespace_add_namespace (ns, VALA_NAMESPACE (result));
		} else {
			vala_namespace_add_interface (ns, VALA_INTERFACE (result));
			vala_source_file_add_node (vala_scanner_get_source_file (self->priv->scanner), VALA_CODE_NODE (result));
		}
		_tmp6 = NULL;
		_tmp5 = NULL;
		result = (_tmp6 = (_tmp5 = VALA_SYMBOL (ns), (_tmp5 == NULL ? NULL : vala_code_node_ref (_tmp5))), (result == NULL ? NULL : (result = (vala_code_node_unref (result), NULL))), _tmp6);
		(ns == NULL ? NULL : (ns = (vala_code_node_unref (ns), NULL)));
	}
	_tmp7 = NULL;
	return (_tmp7 = result, (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), (type_param_list == NULL ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL))), (base_types == NULL ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL))), (iface == NULL ? NULL : (iface = (vala_code_node_unref (iface), NULL))), _tmp7);
}


static void vala_parser_parse_interface_member (ValaParser* self, ValaInterface* iface, GError** error) {
	GError * inner_error;
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_PARSER (self));
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	inner_error = NULL;
	sym = vala_parser_parse_declaration (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return;
	}
	if (VALA_IS_CLASS (sym)) {
		vala_interface_add_class (iface, VALA_CLASS (sym));
	} else {
		if (VALA_IS_STRUCT (sym)) {
			vala_interface_add_struct (iface, VALA_STRUCT (sym));
		} else {
			if (VALA_IS_ENUM (sym)) {
				vala_interface_add_enum (iface, VALA_ENUM (sym));
			} else {
				if (VALA_IS_DELEGATE (sym)) {
					vala_interface_add_delegate (iface, VALA_DELEGATE (sym));
				} else {
					if (VALA_IS_METHOD (sym)) {
						vala_interface_add_method (iface, VALA_METHOD (sym));
					} else {
						if (VALA_IS_SIGNAL (sym)) {
							vala_interface_add_signal (iface, VALA_SIGNAL (sym));
						} else {
							if (VALA_IS_FIELD (sym)) {
								vala_interface_add_field (iface, VALA_FIELD (sym));
							} else {
								if (VALA_IS_PROPERTY (sym)) {
									vala_interface_add_property (iface, VALA_PROPERTY (sym));
								} else {
									if (sym == NULL) {
										/* workaround for current limitation of exception handling*/
										inner_error = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, "syntax error in declaration");
										if (inner_error != NULL) {
											g_propagate_error (error, inner_error);
											(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
											return;
										}
									} else {
										vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (sym)), "unexpected declaration in interface");
									}
								}
							}
						}
					}
				}
			}
		}
	}
	(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
}


static ValaSymbol* vala_parser_parse_enum_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaUnresolvedSymbol* sym;
	ValaSourceReference* _tmp0;
	ValaEnum* _tmp1;
	ValaEnum* en;
	ValaSymbol* _tmp5;
	ValaSymbol* result;
	ValaSymbol* _tmp10;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_type_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_ENUM, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	en = (_tmp1 = vala_enum_new (vala_symbol_get_name (VALA_SYMBOL (sym)), (_tmp0 = vala_parser_get_src_com (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	vala_symbol_set_access (VALA_SYMBOL (en), access);
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_EXTERN)) == (VALA_PARSER_MODIFIER_FLAGS_EXTERN) || vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
		vala_symbol_set_external (VALA_SYMBOL (en), TRUE);
	}
	vala_parser_set_attributes (self, VALA_CODE_NODE (en), attrs);
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		(en == NULL ? NULL : (en = (vala_code_node_unref (en), NULL)));
		return NULL;
	}
	do {
		GeeList* value_attrs;
		ValaSourceLocation value_begin;
		char* id;
		ValaSourceReference* _tmp2;
		ValaEnumValue* _tmp3;
		ValaEnumValue* ev;
		if (vala_parser_current (self) == VALA_TOKEN_TYPE_CLOSE_BRACE) {
			/* allow trailing comma*/
			break;
		}
		value_attrs = vala_parser_parse_attributes (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
			(en == NULL ? NULL : (en = (vala_code_node_unref (en), NULL)));
			return NULL;
		}
		value_begin = vala_parser_get_location (self);
		id = vala_parser_parse_identifier (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(value_attrs == NULL ? NULL : (value_attrs = (gee_collection_object_unref (value_attrs), NULL)));
			(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
			(en == NULL ? NULL : (en = (vala_code_node_unref (en), NULL)));
			return NULL;
		}
		_tmp2 = NULL;
		_tmp3 = NULL;
		ev = (_tmp3 = vala_enum_value_new (id, (_tmp2 = vala_parser_get_src (self, &value_begin))), (_tmp2 == NULL ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL))), _tmp3);
		vala_parser_set_attributes (self, VALA_CODE_NODE (ev), value_attrs);
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_ASSIGN)) {
			ValaExpression* _tmp4;
			_tmp4 = NULL;
			vala_enum_value_set_value (ev, (_tmp4 = vala_parser_parse_expression (self, &inner_error)));
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(value_attrs == NULL ? NULL : (value_attrs = (gee_collection_object_unref (value_attrs), NULL)));
				id = (g_free (id), NULL);
				(ev == NULL ? NULL : (ev = (vala_code_node_unref (ev), NULL)));
				(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
				(en == NULL ? NULL : (en = (vala_code_node_unref (en), NULL)));
				return NULL;
			}
			(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
		}
		vala_enum_add_value (en, ev);
		(value_attrs == NULL ? NULL : (value_attrs = (gee_collection_object_unref (value_attrs), NULL)));
		id = (g_free (id), NULL);
		(ev == NULL ? NULL : (ev = (vala_code_node_unref (ev), NULL)));
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_SEMICOLON)) {
		while (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACE) {
			ValaSymbol* sym;
			sym = vala_parser_parse_declaration (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
				(en == NULL ? NULL : (en = (vala_code_node_unref (en), NULL)));
				return NULL;
			}
			if (VALA_IS_METHOD (sym)) {
				vala_enum_add_method (en, VALA_METHOD (sym));
			} else {
				if (sym == NULL) {
					/* workaround for current limitation of exception handling*/
					inner_error = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, "syntax error in declaration");
					if (inner_error != NULL) {
						g_propagate_error (error, inner_error);
						(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
						(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
						(en == NULL ? NULL : (en = (vala_code_node_unref (en), NULL)));
						return NULL;
					}
				} else {
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (sym)), "unexpected declaration in enum");
				}
			}
			(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		(en == NULL ? NULL : (en = (vala_code_node_unref (en), NULL)));
		return NULL;
	}
	_tmp5 = NULL;
	result = (_tmp5 = VALA_SYMBOL (en), (_tmp5 == NULL ? NULL : vala_code_node_ref (_tmp5)));
	while (vala_unresolved_symbol_get_inner (sym) != NULL) {
		ValaUnresolvedSymbol* _tmp7;
		ValaUnresolvedSymbol* _tmp6;
		ValaNamespace* ns;
		ValaSymbol* _tmp9;
		ValaSymbol* _tmp8;
		_tmp7 = NULL;
		_tmp6 = NULL;
		sym = (_tmp7 = (_tmp6 = vala_unresolved_symbol_get_inner (sym), (_tmp6 == NULL ? NULL : vala_code_node_ref (_tmp6))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp7);
		ns = vala_namespace_new (vala_symbol_get_name (VALA_SYMBOL (sym)), vala_code_node_get_source_reference (VALA_CODE_NODE (en)));
		if (VALA_IS_NAMESPACE (result)) {
			vala_namespace_add_namespace (ns, VALA_NAMESPACE (result));
		} else {
			vala_namespace_add_enum (ns, VALA_ENUM (result));
			vala_source_file_add_node (vala_scanner_get_source_file (self->priv->scanner), VALA_CODE_NODE (result));
		}
		_tmp9 = NULL;
		_tmp8 = NULL;
		result = (_tmp9 = (_tmp8 = VALA_SYMBOL (ns), (_tmp8 == NULL ? NULL : vala_code_node_ref (_tmp8))), (result == NULL ? NULL : (result = (vala_code_node_unref (result), NULL))), _tmp9);
		(ns == NULL ? NULL : (ns = (vala_code_node_unref (ns), NULL)));
	}
	_tmp10 = NULL;
	return (_tmp10 = result, (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), (en == NULL ? NULL : (en = (vala_code_node_unref (en), NULL))), _tmp10);
}


static ValaSymbol* vala_parser_parse_errordomain_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaUnresolvedSymbol* sym;
	ValaSourceReference* _tmp0;
	ValaErrorDomain* _tmp1;
	ValaErrorDomain* ed;
	ValaSymbol* _tmp5;
	ValaSymbol* result;
	ValaSymbol* _tmp10;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_type_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_ERRORDOMAIN, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	ed = (_tmp1 = vala_error_domain_new (vala_symbol_get_name (VALA_SYMBOL (sym)), (_tmp0 = vala_parser_get_src_com (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	vala_symbol_set_access (VALA_SYMBOL (ed), access);
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_EXTERN)) == (VALA_PARSER_MODIFIER_FLAGS_EXTERN) || vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
		vala_symbol_set_external (VALA_SYMBOL (ed), TRUE);
	}
	vala_parser_set_attributes (self, VALA_CODE_NODE (ed), attrs);
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_BRACE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		(ed == NULL ? NULL : (ed = (vala_code_node_unref (ed), NULL)));
		return NULL;
	}
	do {
		GeeList* code_attrs;
		ValaSourceLocation code_begin;
		char* id;
		ValaSourceReference* _tmp2;
		ValaErrorCode* _tmp3;
		ValaErrorCode* ec;
		if (vala_parser_current (self) == VALA_TOKEN_TYPE_CLOSE_BRACE) {
			/* allow trailing comma*/
			break;
		}
		code_attrs = vala_parser_parse_attributes (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
			(ed == NULL ? NULL : (ed = (vala_code_node_unref (ed), NULL)));
			return NULL;
		}
		code_begin = vala_parser_get_location (self);
		id = vala_parser_parse_identifier (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(code_attrs == NULL ? NULL : (code_attrs = (gee_collection_object_unref (code_attrs), NULL)));
			(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
			(ed == NULL ? NULL : (ed = (vala_code_node_unref (ed), NULL)));
			return NULL;
		}
		_tmp2 = NULL;
		_tmp3 = NULL;
		ec = (_tmp3 = vala_error_code_new (id, (_tmp2 = vala_parser_get_src (self, &code_begin))), (_tmp2 == NULL ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL))), _tmp3);
		vala_parser_set_attributes (self, VALA_CODE_NODE (ec), code_attrs);
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_ASSIGN)) {
			ValaExpression* _tmp4;
			_tmp4 = NULL;
			vala_error_code_set_value (ec, (_tmp4 = vala_parser_parse_expression (self, &inner_error)));
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(code_attrs == NULL ? NULL : (code_attrs = (gee_collection_object_unref (code_attrs), NULL)));
				id = (g_free (id), NULL);
				(ec == NULL ? NULL : (ec = (vala_code_node_unref (ec), NULL)));
				(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
				(ed == NULL ? NULL : (ed = (vala_code_node_unref (ed), NULL)));
				return NULL;
			}
			(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
		}
		vala_error_domain_add_code (ed, ec);
		(code_attrs == NULL ? NULL : (code_attrs = (gee_collection_object_unref (code_attrs), NULL)));
		id = (g_free (id), NULL);
		(ec == NULL ? NULL : (ec = (vala_code_node_unref (ec), NULL)));
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_SEMICOLON)) {
		while (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_BRACE) {
			ValaSymbol* sym;
			sym = vala_parser_parse_declaration (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
				(ed == NULL ? NULL : (ed = (vala_code_node_unref (ed), NULL)));
				return NULL;
			}
			if (VALA_IS_METHOD (sym)) {
				vala_error_domain_add_method (ed, VALA_METHOD (sym));
			} else {
				if (sym == NULL) {
					/* workaround for current limitation of exception handling*/
					inner_error = g_error_new (VALA_PARSE_ERROR, VALA_PARSE_ERROR_SYNTAX, "syntax error in declaration");
					if (inner_error != NULL) {
						g_propagate_error (error, inner_error);
						(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
						(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
						(ed == NULL ? NULL : (ed = (vala_code_node_unref (ed), NULL)));
						return NULL;
					}
				} else {
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (sym)), "unexpected declaration in errordomain");
				}
			}
			(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_BRACE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		(ed == NULL ? NULL : (ed = (vala_code_node_unref (ed), NULL)));
		return NULL;
	}
	_tmp5 = NULL;
	result = (_tmp5 = VALA_SYMBOL (ed), (_tmp5 == NULL ? NULL : vala_code_node_ref (_tmp5)));
	while (vala_unresolved_symbol_get_inner (sym) != NULL) {
		ValaUnresolvedSymbol* _tmp7;
		ValaUnresolvedSymbol* _tmp6;
		ValaNamespace* ns;
		ValaSymbol* _tmp9;
		ValaSymbol* _tmp8;
		_tmp7 = NULL;
		_tmp6 = NULL;
		sym = (_tmp7 = (_tmp6 = vala_unresolved_symbol_get_inner (sym), (_tmp6 == NULL ? NULL : vala_code_node_ref (_tmp6))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp7);
		ns = vala_namespace_new (vala_symbol_get_name (VALA_SYMBOL (sym)), vala_code_node_get_source_reference (VALA_CODE_NODE (ed)));
		if (VALA_IS_NAMESPACE (result)) {
			vala_namespace_add_namespace (ns, VALA_NAMESPACE (result));
		} else {
			vala_namespace_add_error_domain (ns, VALA_ERROR_DOMAIN (result));
			vala_source_file_add_node (vala_scanner_get_source_file (self->priv->scanner), VALA_CODE_NODE (result));
		}
		_tmp9 = NULL;
		_tmp8 = NULL;
		result = (_tmp9 = (_tmp8 = VALA_SYMBOL (ns), (_tmp8 == NULL ? NULL : vala_code_node_ref (_tmp8))), (result == NULL ? NULL : (result = (vala_code_node_unref (result), NULL))), _tmp9);
		(ns == NULL ? NULL : (ns = (vala_code_node_unref (ns), NULL)));
	}
	_tmp10 = NULL;
	return (_tmp10 = result, (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), (ed == NULL ? NULL : (ed = (vala_code_node_unref (ed), NULL))), _tmp10);
}


static ValaSymbolAccessibility vala_parser_parse_access_modifier (ValaParser* self, ValaSymbolAccessibility default_access) {
	ValaTokenType _tmp5;
	g_return_val_if_fail (VALA_IS_PARSER (self), 0);
	_tmp5 = vala_parser_current (self);
	if (_tmp5 == VALA_TOKEN_TYPE_PRIVATE)
	do {
		vala_parser_next (self);
		return VALA_SYMBOL_ACCESSIBILITY_PRIVATE;
	} while (0); else if (_tmp5 == VALA_TOKEN_TYPE_PROTECTED)
	do {
		vala_parser_next (self);
		return VALA_SYMBOL_ACCESSIBILITY_PROTECTED;
	} while (0); else if (_tmp5 == VALA_TOKEN_TYPE_INTERNAL)
	do {
		vala_parser_next (self);
		return VALA_SYMBOL_ACCESSIBILITY_INTERNAL;
	} while (0); else if (_tmp5 == VALA_TOKEN_TYPE_PUBLIC)
	do {
		vala_parser_next (self);
		return VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
	} while (0); else
	do {
		return default_access;
	} while (0);
}


static ValaParserModifierFlags vala_parser_parse_type_declaration_modifiers (ValaParser* self) {
	ValaParserModifierFlags flags;
	g_return_val_if_fail (VALA_IS_PARSER (self), 0);
	flags = 0;
	while (TRUE) {
		ValaTokenType _tmp1;
		_tmp1 = vala_parser_current (self);
		if (_tmp1 == VALA_TOKEN_TYPE_ABSTRACT)
		do {
			vala_parser_next (self);
			flags = flags | (VALA_PARSER_MODIFIER_FLAGS_ABSTRACT);
			break;
		} while (0); else if (_tmp1 == VALA_TOKEN_TYPE_EXTERN)
		do {
			vala_parser_next (self);
			flags = flags | (VALA_PARSER_MODIFIER_FLAGS_EXTERN);
			break;
		} while (0); else if (_tmp1 == VALA_TOKEN_TYPE_STATIC)
		do {
			vala_parser_next (self);
			flags = flags | (VALA_PARSER_MODIFIER_FLAGS_STATIC);
			break;
		} while (0); else
		do {
			return flags;
		} while (0);
	}
	return flags;
}


static ValaParserModifierFlags vala_parser_parse_member_declaration_modifiers (ValaParser* self) {
	ValaParserModifierFlags flags;
	g_return_val_if_fail (VALA_IS_PARSER (self), 0);
	flags = 0;
	while (TRUE) {
		ValaTokenType _tmp1;
		_tmp1 = vala_parser_current (self);
		if (_tmp1 == VALA_TOKEN_TYPE_ABSTRACT)
		do {
			vala_parser_next (self);
			flags = flags | (VALA_PARSER_MODIFIER_FLAGS_ABSTRACT);
			break;
		} while (0); else if (_tmp1 == VALA_TOKEN_TYPE_CLASS)
		do {
			vala_parser_next (self);
			flags = flags | (VALA_PARSER_MODIFIER_FLAGS_CLASS);
			break;
		} while (0); else if (_tmp1 == VALA_TOKEN_TYPE_EXTERN)
		do {
			vala_parser_next (self);
			flags = flags | (VALA_PARSER_MODIFIER_FLAGS_EXTERN);
			break;
		} while (0); else if (_tmp1 == VALA_TOKEN_TYPE_INLINE)
		do {
			vala_parser_next (self);
			flags = flags | (VALA_PARSER_MODIFIER_FLAGS_INLINE);
			break;
		} while (0); else if (_tmp1 == VALA_TOKEN_TYPE_OVERRIDE)
		do {
			vala_parser_next (self);
			flags = flags | (VALA_PARSER_MODIFIER_FLAGS_OVERRIDE);
			break;
		} while (0); else if (_tmp1 == VALA_TOKEN_TYPE_STATIC)
		do {
			vala_parser_next (self);
			flags = flags | (VALA_PARSER_MODIFIER_FLAGS_STATIC);
			break;
		} while (0); else if (_tmp1 == VALA_TOKEN_TYPE_VIRTUAL)
		do {
			vala_parser_next (self);
			flags = flags | (VALA_PARSER_MODIFIER_FLAGS_VIRTUAL);
			break;
		} while (0); else
		do {
			return flags;
		} while (0);
	}
	return flags;
}


static ValaFormalParameter* vala_parser_parse_parameter (ValaParser* self, GError** error) {
	GError * inner_error;
	GeeList* attrs;
	ValaSourceLocation begin;
	gboolean construct_param;
	ValaParameterDirection direction;
	ValaDataType* type;
	char* id;
	ValaSourceReference* _tmp6;
	ValaFormalParameter* _tmp7;
	ValaFormalParameter* param;
	ValaFormalParameter* _tmp9;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	attrs = vala_parser_parse_attributes (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	begin = vala_parser_get_location (self);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_ELLIPSIS)) {
		ValaSourceReference* _tmp0;
		ValaFormalParameter* _tmp1;
		ValaFormalParameter* _tmp2;
		/* varargs*/
		_tmp0 = NULL;
		_tmp1 = NULL;
		_tmp2 = NULL;
		return (_tmp2 = (_tmp1 = vala_formal_parameter_new_with_ellipsis ((_tmp0 = vala_parser_get_src (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1), (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), _tmp2);
	}
	construct_param = FALSE;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_CONSTRUCT)) {
		ValaSourceReference* _tmp3;
		_tmp3 = NULL;
		vala_report_warning ((_tmp3 = vala_parser_get_last_src (self)), "deprecated syntax, use assignments in the method body");
		(_tmp3 == NULL ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL)));
		construct_param = TRUE;
	}
	direction = VALA_PARAMETER_DIRECTION_IN;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OUT)) {
		direction = VALA_PARAMETER_DIRECTION_OUT;
	} else {
		if (vala_parser_accept (self, VALA_TOKEN_TYPE_REF)) {
			direction = VALA_PARAMETER_DIRECTION_REF;
		}
	}
	type = NULL;
	if (direction == VALA_PARAMETER_DIRECTION_IN) {
		ValaDataType* _tmp4;
		/* in parameters are weak by default*/
		_tmp4 = NULL;
		type = (_tmp4 = vala_parser_parse_type (self, FALSE, &inner_error), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp4);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			return NULL;
		}
	} else {
		ValaDataType* _tmp5;
		/* out parameters own the value by default*/
		_tmp5 = NULL;
		type = (_tmp5 = vala_parser_parse_type (self, TRUE, &inner_error), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), _tmp5);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			return NULL;
		}
	}
	id = vala_parser_parse_identifier (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		return NULL;
	}
	_tmp6 = NULL;
	_tmp7 = NULL;
	param = (_tmp7 = vala_formal_parameter_new (id, type, (_tmp6 = vala_parser_get_src (self, &begin))), (_tmp6 == NULL ? NULL : (_tmp6 = (vala_source_reference_unref (_tmp6), NULL))), _tmp7);
	vala_parser_set_attributes (self, VALA_CODE_NODE (param), attrs);
	vala_formal_parameter_set_direction (param, direction);
	vala_formal_parameter_set_construct_parameter (param, construct_param);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_ASSIGN)) {
		ValaExpression* _tmp8;
		_tmp8 = NULL;
		vala_formal_parameter_set_default_expression (param, (_tmp8 = vala_parser_parse_expression (self, &inner_error)));
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL)));
			(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			id = (g_free (id), NULL);
			(param == NULL ? NULL : (param = (vala_code_node_unref (param), NULL)));
			return NULL;
		}
		(_tmp8 == NULL ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL)));
	}
	_tmp9 = NULL;
	return (_tmp9 = param, (attrs == NULL ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL))), (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), (id = (g_free (id), NULL)), _tmp9);
}


static ValaCreationMethod* vala_parser_parse_creation_method_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaUnresolvedSymbol* sym;
	ValaCreationMethod* method;
	ValaCreationMethod* _tmp6;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	method = NULL;
	if (vala_unresolved_symbol_get_inner (sym) == NULL) {
		ValaCreationMethod* _tmp1;
		ValaSourceReference* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		method = (_tmp1 = vala_creation_method_new (vala_symbol_get_name (VALA_SYMBOL (sym)), NULL, (_tmp0 = vala_parser_get_src_com (self, &begin))), (method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL))), _tmp1);
		(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
	} else {
		ValaCreationMethod* _tmp3;
		ValaSourceReference* _tmp2;
		_tmp3 = NULL;
		_tmp2 = NULL;
		method = (_tmp3 = vala_creation_method_new (vala_symbol_get_name (VALA_SYMBOL (vala_unresolved_symbol_get_inner (sym))), vala_symbol_get_name (VALA_SYMBOL (sym)), (_tmp2 = vala_parser_get_src_com (self, &begin))), (method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL))), _tmp3);
		(_tmp2 == NULL ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)));
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_EXTERN)) == (VALA_PARSER_MODIFIER_FLAGS_EXTERN)) {
		vala_symbol_set_external (VALA_SYMBOL (method), TRUE);
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		(method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL)));
		return NULL;
	}
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
		do {
			ValaFormalParameter* param;
			param = vala_parser_parse_parameter (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
				(method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL)));
				return NULL;
			}
			vala_method_add_parameter (VALA_METHOD (method), param);
			(param == NULL ? NULL : (param = (vala_code_node_unref (param), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		(method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL)));
		return NULL;
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_THROWS)) {
		do {
			ValaDataType* _tmp4;
			_tmp4 = NULL;
			vala_code_node_add_error_type (VALA_CODE_NODE (method), (_tmp4 = vala_parser_parse_type (self, TRUE, &inner_error)));
			(_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_symbol_set_access (VALA_SYMBOL (method), access);
	vala_parser_set_attributes (self, VALA_CODE_NODE (method), attrs);
	vala_method_set_binding (VALA_METHOD (method), MEMBER_BINDING_STATIC);
	if (!vala_parser_accept (self, VALA_TOKEN_TYPE_SEMICOLON)) {
		ValaBlock* _tmp5;
		_tmp5 = NULL;
		vala_method_set_body (VALA_METHOD (method), (_tmp5 = vala_parser_parse_block (self, &inner_error)));
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
			(method == NULL ? NULL : (method = (vala_code_node_unref (method), NULL)));
			return NULL;
		}
		(_tmp5 == NULL ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL)));
	} else {
		if (vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
			vala_symbol_set_external (VALA_SYMBOL (method), TRUE);
		}
	}
	_tmp6 = NULL;
	return (_tmp6 = method, (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp6);
}


static ValaSymbol* vala_parser_parse_delegate_declaration (ValaParser* self, GeeList* attrs, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaSymbolAccessibility access;
	ValaParserModifierFlags flags;
	ValaDataType* type;
	ValaUnresolvedSymbol* sym;
	GeeList* type_param_list;
	ValaSourceReference* _tmp0;
	ValaDelegate* _tmp1;
	ValaDelegate* d;
	ValaSymbol* _tmp3;
	ValaSymbol* result;
	ValaSymbol* _tmp8;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	g_return_val_if_fail (attrs == NULL || GEE_IS_LIST (attrs), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	access = vala_parser_parse_access_modifier (self, VALA_SYMBOL_ACCESSIBILITY_PRIVATE);
	flags = vala_parser_parse_member_declaration_modifiers (self);
	vala_parser_expect (self, VALA_TOKEN_TYPE_DELEGATE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	type = vala_parser_parse_type (self, TRUE, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	sym = vala_parser_parse_symbol_name (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		return NULL;
	}
	type_param_list = vala_parser_parse_type_parameter_list (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		return NULL;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	d = (_tmp1 = vala_delegate_new (vala_symbol_get_name (VALA_SYMBOL (sym)), type, (_tmp0 = vala_parser_get_src_com (self, &begin))), (_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL))), _tmp1);
	vala_symbol_set_access (VALA_SYMBOL (d), access);
	vala_parser_set_attributes (self, VALA_CODE_NODE (d), attrs);
	if (!(((flags) & (VALA_PARSER_MODIFIER_FLAGS_STATIC)) == (VALA_PARSER_MODIFIER_FLAGS_STATIC))) {
		vala_delegate_set_has_target (d, TRUE);
	}
	if (((flags) & (VALA_PARSER_MODIFIER_FLAGS_EXTERN)) == (VALA_PARSER_MODIFIER_FLAGS_EXTERN) || vala_source_file_get_external_package (vala_scanner_get_source_file (self->priv->scanner))) {
		vala_symbol_set_external (VALA_SYMBOL (d), TRUE);
	}
	{
		GeeList* type_param_collection;
		int type_param_it;
		type_param_collection = type_param_list;
		for (type_param_it = 0; type_param_it < gee_collection_get_size (GEE_COLLECTION (type_param_collection)); type_param_it = type_param_it + 1) {
			ValaTypeParameter* type_param;
			type_param = ((ValaTypeParameter*) (gee_list_get (GEE_LIST (type_param_collection), type_param_it)));
			{
				vala_delegate_add_type_parameter (d, type_param);
				(type_param == NULL ? NULL : (type_param = (vala_code_node_unref (type_param), NULL)));
			}
		}
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_OPEN_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		(type_param_list == NULL ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL)));
		(d == NULL ? NULL : (d = (vala_code_node_unref (d), NULL)));
		return NULL;
	}
	if (vala_parser_current (self) != VALA_TOKEN_TYPE_CLOSE_PARENS) {
		do {
			ValaFormalParameter* param;
			param = vala_parser_parse_parameter (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
				(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
				(type_param_list == NULL ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL)));
				(d == NULL ? NULL : (d = (vala_code_node_unref (d), NULL)));
				return NULL;
			}
			vala_delegate_add_parameter (d, param);
			(param == NULL ? NULL : (param = (vala_code_node_unref (param), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_CLOSE_PARENS, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		(type_param_list == NULL ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL)));
		(d == NULL ? NULL : (d = (vala_code_node_unref (d), NULL)));
		return NULL;
	}
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_THROWS)) {
		do {
			ValaDataType* _tmp2;
			_tmp2 = NULL;
			_tmp2 = vala_parser_parse_type (self, TRUE, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
				(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
				(type_param_list == NULL ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL)));
				(d == NULL ? NULL : (d = (vala_code_node_unref (d), NULL)));
				return NULL;
			}
			(_tmp2 == NULL ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)));
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
	}
	vala_parser_expect (self, VALA_TOKEN_TYPE_SEMICOLON, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
		(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
		(type_param_list == NULL ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL)));
		(d == NULL ? NULL : (d = (vala_code_node_unref (d), NULL)));
		return NULL;
	}
	_tmp3 = NULL;
	result = (_tmp3 = VALA_SYMBOL (d), (_tmp3 == NULL ? NULL : vala_code_node_ref (_tmp3)));
	while (vala_unresolved_symbol_get_inner (sym) != NULL) {
		ValaUnresolvedSymbol* _tmp5;
		ValaUnresolvedSymbol* _tmp4;
		ValaNamespace* ns;
		ValaSymbol* _tmp7;
		ValaSymbol* _tmp6;
		_tmp5 = NULL;
		_tmp4 = NULL;
		sym = (_tmp5 = (_tmp4 = vala_unresolved_symbol_get_inner (sym), (_tmp4 == NULL ? NULL : vala_code_node_ref (_tmp4))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp5);
		ns = vala_namespace_new (vala_symbol_get_name (VALA_SYMBOL (sym)), vala_code_node_get_source_reference (VALA_CODE_NODE (d)));
		if (VALA_IS_NAMESPACE (result)) {
			vala_namespace_add_namespace (ns, VALA_NAMESPACE (result));
		} else {
			vala_namespace_add_delegate (ns, VALA_DELEGATE (result));
			vala_source_file_add_node (vala_scanner_get_source_file (self->priv->scanner), VALA_CODE_NODE (result));
		}
		_tmp7 = NULL;
		_tmp6 = NULL;
		result = (_tmp7 = (_tmp6 = VALA_SYMBOL (ns), (_tmp6 == NULL ? NULL : vala_code_node_ref (_tmp6))), (result == NULL ? NULL : (result = (vala_code_node_unref (result), NULL))), _tmp7);
		(ns == NULL ? NULL : (ns = (vala_code_node_unref (ns), NULL)));
	}
	_tmp8 = NULL;
	return (_tmp8 = result, (type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL))), (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), (type_param_list == NULL ? NULL : (type_param_list = (gee_collection_object_unref (type_param_list), NULL))), (d == NULL ? NULL : (d = (vala_code_node_unref (d), NULL))), _tmp8);
}


static GeeList* vala_parser_parse_type_parameter_list (ValaParser* self, GError** error) {
	GError * inner_error;
	GeeArrayList* list;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	list = gee_array_list_new (VALA_TYPE_TYPEPARAMETER, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OP_LT)) {
		do {
			ValaSourceLocation begin;
			char* id;
			ValaTypeParameter* _tmp1;
			ValaSourceReference* _tmp0;
			begin = vala_parser_get_location (self);
			id = vala_parser_parse_identifier (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				(list == NULL ? NULL : (list = (gee_collection_object_unref (list), NULL)));
				return NULL;
			}
			_tmp1 = NULL;
			_tmp0 = NULL;
			gee_collection_add (GEE_COLLECTION (list), (_tmp1 = vala_typeparameter_new (id, (_tmp0 = vala_parser_get_src (self, &begin)))));
			(_tmp1 == NULL ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL)));
			(_tmp0 == NULL ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)));
			id = (g_free (id), NULL);
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_OP_GT, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(list == NULL ? NULL : (list = (gee_collection_object_unref (list), NULL)));
			return NULL;
		}
	}
	return GEE_LIST (list);
}


static void vala_parser_skip_type_argument_list (ValaParser* self, GError** error) {
	GError * inner_error;
	g_return_if_fail (VALA_IS_PARSER (self));
	inner_error = NULL;
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OP_LT)) {
		do {
			vala_parser_skip_type (self, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				return;
			}
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		vala_parser_expect (self, VALA_TOKEN_TYPE_OP_GT, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			return;
		}
	}
}


/* try to parse type argument list*/
static GeeList* vala_parser_parse_type_argument_list (ValaParser* self, gboolean maybe_expression, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	if (vala_parser_accept (self, VALA_TOKEN_TYPE_OP_LT)) {
		GeeArrayList* list;
		list = gee_array_list_new (VALA_TYPE_DATA_TYPE, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
		do {
			ValaTokenType _tmp1;
			_tmp1 = vala_parser_current (self);
			if (_tmp1 == VALA_TOKEN_TYPE_VOID || _tmp1 == VALA_TOKEN_TYPE_DYNAMIC || _tmp1 == VALA_TOKEN_TYPE_WEAK || _tmp1 == VALA_TOKEN_TYPE_IDENTIFIER)
			do {
				ValaDataType* type;
				type = vala_parser_parse_type (self, TRUE, &inner_error);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					(list == NULL ? NULL : (list = (gee_collection_object_unref (list), NULL)));
					return NULL;
				}
				gee_collection_add (GEE_COLLECTION (list), type);
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
				break;
			} while (0); else
			do {
				GeeList* _tmp0;
				vala_parser_rollback (self, &begin);
				_tmp0 = NULL;
				return (_tmp0 = NULL, (list == NULL ? NULL : (list = (gee_collection_object_unref (list), NULL))), _tmp0);
			} while (0);
		} while (vala_parser_accept (self, VALA_TOKEN_TYPE_COMMA));
		if (!vala_parser_accept (self, VALA_TOKEN_TYPE_OP_GT)) {
			GeeList* _tmp2;
			vala_parser_rollback (self, &begin);
			_tmp2 = NULL;
			return (_tmp2 = NULL, (list == NULL ? NULL : (list = (gee_collection_object_unref (list), NULL))), _tmp2);
		}
		if (maybe_expression) {
			ValaTokenType _tmp4;
			/* check follower to decide whether to keep type argument list*/
			_tmp4 = vala_parser_current (self);
			if (_tmp4 == VALA_TOKEN_TYPE_OPEN_PARENS || _tmp4 == VALA_TOKEN_TYPE_CLOSE_PARENS || _tmp4 == VALA_TOKEN_TYPE_CLOSE_BRACKET || _tmp4 == VALA_TOKEN_TYPE_COLON || _tmp4 == VALA_TOKEN_TYPE_SEMICOLON || _tmp4 == VALA_TOKEN_TYPE_COMMA || _tmp4 == VALA_TOKEN_TYPE_DOT || _tmp4 == VALA_TOKEN_TYPE_INTERR || _tmp4 == VALA_TOKEN_TYPE_OP_EQ || _tmp4 == VALA_TOKEN_TYPE_OP_NE)
			do {
				break;
			} while (0); else
			do {
				GeeList* _tmp3;
				vala_parser_rollback (self, &begin);
				_tmp3 = NULL;
				return (_tmp3 = NULL, (list == NULL ? NULL : (list = (gee_collection_object_unref (list), NULL))), _tmp3);
			} while (0);
		}
		return GEE_LIST (list);
	}
	return NULL;
}


static ValaMemberAccess* vala_parser_parse_member_name (ValaParser* self, GError** error) {
	GError * inner_error;
	ValaSourceLocation begin;
	ValaMemberAccess* expr;
	gboolean first;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	inner_error = NULL;
	begin = vala_parser_get_location (self);
	expr = NULL;
	first = TRUE;
	do {
		char* id;
		gboolean qualified;
		GeeList* type_arg_list;
		ValaMemberAccess* _tmp2;
		ValaSourceReference* _tmp1;
		id = vala_parser_parse_identifier (self, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			return NULL;
		}
		/* The first member access can be global:: qualified*/
		qualified = FALSE;
		if (first && _vala_strcmp0 (id, "global") == 0 && vala_parser_accept (self, VALA_TOKEN_TYPE_DOUBLE_COLON)) {
			char* _tmp0;
			_tmp0 = NULL;
			id = (_tmp0 = vala_parser_parse_identifier (self, &inner_error), (id = (g_free (id), NULL)), _tmp0);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				id = (g_free (id), NULL);
				(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
				return NULL;
			}
			qualified = TRUE;
		}
		type_arg_list = vala_parser_parse_type_argument_list (self, FALSE, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			id = (g_free (id), NULL);
			(expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL)));
			return NULL;
		}
		_tmp2 = NULL;
		_tmp1 = NULL;
		expr = (_tmp2 = vala_member_access_new (VALA_EXPRESSION (expr), id, (_tmp1 = vala_parser_get_src (self, &begin))), (expr == NULL ? NULL : (expr = (vala_code_node_unref (expr), NULL))), _tmp2);
		(_tmp1 == NULL ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL)));
		vala_member_access_set_qualified (expr, qualified);
		if (type_arg_list != NULL) {
			{
				GeeList* type_arg_collection;
				int type_arg_it;
				type_arg_collection = type_arg_list;
				for (type_arg_it = 0; type_arg_it < gee_collection_get_size (GEE_COLLECTION (type_arg_collection)); type_arg_it = type_arg_it + 1) {
					ValaDataType* type_arg;
					type_arg = ((ValaDataType*) (gee_list_get (GEE_LIST (type_arg_collection), type_arg_it)));
					{
						vala_member_access_add_type_argument (expr, type_arg);
						(type_arg == NULL ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL)));
					}
				}
			}
		}
		first = FALSE;
		id = (g_free (id), NULL);
		(type_arg_list == NULL ? NULL : (type_arg_list = (gee_collection_object_unref (type_arg_list), NULL)));
	} while (vala_parser_accept (self, VALA_TOKEN_TYPE_DOT));
	return expr;
}


static gboolean vala_parser_is_declaration_keyword (ValaParser* self, ValaTokenType type) {
	ValaTokenType _tmp2;
	g_return_val_if_fail (VALA_IS_PARSER (self), FALSE);
	_tmp2 = type;
	if (_tmp2 == VALA_TOKEN_TYPE_ABSTRACT || _tmp2 == VALA_TOKEN_TYPE_CLASS || _tmp2 == VALA_TOKEN_TYPE_CONST || _tmp2 == VALA_TOKEN_TYPE_DELEGATE || _tmp2 == VALA_TOKEN_TYPE_ENUM || _tmp2 == VALA_TOKEN_TYPE_ERRORDOMAIN || _tmp2 == VALA_TOKEN_TYPE_EXTERN || _tmp2 == VALA_TOKEN_TYPE_INLINE || _tmp2 == VALA_TOKEN_TYPE_INTERFACE || _tmp2 == VALA_TOKEN_TYPE_INTERNAL || _tmp2 == VALA_TOKEN_TYPE_NAMESPACE || _tmp2 == VALA_TOKEN_TYPE_OVERRIDE || _tmp2 == VALA_TOKEN_TYPE_PRIVATE || _tmp2 == VALA_TOKEN_TYPE_PROTECTED || _tmp2 == VALA_TOKEN_TYPE_PUBLIC || _tmp2 == VALA_TOKEN_TYPE_SIGNAL || _tmp2 == VALA_TOKEN_TYPE_STATIC || _tmp2 == VALA_TOKEN_TYPE_STRUCT || _tmp2 == VALA_TOKEN_TYPE_VIRTUAL || _tmp2 == VALA_TOKEN_TYPE_VOLATILE)
	do {
		return TRUE;
	} while (0); else
	do {
		return FALSE;
	} while (0);
}


static void vala_parser_class_init (ValaParserClass * klass) {
	vala_parser_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_parser_finalize;
	g_type_class_add_private (klass, sizeof (ValaParserPrivate));
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_parser_real_visit_source_file;
}


static void vala_parser_instance_init (ValaParser * self) {
	self->priv = VALA_PARSER_GET_PRIVATE (self);
}


static void vala_parser_finalize (ValaCodeVisitor* obj) {
	ValaParser * self;
	self = VALA_PARSER (obj);
	(self->priv->scanner == NULL ? NULL : (self->priv->scanner = (vala_scanner_unref (self->priv->scanner), NULL)));
	(self->priv->context == NULL ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)));
	self->priv->tokens = (g_free (self->priv->tokens), NULL);
	self->priv->comment = (g_free (self->priv->comment), NULL);
	VALA_CODE_VISITOR_CLASS (vala_parser_parent_class)->finalize (obj);
}


GType vala_parser_get_type (void) {
	static GType vala_parser_type_id = 0;
	if (vala_parser_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaParserClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_parser_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaParser), 0, (GInstanceInitFunc) vala_parser_instance_init, NULL };
		vala_parser_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaParser", &g_define_type_info, 0);
	}
	return vala_parser_type_id;
}


GQuark vala_parse_error_quark (void) {
	return g_quark_from_static_string ("vala_parse_error-quark");
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




