/* valaproperty.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valaproperty.h>
#include <vala/valacodevisitor.h>
#include <vala/valaformalparameter.h>
#include <vala/valafield.h>
#include <vala/valadatatype.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valasourcereference.h>
#include <vala/valaexpression.h>
#include <vala/valasymbol.h>
#include <vala/valaattribute.h>
#include <vala/valaclass.h>
#include <vala/valainterface.h>
#include <vala/valascope.h>
#include <vala/valareport.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valasourcefile.h>
#include <vala/valavaluetype.h>




struct _ValaPropertyPrivate {
	ValaFormalParameter* _this_parameter;
	gboolean _notify;
	gboolean _no_accessor_method;
	gboolean _interface_only;
	gboolean _is_abstract;
	gboolean _is_virtual;
	gboolean _overrides;
	ValaField* _field;
	MemberBinding _binding;
	ValaExpression* _default_expression;
	gboolean lock_used;
	ValaDataType* _data_type;
	char* _nick;
	char* _blurb;
	ValaProperty* _base_property;
	ValaProperty* _base_interface_property;
	gboolean base_properties_valid;
	ValaPropertyAccessor* _get_accessor;
	ValaPropertyAccessor* _set_accessor;
};

#define VALA_PROPERTY_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_PROPERTY, ValaPropertyPrivate))
enum  {
	VALA_PROPERTY_DUMMY_PROPERTY
};
static void vala_property_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_property_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_property_get_canonical_name (ValaProperty* self);
static void vala_property_process_ccode_attribute (ValaProperty* self, ValaAttribute* a);
static gboolean vala_property_real_get_lock_used (ValaLockable* base);
static void vala_property_real_set_lock_used (ValaLockable* base, gboolean used);
static void vala_property_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static void vala_property_find_base_properties (ValaProperty* self);
static void vala_property_find_base_class_property (ValaProperty* self, ValaClass* cl);
static void vala_property_find_base_interface_property (ValaProperty* self, ValaClass* cl);
static gboolean vala_property_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_property_parent_class = NULL;
static ValaLockableIface* vala_property_vala_lockable_parent_iface = NULL;
static void vala_property_finalize (ValaCodeNode* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new property.
 *
 * @param name         property name
 * @param type         property type
 * @param get_accessor get accessor
 * @param set_accessor set/construct accessor
 * @param source       reference to source code
 * @return             newly created property
 */
ValaProperty* vala_property_construct (GType object_type, const char* name, ValaDataType* property_type, ValaPropertyAccessor* get_accessor, ValaPropertyAccessor* set_accessor, ValaSourceReference* source_reference) {
	ValaProperty* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = ((ValaProperty*) (g_type_create_instance (object_type)));
	self = ((ValaProperty*) (vala_member_construct (object_type, name, source_reference)));
	vala_property_set_property_type (self, property_type);
	vala_property_set_get_accessor (self, get_accessor);
	vala_property_set_set_accessor (self, set_accessor);
	return self;
}


ValaProperty* vala_property_new (const char* name, ValaDataType* property_type, ValaPropertyAccessor* get_accessor, ValaPropertyAccessor* set_accessor, ValaSourceReference* source_reference) {
	return vala_property_construct (VALA_TYPE_PROPERTY, name, property_type, get_accessor, set_accessor, source_reference);
}


static void vala_property_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaProperty * self;
	self = ((ValaProperty*) (base));
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_member (visitor, ((ValaMember*) (self)));
	vala_code_visitor_visit_property (visitor, self);
}


static void vala_property_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaProperty * self;
	self = ((ValaProperty*) (base));
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept (((ValaCodeNode*) (vala_property_get_property_type (self))), visitor);
	if (vala_property_get_get_accessor (self) != NULL) {
		vala_code_node_accept (((ValaCodeNode*) (vala_property_get_get_accessor (self))), visitor);
	}
	if (vala_property_get_set_accessor (self) != NULL) {
		vala_code_node_accept (((ValaCodeNode*) (vala_property_get_set_accessor (self))), visitor);
	}
	if (self->priv->_default_expression != NULL) {
		vala_code_node_accept (((ValaCodeNode*) (self->priv->_default_expression)), visitor);
	}
}


/**
 * Returns the C name of this property in upper case. Words are
 * separated by underscores. The upper case C name of the class is
 * prefix of the result.
 *
 * @return the upper case name to be used in C code
 */
char* vala_property_get_upper_case_cname (ValaProperty* self) {
	char* _tmp2;
	char* _tmp1;
	char* _tmp0;
	char* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp3 = NULL;
	return (_tmp3 = g_utf8_strup ((_tmp2 = g_strdup_printf ("%s_%s", (_tmp0 = vala_symbol_get_lower_case_cname (vala_symbol_get_parent_symbol (((ValaSymbol*) (self))), NULL)), (_tmp1 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name (((ValaSymbol*) (self))))))), -1), (_tmp2 = (g_free (_tmp2), NULL)), (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp3);
}


/**
 * Returns the string literal of this property to be used in C code.
 *
 * @return string literal to be used in C code
 */
ValaCCodeConstant* vala_property_get_canonical_cconstant (ValaProperty* self) {
	char* _tmp1;
	char* _tmp0;
	ValaCCodeConstant* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = vala_ccode_constant_new ((_tmp1 = g_strdup_printf ("\"%s\"", (_tmp0 = vala_property_get_canonical_name (self))))), (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2);
}


static char* vala_property_get_canonical_name (ValaProperty* self) {
	GString* str;
	const char* _tmp0;
	char* i;
	const char* _tmp3;
	char* _tmp4;
	g_return_val_if_fail (self != NULL, NULL);
	str = g_string_new ("");
	_tmp0 = NULL;
	i = (_tmp0 = vala_symbol_get_name (((ValaSymbol*) (self))), (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* _tmp2;
		const char* _tmp1;
		c = g_utf8_get_char (i);
		if (c == '_') {
			g_string_append_c (str, '-');
		} else {
			g_string_append_unichar (str, c);
		}
		_tmp2 = NULL;
		_tmp1 = NULL;
		i = (_tmp2 = (_tmp1 = g_utf8_next_char (i), (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (i = (g_free (i), NULL)), _tmp2);
	}
	_tmp3 = NULL;
	_tmp4 = NULL;
	return (_tmp4 = (_tmp3 = str->str, (_tmp3 == NULL ? NULL : g_strdup (_tmp3))), (str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL))), (i = (g_free (i), NULL)), _tmp4);
}


static void vala_property_process_ccode_attribute (ValaProperty* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "notify")) {
		vala_property_set_notify (self, vala_attribute_get_bool (a, "notify"));
	}
}


/**
 * Process all associated attributes.
 */
void vala_property_process_attributes (ValaProperty* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) (self))->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp2;
			ValaAttribute* a;
			_tmp2 = NULL;
			a = (_tmp2 = ((ValaAttribute*) (a_it->data)), (_tmp2 == NULL ? NULL : vala_code_node_ref (_tmp2)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_property_process_ccode_attribute (self, a);
				} else {
					if (_vala_strcmp0 (vala_attribute_get_name (a), "NoAccessorMethod") == 0) {
						vala_property_set_no_accessor_method (self, TRUE);
					} else {
						if (_vala_strcmp0 (vala_attribute_get_name (a), "Description") == 0) {
							if (vala_attribute_has_argument (a, "nick")) {
								char* _tmp0;
								_tmp0 = NULL;
								vala_property_set_nick (self, (_tmp0 = vala_attribute_get_string (a, "nick")));
								_tmp0 = (g_free (_tmp0), NULL);
							}
							if (vala_attribute_has_argument (a, "blurb")) {
								char* _tmp1;
								_tmp1 = NULL;
								vala_property_set_blurb (self, (_tmp1 = vala_attribute_get_string (a, "blurb")));
								_tmp1 = (g_free (_tmp1), NULL);
							}
						}
					}
				}
				(a == NULL ? NULL : (a = (vala_code_node_unref (a), NULL)));
			}
		}
	}
}


static gboolean vala_property_real_get_lock_used (ValaLockable* base) {
	ValaProperty * self;
	self = ((ValaProperty*) (base));
	return self->priv->lock_used;
}


static void vala_property_real_set_lock_used (ValaLockable* base, gboolean used) {
	ValaProperty * self;
	self = ((ValaProperty*) (base));
	self->priv->lock_used = used;
}


/**
 * Checks whether the accessors and type of the specified property
 * matches this property.
 *
 * @param prop a property
 * @return     true if the specified property is compatible to this
 *             property
 */
gboolean vala_property_equals (ValaProperty* self, ValaProperty* prop2) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (prop2 != NULL, FALSE);
	if (!vala_data_type_equals (vala_property_get_property_type (prop2), vala_property_get_property_type (self))) {
		return FALSE;
	}
	if ((vala_property_get_get_accessor (self) == NULL && vala_property_get_get_accessor (prop2) != NULL) || (vala_property_get_get_accessor (self) != NULL && vala_property_get_get_accessor (prop2) == NULL)) {
		return FALSE;
	}
	if ((vala_property_get_set_accessor (self) == NULL && vala_property_get_set_accessor (prop2) != NULL) || (vala_property_get_set_accessor (self) != NULL && vala_property_get_set_accessor (prop2) == NULL)) {
		return FALSE;
	}
	if (vala_property_get_set_accessor (self) != NULL) {
		if (vala_property_accessor_get_writable (vala_property_get_set_accessor (self)) != vala_property_accessor_get_writable (vala_property_get_set_accessor (prop2))) {
			return FALSE;
		}
		if (vala_property_accessor_get_construction (vala_property_get_set_accessor (self)) != vala_property_accessor_get_construction (vala_property_get_set_accessor (prop2))) {
			return FALSE;
		}
	}
	return TRUE;
}


static void vala_property_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaProperty * self;
	self = ((ValaProperty*) (base));
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_property_get_property_type (self) == old_type) {
		vala_property_set_property_type (self, new_type);
	}
}


static void vala_property_find_base_properties (ValaProperty* self) {
	g_return_if_fail (self != NULL);
	if (self->priv->base_properties_valid) {
		return;
	}
	if (VALA_IS_CLASS (vala_symbol_get_parent_symbol (((ValaSymbol*) (self))))) {
		/* VAPI classes don't specify overridden properties */
		if (!vala_symbol_get_external_package (vala_symbol_get_parent_symbol (((ValaSymbol*) (self))))) {
			vala_property_find_base_interface_property (self, VALA_CLASS (vala_symbol_get_parent_symbol (((ValaSymbol*) (self)))));
			if (self->priv->_is_virtual || self->priv->_overrides) {
				vala_property_find_base_class_property (self, VALA_CLASS (vala_symbol_get_parent_symbol (((ValaSymbol*) (self)))));
			}
		} else {
			if (self->priv->_is_virtual || self->priv->_is_abstract) {
				self->priv->_base_property = self;
			}
		}
	} else {
		if (VALA_IS_INTERFACE (vala_symbol_get_parent_symbol (((ValaSymbol*) (self))))) {
			if (self->priv->_is_virtual || self->priv->_is_abstract) {
				ValaProperty* _tmp1;
				ValaProperty* _tmp0;
				_tmp1 = NULL;
				_tmp0 = NULL;
				self->priv->_base_interface_property = (_tmp1 = (_tmp0 = self, (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0))), (self->priv->_base_interface_property == NULL ? NULL : (self->priv->_base_interface_property = (vala_code_node_unref (self->priv->_base_interface_property), NULL))), _tmp1);
			}
		}
	}
	self->priv->base_properties_valid = TRUE;
}


static void vala_property_find_base_class_property (ValaProperty* self, ValaClass* cl) {
	ValaSymbol* sym;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	sym = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) (cl))), vala_symbol_get_name (((ValaSymbol*) (self))));
	if (VALA_IS_PROPERTY (sym)) {
		ValaProperty* _tmp0;
		ValaProperty* base_property;
		_tmp0 = NULL;
		base_property = (_tmp0 = VALA_PROPERTY (sym), (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0)));
		if (base_property->priv->_is_abstract || base_property->priv->_is_virtual) {
			if (!vala_property_equals (self, base_property)) {
				char* _tmp3;
				char* _tmp2;
				char* _tmp1;
				vala_code_node_set_error (((ValaCodeNode*) (self)), TRUE);
				_tmp3 = NULL;
				_tmp2 = NULL;
				_tmp1 = NULL;
				vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) (self))), (_tmp3 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", (_tmp1 = vala_symbol_get_full_name (((ValaSymbol*) (self)))), (_tmp2 = vala_symbol_get_full_name (((ValaSymbol*) (base_property)))))));
				_tmp3 = (g_free (_tmp3), NULL);
				_tmp2 = (g_free (_tmp2), NULL);
				_tmp1 = (g_free (_tmp1), NULL);
				(base_property == NULL ? NULL : (base_property = (vala_code_node_unref (base_property), NULL)));
				(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
				return;
			}
			self->priv->_base_property = base_property;
			(base_property == NULL ? NULL : (base_property = (vala_code_node_unref (base_property), NULL)));
			(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
			return;
		}
		(base_property == NULL ? NULL : (base_property = (vala_code_node_unref (base_property), NULL)));
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_property_find_base_class_property (self, vala_class_get_base_class (cl));
	}
	(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
}


static void vala_property_find_base_interface_property (ValaProperty* self, ValaClass* cl) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	/* FIXME report error if multiple possible base properties are found*/
	{
		GeeList* type_collection;
		int type_it;
		type_collection = vala_class_get_base_types (cl);
		for (type_it = 0; type_it < gee_collection_get_size (GEE_COLLECTION (type_collection)); type_it = type_it + 1) {
			ValaDataType* type;
			type = ((ValaDataType*) (gee_list_get (GEE_LIST (type_collection), type_it)));
			{
				if (VALA_IS_INTERFACE (vala_data_type_get_data_type (type))) {
					ValaSymbol* sym;
					sym = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) (vala_data_type_get_data_type (type)))), vala_symbol_get_name (((ValaSymbol*) (self))));
					if (VALA_IS_PROPERTY (sym)) {
						ValaProperty* _tmp0;
						ValaProperty* base_property;
						_tmp0 = NULL;
						base_property = (_tmp0 = VALA_PROPERTY (sym), (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0)));
						if (base_property->priv->_is_abstract) {
							ValaProperty* _tmp5;
							ValaProperty* _tmp4;
							if (!vala_property_equals (self, base_property)) {
								char* _tmp3;
								char* _tmp2;
								char* _tmp1;
								vala_code_node_set_error (((ValaCodeNode*) (self)), TRUE);
								_tmp3 = NULL;
								_tmp2 = NULL;
								_tmp1 = NULL;
								vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) (self))), (_tmp3 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", (_tmp1 = vala_symbol_get_full_name (((ValaSymbol*) (self)))), (_tmp2 = vala_symbol_get_full_name (((ValaSymbol*) (base_property)))))));
								_tmp3 = (g_free (_tmp3), NULL);
								_tmp2 = (g_free (_tmp2), NULL);
								_tmp1 = (g_free (_tmp1), NULL);
								(base_property == NULL ? NULL : (base_property = (vala_code_node_unref (base_property), NULL)));
								(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
								(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
								(type_collection == NULL ? NULL : (type_collection = (gee_collection_object_unref (type_collection), NULL)));
								return;
							}
							_tmp5 = NULL;
							_tmp4 = NULL;
							self->priv->_base_interface_property = (_tmp5 = (_tmp4 = base_property, (_tmp4 == NULL ? NULL : vala_code_node_ref (_tmp4))), (self->priv->_base_interface_property == NULL ? NULL : (self->priv->_base_interface_property = (vala_code_node_unref (self->priv->_base_interface_property), NULL))), _tmp5);
							(base_property == NULL ? NULL : (base_property = (vala_code_node_unref (base_property), NULL)));
							(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
							(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
							(type_collection == NULL ? NULL : (type_collection = (gee_collection_object_unref (type_collection), NULL)));
							return;
						}
						(base_property == NULL ? NULL : (base_property = (vala_code_node_unref (base_property), NULL)));
					}
					(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
				}
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			}
		}
		(type_collection == NULL ? NULL : (type_collection = (gee_collection_object_unref (type_collection), NULL)));
	}
}


static gboolean vala_property_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaProperty * self;
	ValaSourceFile* _tmp1;
	ValaSourceFile* old_source_file;
	ValaSymbol* _tmp2;
	ValaSymbol* old_symbol;
	gboolean _tmp13;
	self = ((ValaProperty*) (base));
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked (((ValaCodeNode*) (self)))) {
		return !vala_code_node_get_error (((ValaCodeNode*) (self)));
	}
	vala_code_node_set_checked (((ValaCodeNode*) (self)), TRUE);
	_tmp1 = NULL;
	old_source_file = (_tmp1 = vala_semantic_analyzer_get_current_source_file (analyzer), (_tmp1 == NULL ? NULL : vala_source_file_ref (_tmp1)));
	_tmp2 = NULL;
	old_symbol = (_tmp2 = vala_semantic_analyzer_get_current_symbol (analyzer), (_tmp2 == NULL ? NULL : vala_code_node_ref (_tmp2)));
	if (vala_code_node_get_source_reference (((ValaCodeNode*) (self))) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) (self)))));
	}
	vala_semantic_analyzer_set_current_symbol (analyzer, ((ValaSymbol*) (self)));
	vala_code_node_accept (((ValaCodeNode*) (vala_property_get_property_type (self))), ((ValaCodeVisitor*) (analyzer)));
	if (vala_property_get_get_accessor (self) != NULL) {
		vala_code_node_accept (((ValaCodeNode*) (vala_property_get_get_accessor (self))), ((ValaCodeVisitor*) (analyzer)));
	}
	if (vala_property_get_set_accessor (self) != NULL) {
		vala_code_node_accept (((ValaCodeNode*) (vala_property_get_set_accessor (self))), ((ValaCodeVisitor*) (analyzer)));
	}
	if (self->priv->_default_expression != NULL) {
		vala_code_node_accept (((ValaCodeNode*) (self->priv->_default_expression)), ((ValaCodeVisitor*) (analyzer)));
	}
	/* check whether property type is at least as accessible as the property*/
	if (!vala_semantic_analyzer_is_type_accessible (analyzer, ((ValaSymbol*) (self)), vala_property_get_property_type (self))) {
		char* _tmp5;
		char* _tmp4;
		char* _tmp3;
		vala_code_node_set_error (((ValaCodeNode*) (self)), TRUE);
		_tmp5 = NULL;
		_tmp4 = NULL;
		_tmp3 = NULL;
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) (self))), (_tmp5 = g_strdup_printf ("property type `%s` is less accessible than property `%s`", (_tmp3 = vala_code_node_to_string (((ValaCodeNode*) (vala_property_get_property_type (self))))), (_tmp4 = vala_symbol_get_full_name (((ValaSymbol*) (self)))))));
		_tmp5 = (g_free (_tmp5), NULL);
		_tmp4 = (g_free (_tmp4), NULL);
		_tmp3 = (g_free (_tmp3), NULL);
	}
	if (!vala_symbol_is_internal_symbol (((ValaSymbol*) (self)))) {
		if (VALA_IS_VALUE_TYPE (vala_property_get_property_type (self)) && !vala_data_type_is_real_struct_type (vala_property_get_property_type (self))) {
			vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_property_get_property_type (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
		} else {
			vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_property_get_property_type (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		}
	}
	vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_property_get_property_type (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	if (self->priv->_overrides && vala_property_get_base_property (self) == NULL) {
		char* _tmp7;
		char* _tmp6;
		_tmp7 = NULL;
		_tmp6 = NULL;
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) (self))), (_tmp7 = g_strdup_printf ("%s: no suitable property found to override", (_tmp6 = vala_symbol_get_full_name (((ValaSymbol*) (self)))))));
		_tmp7 = (g_free (_tmp7), NULL);
		_tmp6 = (g_free (_tmp6), NULL);
	}
	/* construct properties must be public */
	if (vala_property_get_set_accessor (self) != NULL && vala_property_accessor_get_construction (vala_property_get_set_accessor (self))) {
		if (vala_symbol_get_access (((ValaSymbol*) (self))) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
			char* _tmp9;
			char* _tmp8;
			vala_code_node_set_error (((ValaCodeNode*) (self)), TRUE);
			_tmp9 = NULL;
			_tmp8 = NULL;
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) (self))), (_tmp9 = g_strdup_printf ("%s: construct properties must be public", (_tmp8 = vala_symbol_get_full_name (((ValaSymbol*) (self)))))));
			_tmp9 = (g_free (_tmp9), NULL);
			_tmp8 = (g_free (_tmp8), NULL);
		}
	}
	if (self->priv->_default_expression != NULL && !(vala_data_type_compatible (vala_expression_get_value_type (self->priv->_default_expression), vala_property_get_property_type (self)))) {
		char* _tmp12;
		char* _tmp11;
		char* _tmp10;
		vala_code_node_set_error (((ValaCodeNode*) (self)), TRUE);
		_tmp12 = NULL;
		_tmp11 = NULL;
		_tmp10 = NULL;
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) (self->priv->_default_expression))), (_tmp12 = g_strdup_printf ("Expected initializer of type `%s' but got `%s'", (_tmp10 = vala_code_node_to_string (((ValaCodeNode*) (vala_property_get_property_type (self))))), (_tmp11 = vala_code_node_to_string (((ValaCodeNode*) (vala_expression_get_value_type (self->priv->_default_expression))))))));
		_tmp12 = (g_free (_tmp12), NULL);
		_tmp11 = (g_free (_tmp11), NULL);
		_tmp10 = (g_free (_tmp10), NULL);
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	vala_semantic_analyzer_set_current_symbol (analyzer, old_symbol);
	return (_tmp13 = !vala_code_node_get_error (((ValaCodeNode*) (self))), (old_source_file == NULL ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL))), (old_symbol == NULL ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL))), _tmp13);
}


ValaDataType* vala_property_get_property_type (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_data_type;
}


void vala_property_set_property_type (ValaProperty* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (((ValaCodeNode*) (self->priv->_data_type)), ((ValaCodeNode*) (self)));
}


ValaPropertyAccessor* vala_property_get_get_accessor (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_get_accessor;
}


void vala_property_set_get_accessor (ValaProperty* self, ValaPropertyAccessor* value) {
	ValaPropertyAccessor* _tmp2;
	ValaPropertyAccessor* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_get_accessor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_get_accessor == NULL ? NULL : (self->priv->_get_accessor = (vala_code_node_unref (self->priv->_get_accessor), NULL))), _tmp2);
	if (value != NULL) {
		vala_property_accessor_set_prop (value, self);
	}
}


ValaPropertyAccessor* vala_property_get_set_accessor (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_set_accessor;
}


void vala_property_set_set_accessor (ValaProperty* self, ValaPropertyAccessor* value) {
	ValaPropertyAccessor* _tmp2;
	ValaPropertyAccessor* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_set_accessor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_set_accessor == NULL ? NULL : (self->priv->_set_accessor = (vala_code_node_unref (self->priv->_set_accessor), NULL))), _tmp2);
	if (value != NULL) {
		vala_property_accessor_set_prop (value, self);
	}
}


ValaFormalParameter* vala_property_get_this_parameter (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_this_parameter;
}


void vala_property_set_this_parameter (ValaProperty* self, ValaFormalParameter* value) {
	ValaFormalParameter* _tmp2;
	ValaFormalParameter* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_this_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (vala_code_node_unref (self->priv->_this_parameter), NULL))), _tmp2);
}


gboolean vala_property_get_notify (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_notify;
}


void vala_property_set_notify (ValaProperty* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_notify = value;
}


gboolean vala_property_get_no_accessor_method (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_no_accessor_method;
}


void vala_property_set_no_accessor_method (ValaProperty* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_no_accessor_method = value;
}


gboolean vala_property_get_interface_only (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_interface_only;
}


void vala_property_set_interface_only (ValaProperty* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_interface_only = value;
}


gboolean vala_property_get_is_abstract (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_abstract;
}


void vala_property_set_is_abstract (ValaProperty* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_abstract = value;
}


gboolean vala_property_get_is_virtual (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_virtual;
}


void vala_property_set_is_virtual (ValaProperty* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_virtual = value;
}


gboolean vala_property_get_overrides (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_overrides;
}


void vala_property_set_overrides (ValaProperty* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_overrides = value;
}


ValaField* vala_property_get_field (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_field;
}


void vala_property_set_field (ValaProperty* self, ValaField* value) {
	ValaField* _tmp2;
	ValaField* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_field = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_field == NULL ? NULL : (self->priv->_field = (vala_code_node_unref (self->priv->_field), NULL))), _tmp2);
}


MemberBinding vala_property_get_binding (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_binding;
}


void vala_property_set_binding (ValaProperty* self, MemberBinding value) {
	g_return_if_fail (self != NULL);
	self->priv->_binding = value;
}


ValaProperty* vala_property_get_base_property (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	vala_property_find_base_properties (self);
	return self->priv->_base_property;
}


ValaProperty* vala_property_get_base_interface_property (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	vala_property_find_base_properties (self);
	return self->priv->_base_interface_property;
}


ValaExpression* vala_property_get_default_expression (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_default_expression;
}


void vala_property_set_default_expression (ValaProperty* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_default_expression = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_default_expression == NULL ? NULL : (self->priv->_default_expression = (vala_code_node_unref (self->priv->_default_expression), NULL))), _tmp2);
}


const char* vala_property_get_nick (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_nick == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->_nick = (_tmp0 = vala_property_get_canonical_name (self), (self->priv->_nick = (g_free (self->priv->_nick), NULL)), _tmp0);
	}
	return self->priv->_nick;
}


void vala_property_set_nick (ValaProperty* self, const char* value) {
	char* _tmp3;
	const char* _tmp2;
	g_return_if_fail (self != NULL);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->_nick = (_tmp3 = (_tmp2 = value, (_tmp2 == NULL ? NULL : g_strdup (_tmp2))), (self->priv->_nick = (g_free (self->priv->_nick), NULL)), _tmp3);
}


const char* vala_property_get_blurb (ValaProperty* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_blurb == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->_blurb = (_tmp0 = vala_property_get_canonical_name (self), (self->priv->_blurb = (g_free (self->priv->_blurb), NULL)), _tmp0);
	}
	return self->priv->_blurb;
}


void vala_property_set_blurb (ValaProperty* self, const char* value) {
	char* _tmp3;
	const char* _tmp2;
	g_return_if_fail (self != NULL);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->_blurb = (_tmp3 = (_tmp2 = value, (_tmp2 == NULL ? NULL : g_strdup (_tmp2))), (self->priv->_blurb = (g_free (self->priv->_blurb), NULL)), _tmp3);
}


static void vala_property_class_init (ValaPropertyClass * klass) {
	vala_property_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_property_finalize;
	g_type_class_add_private (klass, sizeof (ValaPropertyPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_property_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_property_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_property_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_property_real_check;
}


static void vala_property_vala_lockable_interface_init (ValaLockableIface * iface) {
	vala_property_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_property_real_get_lock_used;
	iface->set_lock_used = vala_property_real_set_lock_used;
}


static void vala_property_instance_init (ValaProperty * self) {
	self->priv = VALA_PROPERTY_GET_PRIVATE (self);
	self->priv->_notify = TRUE;
	self->priv->_binding = MEMBER_BINDING_INSTANCE;
	self->priv->lock_used = FALSE;
}


static void vala_property_finalize (ValaCodeNode* obj) {
	ValaProperty * self;
	self = VALA_PROPERTY (obj);
	(self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (vala_code_node_unref (self->priv->_this_parameter), NULL)));
	(self->priv->_field == NULL ? NULL : (self->priv->_field = (vala_code_node_unref (self->priv->_field), NULL)));
	(self->priv->_default_expression == NULL ? NULL : (self->priv->_default_expression = (vala_code_node_unref (self->priv->_default_expression), NULL)));
	(self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL)));
	self->priv->_nick = (g_free (self->priv->_nick), NULL);
	self->priv->_blurb = (g_free (self->priv->_blurb), NULL);
	(self->priv->_base_interface_property == NULL ? NULL : (self->priv->_base_interface_property = (vala_code_node_unref (self->priv->_base_interface_property), NULL)));
	(self->priv->_get_accessor == NULL ? NULL : (self->priv->_get_accessor = (vala_code_node_unref (self->priv->_get_accessor), NULL)));
	(self->priv->_set_accessor == NULL ? NULL : (self->priv->_set_accessor = (vala_code_node_unref (self->priv->_set_accessor), NULL)));
	VALA_CODE_NODE_CLASS (vala_property_parent_class)->finalize (obj);
}


GType vala_property_get_type (void) {
	static GType vala_property_type_id = 0;
	if (vala_property_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaPropertyClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_property_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaProperty), 0, (GInstanceInitFunc) vala_property_instance_init, NULL };
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_property_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_property_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaProperty", &g_define_type_info, 0);
		g_type_add_interface_static (vala_property_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_property_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




