/* valagirwriter.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <gobject/valagirwriter.h>
#include <stdio.h>
#include <vala/valatypesymbol.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valacodenode.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valadatatype.h>
#include <vala/valaobjecttype.h>
#include <gee/collection.h>
#include <vala/valaarraytype.h>
#include <vala/valavoidtype.h>
#include <gee/list.h>
#include <vala/valaformalparameter.h>
#include <vala/valamember.h>
#include <gobject/valaccodebasemodule.h>




struct _ValaGIRWriterPrivate {
	ValaCodeContext* context;
	FILE* stream;
	gint indent;
	ValaTypeSymbol* gobject_type;
};

#define VALA_GIR_WRITER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_GIR_WRITER, ValaGIRWriterPrivate))
enum  {
	VALA_GIR_WRITER_DUMMY_PROPERTY
};
static void vala_gir_writer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_gir_writer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_gir_writer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_gir_writer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_gir_writer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_gir_writer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev);
static void vala_gir_writer_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* edomain);
static void vala_gir_writer_real_visit_error_code (ValaCodeVisitor* base, ValaErrorCode* ecode);
static void vala_gir_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_gir_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_gir_writer_write_params (ValaGIRWriter* self, GeeList* params, ValaDataType* instance_type);
static void vala_gir_writer_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* cb);
static void vala_gir_writer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_gir_writer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_gir_writer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_gir_writer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig);
static void vala_gir_writer_write_indent (ValaGIRWriter* self);
static void vala_gir_writer_write_return_type (ValaGIRWriter* self, ValaDataType* type);
static void vala_gir_writer_write_type (ValaGIRWriter* self, ValaDataType* type);
static gboolean vala_gir_writer_check_accessibility (ValaGIRWriter* self, ValaSymbol* sym);
static gpointer vala_gir_writer_parent_class = NULL;
static void vala_gir_writer_finalize (ValaCodeVisitor* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);



/**
 * Writes the public interface of the specified code context into the
 * specified file.
 *
 * @param context  a code context
 * @param filename a relative or absolute filename
 */
void vala_gir_writer_write_file (ValaGIRWriter* self, ValaCodeContext* context, const char* filename) {
	ValaCodeContext* _tmp1;
	ValaCodeContext* _tmp0;
	ValaNamespace* _tmp2;
	ValaNamespace* root_symbol;
	ValaSymbol* glib_ns;
	ValaTypeSymbol* _tmp3;
	FILE* _tmp4;
	FILE* _tmp5;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	g_return_if_fail (filename != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->context = (_tmp1 = (_tmp0 = context, (_tmp0 == NULL) ? NULL : vala_code_context_ref (_tmp0)), (self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)), _tmp1);
	_tmp2 = NULL;
	root_symbol = (_tmp2 = vala_code_context_get_root (context), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	glib_ns = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) root_symbol), "GLib");
	_tmp3 = NULL;
	self->priv->gobject_type = (_tmp3 = VALA_TYPESYMBOL (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Object")), (self->priv->gobject_type == NULL) ? NULL : (self->priv->gobject_type = (vala_code_node_unref (self->priv->gobject_type), NULL)), _tmp3);
	_tmp4 = NULL;
	self->priv->stream = (_tmp4 = fopen (filename, "w"), (self->priv->stream == NULL) ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)), _tmp4);
	fprintf (self->priv->stream, "<?xml version=\"1.0\"?>\n");
	fprintf (self->priv->stream, "<repository version=\"1.0\">\n");
	vala_code_context_accept (context, (ValaCodeVisitor*) self);
	fprintf (self->priv->stream, "</repository>\n");
	_tmp5 = NULL;
	self->priv->stream = (_tmp5 = NULL, (self->priv->stream == NULL) ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)), _tmp5);
	(root_symbol == NULL) ? NULL : (root_symbol = (vala_code_node_unref (root_symbol), NULL));
	(glib_ns == NULL) ? NULL : (glib_ns = (vala_code_node_unref (glib_ns), NULL));
}


static void vala_gir_writer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns) {
	ValaGIRWriter * self;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (ns != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) ns)) {
		return;
	}
	if (vala_symbol_get_name ((ValaSymbol*) ns) == NULL) {
		/* global namespace*/
		vala_code_node_accept_children ((ValaCodeNode*) ns, (ValaCodeVisitor*) self);
		return;
	}
	if (vala_symbol_get_name (vala_symbol_get_parent_symbol ((ValaSymbol*) ns)) != NULL) {
		/* nested namespace
		 not supported in GIR at the moment*/
		return;
	}
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<namespace name=\"%s\" version=\"1.0\">\n", vala_symbol_get_name ((ValaSymbol*) ns));
	self->priv->indent++;
	vala_code_node_accept_children ((ValaCodeNode*) ns, (ValaCodeVisitor*) self);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</namespace>\n");
}


static void vala_gir_writer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaGIRWriter * self;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (cl != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) cl)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) cl)) {
		return;
	}
	if (vala_typesymbol_is_subtype_of ((ValaTypeSymbol*) cl, self->priv->gobject_type)) {
		char* _tmp0;
		char* _tmp1;
		char* _tmp2;
		gboolean first;
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "<class name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) cl));
		_tmp0 = NULL;
		fprintf (self->priv->stream, " parent=\"%s\"", _tmp0 = vala_symbol_get_full_name ((ValaSymbol*) vala_class_get_base_class (cl)));
		_tmp0 = (g_free (_tmp0), NULL);
		_tmp1 = NULL;
		fprintf (self->priv->stream, " glib:type-name=\"%s\"", _tmp1 = vala_typesymbol_get_cname ((ValaTypeSymbol*) cl, FALSE));
		_tmp1 = (g_free (_tmp1), NULL);
		_tmp2 = NULL;
		fprintf (self->priv->stream, " glib:get-type=\"%sget_type\"", _tmp2 = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) cl));
		_tmp2 = (g_free (_tmp2), NULL);
		fprintf (self->priv->stream, ">\n");
		self->priv->indent++;
		/* write implemented interfaces*/
		first = TRUE;
		{
			GeeList* _tmp3;
			GeeIterator* _tmp4;
			GeeIterator* base_type_it;
			_tmp3 = NULL;
			_tmp4 = NULL;
			base_type_it = (_tmp4 = gee_iterable_iterator ((GeeIterable*) (_tmp3 = vala_class_get_base_types (cl))), (_tmp3 == NULL) ? NULL : (_tmp3 = (gee_collection_object_unref (_tmp3), NULL)), _tmp4);
			while (gee_iterator_next (base_type_it)) {
				ValaDataType* base_type;
				ValaObjectType* _tmp5;
				ValaObjectType* object_type;
				base_type = (ValaDataType*) gee_iterator_get (base_type_it);
				_tmp5 = NULL;
				object_type = (_tmp5 = VALA_OBJECT_TYPE (base_type), (_tmp5 == NULL) ? NULL : vala_code_node_ref (_tmp5));
				if (VALA_IS_INTERFACE (vala_object_type_get_type_symbol (object_type))) {
					char* _tmp6;
					if (first) {
						vala_gir_writer_write_indent (self);
						fprintf (self->priv->stream, "<implements>\n");
						self->priv->indent++;
						first = FALSE;
					}
					vala_gir_writer_write_indent (self);
					_tmp6 = NULL;
					fprintf (self->priv->stream, "<interface name=\"%s\"/>\n", _tmp6 = vala_symbol_get_full_name ((ValaSymbol*) vala_object_type_get_type_symbol (object_type)));
					_tmp6 = (g_free (_tmp6), NULL);
				}
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
				(object_type == NULL) ? NULL : (object_type = (vala_code_node_unref (object_type), NULL));
			}
			(base_type_it == NULL) ? NULL : (base_type_it = (gee_collection_object_unref (base_type_it), NULL));
		}
		if (!first) {
			self->priv->indent--;
			vala_gir_writer_write_indent (self);
			fprintf (self->priv->stream, "</implements>\n");
		}
		vala_code_node_accept_children ((ValaCodeNode*) cl, (ValaCodeVisitor*) self);
		self->priv->indent--;
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "</class>\n");
	} else {
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "<record name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) cl));
		fprintf (self->priv->stream, ">\n");
		self->priv->indent++;
		vala_code_node_accept_children ((ValaCodeNode*) cl, (ValaCodeVisitor*) self);
		self->priv->indent--;
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "</record>\n");
	}
}


static void vala_gir_writer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaGIRWriter * self;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (st != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) st)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) st)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<record name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) st));
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_code_node_accept_children ((ValaCodeNode*) st, (ValaCodeVisitor*) self);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</record>\n");
}


static void vala_gir_writer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaGIRWriter * self;
	char* _tmp0;
	GeeList* _tmp1;
	gboolean _tmp2;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (iface != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) iface)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) iface)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<interface name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) iface));
	_tmp0 = NULL;
	fprintf (self->priv->stream, " glib:get-type=\"%sget_type\"", _tmp0 = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) iface));
	_tmp0 = (g_free (_tmp0), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	/* write prerequisites*/
	_tmp1 = NULL;
	if ((_tmp2 = gee_collection_get_size ((GeeCollection*) (_tmp1 = vala_interface_get_prerequisites (iface))) > 0, (_tmp1 == NULL) ? NULL : (_tmp1 = (gee_collection_object_unref (_tmp1), NULL)), _tmp2)) {
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "<requires>\n");
		self->priv->indent++;
		{
			GeeList* _tmp3;
			GeeIterator* _tmp4;
			GeeIterator* base_type_it;
			_tmp3 = NULL;
			_tmp4 = NULL;
			base_type_it = (_tmp4 = gee_iterable_iterator ((GeeIterable*) (_tmp3 = vala_interface_get_prerequisites (iface))), (_tmp3 == NULL) ? NULL : (_tmp3 = (gee_collection_object_unref (_tmp3), NULL)), _tmp4);
			while (gee_iterator_next (base_type_it)) {
				ValaDataType* base_type;
				ValaObjectType* _tmp5;
				ValaObjectType* object_type;
				base_type = (ValaDataType*) gee_iterator_get (base_type_it);
				_tmp5 = NULL;
				object_type = (_tmp5 = VALA_OBJECT_TYPE (base_type), (_tmp5 == NULL) ? NULL : vala_code_node_ref (_tmp5));
				if (VALA_IS_CLASS (vala_object_type_get_type_symbol (object_type))) {
					char* _tmp6;
					vala_gir_writer_write_indent (self);
					_tmp6 = NULL;
					fprintf (self->priv->stream, "<object name=\"%s\"/>\n", _tmp6 = vala_symbol_get_full_name ((ValaSymbol*) vala_object_type_get_type_symbol (object_type)));
					_tmp6 = (g_free (_tmp6), NULL);
				} else {
					if (VALA_IS_INTERFACE (vala_object_type_get_type_symbol (object_type))) {
						char* _tmp7;
						vala_gir_writer_write_indent (self);
						_tmp7 = NULL;
						fprintf (self->priv->stream, "<interface name=\"%s\"/>\n", _tmp7 = vala_symbol_get_full_name ((ValaSymbol*) vala_object_type_get_type_symbol (object_type)));
						_tmp7 = (g_free (_tmp7), NULL);
					} else {
						g_assert_not_reached ();
					}
				}
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
				(object_type == NULL) ? NULL : (object_type = (vala_code_node_unref (object_type), NULL));
			}
			(base_type_it == NULL) ? NULL : (base_type_it = (gee_collection_object_unref (base_type_it), NULL));
		}
		self->priv->indent--;
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "</requires>\n");
	}
	vala_code_node_accept_children ((ValaCodeNode*) iface, (ValaCodeVisitor*) self);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</interface>\n");
}


static void vala_gir_writer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaGIRWriter * self;
	char* _tmp0;
	char* _tmp1;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (en != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) en)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) en)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<enumeration name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) en));
	_tmp0 = NULL;
	fprintf (self->priv->stream, " c:type=\"%s\"", _tmp0 = vala_typesymbol_get_cname ((ValaTypeSymbol*) en, FALSE));
	_tmp0 = (g_free (_tmp0), NULL);
	_tmp1 = NULL;
	fprintf (self->priv->stream, " glib:get-type=\"%sget_type\"", _tmp1 = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) en));
	_tmp1 = (g_free (_tmp1), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_code_node_accept_children ((ValaCodeNode*) en, (ValaCodeVisitor*) self);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</enumeration>\n");
}


static void vala_gir_writer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev) {
	ValaGIRWriter * self;
	char* _tmp2;
	gint _tmp1_length1;
	char** _tmp1;
	char* _tmp0;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (ev != NULL);
	vala_gir_writer_write_indent (self);
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<member name=\"%s\"/>\n", _tmp2 = g_strjoinv ("-", (_tmp1 = g_strsplit (_tmp0 = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) ev), -1), "_", 0), _tmp1_length1 = -1, _tmp1)));
	_tmp2 = (g_free (_tmp2), NULL);
	_tmp1 = (_vala_array_free (_tmp1, _tmp1_length1, (GDestroyNotify) g_free), NULL);
	_tmp0 = (g_free (_tmp0), NULL);
}


static void vala_gir_writer_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* edomain) {
	ValaGIRWriter * self;
	char* _tmp0;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (edomain != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) edomain)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) edomain)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<errordomain name=\"%s\"", _tmp0 = vala_typesymbol_get_cname ((ValaTypeSymbol*) edomain, FALSE));
	_tmp0 = (g_free (_tmp0), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_code_node_accept_children ((ValaCodeNode*) edomain, (ValaCodeVisitor*) self);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</errordomain>\n");
}


static void vala_gir_writer_real_visit_error_code (ValaCodeVisitor* base, ValaErrorCode* ecode) {
	ValaGIRWriter * self;
	char* _tmp0;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (ecode != NULL);
	vala_gir_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<member name=\"%s\"/>\n", _tmp0 = vala_typesymbol_get_cname ((ValaTypeSymbol*) ecode, FALSE));
	_tmp0 = (g_free (_tmp0), NULL);
}


static void vala_gir_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c) {
	ValaGIRWriter * self;
	char* _tmp0;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (c != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) c)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) c)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<constant name=\"%s\"/>\n", _tmp0 = vala_constant_get_cname (c));
	_tmp0 = (g_free (_tmp0), NULL);
}


static void vala_gir_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f) {
	ValaGIRWriter * self;
	char* _tmp0;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (f != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) f)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) f)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<field name=\"%s\">\n", _tmp0 = vala_field_get_cname (f));
	_tmp0 = (g_free (_tmp0), NULL);
	self->priv->indent++;
	vala_gir_writer_write_type (self, vala_field_get_field_type (f));
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</field>\n");
}


static void vala_gir_writer_write_params (ValaGIRWriter* self, GeeList* params, ValaDataType* instance_type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (params != NULL);
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<parameters>\n");
	self->priv->indent++;
	if (instance_type != NULL) {
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "<parameter name=\"self\">\n");
		self->priv->indent++;
		vala_gir_writer_write_type (self, instance_type);
		self->priv->indent--;
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "</parameter>\n");
	}
	{
		GeeIterator* param_it;
		param_it = gee_iterable_iterator ((GeeIterable*) params);
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = (ValaFormalParameter*) gee_iterator_get (param_it);
			vala_gir_writer_write_indent (self);
			fprintf (self->priv->stream, "<parameter name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) param));
			if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_REF) {
				fprintf (self->priv->stream, " direction=\"inout\"");
				/* in/out paramter*/
				if (vala_data_type_get_value_owned (vala_formal_parameter_get_parameter_type (param))) {
					fprintf (self->priv->stream, " transfer-ownership=\"full\"");
				}
			} else {
				if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_OUT) {
					/* out paramter*/
					fprintf (self->priv->stream, " direction=\"out\"");
					if (vala_data_type_get_value_owned (vala_formal_parameter_get_parameter_type (param))) {
						fprintf (self->priv->stream, " transfer-ownership=\"full\"");
					}
				} else {
					/* normal in paramter*/
					if (vala_data_type_get_value_owned (vala_formal_parameter_get_parameter_type (param))) {
						fprintf (self->priv->stream, " transfer-ownership=\"full\"");
					}
				}
			}
			fprintf (self->priv->stream, ">\n");
			self->priv->indent++;
			vala_gir_writer_write_type (self, vala_formal_parameter_get_parameter_type (param));
			self->priv->indent--;
			vala_gir_writer_write_indent (self);
			fprintf (self->priv->stream, "</parameter>\n");
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(param_it == NULL) ? NULL : (param_it = (gee_collection_object_unref (param_it), NULL));
	}
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</parameters>\n");
}


static void vala_gir_writer_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* cb) {
	ValaGIRWriter * self;
	char* _tmp0;
	GeeList* _tmp1;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (cb != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) cb)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) cb)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<callback name=\"%s\"", _tmp0 = vala_typesymbol_get_cname ((ValaTypeSymbol*) cb, FALSE));
	_tmp0 = (g_free (_tmp0), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	_tmp1 = NULL;
	vala_gir_writer_write_params (self, _tmp1 = vala_delegate_get_parameters (cb), NULL);
	(_tmp1 == NULL) ? NULL : (_tmp1 = (gee_collection_object_unref (_tmp1), NULL));
	vala_gir_writer_write_return_type (self, vala_delegate_get_return_type (cb));
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</callback>\n");
}


static void vala_gir_writer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m) {
	ValaGIRWriter * self;
	gboolean _tmp0;
	gboolean _tmp1;
	char* _tmp4;
	ValaDataType* instance_type;
	GeeList* _tmp6;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (m != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) m)) {
		return;
	}
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) m)) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = vala_method_get_overrides (m);
	}
	if (_tmp1) {
		_tmp0 = TRUE;
	} else {
		gboolean _tmp2;
		gboolean _tmp3;
		_tmp2 = FALSE;
		_tmp3 = FALSE;
		if (vala_method_get_base_interface_method (m) != NULL) {
			_tmp3 = !vala_method_get_is_abstract (m);
		} else {
			_tmp3 = FALSE;
		}
		if (_tmp3) {
			_tmp2 = !vala_method_get_is_virtual (m);
		} else {
			_tmp2 = FALSE;
		}
		_tmp0 = (_tmp2);
	}
	/* don't write interface implementation unless it's an abstract or virtual method*/
	if (_tmp0) {
		return;
	}
	vala_gir_writer_write_indent (self);
	_tmp4 = NULL;
	fprintf (self->priv->stream, "<method name=\"%s\" c:identifier=\"%s\"", vala_symbol_get_name ((ValaSymbol*) m), _tmp4 = vala_method_get_cname (m));
	_tmp4 = (g_free (_tmp4), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	instance_type = NULL;
	if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
		ValaDataType* _tmp5;
		_tmp5 = NULL;
		instance_type = (_tmp5 = vala_ccode_base_module_get_data_type_for_symbol (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) m))), (instance_type == NULL) ? NULL : (instance_type = (vala_code_node_unref (instance_type), NULL)), _tmp5);
	}
	_tmp6 = NULL;
	vala_gir_writer_write_params (self, _tmp6 = vala_method_get_parameters (m), instance_type);
	(_tmp6 == NULL) ? NULL : (_tmp6 = (gee_collection_object_unref (_tmp6), NULL));
	vala_gir_writer_write_return_type (self, vala_method_get_return_type (m));
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</method>\n");
	(instance_type == NULL) ? NULL : (instance_type = (vala_code_node_unref (instance_type), NULL));
}


static void vala_gir_writer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m) {
	ValaGIRWriter * self;
	char* _tmp0;
	GeeList* _tmp1;
	ValaDataType* _tmp2;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (m != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) m)) {
		return;
	}
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) m)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<constructor name=\"%s\" c:identifier=\"%s\"", vala_symbol_get_name ((ValaSymbol*) m), _tmp0 = vala_method_get_cname ((ValaMethod*) m));
	_tmp0 = (g_free (_tmp0), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	_tmp1 = NULL;
	vala_gir_writer_write_params (self, _tmp1 = vala_method_get_parameters ((ValaMethod*) m), NULL);
	(_tmp1 == NULL) ? NULL : (_tmp1 = (gee_collection_object_unref (_tmp1), NULL));
	_tmp2 = NULL;
	vala_gir_writer_write_return_type (self, _tmp2 = vala_ccode_base_module_get_data_type_for_symbol (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) m))));
	(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL));
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</constructor>\n");
}


static void vala_gir_writer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop) {
	ValaGIRWriter * self;
	gboolean _tmp0;
	gboolean _tmp1;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (prop != NULL);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) prop)) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = vala_property_get_overrides (prop);
	}
	if (_tmp1) {
		_tmp0 = TRUE;
	} else {
		gboolean _tmp2;
		gboolean _tmp3;
		_tmp2 = FALSE;
		_tmp3 = FALSE;
		if (vala_property_get_base_interface_property (prop) != NULL) {
			_tmp3 = !vala_property_get_is_abstract (prop);
		} else {
			_tmp3 = FALSE;
		}
		if (_tmp3) {
			_tmp2 = !vala_property_get_is_virtual (prop);
		} else {
			_tmp2 = FALSE;
		}
		_tmp0 = (_tmp2);
	}
	if (_tmp0) {
		return;
	}
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<property name=\"%s\"", vala_symbol_get_name ((ValaSymbol*) prop));
	if (vala_property_get_get_accessor (prop) != NULL) {
		fprintf (self->priv->stream, " readable=\"1\"");
	}
	if (vala_property_get_set_accessor (prop) != NULL) {
		fprintf (self->priv->stream, " writable=\"1\"");
	}
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_gir_writer_write_type (self, vala_property_get_property_type (prop));
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</property>\n");
}


static void vala_gir_writer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig) {
	ValaGIRWriter * self;
	char* _tmp0;
	GeeList* _tmp1;
	self = (ValaGIRWriter*) base;
	g_return_if_fail (sig != NULL);
	if (!vala_gir_writer_check_accessibility (self, (ValaSymbol*) sig)) {
		return;
	}
	vala_gir_writer_write_indent (self);
	_tmp0 = NULL;
	fprintf (self->priv->stream, "<glib:signal name=\"%s\"", _tmp0 = vala_signal_get_cname (sig));
	_tmp0 = (g_free (_tmp0), NULL);
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	_tmp1 = NULL;
	vala_gir_writer_write_params (self, _tmp1 = vala_signal_get_parameters (sig), NULL);
	(_tmp1 == NULL) ? NULL : (_tmp1 = (gee_collection_object_unref (_tmp1), NULL));
	vala_gir_writer_write_return_type (self, vala_signal_get_return_type (sig));
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</glib:signal>\n");
}


static void vala_gir_writer_write_indent (ValaGIRWriter* self) {
	gint i;
	g_return_if_fail (self != NULL);
	i = 0;
	for (i = 0; i < self->priv->indent; i++) {
		fputc ('\t', self->priv->stream);
	}
}


static void vala_gir_writer_write_return_type (ValaGIRWriter* self, ValaDataType* type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "<return-value");
	if (vala_data_type_get_value_owned (type)) {
		fprintf (self->priv->stream, " transfer-ownership=\"full\"");
	}
	fprintf (self->priv->stream, ">\n");
	self->priv->indent++;
	vala_gir_writer_write_type (self, type);
	self->priv->indent--;
	vala_gir_writer_write_indent (self);
	fprintf (self->priv->stream, "</return-value>\n");
}


static void vala_gir_writer_write_type (ValaGIRWriter* self, ValaDataType* type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	if (VALA_IS_ARRAY_TYPE (type)) {
		ValaArrayType* _tmp0;
		ValaArrayType* array_type;
		_tmp0 = NULL;
		array_type = (_tmp0 = VALA_ARRAY_TYPE (type), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "<array>\n");
		self->priv->indent++;
		vala_gir_writer_write_type (self, vala_array_type_get_element_type (array_type));
		self->priv->indent--;
		vala_gir_writer_write_indent (self);
		fprintf (self->priv->stream, "</array>\n");
		(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
	} else {
		if (VALA_IS_VOID_TYPE (type)) {
			vala_gir_writer_write_indent (self);
			fprintf (self->priv->stream, "<type name=\"none\"/>\n");
		} else {
			char* _tmp1;
			vala_gir_writer_write_indent (self);
			_tmp1 = NULL;
			fprintf (self->priv->stream, "<type name=\"%s\"/>\n", _tmp1 = vala_code_node_to_string ((ValaCodeNode*) type));
			_tmp1 = (g_free (_tmp1), NULL);
		}
	}
}


static gboolean vala_gir_writer_check_accessibility (ValaGIRWriter* self, ValaSymbol* sym) {
	gboolean _tmp0;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (sym != NULL, FALSE);
	_tmp0 = FALSE;
	if (vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PROTECTED;
	}
	if (_tmp0) {
		return TRUE;
	}
	return FALSE;
}


/**
 * Code visitor generating .gir file for the public interface.
 */
ValaGIRWriter* vala_gir_writer_construct (GType object_type) {
	ValaGIRWriter* self;
	self = (ValaGIRWriter*) g_type_create_instance (object_type);
	return self;
}


ValaGIRWriter* vala_gir_writer_new (void) {
	return vala_gir_writer_construct (VALA_TYPE_GIR_WRITER);
}


static void vala_gir_writer_class_init (ValaGIRWriterClass * klass) {
	vala_gir_writer_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_gir_writer_finalize;
	g_type_class_add_private (klass, sizeof (ValaGIRWriterPrivate));
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_gir_writer_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_gir_writer_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_gir_writer_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_gir_writer_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_gir_writer_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum_value = vala_gir_writer_real_visit_enum_value;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_domain = vala_gir_writer_real_visit_error_domain;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_code = vala_gir_writer_real_visit_error_code;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_gir_writer_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_gir_writer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delegate = vala_gir_writer_real_visit_delegate;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_gir_writer_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_gir_writer_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_gir_writer_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_gir_writer_real_visit_signal;
}


static void vala_gir_writer_instance_init (ValaGIRWriter * self) {
	self->priv = VALA_GIR_WRITER_GET_PRIVATE (self);
}


static void vala_gir_writer_finalize (ValaCodeVisitor* obj) {
	ValaGIRWriter * self;
	self = VALA_GIR_WRITER (obj);
	(self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL));
	(self->priv->stream == NULL) ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL));
	(self->priv->gobject_type == NULL) ? NULL : (self->priv->gobject_type = (vala_code_node_unref (self->priv->gobject_type), NULL));
	VALA_CODE_VISITOR_CLASS (vala_gir_writer_parent_class)->finalize (obj);
}


GType vala_gir_writer_get_type (void) {
	static GType vala_gir_writer_type_id = 0;
	if (vala_gir_writer_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaGIRWriterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_gir_writer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGIRWriter), 0, (GInstanceInitFunc) vala_gir_writer_instance_init, NULL };
		vala_gir_writer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaGIRWriter", &g_define_type_info, 0);
	}
	return vala_gir_writer_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}




