/* valamethodcall.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valamethodcall.h>
#include <gee/arraylist.h>
#include <gee/readonlylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valadatatype.h>
#include <vala/valamemberaccess.h>
#include <vala/valareport.h>
#include <vala/valasymbol.h>
#include <vala/valaobjecttype.h>
#include <vala/valacreationmethod.h>
#include <vala/valastruct.h>
#include <vala/valaobjectcreationexpression.h>
#include <vala/valaclass.h>
#include <vala/valaformalparameter.h>
#include <vala/valaarraytype.h>
#include <vala/valamethodtype.h>
#include <vala/valamethod.h>
#include <vala/valastringliteral.h>
#include <vala/valapointertype.h>
#include <vala/valavoidtype.h>
#include <vala/valaexpressionstatement.h>
#include <vala/valaforstatement.h>
#include <vala/valayieldstatement.h>
#include <vala/valadelegatetype.h>
#include <vala/valadelegate.h>
#include <vala/valalocalvariable.h>
#include <vala/valablock.h>
#include <vala/valadeclarationstatement.h>
#include <vala/valastatement.h>




struct _ValaMethodCallPrivate {
	ValaCCodeExpression* _delegate_target;
	GeeList* argument_list;
	GeeList* array_sizes;
};

#define VALA_METHOD_CALL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_METHOD_CALL, ValaMethodCallPrivate))
enum  {
	VALA_METHOD_CALL_DUMMY_PROPERTY
};
static void vala_method_call_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_method_call_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_method_call_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_method_call_real_is_pure (ValaExpression* base);
static gboolean vala_method_call_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_method_call_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
static void vala_method_call_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
static gpointer vala_method_call_parent_class = NULL;
static void vala_method_call_finalize (ValaCodeNode* obj);



/**
 * Creates a new invocation expression.
 *
 * @param call             method to call
 * @param source_reference reference to source code
 * @return                 newly created invocation expression
 */
ValaMethodCall* vala_method_call_construct (GType object_type, ValaExpression* call, ValaSourceReference* source_reference) {
	ValaMethodCall* self;
	g_return_val_if_fail (call != NULL, NULL);
	self = (ValaMethodCall*) g_type_create_instance (object_type);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_method_call_set_call (self, call);
	return self;
}


ValaMethodCall* vala_method_call_new (ValaExpression* call, ValaSourceReference* source_reference) {
	return vala_method_call_construct (VALA_TYPE_METHOD_CALL, call, source_reference);
}


/**
 * Appends the specified expression to the list of arguments.
 *
 * @param arg an argument
 */
void vala_method_call_add_argument (ValaMethodCall* self, ValaExpression* arg) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (arg != NULL);
	gee_collection_add ((GeeCollection*) self->priv->argument_list, arg);
	vala_code_node_set_parent_node ((ValaCodeNode*) arg, (ValaCodeNode*) self);
}


/**
 * Returns a copy of the argument list.
 *
 * @return argument list
 */
GeeList* vala_method_call_get_argument_list (ValaMethodCall* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->argument_list);
}


/**
 * Add an array size C code expression.
 */
void vala_method_call_append_array_size (ValaMethodCall* self, ValaCCodeExpression* size) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (size != NULL);
	gee_collection_add ((GeeCollection*) self->priv->array_sizes, size);
}


/**
 * Get the C code expression for array sizes for all dimensions
 * ascending from left to right.
 */
GeeList* vala_method_call_get_array_sizes (ValaMethodCall* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, self->priv->array_sizes);
}


static void vala_method_call_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMethodCall * self;
	self = (ValaMethodCall*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_method_call (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static void vala_method_call_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMethodCall * self;
	self = (ValaMethodCall*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_method_call_get_call (self), visitor);
	{
		GeeIterator* _expr_it;
		_expr_it = gee_iterable_iterator ((GeeIterable*) self->priv->argument_list);
		while (gee_iterator_next (_expr_it)) {
			ValaExpression* expr;
			expr = (ValaExpression*) gee_iterator_get (_expr_it);
			vala_code_node_accept ((ValaCodeNode*) expr, visitor);
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
		}
		(_expr_it == NULL) ? NULL : (_expr_it = (gee_collection_object_unref (_expr_it), NULL));
	}
}


static void vala_method_call_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaMethodCall * self;
	gint index;
	gboolean _tmp0;
	self = (ValaMethodCall*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_method_call_get_call (self) == old_node) {
		vala_method_call_set_call (self, new_node);
	}
	index = gee_list_index_of (self->priv->argument_list, old_node);
	_tmp0 = FALSE;
	if (index >= 0) {
		_tmp0 = vala_code_node_get_parent_node ((ValaCodeNode*) new_node) == NULL;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		gee_list_set ((GeeList*) self->priv->argument_list, index, new_node);
		vala_code_node_set_parent_node ((ValaCodeNode*) new_node, (ValaCodeNode*) self);
	}
}


static gboolean vala_method_call_real_is_pure (ValaExpression* base) {
	ValaMethodCall * self;
	self = (ValaMethodCall*) base;
	return FALSE;
}


static gboolean vala_method_call_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaMethodCall * self;
	ValaDataType* target_object_type;
	ValaDataType* _tmp8;
	ValaDataType* mtype;
	gboolean _tmp12;
	GeeList* params;
	gboolean _tmp22;
	ValaExpression* last_arg;
	GeeList* args;
	GeeIterator* arg_it;
	gboolean _tmp30;
	ValaDataType* ret_type;
	GeeList* _tmp88;
	ValaDataType* _tmp92;
	gboolean may_throw;
	GeeList* _tmp99;
	gboolean _tmp100;
	gboolean _tmp111;
	self = (ValaMethodCall*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	if (!vala_code_node_check ((ValaCodeNode*) vala_method_call_get_call (self), analyzer)) {
		/* if method resolving didn't succeed, skip this check */
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	/* type of target object*/
	target_object_type = NULL;
	if (VALA_IS_MEMBER_ACCESS (vala_method_call_get_call (self))) {
		ValaMemberAccess* _tmp2;
		ValaMemberAccess* ma;
		_tmp2 = NULL;
		ma = (_tmp2 = VALA_MEMBER_ACCESS (vala_method_call_get_call (self)), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
		if (vala_member_access_get_prototype_access (ma)) {
			char* _tmp4;
			char* _tmp3;
			gboolean _tmp5;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp4 = NULL;
			_tmp3 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp4 = g_strdup_printf ("Access to instance member `%s' denied", _tmp3 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (vala_method_call_get_call (self)))));
			_tmp4 = (g_free (_tmp4), NULL);
			_tmp3 = (g_free (_tmp3), NULL);
			return (_tmp5 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), _tmp5);
		}
		if (vala_member_access_get_inner (ma) != NULL) {
			ValaDataType* _tmp7;
			ValaDataType* _tmp6;
			_tmp7 = NULL;
			_tmp6 = NULL;
			target_object_type = (_tmp7 = (_tmp6 = vala_expression_get_value_type (vala_member_access_get_inner (ma)), (_tmp6 == NULL) ? NULL : vala_code_node_ref (_tmp6)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), _tmp7);
		}
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	}
	_tmp8 = NULL;
	mtype = (_tmp8 = vala_expression_get_value_type (vala_method_call_get_call (self)), (_tmp8 == NULL) ? NULL : vala_code_node_ref (_tmp8));
	if (VALA_IS_OBJECT_TYPE (mtype)) {
		ValaMethod* _tmp9;
		ValaCreationMethod* cm;
		/* constructor chain-up*/
		_tmp9 = NULL;
		cm = (_tmp9 = vala_semantic_analyzer_find_current_method (analyzer), VALA_IS_CREATION_METHOD (_tmp9) ? ((ValaCreationMethod*) _tmp9) : NULL);
		if (cm == NULL) {
			gboolean _tmp10;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "use `new' operator to create new objects");
			return (_tmp10 = FALSE, (cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), _tmp10);
		} else {
			if (vala_creation_method_get_chain_up (cm)) {
				gboolean _tmp11;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Multiple constructor calls in the same constructor are not permitted");
				return (_tmp11 = FALSE, (cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), _tmp11);
			}
		}
		vala_creation_method_set_chain_up (cm, TRUE);
		(cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL));
	}
	_tmp12 = FALSE;
	if (VALA_IS_MEMBER_ACCESS (vala_method_call_get_call (self))) {
		gboolean _tmp13;
		gboolean _tmp14;
		_tmp13 = FALSE;
		_tmp14 = FALSE;
		if (VALA_IS_CREATION_METHOD (vala_expression_get_symbol_reference (vala_method_call_get_call (self)))) {
			_tmp14 = VALA_IS_STRUCT (vala_symbol_get_parent_symbol (vala_expression_get_symbol_reference (vala_method_call_get_call (self))));
		} else {
			_tmp14 = FALSE;
		}
		if ((_tmp14)) {
			_tmp13 = TRUE;
		} else {
			_tmp13 = VALA_IS_STRUCT (vala_expression_get_symbol_reference (vala_method_call_get_call (self)));
		}
		_tmp12 = (_tmp13);
	} else {
		_tmp12 = FALSE;
	}
	/* check for struct construction*/
	if (_tmp12) {
		ValaObjectCreationExpression* struct_creation_expression;
		gboolean _tmp17;
		struct_creation_expression = vala_object_creation_expression_new (VALA_MEMBER_ACCESS (vala_method_call_get_call (self)), vala_code_node_get_source_reference ((ValaCodeNode*) self));
		vala_object_creation_expression_set_struct_creation (struct_creation_expression, TRUE);
		{
			GeeList* _tmp15;
			GeeIterator* _tmp16;
			GeeIterator* _arg_it;
			_tmp15 = NULL;
			_tmp16 = NULL;
			_arg_it = (_tmp16 = gee_iterable_iterator ((GeeIterable*) (_tmp15 = vala_method_call_get_argument_list (self))), (_tmp15 == NULL) ? NULL : (_tmp15 = (gee_collection_object_unref (_tmp15), NULL)), _tmp16);
			while (gee_iterator_next (_arg_it)) {
				ValaExpression* arg;
				arg = (ValaExpression*) gee_iterator_get (_arg_it);
				vala_object_creation_expression_add_argument (struct_creation_expression, arg);
				(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
			}
			(_arg_it == NULL) ? NULL : (_arg_it = (gee_collection_object_unref (_arg_it), NULL));
		}
		vala_expression_set_target_type ((ValaExpression*) struct_creation_expression, vala_expression_get_target_type ((ValaExpression*) self));
		gee_collection_add ((GeeCollection*) analyzer->replaced_nodes, (ValaCodeNode*) self);
		vala_code_node_replace_expression (vala_code_node_get_parent_node ((ValaCodeNode*) self), (ValaExpression*) self, (ValaExpression*) struct_creation_expression);
		vala_code_node_check ((ValaCodeNode*) struct_creation_expression, analyzer);
		return (_tmp17 = TRUE, (struct_creation_expression == NULL) ? NULL : (struct_creation_expression = (vala_code_node_unref (struct_creation_expression), NULL)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), _tmp17);
	} else {
		gboolean _tmp18;
		_tmp18 = FALSE;
		if (VALA_IS_MEMBER_ACCESS (vala_method_call_get_call (self))) {
			_tmp18 = VALA_IS_CREATION_METHOD (vala_expression_get_symbol_reference (vala_method_call_get_call (self)));
		} else {
			_tmp18 = FALSE;
		}
		if (_tmp18) {
			ValaMethod* _tmp19;
			ValaCreationMethod* cm;
			/* constructor chain-up*/
			_tmp19 = NULL;
			cm = (_tmp19 = vala_semantic_analyzer_find_current_method (analyzer), VALA_IS_CREATION_METHOD (_tmp19) ? ((ValaCreationMethod*) _tmp19) : NULL);
			if (cm == NULL) {
				gboolean _tmp20;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "use `new' operator to create new objects");
				return (_tmp20 = FALSE, (cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), _tmp20);
			} else {
				if (vala_creation_method_get_chain_up (cm)) {
					gboolean _tmp21;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Multiple constructor calls in the same constructor are not permitted");
					return (_tmp21 = FALSE, (cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), _tmp21);
				}
			}
			vala_creation_method_set_chain_up (cm, TRUE);
			(cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL));
		}
	}
	params = NULL;
	_tmp22 = FALSE;
	if (mtype != NULL) {
		_tmp22 = vala_data_type_is_invokable (mtype);
	} else {
		_tmp22 = FALSE;
	}
	if (_tmp22) {
		GeeList* _tmp23;
		_tmp23 = NULL;
		params = (_tmp23 = vala_data_type_get_parameters (mtype), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), _tmp23);
	} else {
		if (VALA_IS_CLASS (vala_expression_get_symbol_reference (vala_method_call_get_call (self)))) {
			gboolean _tmp24;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "use `new' operator to create new objects");
			return (_tmp24 = FALSE, (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), _tmp24);
		} else {
			gboolean _tmp25;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "invocation not supported in this context");
			return (_tmp25 = FALSE, (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), _tmp25);
		}
	}
	last_arg = NULL;
	args = vala_method_call_get_argument_list (self);
	arg_it = gee_iterable_iterator ((GeeIterable*) args);
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) params);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			if (vala_formal_parameter_get_ellipsis (param)) {
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				break;
			}
			if (vala_formal_parameter_get_params_array (param)) {
				ValaArrayType* _tmp26;
				ValaArrayType* array_type;
				_tmp26 = NULL;
				array_type = (_tmp26 = VALA_ARRAY_TYPE (vala_formal_parameter_get_parameter_type (param)), (_tmp26 == NULL) ? NULL : vala_code_node_ref (_tmp26));
				while (gee_iterator_next (arg_it)) {
					ValaExpression* arg;
					arg = (ValaExpression*) gee_iterator_get (arg_it);
					/* store expected type for callback parameters */
					vala_expression_set_target_type (arg, vala_array_type_get_element_type (array_type));
					vala_data_type_set_value_owned (vala_expression_get_target_type (arg), vala_data_type_get_value_owned ((ValaDataType*) array_type));
					(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
				}
				(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				break;
			}
			if (gee_iterator_next (arg_it)) {
				ValaExpression* arg;
				ValaDataType* _tmp27;
				ValaExpression* _tmp29;
				ValaExpression* _tmp28;
				arg = (ValaExpression*) gee_iterator_get (arg_it);
				/* store expected type for callback parameters */
				vala_expression_set_formal_target_type (arg, vala_formal_parameter_get_parameter_type (param));
				_tmp27 = NULL;
				vala_expression_set_target_type (arg, _tmp27 = vala_data_type_get_actual_type (vala_expression_get_formal_target_type (arg), target_object_type, (ValaCodeNode*) self));
				(_tmp27 == NULL) ? NULL : (_tmp27 = (vala_code_node_unref (_tmp27), NULL));
				_tmp29 = NULL;
				_tmp28 = NULL;
				last_arg = (_tmp29 = (_tmp28 = arg, (_tmp28 == NULL) ? NULL : vala_code_node_ref (_tmp28)), (last_arg == NULL) ? NULL : (last_arg = (vala_code_node_unref (last_arg), NULL)), _tmp29);
				(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
			}
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	_tmp30 = FALSE;
	if (VALA_IS_METHOD_TYPE (mtype)) {
		_tmp30 = vala_method_get_printf_format (vala_method_type_get_method_symbol ((VALA_METHOD_TYPE (mtype))));
	} else {
		_tmp30 = FALSE;
	}
	/* printf arguments*/
	if (_tmp30) {
		ValaStringLiteral* format_literal;
		format_literal = NULL;
		if (last_arg != NULL) {
			ValaStringLiteral* _tmp33;
			ValaStringLiteral* _tmp32;
			ValaExpression* _tmp31;
			/* use last argument as format string*/
			_tmp33 = NULL;
			_tmp32 = NULL;
			_tmp31 = NULL;
			format_literal = (_tmp33 = (_tmp32 = (_tmp31 = last_arg, VALA_IS_STRING_LITERAL (_tmp31) ? ((ValaStringLiteral*) _tmp31) : NULL), (_tmp32 == NULL) ? NULL : vala_code_node_ref (_tmp32)), (format_literal == NULL) ? NULL : (format_literal = (vala_code_node_unref (format_literal), NULL)), _tmp33);
		} else {
			ValaMemberAccess* _tmp35;
			ValaExpression* _tmp34;
			ValaMemberAccess* ma;
			/* use instance as format string for string.printf (...)*/
			_tmp35 = NULL;
			_tmp34 = NULL;
			ma = (_tmp35 = (_tmp34 = vala_method_call_get_call (self), VALA_IS_MEMBER_ACCESS (_tmp34) ? ((ValaMemberAccess*) _tmp34) : NULL), (_tmp35 == NULL) ? NULL : vala_code_node_ref (_tmp35));
			if (ma != NULL) {
				ValaStringLiteral* _tmp38;
				ValaStringLiteral* _tmp37;
				ValaExpression* _tmp36;
				_tmp38 = NULL;
				_tmp37 = NULL;
				_tmp36 = NULL;
				format_literal = (_tmp38 = (_tmp37 = (_tmp36 = vala_member_access_get_inner (ma), VALA_IS_STRING_LITERAL (_tmp36) ? ((ValaStringLiteral*) _tmp36) : NULL), (_tmp37 == NULL) ? NULL : vala_code_node_ref (_tmp37)), (format_literal == NULL) ? NULL : (format_literal = (vala_code_node_unref (format_literal), NULL)), _tmp38);
			}
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
		}
		if (format_literal != NULL) {
			char* format;
			gboolean unsupported_format;
			const char* format_it;
			gunichar c;
			gboolean _tmp84;
			format = vala_string_literal_eval (format_literal);
			unsupported_format = FALSE;
			format_it = format;
			c = g_utf8_get_char (format_it);
			while (c != '\0') {
				gint length;
				ValaDataType* param_type;
				gboolean _tmp45;
				gboolean _tmp46;
				if (c != '%') {
					format_it = g_utf8_next_char (format_it);
					c = g_utf8_get_char (format_it);
					continue;
				}
				format_it = g_utf8_next_char (format_it);
				c = g_utf8_get_char (format_it);
				while (TRUE) {
					gboolean _tmp39;
					gboolean _tmp40;
					gboolean _tmp41;
					gboolean _tmp42;
					_tmp39 = FALSE;
					_tmp40 = FALSE;
					_tmp41 = FALSE;
					_tmp42 = FALSE;
					if (c == '#') {
						_tmp42 = TRUE;
					} else {
						_tmp42 = c == '0';
					}
					if (_tmp42) {
						_tmp41 = TRUE;
					} else {
						_tmp41 = c == '-';
					}
					if (_tmp41) {
						_tmp40 = TRUE;
					} else {
						_tmp40 = c == ' ';
					}
					if (_tmp40) {
						_tmp39 = TRUE;
					} else {
						_tmp39 = c == '+';
					}
					if (!_tmp39) {
						break;
					}
					format_it = g_utf8_next_char (format_it);
					c = g_utf8_get_char (format_it);
				}
				while (TRUE) {
					gboolean _tmp43;
					_tmp43 = FALSE;
					if (c >= '0') {
						_tmp43 = c <= '9';
					} else {
						_tmp43 = FALSE;
					}
					if (!_tmp43) {
						break;
					}
					format_it = g_utf8_next_char (format_it);
					c = g_utf8_get_char (format_it);
				}
				/* precision*/
				if (c == '.') {
					format_it = g_utf8_next_char (format_it);
					c = g_utf8_get_char (format_it);
					while (TRUE) {
						gboolean _tmp44;
						_tmp44 = FALSE;
						if (c >= '0') {
							_tmp44 = c <= '9';
						} else {
							_tmp44 = FALSE;
						}
						if (!_tmp44) {
							break;
						}
						format_it = g_utf8_next_char (format_it);
						c = g_utf8_get_char (format_it);
					}
				}
				/* length modifier*/
				length = 0;
				if (c == 'h') {
					length = -1;
					format_it = g_utf8_next_char (format_it);
					c = g_utf8_get_char (format_it);
					if (c == 'h') {
						length = -2;
						format_it = g_utf8_next_char (format_it);
						c = g_utf8_get_char (format_it);
					}
				} else {
					if (c == 'l') {
						length = 1;
						format_it = g_utf8_next_char (format_it);
						c = g_utf8_get_char (format_it);
					} else {
						if (c == 'z') {
							length = 2;
							format_it = g_utf8_next_char (format_it);
							c = g_utf8_get_char (format_it);
						}
					}
				}
				/* conversion specifier*/
				param_type = NULL;
				_tmp45 = FALSE;
				_tmp46 = FALSE;
				if (c == 'd') {
					_tmp46 = TRUE;
				} else {
					_tmp46 = c == 'i';
				}
				if (_tmp46) {
					_tmp45 = TRUE;
				} else {
					_tmp45 = c == 'c';
				}
				if (_tmp45) {
					/* integer*/
					if (length == (-2)) {
						ValaDataType* _tmp48;
						ValaDataType* _tmp47;
						_tmp48 = NULL;
						_tmp47 = NULL;
						param_type = (_tmp48 = (_tmp47 = analyzer->int8_type, (_tmp47 == NULL) ? NULL : vala_code_node_ref (_tmp47)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp48);
					} else {
						if (length == (-1)) {
							ValaDataType* _tmp50;
							ValaDataType* _tmp49;
							_tmp50 = NULL;
							_tmp49 = NULL;
							param_type = (_tmp50 = (_tmp49 = analyzer->short_type, (_tmp49 == NULL) ? NULL : vala_code_node_ref (_tmp49)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp50);
						} else {
							if (length == 0) {
								ValaDataType* _tmp52;
								ValaDataType* _tmp51;
								_tmp52 = NULL;
								_tmp51 = NULL;
								param_type = (_tmp52 = (_tmp51 = analyzer->int_type, (_tmp51 == NULL) ? NULL : vala_code_node_ref (_tmp51)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp52);
							} else {
								if (length == 1) {
									ValaDataType* _tmp54;
									ValaDataType* _tmp53;
									_tmp54 = NULL;
									_tmp53 = NULL;
									param_type = (_tmp54 = (_tmp53 = analyzer->long_type, (_tmp53 == NULL) ? NULL : vala_code_node_ref (_tmp53)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp54);
								} else {
									if (length == 2) {
										ValaDataType* _tmp56;
										ValaDataType* _tmp55;
										_tmp56 = NULL;
										_tmp55 = NULL;
										param_type = (_tmp56 = (_tmp55 = analyzer->ssize_t_type, (_tmp55 == NULL) ? NULL : vala_code_node_ref (_tmp55)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp56);
									}
								}
							}
						}
					}
				} else {
					gboolean _tmp57;
					gboolean _tmp58;
					gboolean _tmp59;
					_tmp57 = FALSE;
					_tmp58 = FALSE;
					_tmp59 = FALSE;
					if (c == 'o') {
						_tmp59 = TRUE;
					} else {
						_tmp59 = c == 'u';
					}
					if (_tmp59) {
						_tmp58 = TRUE;
					} else {
						_tmp58 = c == 'x';
					}
					if (_tmp58) {
						_tmp57 = TRUE;
					} else {
						_tmp57 = c == 'X';
					}
					if (_tmp57) {
						/* unsigned integer*/
						if (length == (-2)) {
							ValaDataType* _tmp61;
							ValaDataType* _tmp60;
							_tmp61 = NULL;
							_tmp60 = NULL;
							param_type = (_tmp61 = (_tmp60 = analyzer->uchar_type, (_tmp60 == NULL) ? NULL : vala_code_node_ref (_tmp60)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp61);
						} else {
							if (length == (-1)) {
								ValaDataType* _tmp63;
								ValaDataType* _tmp62;
								_tmp63 = NULL;
								_tmp62 = NULL;
								param_type = (_tmp63 = (_tmp62 = analyzer->ushort_type, (_tmp62 == NULL) ? NULL : vala_code_node_ref (_tmp62)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp63);
							} else {
								if (length == 0) {
									ValaDataType* _tmp65;
									ValaDataType* _tmp64;
									_tmp65 = NULL;
									_tmp64 = NULL;
									param_type = (_tmp65 = (_tmp64 = analyzer->uint_type, (_tmp64 == NULL) ? NULL : vala_code_node_ref (_tmp64)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp65);
								} else {
									if (length == 1) {
										ValaDataType* _tmp67;
										ValaDataType* _tmp66;
										_tmp67 = NULL;
										_tmp66 = NULL;
										param_type = (_tmp67 = (_tmp66 = analyzer->ulong_type, (_tmp66 == NULL) ? NULL : vala_code_node_ref (_tmp66)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp67);
									} else {
										if (length == 2) {
											ValaDataType* _tmp69;
											ValaDataType* _tmp68;
											_tmp69 = NULL;
											_tmp68 = NULL;
											param_type = (_tmp69 = (_tmp68 = analyzer->size_t_type, (_tmp68 == NULL) ? NULL : vala_code_node_ref (_tmp68)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp69);
										}
									}
								}
							}
						}
					} else {
						gboolean _tmp70;
						gboolean _tmp71;
						gboolean _tmp72;
						gboolean _tmp73;
						gboolean _tmp74;
						gboolean _tmp75;
						gboolean _tmp76;
						_tmp70 = FALSE;
						_tmp71 = FALSE;
						_tmp72 = FALSE;
						_tmp73 = FALSE;
						_tmp74 = FALSE;
						_tmp75 = FALSE;
						_tmp76 = FALSE;
						if (c == 'e') {
							_tmp76 = TRUE;
						} else {
							_tmp76 = c == 'E';
						}
						if (_tmp76) {
							_tmp75 = TRUE;
						} else {
							_tmp75 = c == 'f';
						}
						if (_tmp75) {
							_tmp74 = TRUE;
						} else {
							_tmp74 = c == 'F';
						}
						if (_tmp74) {
							_tmp73 = TRUE;
						} else {
							_tmp73 = c == 'g';
						}
						if (_tmp73) {
							_tmp72 = TRUE;
						} else {
							_tmp72 = c == 'G';
						}
						if (_tmp72) {
							_tmp71 = TRUE;
						} else {
							_tmp71 = c == 'a';
						}
						if (_tmp71) {
							_tmp70 = TRUE;
						} else {
							_tmp70 = c == 'A';
						}
						if (_tmp70) {
							ValaDataType* _tmp78;
							ValaDataType* _tmp77;
							/* double*/
							_tmp78 = NULL;
							_tmp77 = NULL;
							param_type = (_tmp78 = (_tmp77 = analyzer->double_type, (_tmp77 == NULL) ? NULL : vala_code_node_ref (_tmp77)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp78);
						} else {
							if (c == 's') {
								ValaDataType* _tmp80;
								ValaDataType* _tmp79;
								/* string*/
								_tmp80 = NULL;
								_tmp79 = NULL;
								param_type = (_tmp80 = (_tmp79 = analyzer->string_type, (_tmp79 == NULL) ? NULL : vala_code_node_ref (_tmp79)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp80);
							} else {
								if (c == 'p') {
									ValaDataType* _tmp82;
									ValaVoidType* _tmp81;
									/* pointer*/
									_tmp82 = NULL;
									_tmp81 = NULL;
									param_type = (_tmp82 = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp81 = vala_void_type_new (NULL)), NULL), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp82);
									(_tmp81 == NULL) ? NULL : (_tmp81 = (vala_code_node_unref (_tmp81), NULL));
								} else {
									if (c == '%') {
									} else {
										/* literal %*/
										unsupported_format = TRUE;
										(param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL));
										break;
									}
								}
							}
						}
					}
				}
				if (c != '\0') {
					format_it = g_utf8_next_char (format_it);
					c = g_utf8_get_char (format_it);
				}
				if (param_type != NULL) {
					if (gee_iterator_next (arg_it)) {
						ValaExpression* arg;
						arg = (ValaExpression*) gee_iterator_get (arg_it);
						vala_expression_set_target_type (arg, param_type);
						(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
					} else {
						gboolean _tmp83;
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Too few arguments for specified format");
						return (_tmp83 = FALSE, (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), format = (g_free (format), NULL), (format_literal == NULL) ? NULL : (format_literal = (vala_code_node_unref (format_literal), NULL)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), (last_arg == NULL) ? NULL : (last_arg = (vala_code_node_unref (last_arg), NULL)), (args == NULL) ? NULL : (args = (gee_collection_object_unref (args), NULL)), (arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL)), _tmp83);
					}
				}
				(param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL));
			}
			_tmp84 = FALSE;
			if (!unsupported_format) {
				_tmp84 = gee_iterator_next (arg_it);
			} else {
				_tmp84 = FALSE;
			}
			if (_tmp84) {
				gboolean _tmp85;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Too many arguments for specified format");
				return (_tmp85 = FALSE, format = (g_free (format), NULL), (format_literal == NULL) ? NULL : (format_literal = (vala_code_node_unref (format_literal), NULL)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), (last_arg == NULL) ? NULL : (last_arg = (vala_code_node_unref (last_arg), NULL)), (args == NULL) ? NULL : (args = (gee_collection_object_unref (args), NULL)), (arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL)), _tmp85);
			}
			format = (g_free (format), NULL);
		}
		(format_literal == NULL) ? NULL : (format_literal = (vala_code_node_unref (format_literal), NULL));
	}
	{
		GeeList* _tmp86;
		GeeIterator* _tmp87;
		GeeIterator* _arg_it;
		_tmp86 = NULL;
		_tmp87 = NULL;
		_arg_it = (_tmp87 = gee_iterable_iterator ((GeeIterable*) (_tmp86 = vala_method_call_get_argument_list (self))), (_tmp86 == NULL) ? NULL : (_tmp86 = (gee_collection_object_unref (_tmp86), NULL)), _tmp87);
		while (gee_iterator_next (_arg_it)) {
			ValaExpression* arg;
			arg = (ValaExpression*) gee_iterator_get (_arg_it);
			vala_code_node_check ((ValaCodeNode*) arg, analyzer);
			(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
		}
		(_arg_it == NULL) ? NULL : (_arg_it = (gee_collection_object_unref (_arg_it), NULL));
	}
	ret_type = vala_data_type_get_return_type (mtype);
	_tmp88 = NULL;
	params = (_tmp88 = vala_data_type_get_parameters (mtype), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), _tmp88);
	if (VALA_IS_VOID_TYPE (ret_type)) {
		gboolean _tmp89;
		gboolean _tmp90;
		_tmp89 = FALSE;
		_tmp90 = FALSE;
		if (!(VALA_IS_EXPRESSION_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) self)))) {
			_tmp90 = !(VALA_IS_FOR_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) self)));
		} else {
			_tmp90 = FALSE;
		}
		if (_tmp90) {
			_tmp89 = !(VALA_IS_YIELD_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) self)));
		} else {
			_tmp89 = FALSE;
		}
		/* void return type*/
		if (_tmp89) {
			gboolean _tmp91;
			/* A void method invocation can be in the initializer or
			 iterator of a for statement*/
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "invocation of void method not allowed as expression");
			return (_tmp91 = FALSE, (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), (last_arg == NULL) ? NULL : (last_arg = (vala_code_node_unref (last_arg), NULL)), (args == NULL) ? NULL : (args = (gee_collection_object_unref (args), NULL)), (arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL)), (ret_type == NULL) ? NULL : (ret_type = (vala_code_node_unref (ret_type), NULL)), _tmp91);
		}
	}
	vala_expression_set_formal_value_type ((ValaExpression*) self, ret_type);
	_tmp92 = NULL;
	vala_expression_set_value_type ((ValaExpression*) self, _tmp92 = vala_data_type_get_actual_type (vala_expression_get_formal_value_type ((ValaExpression*) self), target_object_type, (ValaCodeNode*) self));
	(_tmp92 == NULL) ? NULL : (_tmp92 = (vala_code_node_unref (_tmp92), NULL));
	may_throw = FALSE;
	if (VALA_IS_METHOD_TYPE (mtype)) {
		ValaMethod* _tmp93;
		ValaMethod* m;
		_tmp93 = NULL;
		m = (_tmp93 = vala_method_type_get_method_symbol ((VALA_METHOD_TYPE (mtype))), (_tmp93 == NULL) ? NULL : vala_code_node_ref (_tmp93));
		{
			GeeList* _tmp94;
			GeeIterator* _tmp95;
			GeeIterator* _error_type_it;
			_tmp94 = NULL;
			_tmp95 = NULL;
			_error_type_it = (_tmp95 = gee_iterable_iterator ((GeeIterable*) (_tmp94 = vala_code_node_get_error_types ((ValaCodeNode*) m))), (_tmp94 == NULL) ? NULL : (_tmp94 = (gee_collection_object_unref (_tmp94), NULL)), _tmp95);
			while (gee_iterator_next (_error_type_it)) {
				ValaDataType* error_type;
				ValaDataType* call_error_type;
				error_type = (ValaDataType*) gee_iterator_get (_error_type_it);
				may_throw = TRUE;
				/* ensure we can trace back which expression may throw errors of this type*/
				call_error_type = vala_data_type_copy (error_type);
				vala_code_node_set_source_reference ((ValaCodeNode*) call_error_type, vala_code_node_get_source_reference ((ValaCodeNode*) self));
				vala_code_node_add_error_type ((ValaCodeNode*) self, call_error_type);
				(error_type == NULL) ? NULL : (error_type = (vala_code_node_unref (error_type), NULL));
				(call_error_type == NULL) ? NULL : (call_error_type = (vala_code_node_unref (call_error_type), NULL));
			}
			(_error_type_it == NULL) ? NULL : (_error_type_it = (gee_collection_object_unref (_error_type_it), NULL));
		}
		(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
	} else {
		if (VALA_IS_DELEGATE_TYPE (mtype)) {
			ValaDelegate* _tmp96;
			ValaDelegate* d;
			_tmp96 = NULL;
			d = (_tmp96 = vala_delegate_type_get_delegate_symbol ((VALA_DELEGATE_TYPE (mtype))), (_tmp96 == NULL) ? NULL : vala_code_node_ref (_tmp96));
			{
				GeeList* _tmp97;
				GeeIterator* _tmp98;
				GeeIterator* _error_type_it;
				_tmp97 = NULL;
				_tmp98 = NULL;
				_error_type_it = (_tmp98 = gee_iterable_iterator ((GeeIterable*) (_tmp97 = vala_code_node_get_error_types ((ValaCodeNode*) d))), (_tmp97 == NULL) ? NULL : (_tmp97 = (gee_collection_object_unref (_tmp97), NULL)), _tmp98);
				while (gee_iterator_next (_error_type_it)) {
					ValaDataType* error_type;
					ValaDataType* call_error_type;
					error_type = (ValaDataType*) gee_iterator_get (_error_type_it);
					may_throw = TRUE;
					/* ensure we can trace back which expression may throw errors of this type*/
					call_error_type = vala_data_type_copy (error_type);
					vala_code_node_set_source_reference ((ValaCodeNode*) call_error_type, vala_code_node_get_source_reference ((ValaCodeNode*) self));
					vala_code_node_add_error_type ((ValaCodeNode*) self, call_error_type);
					(error_type == NULL) ? NULL : (error_type = (vala_code_node_unref (error_type), NULL));
					(call_error_type == NULL) ? NULL : (call_error_type = (vala_code_node_unref (call_error_type), NULL));
				}
				(_error_type_it == NULL) ? NULL : (_error_type_it = (gee_collection_object_unref (_error_type_it), NULL));
			}
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
		}
	}
	_tmp99 = NULL;
	if ((_tmp100 = !vala_semantic_analyzer_check_arguments (analyzer, (ValaExpression*) self, mtype, params, _tmp99 = vala_method_call_get_argument_list (self)), (_tmp99 == NULL) ? NULL : (_tmp99 = (gee_collection_object_unref (_tmp99), NULL)), _tmp100)) {
		gboolean _tmp101;
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return (_tmp101 = FALSE, (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), (last_arg == NULL) ? NULL : (last_arg = (vala_code_node_unref (last_arg), NULL)), (args == NULL) ? NULL : (args = (gee_collection_object_unref (args), NULL)), (arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL)), (ret_type == NULL) ? NULL : (ret_type = (vala_code_node_unref (ret_type), NULL)), _tmp101);
	}
	if (may_throw) {
		gboolean _tmp102;
		_tmp102 = FALSE;
		if (VALA_IS_LOCAL_VARIABLE (vala_code_node_get_parent_node ((ValaCodeNode*) self))) {
			_tmp102 = TRUE;
		} else {
			_tmp102 = VALA_IS_EXPRESSION_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) self));
		}
		if (_tmp102) {
		} else {
			ValaBlock* _tmp103;
			ValaBlock* old_insert_block;
			ValaBlock* _tmp104;
			ValaCodeNode* _tmp105;
			ValaCodeNode* old_parent_node;
			char* _tmp106;
			ValaLocalVariable* _tmp107;
			ValaLocalVariable* local;
			ValaDeclarationStatement* decl;
			ValaExpression* temp_access;
			ValaBlock* _tmp108;
			ValaBlock* block;
			ValaBlock* _tmp110;
			ValaBlock* _tmp109;
			/* simple statements, no side effects after method call*/
			_tmp103 = NULL;
			old_insert_block = (_tmp103 = analyzer->insert_block, (_tmp103 == NULL) ? NULL : vala_code_node_ref (_tmp103));
			_tmp104 = NULL;
			analyzer->insert_block = (_tmp104 = vala_expression_prepare_condition_split ((ValaExpression*) self, analyzer), (analyzer->insert_block == NULL) ? NULL : (analyzer->insert_block = (vala_code_node_unref (analyzer->insert_block), NULL)), _tmp104);
			/* store parent_node as we need to replace the expression in the old parent node later on*/
			_tmp105 = NULL;
			old_parent_node = (_tmp105 = vala_code_node_get_parent_node ((ValaCodeNode*) self), (_tmp105 == NULL) ? NULL : vala_code_node_ref (_tmp105));
			_tmp106 = NULL;
			_tmp107 = NULL;
			local = (_tmp107 = vala_local_variable_new (vala_expression_get_value_type ((ValaExpression*) self), _tmp106 = vala_code_node_get_temp_name ((ValaCodeNode*) self), NULL, vala_code_node_get_source_reference ((ValaCodeNode*) self)), _tmp106 = (g_free (_tmp106), NULL), _tmp107);
			/* use floating variable to avoid unnecessary (and sometimes impossible) copies*/
			vala_local_variable_set_floating (local, TRUE);
			decl = vala_declaration_statement_new ((ValaSymbol*) local, vala_code_node_get_source_reference ((ValaCodeNode*) self));
			vala_expression_insert_statement ((ValaExpression*) self, analyzer->insert_block, (ValaStatement*) decl);
			temp_access = (ValaExpression*) vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) local), vala_code_node_get_source_reference ((ValaCodeNode*) self));
			vala_expression_set_target_type (temp_access, vala_expression_get_target_type ((ValaExpression*) self));
			/* don't set initializer earlier as this changes parent_node and parent_statement*/
			vala_local_variable_set_initializer (local, (ValaExpression*) self);
			vala_code_node_check ((ValaCodeNode*) decl, analyzer);
			vala_code_node_check ((ValaCodeNode*) temp_access, analyzer);
			/* move temp variable to insert block to ensure the
			 variable is in the same block as the declaration
			 otherwise there will be scoping issues in the generated code*/
			_tmp108 = NULL;
			block = (_tmp108 = VALA_BLOCK (vala_semantic_analyzer_get_current_symbol (analyzer)), (_tmp108 == NULL) ? NULL : vala_code_node_ref (_tmp108));
			vala_block_remove_local_variable (block, local);
			vala_block_add_local_variable (analyzer->insert_block, local);
			_tmp110 = NULL;
			_tmp109 = NULL;
			analyzer->insert_block = (_tmp110 = (_tmp109 = old_insert_block, (_tmp109 == NULL) ? NULL : vala_code_node_ref (_tmp109)), (analyzer->insert_block == NULL) ? NULL : (analyzer->insert_block = (vala_code_node_unref (analyzer->insert_block), NULL)), _tmp110);
			vala_code_node_replace_expression (old_parent_node, (ValaExpression*) self, temp_access);
			(old_insert_block == NULL) ? NULL : (old_insert_block = (vala_code_node_unref (old_insert_block), NULL));
			(old_parent_node == NULL) ? NULL : (old_parent_node = (vala_code_node_unref (old_parent_node), NULL));
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
			(decl == NULL) ? NULL : (decl = (vala_code_node_unref (decl), NULL));
			(temp_access == NULL) ? NULL : (temp_access = (vala_code_node_unref (temp_access), NULL));
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
		}
	}
	return (_tmp111 = !vala_code_node_get_error ((ValaCodeNode*) self), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), (last_arg == NULL) ? NULL : (last_arg = (vala_code_node_unref (last_arg), NULL)), (args == NULL) ? NULL : (args = (gee_collection_object_unref (args), NULL)), (arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL)), (ret_type == NULL) ? NULL : (ret_type = (vala_code_node_unref (ret_type), NULL)), _tmp111);
}


static void vala_method_call_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaMethodCall * self;
	self = (ValaMethodCall*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_method_call_get_call (self), collection);
	{
		GeeIterator* _arg_it;
		_arg_it = gee_iterable_iterator ((GeeIterable*) self->priv->argument_list);
		while (gee_iterator_next (_arg_it)) {
			ValaExpression* arg;
			arg = (ValaExpression*) gee_iterator_get (_arg_it);
			vala_code_node_get_defined_variables ((ValaCodeNode*) arg, collection);
			(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
		}
		(_arg_it == NULL) ? NULL : (_arg_it = (gee_collection_object_unref (_arg_it), NULL));
	}
}


static void vala_method_call_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaMethodCall * self;
	self = (ValaMethodCall*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_used_variables ((ValaCodeNode*) vala_method_call_get_call (self), collection);
	{
		GeeIterator* _arg_it;
		_arg_it = gee_iterable_iterator ((GeeIterable*) self->priv->argument_list);
		while (gee_iterator_next (_arg_it)) {
			ValaExpression* arg;
			arg = (ValaExpression*) gee_iterator_get (_arg_it);
			vala_code_node_get_used_variables ((ValaCodeNode*) arg, collection);
			(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
		}
		(_arg_it == NULL) ? NULL : (_arg_it = (gee_collection_object_unref (_arg_it), NULL));
	}
}


ValaExpression* vala_method_call_get_call (ValaMethodCall* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->_call;
}


void vala_method_call_set_call (ValaMethodCall* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->_call = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->_call == NULL) ? NULL : (self->_call = (vala_code_node_unref (self->_call), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->_call, (ValaCodeNode*) self);
}


ValaCCodeExpression* vala_method_call_get_delegate_target (ValaMethodCall* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_delegate_target;
}


void vala_method_call_set_delegate_target (ValaMethodCall* self, ValaCCodeExpression* value) {
	ValaCCodeExpression* _tmp2;
	ValaCCodeExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_delegate_target = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_ccode_node_ref (_tmp1)), (self->priv->_delegate_target == NULL) ? NULL : (self->priv->_delegate_target = (vala_ccode_node_unref (self->priv->_delegate_target), NULL)), _tmp2);
}


static void vala_method_call_class_init (ValaMethodCallClass * klass) {
	vala_method_call_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_method_call_finalize;
	g_type_class_add_private (klass, sizeof (ValaMethodCallPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_method_call_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_method_call_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_method_call_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_method_call_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->check = vala_method_call_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_method_call_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_method_call_real_get_used_variables;
}


static void vala_method_call_instance_init (ValaMethodCall * self) {
	self->priv = VALA_METHOD_CALL_GET_PRIVATE (self);
	self->priv->argument_list = (GeeList*) gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->array_sizes = (GeeList*) gee_array_list_new (VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_equal);
}


static void vala_method_call_finalize (ValaCodeNode* obj) {
	ValaMethodCall * self;
	self = VALA_METHOD_CALL (obj);
	(self->priv->_delegate_target == NULL) ? NULL : (self->priv->_delegate_target = (vala_ccode_node_unref (self->priv->_delegate_target), NULL));
	(self->_call == NULL) ? NULL : (self->_call = (vala_code_node_unref (self->_call), NULL));
	(self->priv->argument_list == NULL) ? NULL : (self->priv->argument_list = (gee_collection_object_unref (self->priv->argument_list), NULL));
	(self->priv->array_sizes == NULL) ? NULL : (self->priv->array_sizes = (gee_collection_object_unref (self->priv->array_sizes), NULL));
	VALA_CODE_NODE_CLASS (vala_method_call_parent_class)->finalize (obj);
}


GType vala_method_call_get_type (void) {
	static GType vala_method_call_type_id = 0;
	if (vala_method_call_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMethodCallClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_method_call_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMethodCall), 0, (GInstanceInitFunc) vala_method_call_instance_init, NULL };
		vala_method_call_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaMethodCall", &g_define_type_info, 0);
	}
	return vala_method_call_type_id;
}




