/* valaassignment.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaassignment.h>
#include <stdlib.h>
#include <string.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/list.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valamemberaccess.h>
#include <vala/valasignal.h>
#include <vala/valadynamicproperty.h>
#include <vala/valareport.h>
#include <vala/valasymbol.h>
#include <vala/valadynamicsignal.h>
#include <vala/valadelegatetype.h>
#include <vala/valadatatype.h>
#include <vala/valadelegate.h>
#include <vala/valaelementaccess.h>
#include <vala/valamethod.h>
#include <vala/valamethodcall.h>
#include <vala/valapointerindirection.h>
#include <vala/valabinaryexpression.h>
#include <vala/valaparenthesizedexpression.h>
#include <vala/valaformalparameter.h>
#include <vala/valaobjecttype.h>
#include <vala/valaobjecttypesymbol.h>
#include <vala/valaproperty.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valacreationmethod.h>
#include <vala/valalocalvariable.h>
#include <vala/valafield.h>
#include <vala/valapointertype.h>
#include <vala/valaarraytype.h>




struct _ValaAssignmentPrivate {
	ValaAssignmentOperator _operator;
	ValaExpression* _left;
	ValaExpression* _right;
};

#define VALA_ASSIGNMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ASSIGNMENT, ValaAssignmentPrivate))
enum  {
	VALA_ASSIGNMENT_DUMMY_PROPERTY
};
static void vala_assignment_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_assignment_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_assignment_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_assignment_real_is_pure (ValaExpression* base);
static gboolean vala_assignment_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_assignment_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
static void vala_assignment_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
static gpointer vala_assignment_parent_class = NULL;
static void vala_assignment_finalize (ValaCodeNode* obj);



/**
 * Creates a new assignment.
 *
 * @param left             left hand side
 * @param operator         assignment operator
 * @param right            right hand side
 * @param source_reference reference to source code
 * @return                 newly created assignment
 */
ValaAssignment* vala_assignment_construct (GType object_type, ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference) {
	ValaAssignment* self;
	g_return_val_if_fail (left != NULL, NULL);
	g_return_val_if_fail (right != NULL, NULL);
	self = (ValaAssignment*) g_type_create_instance (object_type);
	vala_assignment_set_right (self, right);
	vala_assignment_set_operator (self, operator);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_assignment_set_left (self, left);
	return self;
}


ValaAssignment* vala_assignment_new (ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference) {
	return vala_assignment_construct (VALA_TYPE_ASSIGNMENT, left, right, operator, source_reference);
}


static void vala_assignment_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_assignment (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static void vala_assignment_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_assignment_get_left (self), visitor);
	vala_code_node_accept ((ValaCodeNode*) vala_assignment_get_right (self), visitor);
}


static void vala_assignment_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_assignment_get_left (self) == old_node) {
		vala_assignment_set_left (self, new_node);
	}
	if (vala_assignment_get_right (self) == old_node) {
		vala_assignment_set_right (self, new_node);
	}
}


static gboolean vala_assignment_real_is_pure (ValaExpression* base) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	return FALSE;
}


static gboolean vala_assignment_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaAssignment * self;
	gboolean _tmp29;
	GeeList* _tmp113;
	GeeList* _tmp114;
	self = (ValaAssignment*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_expression_set_lvalue (vala_assignment_get_left (self), TRUE);
	if (!vala_code_node_check ((ValaCodeNode*) vala_assignment_get_left (self), analyzer)) {
		/* skip on error in inner expression*/
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self))) {
		ValaMemberAccess* _tmp2;
		ValaMemberAccess* ma;
		gboolean _tmp3;
		gboolean _tmp4;
		gboolean _tmp9;
		_tmp2 = NULL;
		ma = (_tmp2 = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
		_tmp3 = FALSE;
		_tmp4 = FALSE;
		if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
			_tmp4 = TRUE;
		} else {
			_tmp4 = VALA_IS_DYNAMIC_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma));
		}
		if (!(_tmp4)) {
			_tmp3 = vala_expression_get_value_type ((ValaExpression*) ma) == NULL;
		} else {
			_tmp3 = FALSE;
		}
		if (_tmp3) {
			gboolean _tmp5;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "unsupported lvalue in assignment");
			return (_tmp5 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp5);
		}
		if (vala_member_access_get_prototype_access (ma)) {
			char* _tmp7;
			char* _tmp6;
			gboolean _tmp8;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp7 = NULL;
			_tmp6 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp7 = g_strdup_printf ("Access to instance member `%s' denied", _tmp6 = vala_symbol_get_full_name (vala_expression_get_symbol_reference ((ValaExpression*) ma))));
			_tmp7 = (g_free (_tmp7), NULL);
			_tmp6 = (g_free (_tmp6), NULL);
			return (_tmp8 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp8);
		}
		_tmp9 = FALSE;
		if (vala_code_node_get_error ((ValaCodeNode*) ma)) {
			_tmp9 = TRUE;
		} else {
			_tmp9 = vala_expression_get_symbol_reference ((ValaExpression*) ma) == NULL;
		}
		if (_tmp9) {
			gboolean _tmp10;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			/* if no symbol found, skip this check */
			return (_tmp10 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp10);
		}
		if (VALA_IS_DYNAMIC_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
		} else {
			if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
				ValaSignal* _tmp11;
				ValaSignal* sig;
				ValaDelegateType* _tmp13;
				ValaDelegate* _tmp12;
				_tmp11 = NULL;
				sig = (_tmp11 = VALA_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp11 == NULL) ? NULL : vala_code_node_ref (_tmp11));
				_tmp13 = NULL;
				_tmp12 = NULL;
				vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp13 = vala_delegate_type_new (_tmp12 = vala_signal_get_delegate (sig, vala_expression_get_value_type (vala_member_access_get_inner (ma)), (ValaCodeNode*) self))));
				(_tmp13 == NULL) ? NULL : (_tmp13 = (vala_code_node_unref (_tmp13), NULL));
				(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_code_node_unref (_tmp12), NULL));
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			} else {
				vala_expression_set_formal_target_type (vala_assignment_get_right (self), vala_expression_get_formal_value_type ((ValaExpression*) ma));
				vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type ((ValaExpression*) ma));
			}
		}
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	} else {
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (self))) {
			ValaElementAccess* _tmp14;
			ValaElementAccess* ea;
			gboolean _tmp15;
			_tmp14 = NULL;
			ea = (_tmp14 = VALA_ELEMENT_ACCESS (vala_assignment_get_left (self)), (_tmp14 == NULL) ? NULL : vala_code_node_ref (_tmp14));
			_tmp15 = FALSE;
			if (VALA_IS_MEMBER_ACCESS (vala_element_access_get_container (ea))) {
				_tmp15 = VALA_IS_SIGNAL (vala_expression_get_symbol_reference (vala_element_access_get_container (ea)));
			} else {
				_tmp15 = FALSE;
			}
			if (_tmp15) {
				ValaMemberAccess* _tmp16;
				ValaMemberAccess* ma;
				ValaSignal* _tmp17;
				ValaSignal* sig;
				ValaDelegateType* _tmp19;
				ValaDelegate* _tmp18;
				_tmp16 = NULL;
				ma = (_tmp16 = VALA_MEMBER_ACCESS (vala_element_access_get_container (ea)), (_tmp16 == NULL) ? NULL : vala_code_node_ref (_tmp16));
				_tmp17 = NULL;
				sig = (_tmp17 = VALA_SIGNAL (vala_expression_get_symbol_reference (vala_element_access_get_container (ea))), (_tmp17 == NULL) ? NULL : vala_code_node_ref (_tmp17));
				_tmp19 = NULL;
				_tmp18 = NULL;
				vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp19 = vala_delegate_type_new (_tmp18 = vala_signal_get_delegate (sig, vala_expression_get_value_type (vala_member_access_get_inner (ma)), (ValaCodeNode*) self))));
				(_tmp19 == NULL) ? NULL : (_tmp19 = (vala_code_node_unref (_tmp19), NULL));
				(_tmp18 == NULL) ? NULL : (_tmp18 = (vala_code_node_unref (_tmp18), NULL));
				(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			} else {
				ValaSymbol* _tmp20;
				gboolean _tmp21;
				_tmp20 = NULL;
				if ((_tmp21 = VALA_IS_METHOD (_tmp20 = vala_data_type_get_member (vala_expression_get_value_type (vala_element_access_get_container (ea)), "set")), (_tmp20 == NULL) ? NULL : (_tmp20 = (vala_code_node_unref (_tmp20), NULL)), _tmp21)) {
					ValaMemberAccess* _tmp22;
					ValaMethodCall* _tmp23;
					ValaMethodCall* set_call;
					gboolean _tmp26;
					_tmp22 = NULL;
					_tmp23 = NULL;
					set_call = (_tmp23 = vala_method_call_new ((ValaExpression*) (_tmp22 = vala_member_access_new (vala_element_access_get_container (ea), "set", NULL)), NULL), (_tmp22 == NULL) ? NULL : (_tmp22 = (vala_code_node_unref (_tmp22), NULL)), _tmp23);
					{
						GeeList* _tmp24;
						GeeIterator* _tmp25;
						GeeIterator* _e_it;
						_tmp24 = NULL;
						_tmp25 = NULL;
						_e_it = (_tmp25 = gee_iterable_iterator ((GeeIterable*) (_tmp24 = vala_element_access_get_indices (ea))), (_tmp24 == NULL) ? NULL : (_tmp24 = (gee_collection_object_unref (_tmp24), NULL)), _tmp25);
						while (gee_iterator_next (_e_it)) {
							ValaExpression* e;
							e = (ValaExpression*) gee_iterator_get (_e_it);
							vala_method_call_add_argument (set_call, e);
							(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
						}
						(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
					}
					vala_method_call_add_argument (set_call, vala_assignment_get_right (self));
					vala_code_node_replace_expression (vala_code_node_get_parent_node ((ValaCodeNode*) self), (ValaExpression*) self, (ValaExpression*) set_call);
					return (_tmp26 = vala_code_node_check ((ValaCodeNode*) set_call, analyzer), (set_call == NULL) ? NULL : (set_call = (vala_code_node_unref (set_call), NULL)), (ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL)), _tmp26);
				} else {
					vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type (vala_assignment_get_left (self)));
				}
			}
			(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
		} else {
			if (VALA_IS_POINTER_INDIRECTION (vala_assignment_get_left (self))) {
				vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type (vala_assignment_get_left (self)));
			} else {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "unsupported lvalue in assignment");
				return FALSE;
			}
		}
	}
	if (!vala_code_node_check ((ValaCodeNode*) vala_assignment_get_right (self), analyzer)) {
		/* skip on error in inner expression*/
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	_tmp29 = FALSE;
	if (self->priv->_operator != VALA_ASSIGNMENT_OPERATOR_SIMPLE) {
		_tmp29 = VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self));
	} else {
		_tmp29 = FALSE;
	}
	if (_tmp29) {
		ValaMemberAccess* _tmp30;
		ValaMemberAccess* ma;
		/* transform into simple assignment
		 FIXME: only do this if the backend doesn't support
		 the assignment natively*/
		_tmp30 = NULL;
		ma = (_tmp30 = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp30 == NULL) ? NULL : vala_code_node_ref (_tmp30));
		if (!(VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma)))) {
			ValaMemberAccess* old_value;
			ValaParenthesizedExpression* _tmp31;
			ValaBinaryExpression* _tmp32;
			ValaBinaryExpression* bin;
			ValaDataType* _tmp33;
			old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
			_tmp31 = NULL;
			_tmp32 = NULL;
			bin = (_tmp32 = vala_binary_expression_new (VALA_BINARY_OPERATOR_PLUS, (ValaExpression*) old_value, (ValaExpression*) (_tmp31 = vala_parenthesized_expression_new (vala_assignment_get_right (self), vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)))), NULL), (_tmp31 == NULL) ? NULL : (_tmp31 = (vala_code_node_unref (_tmp31), NULL)), _tmp32);
			vala_expression_set_target_type ((ValaExpression*) bin, vala_expression_get_target_type (vala_assignment_get_right (self)));
			_tmp33 = NULL;
			vala_expression_set_target_type (vala_assignment_get_right (self), _tmp33 = vala_data_type_copy (vala_expression_get_target_type (vala_assignment_get_right (self))));
			(_tmp33 == NULL) ? NULL : (_tmp33 = (vala_code_node_unref (_tmp33), NULL));
			vala_data_type_set_value_owned (vala_expression_get_target_type (vala_assignment_get_right (self)), FALSE);
			if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_OR) {
				vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_OR);
			} else {
				if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_AND) {
					vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_AND);
				} else {
					if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR) {
						vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_XOR);
					} else {
						if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_ADD) {
							vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_PLUS);
						} else {
							if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SUB) {
								vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MINUS);
							} else {
								if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_MUL) {
									vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MUL);
								} else {
									if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_DIV) {
										vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_DIV);
									} else {
										if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_PERCENT) {
											vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MOD);
										} else {
											if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT) {
												vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_LEFT);
											} else {
												if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT) {
													vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_RIGHT);
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			vala_assignment_set_right (self, (ValaExpression*) bin);
			vala_code_node_check ((ValaCodeNode*) vala_assignment_get_right (self), analyzer);
			vala_assignment_set_operator (self, VALA_ASSIGNMENT_OPERATOR_SIMPLE);
			(old_value == NULL) ? NULL : (old_value = (vala_code_node_unref (old_value), NULL));
			(bin == NULL) ? NULL : (bin = (vala_code_node_unref (bin), NULL));
		}
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	}
	if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference (vala_assignment_get_left (self)))) {
		ValaSignal* _tmp34;
		ValaSignal* sig;
		ValaMethod* _tmp36;
		ValaSymbol* _tmp35;
		ValaMethod* m;
		ValaDynamicSignal* _tmp39;
		ValaSignal* _tmp38;
		ValaDynamicSignal* dynamic_sig;
		ValaMemberAccess* _tmp41;
		ValaExpression* _tmp40;
		ValaMemberAccess* right_ma;
		_tmp34 = NULL;
		sig = (_tmp34 = VALA_SIGNAL (vala_expression_get_symbol_reference (vala_assignment_get_left (self))), (_tmp34 == NULL) ? NULL : vala_code_node_ref (_tmp34));
		_tmp36 = NULL;
		_tmp35 = NULL;
		m = (_tmp36 = (_tmp35 = vala_expression_get_symbol_reference (vala_assignment_get_right (self)), VALA_IS_METHOD (_tmp35) ? ((ValaMethod*) _tmp35) : NULL), (_tmp36 == NULL) ? NULL : vala_code_node_ref (_tmp36));
		if (m == NULL) {
			gboolean _tmp37;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), "unsupported expression for signal handler");
			return (_tmp37 = FALSE, (sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), _tmp37);
		}
		_tmp39 = NULL;
		_tmp38 = NULL;
		dynamic_sig = (_tmp39 = (_tmp38 = sig, VALA_IS_DYNAMIC_SIGNAL (_tmp38) ? ((ValaDynamicSignal*) _tmp38) : NULL), (_tmp39 == NULL) ? NULL : vala_code_node_ref (_tmp39));
		_tmp41 = NULL;
		_tmp40 = NULL;
		right_ma = (_tmp41 = (_tmp40 = vala_assignment_get_right (self), VALA_IS_MEMBER_ACCESS (_tmp40) ? ((ValaMemberAccess*) _tmp40) : NULL), (_tmp41 == NULL) ? NULL : vala_code_node_ref (_tmp41));
		if (dynamic_sig != NULL) {
			gboolean first;
			ValaDelegateType* _tmp47;
			ValaDelegate* _tmp46;
			ValaObjectType* _tmp45;
			first = TRUE;
			{
				GeeList* _tmp42;
				GeeIterator* _tmp43;
				GeeIterator* _param_it;
				_tmp42 = NULL;
				_tmp43 = NULL;
				_param_it = (_tmp43 = gee_iterable_iterator ((GeeIterable*) (_tmp42 = vala_data_type_get_parameters (vala_expression_get_value_type (vala_dynamic_signal_get_handler (dynamic_sig))))), (_tmp42 == NULL) ? NULL : (_tmp42 = (gee_collection_object_unref (_tmp42), NULL)), _tmp43);
				while (gee_iterator_next (_param_it)) {
					ValaFormalParameter* param;
					param = (ValaFormalParameter*) gee_iterator_get (_param_it);
					if (first) {
						/* skip sender parameter*/
						first = FALSE;
					} else {
						ValaFormalParameter* _tmp44;
						_tmp44 = NULL;
						vala_signal_add_parameter ((ValaSignal*) dynamic_sig, _tmp44 = vala_formal_parameter_copy (param));
						(_tmp44 == NULL) ? NULL : (_tmp44 = (vala_code_node_unref (_tmp44), NULL));
					}
					(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				}
				(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
			}
			_tmp47 = NULL;
			_tmp46 = NULL;
			_tmp45 = NULL;
			vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp47 = vala_delegate_type_new (_tmp46 = vala_signal_get_delegate (sig, (ValaDataType*) (_tmp45 = vala_object_type_new (VALA_OBJECT_TYPE_SYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) sig)))), (ValaCodeNode*) self))));
			(_tmp47 == NULL) ? NULL : (_tmp47 = (vala_code_node_unref (_tmp47), NULL));
			(_tmp46 == NULL) ? NULL : (_tmp46 = (vala_code_node_unref (_tmp46), NULL));
			(_tmp45 == NULL) ? NULL : (_tmp45 = (vala_code_node_unref (_tmp45), NULL));
		} else {
			if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_target_type (vala_assignment_get_right (self)))) {
				ValaDelegateType* _tmp48;
				ValaDelegateType* delegate_type;
				char* _tmp52;
				char* _tmp51;
				char* _tmp50;
				char* _tmp49;
				gboolean _tmp53;
				_tmp48 = NULL;
				delegate_type = (_tmp48 = VALA_DELEGATE_TYPE (vala_expression_get_target_type (vala_assignment_get_right (self))), (_tmp48 == NULL) ? NULL : vala_code_node_ref (_tmp48));
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp52 = NULL;
				_tmp51 = NULL;
				_tmp50 = NULL;
				_tmp49 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), _tmp52 = g_strdup_printf ("method `%s' is incompatible with signal `%s', expected `%s'", _tmp49 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp50 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_target_type (vala_assignment_get_right (self))), _tmp51 = vala_delegate_get_prototype_string (vala_delegate_type_get_delegate_symbol (delegate_type), vala_symbol_get_name ((ValaSymbol*) m))));
				_tmp52 = (g_free (_tmp52), NULL);
				_tmp51 = (g_free (_tmp51), NULL);
				_tmp50 = (g_free (_tmp50), NULL);
				_tmp49 = (g_free (_tmp49), NULL);
				return (_tmp53 = FALSE, (delegate_type == NULL) ? NULL : (delegate_type = (vala_code_node_unref (delegate_type), NULL)), (sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL)), (right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL)), _tmp53);
			} else {
				gboolean _tmp54;
				_tmp54 = FALSE;
				if (right_ma != NULL) {
					_tmp54 = vala_member_access_get_prototype_access (right_ma);
				} else {
					_tmp54 = FALSE;
				}
				if (_tmp54) {
					char* _tmp56;
					char* _tmp55;
					gboolean _tmp57;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp56 = NULL;
					_tmp55 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), _tmp56 = g_strdup_printf ("Access to instance member `%s' denied", _tmp55 = vala_symbol_get_full_name ((ValaSymbol*) m)));
					_tmp56 = (g_free (_tmp56), NULL);
					_tmp55 = (g_free (_tmp55), NULL);
					return (_tmp57 = FALSE, (sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL)), (right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL)), _tmp57);
				}
			}
		}
		(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
		(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		(dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL));
		(right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL));
	} else {
		if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self))) {
			ValaMemberAccess* _tmp58;
			ValaMemberAccess* ma;
			gboolean _tmp94;
			_tmp58 = NULL;
			ma = (_tmp58 = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp58 == NULL) ? NULL : vala_code_node_ref (_tmp58));
			if (VALA_IS_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
				ValaProperty* _tmp59;
				ValaProperty* prop;
				ValaDynamicProperty* _tmp61;
				ValaProperty* _tmp60;
				ValaDynamicProperty* dynamic_prop;
				gboolean _tmp64;
				_tmp59 = NULL;
				prop = (_tmp59 = VALA_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp59 == NULL) ? NULL : vala_code_node_ref (_tmp59));
				_tmp61 = NULL;
				_tmp60 = NULL;
				dynamic_prop = (_tmp61 = (_tmp60 = prop, VALA_IS_DYNAMIC_PROPERTY (_tmp60) ? ((ValaDynamicProperty*) _tmp60) : NULL), (_tmp61 == NULL) ? NULL : vala_code_node_ref (_tmp61));
				if (dynamic_prop != NULL) {
					ValaDataType* _tmp62;
					ValaDataType* _tmp63;
					_tmp62 = NULL;
					vala_property_set_property_type ((ValaProperty*) dynamic_prop, _tmp62 = vala_data_type_copy (vala_expression_get_value_type (vala_assignment_get_right (self))));
					(_tmp62 == NULL) ? NULL : (_tmp62 = (vala_code_node_unref (_tmp62), NULL));
					_tmp63 = NULL;
					vala_expression_set_value_type (vala_assignment_get_left (self), _tmp63 = vala_data_type_copy (vala_property_get_property_type ((ValaProperty*) dynamic_prop)));
					(_tmp63 == NULL) ? NULL : (_tmp63 = (vala_code_node_unref (_tmp63), NULL));
				}
				_tmp64 = FALSE;
				if (vala_property_get_set_accessor (prop) == NULL) {
					_tmp64 = TRUE;
				} else {
					gboolean _tmp65;
					_tmp65 = FALSE;
					if (!vala_property_accessor_get_writable (vala_property_get_set_accessor (prop))) {
						gboolean _tmp66;
						ValaMethod* _tmp67;
						gboolean _tmp68;
						_tmp66 = FALSE;
						_tmp67 = NULL;
						if ((_tmp68 = VALA_IS_CREATION_METHOD (_tmp67 = vala_semantic_analyzer_find_current_method (analyzer)), (_tmp67 == NULL) ? NULL : (_tmp67 = (vala_code_node_unref (_tmp67), NULL)), _tmp68)) {
							_tmp66 = TRUE;
						} else {
							_tmp66 = vala_semantic_analyzer_is_in_constructor (analyzer);
						}
						_tmp65 = !(_tmp66);
					} else {
						_tmp65 = FALSE;
					}
					_tmp64 = (_tmp65);
				}
				if (_tmp64) {
					char* _tmp70;
					char* _tmp69;
					gboolean _tmp71;
					vala_code_node_set_error ((ValaCodeNode*) ma, TRUE);
					_tmp70 = NULL;
					_tmp69 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) ma), _tmp70 = g_strdup_printf ("Property `%s' is read-only", _tmp69 = vala_symbol_get_full_name ((ValaSymbol*) prop)));
					_tmp70 = (g_free (_tmp70), NULL);
					_tmp69 = (g_free (_tmp69), NULL);
					return (_tmp71 = FALSE, (prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL)), (dynamic_prop == NULL) ? NULL : (dynamic_prop = (vala_code_node_unref (dynamic_prop), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp71);
				}
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				(dynamic_prop == NULL) ? NULL : (dynamic_prop = (vala_code_node_unref (dynamic_prop), NULL));
			} else {
				gboolean _tmp72;
				_tmp72 = FALSE;
				if (VALA_IS_LOCAL_VARIABLE (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
					_tmp72 = vala_expression_get_value_type (vala_assignment_get_right (self)) == NULL;
				} else {
					_tmp72 = FALSE;
				}
				if (_tmp72) {
					ValaLocalVariable* _tmp73;
					ValaLocalVariable* local;
					gboolean _tmp74;
					_tmp73 = NULL;
					local = (_tmp73 = VALA_LOCAL_VARIABLE (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp73 == NULL) ? NULL : vala_code_node_ref (_tmp73));
					_tmp74 = FALSE;
					if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self)))) {
						_tmp74 = VALA_IS_DELEGATE_TYPE (vala_local_variable_get_variable_type (local));
					} else {
						_tmp74 = FALSE;
					}
					if (_tmp74) {
						ValaMethod* _tmp75;
						ValaMethod* m;
						ValaDelegateType* _tmp76;
						ValaDelegateType* dt;
						ValaDelegate* _tmp77;
						ValaDelegate* cb;
						_tmp75 = NULL;
						m = (_tmp75 = VALA_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self))), (_tmp75 == NULL) ? NULL : vala_code_node_ref (_tmp75));
						_tmp76 = NULL;
						dt = (_tmp76 = VALA_DELEGATE_TYPE (vala_local_variable_get_variable_type (local)), (_tmp76 == NULL) ? NULL : vala_code_node_ref (_tmp76));
						_tmp77 = NULL;
						cb = (_tmp77 = vala_delegate_type_get_delegate_symbol (dt), (_tmp77 == NULL) ? NULL : vala_code_node_ref (_tmp77));
						/* check whether method matches callback type */
						if (!vala_delegate_matches_method (cb, m)) {
							char* _tmp80;
							char* _tmp79;
							char* _tmp78;
							gboolean _tmp81;
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							_tmp80 = NULL;
							_tmp79 = NULL;
							_tmp78 = NULL;
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp80 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", _tmp78 = vala_symbol_get_full_name ((ValaSymbol*) m), _tmp79 = vala_symbol_get_full_name ((ValaSymbol*) cb)));
							_tmp80 = (g_free (_tmp80), NULL);
							_tmp79 = (g_free (_tmp79), NULL);
							_tmp78 = (g_free (_tmp78), NULL);
							return (_tmp81 = FALSE, (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), (cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL)), (local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp81);
						}
						vala_expression_set_value_type (vala_assignment_get_right (self), vala_local_variable_get_variable_type (local));
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
						(dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL));
						(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
					} else {
						gboolean _tmp82;
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Assignment: Invalid callback assignment attempt");
						return (_tmp82 = FALSE, (local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp82);
					}
					(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
				} else {
					gboolean _tmp83;
					_tmp83 = FALSE;
					if (VALA_IS_FIELD (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
						_tmp83 = vala_expression_get_value_type (vala_assignment_get_right (self)) == NULL;
					} else {
						_tmp83 = FALSE;
					}
					if (_tmp83) {
						ValaField* _tmp84;
						ValaField* f;
						gboolean _tmp85;
						_tmp84 = NULL;
						f = (_tmp84 = VALA_FIELD (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp84 == NULL) ? NULL : vala_code_node_ref (_tmp84));
						_tmp85 = FALSE;
						if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self)))) {
							_tmp85 = VALA_IS_DELEGATE_TYPE (vala_field_get_field_type (f));
						} else {
							_tmp85 = FALSE;
						}
						if (_tmp85) {
							ValaMethod* _tmp86;
							ValaMethod* m;
							ValaDelegateType* _tmp87;
							ValaDelegateType* dt;
							ValaDelegate* _tmp88;
							ValaDelegate* cb;
							_tmp86 = NULL;
							m = (_tmp86 = VALA_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self))), (_tmp86 == NULL) ? NULL : vala_code_node_ref (_tmp86));
							_tmp87 = NULL;
							dt = (_tmp87 = VALA_DELEGATE_TYPE (vala_field_get_field_type (f)), (_tmp87 == NULL) ? NULL : vala_code_node_ref (_tmp87));
							_tmp88 = NULL;
							cb = (_tmp88 = vala_delegate_type_get_delegate_symbol (dt), (_tmp88 == NULL) ? NULL : vala_code_node_ref (_tmp88));
							/* check whether method matches callback type */
							if (!vala_delegate_matches_method (cb, m)) {
								char* _tmp91;
								char* _tmp90;
								char* _tmp89;
								gboolean _tmp92;
								vala_code_node_set_error ((ValaCodeNode*) f, TRUE);
								_tmp91 = NULL;
								_tmp90 = NULL;
								_tmp89 = NULL;
								vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp91 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", _tmp89 = vala_symbol_get_full_name ((ValaSymbol*) m), _tmp90 = vala_symbol_get_full_name ((ValaSymbol*) cb)));
								_tmp91 = (g_free (_tmp91), NULL);
								_tmp90 = (g_free (_tmp90), NULL);
								_tmp89 = (g_free (_tmp89), NULL);
								return (_tmp92 = FALSE, (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), (cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL)), (f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp92);
							}
							vala_expression_set_value_type (vala_assignment_get_right (self), vala_field_get_field_type (f));
							(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
							(dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL));
							(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
						} else {
							gboolean _tmp93;
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Assignment: Invalid callback assignment attempt");
							return (_tmp93 = FALSE, (f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp93);
						}
						(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
					}
				}
			}
			_tmp94 = FALSE;
			if (vala_expression_get_value_type (vala_assignment_get_left (self)) != NULL) {
				_tmp94 = vala_expression_get_value_type (vala_assignment_get_right (self)) != NULL;
			} else {
				_tmp94 = FALSE;
			}
			if (_tmp94) {
				/* if there was an error on either side,
				 * i.e. {left|right}.value_type == null, skip type check */
				if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					char* _tmp97;
					char* _tmp96;
					char* _tmp95;
					gboolean _tmp98;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp97 = NULL;
					_tmp96 = NULL;
					_tmp95 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp97 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", _tmp95 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp96 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_left (self)))));
					_tmp97 = (g_free (_tmp97), NULL);
					_tmp96 = (g_free (_tmp96), NULL);
					_tmp95 = (g_free (_tmp95), NULL);
					return (_tmp98 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp98);
				}
				if (!(VALA_IS_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma)))) {
					if (vala_data_type_is_disposable (vala_expression_get_value_type (vala_assignment_get_right (self)))) {
						gboolean _tmp99;
						_tmp99 = FALSE;
						if (!(VALA_IS_POINTER_TYPE (vala_expression_get_value_type (vala_assignment_get_left (self))))) {
							_tmp99 = !vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)));
						} else {
							_tmp99 = FALSE;
						}
						/* rhs transfers ownership of the expression */
						if (_tmp99) {
							/* lhs doesn't own the value */
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Invalid assignment from owned expression to unowned variable");
						}
					} else {
						if (vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)))) {
						}
					}
				}
			}
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
		} else {
			if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (self))) {
				ValaElementAccess* _tmp100;
				ValaElementAccess* ea;
				_tmp100 = NULL;
				ea = (_tmp100 = VALA_ELEMENT_ACCESS (vala_assignment_get_left (self)), (_tmp100 == NULL) ? NULL : vala_code_node_ref (_tmp100));
				if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					char* _tmp103;
					char* _tmp102;
					char* _tmp101;
					gboolean _tmp104;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp103 = NULL;
					_tmp102 = NULL;
					_tmp101 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp103 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", _tmp101 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp102 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_left (self)))));
					_tmp103 = (g_free (_tmp103), NULL);
					_tmp102 = (g_free (_tmp102), NULL);
					_tmp101 = (g_free (_tmp101), NULL);
					return (_tmp104 = FALSE, (ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL)), _tmp104);
				}
				if (vala_data_type_is_disposable (vala_expression_get_value_type (vala_assignment_get_right (self)))) {
					ValaDataType* element_type;
					gboolean _tmp109;
					/* rhs transfers ownership of the expression */
					element_type = NULL;
					if (VALA_IS_ARRAY_TYPE (vala_expression_get_value_type (vala_element_access_get_container (ea)))) {
						ValaArrayType* _tmp105;
						ValaArrayType* array_type;
						ValaDataType* _tmp107;
						ValaDataType* _tmp106;
						_tmp105 = NULL;
						array_type = (_tmp105 = VALA_ARRAY_TYPE (vala_expression_get_value_type (vala_element_access_get_container (ea))), (_tmp105 == NULL) ? NULL : vala_code_node_ref (_tmp105));
						_tmp107 = NULL;
						_tmp106 = NULL;
						element_type = (_tmp107 = (_tmp106 = vala_array_type_get_element_type (array_type), (_tmp106 == NULL) ? NULL : vala_code_node_ref (_tmp106)), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp107);
						(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
					} else {
						GeeList* args;
						ValaDataType* _tmp108;
						args = vala_data_type_get_type_arguments (vala_expression_get_value_type (vala_element_access_get_container (ea)));
						g_assert (gee_collection_get_size ((GeeCollection*) args) == 1);
						_tmp108 = NULL;
						element_type = (_tmp108 = (ValaDataType*) gee_list_get (args, 0), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp108);
						(args == NULL) ? NULL : (args = (gee_collection_object_unref (args), NULL));
					}
					_tmp109 = FALSE;
					if (!(VALA_IS_POINTER_TYPE (element_type))) {
						_tmp109 = !vala_data_type_get_value_owned (element_type);
					} else {
						_tmp109 = FALSE;
					}
					if (_tmp109) {
						gboolean _tmp110;
						/* lhs doesn't own the value */
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Invalid assignment from owned expression to unowned variable");
						return (_tmp110 = FALSE, (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), (ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL)), _tmp110);
					}
					(element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL));
				} else {
					if (vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					}
				}
				(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
			} else {
				/* lhs wants to own the value
				 * rhs doesn't transfer the ownership
				 * code generator needs to add reference
				 * increment calls */
				return TRUE;
			}
		}
	}
	if (vala_expression_get_value_type (vala_assignment_get_left (self)) != NULL) {
		ValaDataType* _tmp112;
		_tmp112 = NULL;
		vala_expression_set_value_type ((ValaExpression*) self, _tmp112 = vala_data_type_copy (vala_expression_get_value_type (vala_assignment_get_left (self))));
		(_tmp112 == NULL) ? NULL : (_tmp112 = (vala_code_node_unref (_tmp112), NULL));
		vala_data_type_set_value_owned (vala_expression_get_value_type ((ValaExpression*) self), FALSE);
	} else {
		vala_expression_set_value_type ((ValaExpression*) self, NULL);
	}
	_tmp113 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp113 = vala_code_node_get_error_types ((ValaCodeNode*) vala_assignment_get_left (self)));
	(_tmp113 == NULL) ? NULL : (_tmp113 = (gee_collection_object_unref (_tmp113), NULL));
	_tmp114 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp114 = vala_code_node_get_error_types ((ValaCodeNode*) vala_assignment_get_right (self)));
	(_tmp114 == NULL) ? NULL : (_tmp114 = (gee_collection_object_unref (_tmp114), NULL));
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


static void vala_assignment_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaAssignment * self;
	ValaLocalVariable* _tmp1;
	ValaSymbol* _tmp0;
	ValaLocalVariable* local;
	self = (ValaAssignment*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_assignment_get_right (self), collection);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_assignment_get_left (self), collection);
	_tmp1 = NULL;
	_tmp0 = NULL;
	local = (_tmp1 = (_tmp0 = vala_expression_get_symbol_reference (vala_assignment_get_left (self)), VALA_IS_LOCAL_VARIABLE (_tmp0) ? ((ValaLocalVariable*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	if (local != NULL) {
		gee_collection_add (collection, local);
	}
	(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
}


static void vala_assignment_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaAssignment * self;
	ValaMemberAccess* _tmp1;
	ValaExpression* _tmp0;
	ValaMemberAccess* ma;
	ValaElementAccess* _tmp3;
	ValaExpression* _tmp2;
	ValaElementAccess* ea;
	gboolean _tmp4;
	self = (ValaAssignment*) base;
	g_return_if_fail (collection != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	ma = (_tmp1 = (_tmp0 = vala_assignment_get_left (self), VALA_IS_MEMBER_ACCESS (_tmp0) ? ((ValaMemberAccess*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	_tmp3 = NULL;
	_tmp2 = NULL;
	ea = (_tmp3 = (_tmp2 = vala_assignment_get_left (self), VALA_IS_ELEMENT_ACCESS (_tmp2) ? ((ValaElementAccess*) _tmp2) : NULL), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3));
	_tmp4 = FALSE;
	if (ma != NULL) {
		_tmp4 = vala_member_access_get_inner (ma) != NULL;
	} else {
		_tmp4 = FALSE;
	}
	if (_tmp4) {
		vala_code_node_get_used_variables ((ValaCodeNode*) vala_member_access_get_inner (ma), collection);
	} else {
		if (ea != NULL) {
			vala_code_node_get_used_variables ((ValaCodeNode*) ea, collection);
		}
	}
	vala_code_node_get_used_variables ((ValaCodeNode*) vala_assignment_get_right (self), collection);
	(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
}


ValaExpression* vala_assignment_get_left (ValaAssignment* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_left;
}


void vala_assignment_set_left (ValaAssignment* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_left = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_code_node_unref (self->priv->_left), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_left, (ValaCodeNode*) self);
}


ValaAssignmentOperator vala_assignment_get_operator (ValaAssignment* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_operator;
}


void vala_assignment_set_operator (ValaAssignment* self, ValaAssignmentOperator value) {
	g_return_if_fail (self != NULL);
	self->priv->_operator = value;
}


ValaExpression* vala_assignment_get_right (ValaAssignment* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_right;
}


void vala_assignment_set_right (ValaAssignment* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_right = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_code_node_unref (self->priv->_right), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_right, (ValaCodeNode*) self);
}


static void vala_assignment_class_init (ValaAssignmentClass * klass) {
	vala_assignment_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_assignment_finalize;
	g_type_class_add_private (klass, sizeof (ValaAssignmentPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_assignment_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_assignment_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_assignment_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_assignment_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->check = vala_assignment_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_assignment_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_assignment_real_get_used_variables;
}


static void vala_assignment_instance_init (ValaAssignment * self) {
	self->priv = VALA_ASSIGNMENT_GET_PRIVATE (self);
}


static void vala_assignment_finalize (ValaCodeNode* obj) {
	ValaAssignment * self;
	self = VALA_ASSIGNMENT (obj);
	(self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_code_node_unref (self->priv->_left), NULL));
	(self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_code_node_unref (self->priv->_right), NULL));
	VALA_CODE_NODE_CLASS (vala_assignment_parent_class)->finalize (obj);
}


GType vala_assignment_get_type (void) {
	static GType vala_assignment_type_id = 0;
	if (vala_assignment_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaAssignmentClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_assignment_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaAssignment), 0, (GInstanceInitFunc) vala_assignment_instance_init, NULL };
		vala_assignment_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaAssignment", &g_define_type_info, 0);
	}
	return vala_assignment_type_id;
}



GType vala_assignment_operator_get_type (void) {
	static GType vala_assignment_operator_type_id = 0;
	if (G_UNLIKELY (vala_assignment_operator_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_ASSIGNMENT_OPERATOR_NONE, "VALA_ASSIGNMENT_OPERATOR_NONE", "none"}, {VALA_ASSIGNMENT_OPERATOR_SIMPLE, "VALA_ASSIGNMENT_OPERATOR_SIMPLE", "simple"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_OR, "VALA_ASSIGNMENT_OPERATOR_BITWISE_OR", "bitwise-or"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_AND, "VALA_ASSIGNMENT_OPERATOR_BITWISE_AND", "bitwise-and"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR, "VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR", "bitwise-xor"}, {VALA_ASSIGNMENT_OPERATOR_ADD, "VALA_ASSIGNMENT_OPERATOR_ADD", "add"}, {VALA_ASSIGNMENT_OPERATOR_SUB, "VALA_ASSIGNMENT_OPERATOR_SUB", "sub"}, {VALA_ASSIGNMENT_OPERATOR_MUL, "VALA_ASSIGNMENT_OPERATOR_MUL", "mul"}, {VALA_ASSIGNMENT_OPERATOR_DIV, "VALA_ASSIGNMENT_OPERATOR_DIV", "div"}, {VALA_ASSIGNMENT_OPERATOR_PERCENT, "VALA_ASSIGNMENT_OPERATOR_PERCENT", "percent"}, {VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT, "VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT", "shift-left"}, {VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT, "VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT", "shift-right"}, {0, NULL, NULL}};
		vala_assignment_operator_type_id = g_enum_register_static ("ValaAssignmentOperator", values);
	}
	return vala_assignment_operator_type_id;
}




