/* valaassignment.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaassignment.h>
#include <stdlib.h>
#include <string.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/list.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valamemberaccess.h>
#include <vala/valasignal.h>
#include <vala/valadynamicproperty.h>
#include <vala/valareport.h>
#include <vala/valasymbol.h>
#include <vala/valadynamicsignal.h>
#include <vala/valadelegatetype.h>
#include <vala/valadatatype.h>
#include <vala/valadelegate.h>
#include <vala/valaelementaccess.h>
#include <vala/valamethod.h>
#include <vala/valamethodcall.h>
#include <vala/valapointerindirection.h>
#include <vala/valabinaryexpression.h>
#include <vala/valaformalparameter.h>
#include <vala/valaobjecttype.h>
#include <vala/valaobjecttypesymbol.h>
#include <vala/valaproperty.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valacreationmethod.h>
#include <vala/valalocalvariable.h>
#include <vala/valafield.h>
#include <vala/valapointertype.h>
#include <vala/valaarraytype.h>




struct _ValaAssignmentPrivate {
	ValaAssignmentOperator _operator;
	ValaExpression* _left;
	ValaExpression* _right;
};

#define VALA_ASSIGNMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ASSIGNMENT, ValaAssignmentPrivate))
enum  {
	VALA_ASSIGNMENT_DUMMY_PROPERTY
};
static void vala_assignment_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_assignment_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_assignment_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_assignment_real_is_pure (ValaExpression* base);
static gboolean vala_assignment_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_assignment_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
static void vala_assignment_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
static gpointer vala_assignment_parent_class = NULL;
static void vala_assignment_finalize (ValaCodeNode* obj);



/**
 * Creates a new assignment.
 *
 * @param left             left hand side
 * @param operator         assignment operator
 * @param right            right hand side
 * @param source_reference reference to source code
 * @return                 newly created assignment
 */
ValaAssignment* vala_assignment_construct (GType object_type, ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference) {
	ValaAssignment* self;
	g_return_val_if_fail (left != NULL, NULL);
	g_return_val_if_fail (right != NULL, NULL);
	self = (ValaAssignment*) g_type_create_instance (object_type);
	vala_assignment_set_right (self, right);
	vala_assignment_set_operator (self, operator);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_assignment_set_left (self, left);
	return self;
}


ValaAssignment* vala_assignment_new (ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference) {
	return vala_assignment_construct (VALA_TYPE_ASSIGNMENT, left, right, operator, source_reference);
}


static void vala_assignment_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_assignment (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static void vala_assignment_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_assignment_get_left (self), visitor);
	vala_code_node_accept ((ValaCodeNode*) vala_assignment_get_right (self), visitor);
}


static void vala_assignment_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_assignment_get_left (self) == old_node) {
		vala_assignment_set_left (self, new_node);
	}
	if (vala_assignment_get_right (self) == old_node) {
		vala_assignment_set_right (self, new_node);
	}
}


static gboolean vala_assignment_real_is_pure (ValaExpression* base) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	return FALSE;
}


static gboolean vala_assignment_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaAssignment * self;
	gboolean _tmp29;
	GeeList* _tmp111;
	GeeList* _tmp112;
	self = (ValaAssignment*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_expression_set_lvalue (vala_assignment_get_left (self), TRUE);
	if (!vala_code_node_check ((ValaCodeNode*) vala_assignment_get_left (self), analyzer)) {
		/* skip on error in inner expression*/
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self))) {
		ValaMemberAccess* _tmp2;
		ValaMemberAccess* ma;
		gboolean _tmp3;
		gboolean _tmp4;
		gboolean _tmp9;
		_tmp2 = NULL;
		ma = (_tmp2 = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
		_tmp3 = FALSE;
		_tmp4 = FALSE;
		if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
			_tmp4 = TRUE;
		} else {
			_tmp4 = VALA_IS_DYNAMIC_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma));
		}
		if (!_tmp4) {
			_tmp3 = vala_expression_get_value_type ((ValaExpression*) ma) == NULL;
		} else {
			_tmp3 = FALSE;
		}
		if (_tmp3) {
			gboolean _tmp5;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "unsupported lvalue in assignment");
			return (_tmp5 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp5);
		}
		if (vala_member_access_get_prototype_access (ma)) {
			char* _tmp7;
			char* _tmp6;
			gboolean _tmp8;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp7 = NULL;
			_tmp6 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp7 = g_strdup_printf ("Access to instance member `%s' denied", _tmp6 = vala_symbol_get_full_name (vala_expression_get_symbol_reference ((ValaExpression*) ma))));
			_tmp7 = (g_free (_tmp7), NULL);
			_tmp6 = (g_free (_tmp6), NULL);
			return (_tmp8 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp8);
		}
		_tmp9 = FALSE;
		if (vala_code_node_get_error ((ValaCodeNode*) ma)) {
			_tmp9 = TRUE;
		} else {
			_tmp9 = vala_expression_get_symbol_reference ((ValaExpression*) ma) == NULL;
		}
		if (_tmp9) {
			gboolean _tmp10;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			/* if no symbol found, skip this check */
			return (_tmp10 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp10);
		}
		if (VALA_IS_DYNAMIC_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
		} else {
			if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
				ValaSignal* _tmp11;
				ValaSignal* sig;
				ValaDelegateType* _tmp13;
				ValaDelegate* _tmp12;
				_tmp11 = NULL;
				sig = (_tmp11 = VALA_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp11 == NULL) ? NULL : vala_code_node_ref (_tmp11));
				_tmp13 = NULL;
				_tmp12 = NULL;
				vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp13 = vala_delegate_type_new (_tmp12 = vala_signal_get_delegate (sig, vala_expression_get_value_type (vala_member_access_get_inner (ma)), (ValaCodeNode*) self))));
				(_tmp13 == NULL) ? NULL : (_tmp13 = (vala_code_node_unref (_tmp13), NULL));
				(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_code_node_unref (_tmp12), NULL));
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			} else {
				vala_expression_set_formal_target_type (vala_assignment_get_right (self), vala_expression_get_formal_value_type ((ValaExpression*) ma));
				vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type ((ValaExpression*) ma));
			}
		}
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	} else {
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (self))) {
			ValaElementAccess* _tmp14;
			ValaElementAccess* ea;
			gboolean _tmp15;
			_tmp14 = NULL;
			ea = (_tmp14 = VALA_ELEMENT_ACCESS (vala_assignment_get_left (self)), (_tmp14 == NULL) ? NULL : vala_code_node_ref (_tmp14));
			_tmp15 = FALSE;
			if (VALA_IS_MEMBER_ACCESS (vala_element_access_get_container (ea))) {
				_tmp15 = VALA_IS_SIGNAL (vala_expression_get_symbol_reference (vala_element_access_get_container (ea)));
			} else {
				_tmp15 = FALSE;
			}
			if (_tmp15) {
				ValaMemberAccess* _tmp16;
				ValaMemberAccess* ma;
				ValaSignal* _tmp17;
				ValaSignal* sig;
				ValaDelegateType* _tmp19;
				ValaDelegate* _tmp18;
				_tmp16 = NULL;
				ma = (_tmp16 = VALA_MEMBER_ACCESS (vala_element_access_get_container (ea)), (_tmp16 == NULL) ? NULL : vala_code_node_ref (_tmp16));
				_tmp17 = NULL;
				sig = (_tmp17 = VALA_SIGNAL (vala_expression_get_symbol_reference (vala_element_access_get_container (ea))), (_tmp17 == NULL) ? NULL : vala_code_node_ref (_tmp17));
				_tmp19 = NULL;
				_tmp18 = NULL;
				vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp19 = vala_delegate_type_new (_tmp18 = vala_signal_get_delegate (sig, vala_expression_get_value_type (vala_member_access_get_inner (ma)), (ValaCodeNode*) self))));
				(_tmp19 == NULL) ? NULL : (_tmp19 = (vala_code_node_unref (_tmp19), NULL));
				(_tmp18 == NULL) ? NULL : (_tmp18 = (vala_code_node_unref (_tmp18), NULL));
				(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			} else {
				ValaSymbol* _tmp20;
				gboolean _tmp21;
				_tmp20 = NULL;
				if ((_tmp21 = VALA_IS_METHOD (_tmp20 = vala_data_type_get_member (vala_expression_get_value_type (vala_element_access_get_container (ea)), "set")), (_tmp20 == NULL) ? NULL : (_tmp20 = (vala_code_node_unref (_tmp20), NULL)), _tmp21)) {
					ValaMemberAccess* _tmp22;
					ValaMethodCall* _tmp23;
					ValaMethodCall* set_call;
					gboolean _tmp26;
					_tmp22 = NULL;
					_tmp23 = NULL;
					set_call = (_tmp23 = vala_method_call_new ((ValaExpression*) (_tmp22 = vala_member_access_new (vala_element_access_get_container (ea), "set", NULL)), NULL), (_tmp22 == NULL) ? NULL : (_tmp22 = (vala_code_node_unref (_tmp22), NULL)), _tmp23);
					{
						GeeList* _tmp24;
						GeeIterator* _tmp25;
						GeeIterator* _e_it;
						_tmp24 = NULL;
						_tmp25 = NULL;
						_e_it = (_tmp25 = gee_iterable_iterator ((GeeIterable*) (_tmp24 = vala_element_access_get_indices (ea))), (_tmp24 == NULL) ? NULL : (_tmp24 = (gee_collection_object_unref (_tmp24), NULL)), _tmp25);
						while (gee_iterator_next (_e_it)) {
							ValaExpression* e;
							e = (ValaExpression*) gee_iterator_get (_e_it);
							vala_method_call_add_argument (set_call, e);
							(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
						}
						(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
					}
					vala_method_call_add_argument (set_call, vala_assignment_get_right (self));
					vala_code_node_replace_expression (vala_code_node_get_parent_node ((ValaCodeNode*) self), (ValaExpression*) self, (ValaExpression*) set_call);
					return (_tmp26 = vala_code_node_check ((ValaCodeNode*) set_call, analyzer), (set_call == NULL) ? NULL : (set_call = (vala_code_node_unref (set_call), NULL)), (ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL)), _tmp26);
				} else {
					vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type (vala_assignment_get_left (self)));
				}
			}
			(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
		} else {
			if (VALA_IS_POINTER_INDIRECTION (vala_assignment_get_left (self))) {
				vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type (vala_assignment_get_left (self)));
			} else {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "unsupported lvalue in assignment");
				return FALSE;
			}
		}
	}
	if (!vala_code_node_check ((ValaCodeNode*) vala_assignment_get_right (self), analyzer)) {
		/* skip on error in inner expression*/
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	_tmp29 = FALSE;
	if (self->priv->_operator != VALA_ASSIGNMENT_OPERATOR_SIMPLE) {
		_tmp29 = VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self));
	} else {
		_tmp29 = FALSE;
	}
	if (_tmp29) {
		ValaMemberAccess* _tmp30;
		ValaMemberAccess* ma;
		/* transform into simple assignment
		 FIXME: only do this if the backend doesn't support
		 the assignment natively*/
		_tmp30 = NULL;
		ma = (_tmp30 = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp30 == NULL) ? NULL : vala_code_node_ref (_tmp30));
		if (!VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
			ValaMemberAccess* old_value;
			ValaBinaryExpression* bin;
			ValaDataType* _tmp31;
			old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
			bin = vala_binary_expression_new (VALA_BINARY_OPERATOR_PLUS, (ValaExpression*) old_value, vala_assignment_get_right (self), NULL);
			vala_expression_set_target_type ((ValaExpression*) bin, vala_expression_get_target_type (vala_assignment_get_right (self)));
			_tmp31 = NULL;
			vala_expression_set_target_type (vala_assignment_get_right (self), _tmp31 = vala_data_type_copy (vala_expression_get_target_type (vala_assignment_get_right (self))));
			(_tmp31 == NULL) ? NULL : (_tmp31 = (vala_code_node_unref (_tmp31), NULL));
			vala_data_type_set_value_owned (vala_expression_get_target_type (vala_assignment_get_right (self)), FALSE);
			if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_OR) {
				vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_OR);
			} else {
				if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_AND) {
					vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_AND);
				} else {
					if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR) {
						vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_XOR);
					} else {
						if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_ADD) {
							vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_PLUS);
						} else {
							if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SUB) {
								vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MINUS);
							} else {
								if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_MUL) {
									vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MUL);
								} else {
									if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_DIV) {
										vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_DIV);
									} else {
										if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_PERCENT) {
											vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MOD);
										} else {
											if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT) {
												vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_LEFT);
											} else {
												if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT) {
													vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_RIGHT);
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			vala_assignment_set_right (self, (ValaExpression*) bin);
			vala_code_node_check ((ValaCodeNode*) vala_assignment_get_right (self), analyzer);
			vala_assignment_set_operator (self, VALA_ASSIGNMENT_OPERATOR_SIMPLE);
			(old_value == NULL) ? NULL : (old_value = (vala_code_node_unref (old_value), NULL));
			(bin == NULL) ? NULL : (bin = (vala_code_node_unref (bin), NULL));
		}
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	}
	if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference (vala_assignment_get_left (self)))) {
		ValaSignal* _tmp32;
		ValaSignal* sig;
		ValaMethod* _tmp34;
		ValaSymbol* _tmp33;
		ValaMethod* m;
		ValaDynamicSignal* _tmp37;
		ValaSignal* _tmp36;
		ValaDynamicSignal* dynamic_sig;
		ValaMemberAccess* _tmp39;
		ValaExpression* _tmp38;
		ValaMemberAccess* right_ma;
		_tmp32 = NULL;
		sig = (_tmp32 = VALA_SIGNAL (vala_expression_get_symbol_reference (vala_assignment_get_left (self))), (_tmp32 == NULL) ? NULL : vala_code_node_ref (_tmp32));
		_tmp34 = NULL;
		_tmp33 = NULL;
		m = (_tmp34 = (_tmp33 = vala_expression_get_symbol_reference (vala_assignment_get_right (self)), VALA_IS_METHOD (_tmp33) ? ((ValaMethod*) _tmp33) : NULL), (_tmp34 == NULL) ? NULL : vala_code_node_ref (_tmp34));
		if (m == NULL) {
			gboolean _tmp35;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), "unsupported expression for signal handler");
			return (_tmp35 = FALSE, (sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), _tmp35);
		}
		_tmp37 = NULL;
		_tmp36 = NULL;
		dynamic_sig = (_tmp37 = (_tmp36 = sig, VALA_IS_DYNAMIC_SIGNAL (_tmp36) ? ((ValaDynamicSignal*) _tmp36) : NULL), (_tmp37 == NULL) ? NULL : vala_code_node_ref (_tmp37));
		_tmp39 = NULL;
		_tmp38 = NULL;
		right_ma = (_tmp39 = (_tmp38 = vala_assignment_get_right (self), VALA_IS_MEMBER_ACCESS (_tmp38) ? ((ValaMemberAccess*) _tmp38) : NULL), (_tmp39 == NULL) ? NULL : vala_code_node_ref (_tmp39));
		if (dynamic_sig != NULL) {
			gboolean first;
			ValaDelegateType* _tmp45;
			ValaDelegate* _tmp44;
			ValaObjectType* _tmp43;
			first = TRUE;
			{
				GeeList* _tmp40;
				GeeIterator* _tmp41;
				GeeIterator* _param_it;
				_tmp40 = NULL;
				_tmp41 = NULL;
				_param_it = (_tmp41 = gee_iterable_iterator ((GeeIterable*) (_tmp40 = vala_data_type_get_parameters (vala_expression_get_value_type (vala_dynamic_signal_get_handler (dynamic_sig))))), (_tmp40 == NULL) ? NULL : (_tmp40 = (gee_collection_object_unref (_tmp40), NULL)), _tmp41);
				while (gee_iterator_next (_param_it)) {
					ValaFormalParameter* param;
					param = (ValaFormalParameter*) gee_iterator_get (_param_it);
					if (first) {
						/* skip sender parameter*/
						first = FALSE;
					} else {
						ValaFormalParameter* _tmp42;
						_tmp42 = NULL;
						vala_signal_add_parameter ((ValaSignal*) dynamic_sig, _tmp42 = vala_formal_parameter_copy (param));
						(_tmp42 == NULL) ? NULL : (_tmp42 = (vala_code_node_unref (_tmp42), NULL));
					}
					(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				}
				(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
			}
			_tmp45 = NULL;
			_tmp44 = NULL;
			_tmp43 = NULL;
			vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp45 = vala_delegate_type_new (_tmp44 = vala_signal_get_delegate (sig, (ValaDataType*) (_tmp43 = vala_object_type_new (VALA_OBJECT_TYPE_SYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) sig)))), (ValaCodeNode*) self))));
			(_tmp45 == NULL) ? NULL : (_tmp45 = (vala_code_node_unref (_tmp45), NULL));
			(_tmp44 == NULL) ? NULL : (_tmp44 = (vala_code_node_unref (_tmp44), NULL));
			(_tmp43 == NULL) ? NULL : (_tmp43 = (vala_code_node_unref (_tmp43), NULL));
		} else {
			if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_target_type (vala_assignment_get_right (self)))) {
				ValaDelegateType* _tmp46;
				ValaDelegateType* delegate_type;
				char* _tmp50;
				char* _tmp49;
				char* _tmp48;
				char* _tmp47;
				gboolean _tmp51;
				_tmp46 = NULL;
				delegate_type = (_tmp46 = VALA_DELEGATE_TYPE (vala_expression_get_target_type (vala_assignment_get_right (self))), (_tmp46 == NULL) ? NULL : vala_code_node_ref (_tmp46));
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp50 = NULL;
				_tmp49 = NULL;
				_tmp48 = NULL;
				_tmp47 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), _tmp50 = g_strdup_printf ("method `%s' is incompatible with signal `%s', expected `%s'", _tmp47 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp48 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_target_type (vala_assignment_get_right (self))), _tmp49 = vala_delegate_get_prototype_string (vala_delegate_type_get_delegate_symbol (delegate_type), vala_symbol_get_name ((ValaSymbol*) m))));
				_tmp50 = (g_free (_tmp50), NULL);
				_tmp49 = (g_free (_tmp49), NULL);
				_tmp48 = (g_free (_tmp48), NULL);
				_tmp47 = (g_free (_tmp47), NULL);
				return (_tmp51 = FALSE, (delegate_type == NULL) ? NULL : (delegate_type = (vala_code_node_unref (delegate_type), NULL)), (sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL)), (right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL)), _tmp51);
			} else {
				gboolean _tmp52;
				_tmp52 = FALSE;
				if (right_ma != NULL) {
					_tmp52 = vala_member_access_get_prototype_access (right_ma);
				} else {
					_tmp52 = FALSE;
				}
				if (_tmp52) {
					char* _tmp54;
					char* _tmp53;
					gboolean _tmp55;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp54 = NULL;
					_tmp53 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), _tmp54 = g_strdup_printf ("Access to instance member `%s' denied", _tmp53 = vala_symbol_get_full_name ((ValaSymbol*) m)));
					_tmp54 = (g_free (_tmp54), NULL);
					_tmp53 = (g_free (_tmp53), NULL);
					return (_tmp55 = FALSE, (sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL)), (right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL)), _tmp55);
				}
			}
		}
		(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
		(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		(dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL));
		(right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL));
	} else {
		if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self))) {
			ValaMemberAccess* _tmp56;
			ValaMemberAccess* ma;
			gboolean _tmp92;
			_tmp56 = NULL;
			ma = (_tmp56 = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp56 == NULL) ? NULL : vala_code_node_ref (_tmp56));
			if (VALA_IS_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
				ValaProperty* _tmp57;
				ValaProperty* prop;
				ValaDynamicProperty* _tmp59;
				ValaProperty* _tmp58;
				ValaDynamicProperty* dynamic_prop;
				gboolean _tmp62;
				_tmp57 = NULL;
				prop = (_tmp57 = VALA_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp57 == NULL) ? NULL : vala_code_node_ref (_tmp57));
				_tmp59 = NULL;
				_tmp58 = NULL;
				dynamic_prop = (_tmp59 = (_tmp58 = prop, VALA_IS_DYNAMIC_PROPERTY (_tmp58) ? ((ValaDynamicProperty*) _tmp58) : NULL), (_tmp59 == NULL) ? NULL : vala_code_node_ref (_tmp59));
				if (dynamic_prop != NULL) {
					ValaDataType* _tmp60;
					ValaDataType* _tmp61;
					_tmp60 = NULL;
					vala_property_set_property_type ((ValaProperty*) dynamic_prop, _tmp60 = vala_data_type_copy (vala_expression_get_value_type (vala_assignment_get_right (self))));
					(_tmp60 == NULL) ? NULL : (_tmp60 = (vala_code_node_unref (_tmp60), NULL));
					_tmp61 = NULL;
					vala_expression_set_value_type (vala_assignment_get_left (self), _tmp61 = vala_data_type_copy (vala_property_get_property_type ((ValaProperty*) dynamic_prop)));
					(_tmp61 == NULL) ? NULL : (_tmp61 = (vala_code_node_unref (_tmp61), NULL));
				}
				_tmp62 = FALSE;
				if (vala_property_get_set_accessor (prop) == NULL) {
					_tmp62 = TRUE;
				} else {
					gboolean _tmp63;
					_tmp63 = FALSE;
					if (!vala_property_accessor_get_writable (vala_property_get_set_accessor (prop))) {
						gboolean _tmp64;
						ValaMethod* _tmp65;
						gboolean _tmp66;
						_tmp64 = FALSE;
						_tmp65 = NULL;
						if ((_tmp66 = VALA_IS_CREATION_METHOD (_tmp65 = vala_semantic_analyzer_find_current_method (analyzer)), (_tmp65 == NULL) ? NULL : (_tmp65 = (vala_code_node_unref (_tmp65), NULL)), _tmp66)) {
							_tmp64 = TRUE;
						} else {
							_tmp64 = vala_semantic_analyzer_is_in_constructor (analyzer);
						}
						_tmp63 = !_tmp64;
					} else {
						_tmp63 = FALSE;
					}
					_tmp62 = _tmp63;
				}
				if (_tmp62) {
					char* _tmp68;
					char* _tmp67;
					gboolean _tmp69;
					vala_code_node_set_error ((ValaCodeNode*) ma, TRUE);
					_tmp68 = NULL;
					_tmp67 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) ma), _tmp68 = g_strdup_printf ("Property `%s' is read-only", _tmp67 = vala_symbol_get_full_name ((ValaSymbol*) prop)));
					_tmp68 = (g_free (_tmp68), NULL);
					_tmp67 = (g_free (_tmp67), NULL);
					return (_tmp69 = FALSE, (prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL)), (dynamic_prop == NULL) ? NULL : (dynamic_prop = (vala_code_node_unref (dynamic_prop), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp69);
				}
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				(dynamic_prop == NULL) ? NULL : (dynamic_prop = (vala_code_node_unref (dynamic_prop), NULL));
			} else {
				gboolean _tmp70;
				_tmp70 = FALSE;
				if (VALA_IS_LOCAL_VARIABLE (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
					_tmp70 = vala_expression_get_value_type (vala_assignment_get_right (self)) == NULL;
				} else {
					_tmp70 = FALSE;
				}
				if (_tmp70) {
					ValaLocalVariable* _tmp71;
					ValaLocalVariable* local;
					gboolean _tmp72;
					_tmp71 = NULL;
					local = (_tmp71 = VALA_LOCAL_VARIABLE (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp71 == NULL) ? NULL : vala_code_node_ref (_tmp71));
					_tmp72 = FALSE;
					if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self)))) {
						_tmp72 = VALA_IS_DELEGATE_TYPE (vala_local_variable_get_variable_type (local));
					} else {
						_tmp72 = FALSE;
					}
					if (_tmp72) {
						ValaMethod* _tmp73;
						ValaMethod* m;
						ValaDelegateType* _tmp74;
						ValaDelegateType* dt;
						ValaDelegate* _tmp75;
						ValaDelegate* cb;
						_tmp73 = NULL;
						m = (_tmp73 = VALA_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self))), (_tmp73 == NULL) ? NULL : vala_code_node_ref (_tmp73));
						_tmp74 = NULL;
						dt = (_tmp74 = VALA_DELEGATE_TYPE (vala_local_variable_get_variable_type (local)), (_tmp74 == NULL) ? NULL : vala_code_node_ref (_tmp74));
						_tmp75 = NULL;
						cb = (_tmp75 = vala_delegate_type_get_delegate_symbol (dt), (_tmp75 == NULL) ? NULL : vala_code_node_ref (_tmp75));
						/* check whether method matches callback type */
						if (!vala_delegate_matches_method (cb, m)) {
							char* _tmp78;
							char* _tmp77;
							char* _tmp76;
							gboolean _tmp79;
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							_tmp78 = NULL;
							_tmp77 = NULL;
							_tmp76 = NULL;
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp78 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", _tmp76 = vala_symbol_get_full_name ((ValaSymbol*) m), _tmp77 = vala_symbol_get_full_name ((ValaSymbol*) cb)));
							_tmp78 = (g_free (_tmp78), NULL);
							_tmp77 = (g_free (_tmp77), NULL);
							_tmp76 = (g_free (_tmp76), NULL);
							return (_tmp79 = FALSE, (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), (cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL)), (local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp79);
						}
						vala_expression_set_value_type (vala_assignment_get_right (self), vala_local_variable_get_variable_type (local));
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
						(dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL));
						(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
					} else {
						gboolean _tmp80;
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Assignment: Invalid callback assignment attempt");
						return (_tmp80 = FALSE, (local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp80);
					}
					(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
				} else {
					gboolean _tmp81;
					_tmp81 = FALSE;
					if (VALA_IS_FIELD (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
						_tmp81 = vala_expression_get_value_type (vala_assignment_get_right (self)) == NULL;
					} else {
						_tmp81 = FALSE;
					}
					if (_tmp81) {
						ValaField* _tmp82;
						ValaField* f;
						gboolean _tmp83;
						_tmp82 = NULL;
						f = (_tmp82 = VALA_FIELD (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp82 == NULL) ? NULL : vala_code_node_ref (_tmp82));
						_tmp83 = FALSE;
						if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self)))) {
							_tmp83 = VALA_IS_DELEGATE_TYPE (vala_field_get_field_type (f));
						} else {
							_tmp83 = FALSE;
						}
						if (_tmp83) {
							ValaMethod* _tmp84;
							ValaMethod* m;
							ValaDelegateType* _tmp85;
							ValaDelegateType* dt;
							ValaDelegate* _tmp86;
							ValaDelegate* cb;
							_tmp84 = NULL;
							m = (_tmp84 = VALA_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self))), (_tmp84 == NULL) ? NULL : vala_code_node_ref (_tmp84));
							_tmp85 = NULL;
							dt = (_tmp85 = VALA_DELEGATE_TYPE (vala_field_get_field_type (f)), (_tmp85 == NULL) ? NULL : vala_code_node_ref (_tmp85));
							_tmp86 = NULL;
							cb = (_tmp86 = vala_delegate_type_get_delegate_symbol (dt), (_tmp86 == NULL) ? NULL : vala_code_node_ref (_tmp86));
							/* check whether method matches callback type */
							if (!vala_delegate_matches_method (cb, m)) {
								char* _tmp89;
								char* _tmp88;
								char* _tmp87;
								gboolean _tmp90;
								vala_code_node_set_error ((ValaCodeNode*) f, TRUE);
								_tmp89 = NULL;
								_tmp88 = NULL;
								_tmp87 = NULL;
								vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp89 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", _tmp87 = vala_symbol_get_full_name ((ValaSymbol*) m), _tmp88 = vala_symbol_get_full_name ((ValaSymbol*) cb)));
								_tmp89 = (g_free (_tmp89), NULL);
								_tmp88 = (g_free (_tmp88), NULL);
								_tmp87 = (g_free (_tmp87), NULL);
								return (_tmp90 = FALSE, (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), (cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL)), (f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp90);
							}
							vala_expression_set_value_type (vala_assignment_get_right (self), vala_field_get_field_type (f));
							(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
							(dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL));
							(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
						} else {
							gboolean _tmp91;
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Assignment: Invalid callback assignment attempt");
							return (_tmp91 = FALSE, (f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp91);
						}
						(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
					}
				}
			}
			_tmp92 = FALSE;
			if (vala_expression_get_value_type (vala_assignment_get_left (self)) != NULL) {
				_tmp92 = vala_expression_get_value_type (vala_assignment_get_right (self)) != NULL;
			} else {
				_tmp92 = FALSE;
			}
			if (_tmp92) {
				/* if there was an error on either side,
				 * i.e. {left|right}.value_type == null, skip type check */
				if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					char* _tmp95;
					char* _tmp94;
					char* _tmp93;
					gboolean _tmp96;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp95 = NULL;
					_tmp94 = NULL;
					_tmp93 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp95 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", _tmp93 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp94 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_left (self)))));
					_tmp95 = (g_free (_tmp95), NULL);
					_tmp94 = (g_free (_tmp94), NULL);
					_tmp93 = (g_free (_tmp93), NULL);
					return (_tmp96 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp96);
				}
				if (!VALA_IS_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
					if (vala_data_type_is_disposable (vala_expression_get_value_type (vala_assignment_get_right (self)))) {
						gboolean _tmp97;
						_tmp97 = FALSE;
						if (!VALA_IS_POINTER_TYPE (vala_expression_get_value_type (vala_assignment_get_left (self)))) {
							_tmp97 = !vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)));
						} else {
							_tmp97 = FALSE;
						}
						/* rhs transfers ownership of the expression */
						if (_tmp97) {
							/* lhs doesn't own the value */
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Invalid assignment from owned expression to unowned variable");
						}
					} else {
						if (vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)))) {
						}
					}
				}
			}
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
		} else {
			if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (self))) {
				ValaElementAccess* _tmp98;
				ValaElementAccess* ea;
				_tmp98 = NULL;
				ea = (_tmp98 = VALA_ELEMENT_ACCESS (vala_assignment_get_left (self)), (_tmp98 == NULL) ? NULL : vala_code_node_ref (_tmp98));
				if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					char* _tmp101;
					char* _tmp100;
					char* _tmp99;
					gboolean _tmp102;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp101 = NULL;
					_tmp100 = NULL;
					_tmp99 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp101 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", _tmp99 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp100 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_left (self)))));
					_tmp101 = (g_free (_tmp101), NULL);
					_tmp100 = (g_free (_tmp100), NULL);
					_tmp99 = (g_free (_tmp99), NULL);
					return (_tmp102 = FALSE, (ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL)), _tmp102);
				}
				if (vala_data_type_is_disposable (vala_expression_get_value_type (vala_assignment_get_right (self)))) {
					ValaDataType* element_type;
					gboolean _tmp107;
					/* rhs transfers ownership of the expression */
					element_type = NULL;
					if (VALA_IS_ARRAY_TYPE (vala_expression_get_value_type (vala_element_access_get_container (ea)))) {
						ValaArrayType* _tmp103;
						ValaArrayType* array_type;
						ValaDataType* _tmp105;
						ValaDataType* _tmp104;
						_tmp103 = NULL;
						array_type = (_tmp103 = VALA_ARRAY_TYPE (vala_expression_get_value_type (vala_element_access_get_container (ea))), (_tmp103 == NULL) ? NULL : vala_code_node_ref (_tmp103));
						_tmp105 = NULL;
						_tmp104 = NULL;
						element_type = (_tmp105 = (_tmp104 = vala_array_type_get_element_type (array_type), (_tmp104 == NULL) ? NULL : vala_code_node_ref (_tmp104)), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp105);
						(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
					} else {
						GeeList* args;
						ValaDataType* _tmp106;
						args = vala_data_type_get_type_arguments (vala_expression_get_value_type (vala_element_access_get_container (ea)));
						g_assert (gee_collection_get_size ((GeeCollection*) args) == 1);
						_tmp106 = NULL;
						element_type = (_tmp106 = (ValaDataType*) gee_list_get (args, 0), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp106);
						(args == NULL) ? NULL : (args = (gee_collection_object_unref (args), NULL));
					}
					_tmp107 = FALSE;
					if (!VALA_IS_POINTER_TYPE (element_type)) {
						_tmp107 = !vala_data_type_get_value_owned (element_type);
					} else {
						_tmp107 = FALSE;
					}
					if (_tmp107) {
						gboolean _tmp108;
						/* lhs doesn't own the value */
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Invalid assignment from owned expression to unowned variable");
						return (_tmp108 = FALSE, (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), (ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL)), _tmp108);
					}
					(element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL));
				} else {
					if (vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					}
				}
				(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
			} else {
				/* lhs wants to own the value
				 * rhs doesn't transfer the ownership
				 * code generator needs to add reference
				 * increment calls */
				return TRUE;
			}
		}
	}
	if (vala_expression_get_value_type (vala_assignment_get_left (self)) != NULL) {
		ValaDataType* _tmp110;
		_tmp110 = NULL;
		vala_expression_set_value_type ((ValaExpression*) self, _tmp110 = vala_data_type_copy (vala_expression_get_value_type (vala_assignment_get_left (self))));
		(_tmp110 == NULL) ? NULL : (_tmp110 = (vala_code_node_unref (_tmp110), NULL));
		vala_data_type_set_value_owned (vala_expression_get_value_type ((ValaExpression*) self), FALSE);
	} else {
		vala_expression_set_value_type ((ValaExpression*) self, NULL);
	}
	_tmp111 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp111 = vala_code_node_get_error_types ((ValaCodeNode*) vala_assignment_get_left (self)));
	(_tmp111 == NULL) ? NULL : (_tmp111 = (gee_collection_object_unref (_tmp111), NULL));
	_tmp112 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp112 = vala_code_node_get_error_types ((ValaCodeNode*) vala_assignment_get_right (self)));
	(_tmp112 == NULL) ? NULL : (_tmp112 = (gee_collection_object_unref (_tmp112), NULL));
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


static void vala_assignment_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaAssignment * self;
	ValaLocalVariable* _tmp1;
	ValaSymbol* _tmp0;
	ValaLocalVariable* local;
	self = (ValaAssignment*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_assignment_get_right (self), collection);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_assignment_get_left (self), collection);
	_tmp1 = NULL;
	_tmp0 = NULL;
	local = (_tmp1 = (_tmp0 = vala_expression_get_symbol_reference (vala_assignment_get_left (self)), VALA_IS_LOCAL_VARIABLE (_tmp0) ? ((ValaLocalVariable*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	if (local != NULL) {
		gee_collection_add (collection, local);
	}
	(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
}


static void vala_assignment_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaAssignment * self;
	ValaMemberAccess* _tmp1;
	ValaExpression* _tmp0;
	ValaMemberAccess* ma;
	ValaElementAccess* _tmp3;
	ValaExpression* _tmp2;
	ValaElementAccess* ea;
	gboolean _tmp4;
	self = (ValaAssignment*) base;
	g_return_if_fail (collection != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	ma = (_tmp1 = (_tmp0 = vala_assignment_get_left (self), VALA_IS_MEMBER_ACCESS (_tmp0) ? ((ValaMemberAccess*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	_tmp3 = NULL;
	_tmp2 = NULL;
	ea = (_tmp3 = (_tmp2 = vala_assignment_get_left (self), VALA_IS_ELEMENT_ACCESS (_tmp2) ? ((ValaElementAccess*) _tmp2) : NULL), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3));
	_tmp4 = FALSE;
	if (ma != NULL) {
		_tmp4 = vala_member_access_get_inner (ma) != NULL;
	} else {
		_tmp4 = FALSE;
	}
	if (_tmp4) {
		vala_code_node_get_used_variables ((ValaCodeNode*) vala_member_access_get_inner (ma), collection);
	} else {
		if (ea != NULL) {
			vala_code_node_get_used_variables ((ValaCodeNode*) ea, collection);
		}
	}
	vala_code_node_get_used_variables ((ValaCodeNode*) vala_assignment_get_right (self), collection);
	(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
}


ValaExpression* vala_assignment_get_left (ValaAssignment* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_left;
}


void vala_assignment_set_left (ValaAssignment* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_left = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_code_node_unref (self->priv->_left), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_left, (ValaCodeNode*) self);
}


ValaAssignmentOperator vala_assignment_get_operator (ValaAssignment* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_operator;
}


void vala_assignment_set_operator (ValaAssignment* self, ValaAssignmentOperator value) {
	g_return_if_fail (self != NULL);
	self->priv->_operator = value;
}


ValaExpression* vala_assignment_get_right (ValaAssignment* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_right;
}


void vala_assignment_set_right (ValaAssignment* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_right = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_code_node_unref (self->priv->_right), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_right, (ValaCodeNode*) self);
}


static void vala_assignment_class_init (ValaAssignmentClass * klass) {
	vala_assignment_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_assignment_finalize;
	g_type_class_add_private (klass, sizeof (ValaAssignmentPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_assignment_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_assignment_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_assignment_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_assignment_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->check = vala_assignment_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_assignment_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_assignment_real_get_used_variables;
}


static void vala_assignment_instance_init (ValaAssignment * self) {
	self->priv = VALA_ASSIGNMENT_GET_PRIVATE (self);
}


static void vala_assignment_finalize (ValaCodeNode* obj) {
	ValaAssignment * self;
	self = VALA_ASSIGNMENT (obj);
	(self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_code_node_unref (self->priv->_left), NULL));
	(self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_code_node_unref (self->priv->_right), NULL));
	VALA_CODE_NODE_CLASS (vala_assignment_parent_class)->finalize (obj);
}


GType vala_assignment_get_type (void) {
	static GType vala_assignment_type_id = 0;
	if (vala_assignment_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaAssignmentClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_assignment_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaAssignment), 0, (GInstanceInitFunc) vala_assignment_instance_init, NULL };
		vala_assignment_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaAssignment", &g_define_type_info, 0);
	}
	return vala_assignment_type_id;
}



GType vala_assignment_operator_get_type (void) {
	static GType vala_assignment_operator_type_id = 0;
	if (G_UNLIKELY (vala_assignment_operator_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_ASSIGNMENT_OPERATOR_NONE, "VALA_ASSIGNMENT_OPERATOR_NONE", "none"}, {VALA_ASSIGNMENT_OPERATOR_SIMPLE, "VALA_ASSIGNMENT_OPERATOR_SIMPLE", "simple"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_OR, "VALA_ASSIGNMENT_OPERATOR_BITWISE_OR", "bitwise-or"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_AND, "VALA_ASSIGNMENT_OPERATOR_BITWISE_AND", "bitwise-and"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR, "VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR", "bitwise-xor"}, {VALA_ASSIGNMENT_OPERATOR_ADD, "VALA_ASSIGNMENT_OPERATOR_ADD", "add"}, {VALA_ASSIGNMENT_OPERATOR_SUB, "VALA_ASSIGNMENT_OPERATOR_SUB", "sub"}, {VALA_ASSIGNMENT_OPERATOR_MUL, "VALA_ASSIGNMENT_OPERATOR_MUL", "mul"}, {VALA_ASSIGNMENT_OPERATOR_DIV, "VALA_ASSIGNMENT_OPERATOR_DIV", "div"}, {VALA_ASSIGNMENT_OPERATOR_PERCENT, "VALA_ASSIGNMENT_OPERATOR_PERCENT", "percent"}, {VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT, "VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT", "shift-left"}, {VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT, "VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT", "shift-right"}, {0, NULL, NULL}};
		vala_assignment_operator_type_id = g_enum_register_static ("ValaAssignmentOperator", values);
	}
	return vala_assignment_operator_type_id;
}




